#!/bin/bash
#
# Copyright (c) 2016-2017 by Cisco Systems, Inc.
#
# Provides VRF app hosting functions
#
# Neil McGill
#

APP_HOSTING_APPLY_LOCK_FILE=/misc/app_host/app_hosting_apply_cmd/.lock

#
# Can we execute the VRF command in the local VM ?
#
function app_hosting_is_vrf_infra_local()
{
    if app_hosting_is_vm; then
        true
        return
    fi

    if [[ "$VMTYPE" = "" ]]; then
        VMTYPE=`cat /proc/cmdline | sed 's/^.*vmtype=//' | cut -d" " -f1`
    fi

    if [[ "$VMTYPE" = "hostos" ]]; then
        true
        return
    fi

    false
}

#
# Take a lock to ensure only one apply command user at a time.
#
function app_hosting_apply_lock_assert()
{
    HAVE_LOCK_FILE=

    local TRIES=0

    while [ $TRIES -lt 5 ]
    do
        TRIES=$(expr $TRIES + 1)

        if [[ -f $APP_HOSTING_APPLY_LOCK_FILE ]]; then
            platform_log "Waiting on lock, $APP_HOSTING_APPLY_LOCK_FILE"
            sleep 1
            continue
        fi

        echo $MYPID > $APP_HOSTING_APPLY_LOCK_FILE
        if [[ $? -ne 0 ]]; then
            platform_log "Could not write to lock file, $APP_HOSTING_APPLY_LOCK_FILE"
            sleep 1
            continue
        fi

        #
        # Make sure anyone else can remove the lockfile if needed
        #
        chmod aog+w $APP_HOSTING_APPLY_LOCK_FILE 2>/dev/null

        local LOCK_PID=`cat $APP_HOSTING_APPLY_LOCK_FILE 2>/dev/null`
        if [[ $? -ne 0 ]]; then
            platform_log "Could not read lock file, $APP_HOSTING_APPLY_LOCK_FILE"
            sleep 1
            continue
        fi

        if [[ "$LOCK_PID" != "$MYPID" ]]; then
            platform_log "App hosting lock file grabbed by PID $LOCK_PID"
            sleep 1
            continue
        fi

        HAVE_LOCK_FILE=$APP_HOSTING_APPLY_LOCK_FILE
        break
    done

    if [[ "$HAVE_LOCK_FILE" = "" ]]; then
        platform_log_error "Could not grab app hosting lock"
    fi
}

#
# Release the lock.
#
function app_hosting_apply_lock_release()
{
    if [[ "$HAVE_LOCK_FILE" = "" ]]; then
        #
        # In case we grabbed the lock but died shortly before setting
        # HAVE_LOCK_FILE, check if our pid is in there.
        #
        if [[ -f $APP_HOSTING_APPLY_LOCK_FILE ]]; then
            local LOCK_PID=`cat $APP_HOSTING_APPLY_LOCK_FILE 2>/dev/null`
            if [[ $? -ne 0 ]]; then
                return
            fi

            if [[ "$LOCK_PID" != "$MYPID" ]]; then
                return
            fi

            HAVE_LOCK_FILE=1
        fi
    fi

    if [[ "$HAVE_LOCK_FILE" = "" ]]; then
        return
    fi

    rm -f $HAVE_LOCK_FILE
    HAVE_LOCK_FILE=
}

#
# Run a command on the host and then echo stdout/stderr from
# this command locally. It's like a limited form of remote shell.
#
function app_hosting_apply_cmd()
{
    local exitcode=
    local APP_HOSTING_APPLY_CMD_DIR=/misc/app_host/app_hosting_apply_cmd
    local APP_HOSTING_APPLY_CMD_OUTPUT=$APP_HOSTING_APPLY_CMD_DIR/app_hosting_apply_cmd.output
    local APP_HOSTING_APPLY_CMD_STDOUT=$APP_HOSTING_APPLY_CMD_DIR/app_hosting_apply_cmd.stdout
    local APP_HOSTING_APPLY_CMD_STDERR=$APP_HOSTING_APPLY_CMD_DIR/app_hosting_apply_cmd.stderr
    local APP_HOSTING_APPLY_CMD_EXITCODE=$APP_HOSTING_APPLY_CMD_DIR/app_hosting_apply_cmd.exitcode

    mkdir -p $APP_HOSTING_APPLY_CMD_DIR

    /bin/rm -f $APP_HOSTING_APPLY_CMD_OUTPUT &>/dev/null
    /bin/rm -f $APP_HOSTING_APPLY_CMD_STDOUT &>/dev/null
    /bin/rm -f $APP_HOSTING_APPLY_CMD_STDERR &>/dev/null
    /bin/rm -f $APP_HOSTING_APPLY_CMD_EXITCODE &>/dev/null

    app_hosting_apply_lock_assert

    /usr/sbin/app_hosting_apply_cmd_client.py \
        --cmd "$*" \
        --socket $APP_HOSTING_APPLY_CMD_DIR/socket \
        --stdout_path $APP_HOSTING_APPLY_CMD_STDOUT \
        --stderr_path $APP_HOSTING_APPLY_CMD_STDERR \
        --exitcode $APP_HOSTING_APPLY_CMD_EXITCODE \
        >$APP_HOSTING_APPLY_CMD_OUTPUT 2>&1

    exitcode=$?

    app_hosting_apply_lock_release

    #
    # If the tool itself fails, like the socket is removed, for example,
    # then print an error to enlighten.
    #
    cat $APP_HOSTING_APPLY_CMD_OUTPUT >> $APP_HOSTING_LOG_FILE
    if [[ $exitcode -ne 0 ]]; then
        cat $APP_HOSTING_APPLY_CMD_OUTPUT 
    fi

    if [[ -f $APP_HOSTING_APPLY_CMD_STDOUT ]]; then
        cat $APP_HOSTING_APPLY_CMD_STDOUT >&1
    fi

    if [[ -f $APP_HOSTING_APPLY_CMD_STDERR ]]; then
        cat $APP_HOSTING_APPLY_CMD_STDERR >&2
    fi

    if [[ -f $APP_HOSTING_APPLY_CMD_EXITCODE ]]; then
        return $(cat $APP_HOSTING_APPLY_CMD_EXITCODE)
    fi

    return $exitcode
}

function app_hosting_vrf_debug_namespaces()
{
    app_hosting_log "all" "Namespaces:"

    (
        for i in $(ip netns list); do
            /bin/ls -lai /var/run/netns/$i; 
        done
    ) >> $APP_HOSTING_LOG_FILE 2>/dev/null

    true
}

#
# Create a VRF namespace.
#
# On some platforms this runs in the same kernel as XR (asr9k).
#
# On others this runs on the VM host and return values are sent back
# to XR via unix domain socket.
#
function app_hosting_vrf_create_namespace_server_side()
{
    local VRF=$1
    local exitcode=

    app_hosting_vrf_debug_namespaces

    app_hosting_log "$VRF" "Creating namespace on VM host"

    #
    # VRF already exists? Let's check it's sane.
    #
    if [[ -f "/var/run/netns/$VRF" ]]; then
        ip netns exec "$VRF" ifconfig >/dev/null
        exitcode=$?
        if [[ $exitcode -ne 0 ]]; then
            app_hosting_vrf_namespace_server_side_error \
                "$VRF" "Namespace exists but appears to be inaccessible"
            return $exitcode
        else
            app_hosting_log "$VRF" "Namespace exists and is valid"
            true
            return
        fi
    fi

    ip netns add "$VRF"
    exitcode=$?
    if [[ $exitcode -ne 0 ]]; then
        app_hosting_vrf_namespace_server_side_error \
            "$VRF" "Namespace create failed"
        return $exitcode
    fi

    app_hosting_vrf_debug_namespaces

    true
}

#
# Create a VRF namespace.
#
function app_hosting_vrf_create_namespace()
{
    local VRF=$1
    local exitcode=

    app_hosting_vrf_debug_namespaces

    if app_hosting_is_vrf_infra_local; then
        app_hosting_log "$VRF" "Creating namespace locally"

        app_hosting_vrf_create_namespace_server_side "$VRF"
        exitcode=$?
        if [[ $exitcode -ne 0 ]]; then
            app_hosting_log "$VRF" "Namespace create failed locally"
            return $exitcode
        fi
    else
        app_hosting_log "$VRF" "Creating namespace on VM host"

        app_hosting_apply_cmd "vrf add $VRF"
        exitcode=$?
        if [[ $exitcode -ne 0 ]]; then
            app_hosting_log "$VRF" "Creating namespace on VM host failed"
            return $exitcode
        fi

        #
        # Check it really worked. This is a shared mount and the inode
        # should be present now.
        #
        if [[ ! -f "/bindmnt_netns/$VRF" ]]; then
            app_hosting_log "$VRF" "Namespace created but shared mount not present"
            false
            return
        fi

        #
        # Make a symlink to the real inode.
        #
        ln -sf "/bindmnt_netns/$VRF" "/var/run/netns/$VRF"
        exitcode=$?
        if [[ $exitcode -ne 0 ]]; then
            app_hosting_log "$VRF" "Namespace created but symlink failed"
            false
            return
        fi
    fi

    app_hosting_vrf_debug_namespaces

    create_vrf_dirs "$VRF"

    true
}

#
# Ok to remove a VRF namespace?
#
# On some platforms this runs in the same kernel as XR (asr9k).
#
# On others this runs on the VM host and return values are sent back
# to XR via unix domain socket.
#
function app_hosting_vrf_ok_to_remove_namespace_server_side()
{
    local VRF=$1
    local exitcode=

    if [[ ! -f "/var/run/netns/$VRF" ]]; then
        true
        return
    fi

    app_hosting_vrf_destroy_namespace_server_side_check $VRF check-only &>/dev/null
    exitcode=$?
    if [[ $exitcode -ne 0 ]]; then
        app_hosting_log "$VRF" "Not safe to destroy namespace"
        echo "Namespace '$VRF' is in use. "
        echo "Use 'show processes namespace vrf $VRF' to list the processes using it."
        false
        return
    fi

    true
}

#
# Ok to remove a VRF namespace?
#
function app_hosting_vrf_ok_to_remove_namespace()
{
    local VRF=$1
    local exitcode=

    if app_hosting_is_vrf_infra_local; then
        app_hosting_vrf_ok_to_remove_namespace_server_side "$VRF"
        return
    fi

    app_hosting_apply_cmd "vrf ok_to_remove $VRF"
    exitcode=$?
    if [[ $exitcode -eq 0 ]]; then
        app_hosting_log "$VRF" "NOT ok to remove namespace on VM host"
    else
        app_hosting_log "$VRF" "Ok to remove namespace on VM host"
    fi

    return $exitcode
}

#
# Destroy a VRF namespace.
#
# On some platforms this runs in the same kernel as XR (asr9k).
#
# On others this runs on the VM host and return values are sent back
# to XR via unix domain socket.
#
function app_hosting_vrf_destroy_namespace_server_side()
{
    local VRF=$1
    local exitcode=

    app_hosting_vrf_debug_namespaces

    app_hosting_log "$VRF" "Destroying namespace on VM host"

    #
    # VRF is gone? Just ignore.
    #
    if [[ ! -f "/var/run/netns/$VRF" ]]; then
        app_hosting_log "$VRF" "Namespace already removed on VM host"
        true
        return
    fi

    #
    # Check we can destroy the namespace.
    #
    app_hosting_vrf_destroy_namespace_server_side_check "$VRF"
    exitcode=$?
    if [[ $exitcode -ne 0 ]]; then
        app_hosting_log "$VRF" "Not safe to destroy namespace"
        return $exitcode
    fi

    ip netns del "$VRF"
    exitcode=$?
    if [[ $exitcode -ne 0 ]]; then
        app_hosting_log "$VRF" "Failed to destroy namespace"
        return $exitcode
    fi

    app_hosting_vrf_debug_namespaces

    true
}

#
# Destroy a VRF namespace.
#
function app_hosting_vrf_destroy_namespace()
{
    local VRF=$1
    local exitcode=

    app_hosting_vrf_debug_namespaces

    if app_hosting_is_vrf_infra_local; then
        app_hosting_log "$VRF" "Destroying namespace locally"

        #
        # We want to hide stdout but collect stderr as we feed the results 
        # back to KIM. So just in case there is any future stdout, hide it.
        #
        app_hosting_vrf_destroy_namespace_server_side "$VRF" >/dev/null
        exitcode=$?
        if [[ $exitcode -ne 0 ]]; then
            app_hosting_log "$VRF" "Destroying namespace locally failed"
            return $exitcode
        fi
    else
        app_hosting_log "$VRF" "Destroying namespace on VM host"

        app_hosting_apply_cmd "vrf del $VRF"
        exitcode=$?
        if [[ $exitcode -ne 0 ]]; then
            app_hosting_log "$VRF" "Destroying namespace on VM host failed"
            return $exitcode
        fi
    fi

    app_hosting_vrf_debug_namespaces

    destroy_vrf_dirs "$VRF"

    true
}

#
# Show the containers and processes using a namespace.
#
function app_hosting_vrf_show_namespace_server_side()
{
    local VRF=$1
    local exitcode=

    local vrf_processes=""
    local running_containers=""
    local docker_containers=""
    local docker_vrf=()
    local lxc_containers=""
    local lxc_vrf=()

    source /etc/sysconfig/cisco_docker &>/dev/null
    export DOCKER_HOST=$DOCKER_DAEMON_SOCKET

    cat <<%%

Linux processes using VRF namespace $VRF:
================================================================================
%%

    #
    # If the VRF is gone, do we fail this to kim or not? I think allow it
    # but log an error.
    #
    ip netns list 2>/dev/null | grep -q "\<$VRF\>"
    if [[ $? -ne 0 ]]; then
        cat <<%%
Error. VRF namespace \"$VRF\" appears to be invalid. Known namespaces are:
%%
        for i in $(ip netns list); do
            /bin/ls -lai /var/run/netns/$i 2>&1
        done

        return
    fi

    vrf_processes=$(ip netns pids ${VRF} 2>/dev/null)
    ps_args=""
    for p in $vrf_processes
    do
        ps_args="$ps_args -p $p"
    done

    if [[ $ps_args = "" ]]; then
        echo "No process is using this VRF namespace."
    else
        ps $ps_args wwwww
    fi

    #
    # Do we have any docker containers?
    #
    docker_containers=$(docker ps -a 2>/dev/null | awk '{if(NR>1) print $NF}')
    for container_id in $docker_containers
    do
        docker inspect $container_id | grep -q "\"/var/run/netns\(/${VRF}\)\?\"" &>/dev/null
        if [[ $? -eq 0 ]]; then
            docker_vrf+=("$container_id")
        fi
    done

    #
    # Show the docker containers.
    #
    if [[ ${#docker_vrf[@]} -eq 0 ]]; then
        cat <<%%

No Docker container is using this VRF namespace.
%%
    else
        cat <<%%

Docker containers using VRF namespace $VRF:
================================================================================
${docker_vrf[@]}
%%
    fi

    #
    # Check lxc_containers using vrf
    #
    export LIBVIRT_DEFAULT_URI=lxc:///
    lxc_containers=$(virsh list --name)
    for container in $lxc_containers
    do
        virsh dumpxml $container | grep "<sharenet type='netns'" | grep -q "value='${VRF}'"
        if [[ $? -eq 0 ]]; then
            lxc_vrf+=("$container")
        fi
    done

    #
    # Show the lxc containers.
    #
    if [[ ${#lxc_vrf[@]} -eq 0 ]]; then
        cat <<%%

No LXC container is using this VRF namespace.
%%
    else
        cat <<%%

LXC containers using VRF namespace $VRF:
================================================================================
${lxc_vrf[@]}
%%
    fi

    #
    # Check to see if this VRF is removable.
    #
    if [[ "$VRF" != $OPERNS_NAME ]]; then
        app_hosting_vrf_destroy_namespace_server_side_check $VRF check-only &>/dev/null
        if [[ $? -ne 0 ]]; then
            cat <<%%

NOTE: VRF namespace $VRF cannot be removed until containers and processes using it are removed.
%%
        fi
    fi
}

#
# Show the processes running locally in XR using this VRF
#
function app_hosting_vrf_show_namespace_local_side()
{
    local VRF=$1

    local inode=$(ls -i "/bindmnt_netns/$VRF" 2>/dev/null | cut -f1 -d" ")
    if [[ "$inode" = "" ]]; then
        return
    fi

    local vrf_processes=$(find -L /proc/[1-9]*/task/*/ns/net -inum $inode 2>/dev/null | cut -f3 -d"/" | uniq)
    ps_args=""
    for p in $vrf_processes
    do
        ps_args="$ps_args -p $p"
    done

    if [[ $ps_args = "" ]]; then
        return
    fi

    cat <<%%

Linux processes using VRF namespace $VRF: (within the XR container)
================================================================================
%%
    ps $ps_args wwwww
    echo
}

#
# Show the containers and processes using a namespace.
#
function app_hosting_vrf_show_namespace()
{
    local VRF=$1
    local exitcode=

    app_hosting_vrf_debug_namespaces

    if app_hosting_is_vrf_infra_local; then
        app_hosting_log "$VRF" "showing namespace locally"

        app_hosting_vrf_show_namespace_server_side "$VRF"
        exitcode=$?
        if [[ $exitcode -ne 0 ]]; then
            app_hosting_log "$VRF" "showing namespace locally failed"
            return $exitcode
        fi
    else
        app_hosting_log "$VRF" "showing namespace on VM host"

        app_hosting_vrf_show_namespace_local_side "$VRF"

        app_hosting_apply_cmd "vrf show $VRF"
        exitcode=$?
        if [[ $exitcode -ne 0 ]]; then
            app_hosting_log "$VRF" "showing namespace on VM host failed"
            return $exitcode
        fi
    fi

    true
}

function create_vrf_dirs()
{
    local VRF=$1
    local VRF_PATH="/etc/netns/"

    app_hosting_log "$VRF" "Creating VRF dirs ${VRF_PATH}${VRF}"
    $PLATFORM_LOG_EXEC mkdir -p ${VRF_PATH}${VRF}
    $PLATFORM_LOG_EXEC touch /etc/resolv.conf
    $PLATFORM_LOG_EXEC touch ${VRF_PATH}${VRF}/resolv.conf
    $PLATFORM_LOG_EXEC cp /etc/hosts ${VRF_PATH}${VRF}/hosts
}

#
# Remove /etc/netns/$vrf dir when the vrf is deleted 
#
function destroy_vrf_dirs()
{
    local vrf=$1
    local vrf_path="/etc/netns/"

    app_hosting_log "$vrf" "Destroying VRF dirs ${vrf_path}${vrf}"
    $PLATFORM_LOG_EXEC rm -rf --preserve-root --one-file-system ${vrf_path}${vrf}
}

#
# VRF remove has failed. Log an error to stderr and the log file. The stderr
# string will be returned back to KIM for reporting.
#
function app_hosting_vrf_namespace_server_side_error()
{
    local VRF=$1
    shift

    echo "$*" >&2
    app_hosting_log "$VRF" "$*"
}

#
# Can this VRF be safely destroyed? We need to check prior to destroying a VRF
# as the attempt itself can be partial and leave the namespace in a messed up 
# state.
#
function app_hosting_vrf_destroy_namespace_server_side_check()
{
    local VRF=$1
    local arg=$2
    local vrf_processes=""
    local running_containers=""
    local docker_containers=""
    local docker_vrf=()

    source /etc/sysconfig/cisco_docker &>/dev/null
    export DOCKER_HOST=$DOCKER_DAEMON_SOCKET

    #
    # If the VRF is gone, do we fail this to kim or not? I think allow it
    # but log an error.
    #
    ip netns list | grep -q "\<$VRF\>"
    if [[ $? -ne 0 ]]; then
        app_hosting_vrf_namespace_server_side_error \
            "$VRF" \
            "Namespace does not exist."
        true
        return
    fi

    docker_containers=$(docker ps -a | awk '{if(NR>1) print $NF}')
    for container_id in $docker_containers
    do
        docker inspect $container_id | grep -q "\"/var/run/netns\(/${VRF}\)\?\"" &>/dev/null
        if [[ $? -eq 0 ]]; then
            docker_vrf+=("$container_id")
        fi
    done

    if [[ ${#docker_vrf[@]} -ne 0 ]]; then
        if [[ "$arg" != "check-only" ]]; then
            app_hosting_vrf_namespace_server_side_error \
                "$VRF" \
                "There are running docker containers in this namespace. " \
                "Please stop them to proceed with VRF namespace delete. " \
                "Running containers are: ${docker_vrf[@]}"
        fi

        false
        return
    fi

    #
    # Check lxc_containers using vrf
    #
    export LIBVIRT_DEFAULT_URI=lxc:///
    local lxc_containers=$(virsh list --name)
    for container in $lxc_containers
    do
        virsh dumpxml $container | grep "<sharenet type='netns'" | grep -q "value='${VRF}'"
        if [[ $? -eq 0 ]]; then
            lxc_vrf+=("$container")
        fi
    done

    #
    # Show the lxc containers.
    #
    if [[ ${#lxc_vrf[@]} -ne 0 ]]; then
        if [[ "$arg" != "check-only" ]]; then
            app_hosting_vrf_namespace_server_side_error \
                "$VRF" \
                "There are running LXC containers in this namespace. " \
                "Please stop them to proceed with VRF namespace delete. " \
                "Running containers are: ${lxc_vrf[@]}"
        fi

        false
        return
    fi

    vrf_processes=$(ip netns pids ${VRF})
    if [[ $? -ne 0 ]]; then
        #
        # Is this really an error case?
        #
        if [[ "$arg" != "check-only" ]]; then
            app_hosting_vrf_namespace_server_side_error \
                "$VRF" \
                "Cannot get namespace process list"
        fi

        false
        return
    fi

    if [[ -n "$vrf_processes" ]]; then
        if [[ "$arg" != "check-only" ]]; then
            app_hosting_vrf_namespace_server_side_error \
                "$VRF" \
                "Namespace still has active processes: $vrf_processes"
        fi

        false
        return
    fi

    app_hosting_log "$VRF" "No docker container or LXC or process is using this namespace "

    true
}

