#!/bin/bash
#
# pxe_install_memboot.sh
# This PI script is only invoked for memboot scenarios. 
# The source code resides in calvados WS, which is much more handy.
# The script prepares a calvados volume for memory boot.
# Host RPMs are expected to be pre-installed in the base initrd image.
#
# Copyright (c) 2014-2017 by Cisco Systems, Inc.
#
#
#==============================================================
#                     functions
#==============================================================

function redirect_output
{
    local logfile
    logfile=$1
    exec 107> >( exec cat > $logfile )
    redirect_output_pid=$!
    exec 105>&1
    exec 106>&2
    exec 1> >( exec tee -a /dev/fd/107 )
    exec 2> >( exec tee -a /dev/fd/107 >&2 )
    export LVM_SUPPRESS_FD_WARNINGS=1
}

function unredirect_output
{
    exec 1>&105-
    exec 2>&106-
    exec 107>&-
    if [ -n "$redirect_output_pid" ]; then
        kill -HUP $redirect_output_pid
        while [ -e /proc/$redirect_output_pid ]; do :; done
    fi
}

function step_log_console {
    local log_msg=${1}
    local time_str=$(date)
    echo "${time_str}: ${log_msg}"
    if [ $USE_FINAL -gt 0 ]; then
        sync
    fi
}
readonly -f step_log_console

# Verbose output should always go to the file (for debug-ability)
function step_log_file {
    local log_msg=${1}
    local time_str=$(date)
    echo "${time_str}: ${log_msg}" >&107
    if [ $USE_FINAL -gt 0 ]; then
        sync
    fi
}
readonly -f step_log_file

function run_depmod {
    for d in $1/lib/modules/*;
    do

        if [ -d $d ]; then
            depmod -a -b $1 `basename $d`
        fi
    done
}

cleanup() {
    cd /; umount $CALDIR > /dev/null 2>&1
    rm -rf $CALDIR > /dev/null 2>&1
}

exitclean() {
    step_log_console "Cleaning up to exit..."
    cleanup
    exit 1
}

function get_packages_to_be_installed {
    local iso_type=${1}
    local -a pkgs=("${!2}")

    unset packages_to_be_installed
    if [ ${iso_type} == ${PKG_VER} ]; then

        for file in ${pkgs[@]}
        do
            packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
        done
    else
    for file in ${pkgs[@]}
    do
        is_rp_package=0
        is_lc_package=0
        is_sc_package=0
        is_xc_package=0
        is_fc_package=0
        is_common_package=0

        if [[ ${file} == *.all-* ]] ; then
            is_common_package=1;
        else
            if [[ ${file} == *.rp[-,_]* ]] ; then
                is_rp_package=1;
            fi

            if [[ ${file} == *[.,_]sc* ]] ; then
                is_sc_package=1;
            fi

            if [[ ${file} == *[.,_]lc* ]] ; then
                is_lc_package=1;
            fi

            if [[ ${file} == *[.,_]xc* ]] ; then
                is_xc_package=1;
            fi
            
            if [[ ${file} == *[.,_]fc* ]] ; then
                is_fc_package=1;
            fi
        fi

        if [ "$is_rp_package" -ne "1" -a                                      \
             "$is_sc_package" -ne "1" -a                                      \
             "$is_lc_package" -ne "1" -a                                      \
             "$is_xc_package" -ne "1" -a                                      \
             "$is_fc_package" -ne "1" -a                                      \
             "$is_common_package" -ne "1" ]; then
            step_log_file "Error! Pkg ${file} is not applicable to any card"
        fi

        case ${BOARDTYPE} in
            LC)
               if [ "$is_lc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            SC)
               if [ "$is_sc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            XC)
               if [ "$is_xc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            FC)
               if [ "$is_fc_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
            *)
              # default is RP 
               if [ "$is_rp_package" -eq "1" -o "$is_common_package" -eq "1" ]; then
                   packages_to_be_installed[${#packages_to_be_installed[*]}]=${file}
               fi
               ;;
        esac
    done
    fi
}

function uninstall_rpms ()
{
    if [[ ! -f "$1" ]]; then
        return
    fi

    while read line; do
        if [[ ! $line == \#* ]] && [[ ! $line == "" ]]; then
            echo $(date)": Uninstalling rpm $line"

            # don't care if this fails
            chroot ${XROOT} rpm -e --allmatches $line > /dev/null
        fi
    done < $1
}
readonly -f uninstall_rpms

function install_calvados_rpms {
    local -a packages

    cd ${CALDIR}
    time zcat -f /sysadmin-nbi-initrd.img | cpio -id 
    rm /sysadmin-nbi-initrd.img 
    XROOT=${CALDIR} uninstall_rpms /etc/sysconfig/rpm_blacklist_sysadmin-vm.txt    
    # Get card instance information and add it in the mount
    declare -f pd_get_card_inst > /dev/null
    if [ "$?" -eq 0 ]; then
        local CARD_INST
        pd_get_card_inst CARD_INST
        echo "$BOARDTYPE $CARD_INST" > $CARD_INST_FILE
        step_log_file $"Using board ${BOARDTYPE} and inst ${CARD_INST}"
    fi

    step_log_console "sysadmin-vm: Starting $BOARDTYPE $CALDIR based RPM installation"

    # Obtain the list of packages to be installed
    packages=(rpm/*)

    #FRETTA_BRINGUP_HACK - abramach/pvarseka
    # Get list of SMU RPMs to be installed and append to packages
    
    #FRETTA_BRINGUP_HACK - abramach
    # Find a reliable way of determining package version from NBI metadata
    # Alternatively, support only 1.0 packages
    get_packages_to_be_installed "${PKG_VER}" packages[@]

    for file in ${packages_to_be_installed[@]}
    do
        if [ ${PKG_VER} == "1.0" ]; then
            if [ ${file} == *thirdparty-libexec* ]; then
                step_log_file $"Ignoring Install RPM ${file} of size "
            else
                SUPP_CARDS=()
                excl_card=()
                #SUPPCARDS holds a list of supported cards by this rpm; comma separated.
                supp_card=`chroot ${CALDIR} rpm -qp --qf '%{SUPPCARDS}\n' ${file}`
                SUPP_CARDS=(${supp_card//,/ })
                board=`echo ${BOARDTYPE} | tr "a-z" "A-Z"`
                for card in ${SUPP_CARDS[@]}
                do
                    card=`echo $card | tr "a-z" "A-Z"`
                    if [ $card != ${BOARDTYPE} -a  $card != ALL ]; then
                        excl_card[${#excl_card[*]}]=`echo $card | tr "A-Z" "a-z"`
                    fi
                done
                declare -a excl
                excl=()
                prefix=`chroot ${CALDIR} rpm -qp --qf '%{PREFIXES}\n' ${file}`
                pkg_name=`chroot ${CALDIR} rpm -qp --qf '%{NAME}\n' ${file}`
                version=`chroot ${CALDIR} rpm -qp --qf '%{VERSION}\n' ${file}`
                release=`chroot ${CALDIR} rpm -qp --qf '%{RELEASE}\n' ${file}`
                vmtype=`chroot ${CALDIR} rpm -qp --qf '%{VMTYPE}\n' ${file}`
                
                if [ "$vmtype" != "host" ]; then
                    excl_path=${prefix}/${pkg_name}-${version}-${release}
                    for card in ${excl_card[@]}
                    do
                        excl[${#excl[*]}]=--excludepath=${excl_path}/${card}
                    done
                    exclude=`echo ${excl[@]}`
                    step_log_console $"Install RPM with ${exclude}"
                    STARTRPM=$(date +%s)
                    chroot ${CALDIR} rpm -i ${exclude} --nodeps ${file} >> /tmp/rpmlog 2>&1
                    ENDRPM=$(date +%s)
                    DIFFRPM=$(( $ENDRPM - $STARTRPM ))
                    SIZERPM=$(ls -lh ${file}| awk '{ print $5 }')
                    step_log_file $"Install RPM ${file} of size ${SIZERPM} took ${DIFFRPM} sec"
                else
                    step_log_console $"Skipping ${file}"
                    step_log_file $"Skipping ${file}"
                fi
            fi
       else
           case ${file} in
               *.rpm) 
                 if [ ${file} == *thirdparty-libexec* -o \
                      ${file} == *hostos* ] ; then
                    step_log_file $"Ignoring Install RPM ${file} of size "
                 else
                 STARTRPM=$(date +%s)
                 chroot ${CALDIR} rpm -i --nodeps ${file} >> /tmp/rpmlog 2>&1
                 ENDRPM=$(date +%s)
                 DIFFRPM=$(( $ENDRPM - $STARTRPM ))
                 SIZERPM=$(ls -lh ${file}| awk '{ print $5 }')
                 step_log_file $"Install RPM ${file} of size ${SIZERPM} took ${DIFFRPM} sec"
                 fi
                 ;;
           esac
        fi
    done
    
    SMU_INST_SCRIPT_PATH=/usr/bin/calv_memboot_smu_install.py 
    if [ -f ${SMU_INST_SCRIPT_PATH} ]; then
         chmod 744 ${SMU_INST_SCRIPT_PATH}
         step_log_file "Install SMUs"
         STARTRPM=$(date +%s)
         ${SMU_INST_SCRIPT_PATH} -b "${BOARDTYPE}" -c "${CALDIR}" |tee /tmp/calv_memboot_smu_install.log 2>&1
         ENDRPM=$(date +%s)
         DIFFRPM=$(( $ENDRPM - $STARTRPM ))
         step_log_file "SMU installation took ${DIFFRPM} sec"
    fi

    if [ -f ${CALDIR}/usr/bin/calv_setup_ldpath.py ]; then
         chmod 744 ${CALDIR}/usr/bin/calv_setup_ldpath.py
         step_log_file "Install sysadmin symlink setup"
         STARTRPM=$(date +%s)
         arg="`echo ${excl_card[@]}`"
         ${CALDIR}/usr/bin/calv_setup_ldpath.py ${CALDIR} "${arg}" > /tmp/calv_setup_ldpath.log 2>&1
         ENDRPM=$(date +%s)
         DIFFRPM=$(( $ENDRPM - $STARTRPM ))
         step_log_file "sysadmin-vm symlink setup took ${DIFFRPM} sec"
    fi

    step_log_file "Install sysadmin dir setup"
    rm -rf rpm/* > /dev/null 2>&1

    mkdir -p ${CALDIR}/proc ${CALDIR}/sys ${CALDIR}/root/.ssh
    cp ${SSHKEY_DIR}/calvados/* ${CALDIR}/root/.ssh/
    mkdir -p ${CALDIR}/dev 
    chmod -R 700 ${CALDIR}/root
    cd / > /dev/null 2>&1

    # Allow platform specific patching for calvados boot partition 
    declare -F xrnginstall_platform_patch_calvados &>/dev/null &&
         xrnginstall_platform_patch_calvados ${CALDIR} ${PLATFORM}            \
                                      ${SIMULATION} ${VIRT_METHOD}

    run_depmod ${CALDIR}
    sync
    sync
    SIZE=$(du -hs --exclude="/proc" --exclude="/sys" ${CALDIR} | awk '{print $1}')

}

trap exitclean SIGINT SIGTERM
usage() {
    step_log_console $"Usage: $0 {start|stop}"
}

function create_ssh_keys {
    step_log_file "Create ssh keys"
    mkdir -p ${SSHKEY_DIR}/host ${SSHKEY_DIR}/calvados 
    ssh-keygen -q -t rsa -f ${SSHKEY_DIR}/calvados/id_rsa -C root@calvados-vm -N '' >&107 2>&1
    cat ${SSHKEY_DIR}/calvados/id_rsa.pub >> ${SSHKEY_DIR}/host/authorized_keys
    cat ${SSHKEY_DIR}/calvados/id_rsa.pub >> ${SSHKEY_DIR}/calvados/authorized_keys
}
readonly -f create_ssh_keys

function clean_partition {
    local dev=${1}

    mkdir -p /tmp/wipe  > /dev/null 2>&1
    rm -rf /tmp/wipe/*  > /dev/null 2>&1
    mount ${dev} /tmp/wipe  > /dev/null 2>&1
    rm -rf /tmp/wipe/*  > /dev/null 2>&1
    umount /tmp/wipe    > /dev/null 2>&1
    rm -rf /tmp/wipe    > /dev/null 2>&1
}

function save_install_log {
    local dev=${1}

    unredirect_output
    sync
    mkdir -p /tmp/dvlog  > /dev/null 2>&1
    mount ${dev} /tmp/dvlog > /dev/null 2>&1
    cp $LOGDIR/$LOGFILE /tmp/dvlog/
    cp $LOGDIR/$LOGFILE0 /tmp/dvlog/
    sync
    USE_FINAL=1
    redirect_output $LOGDIR/$LOGFILE2
}

function close_install_log {
    sync
    umount /tmp/dvlog  > /dev/null 2>&1
    rm -rf /tmp/dvlog  > /dev/null 2>&1
}

#==============================================================
#                     Main script code
#==============================================================

#---Basic PI variables---
readonly PD_DYNAMIC="/etc/init.d/spirit_pd.sh"

# TEMP_INSTALL_FILE_NAME: This file would be in the ramfs at this stage, 
# but would be copied to the host and calvados boot paritions after they 
# are prepared. The file this copied will be used by install software 
# during steady state install operation.
readonly TEMP_INSTALL_FILE_NAME="/tmp/pxe_install_temp.cfg"
readonly LOGDIR=/var/log
readonly LOGFILE0=host-install0.log
readonly LOGFILE=memboot-install.log
readonly LOGFILE2=sysadmin-install.log
readonly THIS_SCRIPT=pxe_install_memboot.sh
readonly PKG_VER=1.0
readonly CARD_INST_FILE="./root/card_instances.txt"
USE_FINAL=0

# Source function library.
. /etc/init.d/functions
. /etc/init.d/disk-functions
. /etc/init.d/spirit-functions

if [ -f /etc/init.d/pd-functions ]; then
    source /etc/init.d/pd-functions
    platform_log "PXE install"
fi

redirect_output $LOGDIR/$LOGFILE
# Show that this script is starting
step_log_file "Starting to execute $THIS_SCRIPT"

#-------get PD variables--------------
if [ -f $PD_DYNAMIC ]; then
    step_log_file "Sourcing $PD_DYNAMIC PD script"
    . $PD_DYNAMIC
else 
    step_log_console "Error, could not find $PD_DYNAMIC file which contains PD parameters for installation."
    step_log_console "Exiting $THIS_SCRIPT installation script."
    exit 0
fi

step_log_file "Simulation=$SIMULATION"

# PLATFORM is set in spirit_pd.sh by sourcing calvados_bootstrap.cfg 
if [ -z "$PLATFORM" ]; then
      step_log_console "No platform specified. Exiting."
      exit 1
fi

# Only read this once.
cmdline=$(cat /proc/cmdline)

AUTOREBOOT=yes
if strstr "$cmdline" noautoreboot ; then
    AUTOREBOOT=no
fi

CMD_PLATFORM=`echo $cmdline | sed 's/^.*platform=//' | cut -d" " -f1`
if [ "$CMD_PLATFORM" != "$PLATFORM" ]; then
   step_log_console "Platform in cmd line = $CMD_PLATFORM. Using $PLATFORM instead"
fi

#If platform hw has some way of discovering itself then OVERRIDE_PLATFORM is set to it
OVERRIDE_PLATFORM=$(override_platform_type ${PLATFORM}  $SIMULATION)
if [ "$OVERRIDE_PLATFORM" != "$PLATFORM" ]; then
    step_log_console "Wrong image for platform $OVERRIDE_PLATFORM. Exiting."
    exit 1
fi

step_log_console "Preparing disk for PLATFORM=$PLATFORM:"

# Get board type from PD function
get_board_type
step_log_file "Got board type $BOARDTYPE"

# Retrieves platform bootstrap variable and customizes install 
# specific variables based on the board type.

if [ -f $INSTALL_PARAMS_OVERRIDE ]; then
    readonly PI_CTYPE=UNKNOWN
    source $INSTALL_PARAMS_OVERRIDE $PD_BOOTSTRAP  \
           1 $BOARDTYPE $PI_CTYPE $TEMP_INSTALL_FILE_NAME
    if (( $? != 0 )); then
        step_log_file "Cannot read platform bootstrap cfg"
        exit 1 
    fi
fi

SSHKEY_DIR=$(mktemp -d /tmp/sshkeys.XXXXXX)
bootdir=${rootdir}/boot
CALDIR=/lxc_rootfs/panini_vol_grp-calvados_lv0

# This is where things start to happen
case "$1" in
    start)

      [ -d "${CALDIR}" ] && umount ${CALDIR} && rm -rf ${CALDIR}
      mkdir -p ${CALDIR}
      mount -t tmpfs tmpfs ${CALDIR}

      # pd_punch_watchdog is necessary only for some platforms
      declare -F pd_punch_watchdog && pd_punch_watchdog

      #=======================Prepare Host and Calvados on RAM============================
      step_log_console
      step_log_console $"---Starting to prepare Calvados on ${CALDIR} ---"

      # Setup the ssh keys 
      create_ssh_keys 

      # Patch HostOS
      step_log_console $"Patching HostOS on the root filesystem"

      if [ "${VIRT_METHOD}" == "lxc" ]; then
        # Make dir for lxc hooks and create symbolic link to the hook script
        mkdir -p /etc/libvirt/hooks/
        ln -s /etc/init.d/lxc_hook.sh /etc/libvirt/hooks/lxc
      
        # Allow platform specific patching for HostOS boot filesystem
        declare -F xrnginstall_platform_patch_hostos &>/dev/null &&
           xrnginstall_platform_patch_hostos "/" "" "" \
                    ${PLATFORM} ${SIMULATION} ${VIRT_METHOD}
        XROOT=/ uninstall_rpms /etc/sysconfig/rpm_blacklist_hostos.txt
      fi

      # Set up SSH files, permissions and directories
      mkdir -p /root/.ssh
      cp ${SSHKEY_DIR}/host/* /root/.ssh/
      chmod -R 700 /root
      cd /
      for i in `ls /`
        do
          if [ $i != proc -a $i != sys ]; then 
             chown -R root:root $i 2>/dev/null
          fi
        done

      # pd_notify_img_install_started is necessary only for some platforms
      # that need to track HOST events
      declare -F pd_notify_img_install_started && pd_notify_img_install_started

      step_log_console $"Install sysadmin-vm image on ${CALDIR}"
      START=$(date +%s)
      install_calvados_rpms
      sync
      END=$(date +%s)
      DIFF=$(( $END - $START ))
      step_log_console $"Installing sysadmin-vm image size of ${SIZE} took ${DIFF} seconds"
      
      # pd_notify_img_install_done is necessary only for some platforms
      # that need to track HOST events
      declare -F pd_notify_img_install_done && pd_notify_img_install_done

      # Delete temporary ssh keys dir
      rm -fr ${SSHKEY_DIR}
      # Delete all sysadmin RPMs on the Host
      rm -rf /rpm/*

      # Set the mask for loopback interface
      ifconfig lo netmask 255.255.0.0
    
      # to host boot partition
      cp $TEMP_INSTALL_FILE_NAME ${CALDIR}/${PD_BOARD_INST_CONF}
      if (( $? != 0 )); then
          step_log_file $"Copying ${PD_BOARD_INST_CONF} failed"
      fi
        
        ;;
    stop)
        exit 0
        ;;

    restart | reload | force-reload | status | condrestart | try-restart)
        usage
        exit 3
        ;;

    *)
        usage
        exit 2
        ;;
esac
