#!/bin/bash
#
# calvados/hostos_pkg/boot/scripts/Spirit_log.sh
# This script provides default logging to other scripts.
#
# Copyright (c) 2015-2016 by Cisco Systems, Inc.
# All rights reserved.

if [[ "$spirit_log_sourced" != true ]]; then
    spirit_log_sourced=true

    if [[ -f /etc/init.d/operns-functions && "$operns_functions_sourced" != true ]]; then
        source /etc/init.d/operns-functions
    fi
fi

#
# Get the file size in bytes, cross os
#
function filesize
{
    local file=$1
    size=`stat -c %s $file 2>/dev/null` # linux
    if [ $? -eq 0 ]; then
        echo $size
        return 0
    fi

    eval $(stat -s $file) # macos
    if [ $? -eq 0 ]; then
        echo $st_size
        return 0
    fi

    echo 0
    return -1
}

#
# Check we have somewhere to write logs to. At early boot we might not have
# /var/log, so use /tmp until then
#
function platform_log_choose_log_file
{
    #
    # Where to log shell script output to. If /var/log is not available, use
    # /tmp which surely must be usable.
    #
    local PLATFORM_LOG=/var/log/platform.log

    #
    # If we are installing, append onto the host install logs as platform.log 
    # is not saved by xrnginstall in save_install_log (yet)
    #
    grep -q "root=/dev/ram" /proc/cmdline
    if [ $? -eq 0 ]; then
        PLATFORM_LOG=/var/log/host-install.log
    fi

    PLATFORM_LOG_FILE=${PLATFORM_LOG}
    if [ ! -f ${PLATFORM_LOG_FILE} ]; then
        touch ${PLATFORM_LOG_FILE} &>/dev/null
    fi

    if [ ! -w ${PLATFORM_LOG_FILE} ]; then
        local PLATFORM_TMP_LOG=/tmp/platform.early.log

        PLATFORM_LOG_FILE=$PLATFORM_TMP_LOG
    fi

    #
    # Sanity check it doesn't get too large
    #
    if [ -f ${PLATFORM_LOG_FILE} ]; then
        local FILESIZE=`filesize ${PLATFORM_LOG_FILE}`
        local MAX=1048576

        if [ $FILESIZE -ge $MAX ]; then
            #
            # Chop off the start of the file
            #
            sed -i '1,100d' ${PLATFORM_LOG_FILE}
        fi
    fi
}

#
# Utility function to log to our platform log file to help with
# debugging shell script flow.
#
function platform_log
{
    platform_log_choose_log_file

    local DATE=`date`
    echo "$DATE ($0): $*" >> ${PLATFORM_LOG_FILE}
}

#
# Utility function to log to our platform log file to help with
# debugging shell script flow.
#
function platform_log_error
{
    platform_log_console $*

    backtrace
}

#
# Some calvados scripts use log message but it is not defined. Get the output
# into our platform log if log_message is not defined.
#
function log_message
{
    platform_log $*
}

#
# Log to file and console
#
function platform_log_console 
{
    platform_log_choose_log_file

    local DATE=`date`
    echo "$DATE ($0): $*" | tee -a ${PLATFORM_LOG_FILE}

    return ${PIPESTATUS[0]}
}

#
# Log to file and (in xr-vm only) to syslog as well
#
function platform_log_syslog
{
    local level=$1
    shift
    platform_log "$@"
    if [[ -f /pkg/bin/logger ]]; then
        # Careful - if this is invoked directly from a login shell
        # rather than a script, $0 will be '-su' or similar,
        # which confuses basename, logger, etc. into thinking its' an option
        local tag
        shopt -q login_shell
        if [ $? -eq 0 ]; then
            tag=$SHELL
        else
            tag=`basename $0`
        fi
        # /pkg/bin/logger must be called from xrnns or it hangs indefinitely
        declare -F operns_to_xrnns_cmd &>/dev/null
        if [ $? -eq 0 ]; then
            operns_to_xrnns_cmd /pkg/bin/logger -s $level -t "$tag" "$@"
        fi
    fi
}


function platform_log_syslog_notice
{
    platform_log_syslog notice "$@"
}

function platform_log_syslog_warning
{
    platform_log_syslog warning "$@"
}

function platform_log_syslog_error
{
    platform_log_syslog err "$@"
}

#
# Keep this function aliased as a variable, so that if PLATFORM_LOG_EXEC is 
# not defined due to an error in including this file, the callers command 
# still executes
#
PLATFORM_LOG_EXEC=platform_log_exec
PLATFORM_LOG_EXEC_CONSOLE=platform_log_exec_console

#
# Log results to the log file only
#
function platform_log_exec
{
    platform_log_choose_log_file
    platform_log "exec: $*"
    platform_log "    : in cwd" `pwd`

    local PREFIX="`date -u`: -- "
    $* 2>&1 | sed "s/^/${PREFIX}/g" >>${PLATFORM_LOG_FILE} 2>&1

    return ${PIPESTATUS[0]}
}

#
# Log results to the log file and the console
#
function platform_log_exec_console
{
    platform_log_choose_log_file
    platform_log "exec: $*"
    platform_log "    : in cwd" `pwd`

    local PREFIX="`date -u`: -- "
    $* 2>&1 | sed "s/^/${PREFIX}/g" | tee -a ${PLATFORM_LOG_FILE}

    return ${PIPESTATUS[0]}
}

#
# Print a shell backtrace
#
function backtrace () {
    local deptn=${#FUNCNAME[@]}

    for ((i=1; i<$deptn; i++)); do
        local func="${FUNCNAME[$i]}"
        local line="${BASH_LINENO[$((i-1))]}"
        local src="${BASH_SOURCE[$((i-1))]}"
        printf '%*s at: %s(), %s, line %s\n' $i '' $func $src $line # indent
    done
}

function platform_log_backtrace_ () {
    local depth=${#FUNCNAME[@]}

    for ((i=1; i<$depth; i++)); do
        local func="${FUNCNAME[$i]}"
        local line="${BASH_LINENO[$((i-1))]}"
        local src="${BASH_SOURCE[$((i-1))]}"
        printf '%*s at: %s(), %s, line %s\n' $i '' $func $src $line >> ${PLATFORM_LOG_FILE}
    done
}

function platform_log_backtrace () {
    platform_log_choose_log_file

    platform_log_backtrace_
}

