#!/bin/bash
#------------------------------------------------------------------
# xr_pre_launch.sh
#
# June 2014, Nitin chaudhary
#
# Copyright (c) 2014-2017 by cisco Systems, Inc.
# All rights reserved.
#------------------------------------------------------------------
# Add code in this script for XR pre launch
#

# The script is only valid on HostOS
#
# The Calvados VMM manager calls this script,
# in case the XR is running on LXC based container
# The script mount the XR  boot partition on /tmp/xr
# location for the continer to come up.
#

#
# Input arguments expected: 1
# $1 : Name of the LXC, used in the XML.
#      eg: default-sdr--1/default-sdr--2 ...
#

# do not exit on failed commands
set +e

if [ -f /etc/init.d/spirit-functions ]; then
    source /etc/init.d/spirit-functions
fi

if [ -f /etc/init.d/spirit_pd.sh ]; then
    source /etc/init.d/spirit_pd.sh
fi

if [ -f /etc/init.d/pd-functions ]; then
    source /etc/init.d/pd-functions
fi

if [ -f /etc/init.d/app_volume_config.conf ]; then
    source /etc/init.d/app_volume_config.conf
fi

if [ -f /etc/init.d/app-hosting-functions ]; then
    source /etc/init.d/app-hosting-functions
fi

BOOTSTRAP_FILE="/etc/init.d/calvados_bootstrap.cfg"
source $BOOTSTRAP_FILE

# XR_PATH is the mount point for mounting the active XR root partition.
# Its basename has the same name as the XR logical volume name of the
# active root partition. The mount point is also saved in /var/run/xr_path
# which will later on read by the devlist_get script.

# To make dynamic partion name for System Upgrade
# partition name is recieved as argument($1) 
# and XR_MAPPER_LV value would be /dev/mapper/panini_vol_grp-xr_lv0 for v1 
# and /dev/mapper/panini_vol_grp-xr_lv2 for v2 image

if [[ $# -ne 1 ]]; then
    echo "$0: Invalid arguments"
    exit 1
fi
VM_NAME=$1

#Changing Mount Path Logic to rely upon XR Volume Name in addition to lxc_rootfs
MNT_PATH=`cat /etc/libvirt/lxc/"$VM_NAME".xml | \
          grep "/lxc_rootfs/.*xr_" | cut --delimiter="=" -f2 | \
          cut -c 2- | rev | cut -c 4- | rev`
if [[ ! -n "$MNT_PATH" ]]; then
    echo "$0: Failed to find mount point"
    exit 1
fi

#
# Parse and idenitfy the LV, from the lxc rootfs mount point.
# eg: expected mnt point: /lxc_rootfs/panini_vol_grp-xr_lv0
#     Volume group : panini_vol_grp
#     LV name      : xr_lv0
XR_PATH="$MNT_PATH"
XR_MAPPER_LV="/dev/mapper/`basename $MNT_PATH`"

set_rootfs_var XR_PATH "$XR_PATH"

declare -F platform_log &>/dev/null && platform_log \
    "XR pre launch, VMTYPE $VMTYPE VIRT_METHOD $VIRT_METHOD"

is_lxc=0
if [[ $VIRT_METHOD == "lxc" && $VMTYPE == "hostos" ]]; then
    is_lxc=1
fi

if [[ $is_lxc == 1 ]]; then
    declare -F platform_log &>/dev/null && platform_log "XR pre launch, LXC"

    echo "Running in lxc platform"

    # calculate the offset location for mounting XR LV0
    start=`fdisk -u -l ${XR_MAPPER_LV} | grep ${XR_MAPPER_LV}p1 | awk '{print $2}'`
    OFFSET=$(( ${start} * 512 ))
    mkdir -p ${XR_PATH}
    $PLATFORM_LOG_EXEC mount ${XR_MAPPER_LV} ${XR_PATH} -o offset=${OFFSET}
    if [ -f /misc/config/devlistdata.txt ]; then
       mkdir -p ${XR_PATH}/misc/config/
       cp /misc/config/devlistdata.txt  ${XR_PATH}/misc/config/
    fi

    $PLATFORM_LOG_EXEC mkdir -p /var/run/netns

    declare -F app_host_is_feature_supported &>/dev/null && \
        app_host_is_feature_supported vrf_namespace
    if [[ $? -ne 0 ]]; then
        # EXR: need netns directory to be available on XR, so that
        # we can mount this from host.
        # mkdir -p ${XR_PATH}/var/run/netns
        # /var gets mounted on tmpfs. so first mount
        # default ns into bindmnt_netns and then mount on top
        # of var/run/netns from script Rc.sysinit
        mkdir -p ${XR_PATH}/mnt/bindmnt_netns/
    fi
  
    declare -F app_host_is_enabled &>/dev/null && app_host_is_enabled "$VM_NAME"
    if [[ $? -eq 0 ]]; then
        #
        # Mount VRFs on the host into XR. Specifically, just the mount point.
        # XR will create the VRFs when it boots.
        #
        declare -F app_host_mount_vrfs &> /dev/null
        if [[ $? -eq 0 ]]; then
            platform_log "VRF: Calling mount VRFs hook"
            app_host_mount_vrfs $VM_NAME $XR_PATH
            if [[ $? -ne 0 ]]; then
                platform_log_error "VRF: Calling mount VRFs hook failed"
            fi
        fi

        #create app_repo in App Host Volume of XR
        echo ${APP_REPO} > ${XR_PATH}/${APP_REPO_CONF}
        chmod 0444 ${XR_PATH}/${APP_REPO_CONF}

        # EXR: Need app_host volume directory in order to mount the additional 
        # Volume from the Host
        mkdir -p ${XR_PATH}/misc/app_host

        # EXR: Check whether the app_hosting volume is mounted, if not mounting it
        if mount | grep app_host > /dev/null; then
                echo "App_Host is already mounted"
        else   
                mkdir -p $TP_LV_DIR_PATH 
                declare -F pd_tp_lv_name_full >& /dev/null
                if [ $? -eq 0 ]; then
                    # PD app hosting lv name override
                    TP_LV_NAME_FULL=$( pd_tp_lv_name_full )
                fi
                mount $TP_LV_NAME_FULL $TP_LV_DIR_PATH 
        fi

        # After the app volume is mounted, emit an upstart event to start  
        # docker daemon
        initctl emit app-volume-mounted

        # Create staging directory to be shared with the XR lxc
        create_inotify_staging_dir ${TP_LV_DIR_PATH}

        # Create per network namespace resolv.conf,hosts file in XR LXC
        create_global_vrf_dirs_xr ${XR_PATH}

        # Create user scratch space for staging of app tarballs, etc.
        create_user_scratch_space

        # emit upstart event to start inotify on the host
        initctl emit start-inotify
    fi 

else
    declare -F platform_log &>/dev/null && platform_log "XR pre launch, KVM"
    echo "default : running on VM "
fi

declare -F platform_log &>/dev/null && platform_log "XR pre launch, done"

exit 0
