#!/usr/bin/env python
#
# A tool that provides client and server side options and allows the client
# to send set commands to the server. Replies are sent back to the client in
# the form of json output. Hence this is a limited form of remote execution.
#
# Copyright (c) 2017 by cisco Systems, Inc.
# All rights reserved.
#

#
# As this is spawned by upstart, if we fail to import, let's not respawn
# forever.
#
try:
    import getopt
    import imp
    import json
    import logging
    import os
    import select
    import shlex
    import socket
    import sys
    import traceback
    import time
    import argparse
except ImportError:
    while True:
        time.sleep(99999999)

plugins = {}


#
# Load supported plugins for back end commands. It's not very fancy and I want
# to avoid scanning paths for python paths, for speed.
#
def plugin_load():
    sys.path.append('/usr/sbin')
    import app_hosting_apply_cmd_vrf
    plugins["vrf"] = app_hosting_apply_cmd_vrf.VRF()


class ApplyCmdServer(object):

    #
    # Pre main, parse arguments.
    #
    def __init__(self):

        self.stdout = ""
        self.stderr = ""
        self.exitcode = ""

        self.bufsz = 1024
        self.socket_server = None

        self.handler = logging.StreamHandler()
        self.handler.setLevel(logging.INFO)

        self.logger = logging.getLogger(__name__)
        self.logger.addHandler(self.handler)
        self.logger.setLevel(logging.INFO)

        logging.addLevelName(logging.INFO,
                             "\033[1;31m%s\033[1;0m" %
                             logging.getLevelName(logging.INFO))
        logging.addLevelName(logging.ERROR,
                             "\033[1;41m%s\033[1;0m" %
                             logging.getLevelName(logging.ERROR))

        #
        # Initialize the parser
        #
        description = """
A tool (with client and server side) that allows remote execution of a
set of commands. Such commands should be carefully crafted to disallow
general breakout on the remote server.

The backend commands are implemented via plugins.

One example plugin is the VRF plugin which allows us to remotely do
ip netns add/del <vrf> from XR. By remote here we mean the VM host;
so we're using unix domain sockets to break out of the XR LXC.

Usage (server side):

app_hosting_apply_cmd_server.py --socket <file>

Commands supported:
"""

        for key, value in plugins.iteritems():
            description += value.description()

        arger = argparse.ArgumentParser(description="")

        arger.add_argument("-socket", "--socket",
                           help="UNIX domain socket filename",
                           required=True)

        arger.add_argument("-stdout", "--stdout", 
                           help="Place server command output in this file")

        arger.add_argument("-stderr", "--stderr", 
                           help="Place server command errors in this file")

        arger.add_argument("-exitcode", "--exitcode", 
                           help="Place server command exitcode in this file")

        self.opts = arger.parse_args()

    #
    # Process a command from the server.
    #
    def server_handle_client_cmd(self, cmd):
        cmd = shlex.split(cmd)

        p = plugins.get(cmd[0], None)
        if p is not None:
            return p.handle_command(cmd)

        return (-1, "", "unknown command".format(cmd))

    #
    # Start the server. Runs forever.
    #
    def server_run(self):

        if os.path.exists(self.opts.socket):
            self.logger.info("Socket {0} exists, removing".format(
                             self.opts.socket))
            os.remove(self.opts.socket)

        self.socket_open_server()

        os.chmod(self.opts.socket, int('0700', 8))

        self.socket_server.listen(1)
        connection = None

        while True:
            connection, client_address = self.socket_server.accept()

            datagram = connection.recv(self.bufsz)
            if not datagram:
                break

            self.logger.info("Rx [{0}]".format(datagram))
            exitcode, out, err = self.server_handle_client_cmd(datagram)

            output = {}
            output["stdout"] = out
            output["stderr"] = err
            output["exitcode"] = exitcode

            outstr = json.dumps(output)
            self.logger.info("Tx [{0}]".format(outstr))
            connection.send(outstr)

            connection.close()

    #
    # Open server side socket for communication
    #
    def socket_open_server(self):

        if self.socket_server is not None:
            raise ValueError("Socket {0} already open".format(
                             self.opts.socket))

        self.socket_server = socket.socket(socket.AF_UNIX,
                                           socket.SOCK_STREAM)

        self.logger.info("Listen on socket {0}".format(
                         self.opts.socket))

        self.socket_server.bind(self.opts.socket)

        self.socket_server.setblocking(1)

    #
    # Close server sockets
    #
    def socket_cleanup(self):

        if self.socket_server is not None:
            self.socket_server.close()
            self.socket_server = None

        if os.path.exists(self.opts.socket):
            os.remove(self.opts.socket)

    #
    # Save the results of the server to disk if asked for.
    #
    def write_results(self):

        if self.opts.stdout is not None:
            with open(self.opts.stdout, 'w') as f:
                f.write(self.stdout)

        if self.opts.stderr is not None:
            with open(self.opts.stderr, 'w') as f:
                f.write(self.stderr)

        if self.opts.exitcode is not None:
            with open(self.opts.exitcode, 'w') as f:
                f.write(str(self.exitcode))

    #
    # Called on success or failure at exit
    #
    def cleanup(self):

        self.socket_cleanup()
        self.write_results()

    def main(self):

        try:
            self.server_run()

        except Exception as exception:
            self.logger.error(format(exception))
            traceback.print_exc(file=sys.stdout)
            sys.exit(1)

        finally:
            self.cleanup()


plugin_load()

if __name__ == '__main__':
    ApplyCmdServer().main()
