#!/usr/bin/perl

# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
# script to collect shmwin memory usage on XR (for Spirit/eXR platforms)
#
# Copyright (c) 2017-2019 by Cisco Systems, Inc.
# All rights reserved.
# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

use strict;
use warnings;
use File::Basename;
use File::Copy;
use Getopt::Std;
use vars qw/ %opt /;

use lib qw ( /pkg/opt/cisco/pam/ /opt/cisco/calvados/pam/ /opt/pam/ );
use pam_ltrace;

getopts( "p:h", \%opt );

sub usage {
    print STDERR "\n", "~" x 80, "\n";
    print STDERR "\tUsage:\n";
    print STDERR "\t$0 - will print overall shmwin usage summary (default).\n";
    print STDERR "\t$0 -p <process name>, will print shmwin usage for the process.\n";
    print STDERR "\te.g.,\n";
    print STDERR "\t$0 -p emsd\n";
    print STDERR "\t -h - will print this messsage.\n";
    print STDERR "~" x 80, "\n";
    exit(0);
}

##############################################
# collect shmwin for all processes on local node
##############################################
sub collect_shmwin_summary() {
    my $proc_dir = "/proc/";

    my $info;
    my @shmwins = ();
    if (opendir(DIR,$proc_dir)) {
        my @pids = readdir(DIR);
        closedir(DIR);
        foreach my $pid (@pids) {
            next if ($pid !~ /^\d+$/);
            my $smaps   = $proc_dir . "/" . $pid . "/smaps";
            my $exe     = $proc_dir . "/" . $pid ."/exe";
            my $cmdline = $proc_dir . "/" . $pid . "/cmdline";
            if ((!-f $smaps) || (!-f $exe) || (!-f $cmdline)) {
                next;
            }
            my $exe_link = readlink($exe);
            #for thinxr: /opt/cisco/thinxr/pkg/bin/spp
            if ($exe_link !~ /\/opt\/cisco\/(XR\/packages|thinxr|install-iosxr)\//) {
                next;
            }
            if (!open(FD, $cmdline)) {
                next;
            }
            my $proc=<FD>;
            close(FD);
            if (!defined($proc)) {
                next;
            }
            $proc = (split(/\0/, $proc))[0];
            if ($proc !~ /\w+/) {
                next;
            }
            $proc = (split(/\s+/, $proc))[0];
            $proc =~ s/:$//;
            $proc =~ s/^\-//;
            $proc = basename($proc);
            $proc =~ s/\s+$//;

            my $info = get_shmwin_from_smaps($smaps);
            my $str = $proc . "," . $pid;
            #if (defined($info->{total_size})) {
            #    $str .=  "," . $info->{total_size};
            #} else {
            #    $str .=  ",0";
            #}
            if (defined($info->{total_rss})) {
                $str .=  "," . $info->{total_rss};
            } else {
                $str .=  ",0";
            }
            #if (defined($info->{total_pss})) {
            #    $str .=  "," . $info->{total_pss};
            #} else {
            #    $str .=  ",0";
            #}
            push @shmwins, $str;
        }
    }
    $info->{shmwins} = \@shmwins;
    return $info;
} ;# sub collect_shmwin_summary($$$$$$)

sub get_shmwin_from_smaps($) {
    my $file = shift;
    my $info;
    $info->{total_rss} = 0;
    if (!open(FD, $file)) {
        return $info;
    }
    my $total_rss = 0;
    local $/ = 'VmFlags:';
    while(my $buf=<FD>) {
        if ($buf =~ /\s+\/dev\/.*\/shmwin/) {
            foreach my $line (split(/\n/,$buf)) {
                if ($line =~ /^Rss:\s+(\d+)\s+kB/i) {
                    $total_rss += $1;
                }
            }
        }
    }
    close(FD);
    undef $/;
    $info->{total_rss} = $total_rss;
    return $info;
} ;# sub get_memory_from_smaps($)

sub get_shmwin_breakdowns($) {
    my $file = shift;
    my $info;
    $info->{total_size} = 0;
    $info->{total_rss} = 0;
    $info->{total_pss} = 0;
    if (!open(FD, $file)) {
        return $info;
    }
    my $total_size = 0;
    my $total_rss = 0;
    my $total_pss = 0;

    my $sub_total_size = 0;
    my $sub_total_rss = 0;
    my $sub_total_pss = 0;

    local $/ = 'VmFlags:';
    my $function = "";
    my @functionList = ();
    my %isKnownFunc;
    my $val = $1;
    while(my $buf=<FD>) {
        if ($buf =~ /\s+\/dev\/.*\/shmwin/) {
            foreach my $line (split(/\n/,$buf)) {
                if ($line =~ /\s+\/dev\/.*\/(shmwin.*)/) {
                    my $new_function = $1;
                    if (!$isKnownFunc{$new_function}) {
                        $isKnownFunc{$new_function} = 1;
                        push @functionList, $new_function;
                        if ($function ne "") {
                            $info->{$function}->{sub_total_size} = 0.0;
                            if ((defined($sub_total_size)) && ($sub_total_size ne "")) {
                                $info->{$function}->{sub_total_size} = $sub_total_size;
                            }
                            $info->{$function}->{sub_total_rss} = 0;
                            if ((defined($sub_total_rss)) && ($sub_total_rss ne "")) {
                                $info->{$function}->{sub_total_rss} = $sub_total_rss;
                            }
                            $info->{$function}->{sub_total_pss} = 0.0;
                            if ((defined($sub_total_pss)) && ($sub_total_pss ne "")) {
                                $info->{$function}->{sub_total_pss} = $sub_total_pss;
                            }
                        }
                        $function = $new_function;
                        $sub_total_size = 0;
                        $info->{$function}->{sub_total_size} = 0;
                        $sub_total_rss = 0;
                        $info->{$function}->{sub_total_rss} = 0;
                        $sub_total_pss = 0;
                        $info->{$function}->{sub_total_pss} = 0;
                    } else {
                        $function = $new_function;
                        $sub_total_size = $info->{$function}->{sub_total_size};
                        $sub_total_rss = $info->{$function}->{sub_total_rss};
                        $sub_total_pss = $info->{$function}->{sub_total_pss};
                    }
                    next;
                } elsif ($line =~ /^Size:\s+(\d+)\s+kB/i) {
                    $val = $1;
                    $total_size += $val;
                    $sub_total_size += $val;
                } elsif ($line =~ /^Rss:\s+(\d+)\s+kB/i) {
                    $val = $1;
                    $total_rss += $val;
                    $sub_total_rss += $val;
                } elsif ($line =~ /^Pss:\s+(\d+)\s+kB/i) {
                    $val = $1;
                    $total_pss += $val;
                    $sub_total_pss += $val;
                    last;
                }
            } ;# foreach my $line (split(/\n/,$buf))
        }
    }
    close(FD);
    undef $/;

    $info->{$function}->{sub_total_size} = $sub_total_size;
    $info->{$function}->{sub_total_rss} = $sub_total_rss;
    $info->{$function}->{sub_total_pss} = $sub_total_pss;

    $info->{total_size} = $total_size;
    $info->{total_rss} = $total_rss;
    $info->{total_pss} = $total_pss;

    $info->{functionList} = \@functionList;
    return $info;
} ;# sub get_shmwin_breakdowns($)

if (defined($opt{h})) {
    usage();
}
if (defined($opt{p}) && ($opt{p} =~ /\w+/)) {
    my $_pids = `pidof $opt{p}`;
    $_pids =~ s/[\r\n]//g;
    my @pids = split(/\s+/, $_pids);
    if (!scalar(@pids)) {
        print "Cannot find process PID for '$opt{p}' (make sure the process is running?).\n";
    }
    foreach my $pid (@pids) {
        if ((! -f "/proc/$pid/exe") ||
            (! -f "/proc/$pid/smaps") ||
            (! -f "/proc/$pid/cmdline")) {
            #print "Cannot find process PID for '$pid' (make sure the process is running?).\n";
            next;
        } else {
            my $smape_file = "/proc/$pid/smaps";
            my $shmwin_info = get_shmwin_breakdowns($smape_file);
            my @tmp =();
            foreach my $func (@{$shmwin_info->{functionList}}) {
                my $_tmp = "$func,$shmwin_info->{$func}->{sub_total_size}";
                $_tmp .= ",$shmwin_info->{$func}->{sub_total_rss}";
                push @tmp, $_tmp;
            }
            my @sorted = sort { (split /,/, $b)[2] <=> (split /,/, $a)[2] } @tmp;

            print "==>shmwin breakdown (by function) for $opt{p} (pid=$pid)<==\n";
            my $msg = <<END;
------------------------------------------------- -----------
END
            $msg .= sprintf("%-52s", "Function");
            #$msg .= sprintf("%-15s", "Max size (kB)");
            $msg .= sprintf("%-17s", "Usage (kB)");
            $msg .= "\n";
            $msg .= <<END;
------------------------------------------------- -----------
END
            print $msg;
            foreach my $_tmp (@sorted) {
                my ($func, $size, $rss) = split(/,/, $_tmp);
                $func =~ s/.*shmwin\///;
                $msg = sprintf("%-49s", $func);
                #$msg .= sprintf("%16.1f", $size);
                $msg .= sprintf("%11.1f", $rss);
                $msg .= "\n";
                print $msg;
            }
        }
    }
} else {
    my $info = collect_shmwin_summary();
    my @tmp = @{$info->{shmwins}};
    my @sorted = sort { (split /,/, $b)[2] <=> (split /,/, $a)[2] } @tmp;

    #my $grand_total_size = 0;
    my $grand_total_rss = 0;
    my $msg = <<END;
--------------------------------- -------- -----------
Process                           PID      Usage (kB)
--------------------------------- -------- -----------
END
    print $msg;
    foreach my $_tmp (@sorted) {
        my ($process, $pid, $rss) = split(/,/, $_tmp);
        $msg = sprintf("%-34s", $process);
        $msg .= sprintf("%-8s", $pid);
        #$msg .= sprintf("%16.1f", $size);
        $msg .= sprintf("%11.1f", $rss);
        $msg .= "\n";
        print $msg;
        #$grand_total_size += $size;
        $grand_total_rss += $rss;
    }
    $msg = sprintf("%-34s", "Grand total");
    $msg .= sprintf("%-8s", "-");
    #$msg .= sprintf("%16.1f", $grand_total_size);
    $msg .= sprintf("%11.1f", $grand_total_rss);
    $msg .= "\n";
    print $msg;
}
