#!/usr/bin/perl

# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
# library used by PAM
# 
# Feb 2016, Jieming Wang
# 
# Copyright (c) 2016-2019 by Cisco Systems, Inc.
# All rights reserved.
# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

package      pam;
require      Exporter;
use vars     qw (@ISA @EXPORT $VERSION );

use Cwd;

@ISA       = qw (Exporter);
$VERSION   = 1.00;

@EXPORT    = qw (
        getOsVrfIp
        getOsType
        getWS
        getVersionContent
        checkDiskSize
        checkDiskSpace
        getMemoryUsage
        getTopData
        getMemoryUsageViaTop
        checkMemoryNoCount
        color_print
        getNodeInfo
        getRackInfo
        check_process
        getCmdName
        getCrashProcName
        getCrashSignal
        killProcess
        check_monitor_status
        getRemovedDaemons
        convertToM
        getUptime
        getProcesses
        getCpuUsage
        createFolder
        save_log
        getCmdByPid
        getResourceByPid
        get_ctrace_processes
        dump_xr_run_config
        show_platform
        get_runningConfig
        show_platform_calvados
        show_wd_mem
        sdr_show_info
        get_netstat
        get_lwm_debug_proc
        get_attach_process_output
        get_ng_show_version
        get_sbt
        get_slot_id
        get_as9k_slot_id
        get_show_dll
        get_loadavg
        get_calv_active_rp_from_ip
        get_calvados_active_rp
        get_smu_from_show_version
        create_syslog
        confd_cli_wrapper_exe
        confd_cli_wrapper
        get_calvados_sdr
        get_platform_sysdb
        get_platform_sysdb_from_file
        get_rp_ip_from_sysdb
        do_sshfs_mount
        sshfs_mount_proc
        mount_ws_from_peer
        check_stale_sshfs_mount
        umount_sshfs
        sortCrashbyTime
        get_current_crashes
        get_pid_cmd_maps
        get_calv_command_history
        get_active_rp_nodes
        isFirstActiveRP
        get_active_rp_nodes_by_chassis_id
        get_chassis_id_from_node
        get_chassis_id
        get_active_nodes
        create_console_log
        getVrfIp
        localtime_to_yearmonthday
        verify_core_by_builddate
        get_show_install
        get_ddts_from_show_install
        get_smu_binary_path
        get_proc_args
        get_platform_by_rpm
        get_core_osType
        get_xr_hostname
        get_master_pam_conf
        verify_pam_conf_input
        get_df_lxc_vol
        do_log_rotation
        cleanup_tgz
        dump_calvados_core_cli
        dump_xr_core
        get_CoreFiles
        get_LocalCoreFiles
        pam_logger
        get_remote_cmd_proc_name
        get_remote_card_arch
        get_build_info
        get_ws_and_ddts_from_core_txt
        get_xr_pid_info
        get_calvados_pid_info
        get_dec_data
        get_default_commands
        get_edcd_pattern_commands
        get_user_pattern_files
        get_edcd_event_commands
        connect_to_cli_proxy
        get_dec_pattern_data
        cli_agent_exec
        cli_agent_shell
        do_xr_exec_cmd_verify
        do_xr_shell_cmd_verify
        do_admin_exec_cmd_verify
        save_show_log_summary
        save_process_event_summary
        save_process_crash_summary
        create_log_archive
        get_timestamp
        get_node_name_from_core
        verify_remote_proc_via_top
        get_PamLogDir
        create_pid_file
        get_memfree
        get_process_total_memory
        get_first_calv_ip
        getOldInstances
        update_log
        get_calvados_pam_show_sdr
        get_calv_showtech_cli
        get_xr_showtech_name
        do_calvados_showtech
        do_xr_showtech
        get_showtech_data
        get_proc_name_from_sshfs
        get_netns_env
        is_thinxr
        get_pid_proc_map
);

#use English;
use strict;

use warnings;
#Need to disable POSIX or ZMQ will fail
#use POSIX;
#use File::MkTemp; # used by mktemp
use File::Basename;
use Time::HiRes qw( usleep ualarm gettimeofday tv_interval );
use Fcntl ':mode';
use File::Copy;
use Expect;
#use JSON;
use JSON::PP;
use Date::Calc qw(:all);

sub getOsVrfIp ($) {
    my $intfname = shift;

    my $cmdline = `cat /proc/cmdline`;
    my $platform = $cmdline;
    $platform =~ s/.*\s+platform=//;
    $platform =~ s/\s+.*//;

    my $ip;
    my $netns_cmd = "";
    if ((($cmdline =~ /=xr\-vm/i) &&
        ($cmdline !~ /platform=(panini|scapa|ncs[46]k)/i)) ||
        (-d "/opt/cisco/thinxr/")) {
        my $my_pid = $$;
        my $netns_identify = `/sbin/ip netns identify $my_pid`;
        if ($netns_identify !~ /xrnns/) {
            $netns_cmd = "/sbin/ip netns exec xrnns ";
        }
    }
    if (-d "/opt/cisco/thinxr/") {
        my $_ENV = $ENV{"LD_LIBRARY_PATH"};
        $ENV{"LD_LIBRARY_PATH"} = ":/pkg/lib:/pkg/lib/cerrno";
        $ENV{"LD_LIBRARY_PATH"} .= ":/pkg/lib/mib:/pkg/lib/spp_plugins";
        $ENV{"LD_LIBRARY_PATH"} .= ":" . $_ENV if ($_ENV);
    }
    if (($platform =~ 
         /fretta|asr9k|skywarp|ncs5k|ncs5500|ncs540|rosco|ncs1|mystique|ncs560|iosxrwb/i) ||
         (-d "/opt/cisco/thinxr/")) {
         $ip=`$netns_cmd /sbin/ifconfig $intfname|grep "inet addr"`;
    } else {
        if (-d "/opt/cisco/thinxr/") {
            $ip=`$netns_cmd /sbin/ifconfig $intfname|grep "inet addr"`;
        } elsif ( (-f "/opt/cisco/calvados/sbin/ccc_driver") &&
                              ($intfname =~ /eth\-vf1\.513/) ) {
            $ip=`/sbin/chvrf 2 /sbin/ifconfig $intfname|grep "inet addr"`;
        } elsif ( (-f "/opt/cisco/calvados/sbin/ccc_driver") && 
                                    ($intfname =~ /eth\-vf1\.307[34]/) ) {
            $ip=`$netns_cmd /sbin/chvrf 0 /sbin/ifconfig $intfname|grep "inet addr"`;
        } else {
            $ip=`$netns_cmd /sbin/ifconfig $intfname|grep "inet addr"`;
        }
    }
    $ip =~ s/.*inet addr://g;
    $ip =~ s/\s+Bcast.*//g;
    chomp $ip;
    return $ip;
}

sub get_platform_by_rpm() {
    my $rpm_output = `/bin/rpm -qa`;
    #ncs4k-sysadmin-hostos.rp-5.2.1.18S-Default.x86_64
    #ncs4k-sysadmin-hostos.all-5.2.1.18S-Default.x86_64
    my $platform = "";
    foreach my $_rpm (split(/\n/, $rpm_output)) {
        if ( $_rpm =~ /^(ncs[46]k|ucs|asr9k)\-/ ) {
            $platform = $1;
            return $platform;
        }
    }
    return $platform;
}

sub getOsType ($) {
    my $mount = shift || "/";

    my $sys_info;
    my ($vmtype, $hostType, $boardtype, $simulator, $hostname, $platform);
    $boardtype = "unknown";

    if ( -f "/pkg/bin/hostname" ) {
        $hostname = `/pkg/bin/hostname`;
        $hostname =~ s/^node// if ($hostname);
    } elsif ( -f "/pkg/bin/uname" ) {
        $hostname = `/pkg/bin/uname -n`;
    } else {
        $hostname = `uname -n`;
    }
    $hostname =~ s/[\r\n]//g if ($hostname);
    my $is_thinxr = is_thinxr();
    $sys_info->{is_thinxr} = $is_thinxr;

    my $fd;
    $sys_info->{vf1_3073_ip} = "";
    $sys_info->{earms_jobId} = "";
    $sys_info->{ats_logdir} = "";
    if (!open($fd, "${mount}/proc/cmdline")) { 
        $sys_info->{hostType} = "unknown";
        $sys_info->{platform} = "panini";
        $sys_info->{boardtype} = "unknown";
        $sys_info->{simulator} = "unknown";
        $sys_info->{hostname} = "";
        return $sys_info;
    }
    $sys_info->{hostname} = $hostname;

    my $line = <$fd>;
    close($fd);
    chomp $line;
    $vmtype = $line;
    $boardtype = $line;
    $platform = $line;
    $simulator = $line;
    my $is_vxr = $line;
    my $platformtype = $line;
    $vmtype =~ s/.*vmtype=//;
    $vmtype =~ s/\s+.*//;

    $boardtype =~ s/.*boardtype=//;
    $boardtype =~ s/\s+.*//;

    $platform =~ s/.*platform=//;
    $platform =~ s/\s+.*//;

    $platformtype =~ s/.*_platformtype=//;
    $platformtype =~ s/\s+.*//;

    $is_vxr =~ s/.*_vxr=//;
    $is_vxr =~ s/\s+.*//;

    if ($simulator =~ /simulator=/) {
        $simulator =~ s/.*simulator=//;
        $simulator =~ s/\s+.*//;
        #get platform from rpm packages:
    } else {
        $simulator = "no";
    }

    if ( ($vmtype =~ /xr/i) || (-d "/opt/cisco/thinxr/"))  {
        $hostType = 'xr';
    } elsif ( $vmtype =~ /calvados|sysadmin/i ) {
        $hostType = 'calvados';
    } elsif ( $vmtype =~ /host/i ) {
        $hostType = 'host';
    }
    $sys_info->{isPamMaster} = 0;
    $sys_info->{master_xr_ip} = "";
    if ( $platform =~ /fretta|asr9k|skywarp|ncs5k|ncs540|ncs5500|rosco|ncs1|mystique|ncs560|iosxrwb/i ) {
        my $pam_master = "/opt/pam/.pam_master";
        $sys_info->{isPamMaster} = 1 if (-f $pam_master);
        my $master_xr_ip_file = "/opt/pam/master_xr_eth_vf1_ip.txt";
        if (open(FD1, "$master_xr_ip_file")) {
            my $line = <FD1>;
            close(FD1);
            $line =~ s/[\s\r\n]//g;
            $sys_info->{master_xr_ip} = $line;
        }
    }

    $sys_info->{hostType} = $hostType;
    $sys_info->{boardtype} = $boardtype;
    $sys_info->{simulator} = $simulator;
    $sys_info->{platform} = $platform;
    $sys_info->{platformtype} = $platformtype;
    $sys_info->{is_vxr} = $is_vxr;

    my $vf1 = "eth-vf1.3073";
    my $vf1_3073_ip = "";
    if ( $platform =~ /fretta|zermatt|tourin|iosxrwb/i ) {
        $vf1 = "eth-vf1";
        $vf1_3073_ip = getOsVrfIp($vf1);
        $sys_info->{vf1_3073_ip} = $vf1_3073_ip;
    } elsif ($is_thinxr || ($vmtype =~ /xr|calvados|sysadmin/i)) {
        $vf1_3073_ip = getOsVrfIp($vf1);
        $sys_info->{vf1_3073_ip} = $vf1_3073_ip;
    }

    my $_version = getWS($hostType, "/");
    $sys_info->{efr} = $_version->{efr};
    $sys_info->{ws} = $_version->{ws};
    $sys_info->{buildDate} = $_version->{buildDate};
    $sys_info->{buildHost} = $_version->{buildHost};

    #get the earms jobId
    if (open($fd, "${mount}/etc/rc.d/rc.local")) {
        while(my $line=<$fd>) {
            if ($line =~ /^\s*ats_logdir\s*=\s*(\S+)/) {
                $sys_info->{ats_logdir} = $1;
            }
            if ($line =~ /^\s*earms_jobId\s*=\s*(\d+)/) {
                $sys_info->{earms_jobId} = $1;
            }
        }
        close($fd);
    }
    return $sys_info;
}

sub getWS ($$) {
    my $OsType = shift;
    my $mount = shift || "/";

    my $version;
    my ($ws);

    my $show_file = "${mount}/etc/show_version.txt";
    my $build_file = "${mount}/etc/build-info.txt";
    my $buildHost = "";
    $version->{buildHost} = "";
    $version->{buildDate} = "";
    $version->{ws} = "";
    $version->{efr} = "";
    if ( -f $show_file ) {
        if ( !(open (RD, $show_file)) ) {
            #print "Cannot open $show_file: $!";
            return $version;
        }
        my $lines = "";
        while (my $line = <RD>) {
            $line =~ s/[\r\n]*//g;
            #my $ws = `egrep "Workspace" $show_file`;
            if ($line =~ /Workspace/) {
                $ws = $line;
                $ws =~ s/^.*Workspace\s+[=:]\s+//;
                $ws =~ s/^\s+//;
                $ws =~ s/\s+$//;
                $version->{ws} = $ws if ($ws =~ /\w+/);
            }
            if ($line =~ /Host +[:=]*/) {
                my $buildHost = $line;
                $buildHost =~ s/^.*Host\s+[=:]\s+//;
                $buildHost =~ s/^\s+//;
                $buildHost =~ s/\s+$//;
                $version->{buildHost} = $buildHost if ($buildHost =~ /\w+/);
            }
            if ( $line =~ /^\s*(Date|Buil[td] +on)/i ) {
                my $buildDate = $line;
                #Date: Fri Apr  4 03:36:54 PDT 2014
                $buildDate =~ s/^\s*(Date|Buil[td] +on)\s*[=:]\s+//gi;
                $version->{buildDate} = $buildDate if ($buildDate =~ /\w+/);
            }
            if ( $line =~
                    /((Lineup|Devline).*EFR)|\S+\s+EFR\-\d+(\s+Lineup)*/i ) {
                next if ((defined($version->{efr})) &&
                                                  ($version->{efr} =~ /\d+/));
                my $efr = $line;
                $efr =~ s/.*EFR\-//;
                $efr =~ s/\s+Lineup\s*$//;
                $efr =~ s/\s+Project\s*$//;
                $version->{efr} = $efr if ($efr =~ /\d+/);
            }
            if ( ($version->{efr} eq "") && ($line =~ /\S+\s+EFR/) ) {
                my $efr = $line;
                $efr =~ s/.*EFR\-//;
                $efr =~ s/\s+Lineup.*//;
                $efr =~ s/\s+Project\s*$//;
                $version->{efr} = $efr if ($efr =~ /\d+/);
            }
        }
        close(RD);
    } elsif ( -f $build_file ) {
        my $delimiter;
        if ($OsType =~ /xr/i) {
            $delimiter = 'XR Information';
        } elsif ($OsType =~ /calvados|sysadmin/i) {
            $delimiter = 'Calvados Information';
        } elsif ($OsType =~ /host/i) {
            $delimiter = 'Thirdparty Information';
        }
        local $/ = '###';
        if ( !(open (FD, $build_file)) ) {
            #print "Cannot open $build_file: $!";
            return "";
        }
        while (my $section = <FD>) {
            if ($section =~ /$delimiter/ ) {
                foreach my $line (split/\n/, $section) {
                    $line =~ s/[\r\n]*//g;
                    if ( $line =~ /Workspace/ ) {
                        ($ws = $line) =~ s/^.*Workspace\s+[=:]\s+//g;
                        $ws =~ s/^\s+//;
                        $ws =~ s/\s+$//;
                        $version->{ws} = $ws if ($ws =~ /\w+/);
                    }
                    #Build Host   : sjc-ads-261
                    #Host = sjc-ads-124
                    if ( $line =~ /Host/i ) {
                        ($buildHost = $line) =~ s/^.*Host\s+[=:]\s+//gi;
                        $version->{buildHost} = $buildHost 
                                                 if ($buildHost =~ /\w+/);
                    }
                    if ( $line =~ /^\s*(Date|Buil[td] +on)/i ) {
                        my $buildDate = $line;
                        #Date: Fri Apr  4 03:36:54 PDT 2014
                        $buildDate =~ s/^\s*(Date|Buil[td] +on)\s*[=:]\s+//gi;
                        $version->{buildDate} = $buildDate 
                                                     if ($buildDate =~ /\w+/);
                    }
                    if ( $line =~ 
                     /((Lineup|Devline).*EFR)|\S+\s+EFR\-\d+(\s+Lineup)*/i ) {
                        next if ((defined($version->{efr})) &&
                                                  ($version->{efr} =~ /\d+/));
                        my $efr = $line;
                        $efr =~ s/.*EFR\-//;
                        $efr =~ s/\s+Lineup\s*$//;
                        $efr =~ s/\s+Project\s*$//;
                        $version->{efr} = $efr if ($efr =~ /\d+/);
                    }
                }
            }
        }
        close (FD);
    }
    return $version;
}

sub getVersionContent ($) {
    my $mount = shift || "/";
    my $content;
    my $show_file = "${mount}/etc/show_version.txt";
    my $build_file = "${mount}/etc/build-info.txt";
    my $cmdline = `/bin/cat /proc/cmdline` if ( -f '/proc/cmdline' );
    $cmdline =~ s/.*\s+platform=/platform=/;
    $cmdline =~ s/(platform=\S+)\s+.*/$1/;

    if ( -f $show_file ) {
        $content = `/bin/cat $show_file`;
    } elsif ( -f $build_file ) {
        $content = `/bin/cat $build_file`;
    }
    $content .= $cmdline . "\n";
    return $content;
}

sub checkDiskSize() {
    my $mount_info;
    my %isFound;
    my @mounts;
    my $df = `df -kl`;
    foreach my $line (split(/\n/, $df)) {
        next if ($line =~ /Filesystem/);
        next if ($line =~ /\/mnt\/ram\d+/);
        next if ($line =~ /\/tmp\/iso/);
        if ($line =~ /\s+(\d+)\s+([\d\-]+)\s+([\d\-]+)\s+([\d\-%]+)\s+(\S+)$/) {
            my $total = $1;
            my $used = $2;
            my $free = $3;
            my $usage = $4;
            my $mount = $5;
            $usage =~ s/%//g;
            if (!$isFound{$mount}) {
                $isFound{$mount} = 1;
                $mount_info->{$mount}->{total} = $total;
                $mount_info->{$mount}->{used} = $used;
                $mount_info->{$mount}->{free} = $free;
                $mount_info->{$mount}->{usage} = $usage;
                push @mounts, $mount;
            }
        }
    }
    $mount_info->{mountList} = \@mounts;
    return $mount_info;
} ;# sub checkDiskSize()

sub checkDiskSpace($) {
    my $threshold = shift || 99;

    my @mount_info;
    my %isFound;
    my $df = `df -kl`;
    my $disk_pattern = '([\/\w]*)\s+(\d+)\s+([\d\-]+)\s+([\d\-]+)';
    $disk_pattern .= '\s+([\d\-%]+)\s+([\w\/\-]+)/)';
    foreach my $line (split(/\n/, $df)) {
        next if ($line =~ /Filesystem/);
        next if ($line =~ /\/mnt\/ram\d+/);
        next if ($line =~ /\/tmp\/iso/);
        next if ($line =~ /^\/mnt$/);

        if ($line =~ /$disk_pattern/) {
            my $mount = $6;
            my $usage = $5;
            if ($usage =~ /%/) {
                $usage =~ s/%//g;
                if ($usage >= $threshold) {
                    my $minfo = "${mount}:${usage}";
                    if ( !defined($isFound{$mount}) ) {
                        $isFound{$mount} = 1;
                        push @mount_info, $minfo;
                    }
                }
            }
        }
    }
    return @mount_info;
} ;# sub checkDiskSpace ($)

sub getMemoryUsage ($$$) {
    my $_excluded_procs = shift;
    my $mem_threshold = shift || 5; #5 MB by default
    my $getShared = shift || 0;

    my @excluded_procs = @$_excluded_procs;

    my @skiped_processes = (
         'init',
         'agetty',
         'dbus-daemon',
         'dhclient',
         'klogd',
         'libvirtd',
         'oom.sh',
         'pm_startup.sh',
         'ssh',
         'sshd',
         'sshfs',
         'syslogd',
         'xinetd',
         'udevd',
         'bash',
         'gdb',
    );

    my $memory_info;
    my @procList;

    my ($seconds, $microseconds) = gettimeofday;
    $memory_info->{timestamp} = $seconds;

    ######################################################
    #get proc count for each process
    ######################################################
    my $proc_dir = '/proc';
    my ($main_dir, $cmd);
    my $proc_info;
    my $myPid = $$;
    my (@pidList);
    my (@filenames);
    if ( -d "/var/lib/malloc_stat/" ) {
        if (!opendir(DIR, "/var/lib/malloc_stat/")) {
            #print "Can't open /var/lib/malloc_stat/: $!\n";
            return $memory_info;
        }
        (@filenames) = readdir(DIR);
        closedir(DIR);
    } else {
        if (!opendir(DIR, "/proc/")) {
            #print "Can't open /proc: $!\n";
            return $memory_info;
        }
        (@filenames) = readdir(DIR);
        closedir(DIR);
    }
    for (@filenames) {
        next if ($_ !~ /^\d+$/);
        next if ($_ eq $myPid);
        push @pidList, $_;
    }
    #find the actual names (from /proc/pid/status and /proc/pid/cmdline)
    #this only find out the threads
    my $grandtotal = 0;
    local $/ = '\0';
    foreach my $pid (@pidList) {
        #########################################
        #get the actual process name from cmdline
        #########################################
        my $cmdFile = "$proc_dir/$pid/cmdline";
        next if (! -f $cmdFile);
        next if (!open(CFD, $cmdFile));
        my $cmdline=<CFD>;
        close(CFD);
        next if (!defined($cmdline));
        my $cmd = (split(/\0/, $cmdline))[0];
        next if ( (!defined($cmd)) || ($cmd !~ /\w+/) );
        $cmd = (split(/\s+/, $cmd))[0];
        $cmd =~ s/:$//;
        $cmd =~ s/^\-//;
        $cmd = basename($cmd);
        $cmd =~ s/\s+$//;

        next if (grep(/\b$cmd\b/, @excluded_procs));

        $memory_info->{fullname}->{$cmd} = $cmd;

        #########################################
        #get the proc name from ps (status file)
        #########################################
        my $statusFile = "$proc_dir/$pid/status";
        next if (! -f $statusFile);
        next if (!open(SFD, $statusFile));
        my $lines = <SFD>;
        close(SFD);

        my $proc = "";
        my $total = 0;
        my $virt_total = 0;
        my $threads =0;
        #my $got_it = 0;
        foreach my $line (split(/\n/, $lines)) {
            if ($line =~ /^\s*Name:\s+([\w_\-\.\s]+)$/) {
                $proc = $1;
            }
            if ($line =~ /^\s*VmRSS:\s+(\d+)\s+([km])B/) {
                $total = $1;
                my $km = $2;
                if ($km =~ /k/i) {
                    $total = sprintf ("%.1f", $total/1024);
                } elsif ($km =~ /m/i) {
                    $total = sprintf ("%.1f", $total);
                }
                $grandtotal += $total;
                #$got_it++;
            }
            if ($line =~ /^\s*VmSize:\s+(\d+)\s+([km])B/) {
                $virt_total = $1;
                my $km = $2;
                if ($km =~ /k/i) {
                    $virt_total = sprintf ("%.1f", $virt_total/1024);
                } elsif ($km =~ /m/i) {
                    $virt_total = sprintf ("%.1f", $virt_total);
                }
            }
            if ($line =~ /^\s*Threads:\s+(\d+)/) {
                $threads = $1;
                last;
            }
        }
        #close(SFD);
        next if ($proc eq "");

        $memory_info->{total}->{$cmd} = $total;
        $memory_info->{virt_total}->{$cmd} = $virt_total;
        $memory_info->{threads}->{$cmd} = $threads;

        #TODO:
        #check by PID (instead of process name)
        my $shared =0;
        if ( (defined($getShared)) && ($getShared) ) {
            #########################################
            #get the shared memory from smaps file
            #########################################
            my $smapFile = "$proc_dir/$pid/smaps";
            next if (! -f $smapFile);
            next if (!open(SFD, $smapFile));
            while (my $line = <SFD>) {
                 #assume it is in kB
                if ($line =~
                     /^\s*(Shared_Clean|Shared_Dirty):\s+(\d+)\s+([km])B/) {
                    $shared += $2;
                }
            }
            close(SFD);
            #assume all in kB
            $memory_info->{shared}->{$cmd} = sprintf ("%.1f", $shared/1024);
        }

        #########################################
        #get number of open files
        #########################################
        my $filedecs = 0;
        my $fdDir = "$proc_dir/$pid/fd";
        if (opendir(DIR, $fdDir)) {
            my (@filenames) = readdir(DIR);
            closedir(DIR);
            $filedecs = scalar(@filenames) - 2;
        }
        $memory_info->{filedecs}->{$cmd} = $filedecs;

        $memory_info->{pid}->{$cmd} = $pid;
        if (!grep(/\b$cmd\b/, @procList)) {
            push @procList, $cmd;
        }
        # number of processes;
        $memory_info->{count}->{$cmd}++;
    }
    $memory_info->{grandtotal} = $grandtotal;
    $memory_info->{procList} = \@procList;
    return $memory_info;
} ;# sub getMemoryUsage ($)

sub getTopData() {
    my %isExlucded;
    my @exclude_prcos = ('perl');
    foreach my $proc (@exclude_prcos) {
        $isExlucded{$proc} = 1;
    }
    my $memory_info;
    my $opt_output = `/usr/bin/top -b -n1`;
    my $pattern1 = 'Tasks: +(\d+) +total, +(\d+) +running, +(\d+)';
    $pattern1 .= ' +sleeping, +(\d+) +stopped, +(\d+) +zombie';

    my $pattern2 = '[%]?Cpu\(s\): +([\d\.]+)[ %]us, +([\d\.]+)[ %]sy, +([\d\.]+)[ %]ni,';
    $pattern2 .= ' +([\d\.]+)[ %]id, +([\d\.]+)[ %]wa, +([\d\.]+)[ %]hi,';
    $pattern2 .= ' +([\d\.]+)[ %]si, +([\d\.]+)[ %]st/';

    my $pattern3 = 'Mem: +(\d+)k +total, +(\d+)k used,';
    $pattern3 .= ' +(\d+)k free, +(\d+)k +buffers';

    my $pattern3_thinxr = 'KiB +Mem\s*: +(\d+) +total, +(\d+) free,';
    $pattern3_thinxr .= ' +(\d+) used, +(\d+) +buff';

    my $pattern4 = 'Swap: +(\d+)k +total, +(\d+)k used,';
    $pattern4 .= ' +(\d+)k free, +(\d+)k +cached';

    my $pattern4_thinxr = 'KiB Swap\s*: +(\d+) +total, +(\d+) free,';
    $pattern4_thinxr .= ' +(\d+) used[,\.] +(\d+) +avail';

    my $pattern5 = '\s*(\d+)\s+root\s+\d+\s+\d+\s+(\S+)\s+(\S+)\s+(\S+)\s+';
    $pattern5 .= '\S+\s+(\S+)\s+(\S+)\s+(\d+:\d+\.\d+)\s+(\S+)';

    foreach my $line (split(/\n/, $opt_output)) {
        $line =~ s/\r//g;
        $line =~ s/\n//g;
        if ( $line =~ /^$pattern1/) {
            my $proc_proc = $1;
            my $proc_running = $2;
            my $proc_sleeping = $3;
            my $proc_stopped = $4;
            my $proc_zombie = $5;
            $memory_info->{os_proc_total} = $proc_proc;
            $memory_info->{os_proc_running} = $proc_running;
            $memory_info->{os_proc_sleeping} = $proc_sleeping;
            $memory_info->{os_proc_stopped} = $proc_stopped;
            $memory_info->{os_proc_zombie} = $proc_zombie;
            next;
        } elsif ( $line =~ /^$pattern2/) {
            my $cpu_us = $1;
            my $cpu_sy = $2;
            my $cpu_ni = $3;
            my $cpu_id = $4;
            my $cpu_wa = $5;
            my $cpu_hi = $6;
            my $cpu_si = $7;
            my $cpu_st = $8;
            $memory_info->{os_cpu_us} = $cpu_us;
            $memory_info->{os_cpu_sy} = $cpu_sy;
            $memory_info->{os_cpu_ni} = $cpu_ni;
            $memory_info->{os_cpu_id} = $cpu_id;
            $memory_info->{os_cpu_wa} = $cpu_wa;
            $memory_info->{os_cpu_hi} = $cpu_hi;
            $memory_info->{os_cpu_si} = $cpu_si;
            $memory_info->{os_cpu_st} = $cpu_st;
            next;
        } elsif ( $line =~ /^$pattern3/) {
            my $mem_total = $1;
            my $mem_used = $2;
            my $mem_free = $3;
            my $mem_buffers = $4;
            $memory_info->{os_mem_total} = $mem_total;
            $memory_info->{os_mem_used} = $mem_used;
            $memory_info->{os_mem_free} = $mem_free;
            $memory_info->{os_mem_buffers} = $mem_buffers;
            next;
        } elsif ( $line =~ /^$pattern3_thinxr/) {
            my $mem_total = $1;
            my $mem_free = $2;
            my $mem_used = $3;
            my $mem_buffers = $4;
            $memory_info->{os_mem_total} = $mem_total;
            $memory_info->{os_mem_used} = $mem_used;
            $memory_info->{os_mem_free} = $mem_free;
            $memory_info->{os_mem_buffers} = $mem_buffers;
            next;
        } elsif ( $line =~ /^$pattern4/) {
            my $swap_total = $1;
            my $swap_used = $2;
            my $swap_free = $3;
            my $cached = $4;
            $memory_info->{os_swap_total} = $swap_total;
            $memory_info->{os_swap_used} = $swap_used;
            $memory_info->{os_swap_free} = $swap_free;
            $memory_info->{os_cached} = $cached;
            next;
        } elsif ( $line =~ /^$pattern4_thinxr/) {
            my $swap_total = $1;
            my $swap_free = $2;
            my $swap_used = $3;
            my $cached = $4;
            $memory_info->{os_swap_total} = $swap_total;
            $memory_info->{os_swap_used} = $swap_used;
            $memory_info->{os_swap_free} = $swap_free;
            $memory_info->{os_cached} = $cached;
            next;
        } elsif ( $line =~ /^$pattern5/) {
            my $pid = $1;
            my $virt = $2;
            my $res = $3;
            my $shr = $4;
            my $cpup = $5;
            my $memp = $6;
            my $time = $7;
            my $proc = $8;

            next if ($isExlucded{$proc});
            if ($res =~ /g/) {
                $res =~ s/g//;
                $res = sprintf ("%.1f", $res * 1024);
            } elsif ($res =~ /m/) {
                $res =~ s/m//;
                $res = sprintf ("%.1f", $res);
            } else {
                $res = sprintf ("%.1f", $res/1024);
            }
            if ($shr =~ /g/) {
                $shr =~ s/g//;
                $shr = sprintf ("%.1f", $shr * 1024);
            } elsif ($shr =~ /m/) {
                $shr =~ s/m//;
                $shr = sprintf ("%.1f", $shr);
            } else {
                $shr = sprintf ("%.1f", $shr/1024);
            }

            #push @pidList, $pid;
            $memory_info->{virt}->{$pid} = $virt;
            $memory_info->{res}->{$pid} = $res;
            $memory_info->{shr}->{$pid} = $shr;
            $memory_info->{proc}->{$pid} = $proc;
            $memory_info->{cpup}->{$pid} = $cpup;
            $memory_info->{memp}->{$pid} = $memp;
            $memory_info->{time}->{$pid} = $time;
            next;
        }
    } ;#foreach my $line (split(/\n/, $opt_output))
    return $memory_info;
} ;#sub getTopData()

sub getMemoryUsageViaTop ($$) {
    my $_excluded_procs = shift;
    my $mem_threshold = shift || 5; #5 MB by default

    my %isExlucded;
    my @excluded_procs = @$_excluded_procs;
    my @skiped_processes = (
         'init',
         'agetty',
         'dbus-daemon',
         'dhclient',
         'klogd',
         'libvirtd',
         'oom.sh',
         'pm_startup.sh',
         'ssh',
         'sshd',
         'sshfs',
         'syslogd',
         'xinetd',
         'udevd',
         'bash',
         'gdb',
    );
    foreach my $proc (@excluded_procs) {
        $isExlucded{$proc} = 1;
    }
    foreach my $proc (@skiped_processes) {
        $isExlucded{$proc} = 1;
    }

    my $memory_info;
    my @procList;

    my ($seconds, $microseconds) = gettimeofday;
    $memory_info->{timestamp} = $seconds;
    $memory_info->{localtime} = localtime();

    ######################################################
    #get proc count for each process
    ######################################################
    my $proc_dir = '/proc';
    my ($main_dir, $cmd);
    my $proc_info;
    my $myPid = $$;
    my %isPidSeen;
    my (@filenames);
    my $malloc_stat_dir = "/var/lib/malloc_stat/";
    my $minimum_pid = 200; #ignore kernel processes
    my $proc_map_info;
    if ( -d $malloc_stat_dir ) {
        $proc_map_info = get_pid_cmd_maps($malloc_stat_dir, $minimum_pid);
    } else {
        $proc_map_info = get_pid_cmd_maps("/proc", $minimum_pid);
    }
    my @pidList = @{$proc_map_info->{pidList}};

    #find the actual names (from /proc/pid/status and /proc/pid/cmdline)
    #this only find out the threads
    my $grandtotal = 0;

    my $cmdFile;

    my $top_memory_info =&getTopData();
    $memory_info->{os_mem_total} =
                       sprintf ("%.1f",$top_memory_info->{os_mem_total}/1024.0);
    $memory_info->{os_mem_used} =
                        sprintf ("%.1f",$top_memory_info->{os_mem_used}/1024.0);
    $memory_info->{os_mem_free} =
                        sprintf ("%.1f",$top_memory_info->{os_mem_free}/1024.0);
    $memory_info->{os_mem_buffers} =
                     sprintf ("%.1f",$top_memory_info->{os_mem_buffers}/1024.0);
    $memory_info->{os_swap_total} =
                      sprintf ("%.1f",$top_memory_info->{os_swap_total}/1024.0);
    $memory_info->{os_swap_used} =
                       sprintf ("%.1f",$top_memory_info->{os_swap_used}/1024.0);
    $memory_info->{os_swap_free} =
                       sprintf ("%.1f",$top_memory_info->{os_swap_free}/1024.0);
    $memory_info->{os_cached} =
                          sprintf ("%.1f",$top_memory_info->{os_cached}/1024.0);

    #tmpfs mount - os cache
    my @tmpfsList = ('/dev', '/dev/shm', '/var');
    my $local_mount_info = &checkDiskSize();
    my $total_tmpfs_used = 0;
    foreach my $mount (@{$local_mount_info->{mountList}}) {
        if (grep(/\b$mount\b/, @tmpfsList)) {
            $memory_info->{df}->{$mount}->{total} =
                                        $local_mount_info->{$mount}->{total};
            $memory_info->{df}->{$mount}->{used} =
                                         $local_mount_info->{$mount}->{used};
            $memory_info->{df}->{$mount}->{free} =
                                         $local_mount_info->{$mount}->{free};
            $total_tmpfs_used += $local_mount_info->{$mount}->{used};
        }
    }
    $memory_info->{total_tmpfs_used} =
                                  sprintf ("%.1f", $total_tmpfs_used/1024.0);

    #Track timestamp for each process detected!
    my $timestamp = time();

    foreach my $pid (@pidList) {
        #########################################
        #get the actual process name from cmdline
        #########################################
        my $cmd = $proc_map_info->{$pid};
        next if ( (!defined($cmd)) || ($cmd eq "") );
        #next if (grep(/\b$cmd\b/, @excluded_procs));
        next if ($isExlucded{$cmd});

        $memory_info->{fullname}->{$cmd} = $cmd;

        #########################################
        #get the proc name from ps (status file)
        #########################################
        my $statusFile = "/proc/$pid/status";
        next if (! -f $statusFile);
        next if (!open(SFD, $statusFile));
        my $lines = "";
        while (<SFD>) {
            $lines .= $_;
        }
        close(SFD);

        my $proc = "";
        my $total = 0;
        my $virt_total = 0;
        my $threads =0;
        #my $got_it = 0;
        foreach my $line (split(/\n/, $lines)) {
            if ($line =~ /^\s*Name:\s+([\w_\-\.\s]+)$/) {
                $proc = $1;
            }
            if ($line =~ /^\s*VmRSS:\s+(\d+)\s+([km])B/) {
                $total = $1;
                my $km = $2;
                if ($km =~ /k/i) {
                    $total = sprintf ("%.1f", $total/1024);
                } elsif ($km =~ /m/i) {
                    $total = sprintf ("%.1f", $total);
                }
                $grandtotal += $total;
                #$got_it++;
            }
            if ($line =~ /^\s*VmSize:\s+(\d+)\s+([km])B/) {
                $virt_total = $1;
                my $km = $2;
                if ($km =~ /k/i) {
                    $virt_total = sprintf ("%.1f", $virt_total/1024);
                } elsif ($km =~ /m/i) {
                    $virt_total = sprintf ("%.1f", $virt_total);
                }
            }
            if ($line =~ /^\s*Threads:\s+(\d+)/) {
                $threads = $1;
                last;
            }
        }
        #close(SFD);
        next if ($proc eq "");

        $memory_info->{total}->{$cmd} = $total;
        $memory_info->{virt_total}->{$cmd} = $virt_total;
        $memory_info->{threads}->{$cmd} = $threads;
        $memory_info->{shared}->{$cmd} = $top_memory_info->{shr}->{$pid};
        #disable to prevent HASH error
        #$memory_info->{timestamp}->{$cmd} = $timestamp;

        #TODO:
        #check by PID (instead of process name)

        #########################################
        #get number of open files
        #########################################
        my $filedecs = 0;
        my $fdDir = "$proc_dir/$pid/fd";
        if (opendir(DIR, $fdDir)) {
            my (@filenames) = readdir(DIR);
            closedir(DIR);
            $filedecs = scalar(@filenames) - 2;
        }
        $memory_info->{filedecs}->{$cmd} = $filedecs;

        $memory_info->{pid}->{$cmd} = $pid;
        if (!grep(/\b$cmd\b/, @procList)) {
            push @procList, $cmd;
        }
        # number of processes;
        $memory_info->{count}->{$cmd}++;
    } ;#foreach my $pid (@pidList)
    $memory_info->{grandtotal} = $grandtotal;
    $memory_info->{procList} = \@procList;
    return $memory_info;
} ;# sub getMemoryUsageViaTop ($$)

sub checkMemoryNoCount () {
    my $mount_info;
    my @procList;

    my %isFound;
    my ($total, $shared, $procname);
    my $output = `/usr/bin/smap /proc/*`;
    foreach my $line (split(/\n/, $output)) {
        next if ($line =~ /Total Mem/i);
        next if ($line =~ /\-+\s+\-+\s+\-+/);
        #[host:~]$ smap /proc/*
        if ($line =~ /([\d\.]+[MKG])\s+\(([\d\.]+[KMG])\s*\)\s+([\w_\-\.]+)/) {
            $total = $1;
            $shared = $2;
            $procname = $3;
            push @procList, $procname;
            $mount_info->{$procname}->{total} = $total;
            $mount_info->{$procname}->{shared} = $shared;
        }
    }
    $mount_info->{procList} = \@procList;
    return $mount_info;
} ;# sub checkMemoryNoCount ()

sub color_print ($$) {
    my $color = shift;
    my $msg = shift;

    my $end_tag = "\033\[0m";
    if ( $color =~ /red/i ) {
       print "\033[0;31m", $msg, $end_tag, "\n";
    } elsif ( $color =~ /green/i ) {
       print "\033[0;32m", $msg, $end_tag, "\n";
    } elsif ( $color =~ /yellow/i ) {
       print "\033[0;33m", $msg, $end_tag, "\n";
    } else {
       print $msg, "\n";
    }
    return 1;
} ;# sub color_print ($$)

sub getNodeInfo ($) {
    my $nodeset = shift;
    my $tagname = shift;
    
    my $xml = new XML::Simple;
    my $array;
    my ($text, $ref, $name, $ip, $ssh_port, $username, $password, $console);
    foreach my $node ( $nodeset->get_nodelist ) {
         $text = XML::XPath::XMLParser::as_string($node);
         $ref = $xml->XMLin($text);
         $name = $ref->{'name'};
         $ip = $ref->{'ipv4_address'};
         $ssh_port = $ref->{'ssh_port'};
         $username = $ref->{'username'};
         $password = $ref->{'password'};
         $console = $ref->{'console'};
         $array->{$name}->{ip} = $ip;
         $array->{$name}->{ssh_port} = $ssh_port;
         $array->{$name}->{console} = $console;
         $array->{$name}->{username} = $username;
         $array->{$name}->{password} = $password;
    }
    return $array;
} ;# sub getNodeInfo ($)

sub getRackInfo ($) {
    my $nodeset = shift;

    my $xml = new XML::Simple;
    my $array;
    my ($text, $ref, $name, $gateway, $labgateway, $email);

    foreach my $node ( $nodeset->get_nodelist ) {
         $text = XML::XPath::XMLParser::as_string($node);
         $ref = $xml->XMLin($text);
         $name = $ref->{'name'};
         $gateway = $ref->{'gateway'};
         $labgateway = $ref->{'labgateway'};
         $email = $ref->{'email'};
         $array->{gateway} = $gateway;
         $array->{labgateway} = $labgateway;
         $array->{email} = $email;
    }
    return $array;
} ;# sub getRackInfo ($) 

sub check_process () {
    my $processName = shift;
    my $sys_info = shift;
    my $exclude_pid = shift || 0;
    my $min_pid = shift || 200;

    my @allProcs = ();
    my $osType = $sys_info->{hostType};
    my $_osType = "";
    if (defined($osType)) {
        $_osType = substr $osType, 0, 2;
    }

    my $proc_dir = '/proc';
    my ($main_dir);
    my (@processList);

    if (!opendir(DIR, $proc_dir)) {
        #print "Can't open $proc_dir: $!\n";
        return "";
    }
    my (@filenames) = readdir(DIR);
    closedir(DIR);
    for (@filenames) {
        if ( ($_ =~ /^\d+$/) && ($_ > $min_pid) && ($_ ne $exclude_pid) ) {
            push @processList, $_;
        }
    }
    if ( (defined($osType)) && ($osType =~ /host/i) ) {
        foreach my $pid (@processList) {
            my $cmdFile = "$proc_dir/$pid/cmdline";
            next if (!open(FD, $cmdFile));
            my $line=<FD>;
            close(FD);
            next if ( (!defined($line)) || ($line !~ /$processName/));
            my $proc_args_info = get_proc_args($pid);
            if ( (defined($proc_args_info->{o})) && ($proc_args_info->{o}) ) {
                my $osType2 = $proc_args_info->{o};
                my $_osType2 = substr $osType2, 0, 2;
                next if ($_osType ne $_osType2);
            }
            push @allProcs, $pid;
        }
    } else {
        foreach my $pid (@processList) {
            my $cmdFile = "$proc_dir/$pid/cmdline";
            next if (!open(FD, $cmdFile));
            my $line=<FD>;
            close(FD);
            if ( (defined($line)) && ($line =~ /$processName/) ) {
                push @allProcs, $pid;
            }
        }
    }
    return \@allProcs;
} ;# sub check_process ()

sub getCmdName($$) {
    my $pid = shift;

    my $cmdFile = "/proc/$pid/cmdline";
    my $fd;
    open ($fd, $cmdFile);
    my $cmdtext = <$fd>;
    close($fd);
    
    my @args = ();
    my @cmds = (split(/\0/, $cmdtext));
    return @cmds;
} ;# sub getCmdName($$)

sub getCrashProcName ($) {
    my $procName = shift;
    $procName =~ s/.*\///;
    $procName =~ s/\.gz//;
    $procName = basename($procName);

    #ipv6_ma_5588.by.11.20171026-010515.xr-vm_nodeB0_CB0_CPU0.2f25d.core.gz
    my $pattern1 = '([a-zA-Z0-9\-\.]+(_[\.a-zA-Z0-9]+)*\d*)([\._]\d+)?\.by';
    $pattern1 .= '\.(\d+|user)\.\d{8}(\-\d{6})?(\.|_F\d+_SC\d+)\S+\.core';
    my $pattern2 = '([a-zA-Z0-9\-\.]+(_[\.a-zA-Z0-9]+)*\d*)([\._]\d+)?\.by';
    $pattern2 .= '\.(\d+|user)\.\d{8}\-\d{6}\.\S+\.core';
    my $pattern3 = '.*:B?\d+_(R[S]?P|LC|CB)*\d+_\d+_([a-z_\-]+)_\d+_\d+_\d+\.core';

    if ( $procName =~ /^$pattern1/) {
        $procName = $1;
        $procName =~ s/[\._]\d+$//;
    } elsif ( $procName =~ /^$pattern2/) {
        $procName = $1;
        $procName =~ s/[\._]\d+$//;
    } elsif ( $procName =~ /$pattern3/) {
        $procName = $2;
    } elsif ( $procName =~ /^(xr\-vm\.)*([a-z_\d\-]+)\.\d+\.core/ ) {
        #xr-vm.dsr.1691.core
        #fib_mgr_main.2756.core
        $procName = $2;
    } elsif ( $procName =~ /^(\w+)\.gz$/ ) {
        #wd.gz
        #RMF.gz
        $procName = $1;
    }
    return $procName;
} ;# sub getCrashProcName ($)

sub getCrashSignal($) {
    my $signal = shift;
    $signal =~ s/.*\///;
    $signal =~ s/\.gz//;
    $signal = basename($signal);

    my $pattern1 = '([a-zA-Z0-9\-\.]+(_[\.a-zA-Z0-9]+)*\d*)([\._]\d+)?\.by';
    $pattern1 .= '\.(\d+|user)\.\d{8}(\-\d{6})?(\.|_F\d+_SC\d+)\S+\.core';
    my $pattern2 = '([a-zA-Z0-9\-\.]+(_[\.a-zA-Z0-9]+)*\d*)([\._]\d+)?\.by';
    $pattern2 .= '\.(\d+|user)\.\d{8}\-\d{6}\.\S+\.core';

    if ( $signal =~ /^$pattern1/) {
        $signal = $4;
    } elsif ( $signal =~ /^$pattern2/) {
        $signal = $4;
    }
    return $signal;
} ;# sub getCrashSignal($)

sub killProcess() {
    my $proclList = shift;
    my $sys_info = shift;
    my @processList = @$proclList;
    foreach my $process (@processList) {
        my $ret = &check_process($process, $sys_info);
        foreach my $pid (@$ret) {
            print "\tStopping monitor agent ($process) ...\n";
            if (system("kill $pid") ) {
                color_print ("red", "Failed to stop $process.");
            } else {
                color_print ("green", "$process stopped successfully.");
            }
        }
    }
    return 1;
} ;# sub killProcess()

sub check_monitor_status($) {
    my $jobFile = shift;

    my @daemonList = ();
    return @daemonList if ( ! -f $jobFile );
    my $fd;
    if (!open($fd, $jobFile)) {
        print "Failed to open $jobFile: $!\n";
        return 0;
    }
    my $output = "";
    while (<$fd>) {
        $output .= $_;
    }
    close($fd);

    my %isSeen;
    foreach my $line (split(/\n/, $output)) {
        chomp $line;
        next if ( $line =~ /^\s*$/ );
        next if ( $line =~ /^\s*#/ );
        if ( $line =~ /crash/i ) {
            my $proc = "monitor_crash.pl";
            if (!$isSeen{$proc}) {
                $isSeen{$proc} = 1;
                push @daemonList, $proc;
            }
        }
        if ( $line =~ /cpu/i ) {
            my $proc = "monitor_cpu.pl";
            if (!$isSeen{$proc}) {
                $isSeen{$proc} = 1;
                push @daemonList, $proc;
            }
        }
        if ( $line =~ /memory|ram/i ) {
            my $proc = "monitor_memory.pl";
            if (!$isSeen{$proc}) {
                $isSeen{$proc} = 1;
                push @daemonList, $proc;
            }
        }
        if ( $line =~ /disk/i ) {
            my $proc = "monitor_disk.pl";
            if (!$isSeen{$proc}) {
                $isSeen{$proc} = 1;
                push @daemonList, $proc;
            }
        }
        if ( $line =~ /local_log/i ) {
            my $proc = "monitor_local_logs.pl";
            if (!$isSeen{$proc}) {
                $isSeen{$proc} = 1;
                push @daemonList, $proc;
            }
        }
        if ( $line =~ /_syslog/i ) {
            my $proc = "monitor_syslog.pl";
            if (!$isSeen{$proc}) {
                $isSeen{$proc} = 1;
                push @daemonList, $proc;
            }
        }
    }
    return @daemonList;
} ;# sub check_monitor_status($)

sub getRemovedDaemons($$) {
    my $old = shift;
    my $new = shift;
    my @oldList = @$old;
    my @newList = @$new;

    my @removed;
    foreach my $proc (@oldList) {
        if ( !grep(/\b$proc\b/, @newList) ) {
            push @removed, $proc;
        }
    }
    return @removed;
} ;# sub getRemovedDaemons($$)

sub convertToM($) {
    my $size = shift;
    if ($size =~ /G/ ) {
        $size =~ s/G//;
        $size = $size * 1000;
    } elsif ($size =~ /K/ ) {
        $size =~ s/K//;
        $size = sprintf "%.1f", $size / 1000;
    } elsif ($size =~ /M/ ) {
        $size =~ s/M//;
    }
    return $size;
}

sub getUptime_old() {
    my $file = "/proc/uptime";
    my $dd;
    if ( -f $file ) {
        if (!open($dd, $file)) {
            close($dd);
            return -1;
        }
        my $line = (<$dd>);
        close($dd);
        return (split(/\s+/, $line))[0];
    }
    return -1;
} ;# sub getUptime()

sub getUptime() {
    my $file = "/proc/sys";
    if (! -d $file ) {
        return -1;
    }
    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,
              $size,$atime,$mtime,$ctime,@junk) = stat($file);
    return time() - $mtime;
} ;# sub getUptime()

sub getProcesses() {
    my $proc_dir = '/proc';

    my ($main_dir, $cmd);

    my @skiped_processes = (
         'init',
         'agetty',
         'dbus-daemon',
         'dhclient',
         'klogd',
         'libvirtd',
         'oom.sh',
         'pm_startup.sh',
         'ssh',
         'sshd',
         'sshfs',
         'syslogd',
         'xinetd',
         'udevd',
         'bash',
         'gdb',
         'gzip',
         'perl',
    );
    my %isSkipped;
    foreach my $proc (@skiped_processes) {
        $isSkipped{$proc} = 1;
    }

    my $proc_info;
    if (!opendir(DIR, $proc_dir)) {
        print "Can't open $proc_dir: $!\n";
        return $proc_info;
    }
    my $myPid = $$;

    my (@pidList);
    my ($proc, $statusFile);
    my ($cmdFile, $cmdline);

    my ($seconds, $microseconds) = gettimeofday;
    $proc_info->{timestamp} = $seconds;

    my (@filenames) = readdir(DIR);
    closedir(DIR);
    foreach my $pid (@filenames) {
        next if ($pid !~ /^\d+$/);
        next if ($pid eq $myPid);

        $statusFile = "$proc_dir/$pid/status";
        next if (! -f $statusFile);
        next if (!open(SFD, $statusFile));
        my $lines=<SFD>;
        close(SFD);
        next if (!defined($lines));
        next if ($lines !~ /Name:\s+\w+/);
        $proc = "";
        foreach my $line (split(/\n/, $lines)) {
            if ($line =~ /^\s*Name:\s+([\w_\-\.]+)/) {
                $proc = $1;
                last;
            }
        }
        next if ($proc eq "");
        next if ($isSkipped{$proc});
        #$proc_info->{$proc}->{count}++;

        #########################################
        #get the actual process name from cmdline
        #########################################
        #In panini, the output from top doesn't match ps
        #instead, the following map should be used:
        # top - /proc/pid/status
        # ps - /proc/pid/cmdline

        $cmdFile = "$proc_dir/$pid/cmdline";
        next if (! -f $cmdFile);
        next if (!open(CFD, $cmdFile));
        $cmdline=<CFD>;
        close(CFD);

        next if (!defined($cmdline));
        $cmd = (split(/\0/, $cmdline))[0];
        next if ( (!defined($cmd)) || ($cmd !~ /\w+/) );
        if ($cmd !~ /[\s\@]pts\/\d+/) {
            $cmd = basename($cmd);
            $cmd =~ s/\s+$//;
        }
        next if ($isSkipped{$cmd});
        $proc_info->{$pid}->{fullname} = $cmd;
        $proc_info->{$pid}->{shortname} = $proc;
        push @pidList, $pid;
    }
    $proc_info->{pidList} = \@pidList;
    return $proc_info;
} #; sub getProcesses()

sub getCpuUsage($) {
    my $get_raw_data = shift || 0;

    my $raw_data = "";
    my $max_raw_data = 30; # 30 lines of top output

    my $cpu_info;
    my $topoutput = `/usr/bin/top -b -n 1`;

    #top - 02:09:49 up 1 day, 16:00,  5 users,  load average: 0.09, 0.07, 0.01
    #top - 20:43:02 up 43 min,  1 user,  load average: 0.18, 0.13, 0.09

    #Tasks: 516 total,   1 running, 515 sleeping,   0 stopped,   0 zombie
    my $task_pattern = 'Tasks:\s+(\d+)\s+total,\s+(\d+)\s+running,\s+(\d+)';
    $task_pattern .= '\s+sleeping,\s+(\d+)\s+stopped,\s+(\d+)\s+zombie';

    my $load_pattern = 'top\s+\-\s+(\d+:\d+:\d+)\s+up\s+(\d+\s+day[s]*,)*\s*';
    $load_pattern .= '(\d+:\d+|\d+min),\s+(\d+)\s+user[s]*,\s+load\s+average:';
    $load_pattern .= '\s+([\d\.]+),\s+([\d\.]+),\s+([\d\.]+)';

    #2184 root     20   0 4196m 4.0g 3112 S 102.1 34.5  27:15.98 qemu-system-x86
    my $pattern ='(\d+)\s+(\w+)\s+\S+\s+\d+\s+([\d\.]+[mg]*)\s+([\d\.]+[mg]*)';
    $pattern .='\s+([\d\.]+[mg]*)\s+(\S+)\s+([\d\.]+)\s+([\d\.]+)\s+([\d:\.]+)';
    $pattern .='\s+([\w\-\.]+\s?[\w]*)';

    my ($pid, $uid, $size, $rss, $shr, $state, $cpu, $mem, $time, $proc_name);
    my @pidList;

    my ($seconds, $microseconds) = gettimeofday;
    $cpu_info->{timestamp} = $seconds;

    my $rCount = 0;
    foreach my $line (split (/\n/, $topoutput)) {
        if ( ($get_raw_data) && ($rCount < $max_raw_data) ) {
            $raw_data .= $line . "\n";
            $rCount++;
        }

        if ( $line =~ /$task_pattern/ ) {
            $cpu_info->{total} = $1;
            $cpu_info->{running} = $2;
            $cpu_info->{sleeping} = $3;
            $cpu_info->{stopped} = $4;
            $cpu_info->{zombie} = $5;
        }

        if ( $line =~ /$load_pattern/ ) {
            my $current_time = $1;
            my $uptime1 = $2;
            my $uptime2 = $3;
            my $no_users = $4;
            my $load1 = $5;
            my $load5 = $6;
            my $load15 = $7;
            $cpu_info->{current_time} = $1;
            $cpu_info->{uptime} = $uptime2;
            if ($uptime1) {
                $cpu_info->{uptime} = $uptime1 . " " . $uptime2;
            }
            $cpu_info->{no_users} = $no_users;
            $cpu_info->{load1} = $load1;
            $cpu_info->{load5} = $load5;
            $cpu_info->{load15} = $load15;
        }

        if ( $line =~ /$pattern/ ) {
            $pid = $1;
            next if ( $7 == 0 );
            push @pidList, $pid;
            $cpu_info->{$pid}->{uid} = $2;
            $cpu_info->{$pid}->{size} = $3;
            $cpu_info->{$pid}->{rss} = $4;
            $cpu_info->{$pid}->{shr} = $5;
            $cpu_info->{$pid}->{state} = $6;
            $cpu_info->{$pid}->{cpu_pct} = $7;
            $cpu_info->{$pid}->{mem_pct} = $8;
            $cpu_info->{$pid}->{cpu_time} = $9;

            $cpu_info->{$pid}->{proc_name} = $10;
            $cpu_info->{$pid}->{proc_name} =~ s/^\s+//;
            $cpu_info->{$pid}->{proc_name} =~ s/\s+$//;
        }
    }
    $cpu_info->{pidList} = \@pidList;
    if ($get_raw_data) {
        $cpu_info->{raw_data} = $raw_data;
    } else {
        $cpu_info->{raw_data} = "";
    }
    return $cpu_info;
} ;#sub getCpuUsage($)

sub createFolder($) {
    my $logdir = shift;
    if ( ! -d $logdir ) {
        if (system ("/bin/mkdir -p \"$logdir\"")) {
            #print "Failed to create folder: $logdir: $!\n";
            return 0;
        }
    }
    return 1;
}

sub save_log($$$) {
    my $title = shift;
    my $message = shift;
    my $log_dir = shift;

    my $info;
    my $time = localtime();
    my ($wday, $mon, $day, $hms, $year) = split(/\s+/, $time);
    $day = length($day) < 2 ? "0${day}" : $day;
    $hms =~ s/://g;
    my $date = $year . $mon . $day . $hms;
    my $fullpath = $log_dir . "/" . $title . "-" . $date . ".txt";
    if (!open (WD, ">$fullpath")) {
        $info->{msg} = "Failed to write to $fullpath: $!";
        $info->{status} = 0;
        return $info;
    }
    print WD $message;
    close(WD);
    $info->{file} = $fullpath;
    $info->{status} = 1;
    return $info;
} ;# save_log

sub getCmdByPid($) {
    my $pid = shift;

    my $proc = "";
    my $proc_dir = "/proc";
    my $cmdFile = "$proc_dir/$pid/cmdline";
    return $proc if (! -f $cmdFile);
    return $proc if (!open(CFD, $cmdFile));
    my $cmdline=<CFD>;
    close(CFD);
    return $proc if (!defined($cmdline));
    my $cmd = (split(/\0/, $cmdline))[0];
    return $proc if ( (!defined($cmd)) || ($cmd !~ /\w+/) );
    $proc = (split(/\s+/, $cmd))[0];
    $proc =~ s/:$//;
    $proc =~ s/^\-//;
    $proc =~ s/^\s+//;
    $proc =~ s/\s+$//;
    $proc = basename($proc);
    return $cmd;
}

sub getResourceByPid($) {
    my $pid = shift;

    my $memory_info;
    my $proc_dir = "/proc";
    my $statusFile = "$proc_dir/$pid/status";
    return $memory_info if (! -f $statusFile);
    return $memory_info if (!open(SFD, $statusFile));

    my $proc = "";
    my $total = 0;
    my $virt_total = 0;
    my $threads =0;
    while (my $line = <SFD>) {
        if ($line =~ /^\s*Name:\s+([\w_\-\.\s]+)$/) {
            $proc = $1;
        }
        if ($line =~ /^\s*VmRSS:\s+(\d+)\s+([km])B/) {
            $total = $1;
            if ($2 =~ /k/i) {
                $total = sprintf ("%.1f", $total/1024);
            } elsif ($2 =~ /m/i) {
                $total = sprintf ("%.1f", $total);
            }
        }
        if ($line =~ /^\s*VmSize:\s+(\d+)\s+([km])B/) {
            $virt_total = $1;
            if ($2 =~ /k/i) {
                $virt_total = sprintf ("%.1f", $virt_total/1024);
            } elsif ($2 =~ /m/i) {
                $virt_total = sprintf ("%.1f", $virt_total);
            }
        }
        if ($line =~ /^\s*Threads:\s+(\d+)/) {
            $threads = $1;
            last;
        }
    }
    close(SFD);

    $memory_info->{proc_name} = $proc;
    $memory_info->{total} = $total;
    $memory_info->{virt_total} = $virt_total;
    $memory_info->{threads} = $threads;

    return $memory_info;
} ;#sub getResourceByPid($)

sub get_ctrace_processes() {
    my $ctrace_output = "";
    my $_sys_info = &getOsType();
    my $osType = $_sys_info->{hostType};
    my $platform = $_sys_info->{platform};
    my $netns_cmd = &get_netns_env($_sys_info);
    if ($osType =~ /xr/i) {
        $ctrace_output = `$netns_cmd /pkg/bin/ctracedec -ip`;
    } elsif ($osType =~ /calvados|sysadmin/i) {
        my $chvrf = "";
        if ($platform =~ /panini|scapa|ncs[46]k/i) {
            $chvrf = "/sbin/chvrf 0";
        }
        $ctrace_output = `$chvrf /opt/cisco/calvados/bin/ctracedec -ip`;
    }
    my $ctrace_info;
    my @clients;
    if (!defined($ctrace_output)) {
        $ctrace_info->{clientList} =\@clients;
        return $ctrace_info;
    }
    foreach my $c (split(/\n/, $ctrace_output)) {
        $c =~ s/\s+//g;
        my $c1 = $c;
        $c1 =~ s/.*_[s]*bin_//;
        push @clients, $c1;
        $ctrace_info->{$c1} = $c;
    }
    $ctrace_info->{clientList} =\@clients;
    return $ctrace_info;
}

sub dump_xr_run_config($) {
    my $filename = shift;

    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $cmd = $netns_cmd . "/pkg/bin/cfgmgr_show_persistent_config";
    if ( ! -f $cmd ) {
        return 0;
    }
    $cmd = "ulimit -q unlimited; $cmd";
    my $show_cmd = "$cmd > $filename && gzip $filename";
    if (system($show_cmd)) {
        return 0;
    } else {
        return 1;
    }
}

sub show_platform () {
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $cmd = "/pkg/bin/show_platform";
    if ( ! -f $cmd ) {
        $cmd = $netns_cmd . "/pkg/bin/show_platform_sysdb";
        if ( ! -f $cmd ) {
            return "";
        }
    }
    my $show_output =`ulimit -q unlimited; $cmd`;
    return $show_output;
}

sub get_runningConfig () {
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $cmd = "/pkg/sbin/nvgen";
    if ( ! -f $cmd ) {
        return "";
    }
    $cmd .= " -c -l 1 -t 1 -o 1";
    my $show_output =`ulimit -q unlimited; $netns_cmd $cmd`;
    return $show_output;
}

sub show_platform_calvados () {
    my $cmd = "platform";
    my $output = confd_cli_wrapper($cmd);
    return $output;
}

sub show_wd_mem () {
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $cmd = "/pkg/bin/show_watchdog";
    if ( ! -f $cmd ) {
        return "";
    }
    my $show_output =`ulimit -q unlimited; $netns_cmd $cmd`;
    return $show_output;
}

sub sdr_show_info () {
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $cmd = "/pkg/bin/sdr_show_info";
    if ( ! -f $cmd ) {
        return "";
    }
    my $show_output =`ulimit -q unlimited; $netns_cmd $cmd`;
    return $show_output;
}

sub get_netstat($$) {
    my $port = shift;
    my $sys_info = shift;

    my $osType = $sys_info->{hostType};
    my $boardtype = $sys_info->{boardtype};
    my $platform = $sys_info->{platform};

    my $netns_cmd = get_netns_env($sys_info);
    my $cmd = $netns_cmd . "/bin/netstat -an";
    if ($osType =~ /calvados|sysadmin/i) { 
        my $chvrf = "/sbin/chvrf 0";
        if ($platform !~ /(panini|scapa|ncs[46]k)/i) {
            $chvrf = "";
        }
        $cmd = $netns_cmd . "$chvrf /bin/netstat -an";
    }
    my $output = `$cmd`; 

} ;#sub get_netstat

sub get_lwm_debug_proc() {
    my $proc = shift || "all";

    #TODO NETNS - netns_cmd
    my $output = `ulimit -q unlimited; /pkg/bin/lwm_debug_proc -b -p $proc 2>&1`;

    #5174   5932             ntpd Reply        00:00:00.000 2306   gsp
    #5276   5608          lpts_fm Reply        00:00:18.441 3500   lpts_pa
    #[xr-vm_node0_RP0_CPU0:~]$ lwm_debug_proc -b -p all
    # PID    TID    ProcessName      State        TimeInState  MiscInfo
    #   5276   5608          lpts_fm Reply        00:02:26.423 3500   lpts_pa
    #No response from /pkg/sbin/show_logging(pid:7396) past 2 seconds, exit.
    #Try '-t' option if process should respond.

    #[xr-vm_node0_RP0_CPU0:~]$ lwm_debug_proc -b -p all 2>&1
    # PID    TID    ProcessName      State        TimeInState  MiscInfo
    #   5174   5932             ntpd Reply        00:00:00.000 2306   gsp
    #   5276   5608          lpts_fm Reply        00:15:13.631 3500   lpts_pa
    #   5304   5937     eem_ed_stats Reply        00:06:51.324 2267   pm_wd
    #   6010   6166   eem_metric_dir Reply        00:04:54.314 2267   node0_RP1_CPU0 PID:2267
    #  29697  29697    show_platform Mutex        02:09:00.852 0x7ff999510100 29697 /pkg/bin/show_platform lwm_once_per_thread_client_operations:421

    my $lwm_info;
    my ($pid, $tid, $name, $state, $time, $peer_pid, $hex_addr, $misc);
    my @pidList = ();
    my $ignore_list = '(lwm_debug_proc|lspv_server)';
    foreach my $line (split(/\n/, $output)) {
        if ( $line =~ /^\s*(\d+)\s+(\d+)\s+(\S+)\s+(\S+)\s+(\d{2}:\d{2}:\d{2}\.\d{3})\s+(\d+)\s+(.*)/ ) {
            $pid = $1;
            $tid = $2;
            $name = $3;
            $state = $4;
            $time = $5;
            $peer_pid = $6;
            $misc = $7;
            next if ( ($name =~ /lpts_fm/) && ($misc =~ /lpts_pa|node\d+_R[S]?P\d+_CPU0/) );
            next if ( ($name =~ /ntpd/) && ($misc =~ /gsp|node\d+_R[S]?P\d+_CPU0/) );
            next if ( ($name =~ /$ignore_list/) && ($misc =~ /$ignore_list/) );
            #next if ( ($name =~ /lspv_server/) && ($misc =~ /lspv_server/) );
            #5926   6065   eem_metric_dir Reply        02:11:03.789 2285   node0_RP1_CPU0 PID:2285
            next if ( ($name eq "eem_metric_dir") );

            next if ( ($name =~ /l2vpn_mgr|mpls_ldp|te_control/) && ($misc =~ /lspv_server/) );
            #  4922   5865        l2vpn_mgr Reply        01:10:05.166 6609   lspv_server
            #  5006   5367          lpts_fm Reply        00:29:22.160 3575   lpts_pa
            #  6525   7859   mibd_interface Reply        00:00:00.006 6636   node0_RP1_CPU0 PID:6636
            #  6646   6669         mpls_ldp Reply        01:09:55.887 6609   lspv_server
            #  7241   7285       te_control Reply        01:07:11.940 6609   lspv_server

            #$lwm_info->{pid} = $pid;
            $lwm_info->{$pid}->{tid} = $tid;
            $lwm_info->{$pid}->{name} = $name;
            $lwm_info->{$pid}->{state} = $state;
            $lwm_info->{$pid}->{time} = $time;
            $lwm_info->{$pid}->{peer_pid} = $peer_pid;
            $lwm_info->{$pid}->{misc} = $misc;
            $lwm_info->{$pid}->{line} = $line;
            push @pidList, $pid;
        } elsif ( $line =~ /^\s*(\d+)\s+(\d+)\s+(\S+)\s+(\S+)\s+(\d{2}:\d{2}:\d{2}\.\d{3})\s+(0x[a-f0-9]+)\s+(\d+)\s+(.*)/ ) {
            #  29697  29697    show_platform Mutex        02:09:00.852 0x7ff999510100 29697 /pkg/bin/show_platform lwm_once_per_thread_client_operations:421
            $pid = $1;
            $tid = $2;
            $name = $3;
            $state = $4;
            $time = $5;
            $hex_addr = $6;
            $peer_pid = $7;
            $misc = $8;
            #TODO
            next if ( ($name =~ /lpts_fm/) && ($misc =~ /lpts_pa|node\d+_R[S]?P\d+_CPU0/) );
            next if ( ($name =~ /ntpd/) && ($misc =~ /gsp|node\d+_R[S]?P\d+_CPU0/) );
            next if ( ($name =~ /lwm_debug_proc/) && ($misc =~ /lwm_debug_proc/) );

            #$lwm_info->{pid} = $pid;
            $lwm_info->{$pid}->{tid} = $tid;
            $lwm_info->{$pid}->{name} = $name;
            $lwm_info->{$pid}->{state} = $state;
            $lwm_info->{$pid}->{time} = $time;
            $lwm_info->{$pid}->{peer_pid} = $peer_pid;
            $lwm_info->{$pid}->{misc} = $misc;
            $lwm_info->{$pid}->{line} = $line;
            push @pidList, $pid;
        } elsif ( $line =~ /No response from .* past \d+ seconds, exit/ ) {
            $lwm_info->{no_response} = $line;
        }
    }
    $lwm_info->{output} = $output;
    $lwm_info->{pidList} = \@pidList;
    return $lwm_info;
} ;#sub get_lwm_debug_proc()

sub get_attach_process_output($$) {
    my $pid = shift;
    my $iteration = shift || 5;
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $cmd = "ulimit -q unlimited; ulimit -c unlimited;";
    $cmd .= $netns_cmd . "/pkg/bin/attach_process -A -p $pid -i $iteration 2>&1";
    my $output = `$cmd`;
    return $output;
}

sub get_ng_show_version () {
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $cmd = "/pkg/bin/ng_show_version";
    if ( ! -f $cmd ) {
        return "";
    }
    my $show_output =`ulimit -q unlimited; $netns_cmd $cmd`;
    return $show_output;
}

sub get_sbt($) {
    my $_logs = shift;
    #LC/0/0/CPU0:Apr 20 18:29:16.768 : netio[296]: %PKT_INFRA-NETIO-3-ERR_NULL_CALLBACKS : A NetIO DLL invoked API 'get evm' before callbacks table initialized  : netio : (PID=3429) :  -Traceback= 7f28e1d9453d 7f28e1d944f5 7f28cd6d09a5 7f28cd6d0644 7f28cd6d05ed 7f28cd6d0ac2 7f28cd6cee6b
    #LC/0/0/CPU0:Jan 16 15:34:49.959 UTC: fib_mgr[208]:
    #RP/0/RP0:Feb 22 22:44:27.858 : 

    my $month_pat = 'Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec';
    my $time_pat = '\d+\s+\d+:\d+:\d+\.\d+';
    my @node_pids;
    my %isSeen;
    my ($proc, $pid, $jid, $node, $trace);
    my $sbt;
    #we pop to use the last Traceback (as old are cleared)
    #RP/1/RP1/CPU0:
    #LC/1/1/CPU0:
    my $pattern = '\s*((LC|R[S]?P)\/\d+\/(R[S]?P)?\d+(\/(CPU)?\d+)?)?:?(';
    $pattern .= $month_pat . ')\s+(' . $time_pat . ')';
    $pattern .= '\s*([A-Z]{3})?:?\s*.*\s+[xr]?\s?([\w\-]+)\[(\d+)\]:\s+.*\s+([\w+\-]+)\s+:\s+';
    $pattern .= '\(PID=(\d+)\)\s*:\s+\-Traceback=\s*([a-f0-9]{6,}.*)';
    while (@$_logs) {
        my $l = pop @$_logs;
        if ($l =~ /$pattern/) {
            my $node   = $1;
            my $timestamp  = $6 . " " . $7;
            #$proc   = $9;
            $jid    = $10;

            $proc   = $11;
            $pid    = $12;
            $trace  = $13;
            if (defined($node) && ($node)) {
                $node   =~ s/^(LC|R[S]?P)\///;
            } else {
                $node   = "";
            }
            $l =~ s/\.*$//;
            my $node_pid = $node . ":" . $pid;
            if (!$isSeen{$node_pid}) {
                $sbt->{$node_pid}->{node} = $node;
                $sbt->{$node_pid}->{pid} = $pid;
                $sbt->{$node_pid}->{timestamp} = $timestamp;
                $sbt->{$node_pid}->{proc} = $proc;
                $sbt->{$node_pid}->{line} = $l;
                $sbt->{$node_pid}->{jid} = $jid;
                $sbt->{$node_pid}->{trace} = $trace;
                push @node_pids, $node_pid;
                $isSeen{$node_pid} = 1;
            }
        }
    }
    $sbt->{node_pids} = \@node_pids;
    return $sbt;
}

############################
sub get_slot_id($) {
    my $node = shift;
    $node =~ s/^node//;
    $node =~ s/\//_/g;
    $node = "node" . $node;
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $slot_id = `$netns_cmd /pkg/bin/node_conversion -i $node`;
    $slot_id =~ s/[\n\r]//g;
    return $slot_id;
}

sub get_as9k_slot_id ($) {
    my $node = shift;

    #remove proceeding "node"
    $node =~ s/^node//;
    # nslot 0
    #slot 00 -> 0/RSP0/CPU0 (0x1)
    # nslot 0/RSP0/CPU0
    #0/RSP0/CPU0 (0x1) -> slot 0
    # nslot 0_RSP0_CPU0
    #0_RSP0_CPU0 (0x1) -> slot 0
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $output = `$netns_cmd /pkg/bin/nslot $node`;
    $output =~ s/\->\s+slot//;
    my ($_node, $hex_id, $_slot) = split(/\s+/, $output);
    $hex_id =~ s/[\(\)]//g;
    $hex_id =~ s/0x//;
    return hex($hex_id);
}

sub get_show_dll($$) {
    my $node = shift;
    my $hex_jid = shift;
    my $slot_id = &get_slot_id($node);
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    #CRS: my $output = `show_dll -j $hex_jid -n $slot_id`;
    my $output = `$netns_cmd show_dll -p $hex_jid -n $slot_id`;
    #size ~ 100K
    return $output;
}

sub get_loadavg () {
    my $load;
    my $file= "/proc/loadavg";
    open(FD, $file) || return $load;
    my $out = <FD>;
    close(FD);
    my ($a1, $a5, $a15, $run, $cpid) = split(/\s+/, $out);
    $load->{a1}   = $a1;
    $load->{a5}   = $a5;
    $load->{a15}  = $a15;
    $load->{run}  = $run;
    $load->{cpid} = $cpid;
    return $load;
}

sub get_rp_ip_from_sysdb() {
    my $sysdb_info;
    my @nodeList;
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    #spitfire/thinxr:
    #LD_LIBRARY_PATH=/opt/cisco/install-iosxr/base/lib/
    #root@xr:/etc#  /pkg/bin/show_platform_sysdb -v
    #Node              Type                       State             Config state
    #--------------------------------------------------------------------------------
    #0/RP0/CPU0        Sherman(Active)            IOS XR RUN        NSHUT

    if ( -f "/pkg/bin/show_platform_sysdb" ) {
        my $cmd = $netns_cmd . "/pkg/bin/show_platform_sysdb -v";
        my $pattern = '\s*(\S+)\s+(R[S]?P|LC)\s+\((\S+)\)\s+\S+\s+';
        $pattern .= 'FINAL\s+Band\s+(\d+\.\d+\.\d+\.\d+)';
        my $output = `$cmd`;
        foreach my $line (split(/\n/, $output)) {
            if ( $line =~ /^${pattern}/) {
                my $node = $1;
                my $type = $3;
                my $ip = $4;
                push @nodeList, $node;
                $sysdb_info->{$node}->{type} = $type;
                $sysdb_info->{$node}->{IP} = $ip;
            }
        }
    }
    $sysdb_info->{nodeList} = \@nodeList;
    return $sysdb_info;
}

sub get_calv_active_rp_from_ip () {

    #netstat -plano|egrep EST | egrep -v "127.0.0.1"|egrep "4565"

    my $confd_port = 4565;
    my $min_connections = 4;
    my $_sys_info = &getOsType();
    my $platform = $_sys_info->{platform};
    my $netns_cmd = get_netns_env($_sys_info);
    my $chvrf = "/sbin/chvrf 0";
    if ($platform !~ /panini|scapa|ncs[46]k/i) {
        $chvrf = "";
    }
    my $cmd = $netns_cmd . "$chvrf /bin/netstat -plano";
    my $output = `$cmd`;
    my $ip = "";
    my %isSeen;
    my @ipList;
    my $count;
    my $pattern = 'tcp\s+\d+\s+\d+\s+192\.\d+\.\d+\.1:\d+\s+';
    $pattern .= '(192\.\d+\.\d+\.1):' . $confd_port;
    $pattern .= '\s+ESTABLISHED\s+\d+\/(pm|cm|dumper|ccc_driver)';
    foreach my $line (split(/\n/, $output)) {
        #TODO this may not work on MC (SC)
        if ( $line =~ /$pattern/) {
            $ip = $1;
            $count->{$ip}++;
            return $ip if ($count->{$ip} > $min_connections);
            if (!$isSeen{$ip}) {
                $isSeen{$ip} = 1;
                push @ipList, $ip
            }
        }
    }
    return "";
}

sub get_calvados_active_rp () {
    #the following connections should be established between LC and active RP
    # LC(pm bios_fpd calv_alarm_mgr ccc_driver cm ds dumper envmon led_mgr ntp_helper 
    #    obfl_show ship_server slice_manager timezone_notif vm_manager wdmon)
    # <------------------> RP (confd)

    #tcp        0      0 192.0.64.1:53330            192.0.0.1:4565              ESTABLISHED 1752/pm             off (0.00/0/0)
    #tcp        0      0 192.0.64.1:53337            192.0.0.1:4565              ESTABLISHED 1770/obfl_show      off (0.00/0/0)
    #tcp        0      0 192.0.64.1:53296            192.0.0.1:4565              ESTABLISHED 1759/ccc_driver     off (0.00/0/0)
    #tcp        0      0 192.0.64.1:53336            192.0.0.1:4565              ESTABLISHED 1770/obfl_show      off (0.00/0/0)
    #tcp        0      0 192.0.64.1:53304            192.0.0.1:4565              ESTABLISHED 1768/ntp_helper     off (0.00/0/0)

    my $_sys_info = &getOsType();
    my $platform = $_sys_info->{platform};
    my $netns_cmd = get_netns_env($_sys_info);
    my $cmd = $netns_cmd . "netstat -tplano";
    if ( -f "/opt/cisco/calvados/sbin/ccc_driver" ) {
        my $chvrf = "/sbin/chvrf 0";
        if ($platform !~ /panini|scapa|ncs[46]k/i) {
            $chvrf = "";
        }
        $cmd = $netns_cmd . "/sbin/chvrf 0 /bin/netstat -tplano";
    }

    my $min_connections = 8;
    my $output = `$cmd`;
    my $ip = "";
    my %isSeen;
    my @ipList;
    my $count;
    my $pattern = 'tcp\s+\d+\s+\d+\s+192\.0\.\d+\.1:\d+\s+(192\.0\.[04]\.1)';
    $pattern .= ':\d+\s+ESTABLISHED\s+\d+\/\S+';
    foreach my $line (split(/\n/, $output)) {
        if ( $line =~ /$pattern/) {
            $ip = $1;
            $count->{$ip}++;
            return $ip if ($count->{$ip} > $min_connections);
            if (!$isSeen{$ip}) {
                $isSeen{$ip} = 1;
                push @ipList, $ip
            }
        }
    }
    foreach my $ip (@ipList) {
        return $ip if ($count->{$ip} > $min_connections);
    }
    return "";
} ;# sub get_calvados_active_rp ()

sub get_smu_from_show_version($) {
    my $output = shift;

    my @ddtsList = ();
    my %isSeen;
    my $pattern='Name:\s+\w+\-([\d+\.\w]+)\.(CSC[a-z]{2}\d{5})\-*([\d\.a-z]*)';
    foreach my $line (split(/\n/, $output)) {
        if ( $line =~ /^${pattern}$/) {
            my $image = $1;
            my $ddts = $2;
            my $smu_version = $3;
            my $smu_name = "$image,$ddts,$smu_version";
            if ( !$isSeen{$smu_name} ) {
                $isSeen{$smu_name} = 1;
                push @ddtsList, $smu_name;
            }
        }
    }
    return \@ddtsList;
}

sub create_syslog($$$) {
    my $sys_info = shift;
    my $message = shift;
    my $severity = shift || "info";
    # see - infra/syslog/src/syslog_logger.c
    my @default_severity = (
                  'emerg',
                  'alert',
                  'crit',
                  'err',
                  'warning',
                  'notice',
                  'info',
                  'debug');
    if (!grep(/\b$severity\b/i, @default_severity)) {
        $severity = "info";
    }

    my $osType = $sys_info->{hostType};
    my $boardtype = $sys_info->{boardtype};
    my $platform = $sys_info->{platform};

    my $netns_cmd = get_netns_env($sys_info);
    my $cmd = "";
    my $chvrf = "";
    $ENV{LD_LIBRARY_PATH} = "/pkg/lib:/pkg/lib/cerrno:/pkg/lib/mib:/pkg/lib/spp_plugins";
    if ($osType =~ /xr/i) {
        $cmd = $netns_cmd . "/pkg/bin/logger";
        $cmd .= " -s " . $severity;
    } elsif ($osType =~ /calv/i) {
        if ($platform =~ /panini|scapa|ncs[46]k/i) {
            $chvrf = $netns_cmd . "/sbin/chvrf 0";
        }
        $cmd = $netns_cmd . "$chvrf /opt/cisco/calvados/bin/cal_logger";
        $cmd .= " -s " . $severity;
    } else {
        #host os:
        $cmd = "/usr/bin/logger";
    }
    if ($cmd ne "") {
        $cmd .= " " . "\"$message\"";
        system("$cmd");
    }
    return 1;
}

sub confd_cli_wrapper_exe($) {
    my $cli = shift;
    my $chvrf = "";
    my $cmdline = `cat /proc/cmdline`;
    if ($cmdline =~ /platform=.*(panini|scapa|ncs[46]k)/i) {
        $chvrf = "/sbin/chvrf 0";
    }
    my $cmd = "$chvrf /opt/cisco/calvados/bin/admin-cli-proxy<<END
$cli | nomore
END";
    return `$cmd`;
}

sub confd_cli_wrapper($) {
    my $cli = shift;
    $cli =~ s/^\s*show\s+//i;
    my $chvrf = "";
    my $cmdline = `cat /proc/cmdline`;
    if ($cmdline =~ /platform=.*(panini|scapa|ncs[46]k)/i) {
        $chvrf = "/sbin/chvrf 0";
    }
    my $cmd = "$chvrf /opt/cisco/calvados/bin/admin-cli-proxy<<END
show $cli | nomore
END";
    return `$cmd`;
}

sub get_calvados_sdr($) {
    my $get_active_rp = shift;

    my $output = confd_cli_wrapper("sdr");
    my $sdr_data;
    my @nodeList;
    my $node;
    #source 
    #sdr default-sdr
    # location 0/RP0
    #  sdr-id             2
    #  IP Address of VM   192.0.0.4
    #  MAC address of VM  E0:50:72:F4:DD:06
    #  VM State           RUNNING
    #  start-time         2013-08-30T13:36:13.913602+00:00
    #  Last Reload Reason CARD_SHUTDOWN
    #  Reboot Count       1
    my $is_new_show_sdr = 0;
    foreach my $line (split(/\n/, $output)) {
        if ($line =~ /R[S]?P\d+\/VM\d+\s+192\.\d+/) {
            $is_new_show_sdr = 1;
        }
        $line =~ s/[\r]//g;
        if ($line =~ /^\s*(\S+)\s+(\d+\.\d+\.\d+\.\d+)\s+(\w+)\s+\d+\s+(.*)/ ) {
            $node = $1;
            my $ip = $2;
            my $state = $3;
            my $start_time = $4;

            push @nodeList, $node;
            $sdr_data->{$node}->{IP} = $ip;
            $sdr_data->{$node}->{state} = $state;
            $sdr_data->{$node}->{'start-time'} = $start_time;
            if ( ($get_active_rp) &&
                 ($node =~ /R[S]?P|B\d+\/CB\d+/i) &&
                 ($sdr_data->{$node}->{state} =~ /RUNNING/i) ) {
                return $sdr_data->{$node}->{IP};
            }
        }
    }
    if ($is_new_show_sdr) {
        $sdr_data->{nodeList} = \@nodeList;
        return $sdr_data;
    }
    foreach my $line (split(/\n/, $output)) {
        if ($line =~ /location\s+(\S+)/ ) {
            $node = $1;
            push @nodeList, $node;
        }
        if ($line =~ /IP +Address +of +VM +(\d+\.\d+\.\d+\.\d+)/ ) {
            my $ip = $1;
            $sdr_data->{$node}->{IP} = $ip;
        }
        if ($line =~ /sdr\-id +(\d+)/ ) {
            my $sdr_id = $1;
            $sdr_data->{$node}->{'sdr-id'} = $sdr_id;
        }
        if ($line =~ /VM +State +(\S+)/ ) {
            my $state = $1;
            $sdr_data->{$node}->{state} = $state;
            if ( ($get_active_rp) &&
                 ($node =~ /R[S]?P|B\d+\/CB\d+/i) &&
                 ($sdr_data->{$node}->{state} =~ /RUNNING/i) ) {
                return $sdr_data->{$node}->{IP};
            }
        }
        if ($line =~ /start\-time +(\S+)/ ) {
            my $start_time = $1;
            $sdr_data->{$node}->{'start-time'} = $start_time;
        }
    }
    $sdr_data->{nodeList} = \@nodeList;
    return $sdr_data;
} ;#sub get_calvados_sdr()

sub get_platform_sysdb($) {
    my $get_active_rp = shift || 1;

    my $sysdb_data;
    my @nodeList;
    my $node;
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $cmd = $netns_cmd . "/pkg/bin/show_platform_sysdb";
    if ( ! -f $cmd ) {
        print "$cmd not found!\n";
        return $sysdb_data;
    }
    my $output = `$cmd -v`;

    #Panini
    #[xr-vm_node0_0_CPU0:~]$ show_platform_sysdb -v
    #Node name       Node type       Partner name    SW status       IP address
    #--------------- --------------- --------------- --------------- ----------
    #0/0/CPU0        LC (ACTIVE)     NONE            FINAL Band      192.0.64.3
    #0/RP0/CPU0      RP (ACTIVE)     0/RP1/CPU0      FINAL Band      192.0.0.4
    #0/RP1/CPU0      RP (STANDBY)    0/RP0/CPU0      FINAL Band      192.0.4.4

    #scapa
    #Node name       Node type       Partner name    SW status       IP address
    #--------------- --------------- --------------- --------------- ----------
    #0/LC1           LC (STANDBY)    0/LC0           FINAL Band      192.0.88.8
    #0/RP1           RP (STANDBY)    0/RP0           FINAL Band      192.0.88.6
    #0/RP0           RP (ACTIVE)     0/RP1           FINAL Band      192.0.44.4
    #0/LC0           LC (ACTIVE)     0/LC1           FINAL Band      192.0.44.6

    my $pattern = '\s*(\S+)\s+(R[S]?P|LC)\s+\((STANDBY|ACTIVE|UNKNOWN)\)';
    $pattern .= '\s+(\S+)\s+(.*)\s+(\d+\.\d+\.\d+\.\d+)';
    foreach my $line (split(/\n/, $output)) {
        if ($line =~ /^$pattern/) {
             $node = $1;
             my $type = $2;
             my $state = $3;
             my $partner = $4;
             my $ip = $5;
             push @nodeList, $node;
             $sysdb_data->{$node}->{type} = $type;
             $sysdb_data->{$node}->{state} = $state;
             $sysdb_data->{$node}->{partner} = $partner;
             $sysdb_data->{$node}->{IP} = $ip;
        }
        if ( ($get_active_rp) &&
             ($node =~ /R[S]?P|B\d+\/CB\d+/i) &&
             ($sysdb_data->{$node}->{state} =~ /ACTIVE/i) ) {
             return $sysdb_data->{$node}->{IP};
        }
    }
    $sysdb_data->{nodeList} = \@nodeList;
    return $sysdb_data;
} ;#sub get_platform_sysdb()

sub get_platform_sysdb_from_file($) {
    my $file = shift;

    my $sysdb_data;
    my @nodeList;
    my $node;
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $cmd = $netns_cmd . "/pkg/bin/show_platform_sysdb";
    if ( ! open(FD, $file)) {
        print "$file not found!\n";
        return $sysdb_data;
    }

    #Panini
    #[xr-vm_node0_0_CPU0:~]$ show_platform_sysdb -v
    #Node name       Node type       Partner name    SW status       IP address
    #--------------- --------------- --------------- --------------- ----------
    #0/0/CPU0        LC (ACTIVE)     NONE            FINAL Band      192.0.64.3
    #0/RP0/CPU0      RP (ACTIVE)     0/RP1/CPU0      FINAL Band      192.0.0.4
    #0/RP1/CPU0      RP (STANDBY)    0/RP0/CPU0      FINAL Band      192.0.4.4

    #scapa
    #Node name       Node type       Partner name    SW status       IP address
    #--------------- --------------- --------------- --------------- ----------
    #0/LC1           LC (STANDBY)    0/LC0           FINAL Band      192.0.88.8
    #0/RP1           RP (STANDBY)    0/RP0           FINAL Band      192.0.88.6
    #0/RP0           RP (ACTIVE)     0/RP1           FINAL Band      192.0.44.4
    #0/LC0           LC (ACTIVE)     0/LC1           FINAL Band      192.0.44.6

    my $pattern = '\s*(\S+)\s+(R[S]?P|LC)\s+\((STANDBY|ACTIVE|UNKNOWN)\)';
    $pattern .= '\s+(\S+)\s+(.*)\s+(\d+\.\d+\.\d+\.\d+)';
    while (my $line=<FD>) {
        if ($line =~ /^$pattern/) {
            $node = $1;
            my $type = $2;
            my $state = $3;
            my $partner = $4;
            my $sw_status = $5;
            my $ip = $6;
            push @nodeList, $node;
            $sysdb_data->{$node}->{type} = $type;
            $sysdb_data->{$node}->{state} = $state;
            $sysdb_data->{$node}->{partner} = $partner;
            $sysdb_data->{$node}->{sw_status} = $sw_status;
            $sysdb_data->{$node}->{IP} = $ip;
        }
    }
    close(FD);
    $sysdb_data->{nodeList} = \@nodeList;
    return $sysdb_data;
} ;#sub get_platform_sysdb_from_file()

sub do_sshfs_mount($$$$) {
    my $remote_mount_dir = shift;
    my $dlrsc_ip = shift;
    my $local_mount_dir = shift;
    my $username = shift || "root";

    $remote_mount_dir =~ s/\/+$//;
    $local_mount_dir =~ s/\/+$//;

    my $ret = check_stale_sshfs_mount($local_mount_dir, $remote_mount_dir);
    if ($ret eq 1) {
        #already mounted:
        return 1;
    } elsif ($ret eq -1) {
        #stale - remount:
        system("/bin/umount $local_mount_dir >/dev/null 2>&1");
    }
    mkdir $local_mount_dir if (! -d $local_mount_dir);
    my $cmdline = `cat /proc/cmdline`;
    #TODO - use netns for external mount!
    my $chvrf = "";
    if (($cmdline =~ /vmtype=sysadmin/) &&
        ($cmdline =~ /platform=(panini|scapa|ncs[46]k)/i)) {
        $chvrf = "/sbin/chvrf 0";
    }
    my $cmd = "PATH=\"/usr/bin:/bin:/usr/sbin:/sbin\" && $chvrf /usr/bin/sshfs";
    $cmd .= " -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no";
    $cmd .= " $username\@$dlrsc_ip:$remote_mount_dir $local_mount_dir";
    if (system("$cmd >/dev/null 2>&1")) {
        #print "$cmd failed: $!\n";
        return 0;
    }

    #verify - undef to read all in on shot:
    $ret = check_stale_sshfs_mount($local_mount_dir, $remote_mount_dir);
    if ($ret eq 1) {
        return 1;
    } else {
        #print "Failed to mount $remote_mount_dir $local_mount_dir\n";
        return 0;
    }
} ;# sub do_sshfs_mount($$$$)

sub sshfs_mount_proc($$) {
    my $dlrsc_ip = shift;
    my $local_mount_dir = shift;

    my $username = "root";
    my $remote_mount_dir = "/proc/";
    $local_mount_dir =~ s/\/+$//;

    my $ret = check_stale_sshfs_mount($local_mount_dir, $remote_mount_dir);
    if ($ret eq 1) {
        #already mounted:
        return 1;
    } elsif ($ret eq -1) {
        #stale - remount:
        system("/bin/umount $local_mount_dir >/dev/null 2>&1");
    }
    mkdir $local_mount_dir if (! -d $local_mount_dir);
    my $_sys_info = &getOsType();
    my $osType = $_sys_info->{hostType};
    my $platform = $_sys_info->{platform};
    my $netns_cmd = get_netns_env($_sys_info);
    my $chvrf = "";
    if (($osType =~ /sysadmin/i) &&
        ($platform =~ /panini|scapa|ncs[46]k/i)) {
        $chvrf = "/sbin/chvrf 0";
    }
    my $cmd = "PATH=\"/usr/bin:/bin:/usr/sbin:/sbin\" && ";
    $cmd .= $netns_cmd . "$chvrf /usr/bin/sshfs";
    $cmd .= " -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no";
    $cmd .= " -o direct_io";
    $cmd .= " $username\@$dlrsc_ip:$remote_mount_dir $local_mount_dir";
    if (system("$cmd >/dev/null 2>&1")) {
        return 0;
    }

    #verify - undef to read all in on shot:
    $ret = check_stale_sshfs_mount($local_mount_dir, $remote_mount_dir);
    if ($ret eq 1) {
        return 1;
    } else {
        return 0;
    }
} ;# sub sshfs_mount_proc($$)

sub check_stale_sshfs_mount($$) {
    my $local_mount_dir = shift;
    my $remote_mount_dir = shift;

    $local_mount_dir =~ s/\/*$//;
    $local_mount_dir =~ s/\/+/\//g;

    $remote_mount_dir =~ s/\/*$//;
    $remote_mount_dir =~ s/\/+/\//g;

    my $df = `df -k $local_mount_dir 2>/dev/null`;
    my %df_info;
    my (@df_mnts, @mtab_mnts);
    #regular filesystem d%:
    #root@192.0.12.3:/tmp/     65536   708     64828   2% /mnt/junk
    #ramfs (no d%):
    #root@192.0.12.3:/proc/         0     0         0    - /opt/cisco/pam/mnt/0_1_CPU0
    #proc                   0     0         0    - /proc/sys
    foreach my $line ((split/\n/, $df)) {
        if ( $line =~ /[\d+%|\-]\s+(\/[\w\/\-\.]+)/ ) {
            my $mnt = $1;
            #$mnt =~ s/\/+$//;
            $df_info{$mnt} = 1;
        }
    }
    #verify - undef to read all in on shot:
    local $/ = '\0';
    if (!open(FD, "/proc/mounts") ) {
        print "Failed to open /proc/mounts: $!\n";
        return 0;
    }
    my $mounts = <FD>;
    close(FD);
    my $mtab_mnted = 0;

    my $match_line = "[\\/]*\\s+$local_mount_dir\\s+fuse(\\.sshfs)?\\s+";
    if ($remote_mount_dir =~ /[\/\w+]+/) {
        $match_line = "${remote_mount_dir}[\\/]*\\s+";
        $match_line .= "$local_mount_dir\\s+fuse(\\.sshfs)?\\s+";
    }
    foreach my $line (split(/\n/, $mounts)) {
        if ( $line =~ /$match_line/ ) {
            $mtab_mnted = 1;
            last;
        }
    }
    if (!$mtab_mnted) {
        #not mounted
        return 0;
    } elsif ( ($mtab_mnted) && (!$df_info{$local_mount_dir}) ) {
        #stale
        return -1;
    } else {
        #mounted
        return 1;
    }
}

sub umount_sshfs($) {
    my $mount= shift;
    $mount =~ s/\/*$//;
    $mount=~ s/\/+/\//g;
    local $/ = '\0';
    if (!open(FD, "/proc/mounts") ) {
        print "Failed to open /proc/mounts: $!\n";
        return 0;
    }
    my $mounts = <FD>;
    close(FD);
    my $sshfs_mounted = 0;

    my $match_line = "\\s+${mount}\\s+fuse(\\.sshfs)?\\s+";
    foreach my $line (split(/\n/, $mounts)) {
        if ( $line =~ /$match_line/ ) {
            $sshfs_mounted = 1;
            last;
        }
    }
    if ($sshfs_mounted) {
        #umount
        if (system("umount $mount")) {
            #stale
            return -1;
        }
    }
    #umounted
    return 1;
}

sub sortCrashbyTime($$) {
    my $crash_data = shift;
    my $coreList = shift;

    my @temp = ();
    foreach my $c (@$coreList) {
        next if (!defined($c));
        next if ($c !~ /\w+/);
        if (defined($crash_data->{$c}->{ctime})) {
            push @temp, "$crash_data->{$c}->{ctime} $c";
        } else {
            push @temp, "0 $c";
        }
    }
    my @_sorted = ();
    my @sorted = ();
    if (scalar(@temp) > 0) {
        @_sorted = sort { (split ' ', $a)[0].(split ' ', $a)[1] cmp
                            (split ' ', $b)[0].(split ' ', $b)[1] } @temp;
        foreach my $line (@_sorted) {
            my ($_ctime, $_c) = split (/\s+/, $line);
            push @sorted, $_c;
        }
    }
    return @sorted;
}

sub get_current_crashes($$) {
    my $sys_info = shift;
    my $archive_dir = shift || "/misc/disk1/";

    my $osType = $sys_info->{hostType};
    my $version = &getWS($osType);
    my $buildDate = $version->{buildDate};
    my $efr = $version->{efr};

    my $month_map = {
        'Jan' => 1,
        'Feb' => 2,
        'Mar' => 3,
        'Apr' => 4,
        'May' => 5,
        'Jun' => 6,
        'Jul' => 7,
        'Aug' => 8,
        'Sep' => 9,
        'Oct' => 10,
        'Nov' => 11,
        'Dec' => 12,
    };

    #Fri Apr 25 22:43:45 PDT 2014
    my ($month, $day, $year);
    my $y_m_d;
    my $pattern1 = '\s*\w{3}\s+(\w+)\s+(\d+)\s+\d+:\d+:\d+\s+(\S+\s+)*(\d{4})';
    my $pattern2 = '\.by\.\d+\.(\d{8})\-(\d{6})\.(sysadmin\-vm:\d+_[LCRSP]*\d+';
    $pattern2 .= '|xr\-vm_node\d+_[LCRSP]*\d+_CPU\d+)\.[a-f0-9]{5}\.core\.gz';
    if ( $buildDate =~ /^$pattern1/) {
         $month = $1;
         $day = $2;
         $year = $4;
         $month = $$month_map{$1};
         $month = ( length($month) > 1 ) ? $month : "0${month}";
         $day = ( length($day) > 1 ) ? $day : "0$day}";
         $y_m_d = $year . $month . $day;
    }

    my $core_info;
    my @cores = ();
    my @nodelist = ();
    my $last_timestamp = 0;
    my %isSeen;
    if (opendir(DIR, $archive_dir)) {
        my @files = readdir(DIR);
        closedir(DIR);
        #aib_10198.by.11.20140501-080842.xr-vm_node0_6_CPU0.ad733.core.gz
        #slice_manager_1750.by.11.20140121-170332.sysadmin-vm:0_7.ab54f.core.gz
        #led_mgr_2576.by.11.20140321-234437.sysadmin-vm:0_RP0.0bf88.core.gz
        foreach my $file (@files) {
            if ( $file =~ /$pattern2/) {
                my $year_mm_day = $1;
                my $time = $2;
                my $node = $3;
                my $timestamp = $year_mm_day . $time;
                if ($year_mm_day >= $y_m_d) {
                    push @cores, $file;
                    if (!$isSeen{$node}) {
                        $isSeen{$node} = 1;
                        push @nodelist, $node;
                    }
                    if ( defined($core_info->{$node}->{cores}) ) {
                        $core_info->{$node}->{cores} .= "|" . $file;
                    } else {
                        $core_info->{$node}->{cores} = $file;
                    }
                    #$core_info->{$file}->{date} = $year_mm_day;
                    if ($timestamp > $last_timestamp ) {
                        $last_timestamp = $timestamp;
                    }
                }
            }
        }
    }
    $core_info->{last_timestamp} = $last_timestamp;
    $core_info->{nodelist} = \@nodelist;
    return $core_info;
} ;#sub get_current_crashes($)

################################################################################
# copy config/exec history
################################################################################
# show_history  -p 18681 -e -d
#           1 Sun Apr 13 13:05:54.693 show history | last 20
#           2 Sun Apr 13 13:06:00.013 show history last 20
#           3 Sun Apr 13 13:06:11.049 show logging last 10
#           4 Sun Apr 13 13:06:27.494 show logging | inc exec

#[xr-vm_node0_RP0_CPU0:~]$ cfgmgr_show_history -s -f -n 0x5dc -t all -b
#Sno.  Event      Info                           Time Stamp
#~~~~  ~~~~~      ~~~~                           ~~~~~~~~~~
#1     startup    configuration applied          Fri Apr 11 13:54:45 2014
#2     commit     id 1000000001                  Fri Apr 11 14:02:18 2014
#3     commit     id 1000000002                  Fri Apr 11 14:03:11 2014
#4     commit     id 1000000003                  Fri Apr 11 14:03:40 2014
#5     commit     id 1000000004                  Fri Apr 11 14:11:38 2014
#6     backup     Periodic ASCII backup          Fri Apr 11 14:49:31 2014

sub get_pid_cmd_maps($$) {
    my $dir = shift;
    my $minimum_pid = shift || 200; #ignore kernel processes

    my $myPid = $$;

    my $stat;
    my @pidList;
    if (opendir(DIR, $dir)) {
        my @_pidList = readdir(DIR);
        foreach my $pid (@_pidList) {
            if ($pid =~ /^\d+$/) {
                next if ($pid < $minimum_pid);
                next if ($pid eq $myPid);
                next if (! -d "/proc/$pid");
                my $cmdFile = "/proc/$pid/cmdline";
                next if (! -f $cmdFile);
                next if (!open(CFD, $cmdFile));
                my $cmdline=<CFD>;
                close(CFD);
                next if (!defined($cmdline));
                my $cmd = (split(/\0/, $cmdline))[0];
                next if ( (!defined($cmd)) || ($cmd !~ /\w+/) );
                if ($cmd !~ /[\s\@]pts\/\d+/) {
                    $cmd = basename($cmd);
                    $cmd =~ s/\s+$//;
                }
                push @pidList, $pid;
                $stat->{$pid} = $cmd;
            }
        }
    }
    closedir(DIR);
    $stat->{pidList} = \@pidList;
    return $stat;
} ;#sub get_pid_cmd_maps($)

sub get_calv_command_history($$$) {
    my $_confd_audit_logs = shift;
    my $last_timestamp = shift;
    my $last_mtime = shift;
    my @confd_audit_logs = @$_confd_audit_logs;

    my $month_map = {
        'Jan' => '01',
        'Feb' => '02',
        'Mar' => '03',
        'Apr' => '04',
        'May' => '05',
        'Jun' => '06',
        'Jul' => '07',
        'Aug' => '08',
        'Sep' => '09',
        'Oct' => '10',
        'Nov' => '11',
        'Dec' => '12',
    };

    #my $history_commands = shift || "/var/log/confd_audit.log";
    my $month = '(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)';
    my $time_template = "(\\d+)\\-${month}\\-(\\d{4})::";
    $time_template .= "(\\d{2}:\\d{2}:\\d{2}\\.\\d{3})";
    my $template = "<INFO>\\s+${time_template}\\s+\\S+\\s+\\S+\\s+";
    $template .= "audit\\s+user:\\s+(\\w+)\\/(\\d+)\\s+CLI\\s+(\\'[\\S\\s]+\\')";

    my $cli_data;
    my @timestamps;
    my $_timestamp = $last_timestamp;
    my $_last_timestamp = $_timestamp;
    $cli_data->{timestamps} = \@timestamps;
    $cli_data->{last_timestamp} = $last_timestamp;
    $cli_data->{last_mtime} = $last_mtime;
    my ($_year,$_month,$_day, $_hms);
    my ($user,$tty,$mode,$timestamp,$cli);
    foreach my $history_commands (@confd_audit_logs) {
        if ( (! -f $history_commands) || (-s $history_commands < 10) ) {
            next;
        }
        my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
              $atime,$mtime,$ctime,$blksize,$blocks) = stat($history_commands);
        next if ($mtime < $last_mtime);
        $last_mtime = $mtime;
        my $lines = "";
        if (!open(fd1, "$history_commands") ) {
            return $cli_data;
        }
        while (my $line = <fd1>) {
            chomp $line;
            if ( $line =~ /$template/ ) {
                $_day = $1;
                $_month = $2;
                $_year = $3;
                $_hms = $4;
                $user = $5;
                $tty = $6;
                $cli = $7;
                $cli =~ s/'//g;

                $_day = ($_day =~ /\d{2}/) ? $_day : "0${_day}";
                $_month =  $$month_map{$_month};
                $_month = ($_month =~ /\d{2}/) ? $_month : "0${_month}";
                $timestamp = $_year . $_month . $_day . "-" . $_hms;

                my ($hour, $min, $sec) = (split(/:/, $_hms));
                $_timestamp = $_year . $_month . $_day . $hour . $min . $sec;

                next if ($cli eq 'gone');
                next if ($_timestamp <= $last_timestamp);
                if ($_timestamp >= $_last_timestamp) {
                    $_last_timestamp = $_timestamp;
                }

                push @timestamps, $timestamp;
                $cli_data->{$timestamp}->{user} = $user;
                $cli_data->{$timestamp}->{tty} = $tty;
                #TODO - add mode
                $cli_data->{$timestamp}->{mode} = "-";
                $cli_data->{$timestamp}->{cli} = $cli;
            } ;#if ( $line =~ /$template/ ) 
        } ;#while (my $line = <fd1>)
        close(fd1);
    } ;#foreach my $history_commands (@confd_audit_logs)
    #my @sorted_timestamps = reverse sort {$b <=> $a} @timestamps;
    # cannot sort non-numerically
    my @sorted_timestamps = reverse sort (@timestamps);
    #$cli_data->{last_timestamp} = @sorted_timestamps[$#timestamps];
    #$cli_data->{last_timestamp} = $_timestamp;
    $cli_data->{last_timestamp} = $_last_timestamp;
    $cli_data->{last_mtime} = $last_mtime;
    $cli_data->{timestamps} = \@sorted_timestamps;
    return $cli_data;
} ;#sub get_calv_command_history($)

sub get_active_rp_nodes($) {
    my $sys_info = shift;

    my $osType = $sys_info->{hostType};
    my $vf1_3073_ip = $sys_info->{vf1_3073_ip};

    my $node_info;
    my @ipList = ();
    my @rp_ip_addrs = ();
    $node_info->{active_ip} = "";
    $node_info->{ipList} = \@ipList;
    $node_info->{rp_ip_addrs} = \@rp_ip_addrs;
    my %isSeen;
    $node_info->{isActive} = 0;
    my @activeNodes = ();
    if ($osType =~ /xr/i) {
        $node_info = get_rp_ip_from_sysdb();
        foreach my $node (@{$node_info->{nodeList}}) {
            my $type = $node_info->{$node}->{type} ;
            my $_ip = $node_info->{$node}->{IP};
            if (!$isSeen{$_ip}) {
                $isSeen{$_ip} = 1;
                push @ipList, $_ip;
            }
            if ( ($node =~ /R[S]?P|B\d+\/CB\d+/i) && ($type =~ /active/i) ) {
                push @rp_ip_addrs, $_ip;
                $node_info->{$_ip} = $node;
                if ($vf1_3073_ip eq $_ip) {
                    $node_info->{active_ip} = $_ip;
                    $node_info->{isActive} = 1;
                    push @activeNodes, $node;
                }
            }
        }
    } elsif ($osType =~ /calv|sysadmin/i) {
        my $xr_vf1_3073_ip = $vf1_3073_ip;
        #it's 192.0.x.4 !!!
        $xr_vf1_3073_ip =~ s/(.*).\d+$/$1\.4/;
        my $foundRP = 0;
        if (-f "/opt/cisco/calvados/bin/pam_show_node") {
            $node_info = get_calvados_pam_show_sdr($sys_info);
        } else {
            $node_info = get_calvados_sdr(0);
        }
        #skip VM2 (192.0.44.6/192.0.88.6) on scapa
        #e.g., 0/RP0/VM2, 0/RP1/VM2
        #      0/RSP0/VM2, 0/RSP1/VM2 --> Viking
        foreach my $node (@{$node_info->{nodeList}}) {
            my $_ip = $node_info->{$node}->{IP};
            $_ip =~ s/(.*).\d+$/$1\.1/;
            next if ( ($isSeen{$_ip}) || 
                      (($sys_info->{platform} =~ /nsc4k|scapa/i) && ($node =~ /VM2/)) );
            if (!$isSeen{$_ip}) {
                $isSeen{$_ip} = 1;
                push @ipList, $_ip;
            }
            if ($node =~ /R[S]?P|B\d+\/CB\d+/i) {
                push @rp_ip_addrs, $_ip;
            }
            $node_info->{$_ip} = $node;
            $node_info->{$node}->{IP} = $_ip;
            if ( ($foundRP < 1) && ($node =~ /R[S]?P|B\d+\/CB\d+/i) ) {
                $foundRP++;
                if ($vf1_3073_ip eq $_ip) {
                    $node_info->{active_ip} = $_ip;
                    $node_info->{isActive} = 1;
                    push @activeNodes, $node;
                }
            }
        }
    }
    $node_info->{ipList} = \@ipList;
    $node_info->{rp_ip_addrs} = \@rp_ip_addrs;
    return $node_info;
} ;#sub get_active_rp_nodes($)

sub get_active_rp_nodes_by_chassis_id($$) {
    my $sys_info = shift;
    my $chassis_id = shift;

    my $osType = $sys_info->{hostType};
    my $vf1_3073_ip = $sys_info->{vf1_3073_ip};

    my $node_info;

    my @ipList = ();
    my @rp_ip_addrs = ();
    my @nodeList = ();
    $node_info->{active_ip} = "";
    $node_info->{ipList} = \@ipList;
    $node_info->{rp_ip_addrs} = \@rp_ip_addrs;
    $node_info->{nodeList} = \@nodeList;
    $node_info->{isActive} = 0;
    my %isSeen;
    my $node  = "";
    #hardcode for spitfire until infra can allow to dynamically identify node/IP 
    if ($sys_info->{is_thinxr}) {
        my $IP = "172.0.0.1";
        $node  = "0/RP0/CPU0";
        @ipList = ($IP);
        @rp_ip_addrs = ($IP);
        $node_info->{active_ip} = $IP;
        @nodeList = ($node);
        $node_info->{isActive} = 1;
        $node_info->{$node}->{type} = "Active";
        $node_info->{ipList} = \@rp_ip_addrs;
        $node_info->{rp_ip_addrs} = \@rp_ip_addrs;
        $node_info->{nodeList} = \@nodeList;
        return $node_info;
    }

    if ($osType =~ /xr/i) {
        $node_info = get_rp_ip_from_sysdb();
        my $first_xr_ip = get_first_active_xr_ip($node_info);
        foreach my $node (@{$node_info->{nodeList}}) {
            my $_chassis_id = get_chassis_id_from_node($node);
            next if ($_chassis_id ne $chassis_id);

            push @nodeList, $node;
            my $type = $node_info->{$node}->{type} ;
            my $_ip = $node_info->{$node}->{IP};
            if (!$isSeen{$_ip}) {
                $isSeen{$_ip} = 1;
                push @ipList, $_ip;
            }
            #B1/CB0/CPU0     RP (STANDBY)    B0/CB0/CPU0     FINAL Band      192.0.0.4
            #B0/CB0/CPU0     RP (ACTIVE)     B1/CB0/CPU0     FINAL Band      192.3.0.4
            #[xr-vm_node0_RP0_CPU1:~]$show_platform_sysdb -v
            #Node name       Node type       Partner name    SW status       IP address
            #--------------- --------------- --------------- --------------- ---------------
            #B0/CB0/CPU1     RP (ACTIVE)     NONE            FINAL Band      192.1.0.6
            #0/RP0/CPU1      RP (ACTIVE)     0/RP1/CPU1      FINAL Band      192.0.0.4
            #0/RP1/CPU1      RP (STANDBY)    0/RP0/CPU1      FINAL Band      192.0.4.4

            if ( ($node =~ /R[S]?P|B\d+\/CB\d+/i) && ($type =~ /active/i) ) {
                push @rp_ip_addrs, $_ip;
                $node_info->{$_ip} = $node;
                if (($vf1_3073_ip eq $_ip) && ($vf1_3073_ip eq $first_xr_ip)) {
                    $node_info->{active_ip} = $_ip;
                    $node_info->{isActive} = 1;
                }
            }
        }
    } elsif ($osType =~ /calv|sysadmin/i) {
        my $xr_vf1_3073_ip = $vf1_3073_ip;
        #it's 192.0.x.4 !!!
        #TODO - cannot assume 4 !!!
        $xr_vf1_3073_ip =~ s/(.*).\d+$/$1\.4/;
        my $foundRP = 0;
        if (-f "/opt/cisco/calvados/bin/pam_show_node") {
            $node_info = get_calvados_pam_show_sdr($sys_info);
        } else {
            $node_info = get_calvados_sdr(0);
        }
        #skip VM2 (192.0.44.6/192.0.88.6) on scapa
        #e.g., 0/RP0/VM2, 0/RP1/VM2
        #      0/RSP0/VM2, 0/RSP1/VM2 --> Viking

        my $first_calv_ip = get_first_calv_ip($node_info);
        foreach my $node (@{$node_info->{nodeList}}) {
            my $_chassis_id = get_chassis_id_from_node($node);
            next if ($_chassis_id ne $chassis_id);

            push @nodeList, $node;
            my $_ip = $node_info->{$node}->{IP};
            my $xr_ip = $_ip;
            $_ip =~ s/(.*).\d+$/$1\.1/;
            #next if ( ($isSeen{$_ip}) || ($node =~ /VM2/));
            next if ( ($isSeen{$_ip}) );
            if (!$isSeen{$_ip}) {
                $isSeen{$_ip} = 1;
                push @ipList, $_ip;
            }
            if ($node =~ /R[S]?P|B\d+\/CB\d+/i) {
                push @rp_ip_addrs, $_ip;
            }
            $node_info->{$_ip} = $node;
            $node_info->{$node}->{IP} = $_ip;
            $node_info->{$node}->{XR_IP} = $xr_ip;
            if (($node =~ /R[S]?P|B\d+\/CB\d+/i) &&
                ($vf1_3073_ip eq $_ip) &&
                ($vf1_3073_ip eq $first_calv_ip)) {
                $node_info->{active_ip} = $_ip;
                $node_info->{isActive} = 1;
            }
        }
    }
    $node_info->{nodeList} = \@nodeList;
    $node_info->{ipList} = \@ipList;
    $node_info->{rp_ip_addrs} = \@rp_ip_addrs;
    return $node_info;
} ;# sub get_active_rp_nodes_by_chassis_id($$)

sub isFirstActiveRP($) {
    my $sys_info = shift;
    my $osType = $sys_info->{hostType};
    my $vf1_3073_ip = $sys_info->{vf1_3073_ip};
    my $result = 0;
    if ($osType =~ /xr/i) {
        my $node_info = get_rp_ip_from_sysdb();
        foreach my $node (@{$node_info->{nodeList}}) {
            my $type = $node_info->{$node}->{type} ;
            if ( ($node =~ /R[S]?P|B\d+\/CB\d+/i) && ($type =~ /active/i) ) {
                my $_ip = $node_info->{$node}->{IP};
                if ($vf1_3073_ip eq $_ip) {
                    return 1;
                } else {
                    return 0;
                }
            }
        }
    } elsif ($osType =~ /calv|sysadmin/i) {
        my $node_info;
        if (-f "/opt/cisco/calvados/bin/pam_show_node") {
            $node_info = get_calvados_pam_show_sdr($sys_info);
        } else {
            $node_info = get_calvados_sdr(0);
        }
        foreach my $node (@{$node_info->{nodeList}}) {
            if ( $node =~ /R[S]?P|B\d+\/CB\d+/i ) {
                my $_ip = $node_info->{$node}->{IP};
                $_ip =~ s/(.*).\d+$/$1\.1/;
                #next if ($node =~ /VM2/);
                if ($vf1_3073_ip eq $_ip) {
                    return 1;
                } else {
                    return 0;
                }
            }
        }
    }
    return $result;
} ;# sub isFirstActiveRP($)

sub get_chassis_id($) {
    my $sys_info = shift;

    if ($sys_info->{is_thinxr}) {
        return get_chassis_id_from_node($sys_info->{hostname});
    }    
    my $hostType = $sys_info->{hostType};
    my $boardtype = $sys_info->{boardtype};
    my $vf1_3073_ip = $sys_info->{vf1_3073_ip};
    my $osType = $sys_info->{hostType};
    if ($osType =~ /xr/i) {
        my $node_info = get_rp_ip_from_sysdb();
        foreach my $node (@{$node_info->{nodeList}}) {
            my $_ip = $node_info->{$node}->{IP};
            if ($vf1_3073_ip eq $_ip) {
                return get_chassis_id_from_node($node);
            }
        }
    } elsif ($osType =~ /syadmin|calv/i) {
        my $node_info;
        if (-f "/opt/cisco/calvados/bin/pam_show_node") {
            $node_info = get_calvados_pam_show_sdr($sys_info);
        } else {
            $node_info = get_calvados_sdr(0);
        }
        foreach my $node (@{$node_info->{nodeList}}) {
            if ( $node =~ /R[S]?P|B\d+\/CB\d+/i ) {
                my $_ip = $node_info->{$node}->{IP};
                $_ip =~ s/(.*).\d+$/$1\.1/;
                if ($vf1_3073_ip eq $_ip) {
                    return get_chassis_id_from_node($node);
                }
            }
        }
    }
    return "";
} ;# sub get_chassis_id($)

sub get_chassis_id_from_node($) {
    my $node = shift;

    # xr-vm_node0_RP0_CPU0
    # sysadmin-vm:3_RP0
    # sysadmin-vm:2_3
    # sysadmin-vm:F0_SC0:R45-P9-admin
    # 0/RP1/CPU0
    # 0/RP0/VM2
    # 0/RP0 - scapa doesn't return /CPU0

    # xr-vm_nodeB0_CB0_CPU0
    # sysadmin-vm:B1_CB0
    # B1/CB0/CPU0     RP (STANDBY)    B0/CB0/CPU0     FINAL Band      192.0.0.4

    $node =~ s/[\n\r]//g;
    if ($node =~ /\d+\/LC\d+/) {
        $node =~ s/.*(\d+)\/LC\d+.*/$1/;
    } elsif ($node =~ /xr\-vm/) {
        $node =~ s/.*(node)?(\d+)_(R[S]?P|[CB])?\d+_.*/$2/;
        # xr-vm_nodeB0_CB0_CPU0
        $node =~ s/.*(node)?[BC]?(\d+)_[RSPCB]*\d+_.*/$2/;
    } elsif ($node =~ /sysadmin\-vm:/) {
        $node =~ s/.*\-vm:(\d+)_.*/$1/;
    } elsif ($node =~ /\d+\/(R[S]?P)?\d+(\/(CPU|VM)?\d+)?/) {
        $node =~ s/.*(\d+)\/(R[S]?P)?\d+.*/$1/;
    } elsif ($node =~ /[CB]\d+\/[CB]\d+(\/(CPU|VM)?\d+)?/) {
        #B0/CB0/VM2
        #B1/CB0/CPU0     RP (STANDBY)    B0/CB0/CPU0     FINAL Band      192.0.0.4
        $node =~ s/.*[CB](\d+)\/[CB]\d+.*/$1/;
    } elsif ($node =~ /\d+\/LC\d+(\/(CPU|VM)?\d+)?/) {
        $node =~ s/.*(\d+)\/LC\d+.*/$1/;
    } elsif ($node =~ /^(node)?\d+_[RSPCB]*\d+_/) {
        $node =~ s/^(node)?(\d+)_(R[S]?P|[CB]*)?\d+_.*/$2/;
    }
    return $node;
}

sub get_active_nodes($) {
    my $sys_info = shift;

    my $osType = $sys_info->{hostType};
    my $vf1_3073_ip = $sys_info->{vf1_3073_ip};

    my $node_info;
    $node_info->{active_ip} = "";
    if ($osType =~ /xr/i) {
        $node_info = get_rp_ip_from_sysdb();
        foreach my $node (@{$node_info->{nodeList}}) {
            my $type = $node_info->{$node}->{type} ;
            if ( ($node =~ /R[S]?P|B\d+\/CB\d+/i) && ($type =~ /active/i) ) {
                my $_ip = $node_info->{$node}->{IP};
                if ($vf1_3073_ip eq $_ip) {
                    $node_info->{active_ip} = $_ip;
                }
            }
        }
    } elsif ($osType =~ /calv/i) {
        my $foundRP = 0;
        if (-f "/opt/cisco/calvados/bin/pam_show_node") {
            $node_info = get_calvados_pam_show_sdr($sys_info);
        } else {
            $node_info = get_calvados_sdr(0);
        }
        foreach my $node (@{$node_info->{nodeList}}) {
            my $_ip = $node_info->{$node}->{IP};
            $_ip =~ s/(.*).\d+$/$1\.1/;
            $node_info->{$node}->{IP} = $_ip;
            if ( ($foundRP < 1) && ($node =~ /R[S]?P|B\d+\/CB\d+/i) ) {
                $foundRP++;
                if ($vf1_3073_ip eq $_ip) {
                    $node_info->{active_ip} = $_ip;
                }
            }
        }
    }
    return $node_info;
}

sub create_console_log($$) {
    my $msg = shift;
    my $color = shift || "";

    my $start_tag = "\033[0;31m";
    my $end_tag = "\033\[0m";
    if ($color =~ /red/i) {
        $start_tag = "\033[0;31m";
        $end_tag = "\033\[0m";
    } elsif ($color =~ /green/i) {
        $start_tag = "\033[0;32m";
        $end_tag = "\033\[0m";
    } elsif ($color =~ /yellow/i) {
        $start_tag = "\033[0;33m";
        $end_tag = "\033\[0m";
    }

    my $console_dev = "/dev/hvc0";
    if  ( (-f "/dev/console") || ( -l "/dev/console") ) {
        $console_dev = "/dev/console";
    }
    if (readlink($console_dev)) {
        $console_dev = readlink($console_dev);
    }
    my $local_time = localtime();
    $msg = $local_time . " : " . $msg;
    if ( -c $console_dev ) {
        if (open(CONSOLE, ">$console_dev")) {
            #print CONSOLE $msg, "\n";
            print CONSOLE $start_tag, $msg, $end_tag, "\n";
        }
        close(CONSOLE);
    }
}

sub getVrfIp ($) {
    my $file = shift;
    my $ip = "";
    if (!open(FD, $file)) {
        print "ERROR: $file missing.\n";
        return $ip;
    }
    while (my $line = <FD>) {
        if ($line =~ /inet addr/) {
            $line =~ s/.*net\s+addr://;
            $line =~ s/\s+Bcast.*//g;
            chomp $line;
            $ip = $line;
            last;
        }
    }
    close(FD);
    return $ip;
}

sub get_show_install($) {
    my $sys_info = shift;

    my $osType = $sys_info->{hostType};
    my $output = "";

    my $netns_cmd = get_netns_env($sys_info);
    if ($osType =~ /calv|admin/i) {
        my $cmd = "install active";
        $output = confd_cli_wrapper($cmd);
    } elsif ($osType =~ /xr/i) {
        my $boardtype = $sys_info->{boardtype};
        if ( $boardtype =~ /LC/i ) {
            my $ssh_o = " -o UserKnownHostsFile=/dev/null";
            $ssh_o .= " -o StrictHostKeyChecking=no";
            my $rp_node_info = get_active_rp_nodes($sys_info);
            my $rp_xr_ip = @{$rp_node_info->{rp_ip_addrs}}[0];
            if ($rp_xr_ip =~ /\d+\.\d+\.\d+\.\d+/) {
                my $cmd = "/pkg/bin/sdr_instcmd show install active";
                $output = `$netns_cmd /usr/bin/ssh $rp_xr_ip \"$cmd\"`;
            }
        } else {
            if (-x "/pkg/bin/sdr_instcmd" ) {
                $output = `$netns_cmd /pkg/bin/sdr_instcmd show install active`;
            }
        }
    }
    return $output;
}

########################################################
#get ddts from show install (for smu)
########################################################
sub get_ddts_from_show_install($) {
    my $sys_info = shift;

    my $osType = $sys_info->{hostType};

    my $proc_args_info = get_proc_args();
    my $isLocal = 0;
    if ( (defined($proc_args_info->{L})) && ($proc_args_info->{L}) ) {
        $isLocal = $proc_args_info->{L};
    }
    my $show_install = get_show_install($sys_info);

    my %isSeen;
    my $info;
    my @ddtsList = ();
    #$show_install =~ s/\r//g;
    foreach my $line (split(/\n/, $show_install)) {
        $line =~ s/[\r\n]//;
        if ( $line =~ /\.(CSC[a-z]{2}\d{5})[\-\.]?([\w\.]*)$/ ) {
            my $ddts = $1;
            my $smu_version = $2;
            if (!$isSeen{$ddts}) {
                $isSeen{$ddts} = 1;
                push @ddtsList, $ddts;
                $info->{$ddts}->{smu_version} = $smu_version;
            }
            next;
        }
        if ( $line =~ /Version\s+[:]?\s+(\S+)/) {
            my $version = $1;
            $info->{version} = $version;
            next;
        }
        #ncs4k-xr-5.2.1.11S version=5.2.1.11S [Boot image]
        #ncs4k-sysadmin-5.2.1.11S version=5.2.1.11S [Boot image]
        if ( $line =~ /(\S+)\s+version=(\S+)\s+\[Boot +image\]/i ) {
            my $platform = $1;
            my $version = $2;
            if ( !defined($info->{version}) ) {
                $info->{version} = $version;
            }
            $platform =~ s/\-(xr|sysadmin).*//;
            if ( !defined($info->{platform}) ) {
                $info->{platform} = $platform;
            }
            next;
        }
    }
    $info->{ddtsList} = \@ddtsList;
    return $info;
} ;#sub get_ddts_from_show_install()

###############################################
# find out the latest binary path from smu ws
# assume latest is being used!!!
###############################################
sub get_smu_binary_path($$) {
    my $ddts_smu_info = shift;
    my $binary_file = shift;

    my $full_binary_file = "";
    my $smu_ws_info;
    $smu_ws_info->{smu_ws} = "";
    $smu_ws_info->{smu_ws_binary_file} = "";
    foreach my $ddts (@{$ddts_smu_info->{valid_ddts_list}}) {
        my $sym_ws = $ddts_smu_info->{$ddts}->{sym_ws};
        my $smu_ws_binary_file = $sym_ws . "/" . $binary_file;
        if ( -f $smu_ws_binary_file ) {
            $smu_ws_info->{smu_ws} = $sym_ws;
            $smu_ws_info->{smu_ws_binary_file} = $smu_ws_binary_file;
            return $smu_ws_info;
        }
    }
    return $smu_ws_info;
} ;#sub get_smu_binary_path($$)

sub localtime_to_yearmonthday($) {
    my $buildDate = shift;

    my $month_map = {
        'Jan' => 1,
        'Feb' => 2,
        'Mar' => 3,
        'Apr' => 4,
        'May' => 5,
        'Jun' => 6,
        'Jul' => 7,
        'Aug' => 8,
        'Sep' => 9,
        'Oct' => 10,
        'Nov' => 11,
        'Dec' => 12,
    };

    #Fri Apr 25 22:43:45 PDT 2014
    #Tue Feb  4 11:04:25 PST 2014
    my ($month, $day, $year);
    my ($hour, $min, $sec);
    my $y_m_d = "00000000";
    my $date_pat = '\s*\w{3}\s+(\w+)\s+(\d+)\s+(\d+):(\d+)';
    $date_pat .= ':(\d+)\s+(\S+\s+)*(\d{4})';
    if ( $buildDate =~ /^$date_pat/) {
         $month = $1;
         $day = $2;
         $hour = $3;
         $min = $4;
         $sec = $5;
         $year = $7;
         $month = $$month_map{$month};
         $month = "0${month}" if ( length($month) < 2 );
         $day =  "0${day}" if ( length($day) < 2 );
         $y_m_d = $year . $month . $day;
    }
    return $y_m_d;
} ;#sub get_yearmonthday($);

sub verify_core_by_builddate($$) {
    my $build_ymd = shift;
    my $core = shift;

    my $core_pat = '\.by\.\d+\.(\d{8})\-(\d{6})\.(sysadmin\-vm[:_]';
    $core_pat .= '\d+_[R[S]?P]*\d+|xr\-vm_node\d+_[RSPLC]*\d+_CPU\d+)';
    $core_pat .= '\.[a-f0-9]{5}\.core';
    if ( $core =~ /$core_pat/) {
        #XR/Calv processes:
        my $year_mm_day = $1;
        return 1 if ($year_mm_day >= $build_ymd);

    } elsif ( $core =~ /sysadmin\.(\d{8})\-(\d{6})\.core/ ) {
        #host vm:
        #sysadmin.20140604-204853.core.host.lz
        my $year_mm_day = $1;
        return 1 if ($year_mm_day >= $build_ymd);

    } elsif ( $core =~ /\.by\.\d+\.(\d{8})\-(\d{6})\.host\.[a-f0-9]{5}\.core/ ) {
        #host processes:
        #kvm-qemu-system-x86_64_1797.by.11.20140414-215620.host.4cd5c.core
        my $year_mm_day = $1;
        return 1 if ($year_mm_day >= $build_ymd);
    } elsif ( $core =~ /\.by\.\d+\.(\d{8})\-(\d{6})\.uvf\.[a-f0-9]{5}\.core/ ) {
        #uvf processes:
        my $year_mm_day = $1;
        return 1 if ($year_mm_day >= $build_ymd);
    }
    return 0;
} ;# sub verify_core_by_builddate($$)

#################################################
# get all command arguments (-u, -i -p -L -P)
#################################################
sub get_proc_args() {
    my $pid = shift || $$;

    my $cmdline = "/proc/$pid/cmdline";
    my $info;
    if (open(FD, $cmdline)) {
        my $line = <FD>;
        close(CFD);
        return $info if (!defined($line));
        my @cmdlines = (split(/\0/, $line));
        my ($name1, $name2);
        for (my $i = 0; $i<=$#cmdlines; $i++) {
            my $name1 = $cmdlines[$i];
            if ($name1 eq "-u") {
                my $j = $i + 1;
                if ($cmdlines[$j]) {
                    $info->{u} = $cmdlines[$j];
                }
            }
            if ($name1 eq "-i") {
                my $j = $i + 1;
                if ($cmdlines[$j]) {
                    $info->{i} = $cmdlines[$j];
                }
            }
            if ($name1 eq "-p") {
                my $j = $i + 1;
                if ($cmdlines[$j]) {
                    $info->{p} = $cmdlines[$j];
                }
            }
            if ($name1 eq "-L") {
                $info->{L} = 1;
            }
            if ($name1 eq "-P") {
                my $j = $i + 1;
                if ($cmdlines[$j]) {
                    $info->{P} = $cmdlines[$j];
                }
            }
            if ($name1 eq "-o") {
                my $j = $i + 1;
                if ($cmdlines[$j]) {
                    $info->{o} = $cmdlines[$j];
                }
            }
        }
    }
    return $info;
} ;#sub get_proc_args()

sub get_core_osType() {
    my $sys_info = shift;
    my $crash = shift;

    my $osType = $sys_info->{hostType};
    my $platform = $sys_info->{platform};

    my $core_osType = $osType;
    if ( $platform =~ /sunstone|asr9k/i) {
        if ( $crash =~ /xr\-vm/i ) {
           $core_osType = "xr";
        } elsif ( $crash =~ /sysadmin\-vm/i ) {
           $core_osType = "calvados";
        } elsif ( $crash =~ /uvf/i ) {
      #sunstone-x86-fs-prod.mucode_21997.by.11.20141028-113656.uvf.5958b.core.gz
           $core_osType = "uvf";
        } else {
           $core_osType = "unknown";
        }
    } else {
        if ( $crash =~ /\/opt\/host_core/ ) {
            $core_osType = "host";
        }
    }
    return $core_osType;
} ;#sub get_core_osType()

sub get_xr_hostname() {
    my $hostname = `nvgen -c -q gl/a/hostname`;
    if ((defined($hostname)) && ($hostname =~ /\w+/)) {
       $hostname =~ s/.*hostname\s+//;
       $hostname =~ s/[\r\n]//g;
       return $hostname;
    } else {
       return "ios";
    }
    #nvgen -c -q gl/a/hostname
    #hostname antares_TB2-r2
}

###########################
###########################
sub get_master_pam_conf($) {
    my $confFile = shift;
    my $info;
    if ( ! -f $confFile ) {
        my $console_dev = "/dev/console";
        my $local_time = localtime();
        my $msg = $local_time . " - PAM alert : ";
        #if (open(CONSOLE, ">$console_dev")) {
        #    print CONSOLE $msg, "\n";
        #}
        #close(CONSOLE);
        print "$confFile nout found\n";
        return $info;
    }
    if (!open(FD, $confFile)) {
        print "Failed to open $confFile: $!\n";
        return $info;
    }
    while (my $line = <FD>) {
        chomp $line;
        next if ( $line =~ /^\s*$/ );
        next if ( $line =~ /^\s*#/ );
        $line =~ s/= *\"/=/;
        $line =~ s/\" *$//;
        if ($line =~  /total_leak_threshold_pct\s*=\s*([\.\d]+)/i ) {
            $info->{total_leak_threshold_pct} = $1;
        } elsif ($line =~  /leak_threshold_pct\s*=\s*([\.\d]+)/i ) {
            $info->{leak_threshold_pct} = $1;
        } elsif ( $line =~ /delta_threshold\s*=\s*([\.\d]+)/i ) {
            $info->{delta_threshold} = $1;
        } elsif ( $line =~ /(monitor_interval|refresh_interval)\s*=\s*([\.\d]+)/i ) {
            $info->{refresh_interval} = $2;
        } elsif ( $line =~ /excluded_processes\s*=\s*([\w\-\s\,]+)/i ) {
            $info->{excluded_processes} = $1;
        } elsif ( $line =~ /included_processes\s*=\s*([\w\-\s\,]+)/i ) {
            $info->{included_processes} = $1;
        } elsif ( $line =~ /email\s*=\s*([\w\-\@\.\,\s]+)/i ) {
            $info->{email} = $1;
            $info->{email} =~ s/['\"]//g;
        } elsif ( $line =~ /username\s*=\s*(\w+)/i ) {
            $info->{username} = $1;
            $info->{username} =~ s/['\"]//g;
        } elsif ( $line =~ /ssh_ip\s*=\s*(\d+\.\d+\.\d+\.\d+)/i ) {
            $info->{ssh_ip} = $1;
            $info->{ssh_ip} =~ s/['\"]//g;
        } elsif ( $line =~ /ssh_port\s*=\s*(\d+)/i ) {
            $info->{ssh_port} = $1;
            $info->{ssh_port} =~ s/['\"]//g;

        } elsif ( $line =~ /tftp_ip\s*=\s*(\d+\.\d+\.\d+\.\d+)/i ) {
            $info->{tftp_ip} = $1;
            $info->{tftp_ip} =~ s/['\"]//g;
        } elsif ( $line =~ /tftp_path\s*=\s*(\S+)/i ) {
            $info->{tftp_path} = $1;
            $info->{tftp_path} =~ s/[\"']//g;
        } elsif ( $line =~ /extra_alert_patterns\s*=\s*(.*)/i ) {
            $info->{extra_alert_patterns} = $1;
        } elsif ( $line =~ /excl_alert_patterns\s*=\s*(.*)/i ) {
            $info->{excl_alert_patterns} = $1;
        } elsif ( $line =~ /system_load_threshold\s*=\s*(\d+)/i ) {
            $info->{system_load_threshold} = $1;
        } elsif ( $line =~ /process_cpu_threshold\s*=\s*(\d+)/i ) {
            $info->{process_cpu_threshold} = $1;
        } elsif ( $line =~ /(\S+)\s*=\s*(.*)/ ) {
            my $key = $1;
            my $value = $2;
            $value =~ s/^\s*//;
            $value =~ s/\s*$//;
            $info->{$key} = $value;
        }
    }
    close(FD);
    return $info;
} ;#sub get_master_pam_conf($)

sub verify_pam_conf_input($) {
    my $conf_info = shift;
    my @mandatory = ('perl_lib_path',
                     'pam_root',
                     'ssh_ip',
                     'ssh_port',
                     'username',
                     'email');
    foreach my $item (@mandatory) {
        if (!defined($conf_info->{$item})) {
            print "Mandatory '$item' is not defined in the config\n";
            exit;
        }
    }
    return $conf_info;
}

sub get_df_lxc_vol($$) {
    my $sys_info = shift;
    my $vmType = shift;

    if ( (!defined($sys_info->{hostType})) ||
           ($sys_info->{hostType} !~ /host/i) ) {
        return "";
    }

    #/dev/loop0 1460  666 7172  49% /lxc_rootfs/panini_vol_grp-calvados_lv10
    #/dev/loop5 2996 1274 15800 46% /lxc_rootfs/panini_vol_grp-xr_lv10

    my $vol_name;
    if ($vmType =~ /xr/) {
         $vol_name = "xr_lv";
    } elsif ($vmType =~ /uvf/) {
         $vol_name = "uvf_lv";
    } elsif ($vmType =~ /calvados/) {
        $vol_name = "calvados_lv";
    } else {
       return "";
    }

    $ENV{PATH} = "/sbin:/usr/sbin:/usr/bin:/bin:" . $ENV{PATH};
    my $output = `df -kl`;
    foreach my $line (split(/\n/, $output)) {
        if ($line =~ /\s+(\/lxc_rootfs\/panini_vol_grp\-${vol_name}\d+)/ ) {
            return $1;
        }
    }
    return "";
}

sub do_log_rotation($$$$) {
    my $log_dir = shift;
    my $log_rotation_interval = shift;    #7 days: 7*24*3600
    my $log_rotation_size_limit = shift;  #10MB
    my $log_rotation_count = shift || 3;  #3 rotations

    if (!opendir(DIR, $log_dir)) {
        closedir(DIR);
        return 1;
    }
    my @logs = readdir(DIR);
    closedir(DIR);
    my $log_info;
    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
                                  $atime,$mtime,$ctime,$blksize,$blocks);
    my @logList = ();
    foreach my $log_file (@logs) {
        next if ($log_file =~ /\.conf/);
        #next if ($log_file =~ /\.\d+\.gz/);
        next if ($log_file =~ /\.[t]?gz/);
        next if ($log_file !~ /\w+/);
        next if ($log_file =~ /^\./);

        my $fullname = $log_dir . "/" . $log_file;
        next if ( -d $fullname);

        ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
                 $atime,$mtime,$ctime,$blksize,$blocks) = stat($fullname);

        my $log_tracker = $log_dir . "/" . "." . $log_file;
        my $duration = time() - $mtime;
        if (-f $log_tracker) {
            my ($dev1,$ino1,$mode1,$nlink1,$uid1,$gid1,$rdev1,$size1,
                      $atime1,$mtime1,$ctime,@junk) = stat($log_tracker);
            my $now = time();
            $duration = $now - $mtime1;
            if (($duration < $log_rotation_interval) &&
                           ($size < $log_rotation_size_limit)) {
                next;
            }
        } else {
            if ($size < $log_rotation_size_limit) {
                next;
            }
        }
        my $log = new Logfile::Rotate(File    => $fullname,
                                      Count   => $log_rotation_count,
                                      Gzip    => 'lib',
                                      Flock   => 'yes',
                                      Persist => 'yes',
                             );
        $log->rotate();
        undef $log;
        if (open(WD, ">$log_tracker")) {
            print WD "";
        }
        close(WD);
        push @logList, $log_file;
    }
    return @logList;
} ;# sub do_log_rotation($$$$)

sub cleanup_tgz($$) {
    my $log_dir = shift;
    my $retention_period = shift;    #14 days: 14*24*3600
    if (!opendir(DIR, $log_dir)) {
        closedir(DIR);
        return 0;
    }
    my @logs = readdir(DIR);
    closedir(DIR);
    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
                              $atime,$mtime,$ctime,$blksize,$blocks);
    foreach my $log_file (@logs) {
        next if ($log_file !~ /\.([t]?gz|txt|log)$/);
        my $fullname = $log_dir . "/" . $log_file;
        next if ( -d $fullname);
        ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
                      $atime,$mtime,$ctime,$blksize,$blocks) = stat($fullname);
        my $duration = time() - $mtime;
        if (-f $fullname) {
            my ($dev1,$ino1,$mode1,$nlink1,$uid1,$gid1,
                 $rdev1,$size1,$atime1,$mtime1,$ctime,@junk) = stat($fullname);
            my $now = time();
            $duration = $now - $mtime1;
            if ($duration > $retention_period) {
                my $ltime = localtime();
                print "$ltime: Remove $fullname.\n";
                unlink $fullname;
            }
        }
    }
    return 1;
} ;# sub cleanup_tgz($$)

sub dump_calvados_core_cli($$$) {
    my $sys_info = shift; 
    my $node = shift; 
    my $proc = shift; 

    my $corepath = "/misc/disk1/";
    if ($sys_info->{platform} =~ /scapa|ncs6k/i) {
        $corepath = "/mnt/ecu/vdd";
    }

    my $info;
    my %isSeen;
    $info->{core}->{ip} = "";
    $info->{core}->{path} = "";
    $info->{msg} = "";
    $info->{rc} = -1;
    my $old_core_info = get_CoreFiles($sys_info, $node);
    foreach my $ip (@{$old_core_info->{ipList}}) {
        foreach my $c (@{$old_core_info->{$ip}->{coreList}}) {
            next if ($c !~ /$proc/);
            $isSeen{$c} = 1;
        }
    }

    my $exe_cmd = "dumpcure running $proc $node";
    my $ret = confd_cli_wrapper_exe($exe_cmd);
    sleep 20;
    my $found = 0;
    my $max_duration = 45;
    my $duration = 0;
    my $interval = 5;
    while ( (!$found) && ($duration <= $max_duration)) {
        #check if the core is created:
        my $new_core_info = get_CoreFiles($sys_info, $node);
        foreach my $ip (@{$new_core_info->{ipList}}) {
            foreach my $c (@{$new_core_info->{$ip}->{coreList}}) {
                next if ($c !~ /$proc.*core\.gz/);
                if ((!defined($isSeen{$c})) || (!$isSeen{$c})) {
                    $info->{core}->{rp_ip} = $ip;
                    #$info->{core}->{path} = $corepath . "/" . $c;
                    $info->{core}->{path} = $c;
                    $info->{rc} = 1;
                    $info->{msg} = "Core ($c) successfully dumped";
                    $found++;
                    last;
                }
            }
        }
        $duration += $interval;
        sleep $interval;
    }
    return $info;
} ;# sub dump_calvados_core_cli($$$)

sub dump_xr_core($$$$) {
    my $sys_info = shift; 
    my $pid = shift; 
    my $proc = shift; 
    my $node = shift; 

    my $corepath = "/misc/disk1/";
    if ($sys_info->{platform} =~ /scapa|ncs6k/i) {
        $corepath = "/mnt/ecu/vdd";
    }

    if ( $node eq "") {
        $node = `uname -n`;
        if ( -f "/pkg/bin/uname" ) {
            $node = `/pkg/bin/uname -n`;
        }
        $node =~ s/[\n\r]//g;
    }
    my $node_id = &get_slot_id($node);
    my $info;
    my %isSeen;
    $info->{core}->{ip} = "";
    $info->{core}->{path} = "";
    $info->{msg} = "";
    $info->{rc} = -1;

    my $netns_cmd = get_netns_env($sys_info);
    my $cmd = $netns_cmd;
    $cmd .= "/pkg/bin/sh_proc_pid_jid_conversion p2j $pid";
    my $jid = `cmd`;
    $jid =~ s/[\n\r\s]*//g;
    if ($jid !~ /^\d+$/) {
        $info->{msg} = "Unable to find jid from";
        $info->{msg} .= " 'sh_proc_pid_jid_conversion p2j $pid'";
        $info->{rc} = -1;
        return $info;
    }
    my $old_core_info = get_CoreFiles($sys_info, $node);
    foreach my $ip (@{$old_core_info->{ipList}}) {
        foreach my $c (@{$old_core_info->{$ip}->{coreList}}) {
            next if ($c !~ /$proc/);
            $isSeen{$c} = 1;
        }
    }

    my $hex_jid = sprintf("0x%x", $jid);
    $cmd = $netns_cmd;
    $cmd .= "/pkg/bin/corehelper_gen -o running -j $hex_jid ";
    $cmd .= "-n $node_id 2>&1"; 
    my $ret = `$cmd`;
    $ret =~ s/[\n\r]//g;
    $ret =~ s/[\s]*$//g;
    $ret =~ s/^[\s]*//g;
    if ($ret !~ /\w+/) {
        $info->{msg} = "Unable to dumpcore via 'corehelper_gen";
        $info->{msg} .= " -o running -j $hex_jid -n $node_id'";
        $info->{rc} = -1;
        return $info;
    }
    sleep 20;
    #check if the core is created:
    my $max_duration = 45;
    my $found = 0;
    my $duration = 0;
    my $interval = 5;
    while ( (!$found) && ($duration <= $max_duration)) {
        my $new_core_info = get_CoreFiles($sys_info, $node);
        foreach my $ip (@{$new_core_info->{ipList}}) {
            foreach my $c (@{$new_core_info->{$ip}->{coreList}}) {
                next if ($c !~ /$proc.*core\.gz/);
                if ((!defined($isSeen{$c})) || (!$isSeen{$c})) {
                    $info->{core}->{rp_ip} = $ip;
                    #$info->{core}->{path} = $corepath . "/" . $c;
                    $info->{core}->{path} = $c;
                    $info->{rc} = 1;
                    $info->{msg} = "Core ($c) successfully dumped";
                    $found++;
                    last;
                }
            }
        }
        $duration += $interval;
        sleep $interval;
    }
    return $info;
} ;# sub dump_xr_core($$$)

sub get_CoreFiles($$) {
    my $sys_info = shift;
    my $my_node = shift;
    $my_node =~ s/\//_/g;

    my @rp_ip_addrs = ();
    my $chvrf = "";
    my $corepath = "/misc/disk1/";
    if ($sys_info->{platform} =~ /scapa|ncs6k/i) {
        $corepath = "/mnt/ecu/vdd";
    }
    my $vf1_3073_ip = $sys_info->{vf1_3073_ip};
    my $osType = $sys_info->{hostType};

    my $platform = $sys_info->{platform};
    my $netns_cmd = get_netns_env($sys_info);
    my %isSeen;
    if ($osType =~ /xr/i) {
        my $sysdb_info = get_rp_ip_from_sysdb();
        foreach my $node (@{$sysdb_info->{nodeList}}) {
            if ($node =~ /R[S]?P|B\d+\/CB\d+/i) {
                my $_ip = $sysdb_info->{$node}->{IP};
                if (!$isSeen{$_ip}) {
                    $isSeen{$_ip} = 1;
                    push @rp_ip_addrs, $_ip;
                }
            }
        }
    } elsif ($osType =~ /calv/i) {
        my $sysdb_info;
        if (-f "/opt/cisco/calvados/bin/pam_show_node") {
            $sysdb_info = get_calvados_pam_show_sdr($sys_info);
        } else {
            $sysdb_info = get_calvados_sdr(0);
        }
        foreach my $node (@{$sysdb_info->{nodeList}}) {
            if ($node =~ /R[S]?P|B\d+\/CB\d+/i) {
                my $_ip = $sysdb_info->{$node}->{IP};
                $_ip =~ s/(.*).\d+$/$1\.1/;
                if (!$isSeen{$_ip}) {
                    $isSeen{$_ip} = 1;
                    push @rp_ip_addrs, $_ip;
                }
            }
        }
        if ($sys_info->{platform} =~ /panini|scapa/i) {
            $chvrf = "chvrf 0";
        }
    }
    my $ssh_o = " -q -o UserKnownHostsFile=/dev/null";
    $ssh_o .= " -o StrictHostKeyChecking=no";
    my $info;
    $info->{ipList} = \@rp_ip_addrs;
    foreach my $rp_ip (@rp_ip_addrs) {
        my @coreList = ();
        my $cmd = "";
        if ($rp_ip eq $vf1_3073_ip) {
            $cmd = "ls $corepath/*${my_node}*core*gz";
        } else {
            $cmd = $netns_cmd . "$chvrf /usr/bin/ssh $ssh_o $rp_ip";
            $cmd .= " \"ls $corepath/*${my_node}*core*gz\"";
        }
        my $cores = `$cmd`;
        foreach my $core (split(/\s+/,$cores)) {
            push @coreList, $core;
        }
        $info->{$rp_ip}->{coreList} = \@coreList;
    }
    return $info;
} ;# sub get_CoreFiles($$)

sub get_LocalCoreFiles() {
    my $corepath = "/misc/disk1/";
    my $cmdline = `cat /proc/cmdline`;
    if ( ($cmdline =~ /platform=(scapa|ncs6k)/i) && (-d "/mnt/ecu/vdd") ) {
        $corepath = "/mnt/ecu/vdd";
    }
    my @coreList = ();
    if (!opendir(DIR, $corepath)) {
        return @coreList;
    }
    my @files = readdir(DIR);
    closedir(DIR);
    foreach my $core (@files) {
        if ($core =~ /core.*(gz|txt)/) {
            my $full_core = "${corepath}/${core}";
            $full_core =~ s/\/+/\//g;
            push @coreList, $full_core;
        }
    }
    return @coreList;
}

############################
sub pam_logger($$$$) {
    my $sys_info = shift;
    my $pam_log_dir = shift;
    my $bucket = shift;
    my $msg = shift;
    my $file = $pam_log_dir . "/pam-" . $bucket . ".log";

    my $folder = $pam_log_dir;
    my $ret = &createFolder($pam_log_dir);
    if (!$ret) {
        my $msg = "Failed to create $folder";
        my $severity = "warning";
        create_syslog($sys_info, $msg, $severity);
        return 0;
    }

    my $clock = localtime();
    $clock =~ s/[\r\n]//;

    if (open(FD, ">>$file")) {
        #print $msg, "\n";
        print FD "=" x 10, $clock, "=" x 10, "\n";
        print FD $msg, "\n";
        close(FD);
        return 1;
    } else {
        return 0;
    }
} ;# sub pam_logger($$$$)

sub get_build_info($) {
    my $_buff = shift;
    my @buff = @$_buff;

    my $version;
    my ($ws, $buildHost, $buildDate, $efr);
    foreach my $line (@buff) {
        $line =~ s/[\r\n]*//g;
        if ( $line =~ /Workspace/ ) {
            ($ws = $line) =~ s/^.*Workspace\s+[=:]\s+//g;
            $ws =~ s/^\s+//;
            $ws =~ s/\s+$//;
            $version->{ws} = $ws if ($ws =~ /\w+/);
        }
        #Build Host   : sjc-ads-261
        #Host = sjc-ads-124
        if ( $line =~ /Host/i ) {
            ($buildHost = $line) =~ s/^.*Host(name)?\s+[=:]\s+//gi;
            $version->{buildHost} = $buildHost if ($buildHost =~ /\w+/);
        }
        if ( $line =~ /^\s*(Date|Buil[td] +on)/i ) {
            my $buildDate = $line;
            #Date: Fri Apr  4 03:36:54 PDT 2014
            $buildDate =~ s/^\s*(Date|Buil[td] +on)\s*[=:]\s+//gi;
            $version->{buildDate} = $buildDate if ($buildDate =~ /\w+/);
        }
        if ( $line =~ /((Lineup|Devline).*EFR)|\S+\s+EFR\-\d+(\s+Lineup)*/i ) {
            next if ((defined($version->{efr})) && ($version->{efr} =~ /\d+/));
            my $efr = $line;
            $efr =~ s/.*EFR\-//;
            $efr =~ s/\s+Lineup\s*$//;
            $efr =~ s/\s+Project\s*$//;
            $version->{efr} = $efr if ($efr =~ /\d+/);
        }
    }
    return $version;
} ;# sub get_build_info($)

########################################################
#get build_info and ddts from core.txt
########################################################
sub get_ws_and_ddts_from_core_txt ($) {
    my $core_txt = shift;

    my %isSeen;
    my $info;
    my @ddtsList = ();
    my @libraries = ();
    if (!open(FD, "$core_txt")) {
        close(FD);
        $info->{full_bin_path} = "";
        $info->{ddtsList} = \@ddtsList;
        $info->{libraries} = \@libraries;
        return $info;
    }

    #1) find full path of SMU/libraries
    # Mapping information
    # 2568:   /opt/cisco/calvados/sbin/sfe_driver
    # 6040:   bgp

    my $pattern = '[0-9a-f]{16}\s+.*\s+(\/\S+\.(CSC[a-z]{2}\d{5}).*\.so\.\S+)';

    #2) find all SMU installed
    # Calvados
    # ncs6k-sysadmin-ncs6k-5.2.4.CSCuu95426.all-1.0.0-SMU.x86_64
    # ncs6k-sysadmin-system-5.2.4.CSCuu91901.rp-1.0.0-SMU.x86_64
    # ncs6k-sysadmin-mgbl-5.2.4.CSCuv06711.rp-1.0.0-SMU.x86_64
    # XR
    # iosxr-mpls-5.2.4.CSCur86499.rp-1.0.0-SMU.x86_64
    # iosxr-infra-5.2.4.CSCuv14133.rp_lc-1.0.0-SMU.x86_64
    # iosxr-routing-5.2.4.CSCuv02783.rp_lc-1.0.0-SMU.x86_64

    my $xr_delimiter = 'XR Information';
    my $calv_delimiter = 'Calvados Information';
    my $host_delimiter = 'Thirdparty Information';

    my ($_pid, $_procName, $_core);
    my $start_mapping = 0;
    while (my $line = <FD>) {
        $line =~ s/[\r\n]//g;

        if ($line =~ /$xr_delimiter/) {
            my @buff = ($line);
            $line = <FD>;
            if ($line =~ /\w+/) {
                push @buff, $line;
            }
            do {
                $line = <FD>;
                push @buff, $line;
            } while ($line =~ /\w+/);
            my $version=get_build_info(\@buff);
            $info->{build_info}->{xr} = $version;
            next;
        }
        if ($line =~ /$calv_delimiter/) {
            my @buff = ($line);
            $line = <FD>;
            if ($line =~ /\w+/) {
                push @buff, $line;
            }
            do {
                $line = <FD>;
                push @buff, $line;
            } while ($line =~ /\w+/);
            my $version=get_build_info(\@buff);
            $info->{build_info}->{calvados} = $version;
            next;
        }
        if ($line =~ /$host_delimiter/) {
            my @buff = ($line);
            $line = <FD>;
            if ($line =~ /\w+/) {
                push @buff, $line;
            }
            do {
                $line = <FD>;
                push @buff, $line;
            } while ($line =~ /\w+/);
            my $version=get_build_info(\@buff);
            $info->{build_info}->{host} = $version;
            next;
        }

        #pid:
        #Core for pid = 2568 (SFE_Main_Thread)
        if ( $line =~ /Core\s+for\s+pid\s*[:=]?\s+(\S+)/) {
            $_pid = $1;
            next;
        }
        #core:
        if ( $line =~ /^Core\s+for\s+process\s*[:=]?\s+(\S+)/) {
            $_core = $1;
            $info->{coreName} = $_core;
            next;
        }
        #Core was generated by `/opt/cisco/calvados/sbin/sfe_driver'.
        #process path (not necessarily the final path):
        if ( $line =~ /Core was generated by\s+(\S+)/) {
            $_procName = $1;
            $_procName =~ s/\.$//;
            $_procName =~ s/'//g;
            $_procName =~ s/`//g;
            $_procName = basename($_procName);
            $info->{procName} = $_procName;
            next;
        }
        if ( $line =~ /Mapping\s+information/) {
            $start_mapping = 1 if ($_core =~ /${_procName}(\.|_)${_pid}/);
            next;
        }
        if ($start_mapping) {
            if ($line =~ /\s+(\/\S+\.(CSC[a-z]{2}\d{5}).*\/${_procName})$/) {
                my $bin_full_path = $1;
                my $ddts = $2;
                if (!$isSeen{$ddts}) {
                    $isSeen{$ddts} = 1;
                    push @ddtsList, $ddts;
                }
                #$info->{$ddts}->{smu_version} = $smu_version;
                $info->{$ddts}->{bin_path} = $bin_full_path;
                next;
            }
            if ( $line =~ /$pattern/) {
                my $lib_full_path = $1;
                my $ddts = $2;
                if (!$isSeen{$ddts}) {
                    $isSeen{$ddts} = 1;
                    push @ddtsList, $ddts;
                }
                #$info->{$ddts}->{smu_version} = $smu_version;
                if (!$isSeen{$lib_full_path}) {
                    $isSeen{$lib_full_path} = 1;
                    push @libraries, $lib_full_path;
                    my $lib = basename ($lib_full_path);
                    if (!defined($info->{$ddts}->{libs})) {
                        $info->{$ddts}->{libs} = $lib;
                    } else {
                        $info->{$ddts}->{libs} .= ":" . $lib;
                    }
                }
                next;
            }
        }
    }
    close(FD);
    $info->{libraries} = \@libraries;
    $info->{ddtsList} = \@ddtsList;
    return $info;
} ;# sub get_ws_and_ddts_from_core_txt ($)

sub get_remote_cmd_proc_name($$) {
    my $ip = shift;
    my $pid = shift;

    my $ssh_o = " -q -o UserKnownHostsFile=/dev/null";
    $ssh_o .= " -o StrictHostKeyChecking=no";
    my $cmd = "test -f /proc/${pid}/cmdline && cat /proc/${pid}/cmdline";
    my $cmdline = `/usr/bin/ssh $ssh_o $ip "$cmd" 2>/dev/null`;
    my $proc = "";
    if ( (defined($cmdline)) && ($cmdline =~ /\w+/) ) {
        #$proc = (split(/\s+/, $cmdline))[0];
        $proc = (split(/\0/, $cmdline))[0];
        $proc =~ s/:$//;
        $proc =~ s/^\-//;
        #PAM-xrv9k-memory_leak-0_RP0_CPU0-ema_server_sdr
        $proc =~ s/\0//g;
        $proc = basename($proc);
        $proc =~ s/\s+$//;
    }
    return $proc;
}

sub get_remote_card_arch($) {
    my $ip = shift;
    my $ssh_o = " -q -o UserKnownHostsFile=/dev/null";
    $ssh_o .= " -o StrictHostKeyChecking=no";
    my $arch = `/usr/bin/ssh $ssh_o $ip "/bin/uname -m" 2>/dev/null`;
    $arch =~ s/[\r\n]//g;
    return $arch;
}

sub do_xr_exec_cmd($) {
    my $command = shift;
    my $session = new Expect;

    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $tmp_path = "/opt/pam/";
    my $cmd = "export LD_LIBRARY_PATH=/pkg/lib:/pkg/lib/cerrno:";
    $cmd .= "/pkg/lib/mib:/pkg/lib/spp_plugins; ";
    $cmd .= $netns_cmd . "/pkg/sbin/exec -a";
    $session->log_stdout(0);
    $session->log_user(0);
    $session->log_file("$tmp_path/exec_output.txt", "w");
    $session->spawn($cmd);
    my @array = $session->expect(60, -re, ".*#");

    print $session "$command\r";
    # timeout = 600 seocnds to handle show tech support
    @array = $session->expect(600, -re, ".*#");
    $session->clear_accum;

    return $array[3];
    #return (split(/\n/, $array[3]));
}

sub get_xr_pid_info($$) {
    my $process = shift;
    my $location = shift;

    $location =~ s/_/\//g;

    my $node_id = get_slot_id($location);
    my $proc_info;
    my $_sys_info = &getOsType();
    my $netns_cmd = get_netns_env($_sys_info);
    my $output = `$netns_cmd /pkg/sbin/sysmgr_show -o -p $process -n $node_id`;

    foreach my $line (split(/\n/, $output)) {
        if ($line =~ /Job Id: (\d+)/) {
            $proc_info->{jid} = $1;
        }
        if ($line =~ /PID: (\d+)/) {
            $proc_info->{pid} = $1;
        }
        if ($line =~ /Executable path: (\S+)/) {
            $proc_info->{exe_path} = $1;
        }
        if ($line =~ /Last started: (.*)/) {
            $proc_info->{last_started} = $1;
        }
        if ($line =~ /Process state: (.*)/) {
            $proc_info->{state} = $1;
        }
    }
    return $proc_info;
} ;# sub get_xr_pid_info($$)

sub get_calvados_pid_info($$) {
    my $process = shift;
    my $location = shift;

    $location =~ s/_/\//g;
    $location =~ s/\/CPU\d+//;
    $location =~ s/\/VM\d+//;

    my $proc_info;
    my $cmd = "show process $process location $location";
    my $output = confd_cli_wrapper_exe($cmd);
    foreach my $line (split(/\n/, $output)) {
        if ($line =~ /PID: (\d+)/) {
            $proc_info->{pid} = $1;
        }
        if ($line =~ /Executable path: (\S+)/) {
            $proc_info->{exe_path} = $1;
        }
        if ($line =~ /Last started: (.*)/) {
            $proc_info->{last_started} = $1;
        }
        if ($line =~ /Process state: (.*)/) {
            $proc_info->{state} = $1;
        }
    }
    return $proc_info;
} ;# sub get_calvados_pid_info($$)

# patterns,commands,mode
sub get_dec_data($$$$$$) {
    my $platform = shift;
    my $osType = shift;
    my $boardtype = shift;
    my $event_type = shift;
    my $process = shift;
    my $dec_log = shift || "/harddisk\:/cisco_support/dec.json";

    my $commands;
    if (!open(FD, "$dec_log")) {
        return $commands;
    }
    if ( (!defined($platform)) || ($platform !~ /\w+/) ) {
        $platform = "all";
    }
    if ( (!defined($osType)) || ($osType !~ /\w+/) ) {
        $osType = "all";
    }
    if ( (!defined($process)) || ($process !~ /\w+/) ) {
        $process = "all";
    }
    my $dec;
    while (my $line = <FD>) {
       $line =~ s/[\r]//;
       $dec = decode_json $line;
       if (defined($dec->{$platform}->{$osType}->{$event_type}->{$process}->{commands})) {
           $commands = $dec->{$platform}->{$osType}->{$event_type}->{$process}->{commands};
           last;
       }
    }
    close(FD);
    return $commands;
} ;#sub get_dec_data()

#get default commands:
sub get_default_commands($$) {
    my $event_type = shift;
    my $osType = shift;

    my %isCmdSeen;
    #TODO - add Calvados support (all commands are XR specific) 
    #Default commands:
    my @shell_commands = ('show_logging',
                          'ng_show_version',
                          'sdr_instcmd show install active',
                          'show_platform_sysdb');
    my @commands = ('show logging',
                    'show version',
                    'show install active',
                    'show platform');

    #TODO!!!
    # support 'describe' from shell to find out shell equivalent commands
    # show processes memory location 0/RP0/CPU0
    # sh_proc_memory_sort "sh_proc_mem_cli -l 4096"
    # NOTE: when converted XR CLI to shell, <pid> can be hex or dedimal.
    # if pid is in hex, use 0x<pid> (under 'shell_commands')
    # if pid is in decimal, use <pid>
    # To verify, use describe, i.e., 
    # 'describe show dll pid 12345'
    #        -> 'show_dll -p 0x3039' <-- hex
    # 'describe follow process 12345 location 0/RP0/CPU0'
    #        -> 'attach_process -p 12345 -n 4096' <-- decimal
    my $default_commands = {
        'all' => {
             'crash' => {
                 'shell_commands' => [@shell_commands],
                 'commands' => [@commands],
             },
             'memory_leak' => {
                 'shell_commands' => ["show_logging",
                                      "ng_show_version",
                                      "<ssh> smap /proc/*",
                                      "<ssh> free -m"],
                 'commands' => ["show logging",
                                "ng_show_version",
                                "run <ssh> smap /proc/*",
                                "run <ssh> free -m"],
             },
             'cpu_hog' => {
                 'shell_commands' => ["ng_show_version",
                                      "show_platform_sysdb",
                                      "sdr_instcmd show install active",
                                      "<ssh> /usr/bin/pstack <pid>"],
                 'commands' => ["show version",
                                "show platform",
                                "show install active",
                                "run <ssh> /usr/bin/pstack <pid>"],
             },
             'disk_usage' => {
                 'shell_commands' => ["show_logging",
                                      "ng_show_version",
                                      "show_platform_sysdb",
                                      "sdr_instcmd show install active"],
                 'commands' => ["show logging",
                                "show version",
                                "show platform",
                                "show install active"],
             },
        },
        'xr' => {
             'memory_leak' => {
                 'shell_commands' => ["corehelper_gen -o running -p <process> -n <location>",
                                      "ng_show_version",
                                      "show_platform_sysdb",
                                      "sdr_instcmd show install active",
                                      "show_memory_snapshots -p <process> -n <node>",
                                      "show_memory_ng -n all -s"],
                 'commands' => ["dumpcore running <process> location <location>",
                                "show version",
                                "show platform",
                                "show install active",
                                "show memory-snapshots process <process> location <node>",
                                "show memory summary location all"],
             },
             'cpu_hog' => {
                 'shell_commands' => ["<ssh> top -H -b -n 1 <location>",
                                      "ng_show_version",
                                      "show_platform_sysdb",
                                      "sdr_instcmd show install active",
                                      "attach_process -p <pid> -i 5 -n <location>"],
                 'commands' => ["top -H -b -n 1",
                                "show version",
                                "show platform",
                                "show install active",
                                "follow process <pid> iteration 5 location <location>"],
             },
             'traceback' => {
                 'shell_commands' => ["show_dll -p 0x<pid> -n <location>",
                                      "ng_show_version",
                                      "show_platform_sysdb",
                                      "sdr_instcmd show install active",
                                      "show_logging"],
                 'commands' => ["show dll pid <pid> location <location>",
                                "show version",
                                "show install active",
                                "show platform",
                                "show logging"],
             },
        },
        'calvados' => {
             'memory_leak' => {
                 'commands' => ["dumpcore running <process> location <location>",
                                "show version",
                                "show platform",
                                "show install active",
                                "show memory summary location all"],
             },
             'cpu_hog' => {
                 'commands' => ["run <ssh> top -H -b -n 1"],
             },
             'disk_usage' => {
                 'commands' => ["show disk_status"],
             },
        },
    };

    #desc show dll pid 1234 location 0/RP0/CPU0
    #show_dll -p 0x4d2 -n 4096

    #desc dumpcore running bg location 0/RSP0/CPU0
    #corehelper_gen -o running -p <process> -n get_slot_id($node)

    # show_tech_fast -r -c cfgmgr -m cfgmgr_show_tech_support_fast
    # show_tech_fast -r -c issu -m issu_show_tech 
    # show_tech_fast -r -c install -m install_show_tech_support_fast2 

    my $cmd_info;
    my @cmd_types = ('commands', 'shell_commands');
    foreach my $cmd_type (@cmd_types) {
        my @allCommands = ();
        foreach my $_platform ('all', $osType) {
            my $event_keys = $$default_commands{$_platform};
            my $_mode = "exec";
            if ($osType =~ /sysadmin|calv/i) {
                $_mode = "admin";
            }
            foreach my $_cmd (@{$$event_keys{$event_type}{$cmd_type}}) {
                my $_cmd_mode = $_cmd . "," . $_mode;
                if (!$isCmdSeen{$_cmd_mode}) {
                    push @allCommands, $_cmd_mode;
                    $isCmdSeen{$_cmd_mode} = 1;
                }
            }
        };# foreach my $_platform ('all', $osType)
        $cmd_info->{$cmd_type}->{allCommands} = \@allCommands;
    };# foreach my $cmd_type (@cmd_types)
    return $cmd_info;
    #return @allCommands;
} ;# sub get_default_commands($$)

sub get_edcd_event_commands($$$$$) {
    my $platform = shift;
    my $osType = shift;
    my $event_type = shift;
    my $process = shift;
    #my $dec_log = shift || "/opt/cisco/pam/pam_event.json";
    my $_dec_logs = shift;
    my @dec_logs = @$_dec_logs;

    my $cmd_info;
    my @cmd_types = ('commands', 'shell_commands');
    foreach my $cmd_type (@cmd_types) {
        my @commandList = ();
        $cmd_info->{$cmd_type}->{commandList} = \@commandList;
    }
    if ($event_type =~ /memory/i) {
        $event_type = "MEMLEAK";
    } elsif ($event_type =~ /traceback/i) {
        $event_type = "TRACEBACK";
    } elsif ($event_type =~ /crash/i) {
        $event_type = "CRASH";
    } elsif ($event_type =~ /CPU.*HOG/i) {
        $event_type = "CPUHOG";
    }
    if ($event_type !~ /^(TRACEBACK|CPUHOG|MEMLEAK|CRASH)$/ ) {
        return $cmd_info;
    }

    if ($platform =~ /xrv9k|sunstone/i) {
        $platform = "Sunstone";
    } elsif ($platform =~ /panini|ncs6k|xboard/i) {
        $platform = "Panini";
    } elsif ($platform =~ /scapa|ncs4k/i) {
        $platform = "Scapa";
    } elsif ($platform =~ /skywarp|ncs5k/i) {
        $platform = "Skywarp";
    } elsif ($platform =~ /iosxrwbd/i) {
        $platform = "iosxrwbd";
    } elsif ($platform =~ /iosxrwb/i) {
        $platform = "iosxrwb";
    } elsif ($platform =~ /fretta|ncs5500|ncs540/i) {
        $platform = "Fretta";
    } elsif ($platform =~ /asr9k/i) {
        $platform = "ASR9K";
    } elsif ($platform =~ /mystique|rosco|ncs1k/i) {
        $platform = "ncs1k";
    } elsif ($platform =~ /ncs560|uea/i) {
        $platform = "ncs560";
    }
    if ($osType =~ /xr/i) {
        $osType = "XR";
    } elsif ($osType =~ /calvados|sysadmin/i) {
        $osType = "Calvados";
    } elsif ($osType =~ /all/i) {
        $osType = "All";
    }
    my @platformList = ('All');
    my @osTypeList = ('All');
    my @processList = ('All');
    my (%is_plat, %is_os);
    $is_plat{All} = 1;
    $is_os{All} = 1;
    if ( (defined($platform)) && ($platform =~ /\w+/) ) {
        if (!$is_plat{$platform}) {
            $is_plat{$platform} = 1;
            push @platformList, $platform,
        }
    }
    if ( (defined($osType)) && ($osType =~ /\w+/) ) {
        if (!$is_os{$osType}) {
            $is_os{$osType} = 1;
            push @osTypeList, $osType,
        }
    }
    if ( (defined($process)) && ($process =~ /\w+/) ) {
        push @processList, $process,
    }
    my %isSeen;
    foreach my $dec_log (@dec_logs) {
        if (!open(FD, "$dec_log")) {
            #return $cmd_info;
            next;
        }
        my $output = "";
        while (my $line = <FD>) {
            $line =~ s/[\r\n]//g;
            $output .= $line;
        }
        close(FD);
        #use Data::Dumper;
        my $dec = decode_json $output;
        foreach my $cmd_type (@cmd_types) {
            my @commandList = ();
            if (defined($cmd_info->{$cmd_type}->{commandList})) {
                @commandList = @{$cmd_info->{$cmd_type}->{commandList}};
            }
            foreach my $_platform (@platformList) {
                foreach my $_osType (@osTypeList) {
                    foreach my $_process (@processList) {
                        my $base =
                   $dec->{$_platform}->{$_osType}->{$event_type}->{$_process};
                        next if (!defined($base->{$cmd_type}));
                        my $commands = $base->{$cmd_type};
                        foreach my $cmd (split(/\n/, $commands)) {
                            next if ($cmd !~ /\w+/);
                            if (!$isSeen{$cmd_type}{$cmd}) {
                                $isSeen{$cmd_type}{$cmd} = 1;
                                if ($cmd =~ /show.*tech/i) {
                                   $cmd =~ s/ \(/ \"\(/g;
                                   $cmd =~ s/\) /\)\" /g;
                                }
                                push @commandList, $cmd;
                            }
                        }
                        #last;
                    };# foreach my $_process (@processList)
                };# foreach my $_osType (@osTypeList)
            };# foreach my $_platform (@platformList)
            $cmd_info->{$cmd_type}->{commandList} = \@commandList;
        };# foreach my $cmd_type (@cmd_types)
    }
    return $cmd_info;
} ;# sub get_edcd_event_commands($$$$$)

sub get_edcd_pattern_commands($$$) {
    my $platform = shift;
    my $osType = shift;
    #my $dec_log = shift || "/opt/cisco/pam/pam_pattern.json";
    my $_dec_logs = shift;
    my @dec_logs = @$_dec_logs;

    my $pat_cmd_maps;
    my @commandList = ();
    my @platformList = ('All');
    my @osTypeList = ('All');
    my (%is_plat, %is_os);
    $is_plat{All} = 1;
    $is_os{All} = 1;
    if ($platform =~ /xrv9k|sunstone/i) {
        $platform = "Sunstone";
    } elsif ($platform =~ /panini|ncs6k|xboard/i) {
        $platform = "Panini";
    } elsif ($platform =~ /scapa|ncs4k/i) {
        $platform = "Scapa";
    } elsif ($platform =~ /skywarp|ncs5k/i) {
        $platform = "Skywarp";
    } elsif ($platform =~ /iosxrwbd/i) {
        $platform = "iosxrwbd";
    } elsif ($platform =~ /iosxrwb/i) {
        $platform = "iosxrwb";
    } elsif ($platform =~ /fretta|ncs5500|ncs540/i) {
        $platform = "Fretta";
    } elsif ($platform =~ /asr9k/i) {
        $platform = "ASR9K";
    } elsif ($platform =~ /mystique|rosco|ncs1k/i) {
        $platform = "ncs1k";
    } elsif ($platform =~ /ncs560|uea/i) {
        $platform = "ncs560";
    } elsif ($platform =~ /all/i) {
        $platform = "All";
    }
    if ($osType =~ /xr/i) {
        $osType = "XR";
    } elsif ($osType =~ /calvados|sysadmin/i) {
        $osType = "Calvados";
    } elsif ($osType =~ /all/i) {
        $osType = "All";
    }
    if ( (defined($platform)) && ($platform =~ /\w+/) ) {
        if (!$is_plat{$platform}) {
            $is_plat{$platform} = 1;
            push @platformList, $platform,
        }
    }
    if ( (defined($osType)) && ($osType =~ /\w+/) ) {
        if (!$is_os{$osType}) {
            $is_os{$osType} = 1;
            push @osTypeList, $osType,
        }
    }
    my @cmd_types = ('commands', 'shell_commands');
    foreach my $cmd_type (@cmd_types) {
        my @patterns = ();
        $pat_cmd_maps->{$cmd_type}->{patterns} = \@patterns;
    }
    my %isPatSeen;
    foreach my $dec_log (@dec_logs) {
        if (!open(FD, "$dec_log")) {
            #return $pat_cmd_maps;
            next;
        }
        my $output = "";
        while (my $line = <FD>) {
            $line =~ s/[\r\n]//g;
            $output .= $line;
        }
        close(FD);
        my $dec = decode_json $output;
        foreach my $cmd_type (@cmd_types) {
            my @patterns = ();
            if (defined($pat_cmd_maps->{$cmd_type}->{patterns})) {
                @patterns = @{$pat_cmd_maps->{$cmd_type}->{patterns}};
            }
            foreach my $_platform (@platformList) {
                foreach my $_osType (@osTypeList) {
                    #Get all patterns:
                    next if (!defined($dec->{$_platform}->{$_osType}));
                    my $pat_key = $dec->{$_platform}->{$_osType};
                    my @_pat_keys = keys %{$pat_key};
                    foreach my $_pat (@_pat_keys) {
                        next if (!defined($pat_key->{$_pat}->{$cmd_type}));
                        next if ($isPatSeen{$cmd_type}{$_pat});
                        $isPatSeen{$cmd_type}{$_pat} = 1;
                        push @patterns, $_pat;
                        my @commandList = ();
                        if (defined($pat_cmd_maps->{$cmd_type}->{$_pat})) {
                            @commandList =
                               @{$pat_cmd_maps->{$cmd_type}->{$_pat}};
                        }
                        foreach my $cmd (split(/\n/,
                                   $pat_key->{$_pat}->{$cmd_type})) {
                            next if ($cmd !~ /\w+/);
                            if ($cmd =~ /show.*tech/i) {
                                $cmd =~ s/ \(/ \"\(/g;
                                $cmd =~ s/\) /\)\" /g;
                            }
                            push @commandList, $cmd;
                        }
                        $pat_cmd_maps->{$cmd_type}->{$_pat} = \@commandList;
                    } ;# foreach my $_pat (@_pat_keys)
                } ;# foreach my $_osType (@osTypeList)
            } ;# foreach my $_platform (@platformList)
            $pat_cmd_maps->{$cmd_type}->{patterns} = \@patterns;
        } ;# foreach my $cmd_type (@cmd_types)
    } ;# foreach my $dec_log (@dec_logs)
    return $pat_cmd_maps;
} ;# sub get_edcd_pattern_commands($$$$$)

sub get_user_pattern_files($) {
    my $dir       = shift;
    my $info;
    my @pattern_files = ();
    my @event_files = ();
    if (opendir(DIR, $dir)) {
        my @files = readdir(DIR);
        foreach my $file (@files) {
            if ($file =~ /pattern\.json$/i) {
                 push @pattern_files, "$dir/$file";
            } elsif ($file =~ /event\.json$/i) {
                 push @event_files, "$dir/$file";
            } 
        }
        closedir(DIR);
    } 
    $info->{pattern_files} = \@pattern_files;
    $info->{event_files} = \@event_files;
    return $info;
}

sub connect_to_cli_proxy_bulk($$$$$$) {
    my $osType = shift; #"xr";
    my $boardtype = shift; #"rp";
    my $event_type = shift; #"crash";
    my $process = shift; #"bgp";
    my $location = shift; #"0/RP0/CPU0";
    my $dec_log = shift; #"/tmp/dec.json";

    $location =~ s/_/\//g; 

    my $sys_info = &getOsType();
    my $my_osType = $sys_info->{hostType};
    my $platform = $sys_info->{platform};
    my $vf1_3073_ip = $sys_info->{vf1_3073_ip};
    my $server_port = 19999;

    #get default (generic) commands:
    my %isCmdSeen;
    my @allCommands = get_default_commands($event_type, $osType);
    foreach my $cmd_mode (@allCommands) {
        if (!$isCmdSeen{$cmd_mode}) {
            $isCmdSeen{$cmd_mode} = 1;
        }
    } 
    my $_commandList = get_dec_data($platform,
                                    $osType,
                                    $boardtype,
                                    $event_type,
                                    $process,
                                    $dec_log);

    foreach my $cmd_mode (@$_commandList) {
        if (!$isCmdSeen{$cmd_mode}) {
            push @allCommands, $cmd_mode;
            $isCmdSeen{$cmd_mode} = 1;
        }
    }

    # ZMQ:
    use ZMQ::LibZMQ3;
    # Need to quote ":all" or it will complain ZMQ_PULL, ZMQ_RCVMORE
    use ZMQ::Constants qw(:all);

    my %isExecutedCmds;
    foreach my $cmd_mode (@allCommands) {
        next if ($isExecutedCmds{$cmd_mode});
        $isExecutedCmds{$cmd_mode} = 1;

        my ($cmd, $mode) = split(/,/, $cmd_mode);
        #TODO - verify all addresses:
        my $ip = $vf1_3073_ip;
        if ( $mode =~ /calva|admin/i) {
            $ip =~ s/(\d+\.\d+\.\d+)\.\d+/$1\.1/;
        } else {
            $ip =~ s/(\d+\.\d+\.\d+)\.\d+/$1\.4/;
        }
        my $context = zmq_init();
        #Non-blocking....
        my $sender = zmq_socket($context, ZMQ_NOBLOCK);
        $sender = zmq_socket($context, ZMQ_PUSH);
        my $address = "tcp://${ip}:${server_port}";
        zmq_connect($sender, $address);
        my $strings = "{$event_type,$process,$cmd,$mode,$location}";
        my $ret = zmq_send($sender, $strings, -1);
    }
} ;# sub connect_to_cli_proxy_bulk()

sub connect_to_cli_proxy($$$$$) {
    my $osType = shift; #"xr";
    my $event_type = shift; #"crash";
    my $process = shift; #"bgp";
    my $location = shift; #"0/RP0/CPU0";
    my $_allCommands = shift;
    my @allCommands = @$_allCommands;

    $location =~ s/_/\//g; 

    my $sys_info = &getOsType();
    my $vf1_3073_ip = $sys_info->{vf1_3073_ip};
    my $server_port = 19999;

    # ZMQ:
    use ZMQ::LibZMQ3;
    # Need to quote ":all" or it will complain ZMQ_PULL, ZMQ_RCVMORE
    use ZMQ::Constants qw(:all);

    my %isExecutedCmds;
    foreach my $cmd_mode (@allCommands) {
        next if ($isExecutedCmds{$cmd_mode});
        $isExecutedCmds{$cmd_mode} = 1;

        my ($cmd, $mode) = split(/,/, $cmd_mode);
        #TODO - verify all addresses:
        my $ip = $vf1_3073_ip;
        if ( $mode =~ /calva|admin/i) {
            $ip =~ s/(\d+\.\d+\.\d+)\.\d+/$1\.1/;
        } else {
            $ip =~ s/(\d+\.\d+\.\d+)\.\d+/$1\.4/;
        }
        my $context = zmq_init();
        #Non-blocking....
        my $sender = zmq_socket($context, ZMQ_NOBLOCK);
        $sender = zmq_socket($context, ZMQ_PUSH);
        my $address = "tcp://${ip}:${server_port}";
        zmq_connect($sender, $address);
        my $strings = "{$event_type,$process,$cmd,$mode,$location}";
        my $ret = zmq_send($sender, $strings, -1);
    }
} ;# sub connect_to_cli_proxy()

sub get_dec_pattern_data($$$) {
    my $platform = shift || "all";
    my $osType = shift || "all";
    #my $boardtype = shift || "all";
    my $dec_log = shift || "//tmp/dec_log.json";

    my $dec;
    my $pat_cmd_maps;
    my @patterns = ();
    if ($osType =~ /xr/i) {
        #########################
        #some defaul XR patterns
        #########################
        my @mem_patterns = (
            'HA_WD-4-MEMORY_ALARM',
            'HA_WD-4-MEMORY_LIMIT_EXCEEDED',
            'HA_WD-4-TOP_MEMORY_USERS_WARNING',
            'HA_WD-4-TOP_MEMORY_USER_WARNING',
            'HA_WD-6-MEMORY_RECOVERY_KILL_NON_PROCESSMGR',
            'HA_WD-6-TOP_MEMORY_USERS_INFO',
            'HA_WD-6-TOP_MEMORY_USER_INFO',
            'HA_WD-3-FD_CRIT_LIM',
            'HA_WD-4-FD_MAJ_LIM',
            'OS-SHMWIN-\d+-ERROR_ENCOUNTERED',
            'OS-SYSMGR-3-ERROR',
        );
        my @memory_cmd = ('show memory summary location all,exec');
        foreach my $pat (@mem_patterns) {
            $pat_cmd_maps->{$pat} = \@memory_cmd;
            push @patterns, $pat;
        }
        my @other_patterns = (
            'HA_WD-7-UNEXPECTED',
            'HA_WD-5-SYSCALL',
            'OS-LIBSYSMGR-3-SYSCALL',
            'L2-BM-4-ERR_RETRY_EXCEED_IG',
            'IP-ARP-6-INFO_DUPMAC',
            'Cannot allocate memory',
            'reboot_internal: Incomplete graceful reboot cleanup',
        );
            #'OS-SYSMGR-7-DEBUG',
            #'abnormally terminated, restart scheduled',

        my @_tmp = ();
        foreach my $pat (@other_patterns) {
            push @patterns, $pat;
            $pat_cmd_maps->{$pat} = \@_tmp;
        }
    }

    my %isPatSeen;
    if (!open(FD, "$dec_log")) {
        $pat_cmd_maps->{patterns} = \@patterns;
        return $pat_cmd_maps;
    }

    #where commands = cmd,mode
    while (my $line =<FD>) {
        $line =~ s/[\r]//g;
        next if ($line !~ /\w+/);
        $dec = decode_json $line;
        my @_keys = keys %{$dec};
    
        ############################
        #platform:
        ############################
        foreach my $key (@_keys) {
            next if ($platform ne $key);
            my $sub_dec = $dec->{$key};
            my @_sub_keys = keys %{$sub_dec};
            ############################
            #osType:
            ############################
            foreach my $_sub_key (@_sub_keys) {
                next if ($osType ne $_sub_key);
                my $sub_sub_dec = $sub_dec->{$_sub_key};
                my @_sub_sub_keys = keys %{$sub_sub_dec};
                ############################
                #pattern:
                ############################
                foreach my $_sub_sub_key (@_sub_sub_keys) {
                    my $sub_sub_sub_dec = $sub_sub_dec->{$_sub_sub_key};
                    if (defined($sub_sub_sub_dec->{'commands'})) {
                        $pat_cmd_maps->{$_sub_sub_key} =
                                         $sub_sub_sub_dec->{'commands'};
                    }
                    if (!$isPatSeen{$_sub_sub_key}) {
                        $isPatSeen{$_sub_sub_key} = 1;
                        push @patterns, $_sub_sub_key;
                    }
                } ;# foreach my $_sub_sub_key (@_sub_sub_keys)
            } ;# foreach my $_sub_key (@_sub_keys)
        } ;# foreach my $key (@_keys)
    } ;# while (my $line =<FD>)
    close(FD);
    $pat_cmd_maps->{patterns} = \@patterns;
    return $pat_cmd_maps;
} ;#sub get_dec_pattern_data

sub cli_agent_exec($$$$$) {
    my $event_type = shift;
    my $process = shift;
    my $cmd = shift;
    my $mode = shift;
    my $location = shift;

    my $cli_info;
    my @logs = ();
    $cli_info->{logs} = \@logs;
    $cli_info->{msg} = "";
    $cli_info->{rc} = 0;

    if (!defined($cmd) || ($cmd !~ /\w+/)) {
        $cli_info->{msg} = "Failed to collect logs - Empty command.";
        $cli_info->{rc} = -1;
        return $cli_info;
    }
    my $exec_info;
    my $doDumpcore = 0;
    my @oldCores = ();
    my @coreFile = ();
    #TODO - calvados:
    if ($cmd =~ /^\s*(dumpcore|corehelper_gen)\s+/i) {
        $doDumpcore++;
    }
    my $log_dir = get_PamLogDir();
    my $bucket = "monitor_agent";

    if ( ! -d $log_dir ) {
        mkdir $log_dir;
        if ( ! -d $log_dir ) {
            my $msg = "Unable to create log folder '$log_dir'.";
            $cli_info->{msg} = $msg;
            $cli_info->{rc} = -1;
            return $cli_info;
        }
    }
    my $log_dirname = dirname($log_dir);
    my $log_filename = basename($log_dir) . "/";

    my $output = "";
    ######################################
    #fixup location for calvados
    ######################################
    $location =~ s/_/\//g;
    if ($mode =~ /calv|admin/i) {
        $location =~ s/\/CPU\d+//i;
        #0/RP0/ADMIN0
        #0/5/ADMIN0
        $location =~ s/\/ADMIN\d+//;
    }
    if ($cmd =~ /\s+location(\s+\S+)*/i) {
        if ($location =~ /\d+(\/(R[S]?P|FC|SC)\d+)?(\/(CPU)?\d+)?/) {
            $cmd =~ s/location\s+\S+/location $location/i;
            $cmd =~ s/location\s+\S+/location $location/;
            $cmd =~ s/location\s*$/location $location/i;
        }
    }
    if ($cmd =~ /<process>/i) {
        $cmd =~ s/<process>/$process/i;
        #Verify if process is still running:
        my $pid_info;
        if ($mode =~ /exec|xr/i) {
            $pid_info = get_xr_pid_info($process, $location);
        } else {
            $pid_info = get_calvados_pid_info($process, $location);
        }
        if (!defined($pid_info->{pid})) {
            my $msg = "Unable to find pid for $process at $location.";
            $cli_info->{msg} = $msg;
            $cli_info->{rc} = -1;
            return $cli_info;
        }
    }

    if ($mode =~ /exec|xr/i) {
        #TODO - handle <pid|jib|location> first before calling exec:
        ######################################
        #validate pid (jid)
        ######################################
        if ( ($cmd =~ /\s+(pid|jid)/i) &&
             ($location =~ /\w+\/\w+/) &&
             ($process =~ /\w+/) ) {
            my $pid_info = get_xr_pid_info($process, $location);
            if (!defined($pid_info->{jid})) {
                my $msg = "Unable to find pid for $process at $location.";
                $cli_info->{msg} = $msg;
                $cli_info->{rc} = -1;
                return $cli_info;
            }
            if (!defined($pid_info->{pid})) {
                my $msg = "Unable to find jid for $process at $location.";
                $cli_info->{msg} = $msg;
                $cli_info->{rc} = -1;
                return $cli_info;
            }
            my $jid = $pid_info->{jid};
            my $pid = $pid_info->{pid};
            $cmd =~ s/pid\s+\S+/pid $pid/;
            $cmd =~ s/pid\s*$/pid $pid/;
            $cmd =~ s/jid\s+\S+/jid $jid/;
            $cmd =~ s/jid\s*$/jid $jid/;
        }

        my $cmd_file = $cmd;
        $cmd_file =~ s/\s+[\-]?/_/g;
        $cmd_file =~ s/\/+/_/g;
        $cmd_file =~ s/\|/_/g;
        $cmd_file =~ s/[\$\@\!\~\'\"\*%\?\(\){}\+\[\]]//g;

        if ($event_type) {
            #$log_filename .= $event_type . "-";
            if ($process) {
                $log_filename .= $process . "-";
            }
        } else {
            if ($process) {
                $log_filename .= $process . "-";
            }
        }
        $log_filename .= $cmd_file;
        my $_timestamp = get_timestamp();
        $log_filename .= "-" . $_timestamp . ".txt";
        if ($doDumpcore) {
            @oldCores = &get_LocalCoreFiles();
        }
        $exec_info = do_xr_exec_cmd_verify($cmd, $log_dirname, $log_filename);
        if ($doDumpcore) {
            my @newCores = &get_LocalCoreFiles();
            foreach my $core (@newCores) {
                if (!grep(/$core/, @oldCores)) {
                    push @coreFile, $core;
                    last;
                }
            }
        }
    } else {
        #calvados/admin:
        #$output = confd_cli_wrapper_exe($cmd);
        ######################################
        #validate pid
        ######################################
        if ( ($cmd =~ /\s+pid/i) &&
             ($location =~ /\w+\/\w+/) &&
             ($process =~ /\w+/) ) {
            my $pid_info = get_calvados_pid_info($process, $location);
            if (!defined($pid_info->{pid})) {
                my $msg = "Unable to find jid for $process at $location.";
                $cli_info->{msg} = $msg;
                $cli_info->{rc} = -1;
                return $cli_info;
            }
            my $pid = $pid_info->{pid};
            $cmd =~ s/pid\s+\S+/pid $pid/;
            $cmd =~ s/pid\s*$/pid $pid/;
        }
        my $cmd_file = $cmd;
        $cmd_file =~ s/\s+[\-]?/_/g;
        $cmd_file =~ s/\/+/_/g;
        $cmd_file =~ s/\|/_/g;
        $cmd_file =~ s/[\$\@\!\~\'\"\*%\?\(\){}\+\[\]]//g;

        if ($event_type) {
            #$log_filename .= $event_type . "-";
            if ($process) {
                $log_filename .= $process . "-";
            }
        } else {
            if ($process) {
                $log_filename .= $process . "-";
            }
        }
        $log_filename .= $cmd_file;
        my $_timestamp = get_timestamp();
        $log_filename .= "-" . $_timestamp . ".txt";
        if ($doDumpcore) {
            @oldCores = &get_LocalCoreFiles();
        }
        $exec_info = do_admin_exec_cmd_verify($cmd,
                                              $log_dirname,
                                              $log_filename);
        if ($doDumpcore) {
            my @newCores = &get_LocalCoreFiles();
            foreach my $core (@newCores) {
                if (!grep(/$core/, @oldCores)) {
                    push @coreFile, $core;
                    last;
                }
            }
        }
    }

    if ($exec_info->{rc} ne 1) {
        my $msg = "Error during executing '$cmd': $exec_info->{msg}";
        $cli_info->{msg} = $msg;
        $cli_info->{rc} = -1;
        return $cli_info;
    }
    if ($cmd =~ /^\s*(dumpcore|corehelper_gen)\s+/i) {
    }
    $cli_info->{msg} = $exec_info->{msg};
    $cli_info->{cmd} = $exec_info->{cmd};
    $cli_info->{rc} = 1;
    if ($doDumpcore) {
        if (@coreFile) {
            $cli_info->{log_files} = \@coreFile;
        }
    } else {
        $cli_info->{log_files} = $exec_info->{log_files};
    }
    return $cli_info;
} ;# sub cli_agent_exec($$$$$)

sub normalize($) {
    my $cmd = shift;
    $cmd =~ s/-p\s+0x[0-9a-f]+/\-p 0x<pid>/g;
    $cmd =~ s/-j\s+0x[0-9a-f]+/\-j 0x<jid>/g;
    #Location:
    $cmd =~ s/\d+\/(R[S]P)?\d+\/(CPU)?\d+/<location>/g;
    #$cmd =~ s/-j (0x)[0-9a-f]+/-j<location>/g;
}

#####################################
# Note: this sub handles syntax parsing, 
# including pid/jid substitution etc
# TODO - need to identify to value type of pid:
# i.e., hex, decimal, and etc
#####################################
sub cli_agent_shell($$$$$$) {
    my $sys_info   = shift;
    my $event_type = shift;
    my $process    = shift;
    my $pid        = shift;
    my $cmd        = shift;
    my $location   = shift;

    my $osType = $sys_info->{hostType};
    my $mode = "";
    if ($osType =~ /xr/i) {
        $mode = "exec";
    } elsif ($osType =~ /sysadmin|calvados/i) {
        $mode = "admin";
    }
    my $cli_info;
    my @logs = ();
    $cli_info->{logs} = \@logs;
    $cli_info->{msg} = "";
    $cli_info->{rc} = 0;

    if (!defined($cmd) || ($cmd !~ /\w+/)) {
        $cli_info->{msg} = "Failed to collect logs - Empty command.";
        $cli_info->{rc} = -1;
        return $cli_info;
    }
    my $exec_info;
    my $doDumpcore = 0;
    my @oldCores = ();
    my @coreFile = ();
    #TODO Calvados:
    if ($cmd =~ /^\s*(dumpcore|corehelper_gen)\s+/i) {
        $doDumpcore++;
    }
    my $log_dir = get_PamLogDir();
    my $bucket = "monitor_agent";

    if ( ! -d $log_dir ) {
        mkdir $log_dir;
        if ( ! -d $log_dir ) {
            my $msg = "Unable to create log folder '$log_dir'.";
            $cli_info->{msg} = $msg;
            $cli_info->{rc} = -1;
            return $cli_info;
        }
    }
    my $log_dirname = dirname($log_dir);
    my $log_filename = $log_dir . "/";

    my $output = "";
    ######################################
    #fixup location for calvados
    ######################################
    $location =~ s/_/\//g;
    if ($mode =~ /calv|admin/i) {
        $location =~ s/\/CPU\d+//i;
        #0/RP0/ADMIN0
        #0/5/ADMIN0
        $location =~ s/\/ADMIN\d+//;
    }

    my $original_cmd = $cmd;
    #check command path, for OS command, need to run ssh
    #assume if a command is found under /usr/, /bin/, or /sbin/
    #it need to be executed remotely
    my $do_ssh = 0;
    if ($cmd =~ /<ssh>/i) {
        $do_ssh = 1;
    } elsif ($cmd !~ /\s+\-n\s+/i) {
        my $_cmd = $cmd;
        $_cmd =~ s/^\s+//;
        $_cmd = (split(/\s+/, $_cmd))[0];
        #assume no duplicated name 
        foreach my $_path ('/usr/', '/bin/', '/sbin/') {
            my $name=`find $_path -name $_cmd 2>/dev/null`;
            if (-f $name) {
                $do_ssh = 1;
                last;
            }
        }
    }
    if ($do_ssh) {
        #remove <ssh> and <location>
        $cmd =~ s/<ssh>//i;
        $cmd =~ s/$location//;
        $cmd =~ s/<location>//i;
        $cmd =~ s/location//i;
        $cmd =~ s/<\s*>//i;
        #$cmd =~ s/^\s*run\s+//i;
        my $ssh_o = " -o UserKnownHostsFile=/dev/null";
        $ssh_o .= " -o StrictHostKeyChecking=no ";
        my $ssh = "/usr/bin/ssh" . $ssh_o;
        if ($mode =~ /exec|xr/i) {
            $location =~ s/_/\//g;
            my $node_info = get_rp_ip_from_sysdb();
            foreach my $node (@{$node_info->{nodeList}})  {
                if ($location eq $node) {
                    my $ip = $node_info->{$node}->{IP};
                    if (!$sys_info->{is_thinxr}) {
                        $cmd = $ssh . $ip . " \"" . $cmd . "\"";
                    }
                    last;
                }
            }
        } elsif ($mode =~ /sysadmin|calvados/i) {
            $location =~ s/_/\//g;
            my $node_info;
            if (-f "/opt/cisco/calvados/bin/pam_show_node") {
                $node_info = get_calvados_pam_show_sdr($sys_info);
            } else {
                $node_info = get_calvados_sdr(0);
            }
            foreach my $node (@{$node_info->{nodeList}})  {
                #my $_node = $node;
                #$_node =~ s/\/VM.*//; #to match naming
                if ($node =~ /^${location}/) {
                    my $ip = $node_info->{$node}->{IP};
                    $ip =~ s/\.\d+\s*$/\.1/;
                    $cmd = $ssh . $ip . " \"" . $cmd . "\"";
                    last;
                }
            }
        }
    }

    if ($cmd =~ /\s+\-n\s+<location>/i) {
        if ($location =~ /\d+(\/(R[S]?P|FC|SC)\d+)?(\/(CPU)?\d+)?/) {
            my $nodeid = get_slot_id($location);
            $cmd =~ s/<location>/$nodeid/i;
        }
    }
    if ($cmd =~ /<node>/i) {
        if ($location =~ /\d+(\/(R[S]?P|FC|SC)\d+)?(\/(CPU)?\d+)?/) {
            $cmd =~ s/<node>/$location/i;
        }
    }
    #if still see <location>, it assumes its from show tech,
    #which doesn't use -n <nodeid> convention.
    #examples:
    #show_tech_fast -r -c netflow -l 0/RP0/CPU0 -f /harddisk: - -m show_tech_netflow_fast
    #show_tech_fast -r -x (SP) -c cef -m show_tech_cef_mc -a -v4 -a -loc0/RP0/CPU0 -l 0/RP0/CPU0 -f /harddisk: -
    #TODO - check if all show tech have the same convention:
    if ($cmd =~ /\s+<location>/i) {
        my $_location = $location;
        $_location =~ s/_/\//g;
        $cmd =~ s/<location>/$_location/i;
    }

    if ($cmd =~ /<process>/i) {
        $cmd =~ s/<process>/$process/i;
        #Verify if process is still running:
        my $pid_info;
        if ($pid !~ /\d+/) {
            if ($mode =~ /exec|xr/i) {
                $pid_info = get_xr_pid_info($process, $location);
            } else {
                $pid_info = get_calvados_pid_info($process, $location);
            }
            if (!defined($pid_info->{pid})) {
                my $msg = "Unable to find pid for $process at $location.";
                $cli_info->{msg} = $msg;
                $cli_info->{rc} = -1;
                return $cli_info;
            }
        }
    }
    #normalize cmd
    #$cmd = normalize($cmd);
    if ($mode =~ /exec|xr/i) {
        #check if hex or decimal
        my $convert_to_hex = 0;
        if ($cmd =~ /\s+0x[<]?(pid|jid)/i) {
            $convert_to_hex = 1;
        }
        #TODO - handle <pid|jib|location> first before calling exec:
        ######################################
        #validate pid (jid)
        ######################################
        if ( ($cmd =~ /\s+(0x)?[<]?(pid|jid)/i) &&
             ($location =~ /\w+\/\w+/) &&
             ($process =~ /\w+/) ) {
            my $jid;
            if ($pid !~ /\d+/) {
                my $pid_info = get_xr_pid_info($process, $location);
                if (!defined($pid_info->{jid})) {
                    my $msg = "Unable to find pid for $process at $location.";
                    $cli_info->{msg} = $msg;
                    $cli_info->{rc} = -1;
                    return $cli_info;
                }
                if (!defined($pid_info->{pid})) {
                    my $msg = "Unable to find jid for $process at $location.";
                    $cli_info->{msg} = $msg;
                    $cli_info->{rc} = -1;
                    return $cli_info;
                }
                $pid = $pid_info->{pid};
                $jid = $pid_info->{jid};
            } ;# if ($pid !~ /\d+/)
            if (($cmd =~ /<jid/i) && ($pid =~ /^\d+$/)) {
                #TODO NETNS - netns_cmd
                $jid = `/pkg/bin/sh_proc_pid_jid_conversion p2j $pid`;
                $jid =~ s/[\r\n]//g;
            }
            #remove white spaces
            $pid =~ s/\s+//g if (defined($pid));
            $jid =~ s/\s+//g if (defined($jid));

            #converted to hex:
            if ($convert_to_hex) {
                if ($pid =~ /^\d+$/) {
                    $pid = sprintf("%x", $pid);
                }
                if (defined($jid) && ($jid =~ /^\d+$/)) {
                    $jid = sprintf("%x", $jid);
                }
            }
            if (defined($pid) && ($pid =~ /^[0-9a-f]+$/)) {
                $cmd =~ s/<pid>/$pid/;
            }
            if (($cmd =~ /<jid/i) &&
                defined($jid) && ($jid =~ /^[0-9a-f]+$/)) {
                $cmd =~ s/<jid>/$jid/;
            }
            if (((!defined($pid)) || ($pid !~ /^[0-9a-f]+$/)) &&
                ((!defined($jid)) || ($jid !~ /^[0-9a-f]+$/))) {
                my $msg = "Unable to find valid pid|jid (cmd=$cmd).";
                $cli_info->{msg} = $msg;
                $cli_info->{rc} = -1;
                return $cli_info;
            }
        }

        my $cmd_file = $cmd;
        #Remove leading pathname (/usr/bin/, /pkg/bin, etc):
        $cmd_file =~ s/^\s*\/(usr|pkg)\/[s]?bin\///;
        $cmd_file =~ s/\-o\s+UserKnownHostsFile=\/dev\/null//g;
        $cmd_file =~ s/\-o\s+StrictHostKeyChecking=no//g;

        $cmd_file =~ s/\s+[\-]?/_/g;
        $cmd_file =~ s/\/+/_/g;
        $cmd_file =~ s/\|/_/g;
        $cmd_file =~ s/[\$\@\!\~\'\"\*%\?\(\){}\+\[\]]//g;
        $cmd_file =~ s/\s+/_/g;
        $cmd_file =~ s/_+/_/g;
        $cmd_file =~ s/_+$//;

        if ($cmd_file =~ /<|>/) {
            my $msg = "Error: invalid file name: '$cmd_file'.";
            $cli_info->{msg} = $msg;
            $cli_info->{rc} = -1;
            return $cli_info;
        }

        if ($cmd_file =~ /ssh/) {
            $cmd_file =~ s/ssh[_]+\d+\.\d+\.\d+\.\d+//;
            $cmd_file =~ s/^_+//;
            $cmd_file =~ s/_+$//;
        }
        my $_location = $location;
        $_location =~ s/\/+/_/g;
        $log_filename .= $_location . "-";
        if ($process) {
            $log_filename .= $process . "-";
        }
        $log_filename .= $cmd_file;
        my $_timestamp = get_timestamp();
        $log_filename .= "-" . $_timestamp . ".txt";

        $log_filename =~ s/sdr_instcmd_show_inst.*tive/show_install_active/g;
        $log_filename =~ s/ng_show_version/show_version/g;

        my %isCoreSeen;
        if ($doDumpcore) {
            @oldCores = &get_LocalCoreFiles();
            foreach my $core (@oldCores) {
               $isCoreSeen{$core} = 1;
            }
        }
        my $chassis_id = "";
        if ($cmd =~ /show.*logging/i) {
            if ( (defined($location)) && ($location =~ /\w+/) ) {;
                $chassis_id = get_chassis_id_from_node($location);
            } else {
                #get chassis_id from the local noename
                $chassis_id = get_chassis_id_from_node($sys_info->{hostname});
            }
        }
        $log_filename =~ s/_+/_/g;
        $log_filename =~ s/_\-/-/g;
        $log_filename =~ s/\-_/-/g;
        $exec_info = do_xr_shell_cmd_verify($sys_info,
                                            $cmd,
                                            $log_dirname,
                                            $log_filename,
                                            $chassis_id,
                                            $event_type);
        if ($doDumpcore) {
            sleep 15;
#TODO - Copy cores to cisco_support folder
            my @newCores = &get_LocalCoreFiles();
            foreach my $core (@newCores) {
                if (!$isCoreSeen{$core}) {
                    if ($core =~ /$process/i) {
                        push @coreFile, $core;
                    }
                }
            }
        }
    } else {
        #calvados/admin:
        #$output = confd_cli_wrapper_exe($cmd);
        ######################################
        #validate pid
        ######################################
        if ( ($cmd =~ /\s+[<]?pid/i) &&
             ($location =~ /\w+\/\w+/) &&
             ($process =~ /\w+/) ) {
          if ($pid !~ /\d+/) {
            my $pid_info = get_calvados_pid_info($process, $location);
            if (!defined($pid_info->{pid})) {
                my $msg = "Unable to find jid for $process at $location.";
                $cli_info->{msg} = $msg;
                $cli_info->{rc} = -1;
                return $cli_info;
            }
            $pid = $pid_info->{pid};
          }
            $cmd =~ s/<pid>/$pid/;
        }

        my $cmd_file = $cmd;
        $cmd_file =~ s/^\s*\/(usr|pkg)\/[s]?bin\///;
        $cmd_file =~ s/ssh.*\d+\.\d+\.\d+\.\d+//;
        $cmd_file =~ s/\s+[\-]?/_/g;
        $cmd_file =~ s/\/+/_/g;
        $cmd_file =~ s/\|/_/g;
        $cmd_file =~ s/[\$\@\!\~\'\"\*%\?\(\){}\+\[\]<>]//g;

        my $_location = $location;
        $_location =~ s/\/+/_/g;
        $log_filename .= $_location . "-";
        if ($process) {
            $log_filename .= $process . "-";
        }
        $log_filename .= $cmd_file;
        my $_timestamp = get_timestamp();
        $log_filename .= "-" . $_timestamp . ".txt";
        $log_filename =~ s/_+/_/g;
        $log_filename =~ s/_\-/-/g;
        $log_filename =~ s/\-_/-/g;
        my %isCoreSeen;
        if ($doDumpcore) {
            @oldCores = &get_LocalCoreFiles();
            foreach my $core (@oldCores) {
               $isCoreSeen{$core} = 1;
            }
        }
        #TODO - create do_admin_exec_cmd_verify_new
        $exec_info = do_admin_exec_cmd_verify($cmd,
                                              $log_dirname,
                                              $log_filename);
        if ($doDumpcore) {
            sleep 5;
            my @newCores = &get_LocalCoreFiles();
            foreach my $core (@newCores) {
                if (!$isCoreSeen{$core}) {
                    if ($core =~ /$process/i) {
                        push @coreFile, $core;
                    }
                }
            }
        }
    }

    if ($exec_info->{rc} ne 1) {
        my $msg = "Error during executing '$cmd': $exec_info->{msg}";
        $cli_info->{msg} = $msg;
        $cli_info->{rc} = -1;
        return $cli_info;
    }
    # calvados should be using different API
    if ($cmd =~ /^\s*(dumpcore|corehelper_gen)\s+/i) {
    }
    $cli_info->{msg} = $exec_info->{msg};
    $cli_info->{cmd} = $exec_info->{cmd};
    $cli_info->{rc} = 1;
    if ($doDumpcore) {
        my @tmpList = ();
        my %isSeen;
        if (@coreFile) {
            foreach my $a (@coreFile) {
                next if (! -f $a);
                if (!$isSeen{$a}) {
                    $isSeen{$a} = 1;
                    push @tmpList, $a;
                }
            }
            #$cli_info->{log_files} = \@coreFile;
        }
        # add the output log???
        foreach my $a (@{$exec_info->{log_files}}) {
            next if (! -f $a);
            if (!$isSeen{$a}) {
                $isSeen{$a} = 1;
                push @tmpList, $a;
            }
        }
        $cli_info->{log_files} = \@tmpList;
    } else {
        $cli_info->{log_files} = $exec_info->{log_files};
    }
    return $cli_info;
} ;# sub cli_agent_shell($$$$$$)

#TODO - not used, cleanup
sub do_xr_exec_cmd_verify($$$) {
    my $command = shift;
    #my $location = shift;
    my $log_dirname = shift; # only for show command
    my $log_filename = shift; # only for show command

    my $exec_info;
    my @array;
    my @log_files = ();
    $exec_info->{log_files} = \@log_files;
    my $initial_timeout = 10;
    my $timeout = 600; #seocnds to handle show tech support

    #TODO - translate commands:
    #shell_cmd = get_shell_cmd($command);
    #show version -> ng_show_version
    #show install active -> sdr_instcmd show install active

    if ($command =~ /^\s*show\s+log/i) {
        # show logging filtering last 5 hours (XR)
        my $Dh = 0;
        my $Dm = -20;
        my $dateStrings = get_behind_date($Dh, $Dm);
        my ($year,$month,$day,$hour,$min,$sec) = split(/,/, $dateStrings);
        $command = "show logging start $year $month $day $hour:$min:$sec";
        #how to limit max 2000 lines as did in shell ???
        #max last 2000 lines
        #$command = "show_logging -N 0x7d0";
    }
    #RP/0/RP0/CPU0:ios#desc show logging start march 11 11:40:30
    #show_logging -o 0x2 -d 0xb -h 0xb -m 0x28 -s 0x1e

    my $command_maps = {
        'show version' => 'ng_show_version',
        'show logging' => 'show_logging',
        'show install active' => 'sdr_instcmd show install active',
        'show dll pid <pid> location <location>' =>
                                'show_dll -p <pid> -n <location>',
    };

    ########################################
    #Linux shell mode:
    ########################################
    if ($command =~ /^\s*run\s+/i) {
        $log_filename = "/misc/scratch/" . $log_filename;
        $log_filename =~ s/\/+/\//g;
        $command =~ s/^\s*run\s+//i;
        if ($command =~ /(;|^\/+bin\/+)?rm\s+\-[rRf]?/) {
            $exec_info->{rc} = 0;
            $exec_info->{msg} = "ERROR: '$command' not allowed";
            return $exec_info;
        }
        my $output = `$command`;
        if (!open(WD, ">$log_filename")) {
            $exec_info->{rc} = 0;
            $exec_info->{msg} = "ERROR: Unable to save output of";
            $exec_info->{msg} .= " '$command' to $log_filename': $!";
            return $exec_info;
        }
        print WD $output;
        close(WD);
        $exec_info->{rc} = 1;
        $exec_info->{msg} = "Output of '$command' is saved to $log_filename'";
        push @log_files, $log_filename;
        $exec_info->{log_files} = \@log_files;
        $exec_info->{cmd} = "run " . $command;
        return $exec_info;
    }

    ########################################
    #EXEC mode:
    ########################################
    use Expect;
    my $session = new Expect;
    my $re_pattern = ".*#";

    my $tmp_path = "/opt/pam/";
    mkdir $tmp_path if (! -d $tmp_path);

    $ENV{LD_LIBRARY_PATH} = "/pkg/lib:/pkg/lib/cerrno:";
    $ENV{LD_LIBRARY_PATH} .= "/pkg/lib/mib:/pkg/lib/spp_plugins";
    #TODO NETNS - netns_cmd
    my $cmd = "/pkg/sbin/exec -a";
    $session->log_stdout(0);
    $session->log_user(0);
    $session->log_file("$tmp_path/_exec.log", "w");

    $session->spawn($cmd);
    @array = $session->expect($initial_timeout, -re, $re_pattern);
    if ( (defined($array[1])) && ($array[1] =~ /TIMEOUT/) ) {
        $exec_info->{rc} = 0;
        $exec_info->{msg} = "ERROR: executing '${cmd}' TIMEOUT,";
        $exec_info->{msg} .= " Got output: '$array[3]'.";
        $session->clear_accum;
        return $exec_info;
    }

    #set term length to 0
    $cmd = "term length 0";
    print $session "$cmd\r";
    @array = $session->expect($timeout, -re, $re_pattern);
    if ( (defined($array[1])) && ($array[1] =~ /TIMEOUT/) ) {
        $exec_info->{rc} = 0;
        $exec_info->{msg} = "ERROR: executing '${cmd}' TIMEOUT,";
        $exec_info->{msg} .= " Got output: '$array[3]'.";
        $session->clear_accum;
        return $exec_info;
    }

    $cmd = $command;
    my $checkFileCreation = 0;
    if (($command =~ /^\s*show\s+/i) &&
        ($command !~ /^\s*show\s+tech/i)) {
        if ($log_dirname =~ /\/harddisk/) {
            $cmd .= " | file harddisk:/${log_filename}";
        } else {
            $cmd .= " | file $log_filename";
        }
        $checkFileCreation = 1;
    }

    #verify existence of <cr>
    print $session $cmd, " \?\r";
    @array = $session->expect($timeout, -re, $re_pattern);
    if ( (defined($array[1])) && ($array[1] =~ /TIMEOUT/) ) {
        $exec_info->{rc} = 0;
        $exec_info->{msg} = "ERROR: executing '${cmd}' TIMEOUT,";
        $exec_info->{msg} .= " Got output: '$array[3]'.";
        $session->clear_accum;
        return $exec_info;
    }
    if ( $array[3] =~ /% Invalid input detected/ ) {
        $exec_info->{rc} = 0;
        $exec_info->{msg} = "ERROR: Invalid input detected: '$array[3]'";
        $session->clear_accum;
        return $exec_info;
    }
    if ( (!grep(/^\s+<cr>\s*$/, split(/\n/, $array[3]))) ) {
        $exec_info->{rc} = 0;
        $exec_info->{msg} = "ERROR: missing <cr>. Actual output: '$array[3]'.";
        $session->clear_accum;
        return $exec_info;
    }

    print $session $cmd, "\r";
    @array = $session->expect($timeout, -re, $re_pattern);
    if ( (defined($array[1])) && ($array[1] =~ /TIMEOUT/) ) {
        $exec_info->{rc} = 0;
        $exec_info->{msg} = "ERROR: executing '${cmd}' TIMEOUT,";
        $exec_info->{msg} .= " Got output: '$array[3]'";
        $session->clear_accum;
        return $exec_info;
    }
    if ( $array[3] =~ /% Invalid input detected/ ) {
        $exec_info->{rc} = 0;
        $exec_info->{msg} = "ERROR: Invalid input detected: '$array[3]'";
        $session->clear_accum;
        return $exec_info;
    }
    $session->clear_accum;
    $exec_info->{rc} = 1;

    my $log;
    if ($log_dirname =~ /\/harddisk/) {
        $log = $log_dirname . "/" . $log_filename;
    } else {
        my $home = $ENV{HOME};
        if ($log_filename =~ /^${home}/) {
            $log_filename =~ s/${home}//;
        }
        $log = $home . "/" . $log_filename;
    }
    $log =~ s/[\r\n]//g;
    if ($command =~ /^\s*show\s+tech/i) {
        foreach my $line (split(/\n/, $array[3])) {
            if ($line =~ /output\s+available\s+at\s+(.*)/) {
                $log = $1;
                $log =~ s/[\r\n]//g;
                $log =~ s/\s*$//;
                $log =~ s/.*\s+//;
                $log =~ s/\/+/\//g;
                push @log_files, $log;
            }
        }
    } else {
        $log =~ s/\/+/\//g;
        push @log_files, $log;
    }
    $exec_info->{log_files} = \@log_files;
    $exec_info->{msg} .= $array[3];
    $exec_info->{msg} =~ s/\r//g;
    $exec_info->{cmd} = $cmd;
    $exec_info->{cmd} =~ s/\r//g;
    return $exec_info;
} ;# sub do_xr_exec_cmd_verify($$$)

# reference: /etc/rc.d/init.d/operns-functions
sub do_xr_shell_cmd_verify($$$$$$) {
    my $sys_info     = shift;
    my $command      = shift;
    my $log_dirname  = shift; # only for show command
    my $log_filename = shift; # only for show command
    my $chassis_id   = shift; # only for show command
    my $event_type   = shift; #

    my $exec_info;
    my @array;
    my @log_files = ();
    $exec_info->{log_files} = \@log_files;

    #TODO - translate commands:
    #shell_cmd = get_shell_cmd($command);
    #show install active -> sdr_instcmd show install active
    $log_filename =~ s/\s+/_/g;

    #TODO - remove as duplicated!
    $ENV{LD_LIBRARY_PATH}="/pkg/lib:/pkg/lib/cerrno:";
    $ENV{LD_LIBRARY_PATH} .= "/pkg/lib/mib:/pkg/lib/spp_plugins";

    if ($command =~ /^\s*show\s+log|show_logging/i) {
        if (0) {
        # show logging filtering last 20 mins (XR anc Calvados?)
        my $Dh = 0;
        my $Dm = -20;
        my $dateStrings = get_behind_date($Dh, $Dm);
        my ($year,$month,$day,$hour,$min,$sec) = split(/,/, $dateStrings);
        #$command = "show logging start $year $month $day $hour:$min:$sec";
        my $_y = sprintf("0x%x", $year);
        my $_o = sprintf("0x%x", $month-1);
        my $_d = sprintf("0x%x", $day);
        my $_h = sprintf("0x%x", $hour);
        my $_m = sprintf("0x%x", $min);
        my $_s = sprintf("0x%x", $sec);
        #desc show logging start 2016 may 30 01:01:31 
        #show_logging -o 0x4 -d 0x1e -h 0x1 -m 0x1 -s 0x1f -z 0x7e0 
        $command = "show_logging -o $_o -d $_d -h $_h -m $_m -s $_s -z $_y";
        }
        #max last 2000 lines
        $command = "show_logging -N 0x7d0";
        if ($chassis_id =~ /^\d+$/) {
            if (!$sys_info->{is_thinxr}) {
                #TOOOOOO 11111 - waiting for nodename
                $command .= " | egrep \"^(LC|R[S]?P)/$chassis_id/(R[S]?P)?[0-9]+/\"";
            }
        }
    }

    ########################################
    #All commands will be executed in Linux shell mode:
    ########################################
    #$log_filename = "/misc/scratch/" . $log_filename;
    $log_filename =~ s/\/+/\//g;
    if ($command =~ /[\'\"]\s*run\s+/i) {
        $command =~ s/\s*run\s+//i;
    }
    if ($command =~ /(;|^\/+bin\/+)?rm\s+\-[rRf]?/) {
        $exec_info->{rc} = 0;
        $exec_info->{msg} = "ERROR: '$command' not allowed";
        return $exec_info;
    }
    #Make sure PATHs are present:
    $ENV{PATH}="/pkg/sbin/:/pkg/bin/:/bin/:/sbin:/usr/bin:/usr/sbin";
    $ENV{PATH} .=":/usr/local/bin/";
    my $output = "";
    if ((-f "/pkg/bin/xr_cli" ) && ($command !~ /\s*\"?run\+|\/ssh\s+/)) {
        if ($command !~ /show_logging|ng_show_version|sdr_instcmd|show_platform_sysdb/) {
            $command =~ s/^(\s*\/+pkg\/+bin\/+)?xr_cli//;
            $command =~ s/^\s*[\"\']/\"/;
            $command =~ s/[\"\']\s*$/\"/;
            $command = "/pkg/bin/xr_cli \"" . $command . "\"";
            my $netns_cmd = get_xr_cli_netns_env($sys_info);
            $command = $netns_cmd . $command;
        } else {
            my $netns_cmd = get_netns_env($sys_info);
            $command = $netns_cmd . $command;
        }
    } elsif ($command =~ /\s*\"?run\s+|\/ssh\s+/) {
        my $netns_cmd = get_netns_env($sys_info);
        $command = $netns_cmd . $command;
    }
    if ($command !~ /show.*tech/i) {
        if (!open(RD, "$command |") ) {
            $exec_info->{rc} = 0;
            $exec_info->{msg} = "ERROR: Unable to execute '$command': $!";
            return $exec_info;
        }
        while(<RD>) {
            $output .= $_;
        }
        close(RD);
        if (!open(WD, ">$log_filename")) {
            $exec_info->{rc} = 0;
            $exec_info->{msg} = "ERROR: Unable to save output of";
            $exec_info->{msg} .= " '$command' to '$log_filename': $!";
            return $exec_info;
        }
        print WD $output;
    }
    #add /etc/show_version.txt
    if ($command =~ /(^\s*|xr_cli\s+)\"?show.*version/i) {
         my $show_ver_output= "";
         if (open(RD, "/etc/show_version.txt")) {
             while(my $l = <RD>) {
                 $show_ver_output .= $l;
             }
             close(RD);
             print WD "===content of /etc/show_version.txt===\n";
             print WD $show_ver_output;
         }
    }
    close(WD);

    my $platform = `cat /proc/cmdline`;
    my $showtech_info;
    if ($command =~ /show.*tech/i) {
        if (-f "/pkg/bin/xr_cli") {
            #perform async
            my $showtech_dir = $log_dirname . "/cisco_support/showtech/";
            $showtech_info = &do_xr_showtech($sys_info,
                                             $showtech_dir,
                                             $event_type,
                                             $command);
        } else {
            my $found_show_tech = 0;
            foreach my $line (split(/\n/, $output)) {
                if ($line =~ /output\s+available\s+at\s+(.*)/) {
                    my $log = $1;
                    $log =~ s/[\r\n]//g;
                    $log =~ s/\s*$//;
                    $log =~ s/.*\s+//;
                    $log =~ s/\/+/\//g;
                    push @log_files, $log;
                    $found_show_tech++;
                }
            }
            if ($found_show_tech>0) {
                unlink $log_filename if (-f $log_filename);
            }
        }
    } else {
        if ($command =~ /dumpcore|corehelper_gen/i) {
             my $pam_dumpcore_log = dirname($log_filename) . "/pam_dumpcore.log";
             if (open(WD, ">>$pam_dumpcore_log")) {
                 my $time = localtime();
                 print WD $time, " command executed: $command.\n";
                 if (-f $log_filename) {
                     unlink $log_filename;
                 }
                 close(WD);
             }
        }
        push @log_files, $log_filename if (-f $log_filename);
    }
    $exec_info->{rc} = 1;
    if (defined($showtech_info->{output_file})) {
        $exec_info->{msg} = "Showtech output is saved to " .
                              $showtech_info->{output_file};
    } else {
        $exec_info->{msg} = "Output of '$command' is saved to $log_filename'";
    }
    $exec_info->{log_files} = \@log_files;
    if ($command !~ /\s*run +/i) {
        $command = "run " . $command;
    }
    $exec_info->{cmd} = $command;
    return $exec_info;
} ;# sub do_xr_shell_cmd_verify($$$$$)

sub do_admin_exec_cmd_verify($$$) {
    my $command = shift;
    my $log_direname = shift; # only for show command
    my $log_filename = shift; # only for show command

    my $exec_info;
    my ($cmd, $_cmd, $output);
    my @log_files = ();
    $exec_info->{log_files} = \@log_files;

    #TODO - show logging filtering last N lines (Calvados)

    my $platform = `cat /proc/cmdline`;
    if ($platform =~ /vmtype=(sysadmin|calvados)/i) {
        $ENV{PATH} = "/usr/local/sbin:/usr/bin:/bin:/usr/sbin:/sbin/";
        $ENV{PATH} .= ":/opt/cisco/calvados/bin/:/opt/cisco/calvados/sbin/";
        $ENV{PATH} .= ":/usr/local/bin/";
    }
    $log_filename =~ s/^\s+//;
    $log_filename =~ s/\s+$//;
    $log_filename =~ s/\s+/_/g;

    my $log = "";
    ########################################
    #Linux shell mode:
    ########################################
    if ($command =~ /[\'\"]\s*run\s+/i) {
        $command =~ s/\s*run\s+//i;
        #$log_filename = "/misc/scratch/" . $log_filename;
        $command =~ s/\s*run\s+//i;
        if ($command =~ /(;|^\/+bin\/+)?rm\s+\-[rRf]?/) {
            $exec_info->{rc} = 0;
            $exec_info->{msg} = "ERROR: '$command' not allowed";
            return $exec_info;
        }
        my $output = `$command`;
        $log = $log_filename;
        $log =~ s/\/+/\//g;
        if (!open(WD, ">$log")) {
            $exec_info->{rc} = 0;
            $exec_info->{msg} = "ERROR: Unable to save output of";
            $exec_info->{msg} .= " '$command' to $log': $!";
            return $exec_info;
        }
        print WD $output;
        close(WD);
        $exec_info->{rc} = 1;
        $exec_info->{msg} = "Output of '$command' is saved to $log_filename'";
        $exec_info->{cmd} = "run " . $command;
        push @log_files, $log;
        $exec_info->{log_files} = \@log_files;
        return $exec_info;
    }

    $cmd = $command;
    my $checkFileCreation = 0;

    if (($command =~ /^\s*show\s+/i) &&
        ($command !~ /^\s*show\s+tech/i)) {
        $cmd .= " | save $log_filename";
        $checkFileCreation = 1;
    }

    if (0) {
    #verify existence of <cr>
    $_cmd = $cmd . " \?\r";
    $output = confd_cli_wrapper_exe($_cmd);
    if ( $output =~ /(% Invalid input detected|syntax error: )/ ) {
        $exec_info->{rc} = 0;
        $exec_info->{msg} = "ERROR: Invalid input detected: '$output'";
        return $exec_info;
    }
    if ( !grep(/^\s+<cr>\s*$/, split(/\n/, $output)) ) {
        $exec_info->{rc} = 0;
        $exec_info->{msg} = "ERROR: missing <cr>. Actual output: '$output'.";
        return $exec_info;
    }
    }
    $output = confd_cli_wrapper_exe($cmd);
    $exec_info->{rc} = 1;
    $exec_info->{msg} = "command=$cmd\n";
    $exec_info->{msg} .= $output;
    if ( ! $log_direname ) {
        my $home = $ENV{HOME};
        if ( $log_filename =~ /^${home}/) {
            $log_filename =~ s/$home//;
        }
        $log = $home . "/" . $log_filename;
    } else {
        #my $log = $log_direname . "/" . $log_filename;
        $log = $log_filename;
    }
    #add /etc/show_version.txt
    if ($command =~ /^\s*show.*version/i) {
        my $show_ver_output= "";
        if (open(RD, "/etc/show_version.txt")) {
            while(my $l = <RD>) {
                $show_ver_output .= $l;
            }
            close(RD);
        }
        if (open(WD, ">>$log_filename")) {
            print WD "===content of /etc/show_version.txt===\n";
            print WD $show_ver_output;
            close(WD);
        }
    }

    if (0) {
    if ($checkFileCreation) {
        if (!-f $log) {
            $exec_info->{rc} = -1;
            $exec_info->{msg} = "ERROR: failed to create $log";
            return $exec_info;
        }
    }
    }
    if ($command =~ /^\s*show\s+tech/i) {
        foreach my $line (split(/\n/, $output)) {
            if ($line =~ /output\s+available\s+at\s+(.*)/) {
                $log = $1;
                $log =~ s/[\r\n]//g;
                $log =~ s/\s*$//;
                $log =~ s/.*\s+//;
                $log =~ s/\/+/\//g;
                push @log_files, $log;
            }
        }
    } else {
        push @log_files, $log;
    }
    $exec_info->{log_files} = \@log_files;
    $exec_info->{cmd} = $cmd;
    $exec_info->{cmd} =~ s/\r//g;
    return $exec_info;
} ;# sub do_admin_exec_cmd_verify($$)

# this covers show logging based events (except traceback):
sub save_show_log_summary() {
    my $log_dir = shift;
    my $matchedPatterns = shift;
    my $matchedLogs = shift;
    my $_executed_commands = shift;
    my $_saved_log_files = shift;
    my $show_tech_executed = shift;
    my $show_tech_root = shift;
    my $bucket = shift;
 
    my @executed_commands = @$_executed_commands;
    my @saved_log_files = @$_saved_log_files;
 
    my $_sys_info = &getOsType();
    my $osType = $_sys_info->{hostType};
    my $platform = $_sys_info->{platform};
    my $netns_cmd = &get_netns_env($_sys_info);

    my $routername = `uname -n`;
    if ($osType =~ /xr/i) {
        $ENV{PATH}="/pkg/sbin/:/pkg/bin/:/bin/:/sbin:/usr/bin:/usr/sbin";
        $ENV{PATH} .=":/usr/local/bin/";
        $routername = `$netns_cmd /pkg/bin/node_list_generation -f MY`;
    }
    if ($platform =~ /xboard/) {
        $platform = "panini";
    }
    my $logMsg = "Event monitoring summary.\n";
    $logMsg .= "\n(Node name: " . $routername . ").\n";
    $logMsg .= "platform=$platform\n";
    if ($matchedPatterns =~ /\w+/) {
        $logMsg .= "\n" . "#" x 60;
        $matchedPatterns =~ s/[\r\n]*$//;
        $logMsg .= "\n" . "Matched alert patterns = ";
        $logMsg .= $matchedPatterns . ".\n";
        $logMsg .= "#" x 60;
    }
 
    my $debug = 0;
    my %isLogSeen;
    if ($matchedLogs =~ /\w+/) {
        $matchedLogs =~ s/[\r\n]*[\.]*$//;
        $logMsg .= "\n" . $matchedLogs . "\n";
    }
    if (@executed_commands) {
        $logMsg .= "\n" . "#" x 20 . "Commands executed";
        $logMsg .= "#" x 20 . ".\n";
        my $c = 0;
        foreach my $cmd (@executed_commands) {
            if ( (defined($cmd)) && ($cmd =~ /\w+/) ) {
                $cmd =~ s/\r//g;
                $logMsg .= "'" . $cmd . "'.\n";
            }
            my $log = $saved_log_files[$c];
            $c++;
        }
        if (@saved_log_files) {
            $logMsg .= "\n" . "#" x 20 . "Log/core files collected";
            $logMsg .= "#" x 20 . ".\n";
            foreach my $_log_file (@saved_log_files) {
                if ( $_log_file =~ /harddisk[\/]?:/ ) {
                    my $f = $_log_file;
                    $f =~ s/harddisk[\/]?:/misc\/disk1/;
                    $_log_file = $f if (-f $f);
                }
                if (!$isLogSeen{$_log_file}) {
                    $isLogSeen{$_log_file} = 1;
                    if ($_log_file =~ /core\.gz/) {
                         $logMsg .= $_log_file . "\n";
                    } else {
                         $logMsg .= $_log_file . "\n";
                    }
                }
            }
        }
    }
    my $version_content = &getVersionContent("/");
    $logMsg .= "\n" . $version_content;
 
    my $uptime = &getUptime();
    my $localtime = localtime();
    $logMsg .= "\n\n--- router localtime: $localtime.";
    $logMsg .= " system uptime: $uptime (seconds) ---\n";
 
    if ($debug) {
        print "logMsg=$logMsg\n";
    }
    my $_timestamp = get_timestamp();
    my $log_file = $log_dir . "/" . "show_log_report_summary-";
    $log_file .= $_timestamp . ".txt";
    $log_file =~ s/\/+/\//g;
    if (!open(WD, ">$log_file")) {
        my $msg = "Unable to save traceback data to $log_file: $!";
        my $_sys_info = &getOsType();
        &pam_logger($_sys_info, $log_dir, $bucket, $msg);
        my $severity = "warning";
        create_syslog($_sys_info, $msg, $severity);
        return "";
    }
    print WD $logMsg;
    close(WD);
    return $log_file;
} ;# sub save_show_log_summary()

# this covers traceback - to DETETE:
sub save_traceback_summary() {
    my $log_dir = shift;
    my $matchedPatterns = shift;
    my $matchedLogs = shift;
    my $_executed_commands = shift;
    my $_saved_log_files = shift;
    my $show_tech_executed = shift;
    my $show_tech_root = shift;
    my $bucket = shift;
 
    my @executed_commands = @$_executed_commands;
    my @saved_log_files = @$_saved_log_files;
 
    my $routername = `uname -n`;
    if ( -f "/pkg/bin/uname" ) {
        $routername = `/pkg/bin/uname -n`;
    }
    my $platform = `cat /proc/cmdline`;
    $routername =~ s/[\r\n]//g;
    $platform =~ s/.*platform=//;
    $platform =~ s/\s+.*//;
    if ($platform =~ /xboard/) {
        $platform = "panini";
    }
 
    my $logMsg = "Event monitoring summary.\n";
    $logMsg .= "\n(Node name: " . $routername . ").\n";
    $logMsg .= "platform=$platform\n";
    $logMsg .= "\n" . "#" x 60;
    $matchedPatterns =~ s/[\r\n]*$//;
    $logMsg .= "\n" . "Matched alert patterns = " . $matchedPatterns . ".\n";
    $logMsg .= "#" x 60;
 
    my $debug = 0;
    my %isLogSeen;
    $matchedLogs =~ s/[\r\n]*[\.]*$//;
    $logMsg .= "\n" . $matchedLogs . "\n";
    if (@executed_commands) {
        $logMsg .= "\n" . "#" x 20 . "Commands executed" . "#" x 20 . ".\n";
        my $c = 0;
        foreach my $cmd (@executed_commands) {
            $cmd =~ s/\r//g;
            $logMsg .= "'" . $cmd . "'.\n";
            my $log = $saved_log_files[$c];
            $c++;
        }
        if (@saved_log_files) {
            $logMsg .= "\n" . "#" x 20 . "Log/core files collected";
            $logMsg .= "#" x 20 . ".\n";
            foreach my $_log_file (@saved_log_files) {
                if ( $_log_file =~ /harddisk[\/]?:/ ) {
                    my $f = $_log_file;
                    $f =~ s/harddisk[\/]?:/misc\/disk1/;
                    $_log_file = $f if (-f $f);
                }
                if (!$isLogSeen{$_log_file}) {
                    $isLogSeen{$_log_file} = 1;
                    if ($_log_file =~ /core\.gz/) {
                         $logMsg .= $_log_file . "\n";
                    } else {
                         $logMsg .= $_log_file . "\n";
                    }
                }
            }
        }
    }
    my $version_content = &getVersionContent("/");
    $logMsg .= "\n" . $version_content;
 
    my $uptime = &getUptime();
    my $localtime = localtime();
    $logMsg .= "\n\n--- router localtime: $localtime.";
    $logMsg .= " system uptime: $uptime (seconds) ---\n";
 
    if ($debug) {
        print "logMsg=$logMsg\n";
    }
    my $_timestamp = get_timestamp();
    my $log_file = $log_dir . "/" . "show_log_report_summary-";
    $log_file .= $_timestamp . ".txt";
    $log_file =~ s/\/+/\//g;
    if (!open(WD, ">$log_file")) {
        my $msg = "Unable to save traceback data to $log_file: $!";
        my $_sys_info = &getOsType();
        &pam_logger($_sys_info, $log_dir, $bucket, $msg);
        my $severity = "warning";
        create_syslog($_sys_info, $msg, $severity);
        return "";
    }
    print WD $logMsg;
    close(WD);
    return $log_file;
} ;# sub save_traceback_summary()

# this covers process based events:
# process crash, memory leaks, tracebacks
sub save_process_event_summary() {
    my $log_dir = shift;
    my $_executed_commands = shift;
    my $_saved_log_files = shift;
    my $show_tech_executed = shift;
    my $show_tech_root = shift;
    my $event_type = shift;

    my @executed_commands = @$_executed_commands;
    my @saved_log_files = @$_saved_log_files;

    my $routername = `uname -n`;
    if ( -f "/pkg/bin/uname" ) {
        $routername = `/pkg/bin/uname -n`;
    }
    my $platform = `cat /proc/cmdline`;
    $routername =~ s/[\r\n]//g;
    $platform =~ s/.*platform=//;
    $platform =~ s/\s+.*//;
    if ($platform =~ /xboard/) {
        $platform = "panini";
    }

    my $logMsg = "Event monitoring summary.\n";
    $logMsg .= "\n(Node name: " . $routername . ").\n";
    $logMsg .= "platform=$platform\n";

    my $debug = 0;
    my %isLogSeen;
    if (@executed_commands) {
        $logMsg .= "\n" . "#" x 20 . "Commands executed" . "#" x 20 . ".\n";
        my $c = 0;
        foreach my $cmd (@executed_commands) {
            next if (!defined($cmd) || ($cmd !~ /\w+/));
            $cmd =~ s/\r//g;
            $logMsg .= "'" . $cmd . "'.\n";
            my $log = $saved_log_files[$c];
            $c++;
        }
        if (@saved_log_files) {
            $logMsg .= "\n" . "#" x 20 . "Log/core files collected";
            $logMsg .= "#" x 20 . ".\n";
            foreach my $_log_file (@saved_log_files) {
                if ( $_log_file =~ /harddisk[\/]?:/ ) {
                    my $f = $_log_file;
                    $f =~ s/harddisk[\/]?:/misc\/disk1/g;
                    $_log_file = $f if (-f $f);
                }
                if (!$isLogSeen{$_log_file}) {
                    $isLogSeen{$_log_file} = 1;
                    if ($_log_file =~ /core\.gz/) {
                        $logMsg .= $_log_file . "\n";
                    } else {
                        $logMsg .= $_log_file . "\n";
                    }
                }
            }
        }
    }
    my $version_content = &getVersionContent("/");
    $logMsg .= "\n" . $version_content;

    my $uptime = &getUptime();
    my $localtime = localtime();
    $logMsg .= "\n\n--- router localtime: $localtime.";
    $logMsg .= " system uptime: $uptime (seconds) ---\n";

    if ($debug) {
        print "logMsg=$logMsg\n";
    }
    #TODO!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    my $report_summary = $event_type . "_summary";
    if ($event_type =~ /crash/i) {
        $report_summary = "crash_summary";
    } elsif ($event_type =~ /traceback/i) {
        $report_summary = "traceback_summary";
    } elsif ($event_type =~ /cpu_hog/i) {
        $report_summary = "cpu_hog_summary";
    }

    my $_timestamp = get_timestamp();
    #my $log_file = $log_dir . "/" . "log_summary-" . $_timestamp . ".txt";
    $event_type =~ s/\//_/g;
    my $log_file = $log_dir . "/" . $report_summary . "-" . $_timestamp .".txt";
    $log_file =~ s/\/+/\//g;
    if (!open(WD, ">$log_file")) {
        my $msg = "Unable to save traceback data to $log_file: $!";
        my $_sys_info = &getOsType();
        &pam_logger($_sys_info, $log_dir, $event_type, $msg);
        my $severity = "warning";
        create_syslog($_sys_info, $msg, $severity);
        return "";
    }
    print WD $logMsg;
    close(WD);
    return $log_file;
} ;# sub save_process_event_summary()

sub save_process_crash_summary() {
    my $log_dir = shift;
    my $_executed_commands = shift;
    my $_saved_log_files = shift;
    my $show_tech_executed = shift;
    my $show_tech_root = shift;
    my $_fullcores = shift;
 
    my $bucket = "crash";
 
    my @executed_commands = @$_executed_commands;
    my @saved_log_files = @$_saved_log_files;
    my @fullcores = @$_fullcores;
 
    my $routername = `uname -n`;
    if ( -f "/pkg/bin/uname" ) {
        $routername = `/pkg/bin/uname -n`;
    }
    my $platform = `cat /proc/cmdline`;
    $platform =~ s/.*platform=//;
    $platform =~ s/\s+.*//;
    if ($platform =~ /xboard/) {
        $platform = "panini";
    }
    $routername =~ s/[\r\n]//g;

    my $logMsg = "Event monitoring summary.\n";
    $logMsg .= "\n(Node name: " . $routername . ").\n";
    $logMsg .= "platform=$platform\n";
 
    my $debug = 0;
    my %isLogSeen;
    if (@executed_commands) {
        $logMsg .= "\n" . "#" x 20 . "Commands executed" . "#" x 20 . ".\n";
        my $c = 0;
        foreach my $cmd (@executed_commands) {
            $cmd =~ s/\r//g;
            $logMsg .= "'" . $cmd . "'.\n";
            my $log = $saved_log_files[$c];
            $c++;
        }
        if (@saved_log_files) {
            $logMsg .= "\n" . "#" x 20 . "Log/core files collected";
            $logMsg .= "#" x 20 . ".\n";
            foreach my $_log_file (@saved_log_files) {
                if ( $_log_file =~ /harddisk[\/]?:/ ) {
                    my $f = $_log_file;
                    $f =~ s/harddisk[\/]?:/misc\/disk1/g;
                    $_log_file = $f if (-f $f);
                }
                if (!$isLogSeen{$_log_file}) {
                    $isLogSeen{$_log_file} = 1;
                    if ($_log_file =~ /core\.gz/) {
                        $logMsg .= $_log_file . "\n";
                    } else {
                        $logMsg .= $_log_file . "\n";
                    }
                }
            }
        }
    }
    if (@fullcores) {
        $logMsg .= "\n" . "#" x 20 . "Core files" . "#" x 20 . ".\n";
        foreach my $core (@fullcores) {
            if (-f $core) {
                $logMsg .= "Core file is saved at: " . $core . "\n";
                my $core_txt = $core;
                $core_txt =~ s/\.gz$/\.txt/;
                if (-f $core_txt) {
                    $logMsg .= "(trace file is saved at: " . $core_txt . ")\n";
                }
            }
        }
    }
    my $version_content = &getVersionContent("/");
    $logMsg .= "\n" . $version_content;
 
    my $uptime = &getUptime();
    my $localtime = localtime();
    $logMsg .= "\n\n--- router localtime: $localtime. system uptime:";
    $logMsg .= " $uptime (seconds) ---\n";
 
    if ($debug) {
        print "logMsg=$logMsg\n";
    }
    my $_timestamp = get_timestamp();
    my $log_file = $log_dir . "/" . "crash_summary-" . $_timestamp . ".txt";
    $log_file =~ s/\/+/\//g;
    if (!open(WD, ">$log_file")) {
        my $msg = "Unable to save traceback data to $log_file: $!";
        my $_sys_info = &getOsType();
        &pam_logger($_sys_info, $log_dir, $bucket, $msg);
        my $severity = "warning";
        create_syslog($_sys_info, $msg, $severity);
        return "";
    }
    print WD $logMsg;
    close(WD);
    return $log_file;
} ;# sub save_process_crash_summary()

sub create_log_archive() {
    my $log_dir = shift;
    my $_saved_log_files = shift;
    my $_retained_logs = shift;
    my $pr_info = shift;

    my $showtech_cnt = $pr_info->{showtech_cnt};
    my $event_type = $pr_info->{event_type};
    my $delete_original = $pr_info->{delete_original};
    my $node = $pr_info->{node};
    if (defined($node)) {
        $node =~ s/\//_/g;
        $node =~ s/://g;
    }
    my $process = $pr_info->{procName};

    my @saved_log_files = @$_saved_log_files;
    my @retained_logs = @$_retained_logs;

    my $routername = "";
    my $_sys_info = &getOsType();
    my $osType = $_sys_info->{hostType};
    my $platform = $_sys_info->{platform};
    my $netns_cmd = &get_netns_env($_sys_info);

    my $node_prefix = "";
    if ($osType =~ /xr/i) {
        $ENV{PATH}="/pkg/sbin/:/pkg/bin/:/bin/:/sbin:/usr/bin:/usr/sbin";
        $ENV{PATH} .=":/usr/local/bin/";
        $routername = `$netns_cmd /pkg/bin/node_list_generation -f MY`;
        $node_prefix = "xr_";
    } elsif ($platform =~ /sysadmin|calvados/i) {
        $ENV{PATH} = "/usr/local/sbin:/usr/bin:/bin:/usr/sbin:/sbin/";
        $ENV{PATH} .= ":/opt/cisco/calvados/bin/:/opt/cisco/calvados/sbin/";
        $ENV{PATH} .= ":/usr/local/bin/";
        $routername = `uname -n`;
        $node_prefix = "sysadmin_";
    }
    $routername =~ s/[\r\n]//g;
    if ($platform =~ /xboard|panini/) {
        $platform = "ncs6k";
    } elsif ($platform =~ /scapa/) {
        $platform = "ncs4k";
    } elsif ($platform =~ /sunstone/) {
        $platform = "xrv9k";
    } elsif ($platform =~ /fretta/) {
        $platform = "ncs5500";
    } elsif ($platform =~ /ncs560/) {
        $platform = "ncs560";
    } elsif ($platform =~ /ncs540/) {
        $platform = "ncs540";
    } elsif ($platform =~ /skywarp/) {
        $platform = "ncs5k";
    } elsif ($platform =~ /iosxrwbd/) {
        $platform = "iosxrwbd";
    } elsif ($platform =~ /iosxrwb/) {
        $platform = "iosxrwb";
    } elsif ($platform =~ /mystique|rosco|ncs1k/i) {
        $platform = "ncs1k";
    } elsif ($platform =~ /ncs560|uea/i) {
        $platform = "ncs560";
    }
    $platform =~ s/[\r\n]//g;

    my $debug = 0;
    my %isLogSeen;

    chdir "/";
    my $_timestamp = get_timestamp();
    if ($log_dir =~ /harddisk[\/\\]*:/) {
        if (readlink("/harddisk\:")) {
            my $misc_disk1 = readlink("/harddisk\:");
            $log_dir =~ s/harddisk[\/\\]*:/$misc_disk1/;
        }
    }
    #collect show tech files
    #asusme show tech is collected in async mode 
    if ($platform !~ /panini|scapa/i) {
        if ((defined($pr_info->{showtech_cnt})) &&
                     ($pr_info->{showtech_cnt} >= 1)) {
            my $t1 = localtime();
            print "Collecting showtech data at $t1....\n";
            my $pam_showtech_dir = $log_dir . "/showtech/";
            $event_type =~ s/show_logging.*/show_logging/;
            my $showtech_info = &get_showtech_data($pam_showtech_dir, $event_type);
            if (defined($showtech_info->{log_files})) {
                print "Found ", scalar(@{$showtech_info->{log_files}}), " logs at $t1\n";
                foreach my $f (@{$showtech_info->{log_files}}) {
                    push @saved_log_files, $f;
                }
            }
        }
    }

    my $tar_name = $log_dir . "/" . "PAM-" . $platform . "-";
    if ( (defined($event_type)) && ($event_type =~ /\w+/) ) {
        $tar_name .= $event_type . "-";
    }
    if ( (defined($node)) && ($node =~ /\w+/) ) {
        $node =~ s/\s+//g;
        $tar_name .= $node_prefix . $node . "-";
    }
    if ( (defined($process)) && ($process =~ /\w+/) ) {
        $process =~ s/\s+//g;
        $tar_name .= $process . "-";
    }
    $tar_name .=  $_timestamp . ".tgz";
    chdir "/";
    $tar_name =~ s/\/+/\//g;
    my $tar_list = "";

    my $file_map;
    foreach my $_log_file (@saved_log_files) {
        $_log_file =~ s/\0//g;
        next if (! -f $_log_file);
        my $_file = $_log_file;
        if ( $_file =~ /harddisk[\/\\]*:/ ) {
            my $f = $_file;
            $f =~ s/harddisk[\/\\]*:/misc\/disk1/;
            $_file = $f if (-f $f);
        }
        $_file =~ s/^\s*\/+//;
        $_file =~ s/\0//g;
        $_file =~ s/[\n\r]//g;
        $tar_list .= " $_file";
        $file_map->{$_file} = $_log_file;
    }
    ########################################
    # adding clihistory
    ########################################
    if (opendir(DIR, "$log_dir/clihistory/")) {
        my @clifiles = readdir(DIR);
        closedir(DIR);
        foreach my $clifile (@clifiles) {
            next if ($clifile !~ /\-clihistory/);
            my $full_cli_path = $log_dir . "/clihistory/" . $clifile;
            $full_cli_path =~ s/^\s*[\/]+//;
            $full_cli_path =~ s/[\n\r]//g;
            $tar_list .= " " . $full_cli_path;
            push @retained_logs, $full_cli_path;
        }
    }
    ########################################
    # /audit_trail_logs/audit_logs_console
    ########################################
    if (opendir(DIR, "/audit_trail_logs/")) {
        my @audit_files = readdir(DIR);
        closedir(DIR);
        foreach my $file (@audit_files) {
            next if ( $file !~ /\w+/);
            $file =~ s/^\s*\/+//;
            $file =~ s/\0//g;
            my $full_path = "/audit_trail_logs/" . $file;
            $full_path =~ s/^\s*[\/]+//;
            $full_path =~ s/[\n\r]//g;
            next if (! -f $full_path);
            $tar_list .= " " . $full_path;
            push @retained_logs, $full_path;
        }
    }
    my %isRetain;
    foreach my $ret_log (@retained_logs) {
        $isRetain{$ret_log} = 1
    }
    $tar_list =~ s/[\n\r]//g;
    chdir "/";
    if ($tar_list ne "") {
        #Don't use tar v which can lead to TRUNCATED:
        # misc/disk1/cisco_support/ipv6_ea-show_version-2016May30-165319.(TRUNCATED) 
        my $tar_cmd = "/bin/tar czf $tar_name $tar_list";

        if (system("$tar_cmd")) {
            my $msg = "PAM: Unable to create tar file ($tar_name): $!";
            $msg .= "\ntar command: " . $tar_cmd;
            &pam_logger($_sys_info, $log_dir, $event_type, $msg);
            my $severity = "warning";
            create_syslog($_sys_info, $msg, $severity);
            #cleanup .txt files:
            foreach my $_log_file (@saved_log_files) {
                if ((!defined($isRetain{$_log_file})) ||
                              (!$isRetain{$_log_file})) {
                    unlink $_log_file if (-f $_log_file);
                }
            }
            return "";
        }
    }

    if ($debug) {
        print "$tar_name has been created.\n";
    }
    my $foundCore = 0;
    foreach my $_log_file (@saved_log_files) {
        if ($_log_file =~ /\.core\.gz/) {
            $foundCore++;
        }
    }
    if ($delete_original) {
        foreach my $_log_file (@saved_log_files) {
            if ( ($delete_original) &&  
                 ((!defined($isRetain{$_log_file})) ||
                          (!$isRetain{$_log_file})) ) {
                print "delete $_log_file.\n" if ($debug);
                unlink $_log_file if (-f $_log_file);
            }
        }
    }

    my $console_msg = "PAM detected";
    if (defined($event_type)) {
        #if ($event_type =~ /show_logging_patterns/i) {}
        if ($event_type =~ /show_logging/i) {
            $console_msg .= " errors in show logging";
        } elsif ($event_type =~ /cpu.*hog/i) {
            $console_msg .= " CPU hog";
        } elsif ($event_type =~ /disk_usage/i) {
            my $diskList = "";
            my $cnt = 0;
            foreach my $mount (@{$pr_info->{fullDiskList}}) {
                $diskList .= " " . $mount;
                $cnt++;
            }
            if ($cnt > 1) {
                $console_msg .= $diskList . " are full";
            } else {
                $console_msg .= $diskList . " is full";
            }
        } elsif ($event_type =~ /memory.*leak/i) {
            $console_msg .= " significant memory increase";
            if (defined($pr_info->{start_time})&&
                defined($pr_info->{stop_time}) &&
                defined($pr_info->{total_start}) &&
                defined($pr_info->{total_stop})) {
                $console_msg .= " (from " . $pr_info->{total_start} . "MB";
                $console_msg .= " at " . $pr_info->{start_time};
                $console_msg .= " to " . $pr_info->{total_stop} . "MB";
                $console_msg .= " at " . $pr_info->{stop_time} . ")";
            }
        } else {
            $console_msg .= " " . $event_type;
        }
    }
    if ((defined($process)) && ($process =~ /\w+/)) {
        $console_msg .= " for " . $process;
    }
    if ((defined($node)) && ($node =~ /\w+/)) {
        $console_msg .= " on " . $node;
    }
    if ($event_type =~ /disk_usage/i) {
        $console_msg .= " (please clean up to avoid any fault caused by this)";
    }
    $console_msg .= "." . " All necessary files for debug have been";
    $console_msg .= " collected and saved at";
    $tar_name =~ s/[\/]*misc\/+disk1/harddisk:/;
    $console_msg .= " " . $routername . " : " . $tar_name;
    $console_msg .=" (Please copy tgz file out of the router and send to Cisco";
    $console_msg .=" support. This tgz file will be removed after 14 days.)";

    # resmon[321]: %HA-HA_WD-2-DISK_CRIT : A monitored device
    # /misc/config/config/lr ( (null) ) is above 99% utilization. 
    # Current utilization = 100. Please remove unwanted user files
    # and configuration rollback points. 

    my $severity = "warning";
    if ( (defined($event_type)) && ($event_type =~ /disk/i) ) {
        $severity = "alert",
    }
    create_syslog($_sys_info, $console_msg, $severity);
    if (defined($event_type) && ($event_type =~ /memory.*leak/i)) {
         $console_msg = "Before contacting Cisco for support, please run the ";
         $console_msg .= "following command to confirm if the leak is genuine ";
         $console_msg .= "(ignore the above message if memory stops increasing):\n";
         my $location = $node;
         $location =~ s/^node//;
         $location =~ s/_/\//g;
         if ($location =~ /^d+\/(R[S]?P|SC|LC|CB|CC)*\d+\/(CPU)*\d+/) {
             $console_msg .="show memory-snapshots process $process location $location\n";
         } else {
             $console_msg .="show memory-snapshots process $process location <location>\n";
         }
         create_syslog($_sys_info, $console_msg, $severity);
    }

    if ($foundCore > 0) {
        #drop cache
        system("sysctl vm.drop_caches=3 >/dev/null");
    }
    return $tar_name;
} ;# sub create_log_archive()

sub get_timestamp() {
    my $month_map = {
        '01' => 'Jan',
        '02' => 'Feb',
        '03' => 'Mar',
        '04' => 'Apr',
        '05' => 'May',
        '06' => 'Jun',
        '07' => 'Jul',
        '08' => 'Aug',
        '09' => 'Sep',
        '10' => 'Oct',
        '11' => 'Nov',
        '12' => 'Dec',
    };
    my ($_year,$_month,$_day, $_hour,$_min,$_sec) = Today_and_Now();
    $_month = length($_month) < 2 ? "0${_month}" : $_month;
    $_day = length($_day) < 2 ? "0${_day}" : $_day;
    $_hour = length($_hour) < 2 ? "0${_hour}" : $_hour;
    $_min = length($_min) < 2 ? "0${_min}" : $_min;
    $_sec = length($_sec) < 2 ? "0${_sec}" : $_sec;
    my $_timestamp = $_year . $$month_map{$_month};
    $_timestamp .= $_day . "-" . $_hour . $_min . $_sec;
    return $_timestamp;
} ;# sub get_timestamp()

sub get_node_name_from_core($) {
    my $core = shift;

    my $node = "";
    my $pattern1 = '\w+\.by\.(\d+|user)\.\d{8}\-\d{6}';
    $pattern1 .= '\.sysadmin\-vm[:\_](F\d+_SC\d+)[\.:]';
    my $pattern2 = '\w+\.by\.(\d+|user)\.\d{8}\-\d{6}';
    $pattern2 .= '\.sysadmin\-vm[:_](B?\d+_(LC|R[S]?P|CB)\d+)[\.:]';
    my $pattern3 = '\w+\.by\.(\d+|user)\.\d{8}\-\d{6}';
    $pattern3 .= '\.xr\-vm_node(B?\d+_(LC|R[S]?P|CB)?\d+_(CPU)?\d+)[\.:]';

    if ( $core =~ /$pattern1/) {
        $node = $2;
    } elsif ( $core =~ /$pattern2/) {
        $node = $2;
    } elsif ( $core =~ /$pattern3/) {
        $node = $2;
        #tcp_4097.by.11.20160117-074306.xr-vm_node1_RP0_CPU0.07441.core
        #ntpdc_6414.by.11.20160115-143015.xr-vm_node3_3_CPU0.9e212.core
        #pv6_ma_5588.by.11.20171026-010515.xr-vm_nodeB0_CB0_CPU0.2f25d.core.gz
    }
    return $node;
} ;# sub get_node_name_from_core()

sub verify_remote_proc_via_top($$) {
    my $ip = shift;
    my $pid = shift;

    my $ssh_o = " -q -o UserKnownHostsFile=/dev/null";
    $ssh_o .= " -o StrictHostKeyChecking=no";
    my $cmd = "test -f /proc/${pid}/cmdline && top -p ${pid} -b -n 1";
    my $output = "";
    if ( -d "/opt/cisco/thinxr/" ) {
        $output = `$cmd`;
    } else {
        $output = `/usr/bin/ssh $ssh_o $ip "$cmd" 2>/dev/null`;
    }
    my $proc = "";
    my $pattern = '\s*(\d+)\s+\w+\s+\d+\s+\d+\s+(\S+)\s+(\S+)\s+(\S+)';
    $pattern .= '\s+\S+\s+(\S+)\s+(\S+)\s+(\d+:\d+\.\d+)\s+(\S+)';
    foreach my $line (split(/\n/, $output)) {
        if ( $line =~ /^$pattern/) {
            my $_pid = $1;
            my $virt = $2;
            my $res = $3;
            my $shr = $4;
            my $cpup = $5;
            my $memp = $6;
            my $time = $7;
            $proc = $8;
            last;
        }
    }
    return $proc;
}

sub get_PamLogDir() {
    my $log_dir = "/harddisk\:/" . "/cisco_support/";
    if ( ! -d "/harddisk\:/" ) {
        $log_dir = $ENV{HOME} . "/cisco_support/";
    }
    return $log_dir;
} ;#sub get_PamLogDir()

sub create_pid_file($$$) {
    my $sys_info = shift;
    my $pid_dir = shift;
    my $pid = shift;

    my $pid_file = $pid_dir . "/pid";
    &createFolder($pid_dir) if (! -d $pid_dir);
    if (!open(WD, ">$pid_file") ) {
        my $_msg = "Failed to create pid file ($pid_file): $!";
        create_syslog($sys_info, $_msg, "");
        return -1;
    }
    print WD $pid;
    close(WD);
    return 1;
}

# get date of X hour behind - used for show logging on XR:
sub get_behind_date($) {
    my $Dh = shift;
    my $Dm = shift || -30; #30 mins

    my $month_map = {
        'Jan' => 1,
        'Feb' => 2,
        'Mar' => 3,
        'Apr' => 4,
        'May' => 5,
        'Jun' => 6,
        'Jul' => 7,
        'Aug' => 8,
        'Sep' => 9,
        'Oct' => 10,
        'Nov' => 11,
        'Dec' => 12,
    };

    my $gmt = Gmtime();
    #my ($c_year,$c_mon,$c_day, $c_hour,$c_min,$c_sec,
    #                 $c_doy,$c_dow,$c_dst) = System_Clock([$gmt]);
    #Wed Nov  2 08:02:35 2016
    my $localtime = localtime();
    my ($c_dow, $c_mon, $c_day, $mhs, $c_year) = split (/\s+/, $localtime);
    $c_mon = $$month_map{$c_mon};
    my ($c_hour, $c_min, $c_sec) = split(/:/, $mhs);

    my $Dd = 0;
    #my $Dm = 0;
    my $Ds  =0;
    my ($year,$month,$day, $hour,$min,$sec) =
            Add_Delta_DHMS($c_year,$c_mon,$c_day, $c_hour,$c_min,$c_sec,
                                $Dd,$Dh,$Dm,$Ds);
    my $month_map_rev = {
        1 => 'january',
        2 => 'february',
        3 => 'march',
        4 => 'april',
        5 => 'may',
        6 => 'june',
        7 => 'july',
        8 => 'august',
        9 => 'september',
       10 => 'october',
       11 => 'november',
       12 => 'december',
    };
    return "$year,$month,$day,$hour,$min,$sec";
    #return "$year,$$month_map_rev{$month},$day,$hour,$min,$sec";
} ;# sub get_behind_date($$)

sub get_uptime_via_stat($$) {
    my $sys_info = shift;
    my $ip = shift;
    #ssh 192.0.0.4 "stat -c %Z /proc/sys"
    #1252
    my $remote_ip = shift || "172.0.16.1";
    my $cmd = "/pkg/bin/ng_show_version";
    if ( ! -f $cmd ) {
        return "";
    }
    my $ssh_option = "-q -o UserKnownHostsFile=/dev/null ";
    $ssh_option .= "-o StrictHostKeyChecking=no ";
    my $show_option = "\"ulimit -q unlimited; /pkg/bin/ng_show_version\"";
    #TODO NETNS - netns_cmd
    my $show_cmd = "/usr/bin/ssh $ssh_option $remote_ip $show_option";
    my $show_output = `$show_cmd`;
    return $show_output;
}

sub get_memfree() {
    my $memfree = 0;
    my $file = "/proc/meminfo";
    if (!open(FD, $file)) {
         return $memfree;
    }
    while (<FD>) {
        if (/MemFree:\s+(\d+)\s+kB/i) {
	    $memfree = $1;
            last;
        }
    }
    close(FD);
    return $memfree;
}

sub get_process_total_memory($) {
    my $pid = shift;
    my $total_memory = 0;
    my $file = "/proc/$pid/status";
    if (!open(FD, $file)) {
         return $total_memory;
    }
    while (my $l=<FD>) {
        if ($l =~ /VmRSS:\s+(\d+)\s*kB/) {
	    $total_memory = $1;
            last;
        }
    }
    close(FD);
    return $total_memory;
}

sub get_first_active_xr_ip($) {
    my $node_info = shift;
    my @ipList = ();
    foreach my $node (@{$node_info->{nodeList}}) {
        next if ($node !~ /R[S]?P|B\d+\/CB\d+/i);
        next if ($node_info->{$node}->{type} !~ /active/i);
        my $ip = $node_info->{$node}->{IP};
        push @ipList, $ip;
    }
    my $first_xr_ip = (sort (@ipList))[0];
    return $first_xr_ip;
}

sub get_first_calv_ip($) {
    my $node_info = shift;
    my @ipList = ();
    foreach my $node (@{$node_info->{nodeList}}) {
        #if ($sys_info->{platform} =~ /scapa|ncs4k/i) {
        #    next if ($node !~ /VM1/);
        #}
        next if ($node !~ /R[S]?P|B\d+\/CB\d+/i);
        my $ip = $node_info->{$node}->{IP};
        push @ipList, $ip;
    }
    my $first_calv_ip = (sort (@ipList))[0];
    $first_calv_ip =~ s/\.\d+$/\.1/;
    return $first_calv_ip;
}

sub getOldInstances ($) {
    my $logFile = shift;

    my @oldInstances = ();
    if ( -f $logFile ) {
        if (!open(FD, $logFile)) {
            print "Failed to open $logFile: $!\n";
            return @oldInstances;
        }
        while (<FD>) {
            chomp $_;
            #my ($c, $ctime) = split(/,/, $_);
            push @oldInstances, $_;
        }
        close(FD);
    }
    return @oldInstances;
}

sub update_log($$$$) {
    my $logFile = shift;
    my $node    = shift;
    my $proc    = shift;
    my $_fd     = shift;

    my $rt = 0;
    if (!open($_fd, ">>$logFile")) {
        print "Failed to open $logFile: $!\n";
        return $rt;
    }
    my $timestamp = time();
    my $output = "";
    if ($proc ne '') {
        $output = $node . "," . $proc . "," . $timestamp;
    } else {
        $output = $node . ",," . $timestamp;
    }
    print $_fd $output, "\n";
    close($_fd);
    return 1;
}

sub get_calvados_pam_show_sdr($) {
    my $sys_info = shift;
    my $chvrf = "";
    if ($sys_info->{platform} =~ /panini|scapa|ncs[46]/i) {;
        $chvrf = "chvrf 0";
    }
    #this is to get all sdr info on Panini
    my $output = `$chvrf /opt/cisco/calvados/bin/pam_show_node -A`;
    my $sdr_data;
    my @nodeList;
    my ($node, $ip);
    #NODE: name=0/RP0 ip=192.0.0.1 state=GOING_DOWN(1)
    #NODE: name=0/RP0/CPU0/VM1 ip=192.0.0.4 state=OPERATIONAL(4)
    #NODE: name=0/16/VM2 ip=192.0.0.6 state=OPERATIONAL(4)
    #NODE: name=B0/CB0/CPU0/VM1 ip=192.0.0.4 state=OPERATIONAL(4)
    my %isRpSubnets;
    foreach my $line (split(/\n/, $output)) {
        $line =~ s/[\r]//g;
        if ($line =~ /\s+name=(\S+)\s+ip=(\d+\.\d+\.\d+\.\d+)\s+state=OPERATIONAL/ ) {
            $node = $1;
            $ip = $2;
            my $subnet = $ip;
            $subnet =~ s/\.\d+$/\./;
            $isRpSubnets{$subnet} = 1;
            next if ($ip =~ /\.1$/); #ignore calvados IP
            $node =~ s/CPU\d+\///;
            if ($node =~ /\/(R[S]?P|CB|CC)?\d+/) {
                push @nodeList, $node;
                $sdr_data->{$node}->{IP} = $ip;
            }
        }
    }
    foreach my $line (split(/\n/, $output)) {
        $line =~ s/[\r]//g;
        if ($line =~ /\s+name=(\S+)\s+ip=(\d+\.\d+\.\d+\.\d+)\s+state=OPERATIONAL/ ) {
            $node = $1;
            next if ($node =~ /\/(R[S]?P|CB|CC)?\d+/); #ignore RP that has been tracked
            $ip = $2;
            my $subnet = $ip;
            $subnet =~ s/\.\d+$/\./;
            next if ($isRpSubnets{$subnet}); #ignore LC VM that is created on the same RP
            next if ($ip =~ /\.1$/); #ignore calvados IP
            $node =~ s/CPU\d+\///;
            if ($node =~ /([BC]?\d+)\/([BC]?\d+)\/(VM\d+.*)/) {
                my $chassis_id = $1;
                my $slot = $2;
                my $vm_id = $3;
                if (($slot >= 16) && ($sys_info->{platform} =~ /panini|scapa/i)) {
                    $slot = $slot - 16;
                    $node = $chassis_id . "/" . $slot . "/" . $vm_id;
                }
            }
            my $calv_ip = $ip;
            $calv_ip =~ s/\.\d+$/\.1/;
            push @nodeList, $node;
            $sdr_data->{$node}->{IP} = $ip;
        }
    }
    $sdr_data->{nodeList} = \@nodeList;
    return $sdr_data;
}

sub get_calv_showtech_cli($) {
    my $name = shift;
    my $cmd = "describe show tech-support $name verbose";
    my $output = confd_cli_wrapper_exe($cmd);
    my $show_tech_name = "";
    my $show_tech_script = "";
    #print "output=$output\n";
    my @lines = split(/\n/, $output);
    my $i = 0;
    my $pat = 'script\/show_tech_fast\,?.*\s+';
    $pat .= '\-c\,?\s+(\S+)\s+\-m\,?\s+(show_tech_\S+)\,?\s+\-v';
    foreach my $line (@lines) {
        if ($line =~ /$pat/) {
            $show_tech_name = $1;
            $show_tech_script = $2;
            $show_tech_name =~ s/,//g;
            $show_tech_script =~ s/\,//g;
            last;
        } elsif ($line =~ /script\/show_tech_fast\,?/) {
            $line =~ s/[\r\n]//g;
            my $next_line = $lines[$i+1];
            $next_line =~ s/^\s+//;
            $line .= $next_line;
            if ($line =~ /$pat/) {
                $show_tech_name = $1;
                $show_tech_script = $2;
                $show_tech_name =~ s/,//g;
                $show_tech_script =~ s/\,//g;
            }
            last;
        }
        $i++;
    }
    return "$show_tech_name:$show_tech_script";
}

sub get_xr_showtech_name($) {
    my $cmd = shift;
    my $show_tech_name = "";
    #TODO - all *
    my $pat = 'show.*tech(\-support)*\s+(.*)';
    if ($cmd =~ /$pat/) {
       $show_tech_name = $2;
       $show_tech_name =~ s/[\s\/:]+/_/g;
    }
    return $show_tech_name;
}

sub do_calvados_showtech($$$) {
    my $showtech_dir = shift;
    my $event_type   = shift;
    my $cli_cmd      = shift;

    my $location = "";
    my $cmd = $cli_cmd;
    if ($cmd =~ /\s+location\s+(\S+)/) {
        $location = $1;
        $cmd =~ s/\s+location.*//;
        $cmd =~ s/.*tech\s+//;
    }
    my ($show_tech_name, $show_tech_script);
    $show_tech_name = get_calv_showtech_cli($cmd);

    my $info;
    $info->{rc} = 0;
    $info->{output_file} = "";

    my $cmdline = `cat /proc/cmdline`;
    my $chvrf = "";
    if (($cmdline =~ /=sysadmin/i) &&
        ($cmdline =~ /platform=(panini|scapa|ncs[46]k)/i)) {
        $chvrf = "/sbin/chvrf 0 ";
    }
    my $showtech_script_dir = "/opt/cisco/calvados/script/";
    my $base_cmd = $showtech_script_dir . "show_tech_fast -r -c";
    my $output = get_calv_showtech_cli($cmd);
    ($show_tech_name, $show_tech_script) = split(/:/, $output);

    if ((!$show_tech_script) ||
         (! -f "$showtech_script_dir/$show_tech_script")) {
        print "cmd=$cmd not found - skip\n";
        return $info;
    }
    my $output_file = $showtech_dir . "/" . $event_type;
    $output_file .= "-" . $show_tech_name . "-output.txt";

    my $command = "cd /root/; nohup " . $chvrf . $base_cmd;
    $command .= " " . $show_tech_name . " -m " . $show_tech_script;

    if ($location =~ /\w+/) {
        $command .= " -l " . $location;
    }
    $command .= " > " . $output_file . " &";
    #Need to wait or showtech may fail with the following error:
    #cannot remove '//var/log/2017-Oct-01.143834.UTC-pid
    if (system($command)) {
        print "Error: $!\n";
        return $info;
    }
    my $count = 0;
    while (($count <= 5) && (! -f $output_file)) {
        sleep 1;
        $count++;
    }
    if (-f $output_file) {
        $info->{rc} = 1;
        $info->{output_file} = $output_file;
    }
    return $info;
}

sub do_xr_showtech($$$$) {
    my $sys_info     = shift;
    my $showtech_dir = shift;
    my $event_type   = shift;
    my $cmd          = shift;

    my $info;
    $info->{rc} = 0;
    $info->{output_file} = "";

    my $netns_cmd = get_xr_cli_netns_env($sys_info);
    mkdir $showtech_dir if (! -d $showtech_dir);
    my $show_tech_name = get_xr_showtech_name($cmd);
    $show_tech_name =~ s/\"+//g;
    my $output_file = $showtech_dir . "/" . $event_type . "-";
    $output_file .= $show_tech_name . "-output.txt";
    my $command = "cd /root/; " . $netns_cmd . " nohup ";

    $cmd =~ s/^.*xr_cli\s+//;
    $cmd =~ s/^\"+//;
    $cmd =~ s/\"+$//;
    my $xr_cli = "/pkg/bin/xr_cli";
    if (-f $xr_cli) {
        $command .= $xr_cli . " ";
    }
    $command .= " \"" . $cmd . "\"";
    $command .= " > " . $output_file . " & ";
    if (system($command)) {
        print "Error: $!\n";
        return $info;
    }
    my $count = 0;
    while (($count <= 5) && (! -f $output_file)) {
        sleep 1;
        $count++;
    }
    if (-f $output_file) {
        $info->{rc} = 1;
        $info->{output_file} = $output_file;
    }
    return $info;
}

sub get_showtech_data($$) {
    my $dir            = shift;
    my $event_type     = shift;
    #my @showtech_list = shift;

    my $timeout        = 900;
    my $debug = 0;
    my $archive_duration = 3600; #clean data created 1 hour ago
    my $info;
    my @log_files = ();
    $info->{rc} = 0;
    $info->{log_files} = \@log_files;
    print "dir=$dir\n" if ($debug);
    if (!opendir(DIR, $dir)) {
        print "Failed to open $dir\n";
        return $info;
    }
    my @files = readdir(DIR);
    close(DIR);
    my @raw_logs = ();
    foreach my $file (@files) {
        print "file=$file\n" if ($debug);
        next if ($file !~ /$event_type/);
        next if ($file =~ /pids\.txt/);
        my $log = $dir . "/" . $file;

        if (!open(FD, $log)) {
            unlink $log;
            next;
        }
        my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
            $atime,$mtime,$ctime,$blksize,$blocks) = lstat($log);
        my $delta = time() - $mtime;
        if ($delta > $archive_duration) {
            unlink $log;
            next;
        }
        push @raw_logs, $log;
    }
    return $info if (!scalar(@raw_logs));
    my $pending_logs = scalar(@raw_logs) - scalar(@log_files);
    print "\nThere are $pending_logs show tech files to process\n if ($debug)";
    my %isLogSeen;
    my %isTechDataSeen;
    my %log_map;
    my $interval = 5;
    my $log_cnt = 0;
    my $duration = 0;
    my $start_time = time();
    while($duration < $timeout) {
        my $pending_logs = scalar(@raw_logs) - scalar(@log_files);
        print "\nThere are $pending_logs show tech files to process\n" if ($debug);
        foreach my $log (@raw_logs) {
            next if ($isLogSeen{$log});
            my $time = localtime();
            print "Checking $log at $time ...\n" if ($debug);
            if (!open(FD, $log)) {
                print "Unable to open $log: $!\n";
                next;
            }
            while (my $line =<FD>) {
                if ($line =~ /output\s+available\s+at\s+(.*)/) {
                    my $showtech_file = $1;
                    $showtech_file =~ s/[\r\n]//g;
                    $showtech_file =~ s/\s*$//;
                    $showtech_file =~ s/.*\s+//;
                    $showtech_file =~ s/\/+/\//g;
                    if (!$isTechDataSeen{$showtech_file}) {
                        $isTechDataSeen{$showtech_file} = 1; 
                        $log_map{$log} = $showtech_file;
                        push @log_files, $showtech_file;
                    }
                }
                if (($line =~ /Show.*tech\s+end\s+time/i) &&
                                      (-f $log_map{$log})) {
                    $isLogSeen{$log} = 1;
                    $log_cnt++;
                }
            }
            close(FD);
            unlink $log if ($isLogSeen{$log});
            if (($log_cnt >= scalar(@raw_logs)) ||
                (scalar(@log_files) >= scalar(@raw_logs))) {
                $info->{rc} = 1;
                $info->{log_files} = \@log_files;
                print "All logs found\n";
                $duration = time() - $start_time;
                print "Duration: $duration seconds.\n";
                print "Expected logs: ", scalar @raw_logs, ".\n";
                print "Actual logs: ", scalar @log_files, ".\n";
                return $info;
            }
        } ;# foreach my $log (@raw_logs)
        $duration = time() - $start_time;
        print "Not all show tech data found - continue...\n" if ($debug); 
        print "Duration: $duration seconds.\n" if ($debug);
        sleep $interval;
    } ;# while($duration < $timeout)
    foreach my $log (@raw_logs) {
        unlink $log if (-f $log);
    }
    $info->{rc} = -1;
    $info->{log_files} = \@log_files;
    return $info;
} ;# sub get_showtech_data($$$)

##############################################
# get process name from a pid
##############################################
sub get_proc_name_from_sshfs($$$$) {
    my $sys_info     = shift;
    my $ip           = shift;
    my $pid          = shift;
    my $mnt_root_dir = shift || "/opt/cisco/pam/mnt/";

    my $vf1_3073_ip = $sys_info->{vf1_3073_ip};
    my $src_proc_dir = $mnt_root_dir . "/" . $ip;
    system("mkdir -p \"$src_proc_dir\"") if (! -d $src_proc_dir);
    my $ret;
    if ($vf1_3073_ip eq $ip) {
        $src_proc_dir = "/proc/";
    } else {
        $ret = &check_stale_sshfs_mount($src_proc_dir, "/proc/");
        if (!$ret) {
            if (!sshfs_mount_proc($ip, $src_proc_dir)) {
                return "";
            }
        }
    }
    my $cmdline = $src_proc_dir . "/" . $pid . "/cmdline";
    if (!-f $cmdline) {
        return "";
    }
    if (!open(FD, $cmdline)) {
        return "";
    }
    my $proc=<FD>;
    close(FD);
    if (!defined($proc)) {
        return "";
    }
    $proc = (split(/\0/, $proc))[0];
    if ($proc !~ /\w+/) {
        return "";
    }
    $proc = (split(/\s+/, $proc))[0];
    $proc =~ s/:$//;
    $proc =~ s/^\-//;
    $proc = basename($proc);
    $proc =~ s/\s+$//;

    if ($vf1_3073_ip ne $ip) {
        $ret = &pam::umount_sshfs($src_proc_dir);
    }
    return $proc;
} ;# sub get_proc_name_from_sshfs()

sub get_netns_env($) {
    my $sys_info = shift;
    my $osType = $sys_info->{hostType};
    my $platform = $sys_info->{platform};

    my $netns_cmd = "";
    if (($osType =~ /xr/i) &&
        ($platform !~ /panini|scapa|ncs[46]k/i)) {
        my $my_pid = $$;
        my $netns_identify = `/sbin/ip netns identify $my_pid`;
        if ($netns_identify !~ /xrnns/) {
            $netns_cmd = "/sbin/ip netns exec xrnns ";
        }
        if ($sys_info->{is_thinxr}) {
            my $_ENV = $ENV{"LD_LIBRARY_PATH"};
            #$ENV{"LD_LIBRARY_PATH"} = "/opt/cisco/hostos/usr/lib64/";
            #$ENV{"LD_LIBRARY_PATH"} .= ":/opt/cisco/hostos/usr/lib64/";
            $ENV{"LD_LIBRARY_PATH"} = ":/pkg/lib:/pkg/lib/cerrno";
            $ENV{"LD_LIBRARY_PATH"} .= ":/pkg/lib/mib:/pkg/lib/spp_plugins";
            $ENV{"LD_LIBRARY_PATH"} .= ":" . $_ENV if ($_ENV);
        }
    }
    return $netns_cmd;
}

sub get_xr_cli_netns_env($) {
    my $sys_info = shift;
    my $osType = $sys_info->{hostType};
    my $platform = $sys_info->{platform};

    my $netns_cmd = "";
    my $vrf = "xrnns";
    if ($osType =~ /xr/i) {
        $vrf = "xrnns";
        my $my_pid = $$;
        my $netns_identify = `/sbin/ip netns identify $my_pid`;
        if ($netns_identify !~ /$vrf/) {
            $netns_cmd = "/sbin/ip netns exec " . $vrf . " ";
        }
        my $_ENV = $ENV{"LD_LIBRARY_PATH"};
        $ENV{"LD_LIBRARY_PATH"} = "/pkg/lib:/pkg/lib/cerrno";
        $ENV{"LD_LIBRARY_PATH"} .= ":/pkg/lib/mib:/pkg/lib/spp_plugins";
        $ENV{"LD_LIBRARY_PATH"} .= ":" . $_ENV if ($_ENV);
    }
    return $netns_cmd;
}

sub is_thinxr() {
    return 1 if ( -d "/opt/cisco/thinxr/" );
    return 0;
}

#get the PID -> process name map
#the data has been collected as part of ltrace data collection
sub get_pid_proc_map($$) {
    my $node         = shift;
    my $depot_dir    = shift || "/misc/disk1/cisco_support/ltrace/";
    my $node_dir    = $depot_dir . "/" . $node;
    my $info;
    if (! -d $node_dir) {
        return $info;
    }
    if (!opendir(DIR, $node_dir)) {
        return $info;
    }
    my @files = readdir(DIR);
    closedir(DIR);
    foreach my $file (@files) {
        if ($file =~ /^(\d+)\-(\w+)\-/) {
            $info->{$1} = $2;
        }
    }
    return $info;
}
