#!/usr/bin/perl

# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
# pam_ltrace.pm - module for monitoring ltrace, xos, and shwmwin on XR
#
# Feb 2016, Jieming Wang
# 
# Copyright (c) 2016, 2018-2019 by Cisco Systems, Inc.
# All rights reserved.
# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

use strict;
use warnings;
use Time::HiRes qw( usleep ualarm gettimeofday tv_interval );
use File::Basename;
use File::Copy;
use Logfile::Rotate;
use Expect;

use lib qw ( /pkg/opt/cisco/pam/ );
use pam_perf;

package      pam_ltrace;
require      Exporter;
use vars     qw (@ISA @EXPORT $VERSION );
use strict;
use File::Basename;
use Date::Calc qw(:all);

@ISA       = qw (Exporter);
@EXPORT    = qw ();
$VERSION   = 1.00;

@EXPORT    = qw (
    collect_proc_ltrace_memory
    collect_node_ltrace_memory
    collect_local_ltrace_memory
    get_process_ltrace_meminfo
    get_proc_ltrace_memory
    verify_leak_by_ltrace
    get_memory_from_smaps
    update_ltrace
);

my $month = '(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)';
my $name_tpl = "-trace-shmwin.txt";

##############################################
# collect ltrace for a single process
##############################################
sub collect_proc_ltrace_memory($$$$$$$) {
    my $sys_info     = shift;
    my $ip           = shift;
    my $pid          = shift;
    my $node         = shift;
    my $mnt_root_dir = shift || "/opt/cisco/pam/mnt/";
    my $depot_dir    = shift || "/misc/disk1/cisco_support/ltrace/";

    my $vf1_3073_ip = $sys_info->{vf1_3073_ip};
    my $src_proc_dir = $mnt_root_dir . "/" . $node;
    my $dst_ltrace_dir = $depot_dir . "/" . $node;
    system("mkdir -p \"$src_proc_dir\"") if (! -d $src_proc_dir);
    my $ret;
    if ($vf1_3073_ip eq $ip) {
        $src_proc_dir = "/proc/";
    } else {
        system("mkdir -p \"$dst_ltrace_dir\"") if (! -d $dst_ltrace_dir);
        if (! -d $dst_ltrace_dir) {
            return 0;
        }
        $ret = &check_stale_sshfs_mount($src_proc_dir, "/proc/");
        if (!$ret) {
            if (!sshfs_mount_proc($ip, $src_proc_dir)) {
                return 0;
            }
        }
    }
    my $walltime = time();
    my $smaps   = $src_proc_dir . "/" . $pid . "/smaps";
    my $exe     = $src_proc_dir . "/" . $pid ."/exe";
    my $cmdline = $src_proc_dir . "/" . $pid . "/cmdline";
    if ((!-f $smaps) || (!-f $exe) || (!-f $cmdline)) {
        return 0;
    }
    my $exe_link = readlink($exe);
    if ($exe_link !~ /\/opt\/cisco\/(XR\/packages|thinxr|install-iosxr)/) {
        return 0;
    }
    if (!open(FD, $cmdline)) {
        return 0;
    }
    my $proc=<FD>;
    close(FD);
    if (!defined($proc)) {
        return 0;
    }
    $proc = (split(/\0/, $proc))[0];
    if ($proc !~ /\w+/) {
        return 0;
    }
    $proc = (split(/\s+/, $proc))[0];
    $proc =~ s/:$//;
    $proc =~ s/^\-//;
    $proc = basename($proc);
    $proc =~ s/\s+$//;

    my $info = get_memory_from_smaps($smaps);
    my $ltrace_shmwin_file = $dst_ltrace_dir . "/" . $pid . "-" . $proc;
    $ltrace_shmwin_file .= $name_tpl;
    if (open(WD,">>$ltrace_shmwin_file")) {
        my $str = $walltime . "," . $info->{lt_total_rss};
        $str .= "," . $info->{shmwin_total_rss} . "," . $info->{xdt_total_rss};
        print WD $str, "\n";
        close(WD);
    }
    if ($vf1_3073_ip ne $ip) {
        $ret = &pam::umount_sshfs($src_proc_dir);
    }
    return 1;
} ;# sub collect_proc_ltrace_memory($$$$$$)

##############################################
# collect ltrace for all processes on the node
##############################################
sub collect_node_ltrace_memory($$$$$$) {
    my $sys_info     = shift;
    my $ip           = shift;
    my $node         = shift;
    my $mnt_root_dir = shift || "/opt/cisco/pam/mnt/";
    my $depot_dir    = shift || "/misc/disk1/cisco_support/ltrace/";

    my $ret;
    my $vf1_3073_ip = $sys_info->{vf1_3073_ip};
    my $src_proc_dir = $mnt_root_dir . "/" . $node;
    my $dst_ltrace_dir = $depot_dir . "/" . $node;
    system("mkdir -p \"$src_proc_dir\"") if (! -d $src_proc_dir);
    system("mkdir -p \"$dst_ltrace_dir\"") if (! -d $dst_ltrace_dir);
    if (! -d $dst_ltrace_dir) {
        return 0;
    }
    if ($vf1_3073_ip eq $ip) {
        $src_proc_dir = "/proc/";
    } else {
        $ret = &pam::check_stale_sshfs_mount($src_proc_dir, "/proc/");
        if (!$ret) {
            $ret = pam::sshfs_mount_proc($ip, $src_proc_dir);
            if (!$ret) {
                return 0;
            }
        }
    }
    my $walltime = time();
    if (opendir(DIR,$src_proc_dir)) {
        my @pids = readdir(DIR);
        closedir(DIR);
        foreach my $pid (@pids) {
            next if ($pid !~ /^\d+$/);
            next if ($pid < 500);
            my $smaps   = $src_proc_dir . "/" . $pid . "/smaps";
            my $exe     = $src_proc_dir . "/" . $pid ."/exe";
            my $cmdline = $src_proc_dir . "/" . $pid . "/cmdline";
            if ((!-f $smaps) || (!-f $exe) || (!-f $cmdline)) {
                next;
            }
            my $exe_link = readlink($exe);
            if ($exe_link !~ /\/opt\/cisco\/(XR\/packages|thinxr|install-iosxr)/) {
                next;
            }
            if (!open(FD, $cmdline)) {
                next;
            }
            my $proc=<FD>;
            close(FD);
            if (!defined($proc)) {
                next;
            }
            $proc = (split(/\0/, $proc))[0];
            if ($proc !~ /\w+/) {
                next;
            }
            $proc = (split(/\s+/, $proc))[0];
            $proc =~ s/:$//;
            $proc =~ s/^\-//;
            $proc = basename($proc);
            $proc =~ s/\s+$//;

            my $info = get_memory_from_smaps($smaps);
            my $ltrace_shmwin_file = $dst_ltrace_dir . "/" . $pid . "-" . $proc;
            $ltrace_shmwin_file .= $name_tpl;
            if (open(WD,">>$ltrace_shmwin_file")) {
                my $str = $walltime . "," . $info->{lt_total_rss};
                $str .= "," . $info->{shmwin_total_rss} . "," . $info->{xdt_total_rss};
                print WD $str, "\n";
                close(WD);
            }
        }
    }
    if ($vf1_3073_ip ne $ip) {
        $ret = pam::umount_sshfs($src_proc_dir);
    }
    return 1;
} ;# sub collect_node_ltrace_memory()

##############################################
# collect ltrace for all processes on local node
##############################################
sub collect_local_ltrace_memory() {
    my $dump_output = shift || 0;
    my $proc_dir = "/proc/";

    my $info;
    my @ltraces = ();
    my $min_pid = 300;
    if (opendir(DIR,$proc_dir)) {
        my @pids = readdir(DIR);
        closedir(DIR);
        foreach my $pid (@pids) {
            next if (($pid !~ /^\d+$/) || ($pid < $min_pid));
            my $smaps   = $proc_dir . "/" . $pid . "/smaps";
            my $exe     = $proc_dir . "/" . $pid ."/exe";
            my $cmdline = $proc_dir . "/" . $pid . "/cmdline";
            if ((!-f $smaps) || (!-f $exe) || (!-f $cmdline)) {
                next;
            }
            my $exe_link = readlink($exe);
            if ($exe_link !~ /\/opt\/cisco\/(XR\/packages|thinxr|install-iosxr)\//) {
                next;
            }
            if (!open(FD, $cmdline)) {
                next;
            }
            my $proc=<FD>;
            close(FD);
            if (!defined($proc)) {
                next;
            }
            $proc = (split(/\0/, $proc))[0];
            if ($proc !~ /\w+/) {
                next;
            }
            $proc = (split(/\s+/, $proc))[0];
            $proc =~ s/:$//;
            $proc =~ s/^\-//;
            $proc = basename($proc);
            $proc =~ s/\s+$//;

            my $info = get_memory_from_smaps($smaps);
            my $str = $proc . "," . $pid;
            if (defined($info->{lt_total_rss})) {
                $str .=  "," . $info->{lt_total_rss};
            } else {
                $str .=  ",0";
            }
            if (defined($info->{shmwin_total_rss})) {
                $str .=  "," . $info->{shmwin_total_rss};
            } else {
                $str .=  ",0";
            }
            if (defined($info->{xdt_total_rss})) {
                $str .=  "," . $info->{xdt_total_rss};
            } else {
                $str .=  ",0";
            }
            if ($dump_output) {
                print $str, "\n";
            } else {
                push @ltraces, $str;
            }
        }
    }
    if ($dump_output) {
        return 1;
    } else {
        $info->{ltraces} = \@ltraces;
        return $info;
    }
} ;# sub collect_local_ltrace_memory($$$$$$)

##############################################
# get ltrace memory for a process
##############################################
sub get_process_ltrace_meminfo($$$$) {
    my $pid          = shift;
    my $proc         = shift;
    my $node         = shift;
    my $depot_dir    = shift || "/misc/disk1/cisco_support/ltrace/";

    my $info;
    my @timestamps = ();
    my $ltrace_shmwin_file = $depot_dir . "/" . $node . "/" . $pid;
    $ltrace_shmwin_file .= "-" . $proc . $name_tpl;
    if (open(FD,$ltrace_shmwin_file)) {
        while(my $line=<FD>) {
            if ($line =~ /^\s*(\d+),(\d+),(\d+),(\d+)/) {
                my $timestamp  = $1;
                my $lt_rss     = $2;
                my $shmwin_rss = $3;
                my $xdt_rss    = $4;
                push @timestamps, $timestamp;
                $info->{$timestamp}->{lt_toal_rss}     = $lt_rss;
                $info->{$timestamp}->{shmwin_toal_rss} = $shmwin_rss;
                $info->{$timestamp}->{xdt_total_rss}    = $xdt_rss;
            }
        }
    }
    $info->{timestamps} = \@timestamps;
    return $info;
}

sub get_proc_ltrace_memory($$$$$$$) {
    my $node             = shift;
    my $pid              = shift;
    my $proc             = shift;
    my $start_time       = shift;
    my $stop_time        = shift;
    my $interval         = shift;
    my $ltrace_depot_dir = shift;

    #return @ltrace_mem;
    my $start_wtime = 0;
    if ($start_time =~ /\w+/) {
        $start_wtime = pam_perf::convert_to_wall_time($start_time);
    }
    my $stop_wtime = 0;
    if ($stop_time =~ /\w+/) {
        $stop_wtime = pam_perf::convert_to_wall_time($stop_time);
    }
    my $wtime = 0;
    my $isStarted = 0;
    my $isStopped = 0;

    my $found_date = 0;
    my $count = 0;
    my $next_step = 0;
    my $collect_data = 0;

    my $month = '(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)';
    my $ltrace_shmwin_file = $ltrace_depot_dir . "/" . $node . "/" . $pid;
    $ltrace_shmwin_file .= "-" . $proc . $name_tpl;
    my @ltrace_mems = ();
    if (!open(FD,$ltrace_shmwin_file)) {
        return @ltrace_mems;
    }
    my $pat = '(\d{4}\/' . ${month} . '\/\d+\/\d{2}:\d{2}:\d{2})';
    $pat .= ',(\d+),(\d+),(\d+)';
    while(my $line=<FD>) {
        $line =~ s/[\r\n]//g;
        if ($line =~ /$pat/) {
            my $timestamp  = $1;
            my $lt_rss     = $3;
            my $shmwin_rss = $4;
            my $xdt_rss    = $5;

            $collect_data = 0;
            if (!$isStarted) {
                $wtime = pam_perf::convert_to_wall_time($timestamp);
                if ($wtime >= $start_wtime) {
                    $count++;
                    $isStarted = 1;
                    $next_step += $interval;
                    $collect_data = 1;
                }
            } else {
                $count++;
                if ($count == $next_step ) {
                    $next_step += $interval;
                    $collect_data = 1;
                }
            }
            last if ($isStopped);
            if (!$stop_time) {
                $wtime = pam_perf::convert_to_wall_time($timestamp);
                if ( ($wtime >= $stop_time) && ($stop_time =~ /\w+/) ) {
                    $isStopped = 1;
                }
            }
            next if ($isStarted eq 0);
            next if ($collect_data < 1);
            push @ltrace_mems, "$timestamp,$lt_rss,$shmwin_rss,$xdt_rss";
        }
    }
    close(FD);
    return @ltrace_mems;
}

sub verify_leak_by_ltrace($$$) {
    my $_mem_leaks   = shift;
    my $_ltrace_mems = shift;
    my $ltrace_ratio = shift || 0.5;
    my @mem_leaks    = @$_mem_leaks;
    my @ltrace_mems  = @$_ltrace_mems;

    # Here is the logic to ientify leak.
    # since we have identified leak based on trend:
    # i.e., _mem_leaks increased continuously, so we just 
    # compare total delta with ltreace delta.
    # If ltrace delta is close to total delta (>= 50%),
    # it means increase is (likely) due to ltrace and not considered a leak.
    # Otherwise, its genuine leak.

    my $first_shm = 0;
    my $last_shm  = 0;
    my $first_leak   = 0;
    my $last_leak    = 0;

    #TODO --- why $mem_leaks[0]????/
    my @leak_samples = split(/;/, $mem_leaks[0]); #Only one item!!!
    foreach my $leak_sample (@leak_samples) {
        my ($_timestamp, $_pid, $_proc, $_mem, $_delta, $_shared) =
                                             split(/,/, $leak_sample);
        next if ($_mem !~ /\d+/);
        $first_leak = $_mem if ($first_leak eq 0);
        $last_leak  = $_mem;
    }

    foreach my $ltrace_mem (@ltrace_mems) {
        my ($lt_timestamp, $lt_rss, $shmwin_rss, $xdt_rss) = split(/,/, $ltrace_mem);
        next if (($lt_timestamp !~ /\w+/) || ($lt_rss !~ /\d+/));
        #TODO - possible add weighing factor for shmwin
        # with 0.5, meaning 2 x shresholds
        # with 0.25, meaning 4 x shresholds, etc
        $first_shm = $lt_rss + $shmwin_rss + $xdt_rss if ($first_shm eq 0);
        $last_shm = $lt_rss + $shmwin_rss + $xdt_rss;
    }
    my $shm_delta = $last_shm - $first_shm;
    #Its kB vs MB with leak_data (collected via top):
    $shm_delta = sprintf("%d", $shm_delta/1024);
    my $leak_delta   = $last_leak - $first_leak - $shm_delta;
    if ($shm_delta > 0) {
        # should be always true - but just in case (avoid divide by 0)
        if ($leak_delta > 0) {
            if ($leak_delta < $#leak_samples) {
               return 0; # due to trace + shmwin
            } else {
               return 1;
            }
        } else {
            return 0; #???
        }
    } else {
        #no increase in ltrace.
        return 1;
    }
} ;# sub verify_leak_by_ltrace($$$)

###########################################################
# Get the memory usage by ltrace, shmwin, and xso/xdt
# total - the total amount that can be located to trace/shmwin/xdt
# Rss: the actual usage
# Pss: adjusted usage
###########################################################
sub get_memory_from_smaps($) {
    my $file = shift;
    my $info;

    my $shw_pat = 'aib|dpa|ifc-ipv4|ifc-ipv6|ifc-mpls|im_db|im_rd|lrid_svr_shm|spp';
    my $shmwin_total_rss = 0;
    my $lt_total_rss     = 0;
    my $xdt_total_rss    = 0;
    $info->{shmwin_total_rss} = 0;
    $info->{lt_total_rss}     = 0;
    $info->{xdt_total_rss}    = 0;
    if (!open(FD, $file)) {
        return $info;
    }
    my $total_rss = 0;
    local $/ = 'VmFlags:';
    my %isSeen;
    while(my $buf=<FD>) {
        #if ($buf =~ /\/dev\/.*\/shmwin\/([\w\-]+)/) {}
        if ($buf =~ /\/dev\/.*ltrace/) {
            foreach my $line (split(/\n/,$buf)) {
                if ($line =~ /^Rss:\s+(\d+)\s+kB/i) {
                    $lt_total_rss += $1;
                    last;
                }
            }
        } elsif ($buf =~ /\.xdt/) {
            foreach my $line (split(/\n/,$buf)) {
                if ($line =~ /^Rss:\s+(\d+)\s+kB/i) {
                    $xdt_total_rss += $1;
                    last;
                }
            }
        } elsif ($buf =~ /\/dev\/.*shmwin\/($shw_pat)/) {
            my $shwname = $1;
            if (!$isSeen{$shwname}) {
                $isSeen{$shwname} = 1;
                $info->{$shwname}->{total_rss} = 0;
            }
            foreach my $line (split(/\n/,$buf)) {
                if ($line =~ /^Rss:\s+(\d+)\s+kB/i) {
                    $shmwin_total_rss += $1;
                    $info->{$shwname}->{total_rss} += $1;
                    last;
                }
            }
        }
    }
    close(FD);
    undef $/;
    $info->{lt_total_rss}     = $lt_total_rss;
    $info->{shmwin_total_rss} = $shmwin_total_rss;
    $info->{xdt_total_rss}    = $xdt_total_rss;
    return $info;
} ;# sub get_memory_from_smaps($)

sub update_ltrace($$$$) {
    my $output           = shift;
    my $node             = shift;
    my $ltrace_depot_dir = shift;
    my $timestamp        = shift;

    mkdir $ltrace_depot_dir if (! -d $ltrace_depot_dir);
    my $ltrace_node_dir = $ltrace_depot_dir .  "/". $node;
    mkdir $ltrace_node_dir if (! -d $ltrace_node_dir);
    return 0 if (! -d $ltrace_node_dir);

    foreach my $line (split(/\n/, $output)) {
        my ($proc,$pid,$lt_rss,$shmwin_rss,$xdt_rss) = split(/,/, $line);
        next if ($proc !~ /\w+/);
        next if ($pid !~ /^\d+$/);
        my $ltrace_shmwin_file = $ltrace_node_dir . "/";
        $ltrace_shmwin_file .= $pid ."-" . $proc . $name_tpl;
        if (open(WD,">>$ltrace_shmwin_file")) {
            my $str = $timestamp . "," . $lt_rss;
            $str .= "," . $shmwin_rss . "," . $xdt_rss;
            print WD $str, "\n";
            close(WD);
        } else {
            next;
            #return 0;
        }
    }
    return 1;
}

