#!/usr/bin/perl

# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
# Common library used by PAM (for performance/mempory monitoring)
# 
# Feb 2016, Jieming Wang
# 
# Copyright (c) 2016-2019 by Cisco Systems, Inc.
# All rights reserved.
# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

package      pam_perf;
require      Exporter;
use vars     qw (@ISA @EXPORT $VERSION );
use strict;
use pam;
use File::Basename;
use Date::Calc qw(:all);

@ISA       = qw (Exporter);
@EXPORT    = qw ();
$VERSION   = 1.00;

@EXPORT    = qw (
    convert_to_wall_time
    convert_localtime_to_wall_time
    get_memory_by_interval
    get_total_mem_info
    get_mem_info
    get_cpu_info
    get_potential_leaks
    get_dynamic_leaks
    get_total_timestamps
    get_total_timestamps_v2
    get_sub_block_mem_leaks
    get_mem_info_by_pid
    verify_mem_info_by_pid
    get_blocks_by_reload
    get_memory_last_samplings
    get_cpu_hogs
    get_disk_last_samplings
    get_ng_df_info
    get_top_header
);

my $wday = '(Mon|Tue|Wed|Thu|Fri|Sat|Sun)';
my $month = '(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)';
my $month_map = {
     'Jan' => 1,
     'Feb' => 2,
     'Mar' => 3,
     'Apr' => 4,
     'May' => 5,
     'Jun' => 6,
     'Jul' => 7,
     'Aug' => 8,
     'Sep' => 9,
     'Oct' => 10,
     'Nov' => 11,
     'Dec' => 12,
};

#convert spelled months to numerical
sub convert_to_wall_time($) {
    my $timestamp = shift;

    my ($year, $mon, $day, $time) = split(/\//, $timestamp);
    my ($hour, $min, $sec) = split(/:/, $time);
    $mon = $$month_map{$mon};
    $mon = length($mon) < 2 ? "0${mon}" : $mon;
    if ( ($year < 2010) || ($year > 2037)) {
        return 0;
    }
    if ( ($mon < 1) || ($mon > 12)) {
        return 0;
    }
    if ( ($day < 1) || ($day > 31)) {
        return 0;
    }
    if ( ($hour < 0) || ($hour > 24)) {
        return 0;
    }
    if ( ($min < 0) || ($min > 60)) {
        return 0;
    }
    if ( ($sec < 0) || ($sec > 60)) {
        return 0;
    }
    $timestamp = Date_to_Time($year,$mon,$day,$hour,$min,$sec);

    return $timestamp;
} ;# sub convert_to_wall_time($)

#extracts information from input file, returns a hash with all the data
sub get_memory_by_interval($$$$) {
    my $input = shift;
    my $start_time = shift;
    my $stop_time = shift;
    my $_known_pidList = shift;

    my @mem_info_list = ();
    if (!open(FD, $input)) {
        return @mem_info_list;
    }

    my $start_wtime = convert_to_wall_time($start_time);
    my $stop_wtime = convert_to_wall_time($stop_time);
    my $time_info = get_total_timestamps($input, $start_wtime);
    my @timestamps = @{$time_info->{time_lines}};
    my @up_times = @{$time_info->{uptime_lines}};

    #!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    #only select 200 sampling points to reduce runtime
    #!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    my $max_samples = 200;
    my $interval = sprintf("%d", ($stop_wtime - $start_wtime)/$#timestamps);
    if ($#timestamps > (2 * $max_samples)) {
        $interval = sprintf("%d", ($stop_wtime - $start_wtime)/$max_samples);
    }

    my $count = 0;
    my @buffers;
    my $found_date = 0;

    my $isStarted = 1;
    if ($start_time =~ /\w+/) {
        $isStarted = 0;
    }
    my $isStopped = 0;
    my $isSelected = 0;
    my $wtime;
    my $next_wtime = $start_wtime;

    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";

    my $timestamp = "";
    @timestamps = ();

    while (my $line = <FD>) {
        if ( $line =~ /^$date_pattern/) {
            my $_month = $2;
            my $_day = $3;
            my $_time = $4;
            my $_year = $6;
            $timestamp = $_year . "/" . $_month . "/" . $_day . "/" . $_time;
            $wtime = convert_to_wall_time($timestamp);
            #if ( ($timestamp eq $start_time) && ($timestamp =~ /\w+/) ) {}
            $isSelected = 0;
            if ($wtime eq $start_wtime) {
                $isStarted = 1;
                $isSelected = 1;
                $next_wtime += $interval;
            } elsif ($wtime >= $next_wtime) {
                $isSelected = 1;
                $next_wtime += $interval;
            }
            #if ($isStarted) {}
            if ($isSelected) {
                $count++;
                $found_date++;
                push @timestamps, $timestamp;
            }
            #push @timestamps, $timestamp;
            last if ($isStopped);
            if ($wtime >= $stop_wtime) {
                $isStopped = 1;
            }
        }
        next if ($found_date < 1);
        #next if ($isStarted eq 0);
        next if ($isSelected eq 0);
        last if ( $line =~ /(\[\S+\]\$\s+|RP\S+#)exit/);
        next if ($isSelected eq 0);
        if ($line =~ /\w+/) {
            $buffers[$count] .= $line;
        }
    }
    close(FD);
    
    my $no = $#buffers + 1;

    my $mem_data;
    $mem_data->{timestamps} = \@timestamps;

    for (my $i = 1; $i<=$no; $i++) {
        my $buffer = $buffers[$i];
        next if (!$buffer);
        next if ($buffer !~ /\w+/);
        my $mem_info = get_mem_info($buffer, "", "", $_known_pidList);
        push @mem_info_list, $mem_info;
    }
    $mem_data->{mem_info_list} = \@mem_info_list;
    return @mem_info_list;
} ;# sub get_memory_by_interval($$$$)

#get mem info for multi nodes
sub get_total_mem_info($$$$$$) {
    my $input_dir = shift;
    my $efr = shift;
    my $start_time = shift;
    my $stop_time = shift;
    my $osType = shift || "xr";
    #my $ignore_process = shift || 0;
    my $_known_pidList = shift;
    #my @known_pidList = @$_known_pidList;

    $osType = "calvados" if ($osType =~ /cal/i);

    my @inputfiles;
    if ( -d $input_dir) {
        if (!opendir(DIR,$input_dir)) {
            exit;
        }
        my @_inputfiles = readdir(DIR);
        closedir(DIR);
        foreach my $_file (@_inputfiles) {
            next if ($_file !~ /^${osType}\-\d+_/i);
            next if (($_file !~ /$efr/) && ($input_dir !~ /$efr/));
            next if ($_file =~ /\.(gz|zip|tar)$/);
            if ($_file =~ /\.log/) {
                push @inputfiles, $_file;
            }
        }
    }

    my $max_collections = 0;
    my @filelist = ();
    my $total_mem_info;
    @inputfiles =  sort {$a cmp $b} @inputfiles;
    foreach my $file (@inputfiles) {
        my $input_file = $input_dir . "/" . $file;

        if (!open(IN, $input_file)) {
            next;
        }
        close(IN);
        my @mem_info_list = get_memory_by_interval($input_file,
                                                   $start_time,
                                                   $stop_time,
                                                   $_known_pidList);
        $total_mem_info->{$file}->{mem_info_list} = \@mem_info_list;

        $total_mem_info->{reference_file} = $file;
        my $last = $#mem_info_list - 1;
        if ($max_collections < $last) {
            $max_collections = $last;
            my $reference_file = $file;
            $total_mem_info->{reference_file} = $reference_file;
        }
        push @filelist, $file;
    }
    $total_mem_info->{filelist} = \@filelist;
    $total_mem_info->{max_collections} = $max_collections;
    return $total_mem_info;
} ;# sub get_total_mem_info($$$$$$)

#gets the memory info, does the heavy lifting for get_memory
sub get_mem_info($$$$) {
    my $buffer = shift;
    my $start_time = shift;
    my $stop_time = shift;
    #my $ignore_process = shift || 0;
    my $_known_pidList = shift;
    my @known_pidList = @$_known_pidList;
    
    my $skipped_procs = 'perl|top|bash|ssh|\w+\.sh|cron|oom|scsi_eh';
    $skipped_procs .= '|kworker|kswap|\[\S+\]|klogd';
    $skipped_procs .= '|xinetd|rpcbind|syslogd|dbus-daemon';
    $skipped_procs .= '|libvirtd';
    $skipped_procs .= '|sbin\/(init|udevd)';

    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";

    my $top_pattern = 'top +\- +(\d{2}:\d{2}:\d{2}) +up +(.*), +\d+ +user[s]*,';
    $top_pattern .= ' +load average: +(\d+\.\d+), (\d+\.\d+), (\d+\.\d+)';
    my $cpu_pattern = 'Cpu\(s\): +([\d\.]+)%us, +([\d\.]+)%sy, +([\d\.]+)%ni,';
    $cpu_pattern .= ' +([\d\.]+)%id, +([\d\.]+)%wa, +([\d\.]+)%hi,';
    $cpu_pattern .= ' +([\d\.]+)%si, +([\d\.]+)%st';

    my $pattern1 = '\s*(\d+)\s+\w+\s+\d+\s+\d+\s+(\S+)\s+(\S+)\s+(\S+)\s+\S+';
    $pattern1 .= '\s+(\S+)\s+(\S+)\s+(\d+:\d+(\.\d+)?)\s+(\S+)';

    my $isStarted = 1;
    if ($start_time =~ /\w+/) {
         $isStarted = 0;
    }
    my $get_virt = 0;
    my $min_pid = 500;
    my $isStopped = 0;
    my %isDfSeen;

    my $mem_info;
    my @pidList = ();
    my $found_date = 0;
    foreach my $line (split(/\n/, $buffer)) {
        $line =~ s/\r//g;
        if ( $line =~ /^$date_pattern/) {
            my $_month = $2;
            my $_day = $3;
            my $_time = $4;
            my $_year = $6;
            my $timestamp = $_year . "/" . $_month . "/" . $_day . "/" . $_time;
            $mem_info->{timestamp} = $timestamp;
            $found_date++;
            next;

        }
        next if ($found_date < 1);
        last if ( $line =~ /(\[\S+\]\$\s+|RP\S+#)exit/);
     
        if ( $line =~ /^$top_pattern/) {
            my $now = $1;
            my $up = $2;
            my $lavg1 = $3;
            my $lavg5 = $4;
            my $lavg15 = $5;
            $mem_info->{top}->{now} = $now;
            $mem_info->{top}->{up} = $up;
            $mem_info->{top}->{load1} = $lavg1;
            $mem_info->{top}->{load5} = $lavg5;
            $mem_info->{top}->{load15} = $lavg15;
            next;
        }
        if ( $line =~ /^$cpu_pattern/) {
            my $cpu_us = $1;
            my $cpu_sy = $2;
            my $cpu_ni = $3;
            my $cpu_id = $4;
            my $cpu_wa = $5;
            my $cpu_hi = $6;
            my $cpu_si = $7;
            my $cpu_st = $8;
            $mem_info->{top}->{cpu_us} = $cpu_us;
            $mem_info->{top}->{cpu_sy} = $cpu_sy;
            $mem_info->{top}->{cpu_ni} = $cpu_ni;
            $mem_info->{top}->{cpu_id} = $cpu_id;
            $mem_info->{top}->{cpu_wa} = $cpu_wa;
            $mem_info->{top}->{cpu_hi} = $cpu_hi;
            $mem_info->{top}->{cpu_si} = $cpu_si;
            $mem_info->{top}->{cpu_st} = $cpu_st;
            next;
        }

        if ( $line =~ /^$pattern1/) {
            my $pid = $1;
            my $virt = $2;
            my $res = $3;
            my $shr = $4;
            my $cpup = $5;
            #my $memp = $6;
            my $time = $7;
            my $proc = $9;
            next if ((@known_pidList) && (!grep(/\b$pid\b/, @known_pidList)));
            next if ($pid <= $min_pid);
            next if ($proc =~ /$skipped_procs/);
            push @pidList, $pid;
            if ($get_virt) {
                if ( $virt =~ /m/i ) {
                    $virt =~ s/m//gi;
                } elsif ( $virt =~ /g/i ) {
                    $virt =~ s/g//gi;
                    $virt *= 1024;
                } elsif ( $virt =~ /t/i ) {
                    $virt =~ s/t//gi;
                    $virt *= 1024 * 1024;
                } else {
                    $virt /= 1024;
                }
            }
            if ( $res =~ /m/i ) {
                $res =~ s/m//gi;
            } elsif ( $res =~ /g/i ) {
                $res =~ s/g//gi;
                $res *= 1024;
            } elsif ( $res =~ /t/i ) {
                $res =~ s/t//gi;
                $res *= 1024 * 1024;
            } else {
                $res /= 1024;
            }
            if ( $shr =~ /m/i ) {
                $shr =~ s/m//gi;
            } elsif ( $shr =~ /g/i ) {
                $shr =~ s/g//gi;
                $shr *= 1024;
            } elsif ( $shr =~ /t/i ) {
                $shr =~ s/t//gi;
                $shr *= 1024 * 1024;
            } else {
                $shr /= 1024;
            }

            if ($get_virt) {
                $mem_info->{top}->{virt}->{$pid} = $virt;
            }
            $mem_info->{top}->{res}->{$pid} = $res;
            $mem_info->{top}->{shr}->{$pid} = $shr;
            $mem_info->{top}->{proc}->{$pid} = $proc;
            if (0) {
            $mem_info->{top}->{cpup}->{$pid} = $cpup;
            $mem_info->{top}->{time}->{$pid} = $time;
            }
            #$mem_info->{top}->{memp}->{$pid} = $memp;
            next;
        }
    } 
    $mem_info->{top}->{pidList} = \@pidList;
    return $mem_info;
} ;# sub get_mem_info($$$$)

sub get_cpu_info($$$$) {
    my $buffer = shift;
    my $start_time = shift;
    my $stop_time = shift;
    #my $ignore_process = shift || 0;
    my $_known_pidList = shift;
    my @known_pidList = @$_known_pidList;
    
    my $skipped_procs = 'perl|top|bash|ssh|\w+\.sh|cron|oom|scsi_eh';
    $skipped_procs .= '|kworker|kswap|\[\S+\]';
    $skipped_procs .= '|xinetd|rpcbind|syslogd|dbus-daemon';

    my $isStarted = 1;
    if ($start_time =~ /\w+/) {
         $isStarted = 0;
    }
    my $get_virt = 0;
    my $min_pid = 500;
    my $isStopped = 0;
    my %isDfSeen;
    my $pattern1 = '\s*(\d+)\s+\w+\s+\d+\s+\d+\s+(\S+)\s+(\S+)\s+(\S+)\s+\S+';
    $pattern1 .= '\s+(\S+)\s+(\S+)\s+(\d+:\d+(\.\d+)?)\s+(\S+)';

    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";

    my $mem_info;
    my @pidList = ();
    my $found_date = 0;
    foreach my $line (split(/\n/, $buffer)) {
        $line =~ s/\r//g;
        if ( $line =~ /^$date_pattern/) {
            my $_month = $2;
            my $_day = $3;
            my $_time = $4;
            my $_year = $6;
            my $timestamp = $_year . "/" . $_month . "/" . $_day . "/" . $_time;
            $mem_info->{timestamp} = $timestamp;
            $found_date++;
            next;

        }
        next if ($found_date < 1);
        last if ( $line =~ /(\[\S+\]\$\s+|RP\S+#)exit/);

        if ( $line =~ /^$pattern1/) {
            my $pid = $1;
            my $cpup = $5;
            my $time = $7;
            my $proc = $9;
            next if ((@known_pidList) && (!grep(/\b$pid\b/, @known_pidList)));
            next if ($pid <= $min_pid);
            next if ($proc =~ /$skipped_procs/);
            push @pidList, $pid;

            $mem_info->{top}->{proc}->{$pid} = $proc;
            $mem_info->{top}->{cpup}->{$pid} = $cpup;
            $mem_info->{top}->{time}->{$pid} = $time;
            next;
        }
    } 
    $mem_info->{top}->{pidList} = \@pidList;
    return $mem_info;
} ;# sub get_cpu_info($$$$)

sub get_potential_leaks($$$$) {
    my $first_mem_info = shift;
    my $last_mem_info = shift;
    my $total_threshold = shift;
    my $shr_threshold = shift;

    my $skipped_process = '(ssh[d]*|sleep|top|sh|telnet[d]';
    $skipped_process .= '|exec|sftp-server|sshfs|script)';

    my $leak_info;
    my @leak_processList = ();
    my @missPidsFromFirst = ();
    my @missPidsFromLast = ();
    foreach my $pid (@{$first_mem_info->{top}->{pidList}}) {
        my $proc = $first_mem_info->{top}->{proc}->{$pid};
        if ( (!defined($last_mem_info->{top}->{proc}->{$pid})) &&
             ($proc !~ /^${skipped_process}$/) ) {
            push @missPidsFromLast, $pid;
            next;
        }

        my $res1 = $first_mem_info->{top}->{res}->{$pid};
        my $res2 = $last_mem_info->{top}->{res}->{$pid};
        my $shr1 = $first_mem_info->{top}->{shr}->{$pid};
        my $shr2 = $last_mem_info->{top}->{shr}->{$pid};
        my $res_delta = $res2 - $res1;
        my $shr_delta = $shr2 - $shr1;
        my $delta = $res_delta - $shr_delta;
        next if (($delta < $total_threshold) && ($shr_delta < $shr_threshold));
        push @leak_processList, $pid;
        $leak_info->{$pid}->{delta} = $delta;
        #push @leak_procNames, $last_mem_info->{top}->{proc}->{$pid};
    }
    foreach my $pid (@{$last_mem_info->{top}->{pidList}}) {
        my $proc = $last_mem_info->{top}->{proc}->{$pid};
        if ( (!defined($first_mem_info->{top}->{proc}->{$pid})) &&
             ($proc !~ /^${skipped_process}$/) ) {
            push @missPidsFromFirst, $pid;
            next;
        }
    }
    $leak_info->{leak_processList} = \@leak_processList;
    $leak_info->{missPidsFromLast} = \@missPidsFromLast;
    $leak_info->{missPidsFromFirst} = \@missPidsFromFirst;
    return $leak_info;
} ;# sub get_potential_leaks($$$)

sub get_dynamic_leaks($$$$$$$$$$$) {
    my $input_file            = shift;
    my $start_time            = shift;
    my $stop_time             = shift;
    my $start_uptime          = shift;
    my $stop_uptime           = shift;
    my $filtered_leak_info    = shift;
    my $minimum_samples       = shift || 6;
    my $incremental_threshold = shift || 1; #1 MB - minimum resolution from top
    my $false_leak_proc_info  = shift;
    my $pid_proc_map          = shift;
    my $_skippedProcs         = shift;
    my @skippedProcs = @$_skippedProcs;

    my $debug = 1;
    my %isSeen;
    foreach my $_proc (@skippedProcs) {
        $isSeen{$_proc} = 1;
    }

    my $minimum_duration = $minimum_samples;

    my $sampling_size = 6; # size needed to get the trend!

    #for 60 minutes for all processes to initialize
    my $soaking_time = 120;

    my @_tmp = ();
    my @mem_info_list = get_memory_by_interval($input_file,
                                               $start_time,
                                               $stop_time,
                                               \@_tmp);
    my $total_samples = scalar(@mem_info_list);

    #LAST mem info to track wether if pid is live !!!
    my $last_mem_info = @mem_info_list[$#mem_info_list];
    
    my $proc_info;
    my @leaked_pids = ();
    #total elapsed hours
    my $total_duration = ($stop_uptime - $start_uptime) / 60;
    #TODO - change to use minimum samping period (i.e., 5 hours):
    #this is sampling, not real duration
    if ( $total_samples < $minimum_samples ) {
        $proc_info->{leaked_pids} = \@leaked_pids;
        return $proc_info;
    }
    if ( $total_duration < $minimum_duration ) {
        my $msg = "Duration ($total_duration hours) ";
        $msg .= "is too short (need $minimum_duration hours).\n";
        $proc_info->{leaked_pids} = \@leaked_pids;
        return $proc_info;
    }

    my $max_blocks = sprintf("%d", $total_duration / $minimum_duration);

    my $logic =<<END;
    # Identify number of reloads, divide into blocks
    # (note: processes restarted after reload);
    # For each block, dynamically divide into sub-blocks, 
    # smallest sub-block should contain 10 (minimum) samples,
    # and max sub-block contains entire block (all samples);
    
             intv1    intv2      intvY   intvZ
            +-----+ +-----+     +-----+ +-----+
            |sub-1| |sub-1|     |     | |     |
            +-----+ |     |     |     | |     |
            |sub-2| +-----+     |sub-1| |     |
            +-----+ |     |     |     | |     |
            |  .  | |  .  |     |     | |     |
            |  .  | |  .  | ... +-----+ |sub-1|
            |  .  | |  .  |     |     | |     |
            +-----+ |     |     |     | |     |
            |     | +-----+     |sub-2| |     |
            +-----+ |     |     |     | |     |
            |sub-N| |sub-M|     |     | |     |
            +-----+ +-----+     +-----+ +-----+
END
    
    my $sub_blocksize = $total_samples/$sampling_size;

    $proc_info->{leaked_pids} = \@leaked_pids;

    for (my $block_idx=$max_blocks; $block_idx>=1; $block_idx--) {
        ###########################################
        #divide into sub-blocks:
        ###########################################
        # this is the total blocks

        my $samples_per_sub_blocks = $total_samples/$block_idx;
        my $is_equally_divided = $total_samples % $block_idx;

        my ($sub_block_start_idx, $sub_block_stop_idx);
        for (my $sub_blk_idx=0; $sub_blk_idx<$block_idx; $sub_blk_idx++) {
            $sub_block_start_idx = sprintf("%d", $sub_blk_idx *
                                                      $samples_per_sub_blocks);
            $sub_block_stop_idx = sprintf ("%d", ($sub_blk_idx + 1) *
                                                      $samples_per_sub_blocks);
            if ( ($is_equally_divided) && ($sub_blk_idx > 0) ) {
               $sub_block_start_idx -= 1;
            }
            if ( ($is_equally_divided) && ($sub_blk_idx != ($block_idx - 1)) ) {
               $sub_block_stop_idx += 1;
            }

            $sub_block_start_idx = $sub_block_start_idx < 0 ?
                                                    0 : $sub_block_start_idx;
            $sub_block_stop_idx = $sub_block_stop_idx >= $#mem_info_list ?
                                     $#mem_info_list : $sub_block_stop_idx;
            #if it's the last block, use the last sample:
            if ($block_idx eq 1) {
                $sub_block_stop_idx = $#mem_info_list;
            }
            my $sub_block_start_time =
                             $mem_info_list[$sub_block_start_idx]->{timestamp};
            my $sub_block_stop_time =
                              $mem_info_list[$sub_block_stop_idx]->{timestamp};

            ##### wait until fully initialized !!!!!!!!!!!!!!
            my $sub_block_uptime =
                            $mem_info_list[$sub_block_stop_idx]->{top}->{up};
            if ($sub_block_uptime =~ /((\d+) +day[s]*, +)*(\d+:)*(\d+)/) {
                my $up_day = $2 || 0;
                my $up_hour = $3 || 0;
                my $up_min = $4 || 0;
                $up_hour =~ s/://;
                $sub_block_uptime = 24 * 60 * $up_day + 60 * $up_hour + $up_min;
            } elsif ($sub_block_uptime =~ /((\d+) +day[s]?, +)*(\d+) +min/) {
                #5 days, 16 min
                my $up_day = $2 || 0;
                my $up_min = $3 || 0;
                $sub_block_uptime = 24 * 60 * + $up_min;
            }
            next if ($sub_block_uptime < $soaking_time);

            #take minimum_samples (6) samples:
            my @sub_mem_info_list = ();
            my $mem_idx;
            #NOTE: interval - the step counter, not actual sampling interval
            my $interval = 
                   sprintf("%d", $samples_per_sub_blocks/$minimum_samples);
            for (my $idx=0; $idx<=$minimum_samples; $idx++) {
                 $mem_idx = $sub_block_start_idx + $idx * $interval;
                 push @sub_mem_info_list, $mem_info_list[$mem_idx];
            }
            my @mem_leaks = get_sub_block_mem_leaks(\@sub_mem_info_list,
                                                    $incremental_threshold,
                                                    $minimum_samples,
                                                    $filtered_leak_info,
                                                    $false_leak_proc_info,
                                                    $pid_proc_map,
                                                    %isSeen);

            #Find (filter) out the longest (or shortest) interval
            foreach my $leak_data (@mem_leaks) {
                # print "leak_data=$leak_data\n";
                my @leak_lines = split(/;/, $leak_data);
                if (scalar(@leak_lines) < ($minimum_samples-2)) {
                    next;
                }
                foreach my $sample (split(/;/, $leak_data)) {
                    my ($_timestamp,$_pid,$_proc,$_total,$_delta,$_shared) =
                                                           split(/,/, $sample);
                    next if ($_proc =~ /monitor_(cp|cr|sh)/);
                    #$mem_info->{top}->{proc}->{$pid} = $proc;
                    if (!defined($last_mem_info->{top}->{proc}->{$_pid})) {
                        $proc_info->{$_pid}->{terminated} = 1;
                    }
                    if (!$isSeen{$_proc}) {
                        $isSeen{$_proc} = 1;
                        push @leaked_pids, $_pid;
                        $proc_info->{$_pid}->{proc} = $_proc;
                    }
                    #save least and largest!
                    if (!defined($proc_info->{$_pid}->{total_start})) {
                        $proc_info->{$_pid}->{total_start} = $_total;
                        #TODO ????
                        $proc_info->{$_pid}->{shared_start} = $_shared;
                    } else {
                        if ($_total < $proc_info->{$_pid}->{total_start}) {
                            $proc_info->{$_pid}->{total_start} = $_total;
                            #TODO ????
                            $proc_info->{$_pid}->{shared_start} = $_shared;
                        }
                    }
                    if (!defined($proc_info->{$_pid}->{total_stop})) {
                        $proc_info->{$_pid}->{total_stop} = $_total;
                        #TODO ????
                        $proc_info->{$_pid}->{shared_stop} = $_shared;
                    } else {
                        if ($_total > $proc_info->{$_pid}->{total_stop}) {
                            $proc_info->{$_pid}->{total_stop} = $_total;
                            #TODO ????
                            $proc_info->{$_pid}->{shared_stop} = $_shared;
                        }
                    }
                    #save the interval for each process (the longest)
                    if (!defined($proc_info->{$_pid}->{interval})) {
                        $proc_info->{$_pid}->{interval} = $interval;
                    } else {
                        if ($interval < $proc_info->{$_pid}->{interval}) {
                            $proc_info->{$_pid}->{interval} = $interval;
                        }
                    }
                    my $wall_timestamp = convert_to_wall_time($_timestamp);
                    if (!defined($proc_info->{$_pid}->{start_walltime})) {
                        $proc_info->{$_pid}->{start_walltime} = $wall_timestamp;
                        $proc_info->{$_pid}->{stop_walltime} = $wall_timestamp;
                        $proc_info->{$_pid}->{start_time} = $_timestamp;
                        $proc_info->{$_pid}->{stop_time} = $_timestamp;
                    } else {
                        if ($wall_timestamp >
                                      $proc_info->{$_pid}->{stop_walltime}) {
                            $proc_info->{$_pid}->{stop_walltime} =
                                                             $wall_timestamp;
                            $proc_info->{$_pid}->{stop_time} = $_timestamp;
                        }
                    }
                }
            } ;# foreach my $leak_data (@mem_leaks)
        } ;# for (my $sub_blk_idx=0; $sub_blk_idx<$block_idx; $sub_blk_idx++)
    } ;# for (my $block_idx=1; $block_idx<$max_blocks; $block_idx++)
    $proc_info->{leaked_pids} = \@leaked_pids;
    return $proc_info;
} ;# sub get_dynamic_leaks()

sub get_total_timestamps($$) {
    my $input = shift;
    my $start_wtime = shift || 0; #this is the soacking period

    #goto shell and grep (to use less memory)
    my $grep_date_pattern = "$wday *$month";
    #top - 20:53:35 up 2 days,  4:12,  1 user,  load average: 0.05, 0.03, 0.05
    my $grep_top_pattern = 'top +\- +([0-9]+:[0-9]+:[0-9]+)';
    $grep_top_pattern .= ' +up +(.*), +[0-9]+ +user[s]?';

    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";

    my $output = `grep -E "^ *($grep_date_pattern|$grep_top_pattern)" $input`;
    my $info;
    my @time_lines = ();
    my @uptime_lines = ();
    my $started = 0;
    if ($start_wtime eq 0) {
        $started = 1;
    }
    foreach my $time_line (split(/\n/, $output)) {
        #$time_line =~ s/\r//;
        if ( $time_line =~ /^$date_pattern/) {
            my $_time_line =  $6 . '/' . $2 . '/' . $3 . '/' . $4;;
            if ($start_wtime > 0) {
                my $_wtime = convert_to_wall_time($_time_line);
                if ($_wtime >= $start_wtime) {
                    push @time_lines, $_time_line;
                    $started++;
                } else {
                    next;
                }
            } else {
                push @time_lines, $_time_line;
                $started++;
            }
        }
        next if ($started<1);
        if ( $time_line =~ /^\s*$grep_top_pattern/) {
            my $hms = $1;
            my $uptime = $2;
            if ($uptime =~ /((\d+) +day[s]*, +)*(\d+:)*(\d+)/) {
                my $up_day = $2 || 0;
                my $up_hour = $3 || 0;
                my $up_min = $4 || 0;
                $up_hour =~ s/://;
                $uptime = 24 * 60 * $up_day + 60 * $up_hour + $up_min;
            } elsif ($uptime =~ /((\d+) +day[s]?, +)*(\d+) +min/) {
                #5 days, 16 min
                my $up_day = $2 || 0;
                my $up_min = $3 || 0;
                $uptime = 24 * 60 * $up_day + $up_min;
            }
            push @uptime_lines, $uptime;
        }
    }
    $info->{time_lines} = \@time_lines;
    $info->{uptime_lines} = \@uptime_lines;
    return $info;
} ;# sub get_total_timestamps($)

sub get_total_timestamps_v2($) {
    my $input = shift;

    #goto shell and grep (to use less memory)
    my $grep_date_pattern = "$wday *$month";
    #top - 20:53:35 up 2 days,  4:12,  1 user,  load average: 0.05, 0.03, 0.05
    my $grep_top_pattern = 'top +\- +([0-9]+:[0-9]+:[0-9]+) +up +(.*),';
    $grep_top_pattern .= ' +[0-9]+ +user[s]?';

    my $info;
    my @time_lines = ();
    my @uptime_lines = ();
    my ($_month, $_day, $_time, $_year);
    my ($hms, $uptime);
    my ($up_day, $up_hour, $up_min);
    my $_time_line;
    my $block_count = 0;
    my $mem_info;
    my @mem_info_list = ();
    my $buffer;
    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";
    if (!open(FD, $input)) {
        $info->{time_lines} = \@time_lines;
        $info->{uptime_lines} = \@uptime_lines;
        $info->{mem_info_list} = \@mem_info_list;
        return $info;
    }
    my @_tmp = ();
    while (my $line = <FD>) {
        $line =~ s/\r//g;
        if ( $line =~ /^$date_pattern/) {
            $_month = $2;
            $_day = $3;
            $_time = $4;
            $_year = $6;
            $_time_line = $_year . "/" . $_month . "/" . $_day . "/" . $_time;
            push @time_lines, $_time_line;
            if ($block_count eq 1) {
                $mem_info = get_mem_info($buffer, "", "", \@_tmp);
                push @mem_info_list, $mem_info;
            }
            $buffer = $line;
            $block_count++;
            next;
        }
        if ( $line =~ /^\s*$grep_top_pattern/) {
            $hms = $1;
            $uptime = $2;
            if ($uptime =~ /((\d+) +day[s]*, +)*(\d+:)*(\d+)/) {
                $up_day = $2 || 0;
                $up_hour = $3 || 0;
                $up_min = $4 || 0;
                $up_hour =~ s/://;
                $uptime = 24 * 60 * $up_day + 60 * $up_hour + $up_min;
            } elsif ($uptime =~ /((\d+) +day[s]?, +)*(\d+) +min/) {
                #5 days, 16 min
                $up_day = $2 || 0;
                $up_min = $3 || 0;
                $uptime = 24 * 60 * + $up_min;
            }
            push @uptime_lines, $uptime;
            $buffer .= $line;
            next;
        }
        #find first block, and last block, and see leaks
        if ($line =~ /\w+/) {
            $buffer .= $line;
        }
    }
    close(FD);

    $mem_info = get_mem_info($buffer, "", "", \@_tmp);
    push @mem_info_list, $mem_info;

    $info->{time_lines} = \@time_lines;
    $info->{uptime_lines} = \@uptime_lines;
    $info->{mem_info_list} = \@mem_info_list;
    return $info;
} ;# sub get_total_timestamps_v2($)

sub get_sub_block_mem_leaks($$$$$$$) {
    my $_mem_info_list        = shift;
    my $incremental_threshold = shift || 1; #1MB: minimum resolution from top
    my $minimum_samples       = shift || 6;
    my $filtered_leak_info    = shift;
    my $false_leak_proc_info  = shift;
    my $pid_proc_map          = shift;
    my %isKnownLeaks          = shift;

    my @potential_processList = ();
    foreach my $pid (@{$filtered_leak_info->{leak_processList}}) {
        push @potential_processList, $pid;
    }
    foreach my $pid (@{$filtered_leak_info->{missPidsFromFirst}}) {
        push @potential_processList, $pid;
    }
    foreach my $pid (@{$filtered_leak_info->{missPidsFromLast}}) {
        push @potential_processList, $pid;
    }

    my @mem_info_list = @$_mem_info_list;

    my $first_no = 0;
    my $last_no = $#mem_info_list;
    my $first_mem_info = $mem_info_list[$first_no];
    my $last_mem_info = $mem_info_list[$last_no];
    my $wtime1 = convert_to_wall_time($first_mem_info->{timestamp});

    #my @pidList = @{$first_mem_info->{top}->{pidList}};
    my @pidList = ();
    my @procList = ();

    my $total_threshold = $incremental_threshold * ($minimum_samples -1);
    foreach my $pid (@{$first_mem_info->{top}->{pidList}}) {
        next if (!defined($last_mem_info->{top}->{proc}->{$pid}));
        my $_proc = $last_mem_info->{top}->{proc}->{$pid};
        next if ($isKnownLeaks{$_proc});
        # skip process that either has no leaks,
        # or process started/stopped in-between
        next if ((scalar(@potential_processList)) &&
                           (!grep(/\b$pid\b/, @potential_processList)));
        my $res1 = $first_mem_info->{top}->{res}->{$pid};
        my $res2 = $last_mem_info->{top}->{res}->{$pid};
        my $delta = $res2 - $res1;
        next if ($delta < $total_threshold);

        #find out from $false_leak_proc_info if the process need special treatment
        #if yes, ignore if duration is less than minimum hour
        my $proc = $first_mem_info->{top}->{proc}->{$pid};
        if (defined($pid_proc_map->{$pid})) {
            $proc = $pid_proc_map->{$pid};
        }
        my $_proc = $proc;
        if (defined($false_leak_proc_info->{$_proc}->{minimum_hour}) &&
                              defined($first_mem_info->{timestamp}) &&
                               defined($last_mem_info->{timestamp})) {
            my $minimum_hour = $false_leak_proc_info->{$_proc}->{minimum_hour};
            my $wtime2 = convert_to_wall_time($last_mem_info->{timestamp});
            my $delta = $wtime2 - $wtime1;
            $minimum_hour *= 3600;
            next if ($delta < $minimum_hour);
        }
        push @pidList, $pid;
    }
    my $delta_total = 0;
    my @new_pidList = ();
    my @new_procList = ();
    my ($pid, $proc, $res1, $res2, $shr1, $shr2);
    my ($timestamp, $delta_res, $delta_shr);
    my $step = 1;
    my @mem_leaks = ();
    foreach my $pid (@pidList) {
        my $leak_data = "";
        my $continuous_leaking = 1;
        for (my $i=$first_no; $i<=$last_no; $i++) {
            my $mem_info1 = $mem_info_list[$i];
            my $j = $i + $step;
            my $mem_info2 = $mem_info_list[$j];

            next if (!defined($mem_info1->{top}->{proc}->{$pid}));
            $proc = $mem_info1->{top}->{proc}->{$pid};
            next if ($proc =~ /oom\.sh/);
            next if ($proc =~ /crond/);
            next if (!defined($mem_info2->{top}->{proc}->{$pid}));
            my $proc2 = $mem_info2->{top}->{proc}->{$pid};
            if ( ($proc ne $proc2) && ($proc2 =~ /\w+/) ) {
                next;
            }
            $shr1 = $mem_info1->{top}->{shr}->{$pid};
            $res1 = $mem_info1->{top}->{res}->{$pid};
            $shr2 = $mem_info2->{top}->{shr}->{$pid};
            $res2 = $mem_info2->{top}->{res}->{$pid};

            $shr1 = sprintf ("%.2f", $shr1/1.0);
            $res1 = sprintf ("%.2f", $res1/1.0);
            $shr2 = sprintf ("%.2f", $shr2/1.0);
            $res2 = sprintf ("%.2f", $res2/1.0);
            $delta_res = $res2 - $res1;
            $delta_shr = $shr2 - $shr1;
            $delta_res = sprintf("%.2f", $delta_res/1.0);
            $delta_shr = sprintf("%.2f", $delta_shr/1.0);

            #$delta_total = $delta_res - $delta_shr;
            $delta_total = $delta_res;

            #next if ($delta_res < $incremental_threshold);
            #next if ($delta_total < $incremental_threshold);
            #if ($delta_res < $incremental_threshold) {}
            if ($delta_total < $incremental_threshold) {
                #$timestamp = $mem_info1->{timestamp};
                $continuous_leaking = 0;
                last;
            }
            $timestamp = $mem_info1->{timestamp};
            if ( $leak_data eq "") {
                $leak_data .= "$timestamp,$pid,$proc,$res1,";
                $leak_data .= "$delta_res,$delta_shr";
            } else {
                $leak_data .= ";" . 
                            "$timestamp,$pid,$proc,$res1,$delta_res,$delta_shr";
            }
            if ( $i eq $last_no) {
                $timestamp = $mem_info2->{timestamp};
                $leak_data .= ";" . 
                            "$timestamp,$pid,$proc,$res1,$delta_res,$delta_shr";
            }
        } ;# foreach (my $i=$first_no; $i<$last_no; $i++)
        if ($continuous_leaking) {
            #ensure the current time span is larger than minimum_hour
            my $_proc = $first_mem_info->{top}->{proc}->{$pid};
            if (defined($pid_proc_map->{$pid})) {
                $_proc = $pid_proc_map->{$pid};
            }
            my $minimum_hour = $false_leak_proc_info->{$_proc}->{minimum_hour};
            my $wtime2 = convert_to_wall_time($timestamp);
            my $delta = $wtime2 - $wtime1;
            $minimum_hour *= 3600;
            next if ($delta < $minimum_hour);
            #TODO - will use trend
            push @mem_leaks, $leak_data if ($leak_data ne "");
        }
    } ;# foreach my $pid (@pidList)
    return @mem_leaks;
} ;# sub get_sub_block_mem_leaks($$$$$$)

sub get_mem_info_by_pid($$$$$) {
    my $input       = shift;
    my $start_time  = shift;
    my $stop_time   = shift;
    my $interval    = shift;
    my $pid         = shift;

    my $isStarted = 0;
    my $isStopped = 0;

    my $mem_info;
    my $found_date = 0;
    my $next_wall_timestamp = 0;
    my @timestamps = ();
    my $count = 0;
    my $next_step = 0;
    my $collect_proc_data = 0;

    my @mem_leaks = ();
    my $leak_data = "";
    my $idx = 0;

    my ($proc, $res, $prev_res, $shr, $prev_shr);
    my ($timestamp, $prev_timestamp, $delta_res, $delta_shr);
    #my ($cpup, $memp, $cpu_time);
    my ($cpup, $cpu_time);

    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";
    my $pid_pattern = "\\s*${pid}\\s+\\w+\\s+\\d+\\s+\\d+\\s+(\\S+)\\s+(\\S+)";
    $pid_pattern .= '\s+(\S+)\s+\S+\s+(\S+)\s+(\S+)\s+';
    $pid_pattern .= '(\d+:\d+(\.\d+)?)\s+(\S+)';
    if (!open(FD, $input)) {
        #die "cannot open $input: $!\n";;
        return @mem_leaks;
    }
    while (my $line = <FD>) {
        $line =~ s/\r//g;
        $line =~ s/\n//g;
        if ( $line =~ /^$date_pattern/) {
            my $_month = $2;
            my $_day = $3;
            my $_time = $4;
            my $_year = $6;
            $timestamp = $_year . "/" . $_month . "/" . $_day . "/" . $_time;

            $collect_proc_data = 0;
            if ( ($timestamp eq $start_time) && ($timestamp =~ /\w+/) ) {
                $isStarted = 1;
                $next_step += $interval;
                push @timestamps, $timestamp;
                $collect_proc_data = 1;
            }
            if ($isStarted) {
                $count++;
                $found_date++;
                if ($count == $next_step ) {
                    push @timestamps, $timestamp;
                    $next_step += $interval;
                    $collect_proc_data = 1;
                }
            }
            last if ($isStopped);
            if ( ($timestamp eq $stop_time) && ($stop_time =~ /\w+/) ) {
                $isStopped = 1;
            }
        }
        next if ($found_date < 1);
        next if ($isStarted eq 0);
        next if ($collect_proc_data < 1);
        #last if ( $line =~ /(\[\S+\]\$\s+|RP\S+#)exit/);

        if ( $line =~ /^$pid_pattern/) {
            # my $virt = $1;
            $res = $2;
            $shr = $3;
            $cpup = $4;
            #$memp = $5;
            $cpu_time = $6;
            $proc = $8;
            if ( $res =~ /m/ ) {
                $res =~ s/m//g;
                $res *= 1024;
            } elsif ( $res =~ /g/ ) {
                $res =~ s/g//g;
                $res *= 1024 * 1024;
            } elsif ( $res =~ /t/i ) {
                $res =~ s/t//gi;
                $res *= 1024 * 1024 * 1024;
            }
            if ( $shr =~ /m/ ) {
                $shr =~ s/m//g;
                $shr *= 1024;
            } elsif ( $shr =~ /g/i ) {
                $shr =~ s/g//gi;
                $shr *= 1024 * 1024;
            } elsif ( $shr =~ /t/i ) {
                $shr =~ s/t//gi;
                $shr *= 1024 * 1024 * 1024;
            }
            $shr = sprintf ("%.2f", $shr/1024.0);
            $res = sprintf ("%.2f", $res/1024.0);
            if ($idx eq 0) {
                $leak_data = "$timestamp,$pid,$proc,$res,-,$shr";
            } else {
                $delta_res = $res - $prev_res;
                $delta_shr = $shr - $prev_shr;
                $delta_res = sprintf("%.2f", $delta_res/1.0);
                if ($leak_data eq "") {
                    $leak_data .= "$timestamp,$pid,$proc,$res,$delta_res,$shr";
                } else {
                    $leak_data .= ";" . 
                                 "$timestamp,$pid,$proc,$res,$delta_res,$shr";
                }
            }
            $prev_timestamp = $timestamp;
            $prev_res = $res;
            $prev_shr = $shr;
            $idx++;
            next;
        }
    } ;#while (my $line = <FD>)
    close(FD);
    #TODO - change to string (vs array)!
    push @mem_leaks, $leak_data if ($leak_data ne "");
    #$mem_info->{timestamps} = \@timestamps;
    #return $mem_info;
    return @mem_leaks;
} ;# sub get_mem_info_by_pid($$$$$)

#To check if there is any fluctuation
sub verify_mem_info_by_pid($$$$) {
    my $input      = shift;
    my $start_time = shift;
    my $stop_time  = shift;
    my $pid        = shift;

    my $isStarted = 0;
    my $isStopped = 0;

    my $mem_info;
    my $found_date = 0;
    my @timestamps = ();
    my $count = 0;
    my $next_step = 0;
    my $collect_proc_data = 0;

    my @mem_leaks = ();
    my $leak_data = "";
    my $idx = 0;
    my $interval = 1;

    my ($proc, $res, $prev_res, $shr, $prev_shr);
    my ($timestamp, $prev_timestamp, $delta_res, $delta_shr);

    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";
    my $pid_pattern = "\\s*${pid}\\s+\\w+\\s+\\d+\\s+\\d+\\s+(\\S+)\\s+(\\S+)";
    $pid_pattern .= '\s+(\S+)\s+\S+\s+(\S+)\s+(\S+)\s+';
    $pid_pattern .= '(\d+:\d+\.\d+)\s+(\S+)';
    if (!open(FD, $input)) {
        return @mem_leaks;
    }
    while (my $line = <FD>) {
        $line =~ s/\r//g;
        $line =~ s/\n//g;
        if ( $line =~ /^$date_pattern/) {
            my $_month = $2;
            my $_day = $3;
            my $_time = $4;
            my $_year = $6;
            $timestamp = $_year . "/" . $_month . "/" . $_day . "/" . $_time;

            $collect_proc_data = 0;
            if ( ($timestamp eq $start_time) && ($timestamp =~ /\w+/) ) {
                $isStarted = 1;
                $next_step += $interval;
                push @timestamps, $timestamp;
                $collect_proc_data = 1;
            }
            if ($isStarted) {
                $count++;
                $found_date++;
                if ($count == $next_step ) {
                    push @timestamps, $timestamp;
                    $next_step += $interval;
                    $collect_proc_data = 1;
                }
            }
            last if ($isStopped);
            if ( ($timestamp eq $stop_time) && ($stop_time =~ /\w+/) ) {
                $isStopped = 1;
            }
        }
        next if ($found_date < 1);
        next if ($isStarted eq 0);
        next if ($collect_proc_data < 1);

        if ( $line =~ /^$pid_pattern/) {
            $res = $2;
            $shr = $3;
            $proc = $7;
            if ( $res =~ /m/ ) {
                $res =~ s/m//g;
                $res *= 1024;
            } elsif ( $res =~ /g/ ) {
                $res =~ s/g//g;
                $res *= 1024 * 1024;
            } elsif ( $res =~ /t/i ) {
                $res =~ s/t//gi;
                $res *= 1024 * 1024 * 1024;
            }
            if ( $shr =~ /m/ ) {
                $shr =~ s/m//g;
                $shr *= 1024;
            } elsif ( $shr =~ /g/ ) {
                $shr =~ s/g//g;
                $shr *= 1024 * 1024;
            } elsif ( $shr =~ /t/i ) {
                $shr =~ s/t//gi;
                $shr *= 1024 * 1024 * 1024;
            }
            $shr = sprintf ("%.2f", $shr/1024.0);
            $res = sprintf ("%.2f", $res/1024.0);
            if ($idx eq 0) {
                $leak_data = "$timestamp,$pid,$proc,$res,-,$shr";
            } else {
                $delta_res = $res - $prev_res;
                $delta_shr = $shr - $prev_shr;
                if ($delta_res < 0) {
                    #print "Memory fluctuating - $proc($pid) ignored\n";
                    return ();
                }
                $delta_res = sprintf("%.2f", $delta_res/1.0);
                if ($leak_data eq "") {
                    $leak_data .= "$timestamp,$pid,$proc,$res,$delta_res,$shr";
                } else {
                    $leak_data .= ";" . 
                                 "$timestamp,$pid,$proc,$res,$delta_res,$shr";
                }
            }
            $prev_timestamp = $timestamp;
            $prev_res = $res;
            $prev_shr = $shr;
            $idx++;
            next;
        }
    } ;#while (my $line = <FD>)
    close(FD);
    #TODO - change to string (vs array)!
    push @mem_leaks, $leak_data if ($leak_data ne "");
    return @mem_leaks;
} ;# sub verify_mem_info_by_pid($$$$)

sub get_blocks_by_reload($) {
    my $time_info = shift;

    my @timestamps = @{$time_info->{time_lines}};
    my @up_times = @{$time_info->{uptime_lines}};

    my $last_uptime = -1;
    my $last_timestamp = "";
    my @reload_starttimestamps = ();
    my @reload_startuptimes= ();
    my @reload_stopuptimes = ();
    my @reload_stoptimestamps = ();
    my @reload_stoptimes= ();
    my $reload_info;
    for (my $i=0; $i <= $#timestamps; $i++) {
        my $timestamp = @{$time_info->{time_lines}}[$i];
        my $uptime = @{$time_info->{uptime_lines}}[$i];
        if ($last_uptime eq -1) {
            push @reload_starttimestamps, $timestamp;
            push @reload_startuptimes, $uptime;
        } else {
            if ($last_uptime > $uptime) {
                push @reload_starttimestamps, $timestamp;
                push @reload_startuptimes, $uptime;
                push @reload_stoptimestamps, $last_timestamp;
                push @reload_stopuptimes, $last_uptime;
            }
        }
        $last_timestamp = $timestamp;
        $last_uptime = $uptime;
    }
    push @reload_stoptimestamps, $last_timestamp;
    push @reload_stopuptimes, $last_uptime;
    $reload_info->{reload_starttimestamps} = \@reload_starttimestamps;
    $reload_info->{reload_startuptimes} = \@reload_startuptimes;
    $reload_info->{reload_stoptimestamps} = \@reload_stoptimestamps;
    $reload_info->{reload_stopuptimes} = \@reload_stopuptimes;

    return $reload_info;
} ;# sub get_blocks_by_reload($)

#gets cpu hogs, returns in a hash
sub get_cpu_hogs($$$) {
    my $_cpu_info_list = shift;
    my $cpu_threshold = shift;
    my $osType_n = shift;

    my @cpu_info_list = @$_cpu_info_list;
    my $cpu_sampling_interval = 1800;

    my $step = 1;
    my $cpu_info1 = $cpu_info_list[$#cpu_info_list - $step - $step];
    my $cpu_info2 = $cpu_info_list[$#cpu_info_list - $step];
    my $cpu_info3 = $cpu_info_list[$#cpu_info_list];
    my $timestamp1 = $cpu_info1->{timestamp};
    my $timestamp2 = $cpu_info2->{timestamp};
    my $timestamp3 = $cpu_info3->{timestamp};
    my $w_timestamp1 = convert_to_wall_time($timestamp1);
    my $w_timestamp2 = convert_to_wall_time($timestamp2);
    my $w_timestamp3 = convert_to_wall_time($timestamp3);
    my $w_delta1 = $w_timestamp2 - $w_timestamp1;
    my $w_delta2 = $w_timestamp3 - $w_timestamp2;

    my $skipdProcesses = 'sleep|top|ssh|sshfs|vpe(_main)?';
    $skipdProcesses .= '|fia_driver|nf_producer';

    #sort CPU consumers
    my @cpu_data = ();
    my @pidList = @{$cpu_info3->{top}->{pidList}};
    foreach my $pid (@pidList) {
        my $proc = $cpu_info3->{top}->{proc}->{$pid};
        next if ($proc =~ /^($skipdProcesses)$/i);
        if ( (!defined($cpu_info1->{top}->{proc}->{$pid})) ||
             ($proc ne $cpu_info1->{top}->{proc}->{$pid}) ) {
            next;
        }

        my $cpup1 = $cpu_info1->{top}->{cpup}->{$pid};
        my $cpup2 = $cpu_info2->{top}->{cpup}->{$pid};
        my $cpup3 = $cpu_info3->{top}->{cpup}->{$pid};
        next if (($cpup1 < $cpu_threshold) ||
                 ($cpup2 < $cpu_threshold) ||
                 ($cpup3 < $cpu_threshold));

        my $cput1 = $cpu_info1->{top}->{time}->{$pid};
        my $cput2 = $cpu_info2->{top}->{time}->{$pid};
        my $cput3 = $cpu_info3->{top}->{time}->{$pid};
        my ($min1, $sec1) = split(/:/, $cput1);
        my ($min2, $sec2) = split(/:/, $cput2);
        my ($min3, $sec3) = split(/:/, $cput3);
        $cput1 = $min1 * 60 + $sec1;
        $cput2 = $min2 * 60 + $sec2;
        $cput3 = $min3 * 60 + $sec3;
        my $cput_delta1 = $cput2 - $cput1;
        my $cput_delta2 = $cput3 - $cput2;
        next if ($cput_delta1 < ($w_delta1 * $cpu_threshold/100));
        next if ($cput_delta2 < ($w_delta2 * $cpu_threshold/100));
        $cpup1 = sprintf ("%.1f", $cpup1);
        $cpup2 = sprintf ("%.1f", $cpup2);
        $cpup3 = sprintf ("%.1f", $cpup3);
        $cput1 = sprintf ("%.1f", $cput1);
        $cput2 = sprintf ("%.1f", $cput2);
        $cput3 = sprintf ("%.1f", $cput3);
        my $_line = "$pid,$proc,$timestamp1,$cpup1,$cput1,";
        $_line .= "$timestamp2,$cpup2,$cput2,";
        $_line .= "$timestamp3,$cpup3,$cput3";
        push @cpu_data, $_line;
    } ;# foreach my $pid (@pidList)
    my @sorted_cpu_data = reverse sort { (split(/,/, $a))[3]
                                      <=> (split (/,/, $b))[3] } @cpu_data;
    return @sorted_cpu_data;
} ;# sub get_cpu_hogs($$)

#extracts last X samplings
#This is just for checking CPU hogs
sub get_memory_last_samplings($$) {
    my $input = shift;
    my $number_samplings = shift || 2;

    my @mem_info_list = ();
    if (!open(FD, $input)) {
        return @mem_info_list;
    }
    my $count = 0;
    my @buffers;
    my $found_date = 0;

    my ($start_time, $stop_time);
    my ($start_wtime, $stop_wtime);

    #goto shell and grep (to use less memory)
    my $output = `grep -E "^ *$wday *$month" $input`;
    my @time_lines = ();
    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";
    foreach my $time_line (split(/\n/, $output)) {
        #$time_line =~ s/\r//;
        if ( $time_line =~ /^$date_pattern/) {
            push @time_lines, $6 . '/' . $2 . '/' . $3 . '/' . $4;;
        }
    }
    if (($number_samplings > 1) &&
        (scalar(@time_lines) < $number_samplings) ) {
        return @mem_info_list;
    }

    #find out the interval and total samplings
    #my ($wtime1, $wtime2);
    #$wtime1 = convert_localtime_to_wall_time($time_lines[0]);
    #$wtime2 = convert_localtime_to_wall_time($time_lines[$#time_lines]);
    #my $real_sampling_interval = ($wtime2 - $wtime1)/($#time_lines - 1);

    my $first_no = 0;
    my $last_no = $#time_lines;
    if ($number_samplings >= 2) {
        $first_no = $number_samplings >= $#time_lines ?
                         0 : ($#time_lines - $number_samplings + 1);
    }
    $start_time = $time_lines[$first_no];
    $stop_time = $time_lines[$last_no];
    $start_wtime = convert_to_wall_time($start_time);
    $stop_wtime = convert_to_wall_time($stop_time);
    my $isStarted = 1;
    if ((defined($start_time)) && ($start_time =~ /\w+/)) {
         $isStarted = 0;
    }
    my $isStopped = 0;

    my $timestamp = "";
    my @timestamps;
    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";
    while (my $line = <FD>) {
        if ( $line =~ /^$date_pattern/) {
            my $_month = $2;
            my $_day = $3;
            my $_time = $4;
            my $_year = $6;
            $timestamp = $_year . "/" . $_month . "/" . $_day . "/" . $_time;
            if ( ($timestamp eq $start_time) && ($timestamp =~ /\w+/) ) {
                $isStarted = 1;
            }
            if ($isStarted) {
                $count++;
                $found_date++;
            }
            push @timestamps, $timestamp;
            last if ($isStopped);
            if ( ($timestamp eq $stop_time) && ($stop_time =~ /\w+/) ) {
                $isStopped = 1;
            }
        }
        next if ($found_date < 1);
        next if ($isStarted eq 0);
        last if ( $line =~ /(\[\S+\]\$\s+|RP\S+#)exit/);
        if ($line =~ /\w+/) {
            $buffers[$count] .= $line;
        }
    }
    close(FD);

    my $no = $#buffers + 1;
    my $mem_data;
    $mem_data->{timestamps} = \@timestamps;

    my @_tmp = ();
    for (my $i = 1; $i<=$no; $i++) {
        my $buffer = $buffers[$i];
        next if (!$buffer);
        next if ($buffer !~ /\w+/);
        my $mem_info = get_cpu_info($buffer, "", "", \@_tmp);
        push @mem_info_list, $mem_info;
    }
    $mem_data->{mem_info_list} = \@mem_info_list;
    return @mem_info_list;
} ; # sub get_memory_last_samplings($$)


#extracts last X samplings
sub get_disk_last_samplings($$) {
    my $input = shift;
    my $number_samplings = shift || 2;

    my @mem_info_list = ();
    if(!open(FD, $input)) {
        return @mem_info_list;
    }
    my $count = 0;
    my @buffers;
    my $found_date = 0;

    my ($start_time, $stop_time);
    my ($start_wtime, $stop_wtime);

    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";

    #goto shell and grep (to use less memory)
    my $output = `grep -E "^ *$wday *$month" $input`;
    my @time_lines = ();
    foreach my $time_line (split(/\n/, $output)) {
        #$time_line =~ s/\r//;
        if ( $time_line =~ /^$date_pattern/) {
            push @time_lines, $6 . '/' . $2 . '/' . $3 . '/' . $4;;
        }
    }
    if (($number_samplings > 1) &&
        (scalar(@time_lines) < $number_samplings) ) {
        return @mem_info_list;
    }

    my $first_no = 0;
    my $last_no = $#time_lines;
    if ($number_samplings >= 2) {
        $first_no = $number_samplings >= $#time_lines ?
                    0 : ($#time_lines - $number_samplings + 1);
    }
    $start_time = $time_lines[$first_no];
    $stop_time = $time_lines[$last_no];
    $start_wtime = convert_to_wall_time($start_time);
    $stop_wtime = convert_to_wall_time($stop_time);
    my $isStarted = 1;
    if ((defined($start_time)) && ($start_time =~ /\w+/)) {
         $isStarted = 0;
    }
    my $isStopped = 0;

    my $timestamp = "";
    my @timestamps;
    while (my $line = <FD>) {
        if ( $line =~ /^$date_pattern/) {
            my $_month = $2;
            my $_day = $3;
            my $_time = $4;
            my $_year = $6;
            $timestamp = $_year . "/" . $_month . "/" . $_day . "/" . $_time;
            if ( ($timestamp eq $start_time) && ($timestamp =~ /\w+/) ) {
                $isStarted = 1;
            }
            if ($isStarted) {
                $count++;
                $found_date++;
            }
            push @timestamps, $timestamp;
            last if ($isStopped);
            if ( ($timestamp eq $stop_time) && ($stop_time =~ /\w+/) ) {
                $isStopped = 1;
            }
        }
        next if ($found_date < 1);
        next if ($isStarted eq 0);
        last if ( $line =~ /(\[\S+\]\$\s+|RP\S+#)exit/);
        if ($line =~ /\w+/) {
            $buffers[$count] .= $line;
        }
    }
    close(FD);

    my $no = $#buffers + 1;
    my $mem_data;
    $mem_data->{timestamps} = \@timestamps;

    for (my $i = 1; $i<=$no; $i++) {
        my $buffer = $buffers[$i];
        next if (!$buffer);
        next if ($buffer !~ /\w+/);
        my $mem_info = get_ng_df_info($buffer);
        push @mem_info_list, $mem_info;
    }
    $mem_data->{mem_info_list} = \@mem_info_list;
    return @mem_info_list;
} ;#sub get_disk_last_samplings($$)


sub get_ng_df_info($$$) {
    my $buffer = shift;
    my $start_time = shift;
    my $stop_time = shift;

    #my $wday = '(Mon|Tue|Wed|Thu|Fri|Sat|Sun)';
    #my $month = '(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)';

    my $isStarted = 1;
    if ((defined($start_time)) && ($start_time =~ /\w+/)) {
         $isStarted = 0;
    }
    my $isStopped = 0;
    my %isDfSeen;

    my $mem_info;
    my $found_date = 0;
    my @mountList = ();

    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";

    my $pattern1 = 'top +\- +(\d{2}:\d{2}:\d{2}) +up +(.*), +\d+ +user[s]*,';
    $pattern1 .= ' +load average: +(\d+\.\d+), (\d+\.\d+), (\d+\.\d+)';

    my $df_pattern = '\s*(\/dev\/\S+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)';
    $df_pattern .= '%\s+(\/\S+)\s*';

    foreach my $line (split(/\n/, $buffer)) {
        $line =~ s/\r//g;
        $line =~ s/\n//g;
        if ( $line =~ /^$date_pattern/) {
            #Sat Apr 5 00:00:01 EDT 2014
            #Tue Apr 15 08:44:59 GMT+5 2014
            my $_month = $2;
            my $_day = $3;
            my $_time = $4;
            my $_year = $6;
            my $timestamp = $_year . "/" . $_month . "/" . $_day . "/" . $_time;
            $mem_info->{timestamp} = $timestamp;
            $found_date++;
            next;
        }
        next if ($found_date < 1);
        last if ( $line =~ /(\[\S+\]\$\s+|RP\S+#)exit/);

        if ( $line =~ /^$pattern1/) {
            my $now = $1;
            my $up = $2;
            my $lavg1 = $3;
            my $lavg5 = $4;
            my $lavg15 = $5;
            $mem_info->{top}->{now} = $now;
            $mem_info->{top}->{up} = $up;
            $mem_info->{top}->{load1} = $lavg1;
            $mem_info->{top}->{load5} = $lavg5;
            $mem_info->{top}->{load15} = $lavg15;
            next;
        }

        ###############################################################
        #output of df
        ###############################################################
        if ($line =~ /^${df_pattern}$/) {
            my $filesys    = $1;
            my $total      = $2;
            my $used       = $3;
            my $avail      = $4;
            my $usepct     = $5;
            my $mount_name = $6;
            #Make sure we don;t count twice
            next if ($filesys =~ /\/dev\/ram\d+/);
            next if ($mount_name =~ /\/dev\/sh+/);
            if (!$isDfSeen{$mount_name}) {
                $isDfSeen{$mount_name} = 1;
                $mem_info->{df}->{$mount_name}->{total} = $total;
                $mem_info->{df}->{$mount_name}->{used} = $used;
                $mem_info->{df}->{$mount_name}->{avail} = $avail;
                $mem_info->{df}->{$mount_name}->{usepct} = $usepct;
                push @mountList, $mount_name;
            }
            next;
        }
    } ;#foreach my $line (split(/\n/, $buffer))
    $mem_info->{mountList} = \@mountList;
    return $mem_info;
} ;#sub get_ng_df_info

#convert spelled months to numerical
sub convert_localtime_to_wall_time($) {
    my $timestamp = shift;
    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";
    if ( $timestamp =~ /^$date_pattern/) {
        my $_timestamp = $6 . "/" . $2 . "/" . $3 . "/" . $4;
        return convert_to_wall_time($_timestamp);
    }
    return $timestamp;
}

sub get_top_header($$) {
    my $input = shift;
    my $my_timestamp = shift;

    my $found_date = 0;
    my $output = "";
    if (!open (FD, $input)) {
        #print "cannot open $input: $!\n";;
        return $output;
    }
    #  PID USER      PR  NI  VIRT  RES  SHR S %CPU %MEM    TIME+  COMMAND
    my $pid_header ='PID\s+USER\s+PR\s+NI\s+VIRT\s+RES';
    $pid_header .= '\s+SHR.*%CPU\s+%MEM\s+TIME(\+)?\s+COMMAND';
    my $date_pattern = "\\s*$wday\\s+$month\\s+(\\d+)\\s+";
    $date_pattern .= "(\\d{2}:\\d{2}:\\d{2})\\s+(\\S+\\s+)*(\\d{4})";
    while (my $line = <FD>) {
        $line =~ s/[\r\n]//g;
        if ( $line =~ /^$date_pattern/) {
            my $_month = $2;
            my $_day = $3;
            my $_time = $4;
            my $_year = $6;
            my $timestamp = $_year . "/" . $_month . "/" . $_day . "/" . $_time;
            #timestamp=2016/Jan/22/19:15:22
            if ($my_timestamp eq $timestamp) {
                $found_date++;
            }
            next if (!$found_date);
        }
        if ($found_date) {
            do {
                $line = <FD>;
                $line =~ s/[\r\n]//g;
                $output .= $line . "\n";
                #$output .= $line . "\n" if ($line !~ /$pid_header/);
            } while ( $line !~ /$pid_header/);
            for (my $i=0; $i<5; $i++) {
                $line = <FD>;
                $line =~ s/[\r\n]//g;
                $output .= $line . "\n";
            }
            last;
        }
    }
    close(FD);
    return $output;
} ;# sub get_top_header($$)


