#! /usr/bin/python
#------------------------------------------------------------------
# package.py -
#
# Copyright (c) 2015-2018 by cisco Systems, Inc.
# All rights reserved.
#------------------------------------------------------------------
#

import commands 
import subprocess
import os
import sys
import re
import platform
import provides_hint
import time
import inspect
import logging
import shutil
import tempfile

INSTALL_TMP_STAGING_PATH="/install/tmp_staging"
ISO_BOOT_INITRD="/boot/initrd.img"
SYSTEM_IMAGE_ISO="/iso/system_image.iso"
DEFAULT_RPM_PATH="default_rpm_path"
SIGNED_RPM_PATH='signed_rpm_path'                                     
SIGNED_NCS5500_RPM_PATH="signed_ncs5500_rpm_path"
SIGNED_651_NCS5500_RPM_PATH="signed_651_ncs5500_rpm_path"
FREE_SPACE_REQUIRED_FOR_SYS_ISO=1.5*1024*1024*1024 #1.5 GB for system_image

logger = logging.getLogger('install_logger')

def is_cisco_pkg(platform, pkg_name):
    """ Check if the given pkg name is cisco pkg """
    if platform in pkg_name or 'iosxr' in pkg_name:
        return True

def is_tp_pkg(platform, pkg_name):
    """ Return true if not cisco pkg """
    if not is_cisco_pkg(platform, pkg_name):
        return True

def is_tp_smu(platform, pkg_name):
    """ Return true if not cisco pkg """
    if not is_cisco_pkg(platform, pkg_name) and \
        re.search('CSC[a-z][a-z]\d{5}',pkg_name):
        return True

def get_vm_of_tpsmu(release):
    """ Given release field of TP SMU return VM  """
    vmtyoe = 'None'
    vmtype = release.split('.')[-1]
    return vmtype

class Package:
    def __init__(self):
        self.type        = None # MANDATORY, OPTIONAL, SMU, SP, FP,
        self.format      = None # ISO,TAR,RPM
        self.name        = None # 
        self.version     = None # d.d.d.dx
        self.release     = None # d.d.d.dx
        self.arch        = None # x86_64,ppc,arm
        self.name_on_router = None # As seen on router
        self.filename    = None # Name as in repository
        self.size        = 0    # files size in repository
        self.packageid   = None # Only in case of SMU/SP/FP e.g. CSCub12345
        self.requires    = []   # List of packages which are prereq
        self.obsoletes   = []   # Dict 
        self.provides    = []
        

class PackageInfo:
    def __init__(self,options,input=None):
        logger.info("Collecting software state..")
        self.platform_pkgformat_0 = ['ncs4k']
        self.img_name = {
                           'asr9k':'asr9k-mini-x64.iso',
                        }

        self.cal_tool_name = "/pkg/sbin/admin-cli-proxy-xr_static"
        self.options = options
        self.xractive_package = []# Retain the package on SU
        self.calv_active_package = []# Retain the package on SU
        self.arch = platform.machine()
        self.platform = self.get_platform()
        self.supported_archs = self.get_supported_archs()
        self.to_version = input
        self.active_version = None
        self.calv_active_version = None
        self.input_package       = []        
        self.active_package      = self.get_active()
        self.inactive_package    = self.get_inactive()
        self.repo_package = []        # This is external repo
        self.optional_package    = [] # Used during SU + optional packages
        self.downloaded_package  = [] 
        self.to_download_package = []
        self.to_add = []
        self.to_activate = []
        self.to_deactivate = []
        self.rpms_in_bundle = []
        self.required_package = []
        self.get_calvados_active()
        self.get_calvados_inactive()
        self.rpminrepo = self.get_rpm_in_glrepo()

        # Check if we need following three 
        self.active_provides  = self.get_active_provides()
        self.inactive_provides = self.get_inactive_provides()
        self.inactive_requires = self.get_inactive_requires()
        self.obsoleted_package   = self.get_inactive_obsoletes()

        # Keep track of all rpms for which req is calculated
        self.seen_prereq = []
        self.skipped_as_active = []
        self.skipped_as_inactive = []

        self.tmpstage = None
        self.hints = provides_hint.PkgHintsMdata()

        self.bundle_iso_pkgs = []
        self.active_cisco_pkgs = []
        self.bundle_iso_input = False
        self.update_giso = False
        self.upgrade_giso = False
        self.bundle_iso = None

    def run_command(self,command):
        output = ''
        p = subprocess.Popen(command, stdout=subprocess.PIPE, 
                stderr=subprocess.PIPE, shell=True)

        # Read stdout from subprocess and log it in case of errors
        for line in iter(p.stdout.readline, b''):
            if line: # Don't print blank lines
                output = output + line
            # This ensures the process has completed, AND sets the 
            # 'returncode' attr
            while p.poll() is None:
                sleep(.1) #Don't waste CPU-cycles
            err = p.stderr.read()
            if p.returncode != 0:
                logger.error("Command :%s Failed"%(command))
                logger.error(err)
                sys.exit(-1)

    def mount_iso(self,bundle_iso,tmp_mount_path):
        # Create temp dirs 
        os.makedirs(tmp_mount_path)

        cmd = 'mount -o loop %s %s' % (bundle_iso,tmp_mount_path)
        self.run_command(cmd)

    def umount_iso(self,tmp_mount_path):
        ''' Unmount and remove given path '''

        if os.path.exists(tmp_mount_path):
            # In case previous umount failed try again
            try :
                cmd = 'umount %s' % (tmp_mount_path)
                self.run_command(cmd)
            except :
                pass
            shutil.rmtree(tmp_mount_path)

    def skip_tp_base(self,skipped_smus):
        ''' 
            If TP SMU is skipped and only base package is present in list skip
            the base package also else base package will become active 
        ''' 
        filt_pkgs = self.bundle_iso_pkgs[:]
        for smu in skipped_smus : 
            # Name,Arch and VM should match to skip the base packages.
            for bpkg in filt_pkgs :
                if bpkg.name == smu.name :
                    if smu.arch == bpkg.arch:
                        smu_vm = get_vm_of_tpsmu(smu.release)
                        if smu_vm == get_vm_of_tpsmu(bpkg.release) :
                            if bpkg in self.bundle_iso_pkgs :
                                self.bundle_iso_pkgs.remove(bpkg) 
                            logger.info("Skipping %s from GISO (skipped SMUs base pkg)"%(bpkg.filename)) 

    # get RPM Path from giso_info.txt                                               
    def get_rpm_path_from_giso_info(self, giso_info):                                   
        '''                                                                          
        Check if RPM_PATH exist in giso_info.txt                                    
        if RPM_PATH exist retrun same line
        else return NORPMPATH                                           
        '''                                                                          
        with open(giso_info,  'r') as fd :
            for line in fd:
                if "RPM_PATH:" in line :
                    return line
        return ("NORPMPATH")


    def is_optimised_651_ncs5500_giso(self, line):                                
        '''                                                                         
        fretta will have different giso structure than other giso
        In 651 release the additional rpms in optimised giso 
        will be located in following path 
        giso/boot/initrd.img/iso/system_image.iso              
        '''                                                                         
        if "system_image.iso" in line and "/boot/" in line :
            if (line.count("/boot/") == 1):                                            
                return True                                                          
        return False        

    def is_optimised_non_ncs5500_giso(self, line):                                
        '''                                                                          
        In platforms other than fretta the additional rpms in
        optimised giso will be located in following path 
        giso/boot/initrd.img
        '''                                                                          
        if "system_image.iso" not in line and "/boot/" in line :
            if(line.count("/boot/") == 1):                                              
                return True                                                            
        return False    

    def is_optimised_ncs5500_giso(self, line):                                
        '''                                                                          
        fretta will have different giso structure than other giso
        Release after 651(i.e 652, 701, ...) the additional rpms in 
        optimised giso will be located in following path 
        giso/boot/initrd.img/iso/system_image.iso/boot/initrd.img                
        '''                                                                          
        if "system_image.iso" in line and "/boot/" in line :
           if (line.count("/boot/") == 2):                                              
               return True                                                            
        return False  

    def get_giso_rpm_path(self, giso_info):                                        
        rpm_path = self.get_rpm_path_from_giso_info(giso_info)                                
        logger.debug("rpm path = %s" % (rpm_path))                                                               
        result = rpm_path.find("NORPMPATH")
        if result == -1:                                                             
            if self.is_optimised_ncs5500_giso(rpm_path):                                 
                return SIGNED_NCS5500_RPM_PATH
            elif self.is_optimised_651_ncs5500_giso(rpm_path):                           
                return SIGNED_651_NCS5500_RPM_PATH
            elif self.is_optimised_non_ncs5500_giso(rpm_path):                                       
                return SIGNED_RPM_PATH
            else:                                                                     
                return DEFAULT_RPM_PATH 
        else:                                                                        
            return DEFAULT_RPM_PATH 

    def get_fs_freespace(self, pathname):
        #Get the free space of the filesystem containing pathname
        stat= os.statvfs(pathname)
        return stat.f_bfree*stat.f_bsize

    def get_mount_path_for_signed_651_ncs5500_rpms(self, tmp_mount_path):

        freespace = self.get_fs_freespace(INSTALL_TMP_STAGING_PATH)
        logger.debug("freespace in %s = %sGB" %(INSTALL_TMP_STAGING_PATH, freespace/1024/1024/1024))

        if freespace > FREE_SPACE_REQUIRED_FOR_SYS_ISO:
            pwd = os.getcwd() 
            iso_extract_path = tempfile.mkdtemp(dir=INSTALL_TMP_STAGING_PATH)
            logger.debug("iso_extract_path = %s" %(iso_extract_path))
            if iso_extract_path is not None:
                os.chdir(iso_extract_path)
                cmd = "zcat -f %s%s | cpio -id iso/*" % (tmp_mount_path, ISO_BOOT_INITRD)
                logger.debug("cmd = %s" %(cmd))
                self.run_command(cmd)
                os.chdir(pwd)
                self.umount_iso(tmp_mount_path)
                sys_iso='%s%s' % (iso_extract_path, SYSTEM_IMAGE_ISO)
                logger.debug("sys_iso = %s" %(sys_iso))
                self.mount_iso(sys_iso,tmp_mount_path)
                return iso_extract_path, tmp_mount_path
            else:
                logger.error("Couldn't create directory for extacting initrd")
                sys.exit(-1)
        else:
            logger.error("Space available(%sGB) not sufficient for this operation" %(freespace/1024/1024/1024))
            logger.error("Please free up space ( > %sGB) in harddisk and retry" %(FREE_SPACE_REQUIRED_FOR_SYS_ISO/1024/1024/1024))
            sys.exit(-1)

    # place holder function for future implementation > 651 
    def get_mount_path_for_signed_ncs5500_rpms(self, tmp_mount_path):
        return tmp_mount_path 

    # place holder function for future implementation > 651 
    def get_mount_path_for_signed_rpms(self, tmp_mount_path):
        return tmp_mount_path 

    def bundle_iso_get_rpms(self,bundle_iso,upgrade=False):
        ''' Mount bundle ISO, get the RPMs and unmount and clean up 
            (1) Create temp dir for mounting GISO
            (2) Create temp dir for holding RPMs from Giso
            (3) Copy RPMs from GISO to these folders
        '''
        skipped_tp_smu = []
        to_deact = []
        bundle_rpms = []
        rpms_in_bundle = []
        label = ''
        curr_label = ''
        iso_extract_path = ''
        tmp_mount_path = ''

        tmp_mount_path = os.path.join(self.tmpstage,'BUNDLE_ISO')
        tmp_rpm_path = os.path.join(self.tmpstage,'BUNDLE_ISO_RPMS')

        # Delete older dir if exists due to any previous failure
        self.umount_iso(tmp_mount_path)
        self.mount_iso(bundle_iso,tmp_mount_path)

        # If 'replace', get the GISO label and re-validate the versions and label mismatch
        if self.options.replace:
            yaml_file = os.path.join(tmp_mount_path,'iosxr_image_mdata.yml')
            cmd = "/pkg/bin/provides_hint.py -f %s -g -a | grep \"GISO Label\" | awk '{print $3}'" %(yaml_file)
            try :
                status,label = commands.getstatusoutput(cmd)
                if not status :
                    logger.info("Label within GISO: %s" %(label))
                else :
                    logger.debug("Status is non-zero")
            except:
                logger.warning('Warning: Could not get GISO-Label from GISO')

            if self.options.from_version == self.options.to_version :
                # Check if the labels are same
                curr_label = self.options.curr_label
                if curr_label is not None and curr_label == label :
                    logger.info("\nThe current system version and label are same as in the GISO that you are trying to update to. Exitting.\n")
                    self.umount_iso(tmp_mount_path)
                    sys.exit(0)

        # Check if GISO or not
        giso_info_file = os.path.join(tmp_mount_path, 'giso_info.txt')
        if os.path.isfile(giso_info_file):
            logger.debug("This is a Golden ISO")
            result=self.get_giso_rpm_path(giso_info_file)
          
            if result == SIGNED_NCS5500_RPM_PATH :
                logger.debug("Yet to be supported in ncs5500 for > 651")
                tmp_mount_path = self.get_mount_path_for_signed_ncs5500_rpms(tmp_mount_path)

            elif result == SIGNED_651_NCS5500_RPM_PATH :
                logger.debug("Supported in ncs5500 for 651")
                iso_extract_path, tmp_mount_path=self.get_mount_path_for_signed_651_ncs5500_rpms(tmp_mount_path)
                logger.debug("iso_extract_path = %s and tmp_mount_path = %s" % (iso_extract_path, tmp_mount_path))
            elif result == SIGNED_RPM_PATH :
                logger.debug("Yet to be supported in platform other than ncs5500 for > 651")
                tmp_mount_path = self.get_mount_path_for_signed_rpms(tmp_mount_path)
            # DEFAULT_RPM_PATH 
            else :
                logger.debug("This is default rpm path i.e without initrd alteration")

            # Copy the rpms
            host_rpm = os.path.join(tmp_mount_path,'host_rpms')
            cal_rpm = os.path.join(tmp_mount_path,'calvados_rpms')
            xr_rpm = os.path.join(tmp_mount_path,'xr_rpms')

            if os.path.isdir(host_rpm):
                logger.debug("host_rpm directory present")
                counter = 0
                for filename in os.listdir(host_rpm):
                    rpms_in_bundle.append(filename);
                    counter = counter + 1
                    pkgobj = self.parse_rpm_pkg(filename)
                    nvr = "%s-%s-%s"%(pkgobj.name,pkgobj.version,pkgobj.release)
                    bundle_rpms.append(nvr);
                    if nvr not in self.active_cisco_pkgs :
                        shutil.copy(os.path.join(host_rpm,filename), self.tmpstage)
                        self.bundle_iso_pkgs.append(pkgobj)
                    else :
                        logger.info("Skipping %s from GISO as it's active "%(filename))
                        if is_tp_smu(self.platform, filename):
                            skipped_tp_smu.append(pkgobj)

            if os.path.isdir(cal_rpm):
                logger.debug("cal_rpm directory present")
                counter = 0
                for filename in os.listdir(cal_rpm):
                    rpms_in_bundle.append(filename);
                    counter = counter + 1
                    pkgobj = self.parse_rpm_pkg(filename)
                    nvr = "%s-%s-%s"%(pkgobj.name,pkgobj.version,pkgobj.release)
                    bundle_rpms.append(nvr);
                    if nvr not in self.active_cisco_pkgs :
                        shutil.copy(os.path.join(cal_rpm,filename), self.tmpstage)
                        self.bundle_iso_pkgs.append(pkgobj)
                    else :
                        logger.info("Skipping %s from GISO as it's active "%(filename))
                        if is_tp_smu(self.platform, filename):
                            skipped_tp_smu.append(pkgobj)

            if os.path.isdir(xr_rpm):
                logger.debug("xr_rpm directory present")
                counter = 0
                for filename in os.listdir(xr_rpm):
                    rpms_in_bundle.append(filename);
                    counter = counter + 1
                    pkgobj = self.parse_rpm_pkg(filename)
                    nvr = "%s-%s-%s"%(pkgobj.name,pkgobj.version,pkgobj.release)
                    bundle_rpms.append(nvr);
                    if nvr not in self.active_cisco_pkgs :
                        shutil.copy(os.path.join(xr_rpm,filename), self.tmpstage)
                        self.bundle_iso_pkgs.append(pkgobj)
                    else :
                        logger.info("Skipping %s from GISO as it's active "%(filename))
                        if is_tp_smu(self.platform, filename):
                            skipped_tp_smu.append(pkgobj)

        # Unmount tmp mount path and clear the dir
        self.umount_iso(tmp_mount_path)
        if iso_extract_path :
            shutil.rmtree(iso_extract_path)

        if len(skipped_tp_smu) > 0 :
            self.skip_tp_base(skipped_tp_smu)

        ## Write logic to form the to_deact list
        for act_pkg in self.xractive_package :
            if act_pkg.name_on_router not in bundle_rpms :
                if act_pkg.name_on_router.find("-xr-") == -1:
                    to_deact.append(act_pkg);
        for act_pkg in self.calv_active_package :
            if act_pkg.name_on_router not in bundle_rpms :
                if act_pkg.name_on_router.find("-sysadmin-") == -1:
                    to_deact.append(act_pkg);
        self.to_deactivate = to_deact[:]

        self.rpms_in_bundle = rpms_in_bundle[:]

    def get_active_provides(self):
        ''' Get list of provides of all active packages '''
        print("Getting list of provides of all active packages")
        provides = []
        cmd =  "rpm -qa --provides | grep -v '('  | grep -v '/'  | grep ' = '" 
        cmd = self.get_cal_cmd(cmd)
        try : 
            status,pr = commands.getstatusoutput(cmd)
            if not status : 
                provides = provides + pr.splitlines()    
        except:
            logger.warning('Warning: Could not get provides from calvados')

        cmd =  "rpm -qa --provides | grep -v '('  | grep -v '/'  | grep ' = '" 
        try : 
            status,pr = commands.getstatusoutput(cmd)
            if not status : 
                provides = provides + pr.splitlines()    
        except:
            logger.warning('Warning: Could not get provides from XR')
        return provides

    def get_rpm_in_glrepo(self):
        ''' Get list of RPMs in local repo '''
        print("Getting list of RPMs in local repo")
        path = '/install_repo/gl/'
        repo_path = ''
        rpm_list = []
        cmd = "sdr_instcmd show install repository all"
        output = commands.getoutput(cmd)
        y = output.splitlines()
        for l in y :
            if re.search(r'in Admin repository',l):
                repo_path=os.path.join(path, 'calvados')
            if re.search(r'in XR repository',l):
                repo_path=os.path.join(path, 'xr')
            if re.search(r'in Host repository',l):
                repo_path=os.path.join(path, 'host')

            if re.search(r'\.*host-\d+.\d+.\d+.\.*',l):
                continue
            elif re.search(r'\.*-xr-\d+.\d+.\d+.\.*',l):
                continue 
            elif re.search(r'\.*-sysadmin-\d+.\d+.\d+.\.*',l):
                continue
            elif re.search(r'package\(s\)',l):
                continue
            rpm_list.append(os.path.join(repo_path,l.strip()))
        return rpm_list

    def get_inactive_obsoletes(self):
        ''' Collect obsoletes of each rpm in repo '''

        total_obsoletes = {}
        # Filter one which are active before getting obsoletes
        active_rpm =  [x.filename for x in self.active_package]
        for f in self.rpminrepo:
            if os.path.basename(f) in active_rpm :
                continue
            cmd = self.get_cal_cmd("run rpm  -qp --obsoletes  %s"%(f))
            status,output = commands.getstatusoutput(cmd)
            if not status :
                ol = self.filter_cal_output(output)
                req = [ r for r in ol if "/" not in r ]
                f_name  = os.path.splitext(os.path.basename(f))[0]
                if req :
                    for r in req :
                        if r and r in total_obsoletes.keys() :
                            total_obsoletes[r].append(f_name)
                        elif r :
                            total_obsoletes[r] = [f_name]

        return total_obsoletes

    def get_inactive_requires(self):
        ''' Collect requires of each rpm in repo '''
        print("Getting requires of each rpm in repo")
        total_requires = {}
        # Filter one which are active before getting requires
        active_rpm =  [x.filename for x in self.active_package]
        for f in self.rpminrepo:
            if os.path.basename(f) in active_rpm :
                continue
            cmd = self.get_cal_cmd("run rpm  -qpR %s"%(f))
            status,output = commands.getstatusoutput(cmd)
            if not status :
                ol = self.filter_cal_output(output)
                req = [ r for r in ol if "/" not in r ]
                total_requires[os.path.basename(f)] = req
        return total_requires

    def get_inactive_provides(self):
        ''' Collect requires of each rpm in repo '''
        print("Getting provides of each rpm in repo")
        provides = {}
        # Filter one which are active before getting requires
        active_rpm =  [x.filename for x in self.active_package]
        for f in self.rpminrepo:
            if os.path.basename(f) in active_rpm :
                continue
            cmd = self.get_cal_cmd("run rpm  -qp --provides %s"%(f))
            status,output = commands.getstatusoutput(cmd)
            if not status :
                ol = self.filter_cal_output(output)
                pro = [ r for r in ol if "/" not in r ]
                provides[os.path.basename(f)] = pro
        return provides

    def get_platform(self):
        print("Getting platform")
        cmd  = "sdr_instcmd show install active | grep 'Boot image' |sort | uniq | cut -f1 -d'-'"
        try :
            platform = commands.getoutput(cmd)
            return platform.strip()
        except:
            return None


    def get_cal_shellcmd(self, command):
        cmd = """/pkg/sbin/admin-cli-proxy-xr_static -p XR << 'EOF
             run %s \nEOF
             """%(command)
        return cmd

    def get_cal_cmd(self, command):
        cmd_file = "/tmp/ciscoinstalldepcalc.txt"
        fd =open(cmd_file,"w")
        tool_name = self.cal_tool_name + " -p XR <<'EOF'"
        fd.write("%s\n%s\nEOF"%(tool_name,command))
        fd.close()
        os.system("chmod +x %s"%(cmd_file))
        return cmd_file

    def get_supported_archs(self):
        ''' Get the supported architecture from  /etc/init.d/calvados_bootstrap.cfg '''
        print("Getting supported architecture")
        supported_archs = {}
      
        meta_data_file = '/etc/init.d/calvados_bootstrap.cfg'
        #XR archs
        if os.path.exists(meta_data_file):
            cmd = 'cat  %s | grep XR_SUPPORTED_ARCHS'%(meta_data_file)
            archs = commands.getoutput(cmd).strip().split('=')
            if archs and len(archs) > 1 :
                supported_archs['XR'] = archs[1].split(',')
            else :
                supported_archs['XR'] = ['x86_64']
        else :
            supported_archs['XR'] = ['x86_64']
    
        #CAL archs
        if os.path.exists(meta_data_file):
            cmd = 'cat  %s | grep CALV_SUPPORTED_ARCHS'%(meta_data_file)
            cal_archs = commands.getoutput(cmd).strip().split('=')
            if cal_archs and len(cal_archs) > 1 :
                supported_archs['CAL'] = cal_archs[1].split(',')
            else :
                supported_archs['CAL'] = ['x86_64']
        else :
            supported_archs['CAL'] = ['x86_64']
        logger.debug("XR supported archs %s"%(' '.join(supported_archs['XR'])))
        logger.debug("CAL supported archs %s"%(' '.join(supported_archs['CAL'])))
        return supported_archs

    def get_calvados_active(self):
        pkg_names = []
        pkgs = []
        if not os.path.exists(self.cal_tool_name):
            # XR does not have capability to call CAL command
            return
        #cmd = self.get_cal_cmd('show install active | i " "')
        cmd = "/pkg/bin/sdr_actioncmd active"
        status,output = commands.getstatusoutput(cmd)
        if output :
            lines = output.split("\n\t\n")[0].splitlines()
            for line in lines : 
                pkg = self.parse_pkg(line)
                if pkg.name :
                    pkgs.append(pkg)
                    if not self.active_version :
                            self.active_version = pkg.version
        self.calv_active_package = pkgs[:]
                         
        cmd = "/pkg/bin/sdr_actioncmd 'rpm -qa'"
        #cmd = self.get_cal_cmd(cmd)
        status,output = commands.getstatusoutput(cmd)
        if not status :
            #rpm -qa output has architecture as well in the output
            #we want to strip off architecture from rpm -qa output.
            rpms = ['.'.join(line.split('.')[:-1]) for line in output.splitlines()]
            for rpm in rpms: 
                pkg = self.parse_pkg(rpm)
                if pkg.name and pkg.name not in pkg_names:
                    pkgs.append(pkg)
                    pkg_names.append(pkg.name)

        cmd = "/pkg/bin/sdr_actioncmd 'ls /opt/cisco/calvados/bin/release*.txt'"
        status,output = commands.getstatusoutput(cmd)
        if not status:
            cmd1 = output.strip();

        cmd = "/pkg/bin/sdr_actioncmd 'cat "+cmd1+"'"
        #cmd = self.get_cal_cmd(cmd)
        status,output = commands.getstatusoutput(cmd)
        if not status :
            #each line in release-rpms*.txt has following format
            #name-version-release.arch.rpm
            #we want to strip off .arch.rpm part of it
            rpms = ['.'.join(line.split('.')[:-2]) for line in output.splitlines()]
            for rpm in rpms: 
                pkg = self.parse_pkg(rpm)
                if not pkg:
                    continue
                if pkg.name and pkg.name not in pkg_names:
                    pkgs.append(pkg)
                    pkg_names.append(pkg.name)
        
        self.active_package.extend(pkgs)
        

    def get_calvados_inactive(self):
        pkgs = []
        if not os.path.exists(self.cal_tool_name):
            # XR does not have capability to call CAL command
            return
        #cmd = self.get_cal_cmd('show install inactive | i " "')
        cmd = "/pkg/bin/sdr_actioncmd inactive"
        status,output = commands.getstatusoutput(cmd)
        if output :
            lines = output.split("\n\t\n")[0].splitlines()
            for line in lines : 
                pkg = self.parse_pkg(line)
                if pkg.name :
                    pkgs.append(pkg)
        self.inactive_package.extend(pkgs)

    def parse_rpm_pkg(self,line):
        pkg = Package()
        line = line.strip()

        m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)\.rpm', line)
        if m:
            (path, name, version, release, arch) = m.groups()
            pkg.name = name
            pkg.version = version
            pkg.release = release
            pkg.format = 'RPM'
            pkg.filename = line
            pkg.packageid = release.split('.')[0]
            pkg.name_on_router = line.split()[0]
            if pkg.name_on_router.endswith(".rpm"):
                pkg.name_on_router = pkg.name_on_router.split('.rpm')[0]

            if re.search(r'.CSC',line):
                pkg.type = 'SMU'
            else :
                pkg.type = 'OPTIONAL'
        return pkg

    def parse_pkg(self,line):
        pkg = Package()
        line = line.strip()

        #Active mini
        if re.search(r'Boot image',line):
            pkg.type = 'MANDATORY'
            pkg.format = 'ISO'
            mini = re.search(r'(\w*)-(\w*)-(\d+.\d+.\d+.\d+\w*) \.*',line)
            if not mini:
                mini = re.search(r'(\w*)-(\w*)-(\d+.\d+.\d+) \.*',line)
            if mini:
                pkg.version = mini.group(3)
                if self.img_name.has_key(mini.group(1)):
                    pkg.filename= "%s-%s"%(self.img_name[mini.group(1)],mini.group(3))
                else:
                    pkg.filename= "%s-mini-x.iso-%s"%(mini.group(1),mini.group(3))
                if self.img_name.has_key(mini.group(1)):
                    pkg.filename= "%s-%s"%(self.img_name[mini.group(1)],mini.group(3))
                else:
                    pkg.filename= "%s-mini-x.iso-%s"%(mini.group(1),mini.group(3))
                pkg.name_on_router = line.split()[0]
                pkg.name = "%s-mini"%(mini.group(1))
                if not self.platform :
                    self.platform = mini.group(1)
        # Inactive mini ISO
        elif re.search(r'mini',line):
            mini = re.search(r'(\w*)-(mini)-(\w*)-(\d+.\d+.\d+)$',line)
            if not mini :
                mini = re.search(r'(\w*)-(mini)-(\w*)-(\d+.\d+.\d+)$',line)
            if mini:
                pkg.version = mini.group(4)
                pkg.name = "%s-mini-%s"%(mini.group(1),mini.group(3))
                pkg.type = 'MANDATORY'
                pkg.format = 'ISO'
                pkg.filename = "%s-mini-%s.iso-%s"%(mini.group(1),mini.group(3),mini.group(4))
                if self.img_name.has_key(mini.group(1)):
                    pkg.filename= "%s-%s"%(self.img_name[mini.group(1)],mini.group(3))
                else:
                    pkg.filename= "%s-mini-x.iso-%s"%(mini.group(1),mini.group(3))

                pkg.name_on_router = line.split()[0]

        ## Golden ISO
        elif self.options.replace and re.search(r'-golden',line):
            giso = re.search(r'(\w*)-(golden\w*)-(\w*)-(\d+.\d+.\d+.\d+)\w*-(\w*)$',line)
            ## Handle renumbered images
            if not giso :
                giso = re.search(r'(\w*)-(golden\w*)-(\w*)-(\d+.\d+.\d+)\w*-(\w*)$',line)
            if giso :
                label_name = giso.groups()[4];
            if not giso :
                giso = re.search(r'(\w*)-(golden\w*)-(\w*)-(\d+.\d+.\d+.\d+)$',line)
                if not giso :
                    giso = re.search(r'(\w*)-(golden\w*)-(\w*)-(\d+.\d+.\d+)$',line)
            if giso :
                pkg.version = giso.group(4)
                pkg.name = "%s-golden-%s"%(giso.group(1),giso.group(3))
                pkg.type = 'MANDATORY'
                pkg.format = 'ISO'
                pkg.filename = "%s-golden-%s.iso-%s"%(giso.group(1),giso.group(3),giso.group(4))
                pkg.name_on_router = line.split()[0]
        # SMU
        elif re.search(r'.CSC',line):
            if self.platform in self.platform_pkgformat_0 :
                # .pkg format
                smu = re.search(r'(\w*)-(\d+.\d+.\d+.\d+\w*).(CSC\w*)-\.*',line)
                if not smu:
                    smu = re.search(r'(\w*)-(\d+.\d+.\d+).(CSC\w*)-\.*',line)
                if smu:
                    pkg.type = 'SMU'
                    pkg.format = 'TAR'
                    pkg.name = "%s-%s.%s"%(smu.group(1),smu.group(2),smu.group(3))
                    pkg.filename = "%s-%s.%s"%(smu.group(1),smu.group(2),smu.group(3))
                    pkg.version = smu.group(2)
                    pkg.packageid = smu.group(3)
                    pkg.name_on_router = line.split()[0]
            else :
                # .rpm format
                m = re.search(r'(.*/)*(.*)-(.*)-(.*)', line)
                if m:
                    (path, name, version, release) = m.groups()
                    pkg.name = name
                    pkg.version = version
                    pkg.release = release.strip().split(' ')[0]
                    pkg.type = 'SMU'
                    pkg.format = 'RPM'
                    pkg.filename = "%s-%s-%s.%s.rpm"%(name, version, release, self.arch)
                    pkg.packageid = release.split('.')[0]
                    pkg.name_on_router = line.split()[0]

        # Optional packages
        else :
            if self.platform in self.platform_pkgformat_0 :
                # .pkg format
                opt = re.search(r'(\S*)-(\d+.\d+.\d+\.*)',line)
                if not opt:
                    opt = re.search(r'(\w*)-(\w*)-(\d+.\d+.\d+) \.*',line)
                if opt:
                    pkg.version = opt.group()[1]
                    pkg.filename = "%s.pkg-%s"%(opt.group(1),opt.group(2))
                    pkg.name = opt.group(1)
                    pkg.type = 'OPTIONAL'
                    pkg.format = 'TAR'
                    pkg.name_on_router = line.split()[0]
            else :
                # .rpm format
                m = re.search(r'(.*/)*(.*)-(.*)-(.*)', line)
                if m:
                    (path, name, version, release) = m.groups()
                    pkg.name = name
                    pkg.version = version
                    pkg.release = release.strip().split(' ')[0]
                    pkg.type = 'OPTIONAL'
                    pkg.format = 'RPM'
                    pkg.filename = "%s-%s-%s.%s.rpm"%(name, version, release, self.arch)
                    pkg.packageid = release.split('.')[0]
                    pkg.name_on_router = line.split()[0]


        return pkg

    def get_active(self):
        """ Get active packages from XR """
        print("Getting active packages from XR")
        pkg_names = []
        pkgs = []

        cmd = "sdr_instcmd show install active"
        status,output = commands.getstatusoutput(cmd)
        if output :
            lines = output.split("\n\t\n")[0].splitlines()
            for line in lines : 
                pkg = self.parse_pkg(line)
                if self.options.replace and re.search(r'.CSC',line):
                    pkg_dup = [ x for x in pkgs if (x.name == pkg.name and x.release == pkg.release)]
                else:
                    pkg_dup = [ x for x in pkgs if x.name == pkg.name ]
                if pkg.name and not pkg_dup:
                    pkgs.append(pkg)
                    if not self.active_version :
                            self.active_version = pkg.version
        self.xractive_package = pkgs[:]
        cmd = "rpm -qa"
        status,output = commands.getstatusoutput(cmd)
        if not status :
            rpms = ['.'.join(line.split('.')[:-1]) for line in output.splitlines()]
            for rpm in rpms: 
                pkg = self.parse_pkg(rpm)
                if pkg.name and pkg.name not in pkg_names:
                    pkgs.append(pkg)
                    pkg_names.append(pkg.name)

        cmd = "cat /pkg/bin/release-rpms*.txt"
        status,output = commands.getstatusoutput(cmd)
        if not status :
            rpms = ['.'.join(line.split('.')[:-2]) for line in output.splitlines()]
            for rpm in rpms: 
                pkg = self.parse_pkg(rpm)
                if not pkg:
                    continue
                if pkg.name and pkg.name not in pkg_names:
                    pkgs.append(pkg)
                    pkg_names.append(pkg.name)
        return pkgs

    def get_inactive(self):
        """ Get inactive packages from XR VM """
        print("Getting inactive packages from XR")
        pkgs = []
        rel_to_match = ""
        cmd = "sdr_instcmd show install inactive"
        status,output = commands.getstatusoutput(cmd)
        if output :
            if self.to_version:
                rel_to_match = 'r' + ''.join(self.to_version.split('.'))
            lines = output.split("\n\t\n")[0].splitlines()
            for line in lines : 
                pkg = self.parse_pkg(line)
                if pkg.name and rel_to_match:
                    if pkg.type == 'SMU':
                        pkg_rel = re.sub('.CSC[a-z][a-z]\d{5}','',pkg.release)
                        if pkg_rel == rel_to_match:
                            pkgs.append(pkg)
                    elif pkg.release == rel_to_match:
                        pkgs.append(pkg)
                elif pkg.name:
                    pkgs.append(pkg)
                        
        return pkgs


    def obsoleted_by_active(self):
        """ Get all the packages which are obsolted by installed packages."""
       
        for pkg in self.active_package:
            cmd_supersed = "rpm -qa --obsoletes %s | grep CSC" % ()
            status,supersede = commands.getstatusoutput(cmd_supersed)
            if supersede:
                supersede = supersede.splitlines()
                for s in supersede :
                    ddts = ".%s."%(s.split('=')[0].split('.')[-2])
                    ss = re.sub('.CSC\w*\d+.',ddts,pkg)
                    pkg.obsoletes.append(ss)

        for pkg in self.inactive_package:
            cmd_supersed = "rpm -qa --obsoletes | grep CSC"
            status,supersede = commands.getstatusoutput(cmd_supersed)
            if supersede:
                supersede = supersede.splitlines()
                for s in supersede :
                    ddts = ".%s."%(s.split('=')[0].split('.')[-2])
                    ss = re.sub('.CSC\w*\d+.',ddts,pkg)
                    pkg.self.obsoletes.append(ss)

    def not_in_glrepo(self, name):
        """ Check if given package is not in gl repo """
        for repo_p in self.rpminrepo :
            if re.match(repo_p,name):
                return True
        return False

    def sanitize_remove_duplicates(self, pkglist):
        for cli_pkg in self.options.userpkgs:
            cli_name = ''
            cli_release = ''
            if ".rpm" in cli_pkg:
                m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)\.rpm', cli_pkg)
                if m:
                    (path, cli_name, version, cli_release, arch) = m.groups()
                    cli_release = cli_release.split('.')[0]
                else:
                    logger.error("Error: Unable to parse %s while sanitizing. " 
                                 " Please check if pkg name is modified and "
                                 " not following standard RPM format" %(cli_pkg))
                    sys.exit(-1)
            else:
                continue
            pkgList = pkglist[:]
            for pkg in pkgList:
                if ".rpm" in pkg.filename:
                    m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)\.rpm', pkg.filename)
                    if m:
                        (path, name, version, release, arch) = m.groups()
                        release = release.split('.')[0]
                        if name == cli_name and release != cli_release:
                            logger.debug("Removing %s from package list:  "
                                         " cli pkg will override the pkg in the "
                                         " list if same pkg  of another release "
                                         " present in the list"
                                          %(pkg.filename))
                            pkglist.remove(pkg)
                    else:
                        logger.error("Error: Unable to parse %s while sanitizing" 
                                     " Please check if pkg name is modified and "
                                     " not following standard RPM format"
                                      %(pkg.filename))
                        sys.exit(-1)
        return pkglist

    def sanitize_input_package(self, pkglist):
        ''' Sanitize the set of input packages calculated for
            the same release as what self.to_version is set to. '''
        rel_to_match = 'r' + ''.join(self.to_version.split('.'))
        #logger.debug("Sanitizing input package list for release %s " 
        #             "unless the package is specified explicitly in CLI" %rel_to_match)
        logger.debug("Sanitizing input SMU list for release %s " 
                     "unless the base package is specified explicitly in CLI" %rel_to_match)

        # Remove any duplicate packages in pkglist and override them with cli
        # package list.
        pkglist = self.sanitize_remove_duplicates(pkglist)
        pkg_list = pkglist[:]
        for pkg in pkg_list:
            do_exit = True
            if ".iso" in pkg.filename:
                continue
            if is_tp_pkg(self.platform, pkg.filename):
                continue
            pkg_rel = pkg.release.split('.')[0]
            if pkg_rel != rel_to_match:    
                logger.debug("Pkg %s doesn't match current release %s"
                             %(pkg.filename, rel_to_match))
                # Check if any package is specified in CLI.
                # Continue with SMUs for base packages listed in CLI.
                for cli_pkg in self.options.userpkgs:
                    if ".iso" in cli_pkg:
                        continue
                    if is_tp_pkg(self.platform, cli_pkg):
                        continue
                    # If SMU is picked up, need to check if 
                    # its base package has been specified in CLI.
                    if "CSC" in pkg.filename:    
                        m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)\.rpm', cli_pkg)
                        if m:
                            (path, name, version, release, arch) = m.groups()
                            if pkg.name == name and pkg_rel == release:
                                do_exit = False
                                break
                        else:
                            logger.error("Error: Unable to parse %s while sanitizing"   
                                         " Please check if pkg name is modified and "
                                         " not following standard RPM format" %(cli_pkg))
                            sys.exit(-1)
                    elif pkg.filename == cli_pkg:
                        do_exit = False
                        break

                # Possible that the base package is already active
                if do_exit == True:
                    for act_pkg in self.xractive_package:
                        if "CSC" in pkg.filename:
                            if pkg.name == act_pkg.name and pkg_rel == act_pkg.release:
                                do_exit = False
                                break

                if do_exit == True:
                    logger.error("Error: Exiting operation as smu %s of release %s "
                                 "does not match current release %s " 
                                 %(pkg.filename, pkg_rel, rel_to_match))
                    sys.exit(-1)

        logger.debug("Sanitized list %s" %(', '.join([x.filename for x in self.input_package])))
        return pkglist
                

    def filter(self,v2=None):
        """ Filter and fine tune packages needed for download 
           (1) Packages already in active or inactive do not need download
           (2) Packages superseded/obsoleted by packages in active and inactive 
               do not need download.
           (3) If v2 specified , all packages should match V2
        """
        pkg_abs_name = []
        pkg_inact_name_skip = []
        pkg = []
       
        if not self.input_package :
            return

        # Sanitize input package based on version initialization.
        if self.input_package:
            self.input_package = self.sanitize_input_package(self.input_package)

        # Before we filter active and inactive package , get the package
        # required by given package. 
        self.input_package = self.input_package + self.checkcompat()
            
        pkg = [x for x in self.input_package]

        inact_pkgs = [x.name_on_router for x in self.inactive_package]
        act_pkgs = [x.name_on_router for x in self.active_package]

        if pkg :
            pkg_abs_name = pkg[:]
            message = False
        for x in pkg : 
            for act_pk in act_pkgs :
                #If package is already active , no need to download and activate

                if act_pk in x.filename :
                    m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)(\.rpm)', x.filename)
                    if m:
                        (path, name, version, release, arch, _) = m.groups()
                        nvr = "%s-%s-%s"%(name,version,release)
                        if not act_pk == nvr :
                            continue
                    if x in pkg_abs_name :
                        # If it's upgrade option , if it's not in repo get that
                        if self.not_in_glrepo(x.filename) :
                            self.skipped_as_active.append(x.filename)
                            pkg_abs_name.remove(x)

        if pkg_abs_name :
            xpkgs = pkg_abs_name[:]
            message = False
        else :
            xpkgs = []
        for x in xpkgs : 
            if not x : 
                continue
            for inact_pk in inact_pkgs :
                #If package is inactive , no need to download and activate
                if inact_pk in x.name_on_router and x in pkg_abs_name:
                    if self.options.replace and inact_pk != x.name_on_router :
                        continue
                    if x in pkg_abs_name :
                        self.skipped_as_inactive.append(x.filename)
                        pkg_abs_name.remove(x)
                    pkg_inact_name_skip.append(x.name)
                    self.to_activate.append(x)
                    self.check_required()
        pkg_act_name_skip = [x.filename for x in pkg if x.name in act_pkgs and x.version == self.active_version]

        if pkg_inact_name_skip :
            # Skip downloading inactive package , but we need to account for 
            # Packages required by these inactive packages.
            pkg_req_by_inact = self.get_req_by_inact(pkg_inact_name_skip)

        self.to_download_package.extend([a for a in pkg_abs_name if a])
        self.input_package = []
       
    def filter_cal_output(self,output):
        ''' 
        1. First message of connecting 
        2. second message of command line
        3. Third line is Date
        4. Last line is cal promt'''
        ol = output.splitlines()
        y = []
        for r in ol :
            if not r.strip():
                continue 
            if 'connected from' in r  or 'run ' in r or 'No such file' in r:
                continue
            elif ' UTC' in r  or '#' in r :
                continue
            y.append(r)
        return y

    def get_req_by_inact(self, pkg_inact):
        total_requires = []
        seen_pkg = []
        required_package = []

        while pkg_inact :
            input_pkg = pkg_inact.pop(0)
            if input_pkg[-4:] == '.rpm' :
                input_pkg = os.path.splitext(input_pkg)[0]
            for pkg in self.inactive_requires.keys() :
                if pkg == input_pkg :
                    total_requires = total_requires + self.inactive_requires[pkg]
                    required_package = self.add_requires(total_requires)
        return required_package

    def reverse_key_value(self, supersede):
        """ Write the comments """
        rev_ss = {}
        for ss in supersede.keys():
            if supersede[ss] :
                for s in supersede[ss] : 
                    rev_ss[s] = ss
        return rev_ss

    def filter_supersede(self, pkgs,pr,ss):       
        """ filter out superceded SMUs from the all package list"""  
        all_pkg = pr.keys()
        all_pkg.extend(pr.values())
        all_pkg.extend(pkgs)
        ss_in_pkgs = [x for x in all_pkg if x in ss.keys()]

        # Replace superseded ones
        while ss_in_pkgs :
            for supersede in ss.keys() :
                if supersede in all_pkg :
                    all_pkg.pop(all_pkg.index(supersede))
                    all_pkg.insert(0,ss[supersede])
            ss_in_pkgs = [x for x in all_pkg if x in ss.keys()]
        return all_pkg
        

    def check_required(self):
        """ check if all the required packages are present in provides list """ 
        pkginput = [ p for p in (self.downloaded_package[:] + self.to_activate[:]) if p ]
        to_add_packages = []
        requires_total = []
        all_pkgs       = []
        unmet_reqs = None

        for pkg in pkginput:
            if pkg.filename in self.seen_prereq :
                continue
            self.seen_prereq.append(pkg.filename)
            if not pkg.requires :
                pkg.requires = self.get_req_by_inact([pkg.filename])
            for req in pkg.requires:
                requires_total.append(req)
            
            for pr in pkg.provides:
                all_pkgs.append(os.path.splitext(pr)[0])
        #get all provides 
        all_pkgs.extend(self.active_provides)

        all_pkgs = list(set(all_pkgs))
        requires_total = list(set(requires_total))

        # Because same list is being modified need to make a copy
        # else loop does not work as expected
        tmp_req = requires_total[:]
        for req in tmp_req:
            if req in all_pkgs:
                if req in requires_total:
                    requires_total.remove(req)
                continue

        if requires_total:
            requires_total = self.add_requires(requires_total)
            for x in requires_total:
                for pkg in pkginput:
                    temp = [y for y in pkg.requires if re.search(x.split(' ')[0],y)]
                    if temp and ((('CSC' in x and 'CSC' in pkg.name) or \
                                ('CSC' not in x and 'CSC' not in pkg.name))):
                        if pkg.name.endswith('.smu') and 'CSC' in x :
                            #.smu type patch format
                            package_id = re.search("CSC\w+",x).group()
                            if package_id :
                                to_add_packages.append(re.sub("CSC\w+",package_id,pkg.name))

                        elif pkg.filename.endswith('.rpm') and 'CSC[a-z][a-z]\d{5}' in x :
                            #.rmp type patch format
                            package_id = re.search("CSC[a-z][a-z]\d{5}",x).group()
                            if package_id :
                                to_add_packages.append(re.sub("CSC[a-z][a-z]\d{5}",package_id,pkg.name))

                            unmet_reqs = self.add_requires(x)
                            if unmet_reqs :
                                logger.debug('%s required by %s but not present.'%(unmet_reqs, pkg.filename))

                        elif pkg.filename.endswith('.rpm') and not 'CSC[a-z][a-z]\d{5}' in x :
                            # Package 
                            unmet_reqs = self.add_requires(x)
                            if unmet_reqs :
                                logger.debug('%s required by %s but not present.'%(unmet_reqs, pkg.filename))

                        break
                    elif 'CSC[a-z][a-z]\d{5}' in pkg.name and 'CSC[a-z][a-z]\d{5}' not in x and '.rpm' not in x:
                        logger.info("Skipping %s as its base package %s is not installed"%(pkg.name,x))
                        if pkg in pkginput :
                            pkginput.remove(pkg)
                        self.del_from_to_activate(pkg.name)
        return to_add_packages

    def match_active(self,required):
        package = self.match(required, self.active_package)
        if package :
            return True
        else :
            return False


    def match_inactive(self,required):
        package = self.match(required, self.inactive_package)
        if package :
            # If not already in to_activate list , add  to it
            for p in  self.to_activate :
                if p.name == required.split(' ')[0]:
                    v = re.search("\.*(\d+.\d+.\d+.\d+)\.*",required)
                    if v :
                        # Version is specified, so version also should match
                        if v.groups()[0] == p.version :
                            # Already package is in to activate list
                            return True
                    else :
                        # No version specified, one version is in to_activate 
                        return True

            self.to_activate.append(package)
            #Since new package is added do prereq check again
            self.check_required()
            return True
        else :
            return False

    def match_repo(self,required):
        #IMP --- This is remote repo
        package = self.match(required, self.repo_package)

        # If not already downloaded , add to _to_download
        if package and not self.downloaded(package) :
            self.to_download_package.append(package)

        if package :
            return True
        else :
            return False

    def downloaded(self,package):
        if not self.downloaded_package or not package:
            return False
        for d in self.downloaded_package :
            if d.filename  == package.filename : 
                return True
        return False

    def match(self,required,pkglist):
        required = re.split(' ',required)
        name = ''
        if len(required) == 3 :
            (name,oper,version) = required
            provider = self.hints.whatprovides(name,self.platform)
            for pkg in pkglist :
                if name == pkg.name or provider == pkg.name : 
                    v1 = tuple(pkg.version.split('.'))
                    v2 = tuple(version.split('.'))
                    # SMU dependencies are always = , 
                    if 'CSC' in pkg.release and not oper == '=' :
                        continue

                    if version.count('.') != pkg.version.count('.') :
                        return False
                    elif oper == '>' :
                        if v1 > v2 :
                            return pkg
                    elif oper == '<' :
                        if v1 < v2 :
                            return pkg
                    elif oper == '=' :
                        if self.is_smu(version) :
                            # only if SMU id is in version field
                            ver_rel_ddts = re.search("\.*(r\d*\S*)\S(CSC\S{2}\d{5})\.*",version)
                            if ver_rel_ddts :
                                (rel,ddts) = ver_rel_ddts.groups()
                                if rel in pkg.release and ddts in pkg.release :
                                    return pkg


                        if v1 == v2 :
                            return pkg

                    elif oper == '>=' :
                        if v1 >= v2 :
                            return pkg
                    elif oper == '<=' :
                        if v1 <= v2 :
                            return pkg
                    else :
                        continue

        elif len(required) == 1 :
            for pkg in pkglist :
                if name == pkg.name : 
                    return pkg
        return None



    def is_smu(self, package):
        if re.search('CSC[a-z][a-z]\d{5}',package):
            return True
        return False

    def add_requires(self,required):
        ''' 
        1. See if in active, skip
        2. if in inactive , add to to_activate
        3. If neither active not inactive , look in repo
        '''
        new_required = required[:] 
        inactive_requires = []
        for r in required :
            rp = r.split(" ")[0].strip()
            provider = self.hints.whatprovides(rp,self.platform)
            if provider and not self.match_active(r):
                nr = r.replace(rp,provider)
                new_required[new_required.index(r)] = nr
                r = nr

            if self.match_active(r):
                new_required.remove(r)
                #self.skipped_as_active.append(r)

            elif self.match_inactive(r):
                new_required.remove(r)

            elif self.match_repo(r):
                new_required.remove(r)
                self.to_add.append(r)                
        return new_required

    def eliminate_obsolete_pkgs_from_activation(self):
        pkginput = self.to_download_package
        all_pkgs = self.downloaded_package
        all_obsolete = []
        all_requires = []
        for pkg in all_pkgs:
            if pkg.requires:
                all_requires.extend(pkg.requires)
            if pkg.obsoletes:
                all_obsolete.extend(pkg.obsoletes)

        for obs in all_obsolete:
            for req in all_requires:
                if re.search(obs, req): continue
            else:
                #we found a package which can now be removed from activation list
                for pkg in all_pkgs:
                    for x in pkg.provides:
                        if re.search(obs,x):
                            #we can remove pkg from the list
                            all_pkgs.remove(pkg)

    def del_inact_from_to_add(self):
        """ GIven a package name , remove entry from to_activate list """
        modified_add_list = self.to_add[:]
        for p in self.inactive_package:
            for pkg in self.to_add:
                if pkg.name == p.name :
                    modified_add_list.remove(pkg)
        self.to_add = modified_add_list    

    def del_from_to_activate(self,name):
        """ GIven a package name , remove entry from to_activate list """
        for pkg in self.to_activate:
            if pkg.name == name :
                self.to_activate.remove(pkg)
                break

    def chk_giso_in_inactive(self,name):
        for pkg in self.inactive_package:
            if pkg.name_on_router == name:
                return True
        return False
            
    def input_inactive(self):
        inactive = [x.name_on_router for x in self.inactive_package]
        inact_list = []
        for p in self.input_package :
            if p.name_on_router in inactive :
                inact_list.append(p)
        return inact_list

    def checkcompat(self):
        """ Package dependency and supersede calculation 
            1. Neak into dowanloaded package , see if that requires another 
               package add that not in active/inactive/downloaded then add 
               it to to_be_downloaded list.
            2. Add the downloaded package to _to_activate list with version
               added, so that it can be used during activate
        """
        TMPDIR = "/tmp/tmpdepcaldonotuse"
        global is_new_label_format

        pkinput = self.downloaded_package[:] + self.to_activate[:] + self.input_inactive()
        pkinput = [ p for p in pkinput if p ]
        if pkinput:
            pkinput = self.sanitize_input_package(pkinput)
        pkginput = pkinput[:]
        files = []
        prereq = []
        
        uniq_pkinput = []
        seen_pkgs = []
        for pk in pkinput:
            if pk.filename not in seen_pkgs:
                seen_pkgs.append(pk.filename)
                uniq_pkinput.append(pk)
 
        for pk in uniq_pkinput:
            pkg = pk.filename
            if '.iso' in pkg :
                self.to_activate.append(pk)
                continue 
            if os.path.exists(TMPDIR) :
                os.system("rm -rf %s"%(TMPDIR))
                os.makedirs(TMPDIR)
            if pk.filename.endswith('.rpm') :
                rpm_file = os.path.join(self.tmpstage,pkg)
                if os.path.exists(rpm_file) :
                    pk.format = 'RPM' 
                    pattern = '\'%{VERSION} %{NAME} %{RELEASE}\''
                    cmd  = "rpm -qp --qf %s %s"%(pattern, rpm_file)
                    status,result =  commands.getstatusoutput(cmd)
                    if not status :
                        version,pkg_name,release = result.split()
                        pk.name_on_router = "%s-%s-%s"%(pkg_name,version,release)
                        files = [os.path.join(self.tmpstage,pkg)]
                else :
                    prereq = self.get_req_by_inact([pk.filename])
                    pk.requires.extend(prereq)


            for file in files:  
                if file[-4:] == '.rpm' :
                    rpmfile = os.path.join(TMPDIR,file)
                    cmd_prereq = "rpm -qpR %s | grep -v \/"%(rpmfile)
                    cmd_supersed = "rpm -qp --obsoletes %s | grep CSC "%(rpmfile)
                    cmd_provide = "rpm -qp --provides %s "%(rpmfile)

                    ver = '%{VERSION}'
                    cmd = "rpm -qp --queryformat %s %s"%(ver,rpmfile)
                    status,version= commands.getstatusoutput(cmd)

                    ver = '%{NAME}'
                    cmd = "rpm -qp --queryformat %s %s"%(ver,rpmfile)
                    status,pkg_name= commands.getstatusoutput(cmd)

                    if not pk.version and version :
                        pk.version = version
                    status,prereq = commands.getstatusoutput(cmd_prereq)
                    if status :
                        logger.debug('Failed to execute :\n%s',cmd_prereq)

                    if prereq :
                        prereq = prereq.splitlines()
                        for d in self.downloaded_package :
                            if pk.name == d.name :
                                d.requires.extend(prereq)
                        pk.requires.extend(prereq)
    
                    status,supersede = commands.getstatusoutput(cmd_supersed)
                    if supersede:
                        supersede = supersede.splitlines()
                        for s in supersede :
                            if 'sysadmin' in pkg:
                                ss = os.path.splitext(s.split("=")[0].split("/")[1])[0]
                            else:
                                ss = os.path.splitext(s.split("=")[0])[0]
                            pk.obsoletes.append(ss)
                 
            self.to_activate.append(pk)
            # If a SMU does not have any prereq or supersede, need not be tracked
            for pkg in pkginput :
                if not pkg :
                    continue
                obs_string  = "%s = %s"%(pkg.name, pkg.version)
                if obs_string in self.obsoleted_package.keys(): 
                    message = "%s is obsoleted by %s "%(pkg.name_on_router, \
                            " ".join(self.obsoleted_package[obs_string]))
                    #logger.warning(message)
                if pkg.obsoletes:
                    # Get rid of partial supersede we dont track them either
                    ss = []
                    for s in pkg.obsoletes :
                        if s not in pkg.requires:
                            ss.append(s)
                    pkg.obsoletes = ss
            self.eliminate_obsolete_pkgs_from_activation()

        return self.check_required()
