#!/bin/bash

# Copyright (c) 2015-2019 by cisco Systems, Inc.
# All rights reserved.
# Created by: Santhosh N
#
# This script contains functions used by the NG install component.

SCRIPTPATH="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
# Now use that directory to call the other script
source $SCRIPTPATH/disk_space_precheck.sh

set -e
set -o pipefail

if [ -f /etc/cisco/xr_sysctl.rc ]; then
    source /etc/cisco/xr_sysctl.rc
fi

export USER=root
if grep cpu /proc/1/cgroup >/dev/null 2>&1; then
    export HOSTVM_IP="10.0.2.16"
else
    export HOSTVM_IP="10.0.2.2"
fi

export CHVRF_CMD="/opt/cisco/calvados/bin/vrfch.sh CTRL_VRF "

export INSTALL_SCRIPT=${BASH_SOURCE[0]}
export NEWROOT="/install_repo/tmp/newroot"
export MNTDIR_NAME=".su_mntdir"
repo_old_marker_file="/install/old_repo_marker"
srch_str="Repo is big"
#Repo migration with patching infra
repo_marker_file="/install_repo/local/new_repo_marker"
old_repo_marker_file="/old_install_repo/local/new_repo_marker"

#SDR_KEY is exported before this script is sourced.
#If not, use default
if [ "$SDR_KEY" == "" ]; then
    SDR_KEY="default-sdr"
fi

#SDR local dir has changed between 5.* and 6.*
export SDR_LOCAL1="/install_repo/local/sdr/$SDR_KEY/"
export SDR_LOCAL2="/install_repo/local/"

#SDR repository and instdb dir
export XR_REPO="/install_repo/gl/xr/"
export INSTALL_INSTDB="/install_repo/gl/instdb/"

trap 'error_trap' ERR

## Echo to stderr
function echoerr() {
    echo -e "$@" 1>&2;
}

## Print the command being executed
## Args: Command and its arguments
function run_cmd() {
    echo -e "\nRun -> $@"
    eval $@
    echo ""
}

function dump_stack() {
    local i=0
    local FRAMES=${#BASH_LINENO[@]}
    echoerr "\nCall stack:"
    for ((i=0; i<FRAMES-1; i++)); do
        echoerr '  File' \"${BASH_SOURCE[i+1]}\", Function \"${FUNCNAME[i+1]}\", Line $((${BASH_LINENO[i]}))
    done
   
    return 1
}

function error_trap() {
    echo "Error trap :::: $@ | Func: ${FUNCNAME[$@]}" >/dev/null
}

## Call this below function while sourcing script to avoid ssh login banner msg
function calvms_sshlogin() {
    rm -f /root/.ssh/known_hosts
    
    for i in `ips_cal`; do
        $CHVRF_CMD ssh $i "true" >/dev/null 2>&1 || true
    done
}

## Setup env for running functions dumped from xrnginstall script
function setup_vars() {
    set +e
    if [ -f /iso/sim.txt ]; then
        readonly SIMULATION=1
        echo "Running on SIMULATION PLATFORM"
    else
        readonly SIMULATION=0
    fi            

    source /etc/init.d/disk-functions ||  true
        
    echo "Running on $PLATFORM platform"
    
    export ISOMNT=`mktemp -d /tmp/.su_isomnt.XXXXXX`
    export PARTMNT=`mktemp -d /tmp/.su_prtmnt.XXXXXX`
    export ISO_NAME_FILE=/iso_name.txt
    export LVM_SUPPORTED=1
    export DATA_LV_SUPPORTED=1
    export ISO_INFO_FILE="iso_info.txt"
    export PKG_VER=1.0
    export SSHKEY_DIR="/tmp/sshkeys"
    exec 107>&1

    if (( $SIMULATION )); then
        if [ -f /etc/ucs/xrnginstall-ucs-sim ]; then
            step_log_file "UCS-SIM: source /etc/ucs/xrnginstall-ucs-sim"
            source /etc/ucs/xrnginstall-ucs-sim
        fi
    else
        if [ -f /etc/ucs/xrnginstall-ucs ]; then
            step_log_file "UCS: source /etc/ucs/xrnginstall-ucs"
            source /etc/ucs/xrnginstall-ucs
        fi
    fi

    bootdir=${PARTMNT}/boot
    grubdir=${bootdir}/grub
    grub_shell=/sbin/grub

    set -e
}

## Function called by xrnginstall script. Dummy replacement to original one.
function step_log_file() {
    echo $@
}

## Function called by xrnginstall script. Dummy replacement to original one.
function step_log_console() {
    echo $@
}

## Function called by xrnginstall script. Dummy replacement to original one.
function exitclean() {
    echo "Cleanup exit..."
    exit 1
}

function dump_xrnginstall_funcs() {
    funcs="$@"; fg=0; OLDIFS="$IFS"

    SCRFILE=/etc/init.d/xrnginstall
    if [ -f /etc/rc.d/init.d/pxe_install.sh ]; then
        SCRFILE=/etc/rc.d/init.d/pxe_install.sh
    fi

    >/tmp/fdump
    while IFS= read -r line ; do
        if [[ $line =~ "^function " ]]; then
            set -- $line
            for i in $funcs; do
                if [[ "$i" == "$2" ]]; then
                    fg=1; break
                fi
            done
        fi
        if [ $fg -eq 1 ]; then
            echo "$line"
        fi
        if [[ $line =~ "^}" ]]; then
            fg=0
        fi
    done < $SCRFILE >/tmp/fdump
    IFS="$OLDIFS"
    sync
}

# Sort provided list of packages (rpms) to be
# in the follow order:
#   - PI Mandatory, PD Mandatory, PI Optional, PD Optional 
function sort_xr_packages {
    local -a __packages=("${!1}")
    local -a __sorted_packages
    local __sorted_packages_ref=${2}

    local -a opt_package_strings=('mcast' 'mgbl' 'mpls' 'k9sec')
    local -a pi_packages
    local -a pd_packages
    local -a pi_opt_packages
    local -a pd_opt_packages

    for file in ${__packages[@]}
    do
        is_optional=0
        for opt_str in ${opt_package_strings[@]}
        do
            case ${file} in
                *-${opt_str}-*)
                    # Optional package
                    is_optional=1;
                    break
                    ;;
                *)
                    ;;
            esac
        done


        case ${file} in
            iosxr-*)
                # PI package
                if ((${is_optional})) ; then
                    pi_opt_packages[${#pi_opt_packages[*]}]=${file}
                else
                    pi_packages[${#pi_packages[*]}]=${file}
                fi
                ;;
            *)
                # PD package 
                if ((${is_optional})) ; then
                    pd_opt_packages[${#pd_opt_packages[*]}]=${file}
                else
                    pd_packages[${#pd_packages[*]}]=${file}
                fi
                ;;
        esac
    done

    __sorted_packages=( ${pi_packages[@]} ${pd_packages[@]} ${pi_opt_packages[@]} ${pd_opt_packages[@]} )

    eval ${__sorted_packages_ref}="(${__sorted_packages[@]})"
}

## install_xr_iso - picked directly from xrnginstall script and made
## some  modification, as it does not work directly.
function install_xr_iso {
    local dev_name=${1}
    local dev=/dev/${dev_name}
    local device_part=${2}

    local -a packages
    local -a sorted_packages

    # XR iso is not the same as host iso
    step_log_file "Install xr mount src iso ${ISOMNT}"
    if [ -f /iso/$SDR_ISO ]; then
        mount -o loop /iso/$SDR_ISO ${ISOMNT} || exitclean
    fi
     
    # Mount dest partition
    step_log_file "Install xr mount dest parition ${dev}${device_part}"
    mount -o discard ${dev}${device_part} ${PARTMNT} -o offset=${3},sizelimit=${4} || exitclean
    # Remove old files, maintained across mkfs
    step_log_file "Install xr clean ${PARTMNT}"
    if [ ! -z ${PARTMNT} ]; then
      rm -fr ${PARTMNT}/* > /dev/null 2>&1
    fi
    
    cd ${PARTMNT}
    if [ "$new_iso_format" == "yes" ]; then
	    step_log_file $"XR: new XR ISO format"
        if [ -d ${ISOMNT}/rpm/xr ]; then
            step_log_file "Install xr copy rpms"
            mkdir -p ${PARTMNT}/rpm/xr
            cp ${ISOMNT}/rpm/xr/* ${PARTMNT}/rpm/xr/
        fi
        zcat  ${ISOMNT}/boot/initrd.img | cpio -d -i >&107 2>&1
    else
        step_log_file "Install xr copy rpms"
        mkdir -p rpm/xr
        #cp /rpm/xr/* rpm/xr/
	    step_log_file "Install xr initrd.img"
        zcat /boot/initrd.img | cpio -d -i >&107 2>&1
    fi
    
    if [ -f /boot/xr-image.img ]; then
        step_log_file "Install xr xr-image.img"
        zcat /boot/xr-image.img | cpio -d -i >&107 2>&1
    fi

    if [ "$BOARDTYPE" == "LC" ]; then
      packages=(rpm/xr/*[.,_]lc*)
    else  # RP & SC case
      packages=(`ls rpm/xr/*.rp[-,_]*`)
    fi

    sort_xr_packages packages[@] sorted_packages

    for file in ${sorted_packages[@]}
    do
      case ${file} in
          *.rpm)
             STARTRPM=$(date +%s)
             chroot ${PARTMNT} rpm -i --replacefiles --nodeps ${file}  >> /tmp/rpmlog 2>&1
             ENDRPM=$(date +%s)
             DIFFRPM=$(( $ENDRPM - $STARTRPM ))
             SIZERPM=$(ls -lh ${file}| awk '{ print $5 }')
             step_log_file $"Install RPM ${file} of size ${SIZERPM} took ${DIFFRPM} sec"
             ;;
         esac
    done

    step_log_file "Install xr dir setup"
    rm -rf rpm/xr > /dev/null 2>&1
	
    mkdir -p ${PARTMNT}/proc ${PARTMNT}/sys ${PARTMNT}/root/.ssh
    mkdir -p ${PARTMNT}/tmp ${PARTMNT}/dev ${PARTMNT}/var
    chmod -R 700 ${PARTMNT}/root
    chown -R root ${PARTMNT}/*
    chgrp -R root ${PARTMNT}/*
    cd / > /dev/null 2>&1

    step_log_file "Install xr grub setup ${dev}${device_part}"
    if ! test -e ${PARTMNT}/boot/grub/grub.conf ; then
        test -e ${PARTMNT}/boot/grub/menu.lst && ln -s ./menu.lst ${PARTMNT}/boot/grub/grub.conf
    fi
   
    ##### Grub related settings
    local imgver=`grep "^XR version" ${PARTMNT}/etc/build-info.txt | awk '{print $4}'`
    echo "Image version: $imgver"
    
    ## Calculate the Bigphysarea based on cardtype
    source /etc/rc.d/init.d/pd-functions
    local bigphysarea_pd=`pd_get_bigphysarea $BOARDTYPE $PD_CARDTYPE $VM`

    if [ -z "$bigphysarea_pd" ]; then
        local bigphysarea_rp=200M
        local bigphysarea_lc_low=250M
        local bigphysarea_lc_high=1100M
        if [ "$BOARDTYPE" == "LC" ] || [ "$VM" == "xr_lcp" ]; then
            bigphysarea_pd="$bigphysarea_lc_low\\;$bigphysarea_lc_high"
        elif [ "$BOARDTYPE" == "RP" ]; then
            bigphysarea_pd="$bigphysarea_rp"
        fi
    fi

    local menu_lst="${PARTMNT}/boot/grub/menu.lst"
    if [ "$imgver" == "5.0.1" ] || [ "$imgver" == "5.2.1" ]; then
        sed -i -e "s;root=/dev/ram;root=/dev/vda1 platform=$PLATFORM boardtype=$BOARDTYPE vmtype=xr-vm;" $menu_lst
        sed -i -e "s;crashkernel=[^ ]*;crashkernel=0;" $menu_lst
        sed -i -e "s;console=ttyS0;console=tty0 console=hvc0;" $menu_lst
    else
        sed -i -e "s;root=.*;${IOS_XR_CMDLINE};" $menu_lst
        sed -i -e "s;root=[^ ]* ;root=LABEL=XR ;" $menu_lst
        sed -i -e "s;vmtype=[^ ]* ;vmtype=xr-vm ;" $menu_lst
    fi
    sed -i -e "s;bigphysarea=[^ ]* ;bigphysarea=$bigphysarea_pd ;" $menu_lst
    if [ "$BOARDTYPE" == "LC" ] || [ "$VM" == "xr_lcp" ]; then
        sed -i -e "s;boardtype=[^ ]* ;boardtype=LC ;" $menu_lst
    fi

    echo "****************************"
    echo "Menu.lst after changes"
    cat $menu_lst

    sync
   
    $grub_shell --batch --no-floppy --device-map=/dev/null <<EOF >&107 2>&1
device (hd0) ${dev}${device_part}
root (hd0,0)
setup $force_lba --stage2=${PARTMNT}/boot/grub/stage2 (hd0)
quit
EOF

    sed -i -e "s;root (cd); root (hd0,0);" $menu_lst
    sed -i -e "s;ONBOOT=yes;ONBOOT=no;" ${PARTMNT}/etc/sysconfig/network-scripts/ifcfg-eth0
    sed -i -e "s;HOSTNAME=host;HOSTNAME=xr-vm;" ${PARTMNT}/etc/sysconfig/network

    run_depmod ${PARTMNT}
    sync
    SIZE=$(du -hs --exclude="/proc" --exclude="/sys" ${PARTMNT} | awk '{print $1}')
    [ -d "${PARTMNT}" ] && umount ${PARTMNT}
    [ -d "${ISOMNT}" ] && umount ${ISOMNT} >&107 2>&1
}

## install_sim_tools is called from install_host_iso
function install_sim_tools() {
    # we need to untar the binaries we need in the target partition
    step_log_console "SIMULATION PLATFORM: Unpacking tools.."
    mkdir -p ${PARTMNT}/opt/simulation
    pushd ${PARTMNT}/opt/simulation >&107 2>&1
    bzip2 -dc /opt/simulation/install.tar.bz2 |  tar --no-acls -xf -
    popd >&107 2>&1
    
    sed -i "s;is_flat=0;is_flat=1;g" ${PARTMNT}/opt/simulation/install/patches/*
    sed -i "s;/sbin/ifconfig -a | grep eth1;cat /tmp/eth1val.txt;g" ${PARTMNT}/opt/simulation/install/patches/*
    rm -f /dev/vda1
}

## Calls install_host_iso from xrnginstall script
## Note: In case if a host_smu is given, the host ISO in install_repo from mini ISO is 
## replaced by the host ISO from host SMU in instmgr_check_for_host_upgrade_extract_iso
## function.
function create_host_vmpart() {
    local lv=$1; local lvid=$2; local card_type=$3; local iso=$5; local host_smu=$6
    local lvname=$lv$lvid

    run_cmd setup_vars
    run_cmd dump_xrnginstall_funcs "\"install_host_iso remove_sim_tools \
                           vxr_sim_mode_check run_depmod set_iso_file_name check_iso_type create_ssh_keys\""
    source /tmp/fdump

    ## Create SSH keys
    if [ ! -d $SSHKEY_DIR ]; then
        run_cmd create_ssh_keys
    fi

    ## Copy rpms from host iso
    echo "Copying RPMs from Host ISO."
    isomnt=`mktemp -d /tmp/.su_isomnt.XXXXXX`
    sync
    run_cmd mount -o loop /iso/$iso $isomnt
    run_cmd cp $isomnt/rpm/*.rpm /rpm/calvados/
    run_cmd umount $isomnt
    rm -rf $isomnt

    export HOST_ISO="$iso"
    if [ -f /iso_name.txt ]; then
        export SDR_ISO=`cat    /iso_name.txt | grep SDR_ISO:    | awk '{print $2}'`
    fi
    local devname=`basename $4`
    local devpath="/dev/$devname"
    echo "Using device node $devpath"
    set +e
    run_cmd check_fs_partition "$devpath" "HostOs_lv$lvid"
    run_cmd install_host_iso "$devname" ""
    set -e
    sync

    ## Make some more changes to host vm partition
    run_cmd mount -o discard $devpath ${PARTMNT}
    
    run_cmd mkdir -p ${PARTMNT}/install/instdb/local/
    run_cmd "echo $lvid >${PARTMNT}/install/instdb/local/self_lv_index.txt"
    
    if [ "$host_smu" != "" ] && [ ! -f /iso/sim.txt ]; then
        echo "Installing thirdparty RPMs..."
        mkdir ${PARTMNT}/rpm/calvados/
        cp /rpm/calvados/*thirdparty*.rpm ${PARTMNT}/rpm/calvados/; sync

        if [ "$card_type" == "lc" ]; then
            packages=(/rpm/calvados/*thirdparty*all* /rpm/calvados/*thirdparty*[.,_]lc*)
        elif [ "$card_type" == "rp" ]; then
            packages=(/rpm/calvados/*thirdparty*all* /rpm/calvados/*thirdparty*.rp[-,_]*)
        elif [ "$card_type" == "sc" ]; then
            packages=(/rpm/calvados/*thirdparty*all* /rpm/calvados/*thirdparty*.sc[-,_]*)
        else
            echo "Error: Unknown card type."
            return 1
        fi
       
        for file in ${packages[@]}; do
            STARTRPM=$(date +%s)
            chroot ${PARTMNT} rpm -i --nodeps ${file} >>/dev/null 2>&1
            ENDRPM=$(date +%s)
            DIFFRPM=$(( $ENDRPM - $STARTRPM ))
            SIZERPM=$(ls -lh ${file}| awk '{ print $5 }')
            echo "Install RPM ${file} of size ${SIZERPM} took ${DIFFRPM} sec"
        done
        rm -rf ${PARTMNT}/rpm/calvados/
    fi

    if [ -f /iso/sim.txt ]; then
        ## Hack to make it work on Simulator: Remove *simBridge files to avoid
        ## running tunctl command during boot.
        run_cmd "find ${PARTMNT}/etc -name \"*simBridge\" -exec rm -f {} \;" || true
    fi
    
    run_cmd "cp -arf /boot/grub2/* ${PARTMNT}/boot/grub2/" || true
    run_cmd umount ${PARTMNT}
    sync
}

function create_calxr_vmpart() {
    lv=$1; lvid=$2; card_type=$3; pd_cardtype=$4
    lvname=$lv${lvid}
    vm=`echo $lv | sed 's/_lv$//'`
  
    run_cmd setup_vars
    run_cmd dump_xrnginstall_funcs "\"install_calvados_iso remove_sim_tools run_depmod \
            create_sub_part_file_disk_boot check_fix_disk_part create_fdisk_partition \ 
            find_fdisk_total_partition mark_part_bootable setup_loop_offset set_iso_file_name \
            find_fs_offset find_fs_sizelimit get_packages_to_be_installed \
            vxr_sim_mode_check check_iso_type create_ssh_keys\""
    source /tmp/fdump

    if [ "$vm" == "calvados" ]; then
        local vname="Calvados"; local fname="install_calvados_iso"
        ## Create SSH keys
        if [ ! -d $SSHKEY_DIR ]; then
            run_cmd create_ssh_keys
        fi
    elif [ "$vm" == "xr" ] || [ "$vm" == "xr_lcp" ]; then
        local vname="XR"; local fname="install_xr_iso"
    else
        echo "Invalid VM."
        return 1
    fi
    local iso=$6; local part="1"
    local devname=`basename $5`
    local devpath="/dev/$devname"
    echo "Using device node $devpath"

    export ADMIN_ISO="$iso"
    export SDR_ISO="$iso"
    export BOARDTYPE=`echo $card_type | tr '[:lower:]' '[:upper:]'`
    export PD_CARDTYPE="$pd_cardtype"
    export new_iso_format=yes
    export VM="$vm"
    echo "ADMIN_ISO=$iso SDR_ISO=$iso BOARDTYPE=$BOARDTYPE VM=$VM"
    run_cmd create_sub_part_file_disk_boot
    set +e
    START=$(date +%s)
    run_cmd check_fix_disk_part "$devname" "1" "/tmp/fdisk_boot.input"
    run_cmd mark_part_bootable "$devname" "${devname}${part}"
    for subprt in ${part} ; do
        loop_dev=$(setup_loop_offset "$devpath" "${subprt}")
        if [ -z "${loop_dev}" ]; then
            step_log_console "Cannot setup loop device for ${devname}${subprt}"
        fi     
        check_fs_partition "${loop_dev}" "$vname"
        losetup -d ${loop_dev}
    done
    END=$(date +%s)
    DIFF=$(( $END - $START ))
    step_log_console "File system creation on $devpath took ${DIFF} seconds"
    loop_dev=$(setup_loop_offset "$devpath" "${part}")
    part_offset=$(find_fs_offset "$devpath" "${part}")
    part_szlmt=$(find_fs_sizelimit "$devpath" "${part}")
    step_log_console "Install ${vm}-vm image on $devpath"
    START=$(date +%s)
    run_cmd "$fname \"$devname\" \"\" ${part_offset} ${part_szlmt}"
    sync
    END=$(date +%s)
    DIFF=$(( $END - $START ))
    step_log_console "Installing ${vm}-vm image took ${DIFF} seconds"
    sleep 2
    losetup -d ${loop_dev}
    set -e

    run_cmd "mount -o discard ${devpath} ${PARTMNT} -o offset=${part_offset},sizelimit=${part_szlmt} || exitclean"
    run_cmd mkdir -p ${PARTMNT}/install/instdb/local/
    run_cmd "echo $lvid >${PARTMNT}/install/instdb/local/self_lv_index.txt"
    run_cmd "umount ${PARTMNT}"
    run_cmd rm -rf ${ISOMNT} ${PARTMNT}
}

function ips_cal() {
    if $CHVRF_CMD /opt/cisco/calvados/bin/show_cmd "show vm | nomore" | grep "VMs found at" >/dev/null; then
        $CHVRF_CMD /opt/cisco/calvados/bin/show_cmd "show vm | nomore" | \
                grep sysadmin -A 2 | grep IP | awk '{print $3}'
    else
        $CHVRF_CMD /opt/cisco/calvados/bin/show_cmd "show vm | nomore" | \
                grep sysadmin | awk '{print $3}'
    fi
}

function ips_xr() {
    if $CHVRF_CMD /opt/cisco/calvados/bin/show_cmd "show vm | nomore" | grep "VMs found at" >/dev/null; then
        $CHVRF_CMD /opt/cisco/calvados/bin/show_cmd "show vm | nomore" | \
                grep $SDR_KEY -A 2 | grep IP | awk '{print $3}'
    else
        $CHVRF_CMD /opt/cisco/calvados/bin/show_cmd "show vm | nomore" | \
                grep $SDR_KEY | awk '{print $3}'
    fi
}

function ips_all_nodes() {
    $CHVRF_CMD /opt/cisco/calvados/bin/show_cmd "show vm | nomore"
}

function runon_host() {
    $CHVRF_CMD ssh $HOSTVM_IP "$*"
}

function runon_all_cal() {
    for i in `ips_cal`; do
        echo "${i}: $*"
        $CHVRF_CMD ssh $i "$*"
    done
}

function runon_all_xr() {
    for i in `ips_xr`; do
        echo "${i}:"
        $CHVRF_CMD ssh $i "$*"
    done
}

function list_volumes() {
    if runon_host /sbin/lvdisplay 2>/dev/null | grep -e "LV Path" >/dev/null; then
        runon_host /sbin/lvdisplay 2>/dev/null | grep -e "LV Path" | awk '{print $3}'
    elif runon_host /sbin/lvdisplay 2>/dev/null | grep -e "LV Name" >/dev/null; then
        runon_host /sbin/lvdisplay 2>/dev/null | grep -e "LV Name" | awk '{print $3}'
    else
        false
    fi
}

function create_volume() {
    runon_host /sbin/lvcreate -L $1 -n $2 $3; sync
}

function delete_volume() {
    runon_host /sbin/lvremove -f $1; sync
}

function attach_volume() {
    volname="$1"
   
    (ls -1 /dev/vd* | sort)>/tmp/l1.txt; sync
    
    for i in {e..z}; do
        if runon_host /usr/bin/virsh attach-disk sysadmin ${volname} vd$i --cache=none >/dev/null 2>/dev/null; then
            break
        else
            true
        fi
    done
    if [ "$i" == "z" ]; then
        echoerr "Attach volume failed for $1."
        return 1
    fi
 
    ## Look out for new device node that appears in /dev
    local ctr=0
    while [ $ctr -ne 5 ]; do
        (ls -1 /dev/vd* | sort)>/tmp/l2.txt
        if vdev=`diff /tmp/l1.txt /tmp/l2.txt | awk '/^>/ {print $2}' | tail -n 1`; then
            sleep 1; ctr=$(($ctr + 1))
        else
            break
        fi
    done

    if [ $ctr -eq 5 ] ; then
        echoerr "Virtual device not available."
        return 1
    else
        echo "$vdev" | sed -e "s/[0-9]*$//"
        return 0
    fi
    sync
}

function detach_volume() {
    runon_host /usr/bin/virsh detach-disk sysadmin $1
}

function get_instmgr_ip() {
    for i in `ips_cal`; do
        if $CHVRF_CMD ssh $i "ps -e | grep inst_mgr" >/dev/null 2>&1; then
            echo $i
            return
        else
            true
        fi
    done
}

function get_instmgr_shell() {
    $CHVRF_CMD ssh `get_instmgr_ip`
}

function cp_all_cal() {
    for i in `ips_cal`; do
        $CHVRF_CMD scp $1 $i:$2
    done
    sync
}

function cp_all_xr() {
    for i in `ips_xr`; do
        $CHVRF_CMD scp $1 $i:$2
    done
    sync
}

function list_repo() {
    vm=$1
    find /install_repo/gl/$vm/
}

## Get files from node where instmgr is running
## Usage: get_repo_files local_dir_to_copy remote_dir_having_files files
function get_repo_files() {
    dest=$1; src=$2
    shift 2

    set -xv
    instmgr_ip=`get_instmgr_ip`

    for i in $*; do
        echo "Downloading file $i ..."
        ctr=0
        while [ $ctr -lt 5 ]; do
            run_cmd rm -f $dest/$i
            if $CHVRF_CMD scp -q $instmgr_ip:$src/$i $dest; then
                break
            else
                ctr=$(( $ctr + 1 ))
            fi
            echo "File download $i try $ctr failed..."
            sleep 5
        done
          
        run_cmd ls -l $dest/$i
        if [ $ctr -eq 5 ]; then
            return 1
        fi
    done
    sync
    set +xv
}

function prune_volumes() {
    run_cmd umount_sysdirs
   
    vlist=`list_volumes | grep -e "calvados_lv[0-9]*$" -e "host_lv[0-9]*$" \
                                -e "xr_lv[0-9]*$" -e "xr_lcp_lv[0-9]*$" \
                                -e "calvados_data_lv[0-9]*$" -e "xr_data_lv[0-9]*$" \
                                -e "xr_lcp_data_lv[0-9]*$"`
    echo -e "Linux volumes list:\n$vlist"

    for i in `find /install_repo/local -maxdepth 1 -name "calvados*lv_state.txt" \
        -o -name "host*lv_state.txt"; find $SDR_LOCAL1 -maxdepth 1 -name "xr*lv_state.txt"`; do
        for j in `awk '{print $1" "$2}' $i`; do
             nd="`basename $i _state.txt`$j"
             vlist=`echo "$vlist" | grep -v -w ${nd}$` || true
        done
    done
    echo -e "\nLinux volumes list for removal:\n$vlist"

    echo -e "\nLV state files content:\n`print_lv_states`"

    echo -e "\nRemoving linux volumes:"
    for i in $vlist; do
        detach_volume $i 2>/dev/null || true
        run_cmd delete_volume $i
    done
}

function get_unique_lvid() {
    lvid=$1
    vlist=`list_volumes`

    fg=0
    while [ $fg -eq 0 ]; do
        fg=1
        
        # First, compare with available logical volumes
        for i in $vlist; do
            declare -a vms=('host' 'calvados' 'xr' 'xr_lcp');
            for j in ${vms[@]}; do
                if [ "/dev/panini_vol_grp/${j}_lv${lvid}" == "$i" ] || \
                    [ "/dev/panini_vol_grp/${j}_data_lv${lvid}" == "$i" ] || \
                        [ "/dev/pci_disk1/${j}_data_lv${lvid}" == "$i" ]; then
                    lvid=`expr $lvid + 1`
                    fg=0
                    break;
                fi
            done
            if [ $fg -eq 0 ]; then
                break
            fi
        done
   
        if [ $fg -eq 0 ]; then
            continue
        fi
       
        # Second, compare with lv_state file entries
        for i in `find /install_repo/ -name "*lv_state*" -exec cat {} \; -exec echo"" \;`; do
            if [ "$lvid" -eq "$i" ]; then
                lvid=`expr $lvid + 1`
                fg=0
                break;
            fi
        done
    done
    
    echo $lvid
}

function get_lv_state() {
    vm=$1; part=$2
    cxx=0; axx=1; pxx=2

    if [ "$vm" == "xr" ] || [ "$vm" == "xr_lcp" ]; then
        read -a lvarr <<< `cat $SDR_LOCAL1/${vm}_lv_state.txt`
        echo ${lvarr[`eval echo \\$${part}xx`]}
    else
        read -a lvarr <<< `cat /install_repo/local/${vm}_lv_state.txt`
        echo ${lvarr[`eval echo \\$${part}xx`]}
    fi
}

function set_lv_state() {
    lv=$1; part=$2; partval=$3
    vm=`echo $lv | sed 's/_lv$//'`
    cxx=0; axx=1; pxx=2

    if [ "$vm" == "xr" ] || [ "$vm" == "xr_lcp" ]; then
        read -a lvarr <<< `cat $SDR_LOCAL1/${vm}_lv_state.txt`
        lvarr[`eval echo \\$${part}xx`]=$partval
        echo ${lvarr[@]} > $SDR_LOCAL1/${vm}_lv_state.txt
    else
        read -a lvarr <<< `cat /install_repo/local/${vm}_lv_state.txt`
        lvarr[`eval echo \\$${part}xx`]=$partval
        echo ${lvarr[@]} > /install_repo/local/${vm}_lv_state.txt
    fi
    sync
}

## Prints lv_state file contents for all VMs.
function print_lv_states() {
    for i in `find /install_repo/local -maxdepth 1 -name "calvados*lv_state.txt" \
        -o -name "host*lv_state.txt"; find $SDR_LOCAL1 -maxdepth 1 -name "xr*lv_state.txt"`; do
        echo $i; cat $i; echo ""
    done
}

## Print Volumes used to run VMs.
function print_vm_vols() {
    runon_host "echo -n \"root=\"; $(/usr/bin/xr_sysctl -n kernel.boot.root)"
    runon_host "ps aux | grep qemu | tr \"-\" \"\\n\" | grep ^drive | grep panini"
}

## Extract mini ISO contents
## Usage: process_bundle_iso mini-iso-image-file
function process_bundle_iso() {
    miniiso=$1
    
    isomnt=`mktemp -d /tmp/.su_isomnt.XXXXXX`
    sync
    mount -o loop $miniiso $isomnt
  
    ver=`sed -e "s/Version:/\nVersion:/" $isomnt/iso_info.txt | grep "^Version:" | awk '{print $2}'`

    mkdir -p /install_repo/tmp /install_repo/tmp_staging
    rm -rf /install_repo/tmp/* /install_repo/tmp_staging/*
    cd /install_repo/tmp 
    zcat $isomnt/boot/initrd.img | cpio -idu --quiet
    mv iso/host.iso ../gl/host/host-$ver
    mv iso/ncs6k-sysadmin.iso ../gl/calvados/ncs6k-sysadmin-$ver
    mv iso/ncs6k-xr.iso ../gl/xr/ncs6k-xr-$ver
    rm -rf usr/lib64/valgrind/ usr/bin/crash usr/bin/qemu-system-x86_64 || true
    tar -zcf /install_repo/gl/host/newroot.tar.gz *
    rm -rf *
    cd /

    umount $isomnt
    rm -rf $isomnt
    sync
}

function validate_iso() {
    set +e
    miniiso=$1
    isomnt=`mktemp -d /tmp/.su_isomnt.XXXXXX`
    sync

    run_cmd mount -o loop $miniiso $isomnt

    initrd_md5=$(md5sum $isomnt/boot/initrd.img | awk '{ print $1 }' )
    echo "Initrd md5 = $initrd_md5"
    if grep "^Initrd" $isomnt/iso_info.txt; then
        iso_md5=$(grep "^Initrd" $isomnt/iso_info.txt | awk '{print $3}' )
    else
        umount $isomnt
        rm -rf $isomnt
        return 0
    fi

    umount $isomnt
    rm -rf $isomnt
    sync

    if [ "$initrd_md5" == "$iso_md5" ]; then 
        echo "iso_md5 = $iso_md5"
        return 0 
    else
        return 1  
    fi;
    set -e
}

## extract_v2install_script is run by Cal instmgr. This extracts the 
## install script from V2 ISO and stores in /install_repo/tmp_staging.
function extract_v2install_script() {
    iso=$1; shift 1

    if [ ! -f /tmp/support_v2_script ]; then
        return 0
    fi

    v2scr="/install_repo/tmp_staging/`basename $INSTALL_SCRIPT`.v2"
    mkdir -p /install_repo/tmp /install_repo/tmp_staging/
    run_cmd rm -f $v2scr

    ## Check sys-admin SMUs
    echo "Looking for install script in SMUs."
    for i in $*; do
        tmpdir=`mktemp -d /install_repo/tmp/.smu.XXXXXX`
        mkdir -p $tmpdir; sync; cd $tmpdir
        tar xf /install_repo/gl/calvados/$i >/dev/null 2>&1
        for j in *.rpm; do
            if rpm -qpl $j | grep "$INSTALL_SCRIPT" >/dev/null 2>&1; then
                ifile=`rpm -qpl $j | grep "$INSTALL_SCRIPT" | head -n 1`
                echo "Found $INSTALL_SCRIPT in SMU:$i RPM:$j"
                /usr/lib/rpm-5.1.9/rpm2cpio $j | cpio -i --to-stdout ".$ifile" >$v2scr
                if [ ! -s $v2scr ]; then
                    rm -f $v2scr
                fi
                cd /; rm -rf $tmpdir
                if [ -f $v2scr ]; then
                    run_cmd bash -n $v2scr; sync
                fi
                return 0
            fi
        done
        cd /; rm -rf $tmpdir
    done

    ## Check ISO
    echo "Looking for install script in sys-admin ISO."
    isomnt=`mktemp -d /tmp/.su_isomnt.XXXXXX`
    sync
    mount -o loop /install_repo/gl/calvados/$iso $isomnt
    cd $isomnt/rpm/calvados
    for j in *.rpm; do
        if rpm -qpl $j | grep "$INSTALL_SCRIPT" >/dev/null 2>&1; then
            ifile=`rpm -qpl $j | grep "$INSTALL_SCRIPT" | head -n 1`
            echo "Found $INSTALL_SCRIPT in ISO:$iso RPM:$j"
            run_cmd "/usr/lib/rpm-5.1.9/rpm2cpio $j | cpio -i --to-stdout \".$ifile\" >$v2scr"
            if [ ! -s $v2scr ]; then
                rm -f $v2scr
            fi
            break
        fi
    done
    cd /
    umount $isomnt
    rm -rf $isomnt
    if [ -f $v2scr ]; then
        run_cmd bash -n $v2scr; sync
    fi
    return 0
}

## get_v2install_script is run by all Cal instagts. This gets the V2
## install script from repo to local /install
function get_v2install_script() {
    run_cmd rm -f ${INSTALL_SCRIPT}.v2; sync
    if [ ! -f /tmp/support_v2_script ]; then
        return 0
    fi
    
    run_cmd get_repo_files `dirname $INSTALL_SCRIPT` /install_repo/tmp_staging/ "`basename $INSTALL_SCRIPT`.v2" || true
    if [ ! -s ${INSTALL_SCRIPT}.v2 ]; then
        rm -f ${INSTALL_SCRIPT}.v2
    fi
    sync
}

function mount_sysdirs() {
    if [ $# -ne 1 ]; then
        echo "Provide root dir."
        return 1
    fi
    rootdir=$1
    for i in proc sys; do
        run_cmd "mount -o ro --bind /$i $rootdir/$i || true"
    done
    sync
}

function umount_sysdirs() {
    ## Kill any process making use of new fs and umount sysdirs
    cd /
    run_cmd losetup -a
    for i in /dev/loop[0-9]*; do losetup -d $i 2>/dev/null || true; done
    run_cmd "lsof | grep -e $NEWROOT -e $MNTDIR_NAME | sort -u -k1,2 | \
                awk '{system(\"echo Killing: \"\$0\";kill -9 \"\$2)}'" || true
                
    for retry in seq 10; do
        ufg=0
        for i in `mount | grep -e $NEWROOT -e $MNTDIR_NAME | awk '{print $3}' | tac`; do
            if run_cmd umount $i; then
                sync
                continue
            else
                ufg=1
                break
            fi
        done
        if [ $ufg -eq 0 ]; then break; fi
        sleep 3
    done

    run_cmd mount
}

## Setup env to prepare VM partition. Get and extract the newroot.tar.gz file. 
## This is the tar of the rootfs V2 ISO's initrd.
function setup_prepvm_env() {
    newroot_path=$1; host_iso=$2

    run_cmd umount_sysdirs
    run_cmd "rm -rf $NEWROOT; mkdir -p $NEWROOT; cd $NEWROOT; tar zxf $newroot_path"
    run_cmd "cd $NEWROOT; mkdir -p proc sys dev dev/pts ${NEWROOT}/`dirname $INSTALL_SCRIPT` iso cdrom tmp opt/simulation"
    run_cmd "cp $INSTALL_SCRIPT ${NEWROOT}/$INSTALL_SCRIPT"
    ## For Simulator - only flat supported now
    if [ -f /iso/sim.txt ]; then
        run_cmd $CHVRF_CMD scp $HOSTVM_IP:/opt/simulation/Install.tar.bz2 $NEWROOT/opt/simulation/install.tar.bz2.loader
        run_cmd $CHVRF_CMD scp $HOSTVM_IP:/opt/simulation/Patch.tar.bz2 $NEWROOT/opt/simulation/install.tar.bz2
        run_cmd $CHVRF_CMD scp $HOSTVM_IP:/opt/simulation/configvector.txt $NEWROOT/opt/simulation/
        run_cmd cp /iso/sim.txt $NEWROOT/iso/
        run_cmd "$CHVRF_CMD ssh $HOSTVM_IP /sbin/ifconfig -a | grep eth1 > $NEWROOT/tmp/eth1val.txt"
    fi
        
    # Install RPMs from host ISO to newroot. This is done during PXE 
    # install as well. We will follow the same here.
    isomnt=`mktemp -d /tmp/.su_isomnt.XXXXXX`
    sync
    run_cmd mount -o loop $host_iso $isomnt
    HOSTOS_RPM_SRC=`ls -1 ${isomnt}/rpm/*hostos.all*.rpm | head -1`
    run_cmd rpm -iv --nodeps --noscripts --ignoresize --excludepath /opt/cisco/hostos \
                            --root=$NEWROOT $HOSTOS_RPM_SRC
    run_cmd umount $isomnt
    rm -rf $isomnt
    
    cd /
}

function cleanup_prepvm_env() {
    run_cmd umount_sysdirs
    run_cmd "rm -rf $NEWROOT"; cd /
}

## prepare_vmpart can be called to create host, calvados and xr vm partitions.
## It must it called in that order to work properly on Simulator. The mini iso
## must be broken up before making a call to this. This will be done in the install
## code. For running this function in the bash shell, the mini iso can be broken
## up using the process_bundle_iso function.
## Note: In case if a host_smu is given, the host ISO in install_repo from mini ISO is 
## replaced by the host ISO from host SMU in instmgr_check_for_host_upgrade_extract_iso
## function.
function prepare_vmpart() {
    echo "prepare vm part: $@"
    local ret_val=0
    lv=$1; lvid=$2; card_type=$3; pd_cardtype=$4; iso_path=$5; local host_smu=$6
    vm=`echo $lv | sed 's/_lv$//'`

    if [ ! -f ${NEWROOT}/$INSTALL_SCRIPT ]; then
        echo "Newroot not available. Run setup_prepvm_env first."
        return 1
    fi

    if [ ! -f $iso_path ]; then
        echo "ISO not found: $iso_path"
        return 1
    fi
    iso=`basename $iso_path`

    echo "Preparing \"$vm\" VM now..."
    sync
    run_cmd md5sum $iso_path
    run_cmd cp $iso_path $NEWROOT/iso
    run_cmd umount_sysdirs
    run_cmd mount_sysdirs "$NEWROOT"
    sync
    
    ## Host does not have partitions. Calvados and XR have similar partition format.
    ## Creating Host before Calvados a must on Simulator as during Host prepare
    ## some Sim related changes are made to newroot needed during Cal prepare.
    if [ "$vm" == "host" ]; then
        run_cmd create_volume 1000M $lv$lvid panini_vol_grp; sleep 2
        atdev=`attach_volume /dev/panini_vol_grp/${lv}${lvid}`
        echo "Attached volume is $atdev"
        run_cmd "cp -af /dev/* $NEWROOT/dev 2>/dev/null || true"; sync
        run_cmd "chroot $NEWROOT bash -c \"source $INSTALL_SCRIPT; unset LD_PRELOAD;" \
                                "create_host_vmpart $lv $lvid $card_type $atdev $iso $host_smu\""

        if [ -f /iso/sim.txt ]; then
            run_cmd runon_host "dd if=/dev/flat_disks/$lv$lvid of=/dev/panini_vol_grp/$lv$lvid bs=4M"
        fi
    elif [ "$vm" == "calvados" ]; then
        run_cmd create_volume 2500M $lv$lvid panini_vol_grp; sleep 2
        atdev=`attach_volume /dev/panini_vol_grp/${lv}${lvid}`
        echo "Attached volume is $atdev"
        run_cmd "cp -af /dev/* $NEWROOT/dev 2>/dev/null || true"; sync
        run_cmd "chroot $NEWROOT bash -c \"source $INSTALL_SCRIPT; unset LD_PRELOAD;" \
                                "create_calxr_vmpart $lv $lvid $card_type $pd_cardtype $atdev $iso\""

        # Install repo migration: Copy new to old 
        inst_is_large_repo_supported 
        if [ $? -eq 1 ]; then
            if [ ${card_type} == "rp" ] || [ ${card_type} == "cc" ]; then
                PARTMNT=`mktemp -d /tmp/${MNTDIR_NAME}.XXXXXX`
                sync
                mount ${atdev}1 ${PARTMNT}
                ret_log=$(copy_install_repo_new2old "${PARTMNT}")
                echo "$ret_log" | grep -q "$srch_str"
                if [ $? -eq 0 ]; then
                    echo "$ret_log"
                    ret_val=1
                fi
                umount ${PARTMNT}
                rm -rf ${PARTMNT}
            else
                step_log_console "Not running the repo replication for $vm and $card_type"
            fi
        else 
            step_log_console "This platform or card does not support large repo "
        fi

    elif [ "$vm" == "xr" ] || [ "$vm" == "xr_lcp" ]; then
        run_cmd create_volume 4000M $lv$lvid panini_vol_grp; sleep 2
        atdev=`attach_volume /dev/panini_vol_grp/${lv}${lvid}`
        echo "Attached volume is $atdev"
        run_cmd "cp -af /dev/* $NEWROOT/dev 2>/dev/null || true"; sync
        run_cmd "chroot $NEWROOT bash -c \"source $INSTALL_SCRIPT; unset LD_PRELOAD;" \
                                "create_calxr_vmpart $lv $lvid $card_type $pd_cardtype $atdev $iso\""
   
        if [ -f /iso/sim.txt ]; then
            echo "Applying Sim patches..."
            PARTMNT=`mktemp -d /tmp/${MNTDIR_NAME}.XXXXXX`
            sync
            mount ${atdev}1 ${PARTMNT}
            if [ -f /opt/simulation/xr_patch_from_calvados.sh ]; then
                echo "Running /opt/simulation/xr_patch_from_calvados.sh"
                /opt/simulation/xr_patch_from_calvados.sh ${PARTMNT}
            fi

            if [ -f /opt/cisco/calvados/bin/calvados_patch_xr_iso.sh ]; then
                echo "Running /opt/cisco/calvados/bin/calvados_patch_xr_iso.sh"
                /opt/cisco/calvados/bin/calvados_patch_xr_iso.sh ${PARTMNT}
            fi
            umount ${PARTMNT}
            rm -rf ${PARTMNT}
        fi
    else
        echoerr "Invalid VM: $vm"
    fi

    run_cmd umount_sysdirs
    run_cmd detach_volume /dev/panini_vol_grp/${lv}${lvid} 2>/dev/null || true
    run_cmd rm -f $NEWROOT/iso/$iso

    cd /; sync
    return $ret_val
}

## create_swp_install_smu goes through the list of /install_repo/cache/${lvname}_pkgs.txt 
## files and creates SWP. Based on the platform and card type these files are already
## created by caller to this functions via prepare_vmpart function. This function 
## creates prepared SWP for calvados and loadpath & prepared SWP for xr vm.
function create_swp_install_smu() {
    lv=$1; lvid=$2; card_type=$3; vm_pkgs=$4

    if [ ! -f $vm_pkgs ]; then
        echo "$vm_pkgs file not found."
        return 1
    fi

    local prtmnt=`mktemp -d /tmp/${MNTDIR_NAME}.XXXXXX`; sync
    
    echo "Attach and mount V2 $lv"
    atdev=`attach_volume /dev/panini_vol_grp/${lv}${lvid}`
    sleep 3
    run_cmd mount -o discard ${atdev}1 $prtmnt; sync

    echo "Install SMU/Pkgs on mounted V2 $lv"
    run_cmd rm -f $prtmnt/rpmdb/lib/rpm/__db.00*
    mkdir -p $prtmnt/install_repo
    run_cmd mount -o ro --rbind /install_repo $prtmnt/install_repo
    run_cmd mount --make-slave $prtmnt/install_repo
    run_cmd /opt/cisco/calvados/bin/inst_agent -s $lv $lvid \
                    $card_type $SDR_KEY $prtmnt `cat $vm_pkgs` 2>&1
    run_cmd rm -f $prtmnt/rpmdb/lib/rpm/__db.00*
    run_cmd chroot $prtmnt rpm -qa >/dev/null 2>&1

    run_cmd umount_sysdirs
    
    run_cmd detach_volume /dev/panini_vol_grp/${lv}${lvid}

    run_cmd rm -rf $prtmnt || true
    sync
}

function su_version_fixes() {
    mkdir -p $SDR_LOCAL2
    if ls $SDR_LOCAL1/xr* >/dev/null 2>&1; then
        run_cmd cp $SDR_LOCAL1/xr* $SDR_LOCAL2
    fi

    mkdir -p /install_repo/gl/instdb/sdr/$SDR_KEY/pkg/
    for i in /install_repo/gl/xr/*; do
        ln -s $i /install_repo/gl/instdb/sdr/$SDR_KEY/pkg/ || true
    done
}

function refresh_install_script() {
    calvms_sshlogin
    cp /ws/calvados/system_pkg/install/inst_agent/`basename $INSTALL_SCRIPT` /tmp
    cp_all_cal /tmp/`basename $INSTALL_SCRIPT` $INSTALL_SCRIPT
}

function refresh_router_cal() {
    calvms_sshlogin
    runon_all_cal rm -rf /tmp/dir_to_tar /tmp/refresh_calv.tgz /tmp/refresh_router3
    $CHVRF_CMD /ws/calvados/system_pkg/install/inst_common/scripts/refresh_router3 -d
    cp_all_cal /ws/calvados/system_pkg/install/inst_common/scripts/refresh_router3 /tmp
    cp_all_cal /tmp/refresh_calv.tgz /tmp
    runon_all_cal /tmp/refresh_router3 >/tmp/logs.txt
    grep "^Done" /tmp/logs.txt

    instmgr_ip=`get_instmgr_ip`
    echo "Kill instmgr on $instmgr_ip"
    $CHVRF_CMD ssh $instmgr_ip killall -9 inst_mgr
    sleep 4
    echo "Kill instagts on all nodes"
    runon_all_cal killall -9 inst_agent
    runon_all_cal rm -rf /tmp/dir_to_tar /tmp/refresh_calv.tgz /tmp/refresh_router3
}

function dump_all_install_logs() {
    rm -f inst_mgr.txt
    touch inst_mgr.txt
    \ls -1 inst_mgr.log.* | sort -t "." -k 3 -n | while read line; do zcat "$line" >>inst_mgr.txt; done
    cat inst_mgr.log >>inst_mgr.txt
    ls -l

    rm -f inst_agent.txt
    touch inst_agent.txt
    \ls -1 inst_agent.log.* | sort -t "." -k 3 -n | while read line; do zcat "$line" >>inst_agent.txt; done
    cat inst_agent.log >>inst_agent.txt
    ls -l 
}

function install_precheck_repo ()
{
    # This is used to check the existence of install repository 
    # and sub directories inside the Install repository 
    # For example /install_repo should be mounted and
    # /install_repo/gl/xr and  /install_repo/gl/calvados 
    # should exist, For clients the script should be passed
    # with argument 1.

    if grep -qs '/install_repo' /proc/mounts; then
       echo "Install repository is Mounted"
    else
       echo "Install repository is not mounted"
       return 1
    fi

    if [ -d "/install_repo" ]; then 
        echo "Install repository exists"
    else 
        echo "Install repository Do not exist"
        return 1
    fi

    local client="$1"
    if [ "$client" -eq 1 ]; then
        return 0
    fi

    if [ -d "/install_repo/gl/xr" ]; then 
        echo "Install XR repository exists"
    else 
        echo "Install XR repository Do not exist"
        return 1
    fi

    if [ -d "/install_repo/gl/host" ]; then 
        echo "Install host repository exists"
    else 
        echo "Install host repository Do not exist"
        return 1
    fi

    if [ -d "/install_repo/gl/calvados" ]; then 
        echo "Install calvados repository exists"
    else 
        echo "Install calvados repository Do not exist"
        return 1
    fi

    return 0
}

function install_precheck_disk_check()
{
    #This checks the disk usage for the partition
    #sent as the parameter to the script. If disk usage
    #exceeds threshold,the script returns error
    local threshold=95
    local partition="$1"
    local usage=$(df -Ph "$partition" | sed 's/\s\+/ /' | awk -F' ' '{ print $5 }' | tail -n +2 | sed 's/%//')
    if [ $usage -ge $threshold ]; then
        echo "$partition Usage exceeds threshold, Error"
        echo $(df -Ph ${partition})
        return 1
    else 
        echo "$partition Usage well within the Threshold"
    fi
    return 0

}

#
# list files in rpm package
#
function inst_get_files_in_rpm()
{
   pkg_name=$1
   dest_path=$2
   is_su_operation=$3
   req_vm=$4

   # SU case
   if [ "$is_su_operation" = "TRUE" ]; then
    rm -rf  /tmp/smu_files
    pkg_path=$(find /install_repo/ -name "$pkg_name" | head -1)

    mkdir -p /tmp/smu_files
    cd /tmp/smu_files
    echo "$pkg_path"
    tar vxf $pkg_path >/dev/null
    if [ $? = 0 ]; then
        rpm -qp --qf '[%{FILENAMES}\n]' *.rpm |\
            sort -u | sed "s|CSC[a-z0-9]\{7\}|CSCxxxxxxx|g" | \
            sed "s|CSCxxxxxxx_[0-9]*|CSCxxxxxxx|g" > $dest_path/$pkg_name
    else
       echo "package or path doesn't exist"
    fi
    rm -rf /tmp/smu_files

   else 

     pkg_pfx=$(echo ${pkg_name%-*})
     ddts="${pkg_pfx##*.}"
     echo "PKG : $pkg_pfx, ddts $ddts"
     if [ "$req_vm" = "XR" ]; then
      pkgs=$(ls /opt/cisco/XR/packages/ | grep "$ddts")
     else
      pkgs=$(ls /opt/cisco/calvados/packages/ | grep "$ddts") 
     fi

     if [ -z "$pkgs" ]; then
         echo "No packages found  $pkgs"
     else
        rpm -q --qf '[%{FILENAMES}\n]' $pkgs | sort -u | \
           sed "s|CSC[a-z0-9]\{7\}|CSCxxxxxxx|g" | \
           sed "s|CSCxxxxxxx_[0-9]*|CSCxxxxxxx|g" >$dest_path/$pkg_name
     fi

   fi
}

# get obsoletes information from
# RPMs

function inst_get_obsoletes_from_rpm()
{
   pkg_name=$1
   is_su_operation=$2
   admin=$3

   if [ "$is_su_operation" = "TRUE" ]; then
       rm -rf /tmp/my_sup_dir
       pkg_path=$(find /install_repo/ -name "$pkg_name" | head -1)
       mkdir -p /tmp/my_sup_dir
       cd /tmp/my_sup_dir
       echo "$pkg_path"
       tar vxf $pkg_path >/dev/null
       if [ $? = 0 ]; then 
          if [ "$admin" = "FALSE" ]; then
            ddts_lines=$(rpm -qp --obsoletes *.rpm | cut -d = -f1 | cut -d - -f3)
          else
            ddts_lines=$(rpm -qp --obsoletes *.rpm | cut -d - -f2 | cut -d = -f1)
          fi
          for line in $ddts_lines
          do
           echo ${line%.*} >>/tmp/my_sup_dir/sup_pkgs_from_rpms
          done
          if [ -e "/tmp/my_sup_dir/sup_pkgs_from_rpms" ]; then
              cat /tmp/my_sup_dir/sup_pkgs_from_rpms | sort -u >/tmp/obsoletes_smus_data_in_bundle
          fi
       else
          echo "package or path doesn't exist"
       fi
       rm -rf /tmp/my_sup_dir
   else

       pkg_pfx=$(echo ${pkg_name%-*})
       ddts="${pkg_pfx##*.}"
       echo "ddts: $ddts"
       rm -rf /tmp/sup_pkgs_from_rpms

       if [ "$admin" = "FALSE" ]; then
         pkgs=$(ls /opt/cisco/XR/packages/ | grep "$ddts")
       else
         pkgs=$(ls /opt/cisco/calvados/packages/ | grep "$ddts")
       fi

       if [ -z "$pkgs" ]; then
           echo "No packages found !!"
       else
         if [ "$admin" = "FALSE" ]; then
           ddts_lines=$(rpm -q --obsoletes $pkgs | cut -d = -f1 | cut -d - -f3)
         else
           ddts_lines=$(rpm -q --obsoletes $pkgs | cut -d - -f2 | cut -d = -f1)
         fi
         for line in $ddts_lines
         do
           echo ${line%.*} >>/tmp/sup_pkgs_from_rpms
         done
         if [ -e "/tmp/sup_pkgs_from_rpms" ]; then
             cat /tmp/sup_pkgs_from_rpms | sort -u >/tmp/obsoletes_smus_data_in_bundle
         fi
       fi
       rm -rf /tmp/sup_pkgs_from_rpms
   fi
}

# check whether the given file has any link to it
# $1 is repo path, $2 file to be searched
# This function goes through all symlinks and checks whether they 
# link to given file or not
# if link existsm then return 1, else 0 
function find_link_in_repo ()
{
    local fname=""
    local lfile=""
    local repo_path=$1
    local find_file=$2

    echo "****Function find_link_in_repo $repo_path $find_file" >> /tmp/tftp_cleanup_log
    for fname in `find $repo_path -type l `
    do
        echo "$fname is symbolic link to  " `readlink $fname ` >> /tmp/tftp_cleanup_log
        lfile=`readlink $fname | xargs basename`
        echo $lfile >> /tmp/tftp_cleanup_log
        if [ $lfile = $find_file ];
        then
            echo "Found the match " >> /tmp/tftp_cleanup_log
            return 1
        fi
    done
    return 0
}
# Delete the mini ISO in the tftpboot directory if there is no link
# to those files in repo and tftpboot
function tftp_cleanup ()
{
    local fname=""
    local DIR=/misc/disk1/tftpboot
    local r1=0
    local r2=0
    local r3=0
    local r4=0
    set +e

    echo "Function tftpcleanup" > /tmp/tftp_cleanup_log
    for fname in `ls $DIR | grep mini `
    do
       echo $fname >> /tmp/tftp_cleanup_log
       find_link_in_repo "/install_repo/gl/calvados"  $fname
       r1=$?
       find_link_in_repo "/install_repo/gl/xr" "$fname"
       r2=$?
       find_link_in_repo $DIR $fname
       r3=$?
       for dname in `ls -d /install_repo/gl/instdb/sdr/*/pkg`; do
           find_link_in_repo "$dname" "$fname"
           r4=$?
           if [ $r4 == 1 ]; then
               break
           fi
       done

       if [ $r1 == 1 ] || [ $r2 == 1 ] || [ $r3 == 1 ] || [ $r4 == 1 ]
       then
           echo "Links exists in repo for $fname " >> /tmp/tftp_cleanup_log
           echo "Links exists in repo for $fname "
       else
           echo "Deleting $DIR/$fname " >> /tmp/tftp_cleanup_log
           echo "Deleting $DIR/$fname "
           rm $DIR/$fname
       fi
    done
}

function list_dr_partition () 
{
    local dev_name=${1}
    local sysimg=./system_image.iso
    local grubefi=./EFI/Recovery/grub.efi
    local grubcfg=./EFI/Recovery/grub.cfg
    local system_img_iso_mount=`mktemp -d /tmp/mount_sysimgiso.XXXXXX`

    
    set +e

    cd ${dev_name}
    mount -o loop ${sysimg} ${system_img_iso_mount} || exitclean
    echo Disater Recovery Image Version :`cat ${system_img_iso_mount}/iso_info.txt| head -1 | awk '{ print $4 }'` >> /tmp/inst_veri_dr_log
    run_cmd "umount ${system_img_iso_mount}"
    run_cmd rm -rf ${system_img_iso_mount}

    if [ -f ${sysimg} ]; then
       echo `ls -l  ${sysimg} | awk '{ print $9,$5 }' ` >> /tmp/inst_veri_dr_log
       echo MD5: `md5sum ${sysimg}` >> /tmp/inst_veri_dr_log
    else
       echo system_image.iso not found >> /tmp/inst_veri_dr_log
    fi
    sync

    if [ -f ${grubcfg} ]; then
       echo `ls -l  ${grubcfg} | awk '{ print $9,$5 }' ` >> /tmp/inst_veri_dr_log
       echo MD5: `md5sum ${grubcfg}` >> /tmp/inst_veri_dr_log
    else
       echo grub.cfg not found >> /tmp/inst_veri_dr_log
    fi
    sync

    if [ -f ${grubefi} ]; then
       echo `ls -l  ${grubefi} | awk '{ print $9,$5 }' ` >> /tmp/inst_veri_dr_log
       echo MD5: `md5sum ${grubefi}` >> /tmp/inst_veri_dr_log
    else
       echo grub.efi not found  >> /tmp/inst_veri_dr_log
    fi

    sync
    sleep 2
    sync
    cd - > /dev/null
}

function mount_subpart() {
    local ROOTDEV=$1
    local part=$2
    local dest_dir=$3
    dev_present=$(fdisk -u -l ${ROOTDEV} 2> /dev/null | grep ${ROOTDEV}${part} | wc -l )
    if [[ "${dev_present}" != "1" ]]; then
        echo "Cannot find $ROOTDEV $1"
        return -1 
    fi
    start=$(fdisk -u -l ${ROOTDEV} 2> /dev/null | grep ${ROOTDEV}${part} | awk '{print $2}')
    endlimit=$(fdisk -u -l ${ROOTDEV} 2> /dev/null | grep ${ROOTDEV}${part} | awk '{print $3}')
    result=$((${endlimit} - ${start}))
    sizelimit=$(expr ${result} \* 512)
    offset=$(( ${start} * 512 ))
    loop=$(losetup --find)
    if (( $? != 0 )); then
        echo "Loop devices are unavailable to mount"
        return -1
    fi

    losetup ${loop} ${ROOTDEV} -o ${offset} --sizelimit ${sizelimit}
    if (( $? != 0 )); then
        echo "Failed to losetup ${loop}"
        losetup -d ${loop}
        echo ""
        return -1
    fi
    
    mount -o discard ${loop} ${dest_dir}
    if (( $? != 0 )); then
        echo "Failed to mount  ${loop}"
        losetup -d ${loop}
        return -1
    fi
    echo ${loop}
}

function get_dr_dev() {
   local retval="Failed"

   echo "Function get_dr_dev begin" > /tmp/get_dr_dev 
   set +e
   for device in `ls /dev/vd* `; do
        out=$(blkid -s LABEL -o value ${device} 2>/dev/null)
        echo $out  $device >> /tmp/get_dr_dev
        if [ "${out}" == "RecoveryOs" ]; then
            echo "found the Recovery partition " >> /tmp/get_dr_dev 
            retval=$device
            break
        fi
    done
    echo $retval
    echo "Ret val " $retval >> /tmp/get_dr_dev
}

function get_disk1_dev() {
   local retval="Failed"

   echo "Function get_disk1_dev begin" > /tmp/get_disk1_dev 
   set +e
   for device in `ls /dev/vd* `; do
        out=$(blkid -s LABEL -o value ${device} 2>/dev/null)
        echo $out  $device >> /tmp/get_disk1_dev
        if [ "${out}" == "DISK_ONE" ]; then
            echo "found the Recovery partition " >> /tmp/get_disk1_dev 
            retval=$device
            break
        fi
    done
    echo $retval
    echo "Ret val " $retval >> /tmp/get_disk1_dev
}

function get_repo_dev() {
   local retval="Failed"

   echo "Function get_repo_dev begin" > /tmp/get_repo_dev 
   set +e
   for device in `ls /dev/vd* `; do
        out=$(blkid -s LABEL -o value ${device} 2>/dev/null)
        echo $out  $device >> /tmp/get_repo_dev
        if [ "${out}" == "Repository" ]; then
            echo "found the Repository partition " >> /tmp/get_repo_dev 
            retval=$device
            break
        fi
    done
    echo $retval
    echo "Ret val " $retval >> /tmp/get_repo_dev
}

function get_calvados_lv_method_1() {
    PLATFORM=${IOS_XR_PLATFORM}
    if grep cpu /proc/1/cgroup >/dev/null 2>&1; then
        eval "$CHVRF_CMD ssh $HOSTVM_IP cat /misc/config/sysadmin-lxc-$PLATFORM-0.xml  | grep calvados_lv |  sed   's/^.*calvados_lv//'  | head -n 1 | tr \"'/>\" \" \" "
    else
        eval "$CHVRF_CMD ssh $HOSTVM_IP cat  /misc/config/sysadmin-qemu-$PLATFORM-0.xml | grep calvados_lv | sed   's/^.*calvados_lv//'  | tr \"'/>\" \" \" "
    fi
}

function get_calvados_lv_method_2() {
    PLATFORM=${IOS_XR_PLATFORM}
    if grep cpu /proc/1/cgroup >/dev/null 2>&1; then
        eval "$CHVRF_CMD ssh $HOSTVM_IP cat /misc/config/sysadmin-lxc-$PLATFORM-0.xml  | grep calvados_lv |  sed   's/^.*calvados_lv//'  | head -n 1 | tr \"'/>\" \" \" "
    else
        eval "$CHVRF_CMD ssh $HOSTVM_IP ps aux | grep \"/usr/bin/kvm-qemu-system-x86_64 -name sysadmin\" | grep calvados_lv | sed 's/^.*calvados_lv//'  | sed 's/,.*//' "
    fi
}

function copy_logs_to_debug_part() {
    local TEMP_DATE=`date +%Y-%m-%d_%H_%M_%S`
    local TEMP_FILE="debug_install_logs"
    local TAR_FILE=$TEMP_FILE"_"$TEMP_DATE".tar"
    local GZ_FILE=$TAR_FILE.gz
    local MAX_FILES=1
    set +e

    if [ -f /iso/sim.txt ]; then
        echo "Running on SIMULATION PLATFORM, exiting"
        exit 0
    fi          

    if [ -f /etc/init.d/spirit_pd.sh ]; then
        source /etc/init.d/spirit_pd.sh
    fi
  
    # Mount the debug partition
    declare -F pd_check_if_debug_part_needed >/dev/null
    if  [[ $? -ne 0 ]]; then
        echo "debug partition is not supported "
        exit 0 
    else
        echo "Mounting the debug parition"
    fi
    mnt_point=$(runon_host "source /opt/cisco/hostos/bin/part_script.sh; mount_debug_parition | grep MNT_POINT= | sed 's/MNT_POINT=//' ")
    if [ -z "$mnt_point" ]; then
        echo "Unable to mount, exiting"
        exit -1
    fi
    echo "Debug partition mounted at $mnt_point"

    # Put all logs into gz file
    cd /var/log/install >/dev/null
    rm $TEMP_FILE* > /dev/null 2>&1
    set -e
    tar -czf $GZ_FILE inst_* >/dev/null
    sync

    #In debug partition, we can keep max 2 files. So delete all except one
    while [ true ] ; do
        file_count=$(runon_host "ls -1 ${mnt_point}/${TEMP_FILE}* 2>/dev/null | wc -l")
        echo ${file_count}
        if [ $file_count -gt $MAX_FILES ]; then
            del_file=$(runon_host "ls -1 ${mnt_point}/${TEMP_FILE}* | head -n 1")
            echo "Deleting one file in host ${del_file}"
            runon_host "rm ${del_file} "
        else
            break
        fi
    done

    #Copy the new log file one
    $CHVRF_CMD scp $GZ_FILE $HOSTVM_IP:$mnt_point
    runon_host "sync"
    rm $GZ_FILE

    # Unmount the debug partition
    runon_host "source /opt/cisco/hostos/bin/part_script.sh; unmount_debug_partition "
    cd -     >/dev/null
}
function cleanup_debug_part() {
    local TEMP_FILE="debug_install_logs"

    if [ -f /iso/sim.txt ]; then
        echo "Running on SIMULATION PLATFORM, exiting"
        exit 0
    fi            
    # delete if some temporary files exists 
    cd /var/log/install
    set +e
    rm $TEMP_FILE* > /dev/null 2>&1
    set -e
    sync

    # Try Unmount debug partition
    runon_host "source /opt/cisco/hostos/bin/part_script.sh; unmount_debug_partition "
    cd -    >/dev/null
}

function get_stale_xr_pkgs() {
    retval=0
    rm -rf /tmp/get_stale_xr_pkgs
    touch /tmp/get_stale_xr_pkgs
    for f in $XR_REPO/*
    do
        link=$(find -L $INSTALL_INSTDB -samefile $f | wc -l)
        if [ "$link" == "0" ]; then
            echo $f >> /tmp/get_stale_xr_pkgs
        fi
    done
    retval=$(cat /tmp/get_stale_xr_pkgs | wc -l)
    echo $retval
}
# Delete the .tmp files from install_repo/gl and  tftpboot directory 
function dot_tmp_cleanup ()
{
    set +e
    declare -a repo=("/misc/disk1/tftpboot"
       "/install_repo/gl/calvados"
       "/install_repo/gl/host"
       "/install_repo/gl/xr"
       "/install_repo/gl/instdb/sdr"
    )

    #Loop through all repo and delete .tmp files
    for i in "${repo[@]}"
    do
       if [ -d ${i} ]; then
           #Deleting .tmp files
           echo "Deleting following files from ${i}"
           find ${i} -maxdepth 1 -name "*.tmp" -mmin +360 -print -delete
       fi
    done
}

######################################################################
# generating/cache superseded SMU info while doing show install 
# active and superseded CLIs
######################################################################

SYSTEM_RPM_DATA="/install/obsoletes/__installed_rpm_smu__.txt"
SUPERSEDED_SMUS="/install/obsoletes/__superseded_smus__.txt"
SUP_LOG_FILE="/var/log/install/superseded__check.logs"

#
# get obsoletes data from the installed SMUs
# and cache 
# IN: file containing rpm -qa data 
# OUT: list of DDTS which are fully superseded
#      will be added to SUPERSEDED_SMUS file

function get_superseded_data_from_smus()
{
 set +o pipefail 
 set +e

 #$1 is input rpm -qa data
 rpm_qa="$1"

 smu_list=$(cat $rpm_qa | grep "CSC" | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/' | sort -u)
 if [ -z "$smu_list" ]; then
     return
 fi

 for smu in ${smu_list[@]}
 do
   echo "Requires: $smu " >>$SUP_LOG_FILE
   inst_rpm=$(grep "$smu" "$rpm_qa")

   # Requires 
   grep $smu "$rpm_qa" | xargs rpm -qR | grep "CSC" | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/' | sort -u >/tmp/rlist
   if [ -e /tmp/rlist ]; then
        cat /tmp/rlist >>$SUP_LOG_FILE
   else
       touch /tmp/rlist
   fi

   #echo "Obsoletes ######"
   s_list=$(grep $smu "$rpm_qa" | xargs rpm -q --obsoletes  |\
         grep "CSC" | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/' | sort -u)

   echo "obsoletes: $s_list">>$SUP_LOG_FILE

   for ddts in ${s_list[@]}
   do
     ddts_in_req=$(grep $ddts /tmp/rlist)
     if [ -z "$ddts_in_req" ]; then
         ddts_active=$(grep $ddts "$rpm_qa")
         if [ -z "$ddts_active" ]; then
             continue
         else
            echo "$ddts">>$SUPERSEDED_SMUS
         fi
     else
         continue
     fi
     done
 done
}

function inst_get_superseded_smus()
{
 set +o pipefail
 set +e
 smu=$2

 if [ -e /tmp/superseded_data.txt ]; then
      rm /tmp/superseded_data.txt
 fi
  
 grep $smu "$1" | xargs rpm -qR | grep "CSC" | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/' | sort -u >/tmp/rlist
 if [ -e /tmp/rlist ]; then
     cat /tmp/rlist >>$SUP_LOG_FILE
 else
      touch /tmp/rlist
 fi 

 #echo "Obsoletes ######"
 s_list=$(grep $smu "$1" | xargs rpm -q --obsoletes  |\
      grep "CSC" | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/' | sort -u) 

 echo "inst_get_superseded_smus:obsoletes: $s_list">>$SUP_LOG_FILE
 
 for ddts in ${s_list[@]}
   do
     ddts_in_req=$(grep $ddts /tmp/rlist)
     if [ -z "$ddts_in_req" ]; then
         ddts_active=$(grep $ddts "$1")
         if [ -z "$ddts_active" ]; then
             echo ""
         else
            echo "$ddts">>/tmp/superseded_data.txt
         fi
     else
        echo ""
     fi
   done
 if [ -e /tmp/rlist ]; then
       rm -rf /tmp/rlist
 fi
}
 
#
# Check SMU fully supersedes any other installed SMU
# on the system 
# IN: SMU name
# OUT: creates file with SMU name and stores 
#      SMU IDs which are fully superseded by input SMU 
#

function inst_get_superseded_info()
{
 set +o pipefail
 set +e

 if [ -e /install/obsoletes/$1 ]; then
     cat /install/obsoletes/$1 
     return 
 fi

 sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/'
 in_smu=$(echo $1 | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/')
 if [ -e /install/obsoletes ]; then
     echo ""
  else
     mkdir -p /install/obsoletes
 fi

 rpm -qa >$SYSTEM_RPM_DATA
 touch /install/obsoletes/$1
 inst_get_superseded_smus "$SYSTEM_RPM_DATA" $in_smu
 if [ -e /tmp/superseded_data.txt ]; then
      cp  /tmp/superseded_data.txt /install/obsoletes/$1 
      cat /install/obsoletes/$1
 fi

}

function inst_superseded_validate_cache()
{
   rm -rf /install/obsoletes/*
}

function inst_get_sup_display()
{
  smu="$1"
  ddts="$2"
  echo $smu | sed  's/CSC\w\+/'$ddts'/'

}

####
# check given smu is present in superseded list
####
function inst_check_smu_superseded()
{
 set +o pipefail
 set +e
   in_smu=$(echo $1 | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/')
   #is it optional package ?
   if [ -z "$in_smu" ]; then
       return "FALSE"
   fi

   if [ -e "$SUPERSEDED_SMUS" ]; then
       is_in_sup=$(grep "$in_smu" $SUPERSEDED_SMUS)
       if [ -z "$is_in_sup" ]; then
           echo  "FALSE" 
       else
           echo "TRUE" 
       fi
   else
       #running show install active before running show install superseded
       mkdir -p /install/obsoletes
       rpm -qa >$SYSTEM_RPM_DATA
       get_superseded_data_from_smus $SYSTEM_RPM_DATA

       if [ -e "$SUPERSEDED_SMUS" ]; then
           is_in_sup=$(grep "$in_smu" $SUPERSEDED_SMUS)
           if [ -z "$is_in_sup" ]; then
               echo "FALSE" 
           else
               echo "TRUE" 
           fi
       else
           # no superseded package exist, so just touch it
           # to avoid sup check 
           touch $SUPERSEDED_SMUS 
           echo "FALSE" 
       fi
   fi
}

function compute_changed_and_deleted_files()
{
    local   snapshot1=$1
    local   snapshot2=$2
    local   dir_name=$3
    local   changed_list=$4
    local   deleted_list=$5
    start_time=$(head -n1 ${snapshot1})
    end_time=$(head -n1 ${snapshot2})
    diff_time=$(($end_time - $start_time))
    diff_in_minute=$(($diff_time/60 + 1))
    cd $dir_name && find . -type f  -mmin -$diff_in_minute > $changed_list
    sed -i -e '1d' $snapshot1 $snapshot2
    comm -23 $snapshot1 $snapshot2 > $deleted_list
}
function copy_cal_relnote() {
    set +e
    local caliso=$1
    local tmpdir=$2
    local relnote_fname=$3

    isomnt=`mktemp -d /tmp/.cmpchk_isomnt.XXXXXX`
    sync

    run_cmd mount -o loop $caliso $isomnt
    if [ -f $isomnt/SDK-RELEASE-NOTES ]; then
        cp $isomnt/SDK-RELEASE-NOTES $tmpdir/$relnote_fname
    fi
    #Testing purpose
    if [ -f /tmp/inst_ut_grail_1  ]; then
        cp /tmp/SDK-RELEASE-NOTES-1 $tmpdir/$relnote_fname
    fi

    umount $isomnt
    rm -rf $isomnt
    sync

    if [ -f $tmpdir/$relnote_fname ]; then
        return 0 
    else 
        return 1 
    fi
}
function compare_relnotes() {
    set +e
    local xriso=$2
    local calrelnote=$1
    local relnote_fname="SDK-REL-XR.1"
    local retval=0
    local relnote_diff=$3
    local is_calvados_upgrade=1

    isomnt=`mktemp -d /tmp/.cmpchk_isomnt.XXXXXX`
    tmpdir=`mktemp -d /tmp/.cmpchk_dir.XXXXXX`
    sync >/dev/null 2>&1
    
    run_cmd mount -o loop $xriso $isomnt >/dev/null 2>&1
    if [ -f $isomnt/SDK-RELEASE-NOTES ]; then
        cp $isomnt/SDK-RELEASE-NOTES $tmpdir/$relnote_fname >/dev/null 2>&1
    fi 
    #Testing purpose
    if [ -f /tmp/inst_ut_grail_1  ]; then
        cp /tmp/SDK-RELEASE-NOTES-1 $calrelnote
        cp /tmp/SDK-RELEASE-NOTES-2 $tmpdir/$relnote_fname
    fi
    if [ -f $tmpdir/$relnote_fname ]; then
        python /opt/cisco/calvados/bin/install_rel_notes.py $calrelnote $tmpdir/$relnote_fname $is_calvados_upgrade > $relnote_diff
        retval=$?
    fi 
    umount $isomnt >/dev/null 2>&1
    rm -rf $isomnt >/dev/null 2>&1
    rm -rf $tmpdir >/dev/null 2>&1
    sync >/dev/null 2>&1

    return $retval 
}

function compare_admin_relnotes() {
    set +e
    local xriso=$2
    local caliso=$1
    local relnote_fname="SDK-REL-XR.1"
    local relnote_fname1="SDK-REL-CAL.1"
    local retval=0
    local relnote_diff=$3
    local is_calvados_upgrade=0

    isomnt=`mktemp -d /tmp/.cmpchk_isomnt.XXXXXX`
    tmpdir=`mktemp -d /tmp/.cmpchk_dir.XXXXXX`
    sync >/dev/null 2>&1
    
    run_cmd mount -o loop $xriso $isomnt >/dev/null 2>&1
    if [ -f $isomnt/SDK-RELEASE-NOTES ]; then
        cp $isomnt/SDK-RELEASE-NOTES $tmpdir/$relnote_fname >/dev/null 2>&1
    fi 
    #Testing purpose
    if [ -f /tmp/inst_ut_grail_1  ]; then
        cp /tmp/SDK-RELEASE-NOTES-1 $tmpdir/$relnote_fname
    fi

    isomnt1=`mktemp -d /tmp/.cmpchk_isomnt.XXXXXX`
    tmpdir1=`mktemp -d /tmp/.cmpchk_dir.XXXXXX`
    sync >/dev/null 2>&1
    
    run_cmd mount -o loop $caliso $isomnt1 >/dev/null 2>&1
    if [ -f $isomnt1/SDK-RELEASE-NOTES ]; then
        cp $isomnt1/SDK-RELEASE-NOTES $tmpdir1/$relnote_fname1 >/dev/null 2>&1
    fi 
    #Testing purpose
    if [ -f /tmp/inst_ut_grail_1  ]; then
        cp /tmp/SDK-RELEASE-NOTES-2 $tmpdir1/$relnote_fname1
    fi

    if [ -f $tmpdir1/$relnote_fname1 ] && [ -f $tmpdir/$relnote_fname ]; then
        python /opt/cisco/calvados/bin/install_rel_notes.py $tmpdir1/$relnote_fname1 $tmpdir/$relnote_fname $is_calvados_upgrade > $relnote_diff
        retval=$?
    fi 

    umount $isomnt >/dev/null 2>&1
    umount $isomnt1 >/dev/null 2>&1
    rm -rf $isomnt >/dev/null 2>&1
    rm -rf $isomnt1 >/dev/null 2>&1
    rm -rf $tmpdir >/dev/null 2>&1
    rm -rf $tmpdir1 >/dev/null 2>&1
    sync >/dev/null 2>&1

    return $retval 
}

#For rpm signature and verifcation, check the list of hardcoded releases for the rpm
#If rpm belongs to any of these hardcoded releases do not verify signature for the rpm
#as it is not signed
 
function check_release()
{
    set +o pipefail
    set +e
    local IS_NEW_PKG_FMT=$2
    releases_new_pkg_fmt=("r601" "r602" "r611" "r612" "r6131" "r6121" "r613" "r614" "r622" "r623" "r631" "r60" "r61" "r62")
    arraylth_new=${#releases_new_pkg_fmt[@]}

    releases_old_fmt=("5.2.5." "5.2.6." "6.0.1." "6.0.2." "6.1.1." "6.1.2." "6.1.31." "6.1.21." "6.1.3." "6.1.4." "6.2.2." "6.3.1." "6.0." "6.1." "6.2.")
    arraylth_old=${#releases_old_fmt[@]}

    i=0
    if [ $IS_NEW_PKG_FMT -eq 1 ]
    then
        for (( i=1; i<${arraylth_new}+1; i++ ));
        do
            echo "$1" | grep ${releases_new_pkg_fmt[i-1]}
            if (( $? == 0 )); then
                break;
            fi
        done
        if [ $i -gt ${arraylth_new} ]
        then
            return 0
        fi
    else
        for (( i=1; i<${arraylth_old}+1; i++ ));
        do
            echo "$1" | grep ${releases_old_fmt[i-1]}
            if (( $? == 0 )); then
                break;
            fi
        done
        if [ $i -gt ${arraylth_old} ]
        then
            return 0
        fi
    fi

    return 1

}

#Check if the rpm is a tp smu. If yes, gets its release from xrrelease rpm query
#and verify accordingly
#Return val: check_release returns either 0 or 1 depending on whether release 
#            needs to be verified or not.
#            A return value of 2 is returned to the callee if the rpm is not a tp smu

function check_tp_smu_and_rel()
{
    set +o pipefail
    set +e
    local RPM=$1
    local IS_NEW_PKG_FMT=$2
    tp_smu_rel=`rpm -qp --queryformat "[%{xrrelease}\n]" $RPM`
    tp_smu_rel="$(echo -e "${tp_smu_rel}" | tr -d ' ')"
 
    if [ ! -z "$tp_smu_rel" ]
    then
        if [ `echo $tp_smu_rel | grep ^[0-9]` ]
        then
            tp_smu_rel="r$tp_smu_rel"
            check_release tp_smu_rel $IS_NEW_PKG_FMT
            return $?
        fi
    fi
    return 2
}

#validates the signature of rpm based on the release

function verify_signature()
{
  set +o pipefail
  set +e

  local RPM_PATH=$1
  local retval="Success"
  local IS_NEW_PKG_FMT=0
  local is_tp_smu_and_verify=0
  local check_rel_ret_val=-1

  local RELEASE=`rpm -qp --queryformat "[%{RELEASE}\n]" $RPM_PATH`
  RELEASE="$(echo -e "${RELEASE}" | tr -d ' ')"
  if [ ! -z "$RELEASE" ]
  then
      #if release name is prefixed with 'r', rpm belongs to  new package format
      if [ `echo "$RELEASE" | grep ^r` ]
      then
          IS_NEW_PKG_FMT=1
          check_tp_smu_and_rel $RPM_PATH $IS_NEW_PKG_FMT

          check_rel_ret_val=$?

          if [ $check_rel_ret_val -eq 2 ]
          then
              check_release $RELEASE $IS_NEW_PKG_FMT
              check_rel_ret_val=$?
          fi

      elif [ `echo "$RELEASE" | grep "Default"` ]
      then
          check_release $RPM_PATH $IS_NEW_PKG_FMT
          check_rel_ret_val=$?

      elif [ `echo "$RELEASE" | grep "SMU"` ]
      then
          check_release $RPM_PATH $IS_NEW_PKG_FMT
          check_rel_ret_val=$?
      fi
  fi
  if [ $check_rel_ret_val -eq 0 ]
  then
      rpm -K $RPM_PATH >> /dev/null
      if (( $? != 0 )); then
          retval="Failed"
      else
          rpm -Kv $RPM_PATH | grep -i "signature:" | grep ": OK"
          if (( $? != 0 )); then
              retval="Failed"
          fi
      fi
  fi
  
   echo $retval
}
function sign_error() {
    set +e
    set +o pipefail
    rpm_name="$1"
    base_name="$(basename $rpm_name)"
    err_msg=""

    has_signature="$(rpm -qpi $rpm_name 2>/dev/null | grep "Signature" | grep "Key ID" | cut -d"," -f3)"

    if [[ -z "$has_signature" ]]
    then
        err_msg="$(echo -e "RPM: $base_name is not signed")"
    else
        signature2="$(rpm -qa | head -1 | xargs -I '{}' rpm -qi "{}" | grep Signature | cut -d "," -f3)"

        if [[ $has_signature != $signature2 ]]
        then
            err_msg="$(echo "Key Mismatch Error for $base_name. Please check if both image and RPMS are signed with same keys")"
        fi
    fi
    echo "$err_msg"
    set -e
}

# Delete the corrupted files from install repo
function clean_corrupted_repo() 
{
    local loc=$1
    if [ -z "$loc" ]; then
        echo "Repo location is NULL"
        return 0
    fi
    for fname in `ls $loc -I "*.tmp" `; do
        ftype=`file -b $loc/$fname` 
        # RP Sync starts before BIV begins, so there will be .tmp files
        # which should not be deleted because if those pkgs are required for
        # BIV partition preparation then they are not downloaded again
        if [ "${ftype}" == "data" ]; then 
            echo "Corrupted file found $loc/$fname"
            echo "File Type: ${ftype}"
            echo "MD5: `md5sum $loc/$fname` "
            echo "File Details: `ls -l $loc/$fname` "
            echo "File stat: `stat $loc/$fname` "
            echo "Deleting file $loc/$fname"
            rm -f $loc/$fname
        fi 
    done
}

function show_running_config_sdr_sdrname() {                                                    
 
    local LOCAL_SDR_NAME=$1                                                            
    local LOCAL_OUTPUT_FILENAME=$2                                                     
    local LOCAL_ERROR_FILENAME=$3                                                      

    if [[ -f "$LOCAL_OUTPUT_FILENAME" ]]; then                                   
        echo "Removing old output file: $LOCAL_OUTPUT_FILENAME"
        rm -f $LOCAL_OUTPUT_FILENAME                                             
    fi                                                                           
 
    if [[ -s "$LOCAL_ERROR_FILENAME" ]]; then                                    
        echo "Removing old error file: $LOCAL_ERROR_FILENAME"
        rm -f $LOCAL_ERROR_FILENAME   
    fi                                                                           
 
    $CHVRF_CMD /opt/cisco/calvados/bin/show_cmd "show running-config sdr $LOCAL_SDR_NAME | nomore" > $LOCAL_OUTPUT_FILENAME 2> $LOCAL_ERROR_FILENAME
}

function revert_mswp()
{
    set +e
    ctype=${IOS_XR_BOARD_TYPE}
    if [ "$ctype" == "RP" ] || [ "$ctype" == "CC" ]; then
        run_cmd cp /install_repo/gl/instdb/clos-master-swprofile-committed.bin /install_repo/gl/instdb/clos-master-swprofile-active.bin
        run_cmd cp /install_repo/gl/instdb/clos-master-swprofile-committed.txt /install_repo/gl/instdb/clos-master-swprofile-active.txt
        echo "Copied master committed swp to master active swp"
    fi
}  

function rem_calvados_launch_path_active()
{
    runon_host "ls -l /misc/config/calvados_launch_path_active.txt > /dev/null 2>&1 && rm -f /misc/config/calvados_launch_path_active.txt || echo "/misc/config/calvados_launch_path_active.txt not found""
    echo "Clean up on host completed"
}

function rm_ocu_marker_file()
{
    set +e
    ctype=${IOS_XR_BOARD_TYPE}
    if [ "$ctype" == "RP" ] || [ "$ctype" == "CC" ]; then
        ocu_marker_file="/install_repo/gl/instdb/ocu_marker.txt"
        if [ -f $ocu_marker_file ]; then
            run_cmd rm -f $ocu_marker_file
        fi
        su_marker_file="/install_repo/gl/instdb/su_active_marker.txt"
        if [ -f $su_mark_file ]; then
            run_cmd rm -f $su_marker_file
        fi
    fi
}

function ocu_cleanup()
{   
    runon_all_cal "source /opt/cisco/calvados/bin/install-functions.sh ; revert_mswp"
    
    runon_all_cal "source /opt/cisco/calvados/bin/install-functions.sh ; rem_calvados_launch_path_active"
}

function dump_xrnginstall_funcs_new() {
    funcs="$@"; fg=0; OLDIFS="$IFS"

    SCRFILE=/etc/init.d/xrnginstall
    if [ -f /etc/rc.d/init.d/pxe_install.sh ]; then
        SCRFILE=/etc/rc.d/init.d/pxe_install.sh
    fi

    >/tmp/fdump
    while IFS= read -r line ; do
        if [[ $line =~ ^function ]]; then
            set -- $line
            for i in $funcs; do
                if [[ "$i" == "$2" ]]; then
                    fg=1; break
                fi
            done
        fi
        if [ $fg -eq 1 ]; then
            echo "$line"
        fi
        if [[ $line =~ ^} ]]; then
            fg=0
        fi
    done < $SCRFILE >/tmp/fdump
    IFS="$OLDIFS"
    sync
}

# This functions copy the new repo contents to old repo location
# $1 is the destination dir where V2 partition in mounted
function copy_install_repo_new2old()
{
    set +e
    #Delete old marker file if any 
    rm -f ${repo_old_marker_file} >/var/log/install/cpy_repo_log1 2>&1
    source $1/etc/init.d/calvados_bootstrap.cfg >>/var/log/install/cpy_repo_log1 2>&1
    if [ -z "$DISK_SSD_INSTALL_REPO_SIZE" ]; then

        new_repo_size=`du -ks /install_repo/gl | awk '{ print $1 }'`
        if [[ ${new_repo_size} > 3900000 ]]; then 
            echo "Repo is greater than 4GB, cannot copy"  >>/var/log/install/cpy_repo_log1 2>&1
            echo $srch_str
            return 0;
        else 
            echo "Repo is less than 4GB"  >>/var/log/install/cpy_repo_log1 2>&1
        fi

        echo "V2 has Old repo, copy the content" >>/var/log/install/cpy_repo_log1 2>&1
        run_cmd rm -rf /old_install_repo/* >>/var/log/install/cpy_repo_log1 2>&1
        run_cmd cp -rf /install_repo/gl /old_install_repo/ >>/var/log/install/cpy_repo_log1 2>&1 ; sleep 2 
        run_cmd cp -rf /install_repo/local /old_install_repo/ >>/tmp/cpy_repo_log1 2>&1 ; sleep 2
        run_cmd sync 
        # Create a marker file in the new install repo local, will be used and    removed during activate.
        touch ${repo_old_marker_file}
    else
        echo "nothing has to be copied, V2 has new repo" >>/var/log/install/cpy_repo_log1 2>&1
    fi
    return 0
}
# This functions copy the new repo instdb contents to old repo location
function copy_install_repo_instdb_new2old()
{

    set +e
    if [ ! -f ${repo_old_marker_file} ]; then
        echo "File ${repo_old_marker_file} not found. Skipping repo copy.. "
        return 0
    fi

    run_cmd rm -rf /old_install_repo/local >/var/log/install/cpy_repo_log2 2>&1 
    run_cmd rm -rf /old_install_repo/gl/instdb >>/var/log/install/cpy_repo_log2 2>&1
    run_cmd cp -rf /install_repo/local /old_install_repo/ >>/var/log/install/cpy_repo_log2 2>&1; sleep 2
    run_cmd cp -rf /install_repo/gl /old_install_repo/ >>/tmp/cpy_repo_log2 2>&1; sleep 2
    run_cmd cp -rf /install_repo/gl/instdb /old_install_repo/gl/ >>/var/log/install/cpy_repo_log2 2>&1; sleep 2
    run_cmd sync >>/var/log/install/cpy_repo_log2 2>&1 
    return 0
}

#Copy master SWP as well at the end of SU/OCU
function copy_repo_on_all_rps()
{
    set +e
    if [ ! -f ${repo_old_marker_file} ]; then
        echo "File ${repo_old_marker_file} not found. Skipping repo copy"
        return 0
    fi
    runon_all_cal "source /opt/cisco/calvados/bin/install-functions.sh ; copy_repo_mswp"
}

#Copy install repo on RP/CC
function copy_repo_mswp()
{
    set +e
    ctype=${IOS_XR_BOARD_TYPE}
    if [ "$ctype" == "RP" ] || [ "$ctype" == "CC" ]; then
        rm -f ${repo_old_marker_file} >/var/log/install/cpy_repo_log3
        run_cmd cp -rf /install_repo/local /old_install_repo/ >>/var/log/install/cpy_repo_log3
        run_cmd cp -rf /install_repo/gl/instdb /old_install_repo/gl >>/var/log/install/cpy_repo_log3
        run_cmd rm /install_repo/gl/instdb/sdr/default-sdr/xr_boot_active
        run_cmd sync  >>/var/log/install/cpy_repo_log3
        run_cmd rm -f /old_install_repo/gl/host/newroot.tar.gz;          
    fi
    
}

#Function to check whether local platform supports large install repo or not
function inst_is_large_repo_supported() 
{
    set +e
    if [ -f /iso/sim.txt ]; then
        return 2
    fi 
    if [ "${IOS_XR_PLATFORM}" == "panini" ]; then
        ctype=${IOS_XR_BOARD_TYPE}
        if [ "$ctype" == "RP" ] || [ "$ctype" == "CC" ]; then
            return 1
        else 
            return 2
        fi
    else
        return 2
    fi
}
# Prepare clean handling 
function prep_repo_clean()
{
    set +e
    inst_is_large_repo_supported 
    if [ $? -eq 1 ]; then
        ctype=${IOS_XR_BOARD_TYPE}
        if [ "$ctype" == "RP" ] || [ "$ctype" == "CC" ]; then
            rm -f ${repo_old_marker_file} >/var/log/install/cpy_repo_log4
        fi
    fi
}


# Copy the instdb content of install repo into new install repo on boot
function copy_repo_on_boot() {
    if [ ! -f ${old_repo_marker_file} ]; then
        echo "File ${old_repo_marker_file} not found."
        return 0
    fi
    run_cmd rm -rf /install_repo/local/*
    run_cmd rm -rf /install_repo/gl/instdb/*
    run_cmd cp -rf /old_install_repo/local/* /install_repo/local/ ; sleep 2
    run_cmd cp -rf /old_install_repo/gl/* /install_repo/gl/ ; sleep 2
    rm -f /old_install_repo/gl/instdb/sdr/default-sdr/xr_boot_active
    rm -f ${old_repo_marker_file}
    sync

}
function attach_volume_ssd() {
    volname="/dev/pci_disk1/$1"

    (ls -1 /dev/vd* | sort)>/tmp/l1.txt; sync

    for i in {e..z}; do
        if runon_host /usr/bin/virsh attach-disk sysadmin ${volname} vd$i --cache=none >/dev/null 2>/dev/null; then
            break
        else
            true
        fi
    done
    if [ "$i" == "z" ]; then
        echoerr "Attach volume failed for $1."
        return 1
    fi

    ## Look out for new device node that appears in /dev
    local ctr=0
    while [ $ctr -ne 5 ]; do
        (ls -1 /dev/vd* | sort)>/tmp/l2.txt
        if vdev=`diff /tmp/l1.txt /tmp/l2.txt | awk '/^>/ {print $2}' | tail -n 1`; then
            sleep 1; ctr=$(($ctr + 1))
        else
            break
        fi
    done

    if [ $ctr -eq 5 ] ; then
        echoerr "Virtual device not available."
        return 1
    else
        echo "$vdev" | sed -e "s/[0-9]*$//"
        return 0
    fi
    sync
}

function check_repo_mount() {
    run_cmd dump_xrnginstall_funcs_new "\"create_sub_part_file_disk_boot check_fix_disk_part \
            create_fdisk_partition find_fdisk_total_partition mark_part_bootable \
            setup_loop_offset find_fs_offset find_fs_sizelimit\""
    source /tmp/fdump
    source /etc/init.d/disk-functions
    set +e
    inst_is_large_repo_supported 
    if [ $? -eq 2 ]; then
        echo "Large repo not supported platform/card"
        return 0
    fi
    echo "Check whether install_repo is mounted or not"
    df |  grep "/install_repo" 
    if [ $? -eq 0 ]; then
        echo "repo is mounted"
        return 0
    fi 
    atdev=`attach_volume_ssd install_repo_lv`
    dev=`basename ${atdev}`
    echo "Attached volume is $atdev: $dev"
    loop_dev=$(setup_loop_offset "/dev/${dev}" "1")
    if [ -z "${loop_dev}" ]; then
        echo "Cannot setup loop device for /dev/${dev}"
    fi
    # Mount the repo 
    run_cmd mount -o discard ${loop_dev} /install_repo 
}

#Used during add operation of a tar, the function 
#extracts the tar and in case absolute/relative path
#was specified in tar, the function copies all the files
#from all the extracted directories under TARGET_DIR and
#copies it under TARGET_DIR. It is done this way inorder to
#not disturb the existing logic of add operation, which at
#present, reads files directly under TARGET_DIR

function extract_tar()
{
    set +e

    local TAR_FILE=$1
    local TARGET_DIR=$2
    local retval="Success"

    tar -C $TARGET_DIR -xf $TAR_FILE --transform='s/.*\///'
    if (( $? != 0 )) ; then
        retval="Failed"
    fi  
    echo $retval
}
# This function will check all mount and links 
# related to scapa ECU disk
function install_verify_disk ()
{
    if [ -f /iso/sim.txt ]; then
        return 0
    fi 
    
    #Only for scapa 
    if [ "${IOS_XR_PLATFORM}" == "scapa" ]; then
        local ctype=${IOS_XR_BOARD_TYPE}
        if [ "$ctype" == "RP" ]; then

            #check ECU is mounted or not
            if  grep -qs '/mnt/ecu/' /proc/mounts; then
                echo "ECU disk is mounted."
            else
                echo "It's not mounted."
                return 1
            fi

            #check whether /disk1: link to /mnt/ecu/ 
            if echo "$(readlink /disk1:)" | grep -qs '/mnt/ecu/'; then 
                echo "/disk1: is linked"
            else 
                echo "/disk1: is not linked  $(readlink /disk1:)"
                return 1
            fi 

            #check whether /misc/disk1 link to /disk1: 
            if echo "$(readlink /misc/disk1)" | grep -qs '/disk1:'; then 
                echo "/misc/disk1: is linked"
            else 
                echo "/misc/disk1: is not linked  $(readlink /misc/disk1)"
                return 1
            fi 
            return 0
        fi
    fi
    return 0
}
# This function get the obsolete rpm meta data for first arg
# and search for arg2's version in that. If found means,
# arg1 obsolets args2
function find_obsolete () 
{
   local arg1=$1
   local arg2=$2
   local repo_path="/install_repo/gl/"

   local path_arg1=` find $repo_path -name $arg1 | head -n 1 ` ;
   if [ ! -z $path_arg1 ]
   then
       local res=`rpm -qp --obsoletes $path_arg1`;     
       #local res=`rpm -qp --provides $path_arg1`;     
       #echo $res;
   fi 

   local arg2_trim=${arg2%-*}

   local arg2_version=${arg2_trim##*-}
   #echo $arg2_version 

   if echo "$res" | grep -q "$arg2_version"; then
      echo "Obsoletes the other pkg"
   else
      echo "Donot obsoletes the other pkg"
   fi
 
}

#Arg1: Directory path
function cal_dir_usage_in_pcent() {

    local node_name=""
    local node_ip=""
    local usage_avail=""
    local usage_pcent=""
    local avail_kb=""

    name_ip_arr=( `$CHVRF_CMD /opt/cisco/calvados/bin/show_cmd "show vm | \
                   nomore" | grep -e Location -e sysadmin | \
                   awk '{printf "%s%s", $0, NR%2?" ":ORS}' | \
                   awk '{printf "%s:%s\n", $2, $5}'` )

    for element in "${name_ip_arr[@]}"
    do
        node_name=`echo ${element} | cut -d ':' -f1`
        node_ip=` echo ${element} | cut -d ':' -f2`
        # if directory exist get value else -1
        if $CHVRF_CMD ssh  -q $node_ip  [[ -d "$1" ]]; then
            usage_avail=`$CHVRF_CMD ssh $node_ip "df --output=pcent,avail $1 | \
                          tail -n1 "`
            usage_pcent=`echo $usage_avail | cut -d " " -f1 | sed "s/[^0-9]//g"`
            avail_kb=`echo $usage_avail | cut -d " " -f2`
        else
            usage_pcent=-1
            avail_kb=-1
        fi
        echo -n "$node_name:$node_ip:$usage_pcent:$avail_kb#"
    done
}
