#! /usr/bin/python

import yaml
import os
import sys
import optparse
import logging
import logging.handlers

NAME = 'name'
PROVIDES = 'provides'
REQUIRES = 'requires'
PKGMDATA = 'package_mdata'
ISOMDATA = 'iso_mdata'
ISOTYPE = 'iso_type'
BUNDNAME = 'bundle_name'
LABEL = 'label'
ISORPMS = 'iso_rpms'
GISORPMS = 'golden ISO rpms'
MDATA_YAML = '/etc/init.d/iosxr_image_mdata.yml'

vm_dict = {
        "xr" : "rpms in xr ISO",
        "sysadmin" : "rpms in sysadmin ISO",
        "host" : "rpms in host ISO"
        }

vm_dict_giso = {
        "xr" : "xr rpms in golden ISO",
        "sysadmin" : "sysadmin rpms in golden ISO",
        "host" : "host rpms in golden ISO"
        }

class PkgHintsMdata:
    def __init__(self,mdata_yaml=MDATA_YAML):
        self.mdata_dict = {}
        if os.path.exists(mdata_yaml):
            yaml_fd = open(mdata_yaml,'r')
            self.mdata_dict = yaml.load(yaml_fd)

    def is_pkg_bundle(self, p_name):
        if self.mdata_dict.has_key(PKGMDATA) :
            pkgs = self.mdata_dict[PKGMDATA]
            for pkg in pkgs.keys() :
                if pkgs[pkg][NAME] == p_name:
                    return 1
        return 0

    def whatprovides(self,p_name,platform):
        p_name = p_name.strip()
        platform = platform.strip()
        if self.mdata_dict.has_key(PKGMDATA) :
            pkgs = self.mdata_dict[PKGMDATA]
            for pkg in pkgs.keys() :
                if (pkgs[pkg].has_key(PROVIDES)) and pkgs[pkg][PROVIDES]:
                    provides = (pkgs[pkg][PROVIDES]).split(",")
                    for p in provides:
                        p = p.split('=')[0].strip()
                        if p and p == p_name and platform in pkgs[pkg][NAME]:
                            return pkgs[pkg][NAME]
        return None

    def get_mdata_for_key(self, mdata):
        if self.mdata_dict.has_key(mdata):
            return self.mdata_dict[mdata]
        return ''

def get_giso_mdata_for_vm(hints, vmtype):
    ret_dict = {}
    rpm_dict_list = hints.get_mdata_for_key(GISORPMS)
    for rpm_data in rpm_dict_list:
        for k in rpm_data.keys():
             if k == vm_dict_giso[vmtype]:
                return rpm_data[k]
    return None

def get_mdata_for_vm(hints, vmtype):

    ret_dict = {}
    iso_mdata = hints.get_mdata_for_key(ISOMDATA)
    
    if not iso_mdata:
        logger.error("Necessary metadata not available")
        sys.exit(-1)

    if vmtype and vmtype in vm_dict.keys():
        if iso_mdata[ISOTYPE] == vmtype:
            ret_dict = iso_mdata
        elif iso_mdata[ISOTYPE] == 'bundle':
            rpm_dict_list = hints.get_mdata_for_key(ISORPMS)
            for rpm_data in rpm_dict_list:
                if rpm_data[ISOTYPE] == vmtype:
                    ret_dict = rpm_data
                    break
        else:
            logger.error("Invalid query for this iso")
            sys.exit(-1)
    else:
        logger.error("Invalid vmtype as input")
        sys.exit(-1)

    return ret_dict

def raw_output(data):
    ret_list = ''
    for x in data.keys():
        logger.debug("rpm_data for %s in bundle iso is %s"%(x,data[x]))
        rpm_data = data[x]
        ret_list = "%s"%(rpm_data[vm_dict[rpm_data[ISOTYPE]]])
    return ret_list 

def format_output(data, is_bundle):
    ret_list = ''
    extra_chr = ' ' * 8
    for x in data.keys():
        logger.debug("rpm_data for %s in bundle iso is %s"%(x,data[x]))
        rpm_data = data[x]
        if not is_bundle:
            tmp_list = \
''' 
    ISO Name: %s
    ISO Type: %s
    %s:
'''%(rpm_data[NAME], rpm_data[ISOTYPE], vm_dict[rpm_data[ISOTYPE]])
        else:
            extra_chr = ' ' * 12
            tmp_list = \
''' 
        ISO Name: %s
        ISO Type: %s
        %s:
'''%(rpm_data[NAME], rpm_data[ISOTYPE], vm_dict[rpm_data[ISOTYPE]])

        for pkg in rpm_data[vm_dict[rpm_data[ISOTYPE]]].split(' '):
            if pkg:
                tmp_list += "%s%s\n"%(extra_chr,pkg)
        ret_list += tmp_list
    
    return ret_list    

def main(options, args):
    ret_list = ''
    giso_list = ''
    bundle_data = ''
    is_bundle = False
    data = {}
    giso_data = {}
    hints = PkgHintsMdata(options.yml_f)
    vm_query_list = []

    if options.vmtype and options.all_vms:
        logger.error("Please check the input parameters passed to this tool")
        sys.exit(-1)
    
    if options.all_vms:
        if options.raw_op:
            logger.error("Please provide vm type when collecting raw output")
            sys.exit(-1)

    if options.vmtype:
        vm_query_list.append(options.vmtype)

    if options.all_vms:
        ''' Populate the bundle meta data info into reply'''
        iso_mdata = hints.get_mdata_for_key(ISOMDATA)
        if options.is_giso and iso_mdata[ISOTYPE] == 'bundle':
            logger.debug("Querying golden iso %s"%(iso_mdata[NAME]))
            is_bundle = True
            bundle_data = \
'''
    ISO Name: %s
    ISO Type: %s
    ISO Bundled: %s
    GISO Label: %s
    ISO Contents:
'''%(iso_mdata[NAME], iso_mdata[ISOTYPE], iso_mdata[BUNDNAME], iso_mdata[LABEL])
            giso_list = \
'''
        Golden ISO Rpms:
'''
        elif iso_mdata[ISOTYPE] == 'bundle':
            logger.debug("Querying bundle iso %s"%(iso_mdata[NAME]))
            is_bundle = True
            bundle_data = \
'''
    ISO Name: %s
    ISO Type: %s
    ISO Contents:
'''%(iso_mdata[NAME], iso_mdata[ISOTYPE])
        else:
            logger.error("All vmtypes can only be queried for bundle isos.")
            sys.exit(-1)
        
        for vm in vm_dict.keys():
            vm_query_list.append(vm)

    for vm in vm_query_list:
        logger.debug("Getting meta data for vm %s"%(vm))
        data[vm] = get_mdata_for_vm(hints, vm)
        if options.is_giso:
            logger.debug("Getting meta data for rpms for %s in giso"%(vm))
            giso_data[vm] = get_giso_mdata_for_vm(hints, vm)

    if options.raw_op:
        logger.debug("Output in raw mode")
        ret_list = raw_output(data)
        for key in giso_data.keys():
            if giso_data[key]:
                logger.debug("rpms in golden iso for %s is %s"%(key, giso_data[key]))
                ret_list += giso_data[key]
    else:
        logger.debug("Output in formatted mode")
        ret_list = format_output(data, is_bundle)
        extra_chr = ' ' * 16
        for key in giso_data.keys():
            if giso_data[key]:
                logger.debug("rpms in golden iso for %s is %s"%(key, giso_data[key]))
                giso_list += "%s%s:\n"%(' ' * 12, vm_dict_giso[key])
                if giso_data[key]:
                    for pkg in giso_data[key].split(' '):
                        if pkg:
                            giso_list += "%s%s\n"%(extra_chr, pkg)
        if bundle_data:
            ret_list = "%s%s"%(bundle_data, ret_list)
        if giso_list:
            ret_list = "%s%s"%(ret_list, giso_list)
    
    return ret_list

def parseCli():
    oparser = optparse.OptionParser()
  
    oparser.add_option(
        "-v",
        "--vmtype",
        dest="vmtype",
        default=None,
        help='VM tye (e.g. host | sysadmin | xr )')

    oparser.add_option(
        "-f",
        "--file",
        dest="yml_f",
        default=None,
        help='Yaml file to be considered')

    oparser.add_option(
        "-r",
        "--raw",
        dest="raw_op",
        action='store_true',
        default=False,
        help='True if raw data is being requested for.')

    oparser.add_option(
        "-g",
        "--giso",
        dest="is_giso",
        action='store_true',
        default=False,
        help='True if data is being retrieved for GISO bundle')

    oparser.add_option(
        "-a",
        "--all",
        dest="all_vms",
        action='store_true',
        default=False,
        help='True if data is being retrieved for all vmtypes')

    options, args = oparser.parse_args()
    return options, args

if __name__ == '__main__':
    LOGFILE = "/var/log/install/inst_hooks.log"
    # create logger
    logger = logging.getLogger('install_hook_logger')
    logger.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s::  %(message)s',"%Y-%m-%d %H:%M:%S")

    # Logs to logfile
    fh = logging.handlers.RotatingFileHandler(LOGFILE, maxBytes=(1024*100))
    fh.setLevel(logging.DEBUG)
    fh.setFormatter(formatter)
    logger.addHandler(fh)

    cli = sys.argv[:]
    logger.debug("+"*80)
    logger.debug(' '.join(cli))

    options, args = parseCli()
    if not options.yml_f:
        logger.debug("No Yaml file in input. Initializing to %s"%(MDATA_YAML))
        options.yml_f = MDATA_YAML
    print main(options, args)
    logger.debug("+"*80)
