#!/bin/bash
# Convert a live iso image so that it's bootable off of a USB stick
# Copyright 2007  Red Hat, Inc.
# Jeremy Katz <katzj@redhat.com>
#
# overlay/persistence enhancements by Douglas McClendon <dmc@viros.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

export PATH=/sbin:/usr/sbin:$PATH

usage() {
##
# This script is used to make USB/Pen Drive sticked to RP 
#  on Host USB port. This script supports making EFI type 
#  and Legacy type of linux images. In case if user does 
#  not need any of EFI features can use legacy boot type 
##
    echo "$0 <isopath> <usbstick device> <legacy|EFI> "
    echo "HELP: This script is used to make USB/Pen Drive sticked to "
    echo "      Host USB port. This script supports making EFI type "
    echo "      and Legacy type of linux images. " 
    exit 1
}

cleanup() {
    [ -d "$CDMNT" ] && umount $CDMNT && rmdir $CDMNT
    [ -d "$USBMNT" ] && umount $USBMNT && rmdir $USBMNT
    [ -d "$ROOTFSMNT" ] && umount $ROOTFSMNT && rmdir $ROOTFSMNT
}

efiexitclean() {
    echo "USB EFI setup unsuccessful. Cleaning up to exit..."
    rm -rf ${USBMNT}/EFI/boot ${USBMNT}/boot
    cleanup
    exit 1
}

exitclean() {
    echo "Cleaning up to exit..."
    cleanup
    exit 1
}

function create_part_file_disk {
echo "

n
p
1

+1900M
w" > /tmp/fdisk.input
}
readonly -f create_part_file_disk

function get_partnum {
  local fullname=$1
  echo ${fullname} | tr -d a-z
  return
}

function remove_partition {
    echo "No need for remove_partition for disk"
}

host_device_in_part() {
    devname=$1
    if [ $(grep -c ${devname%%[0-9]*} /proc/partitions) -gt 1 ]; then
        if [ ${devname} = ${devname%%[0-9]*} ]; then
            device_in_part=1
        else
            device_in_part=$(get_partnum $devname)
        fi
    fi
}

getdisk() {
    DEV=$1

    if [[ $(lvscan 2> /dev/null | grep -c panini_vol_grp) -gt 1 ]]; then
        echo "Run usb install from host"
    else
        echo "Running usb install from XR"
        device_in_vm=1
    fi

    udevadmpath=$(which udevadm)
    if [ -x $udevadmpath ]; then
        p=$(udevadm info -q path -n $DEV)
        if [ -n "$device_in_vm" ]; then
            device_serial=$(udevadm info -n $DEV --attribute-walk | grep serial | head -1 | cut -f2 -d'-')
        fi
    else
        p=$(udevinfo -q path -n $DEV)
        if [ -n "$device_in_vm" ]; then
            device_serial=$(udevinfo -n $DEV --attribute-walk | grep serial | head -1 | cut -f2 -d'-' | tr -d a-z)
        fi
    fi
    if [ -e /sys/$p/device ]; then
        device=$(basename /sys/$p)
    else
        device=$(basename $(readlink -f /sys/$p/../))
    fi

    # For ncs5500/fretta, usb is passed thru to XR LXC
    #  as /dev/vdusb, this script runs from XR
    #  so, just have device to /dev/vdusb
    #  no partitions are supported, entire usb
    #  device needs to be passed
    if [ "$platform_type" == "fretta" ]; then
        device=$DEV
    else  

        if [ ! -e /sys/block/$device -o ! -e /dev/$device ]; then
            echo "Error finding block device of $DEV.  Aborting!"
            echo $device
            exitclean
        fi

        device_sd=$device
        device="/dev/$device"
    fi

    len=${#device}
    device_part=1
    if [ -n "$device_serial" ]; then
       device_in_part_n=${device_serial%[^0-9]*}
       device_in_part=${device_in_part_n##*[a-z]}
    else
       host_device_in_part ${device_sd}
    fi
    if [ -n  ${device_in_part} ]; then
      device_part=$device_in_part
    fi
}

function preparedisk {
    local dev_name=${1}
    echo "d
    w" | fdisk ${dev_name}
    sync
    sleep 2
    echo "prepare disk success"
}


function format_ext2_partition {
    local dev_name=${1} label=${2}

    # Reformat partition with an ext4 file system
    for i in {1..3} ; do
        mkfs.ext2 -L ${label} -b 4096 -vv ${dev_name} > /dev/null 2>&1
        sync
        sync
        if [ $i == 1 ]; then
            sleep 2
        fi

        # make sure file system creation is successful
        dumpe2fs -h ${dev_name}  > /dev/null 2>&1
        if [ $? == 0 ]; then
          break;
        fi
    done

    if [ $? != 0 ]; then
        echo "Cannot create ext2 partition exiting"
        exit 1
    fi
}
readonly -f format_ext2_partition

function format_vfat_partition {
    local dev_name=${1} label=${2}

    local disk_size=$(blockdev --getsize64 ${dev_name})

    # Use FAT32 for disks 256M and larger
    if [ ${disk_size} -gt 268435455 ]; then
        local fat_size=32
    else
        local fat_size=16    
    fi

    # Reformat partition with a FAT16/32 file system
    for i in {1..10} ; do
        sync
        mkfs.vfat -n ${label} -I ${dev_name} -F ${fat_size} > /dev/null 2>&1
        rc=$?
        if [ "$rc" -eq "0" ]; then
          partsuccess=yes
          break;
        else
          partsuccess=no
        fi
    done
    if [ "$partsuccess" != "yes" ]; then
        echo "Cannot create vFat partition exiting"
        exit 1
    fi
}

function create_parted_partition {
    local dev=${1}
    # zero the first 10 1024 blocks
    /bin/dd if=/dev/zero of=${dev} bs=1024 count=10000 > /dev/null 2>&1
    if [ $? != 0 ]; then
        echo "/bin/dd: Failed to create partition"
        exitclean
    fi
    parted -s ${dev} mklabel gpt  > /dev/null 2>&1
    if [ $? != 0 ]; then
        echo "parted gpt: Failed to create partition"
        exitclean
    fi
    sleep 1
    if [ "$2" == "fretta" ]; then
        parted -s --align=min ${dev} mkpart usb_install 1 3G > /dev/null 2>&1
    else
        parted -s --align=min ${dev} mkpart usb_install 1 2G > /dev/null 2>&1
    fi 

    if [ $? != 0 ]; then
        echo "parted mkpart: Failed to create partition"
        exitclean
    fi
    sleep 1
    parted -s ${dev} set 1 boot on > /dev/null 2>&1
    if [ $? != 0 ]; then
        echo "parted set 1 boot: Failed to create partition"
        exitclean
    fi
    sleep 1
}

function create_fdisk_partition {
    local dev_name=${1}
    # zero the first 10000 1024 blocks
    /bin/dd if=/dev/zero of=${dev_name}1 bs=1024 count=10000 > /dev/null 2>&1
    if [ $? != 0 ]; then
        echo "create_fdisk_partition failed"
        exitclean
    fi

    /bin/dd if=/dev/zero of=${dev_name} bs=1024 count=10000 > /dev/null 2>&1
    if [ $? != 0 ]; then
        echo "create_fdisk_partition failed"
        exitclean
    fi

    /sbin/fdisk ${dev_name} < ${2} > /dev/null 2>&1
    if [ $? != 0 ]; then
        echo "create_fdisk_partition failed"
        exitclean
    fi
}


checkFilesystem() {
    dev=$1

    USBFS=$(/bin/mount -f --guess-fstype $dev)
    if [ "$USBFS" != "ext2" -a "$USBFS" != "ext3" -a "$USBFS" != "ext4" ]; then
        echo "USB filesystem must be ext[234]"
        exitclean
    fi
}

checkMakePartActive() {
    dev=$1

    if [ "$(/sbin/fdisk -l $device 2>/dev/null |grep $dev |awk {'print $2;'})" != "*" ]; then
        /sbin/fdisk $device << EOF > /dev/null 2>/dev/null
a
1
w
EOF
    fi
}

if [ $(id -u) != 0 ]; then
    echo "You need to be root to run this script"
    exit 1
fi

ISO=$(readlink -f "$1")
USBDEV=$2
platform_type=$(cat /proc/cmdline | awk -F "platform=" ' { print $2 }' | cut -d " "  -f1)
echo "Platform is: $platform_type"
if [ "$#" -eq 2 ]; then
  installtype=EFI
fi
if [ "$#" -eq 3 ]; then
  installtype="$3"
fi
if [ "$#" -eq 4 ]; then
  installtype="$3"
fi

if [ "${installtype}" != "legacy" -a "${installtype}" != "EFI" ]; then
    usage
fi

if [ -z "$ISO" -o ! -f "$ISO" ]; then
    usage
fi

if [ -z "$USBDEV" -o ! -b "$USBDEV" ]; then
    usage
fi

getdisk $USBDEV
if [ "$platform_type" == "fretta" ]; then
  preparedisk ${device}
fi

if [ -n "$device_in_part" ]; then
  if [ -n "$device_serial" ]; then
      device_with_part=${device}
  else
      device_with_part=${device}${device_in_part}
      # remove USB disk partitions from XR VM
      part_num=$(grep -c ${devname%%[0-9]*} /proc/partitions)
      echo "Remove all USB disk partitions from XR VM, part_num = $i{part_num} should > 1"

      for (( i=1; i<part_num; i++ ))
      do
        remove_partition usb${devname}${i}
      done

  fi
else
  device_with_part=${device}1
  if [ ! -n "$device_serial" ]; then
      # remove USB disk from XR VM
      remove_partition usb${devname}
  fi
fi
umount ${device}1 > /dev/null 2>&1 # Sometime host automatically mounts the disk
umount ${device} > /dev/null 2>&1


if [ "$installtype" == "legacy" ]; then
  echo "Preparing USB stick for legacy grub"
  create_part_file_disk
  if [ -z ${device_in_part} ]; then
      if [ -n "$device_serial" ]; then
        echo "need create partition, please run usb-install from host"
        exit
      else
        echo "create partition for ${device}"
        create_fdisk_partition ${device} "/tmp/fdisk.input"
        sleep 1
        # udev rule be invoked and passthrough disk partition to XR VM
        # need remove disk from XR VM
        remove_partition usb${devname}1
      fi
  fi
  echo "Create filesystem on ${device_with_part}"
  if [ "$platform_type" == "fretta" ]; then
    format_ext2_partition ${device_with_part} "Fretta_USB"
  else
    format_ext2_partition ${device_with_part} "Panini_USB"
  fi
  echo "Making ${device_with_path} partition bootable"
  checkMakePartActive ${device_with_part}
fi

if [ "$installtype" == "EFI" ]; then
  parted -l -s > /dev/null 2>&1
  rc=$?
  if [ "$rc" -ne "0" ]; then
        echo "    Cannot locate gnu parted binary"
        echo "    Gnu parted installation is a requirement for EFI based install"
        echo "    Please install gnu parted in host machine"
        exit 1
  fi

  echo "Preparing USB stick for EFI"

  if [ -z ${device_in_part} ]; then
    if [ -n "$device_serial" ]; then
        echo "need create partition, please run usb-install from host"
        exit
    else
        echo "create_parted_partition for ${device}"
        create_parted_partition ${device} ${platform_type}
        sleep 1
        # udev rule be invoked and passthrough disk partition to XR VM
        # need remove disk from XR VM
        remove_partition usb${devname}1
    fi
  fi
  echo "Create filesystem on ${device_with_part}"
  if [ "$platform_type" == "fretta" ]; then
      format_vfat_partition ${device_with_part} "Fretta_USB"
  else
      format_vfat_partition ${device_with_part} "Panini_USB"
  fi
fi

# FIXME: would be better if we had better mountpoints
CDMNT=$(mktemp -d /tmp/cdtmp.XXXXXX)
echo "Mounting source iso at $CDMNT"
mount -o loop,ro $ISO $CDMNT || exitclean
USBMNT=$(mktemp -d /tmp/usbdev.XXXXXX)
#Fretta specific code to get right signed grub image(bootx64.efi'
if [ "$platform_type" == "fretta" ]; then
    ROOTFSMNT=$(mktemp -d /tmp/rootfs.XXXXXX)
    mkdir -p /tmp/frettarfs
    (cat ${CDMNT}/boot/initrd.img | (cd /tmp/frettarfs;cpio -id)) || exitclean
    cd /tmp/frettarfs/iso
    mount -o loop,ro /tmp/frettarfs/iso/system_image.iso ${ROOTFSMNT} || exitclean
fi
#end of Fretta specific code
echo "Mounting destination ${device_with_part} at $USBMNT"
    for i in {1..10} ; do
        sync
        mount ${device_with_part} $USBMNT > /dev/null 2>&1
        rc=$?
        if [ "$rc" -eq "0" ]; then
          mountsuccess=yes
          break;
        else
          mountsuccess=no
        fi
    done
if [ "$mountsuccess" != "yes" ]; then
    echo "Cannot mount partition exiting"
    exitclean
fi

trap exitclean SIGINT SIGTERM

echo "Copying image to USB stick"
if [ "$installtype" == "EFI" ]; then
  mkdir -p ${USBMNT}/EFI/boot ${USBMNT}/boot || exitclean
  # copy image with sync flag to avoid slow usb flash disk
  # which is causing the "hung" message,
  # as sync writing 600+M of data to the flash will take time.
  dd if=$ISO of=${USBMNT}/boot/`basename $ISO` bs=1048576 oflag=sync || efiexitclean
  cp -r ${CDMNT}/boot/grub2/* ${USBMNT}/EFI/boot/ || efiexitclean
  sed -i -e "s;root=<P.*>;root=/dev/ram install=/dev/sda;" \
           -e "s;<V.*>;Panini_USB;" \
           -e "s;nointremap msireserve=133-182:0x2345,183-232:0x2346;;" \
           -e "s;from Disk;from USB;" \
           ${USBMNT}/EFI/boot/grub.cfg

  if [ $? != 0 ]; then
    efiexitclean
  fi

  if [ "$platform_type" == "fretta" ]; then
      cp -f ${ROOTFSMNT}/boot/grub2/bootx64.efi ${USBMNT}/EFI/boot/ || efiexitclean
      cp -f /opt/cisco/hostos/etc/usb-install/grub.cfg ${USBMNT}/EFI/boot/ || efiexitclean
      sed -i -e "s;loopback loop /ncs5500-mini-x.iso;loopback loop /boot/`basename $ISO`;" \
           ${USBMNT}/EFI/boot/grub.cfg
  fi

  if [ -d ${CDMNT}/boot/elilo ]; then
    cp -r ${CDMNT}/boot/elilo/* ${USBMNT}/EFI/boot/ || efiexitclean
  fi
  cp -r ${CDMNT}/boot/bzImage* ${USBMNT}/boot/ || efiexitclean
  for i in ${CDMNT}/boot/initrd*; do
    if [ -e $i ]; then
      dd if=$i of=${USBMNT}/boot/`basename $i` bs=1048576 oflag=sync || efiexitclean
    fi
  done
  if [ -d ${CDMNT}/boot/certs ]; then
    cp -r ${CDMNT}/boot/signature* ${USBMNT}/boot/ || efiexitclean
    cp -r -f ${CDMNT}/boot/certs ${USBMNT}/boot/ || efiexitclean
  fi
  sync
  sync
  cleanup
  echo "USB stick set up for EFI boot!"
  exit 0
fi

cp -r $CDMNT/* $USBMNT/
#mkdir -p ${USBMNT}/proc ${USBMNT}/sys ${USBMNT}/root
#mkdir -p ${USBMNT}/tmp ${USBMNT}/dev
chown -R root ${USBMNT}/*
chgrp -R root ${USBMNT}/*


rootdir=$USBMNT
bootdir=${rootdir}/boot
grubdir=${bootdir}/grub
device_map=${grubdir}/device.map
grub_prefix=${grubdir}/
grub_shell=/sbin/grub

# Get the drive name.
grub_drive=`grep -v '^#' $device_map | grep "$device *$" \
        | sed 's%.*\(([hf]d[0-9][a-g0-9,]*)\).*%\1%'`

# If a partition is specified, we need to translate it into the
# GRUB's syntax.
install_drive=$grub_drive
install_part=$device_part
grub_part=`expr $device_part - 1`
root_drive=`echo "$grub_drive" | sed "s%)$%,$grub_part)%"`

if [ "x$root_drive" = x ]; then
    echo "Cannot determin the grub root drive"
    echo "Image install failed"
    exitclean
    exit 1
fi

if ! test -e ${grubdir}/grub.conf ; then
    test -e ${grubdir}/menu.lst && ln -s ./menu.lst ${grubdir}/grub.conf
fi
sed -i -e "s;root=/dev/ram; root=/dev/ram boot_type=legacy;" \
         ${grubdir}/menu.lst

# Before all invocations of the grub shell, call sync to make sure
# the raw device is in sync with any bufferring in filesystems.
sync


# Now perform the installation.
echo "Installing boot loader"
$grub_shell --batch $no_floppy --device-map=$device_map <<EOF >> /tmp/usbinstall.log
root $root_drive
setup $force_lba --stage2=$grubdir/stage2 $install_drive
quit
EOF

#fix menu.lst script
sed -i~ -e "s;root (cd); root (hd0,$grub_part);" ${grubdir}/menu.lst


[ -d "$CDMNT" ] && umount $CDMNT && rmdir $CDMNT
[ -d "$USBMNT" ] && umount $USBMNT && rmdir $USBMNT
[ -d "$ROOTFSMNT" ] && umount $ROOTFSMNT && rmdir $ROOTFSMNT
rm -rf /tmp/rootfs

echo "USB stick set up for boot!"
