#!/bin/bash

. /etc/init.d/spirit-functions
. /etc/init.d/operns-functions

# On some platforms, sourcing spirit-functions (and its dependent libraries)
# has the side-effect of setting traps on SIGINT and SIGQUIT. That may be 
# fine for other processes but we don't want that for our interactive shell!
# Clear those traps
trap - SIGINT || true
trap - SIGQUIT || true
stty quit '^\'
stty susp '^z'

if in_xrnns ; then
    # In xrnns, legacy behavior is to use the node name instead of the
    # Linux hostname in the prompt.
    export PS1="[\$(/pkg/bin/hostname):\\w]$"
else
    # In non-xrnns (OperNS / Global-VRF environment), we need to
    # remove the nonstandard ulimits that XR places on XR processes
    # and instead provide more typical Linux limits.

    # Hard limits (absolute maximum processes can request)

    # No limit on stack size
    ulimit -H -s unlimited

    # Soft limits (default maximum for processes)

    # No data segment size limit
    ulimit -S -d unlimited
    # 8192k stack size limit
    ulimit -S -s 8192
fi

# docker client export to establish connection with docker daemon
export DOCKER_HOST=unix:///misc/app_host/docker.sock

if [[ -z $VMTYPE ]]; then
    VMTYPE=`cat /proc/cmdline | sed 's/^.*vmtype=//' | cut -d " " -f1`
fi
if [[ $VMTYPE == "xr-vm" ]]; then
    if [[ -z $VIRT_METHOD ]]; then
        VIRT_METHOD=$(grep VIRT_METHOD /etc/init.d/calvados_bootstrap.cfg | awk -F '=' '{print $2}')
    fi
    if [[ $VIRT_METHOD == "lxc" ]]; then
        # On LXC systems, libvirtd runs on the host and is port-forwarded via sysadmin.
        #
        # Determine the IP address to connect to Calvados (sysadmin) on
        # Some example values we might find:
        # 192.0.0.1    host in single RP system
        # 192.0.108.1  host RP0 in dual RP system
        # 192.0.112.1  host RP1 in dual RP system
        #
        eval $(grep -a CALVADOS_IP_ADDR /dev/xr_bootstrap);

        export LIBVIRT_DEFAULT_URI="lxc+tcp://${CALVADOS_IP_ADDR}:$(get_tcp_port)"
    else
        # On KVM systems, libvirtd is running locally within the VM,
        # but will default to KVM instead of LXC, which we don't support for apps.
        export LIBVIRT_DEFAULT_URI="lxc://"
    fi
fi
