#!/bin/bash
#
# calvados/asr9k_pkg/boot/scripts/rp_resync.sh
# This script allows RSP to resync it's grub and 
# command line arguments if it has been moved
# across chassis.
#
# Copyright (c) 2014-2018 by Cisco Systems, Inc.
# All rights reserved.

function rp_cmdline_resync()
{
    local VM_TYPE=$1
    
    get_chassis_details_ipu
    get_chassis_details_cmd

    umount /proc/cmdline
    local PROC_CMDLINE=/root/cmdline

    if [ ! -z "${CHASSIS_SERIAL}" ]; then
        if [ "${CMD_CHASSIS_SERIAL}" != "${CHASSIS_SERIAL}" ]; then
            strip_replace "chassis_serial" "chassis_serial=${CHASSIS_SERIAL}" ${PROC_CMDLINE}
        fi
    fi

    if [ ! -z "${CHASSIS_TYPE}" ]; then
        if [ "${CMD_CHASSIS_TYPE}" != "${CHASSIS_TYPE}" ]; then
            strip_replace "chassis_type" "chassis_type=${CHASSIS_TYPE}" ${PROC_CMDLINE}
            bash /etc/init.d/genhosts.sh / $VM_TYPE
        fi
    fi

    if [ ! -z "${CHASSIS_PID}" ]; then
        if [ "${CMD_CHASSIS_PID}" != "${CHASSIS_PID}" ]; then
            strip_replace "chassis_pid" "chassis_pid=${CHASSIS_PID}" ${PROC_CMDLINE}
        fi
    fi

    mount -n -o ro --bind /root/cmdline /proc/cmdline
    # Newly modified /proc/cmdline requires re-parsing
    . /etc/init.d/xr_sysctl.sh restart
    . /etc/cisco/xr_sysctl.rc

    kmsg_log "cmdline_resync: IPU: SERIAL: ${CHASSIS_SERIAL} TYPE: ${CHASSIS_TYPE} PID: ${CHASSIS_PID}"
    kmsg_log "cmdline_resync: CMD: SERIAL: ${CMD_CHASSIS_SERIAL} TYPE: ${CMD_CHASSIS_TYPE} PID: ${CMD_CHASSIS_PID}"
}

function sync_chassis_details()
{
    local GRUB_FILE=$1
    get_chassis_details_ipu

    local CUR_CHASSIS_SERIAL=$(cat ${GRUB_FILE} | grep chassis_serial= | sed "s;^.*chassis_serial=;;" | sed "s; .*$;;")
    if [ ! -z "${CHASSIS_SERIAL}" ]; then
        if [ "${CUR_CHASSIS_SERIAL}" != "${CHASSIS_SERIAL}" ]; then
            strip_replace "chassis_serial" "chassis_serial=${CHASSIS_SERIAL}" ${GRUB_FILE}
        fi
    fi

    local CUR_CHASSIS_TYPE=$(cat ${GRUB_FILE} | grep chassis_type= | sed "s;^.*chassis_type=;;" | sed "s; .*$;;")
    if [ ! -z "${CHASSIS_TYPE}" ]; then
        if [ "${CUR_CHASSIS_TYPE}" != "${CHASSIS_TYPE}" ]; then
            strip_replace "chassis_type" "chassis_type=${CHASSIS_TYPE}" ${GRUB_FILE}
        fi
    fi

    local CUR_CHASSIS_PID=$(cat ${GRUB_FILE} | grep chassis_pid= | sed "s;^.*chassis_pid=;;" | sed "s; .*$;;")
    if [ ! -z "${CHASSIS_PID}" ]; then
        if [ "${CUR_CHASSIS_PID}" != "${CHASSIS_PID}" ]; then
            strip_replace "chassis_pid" "chassis_pid=${CHASSIS_PID}" ${GRUB_FILE}
        fi
    fi
    kmsg_log "sync_chassis: IPU: SERIAL: ${CHASSIS_SERIAL} TYPE: ${CHASSIS_TYPE} PID: ${CHASSIS_PID}"
    kmsg_log "sync_chassis: CUR: SERIAL: ${CUR_CHASSIS_SERIAL} TYPE: ${CUR_CHASSIS_TYPE} PID: ${CUR_CHASSIS_PID}"
}

function grub_resync()
{
    local GRUB_FILE=/misc/disk1/tftpboot/grub.cfg

    sync_chassis_details $GRUB_FILE
}

function menu_resync()
{   
    local MENU_LST=/boot/grub/menu.lst

    sync_chassis_details $MENU_LST

}

function hostos_grub_resync()
{
    local GRUB_PART=/dev/sda4
    local GRUB_DIR=/tmp/data
    local GRUB_FILE=${GRUB_DIR}/EFI/Cisco/grub.cfg

    mkdir -p $GRUB_DIR
    mount $GRUB_PART $GRUB_DIR
    sync_chassis_details $GRUB_FILE
    umount $GRUB_DIR
    rmdir  $GRUB_DIR
}

function wait_for_ts_done() {
    local npass
    local i
    let "npass = $1 * 4"

    for  (( i = 0; i < $npass; i++ )); do
        if [[ -f /tmp/ts_done ]]; then
            rm -rf /tmp/ts_done
            break
        fi
        sleep 0.25
    done

    if [[ $i == $npass ]]; then
        kmsg_log "ts_done not signalled"
    else
        kmsg_log "ts_done after $i tries"
    fi
}

function grub_cmdline_resync()
{
    local VM_TYPE=$1
    local BOARDTYPE=$2
    local hw_ct=$3

    if [ "$VM_TYPE" == "hostos" ]; then
        if [ "$BOARDTYPE" == "RP" ]; then
            hostos_grub_resync
            case "$hw_ct" in
            26) # RSP4-Lite
                initctl emit ts_agt_rsp4l_start
                ;;
            *)
                initctl emit ts_agt_rp_start
                ;;
            esac

        else
            initctl emit ts_agt_lc_start
            wait_for_ts_done 8
        fi

        return
    fi

    # Do not run the cmdline and grub routines for LC
    # LC does not have grub. And if LC cmdline has incorrect 
    # information we reimage that LC via ts_agt.c
    if [ "$BOARDTYPE" == "LC" ]; then
        return
    fi

    if [ "$VM_TYPE" == "sysadmin-vm" ]; then
        grub_resync
    fi
    menu_resync
    rp_cmdline_resync $VM_TYPE
}

