#!/bin/sh
#
# tp_app_cgrp_cfg.sh:
#     This script is responsible, for setting the CGROUP root node for third party 
#     application hosting, the purpose of this node is to constrain system resources
#     for all third party applications running in any containers.
# 
# Feb. 2016, Mohamed Mahmoud
#
# Copyright (c) 2016-2017 by cisco Systems, Inc.
#

if [ -f /etc/init.d/spirit_pd.sh ]; then
    source /etc/init.d/spirit_pd.sh
fi

if [ -f /etc/init.d/pd-functions ]; then
    source /etc/init.d/pd-functions
    declare -F pd_tpa_cgroup_settings &>/dev/null && pd_tpa_cgroup_settings
fi

if [[ -f /etc/init.d/spirit_log.sh ]]; then
    source /etc/init.d/spirit_log.sh
fi

cgroup_mount_pt=""
cgroups_to_mount=()
tp_app_cgroup_node="tp_app.partition"
tp_app_docker_cgroup_node="tp_app.partition/docker"
tp_app_lxc_cgroup_node="tp_app.partition/lxc.partition"

OSTYPE=$1

if [[ "$OSTYPE" == "thinxr" ]]; then 
    cgroup_mount_pt="/sys/fs/cgroup"
    cgroups_to_mount=( "blkio" "cpu" "cpuacct" "cpuset" "devices" "freezer" "hugetlb" "memory" "perf_event" )
else
    VMTYPE=`cat /proc/cmdline | sed 's/^.*vmtype=//' | cut -d" " -f1`

    if [[ (("$VIRT_METHOD" == "lxc" && "$VMTYPE" == "hostos") ||
         ("$VIRT_METHOD" == "vm" && "$VMTYPE" == "xr-vm")) ]]; then
        platform_log "Virt method $VIRT_METHOD, VM type $VMTYPE - Create TP-APP cgroup"
    else
        platform_log "Virt method $VIRT_METHOD, VM type $VMTYPE - Need not create TP-APP cgroup"
        exit 0
    fi
    
    cgroup_mount_pt="/dev/cgroup"
    cgroups_to_mount=( "cpu" "cpuacct" "cpuset" "devices" "freezer" "memory" )
fi

# app_hosting_cgroup_set_attribute
# set function to set values to cgroup files 
#

function app_hosting_cgroup_set_attribute
{
    local value=$1
    local cgroup_file=$2

    platform_log "Setting $value to $cgroup_file"
    echo "$value" > "$cgroup_file"
    if [[ $? -ne 0 ]]; then 
        platform_log_error "Error setting $value to $cgroup_file"
    fi
}

# app_hosting_cgroup_set_machine_cgroup
#
# /dev/cgroup/<subsystem>/machine cgroup has to created inside xr "vm" 
# Inherit the default values from the parent
# Parameters in this cgroup has to be set before setting the values 
# in the children

function app_hosting_cgroup_set_machine_cgroup
{
    # In XR, machine cgroup is created before creating third party cgroup
    if [[ "$OSTYPE" != "thinxr" ]]; then 
        if ! [[ "$VIRT_METHOD" == "vm" && "$VMTYPE" == "xr-vm" ]]; then 
            true
            return 
        fi
    fi

    for subsystem in "${cgroups_to_mount[@]}"
    do 
        mkdir -p $cgroup_mount_pt/$subsystem/machine
    done
        
    $PLATFORM_LOG_EXEC cp $cgroup_mount_pt/cpuset/cpuset.mems $cgroup_mount_pt/cpuset/machine/cpuset.mems
    $PLATFORM_LOG_EXEC cp $cgroup_mount_pt/cpuset/cpuset.cpus $cgroup_mount_pt/cpuset/machine/cpuset.cpus
    $PLATFORM_LOG_EXEC cp $cgroup_mount_pt/cpu/cpu.shares $cgroup_mount_pt/cpu/machine/cpu.shares
    $PLATFORM_LOG_EXEC cp $cgroup_mount_pt/memory/memory.use_hierarchy $cgroup_mount_pt/memory/machine/memory.use_hierarchy 
   
}

# app_hosting_cgroup_set_tp_app_partition
# Creates /dev/cgroup/<subsystem>/machine/tp_app.partition and sets it 
# To avoid errors while writing values to cgroup files, it is better to 
# set attributes in the hierarchical order
# Set attributes in tp_app.partition cgroup before creating docker 
# and lxc.partition cgroups

function app_hosting_cgroup_set_tp_app_partition 
{
    platform_log "Create $tp_app_cgroup_node cgroup"

    for subsystem in "${cgroups_to_mount[@]}"
    do 
        mkdir -p $cgroup_mount_pt/$subsystem/machine/$tp_app_cgroup_node
    done
 
    # MEM_NUMA_NODES is used to set CGROUP NUMA memory nodes currently 
    # there is only one node.
    if [[ -z "$APP_CGROUP_MEM_NUMA_NODES" ]] ; then
        platform_log_error "Missing APP_CGROUP_MEM_NUMA_NODES setting for tp_app cgroup"
    fi

    app_hosting_cgroup_set_attribute "$APP_CGROUP_MEM_NUMA_NODES" "$cgroup_mount_pt/cpuset/machine/$tp_app_cgroup_node/cpuset.mems"
    app_hosting_cgroup_set_attribute 1 "$cgroup_mount_pt/cpuset/machine/$tp_app_cgroup_node/cpuset.memory_migrate"

    # CPUSET_CPUS configure how many CPU cores will be used to third party 
    # application hosting
    if [[ -z "$APP_CGROUP_CPUSET_CPUS" ]] ; then
        platform_log_error "Missing APP_CGROUP_CPUSET_CPUS setting for tp_app cgroup"
    fi

    app_hosting_cgroup_set_attribute "$APP_CGROUP_CPUSET_CPUS" "$cgroup_mount_pt/cpuset/machine/$tp_app_cgroup_node/cpuset.cpus"

    # APP_CGROUP_CPU_SHARES it defines how much tp_app CPU share relative to 
    # its peer CGROUP nodes
    if [[ -z "$APP_CGROUP_CPU_SHARES" ]] ; then
        platform_log_error "Missing APP_CGROUP_CPU_SHARES setting for tp_app cgroup"
    fi

    app_hosting_cgroup_set_attribute "$APP_CGROUP_CPU_SHARES" "$cgroup_mount_pt/cpu/machine/$tp_app_cgroup_node/cpu.shares"
    
    # MEM_LIMIT will limit how much system memory will be used for 
    # application hosting, 
    # Note: need to enable use_hierarchy flag to propagate this memory limits 
    # from tp_app CGROUP node to all of its children.
    if [[ -z "$APP_CGROUP_MEM_LIMIT_IN_BYTES" ]] ; then
        platform_log_error "Missing APP_CGROUP_MEM_LIMIT_IN_BYTES setting for tp_app cgroup"
    fi

    app_hosting_cgroup_set_attribute "$APP_CGROUP_MEM_LIMIT_IN_BYTES" "$cgroup_mount_pt/memory/machine/$tp_app_cgroup_node/memory.limit_in_bytes"
    app_hosting_cgroup_set_attribute "$APP_CGROUP_MEM_LIMIT_IN_BYTES" "$cgroup_mount_pt/memory/machine/$tp_app_cgroup_node/memory.memsw.limit_in_bytes"
    app_hosting_cgroup_set_attribute 1 "$cgroup_mount_pt/memory/machine/$tp_app_cgroup_node/memory.use_hierarchy"
}

        
# app_hosting_cgroup_mkdir_tp_docker_lxc
#
# Creates child level cgroup directory for third party application, which
# will be the mounting point for the daemon process to install all containers 
# instances.

function app_hosting_cgroup_mkdir_tp_docker_lxc 
{
    platform_log "Create TP-APP cgroup directory"

    for subsystem in "${cgroups_to_mount[@]}"
    do 
        mkdir -p $cgroup_mount_pt/$subsystem/machine/$tp_app_docker_cgroup_node
        mkdir -p $cgroup_mount_pt/$subsystem/machine/$tp_app_lxc_cgroup_node
    done
}

# app_hosting_cgroup_set_tp_docker_lxc
#
# Configure third party's cgroup children node resources limit based on per
# platform settings to constrain how much system resources to be used for 
# third party applications.
#

function app_hosting_cgroup_set_tp_docker_lxc
{
    platform_log "Setting up docker and lxc cgroup node"

    # MEM_NUMA_NODES is used to set CGROUP NUMA memory nodes currently 
    # there is only one node.
    if [[ -z "$APP_CGROUP_MEM_NUMA_NODES" ]] ; then
        platform_log_error "Missing APP_CGROUP_MEM_NUMA_NODES setting for tp_app cgroup"
    fi

    app_hosting_cgroup_set_attribute "$APP_CGROUP_MEM_NUMA_NODES" "$cgroup_mount_pt/cpuset/machine/$tp_app_docker_cgroup_node/cpuset.mems"
    app_hosting_cgroup_set_attribute 1 "$cgroup_mount_pt/cpuset/machine/$tp_app_docker_cgroup_node/cpuset.memory_migrate"
    app_hosting_cgroup_set_attribute "$APP_CGROUP_MEM_NUMA_NODES" "$cgroup_mount_pt/cpuset/machine/$tp_app_lxc_cgroup_node/cpuset.mems"
    app_hosting_cgroup_set_attribute 1 "$cgroup_mount_pt/cpuset/machine/$tp_app_lxc_cgroup_node/cpuset.memory_migrate"

    # CPUSET_CPUS configure how many CPU cores will be used to third party 
    # application hosting
    if [[ -z "$APP_CGROUP_CPUSET_CPUS" ]] ; then
        platform_log_error "Missing APP_CGROUP_CPUSET_CPUS setting for tp_app cgroup"
    fi

    app_hosting_cgroup_set_attribute "$APP_CGROUP_CPUSET_CPUS" "$cgroup_mount_pt/cpuset/machine/$tp_app_docker_cgroup_node/cpuset.cpus"
    app_hosting_cgroup_set_attribute "$APP_CGROUP_CPUSET_CPUS" "$cgroup_mount_pt/cpuset/machine/$tp_app_lxc_cgroup_node/cpuset.cpus"

    # Set cpu.shares for docker and lxc partition
    
    app_hosting_cgroup_set_attribute 1024 "$cgroup_mount_pt/cpu/machine/$tp_app_docker_cgroup_node/cpu.shares"
    app_hosting_cgroup_set_attribute 1024 "$cgroup_mount_pt/cpu/machine/$tp_app_lxc_cgroup_node/cpu.shares"


    # MEM_LIMIT will limit how much system memory will be used for 
    # application hosting, 
    # Note: need to enable use_hierarchy flag to propagate this memory limits 
    # from tp_app CGROUP node to all of its children.
    
    if [[ -z "$APP_CGROUP_MEM_LIMIT_IN_BYTES" ]] ; then
        platform_log_error "Missing APP_CGROUP_MEM_LIMIT_IN_BYTES setting for tp_app cgroup"
    fi
    
    app_hosting_cgroup_set_attribute "$APP_CGROUP_MEM_LIMIT_IN_BYTES" "$cgroup_mount_pt/memory/machine/$tp_app_docker_cgroup_node/memory.limit_in_bytes"
    app_hosting_cgroup_set_attribute "$APP_CGROUP_MEM_LIMIT_IN_BYTES" "$cgroup_mount_pt/memory/machine/$tp_app_lxc_cgroup_node/memory.limit_in_bytes"
    
    app_hosting_cgroup_set_attribute "$APP_CGROUP_MEM_LIMIT_IN_BYTES" "$cgroup_mount_pt/memory/machine/$tp_app_docker_cgroup_node/memory.memsw.limit_in_bytes"
    app_hosting_cgroup_set_attribute "$APP_CGROUP_MEM_LIMIT_IN_BYTES" "$cgroup_mount_pt/memory/machine/$tp_app_lxc_cgroup_node/memory.memsw.limit_in_bytes"
    
    app_hosting_cgroup_set_attribute 1 "$cgroup_mount_pt/memory/machine/$tp_app_docker_cgroup_node/memory.use_hierarchy"
    app_hosting_cgroup_set_attribute 1 "$cgroup_mount_pt/memory/machine/$tp_app_lxc_cgroup_node/memory.use_hierarchy"
}

# app_hosting_cgroup_tp_app_process
#
# it checks if cgroup processing was done it will return w/o doing anything 
# else it will create required cgroup directories and set them based 
# on PD configuration

function app_hosting_cgroup_tp_app_process
{
    # the following check verify that cgroup has been run and 
    # all the way to the end and if that is the case then avoid rerunning it.
    chk_pt_file=$cgroup_mount_pt/memory/machine/$tp_app_lxc_cgroup_node/memory.use_hierarchy
    if [[ -f "$chk_pt_file" ]]; then
        if [[ "$( cat $chk_pt_file )" = "1" ]]; then
            platform_log "TPA-CGROUP already exists"
            return
        fi
    fi
    
    app_hosting_cgroup_set_machine_cgroup
    app_hosting_cgroup_set_tp_app_partition
    app_hosting_cgroup_mkdir_tp_docker_lxc
    app_hosting_cgroup_set_tp_docker_lxc
}

app_hosting_cgroup_tp_app_process
