#!/bin/bash
# 
# lxc - libvirt pd hook script
# This script is added as a pd hack to pass interface from host to XR container for ucs
#
# Copyright (c) 2014-2016 by Cisco Systems, Inc.

#invoked with $1 $2 $3 $4 as received by the caller

source /etc/rc.d/init.d/spirit_pd.sh

function pd_handle_init_start()
{
    local vm_name=$1
    local vmm_path="/usr/local/calvados/vm_manager"

    case "$vm_name" in
    default-sdr--*)
        log_msg "pd_handle_init_start: VM $vm_name no op"
        ;;

    *)
        ;;
    esac

   return 0
}
readonly -f pd_handle_init_start

function pd_handle_sched_runtime()
{
    local vm_name=$1
    local src_file=$2
    local tgt_file=$3

    case "$vm_name" in
    default-sdr--*)
        log_msg "pd_handle_sched_runtime: VM $vm_name no op"
        ;;

    *)
        ;;
    esac

   return 0
}
readonly -f pd_handle_sched_runtime

function kmsg_lxc_hook()
{
    local CMD=$@
    printf "%s CMD: %s\n" "$timestamp" "$CMD" >> $LXC_HOOK_LOG_FILE
    kmsg_exec $CMD >> $LXC_HOOK_LOG_FILE 2>&1
    local ret=$?
    return $ret
}

#invoked with $1 $2 $3 $4 as received by the caller
function lxc_reclaim_sysadmin_intf_hook()
{
    kmsg_log "lxc_reclaim_sysadmin_intf_hook"

    kmsg_lxc_hook "vconfig rem eth-vf1.3073"

    kmsg_lxc_hook "vconfig rem eth-vf1.513"

    if [[ -L /sys/class/net/eth-vf1 ]]; then
        kmsg_lxc_hook "ip link set eth-vf1 name eth-vnic1"
    fi

    BOARDTYPE=`cat /proc/cmdline | sed 's/^.*boardtype=//' | cut -d" " -f1`
    if [ "${BOARDTYPE}" == "RP" ]; then 
        kmsg_lxc_hook "ip link set dev eth7 name eth-mgmt2"
    fi
}

#pass interfaces to admin, invoked with $1 $2 $3 $4 as received by caller
function lxc_set_sysadmin_intf_hook()
{
    #
    # Host needs to move vnic interfaces to respective lxcs.
    # sim environment probably requires eth1 to be transported
    # to default-sdr--1. This will have to be handled based on
    # PI-PD separation.
    #

    kmsg_log "lxc_set_sysadmin_intf_hook"

    update_virt_respawn_count sysadmin-vm 0
    kmsg_lxc_hook "ip link set eth-vnic1 netns $4"
    ret=$?
    if [[ "$ret" -ne 0 ]]
    then
        kmsg_log "lxc_set_sysadmin_intf_hook exiting" 
        exit 1
    fi
    
    BOARDTYPE=`cat /proc/cmdline | sed 's/^.*boardtype=//' | cut -d" " -f1`
    if [ "${BOARDTYPE}" == "RP" ]; then 
        kmsg_lxc_hook "ip link set eth-mgmt2 netns $4"
    fi
}

function unhook_vnic()
{
    local sysfs_vnic=$1
    if [[ ! -L $sysfs_vnic ]]; then
        return
    fi

    local bdf_filename
    if [[ -f /tmp/eth_bdf ]]; then
        bdf_filename=/tmp/eth_bdf
    elif [[ -f /etc/eth_bdf ]]; then
        bdf_filename=/etc/eth_bdf
    else
        kmsg_log "Cannot find NIC/BDF mapping file"
        return
    fi

    local curr_vnic=$( basename $sysfs_vnic )
    local nic_bdf=$( ls -l $sysfs_vnic | awk -F '/' '{print $10}' )
    local orig_vnic=$( grep $nic_bdf $bdf_filename | awk '{print $1 }' )

    if [[ -z $orig_vnic ]]; then
        kmsg_log "Cannot find original VNIC for $sysfs_vnic/$nic_bdf"
    else
        kmsg_lxc_hook "ip link set $curr_vnic name $orig_vnic"
    fi
}

function lxc_reclaim_sdr_intf_hook()
{
    kmsg_log "lxc_reclaim_sdr_intf_hook"

    BOARDTYPE=`cat /proc/cmdline | sed 's/^.*boardtype=//' | cut -d" " -f1`
    if [ "${BOARDTYPE}" == "RP" ]; then
        kmsg_lxc_hook "ip link set dev eth6 name eth-mgmt1"
    fi

    kmsg_lxc_hook "vconfig rem eth-vf1.3073"
    kmsg_lxc_hook "vconfig rem eth-vf1.3074"
    unhook_vnic /sys/class/net/eth-vf1

    kmsg_lxc_hook "vconfig rem eth-punt.1282"
    unhook_vnic /sys/class/net/eth-punt

    if [[ "${BOARDTYPE}" == "RP" ]]; then
        kmsg_lxc_hook "vconfig rem eth-tpa.1282"
        unhook_vnic /sys/class/net/eth-tpa
    fi

    if [[ "${BOARDTYPE}" == "LC" ]]; then
        kmsg_lxc_hook "vconfig rem eth-bfd.1282"
        unhook_vnic /sys/class/net/eth-bfd
    fi
}

# read field=value pair from vmm cfg (a.k.a /dev/xr_bootstrap in xr vm) file
# example : lxc_cfg_field_value default-sdr--1 ISSU_ROLE
function lxc_cfg_field_value()
{
    local vmm_cfg="/usr/local/calvados/vm_manager/$1/vmm_guest_bootstrap.cfg"

    echo `strings $vmm_cfg | grep $2 | cut -d"=" -f2`
}

####################################################
# Interface assignment scheme
#
#  eth-vnic1     Calvados
#  eth-vnic2     RP XR V1 or V2,       LC XR V1
#  eth-vnic3     Calvados (unused)
#  eth-vnic4     RP unused             LC XR V2
#  eth-vnic5     RP XR V1 or V2,       LC XR V1
#  eth-vnic6     RP unused             LC XR V2
#  eth-vnic7     RP XR V1 or V2,       LC XR V1
#  eth-vnic8     RP unused             LC XR V2
#  eth-mgmt1     Only for RP
#  eth-mgmt2     Only for RP 
#
#  Note: on RP, either V1 or V2 is active but not both
#
####################################################
#pass interfaces to xr, invoked with $1 $2 $3 $4 as received by caller
function lxc_set_sdr_intf_hook()
{
    kmsg_log "lxc_set_sdr_intf_hook"

    local issu_role=`lxc_cfg_field_value $1 ISSU_ROLE`
    local issu_id=`lxc_cfg_field_value $1 ISSU_ID`

    log_msg "lxc_set_sdr_intf_hook VM $1 ISSU role $issu_role id $issu_id $4"

    # Host needs to move vnic interfaces to respective lxcs.
    # In case of 9k hardware, move the eth-mgmt1 interface from
    # host kernel to XR guest.
    BOARDTYPE=`cat /proc/cmdline | sed 's/^.*boardtype=//' | cut -d" " -f1`
    update_virt_respawn_count xr-vm $issu_id

    if [ $issu_id -eq 0 ]; then
        kmsg_lxc_hook "ip link set eth-vnic2 netns $4"
        kmsg_lxc_hook "ip link set eth-vnic5 netns $4"
        if [ "${BOARDTYPE}" == "LC" ]; then
            kmsg_lxc_hook "ip link set eth-vnic7 netns $4"
        else
            kmsg_lxc_hook "ip link set eth-vnic7 netns $4"
            kmsg_lxc_hook "ip link set eth-mgmt1 netns $4"
        fi        
    fi    

    if [ $issu_id -eq 1 ]; then
        if [ "${BOARDTYPE}" == "LC" ]; then
            kmsg_lxc_hook "ip link set eth-vnic4 netns $4"
            kmsg_lxc_hook "ip link set eth-vnic6 netns $4"
            kmsg_lxc_hook "ip link set eth-vnic8 netns $4"
        else    
            kmsg_lxc_hook "ip link set eth-vnic2 netns $4"
            kmsg_lxc_hook "ip link set eth-vnic5 netns $4"
            kmsg_lxc_hook "ip link set eth-vnic7 netns $4"
            kmsg_lxc_hook "ip link set eth-mgmt1 netns $4"
        fi        
    fi    

    virt_fabric_init xr-vm $issu_id $issu_role
}

#
# pd_unmount_lxc_rootfs
#   PI counter part is present in lxc_hook.sh
#
function pd_unmount_lxc_rootfs ()
{
    local vm_name=$1
    local func_exit_code=0

    case "$vm_name" in
    sysadmin)
        unmount_lxc_rootfs "$vm_name"
        func_exit_code=$?
        ;;

    default-sdr--*)
        unmount_lxc_rootfs "$vm_name"
        func_exit_code=$?
        ;;

    *)
        log_err "pd_unmount_lxc_rootfs: VM $vm_name, not supported"
        ;;
    esac

    return $func_exit_code
}
readonly -f pd_unmount_lxc_rootfs

#
# pd_prelaunch_setup
#   PI counter part is present in lxc_hook.sh
#
function pd_prelaunch_setup ()
{
    local vm_name=$1
    local func_exit_code=0

    case "$vm_name" in
    sysadmin)
        log_msg "pd_prelaunch_setup: Nothing to do for VM $vm_name"
        ;;

    default-sdr--*)
        prelaunch_setup "$vm_name"
        func_exit_code=$?
        ;;

    *)
        log_err "pd_prelaunch_setup: VM $vm_name, not supported"
        ;;
    esac

    return $func_exit_code
}
readonly -f pd_prelaunch_setup

#
# Gracefully unbind the devices from pci_proxy when SDR reset or reload
#
function pd_pci_cleanup ()
{
    local vm_name=$1
    local func_exit_code=0

    case "$vm_name" in
    default-sdr--*)
        log_msg "pd_pci_cleanup.... $vm_name"
        /bin/sh /etc/init.d/unbind_pci_proxy 255
        func_exit_code=$?
        ;;

    *)
        log_err "pd_pci_cleanup: VM $vm_name, not supported"
        ;;
    esac

    return $func_exit_code
}
readonly -f pd_pci_cleanup
