#!/usr/bin/python
# =============================================================================  
#
# File : sequence_n_install_rpms.py
#
# Following script does the following things.
# 1. Pick all rpms from a folder mathing local card arhitecture.
# 2. Sequence the rpms based on requires relationship and build time.
# 3. Install rpms in the sequenced order. 
#        Host/TP pkgs: rpm -Uv <rpm file> --nodeps --replacefiles
#        Cisco pkgs  : rpm -iv <rpm file> --nodeps --replacefiles
#
# Copyright (c) 2016-2018 by cisco Systems, Inc.                                 
# All rights reserved.                                                           
# ============================================================================= 

from datetime import datetime
import sys
import os
import re
import subprocess
import functools 
import traceback
import optparse
import errno

LOG_FILE="/root/xtra_rpm_install.log"
log_file=open(LOG_FILE,"w")
RPM_BIN="rpm"

#Override RPM_BIN  for running the script in workspace
#RPM_BIN="/nobackup/avastrad/xr-dev/thirdparty/opensource/sdk/x86_64/rpm_dir/sysroots/x86_64-wrlinuxsdk-linux/usr/bin/rpm"
def compare_dotted_string_version(s1,s2,operator):
    slist1 = s1.split('.')
    slist2 = s2.split('.')
    ilist1 = []
    ilist2 = []
    for value in slist1: ilist1.append(int(value))
    for value in slist2: ilist2.append(int(value))

    if (operator == ">"):
       return ilist1 > ilist2
    elif (operator == "<"):
       return ilist1 < ilist2
    elif (operator == ">="):
        return ilist1 >= ilist2
    elif (operator == "<="):
        return ilist1 <= ilist2
    elif (operator == "=="):
        return ilist1 == ilist2
    return False 
        
def runcmd(cmd):
    sprc = 0
    process = subprocess.Popen(cmd, stdout=subprocess.PIPE,                    \
                               stderr=subprocess.PIPE, shell=True)
    out, error = process.communicate()
    sprc = process.returncode
    if sprc == None or sprc != 0:
       out = error
       raise RuntimeError("Error CMD=%s returned [%s]" %(cmd, out))
    log_file.write("cmd=%s out=%s\n"%(cmd,out))
    return dict(rc=sprc, output=out)

class Rpm:

    def __init__(self):
        self.name            = None
        self.version         = None 
        self.release         = None 
        self.arch            = None 
        self.package_type    = None 
        self.package_presence= None 
        self.package_PIPD    = None 
        self.build_time      = None 
        self.platform        = None 
        self.card_type       = None 
        self.provides        = None 
        self.requires        = None 
        self.group           = None
        self.vm_type         = None
        self.supp_cards      = None
        self.prefixes        = None

    def populate_mdata (self, rpm):
        result = runcmd(RPM_BIN+" -qp --qf '%{NAME};%{VERSION};%{RELEASE};"    \
                        "%{ARCH};%{PACKAGETYPE};%{PACKAGEPRESENCE};%{PIPD};"   \
                        "%{PLATFORM};%{CARDTYPE};%{BUILDTIME};%{GROUP};"       \
                        "%{VMTYPE};%{SUPPCARDS};%{PREFIXES}' "+rpm)

        resultStrList=result["output"].split(";")
        self.name            =resultStrList[0]
        self.version         =resultStrList[1]
        self.release         =resultStrList[2]
        self.arch            =resultStrList[3]
        self.package_type    =resultStrList[4]
        self.package_presence=resultStrList[5]
        self.package_pipd    =resultStrList[6]
        self.package_platform=resultStrList[7]
        self.card_type       =resultStrList[8]
        self.build_time      =resultStrList[9]
        self.group           =resultStrList[10]
        self.vm_type         =resultStrList[11]
        self.supp_cards      =resultStrList[12].split(",")
        self.prefixes        =resultStrList[13]

        result = runcmd("%s -qp --provides %s"%(RPM_BIN,rpm)) 
        self.provides=result["output"]

        result = runcmd("%s -qp --requires %s"%(RPM_BIN,rpm)) 
        #
        # There can be more than one requires.
        # Ignore requires starting with /
        # example /bin/sh
        # Ignore /bin/sh requires. 
        #
        resultStrList=result["output"].split("\n")
        requiresList =[]
        for requires in resultStrList:
            if not requires.startswith("/"):
                requiresList.append(requires)
           
        self.requires=requiresList

    #
    # RPM is Hostos RPM if rpm name has hostos keyword and platform name.
    #
    def is_hostos_pkg(self, platform):
        return ("hostos" in self.name) and  (platform in self.name)


    #
    # RPM is Thirdparty(TP) rpm if rpm name  does not have rpm name and ISOXR.
    #
    def is_tp_pkg(self, platform):
        return not ((platform in self.name) or ("IOSXR" in self.name.upper()))
    
def is_pkg_prereq(rpm1, rpm2):    
    is_prereq=False
    for requires in rpm1.requires:
        if not requires:
            continue 
        #[CHM:] Need to talk to surya
        #req_pkg_name,req_op,rversion = requires.split(" ")
        req_info = requires.split(" ")
        if len(req_info) > 2:
            req_pkg_name,req_op,rversion = req_info 
        # some of the tp smu has the require info just name
        else:
            req_pkg_name = req_info
            req_op=""
            rversion=""
            
       #
       # Split requires pkg_name,release,ddts if the requires pkg name has CSC
       # example :
       #rpm -qR ncs5500-sysadmin-hostos-6.0.2.02-r60201I.CSCus12345.admin.x86_64
       #   /bin/sh
       #   /bin/sh
       #   /bin/sh
       #   /bin/sh
       #   ncs5500-sysadmin-system = 6.0.2.02_r60201I_CSCus12345
       #
       # N0te: /bin/sh or any requires biginning with / are ignored while 
       #       generating rpm metadata (rpm.populate_mdata())

        if ("CSC" in rversion):
            req_version,req_rel,req_ddts = rversion.split('_')
            if not ((req_ddts in rpm2.release) and (req_rel in rpm2.release)):
                continue
        else: 
            req_version=rversion
              
        if req_version:
            req_pkg_name_version="%s-%s"%(req_pkg_name,req_version)
            rpm2_name_version="%s-%s"%(rpm2.name,rpm2.version) 

        if (req_pkg_name != rpm2.name):
            is_prereq=False
        elif (req_op=="="):
            if (req_pkg_name_version == rpm2_name_version):
                is_prereq=True
                break
        elif (compare_dotted_string_version(rpm2.version, req_version, req_op)):
            is_prereq=True
            break
    return is_prereq
        
def rpm_requires_cmp(rpm1, rpm2):
    compare=0
    is_prereq1 = is_pkg_prereq(rpm2, rpm1)
    is_prereq2 = is_pkg_prereq(rpm1, rpm2)
    if (is_prereq1 and not is_prereq2):
        compare = -1
    elif (not is_prereq1 and is_prereq2):
        compare = 1
    else:
        if (int(rpm1.build_time) > int(rpm2.build_time)):
            compare = 1
        elif (int(rpm1.build_time) < int(rpm2.build_time)):
            compare = -1
        else:
            compare=0    
    return compare

def decode_fancy_platform_name(plat):
    PLATFORM={'fretta':'ncs5500',
              'zermatt':'ncs5500'}
    if plat in PLATFORM.keys():
        return PLATFORM[plat]
    return plat

def cliparser():

    parser = optparse.OptionParser(description='Python script to order         \
                                             and install rpm[s]')

    parser.add_option('--repo', dest='rpmRepo', type=str,                      \
                      help='Path to RPM repository')

    parser.add_option('--file', dest='repofile', type=str,                      \
                      help='File includes the list of packages to sequence')

    parser.add_option('--root', dest='installRoot', type=str,                  \
                      default='/',                                             \
                            help='Root of filesystem where rpms need to be     \
                                  installed')

    parser.add_option('--cardtype', dest='localCardType', type=str,            \
                      help='Local card type RP/LC')

    parser.add_option('--plat', dest='platform', type=str,                    \
                      help='Platform name ncs6k/ncs1k')

    return parser.parse_args()

def main():
    try:
        try:
            args,tt = cliparser()
        except SystemExit as exc:
            log_file.write("Incorrect inputs to this script\n")
            log_file.write("%s\n"%(sys.argv[:]))
            raise
        args.platform=decode_fancy_platform_name(args.platform)
        result=runcmd("uname -m")
        localCardArch=result["output"].replace("\n","")
        log_file.write("Card Arch: " + localCardArch +"\n")
        rpmFileList = []
        if args.rpmRepo:
            rpmPathList = os.listdir(args.rpmRepo)
            for rpmPath in rpmPathList:
                result=runcmd("file %s/%s"%(args.rpmRepo,rpmPath))
                if not ("RPM" in result["output"]):
                    continue
                rpmFileList.append("%s/%s"%(args.rpmRepo,rpmPath))    
        if args.repofile:
            with open(args.repofile, 'r') as f:
                for line in f:
                    for x in line.strip().split(' '):
                        result = runcmd("file %s"%(x))
                        if not ("RPM" in result["output"]):
                            continue
                        rpmFileList.append(x)
        rpmList = []
        for rpmFile in rpmFileList:
            rpm = Rpm()
            #tp base rpm should not take part in installation
            #only tp smu will be installed
            rpm.populate_mdata(rpmFile)
            if rpm.is_tp_pkg(args.platform) and not "CSC" in rpmFile:
                log_file.write("%s: Skipping tp base rpm(%s) from installation" % 
                       (datetime.utcnow().strftime("%a %b %d %H:%M:%S UTC %Y"), rpmFile))
                continue
            if (localCardArch == rpm.arch):
                rpmList.append(rpm)

        if len(rpmList) == 0:
           log_file.write("No RPMS found in %s [Exiting]"%(args.rpmRepo)) 
           return []
        sortedRpmList = sorted(rpmList,                                        \
                               key=functools.cmp_to_key(rpm_requires_cmp))

        try:
            os.makedirs("%s/rpm/giso_rpms/"%(args.installRoot))
        except OSError as exc:
            if exc.errno != errno.EEXIST:
                raise

        for rpmFile in rpmFileList:
            result=runcmd("cp %s %s/rpm/giso_rpms/"                           \
                      %(rpmFile, args.installRoot))
        ret_list = []
        for rpmi in sortedRpmList:
            excludePath = ""
            for supp_card  in rpmi.supp_cards:
                if not supp_card:
                    continue
                elif ((args.localCardType.upper() != supp_card.upper())        \
                       and (supp_card.upper() != "ALL")):
                    if not rpmi.is_tp_pkg(args.platform):
                        excludePath=excludePath+ " --excludepath=%s/%s-%s-%s/%s"   \
                        %(rpmi.prefixes, rpmi.name, rpmi.version, rpmi.release,\
                        supp_card)
            #nvra: name version release arch
            rpmi_nvra='%s-%s-%s.%s'%(rpmi.name, rpmi.version, \
                                     rpmi.release, rpmi.arch)

            if (rpmi.is_hostos_pkg(args.platform)) or                          \
                (rpmi.is_tp_pkg(args.platform)): 
                # Will be removed once Hostos pkg UT is done
                rpm_options = " rpm -U --nodeps --replacepkgs --force " 
            else :
                rpm_options = " rpm -i --nodeps --replacefiles --force " 
            if args.repofile:    
                result=runcmd("chroot %s %s %s /rpm/giso_rpms/%s-%s-%s.%s" \
                          %(args.installRoot, rpm_options, excludePath,        \
                            rpmi.name, rpmi.version, rpmi.release, rpmi.arch)) 
            else:                
                result=runcmd("chroot %s %s %s /rpm/giso_rpms/%s-%s-%s.%s.rpm" \
                          %(args.installRoot, rpm_options, excludePath,        \
                            rpmi.name, rpmi.version, rpmi.release, rpmi.arch)) 

            log_file.write("%s: Installing %s-%s-%s.%s [SUCCESS] [out=%s]\n"       \
                           %(datetime.utcnow().strftime("%a %b %d %H:%M:%S UTC %Y"),
                             rpmi.name, rpmi.version, rpmi.release, rpmi.arch, \
                             result["output"]))
            ret_list.append(rpmi_nvra)
        return ret_list
    except Exception as inst:
        exc_info = sys.exc_info()
        TB = traceback.format_exc()
        log_file.write("\nException:\n %s\n"%(exc_info[0]))
        log_file.write("%s\n"%(exc_info[1]))
        log_file.write("\nTraceBack: %s\n "%(TB))
        return []
    finally:
        log_file.write("Done\n")
        log_file.close()

if __name__ == '__main__':
    ret_list = main()
    #print(' '.join(ret_list)) 
    sys.exit(0)
