#!/bin/bash
#
# Copyright (c) 2015 by Cisco Systems, Inc.
# All rights reserved.
#
# The purpose of this script is to generate a modified version of
# calvados bootstrap install parameters that are specific to a 
# board type. While calvados bootstrap in itself a platform 
# specific file, some of the board specific variants need to be
# determined during runtime.  This script serves that purpose. 
# Customization can be done either based on PD boardtype (used 
# during pxe boot where the only board identifier comes from 
# hardware) -OR- from PI card type (used during steady state
# install using install infra).
# It takes multiple parameters:
# parameter 1:  The original bootstrap config file (as input)
# parameter 2:  Use PD boardtype (1) or PI card type (2)
# parameter 3:  PD Board type (used during int/ext PXE boot)
# parameter 4:  PI card type (used during steady state install)
# parameter 5:  An optional parameter specifying the output file
#   to which the modified install params are written to.
#
# This script can be invoked from multiple places:
# pxe_install.sh script : During first time external/internal PXE 
#     boot procedure. 
# install agent code: During steady-state install, install agent
#     invokes this script to determine the install parameters of
#     the V2 calvados/XR/host partitions.
# spirit script: For backward compatibility reasons, for systems
#     upgrading for releases < 6.0, spirit script uses this 
#     script to build an o/p file that reflects the current 
#     setting of install parameters.
#

# spirit-functions host the dump_board_install_parameters_to_file fn.
. /etc/init.d/spirit-functions

# spirit-pd.sh hosts pd_board_specific_override fn.
. /etc/init.d/spirit_pd.sh

# Inputs to the script
# calvados_bootstrap.cfg file passed in as an argument
infile=$1

# specifies which parameter to use ($3(boardtype) or $4(cardtype))
board_or_card=$2

# boardtype
boardtype=$3

# PI cardtype
cardtype=$4

# file to which the modified install (board specific)
# install parameters are written to
outfile=$5


# basic validation
if [ ! -f "${infile}" ]; then
   echo "${infile} does not exist" 
   exit
fi

if [[ $board_or_card -ne 1  &&  $board_or_card -ne 2 ]]; then 
   echo "board_or_card parameter is invalid: $board_or_card"
   exit
fi

#-------get PD variables--------------
if [ -f $infile ]; then
    # calvados_bootstrap to setup most install specific variables
    . $infile
else
    return 1
fi

if [ $board_or_card -eq 1 ]; then
    # Platforms may choose to override the install parameters like disks, 
    # partition size and logical volume size based on runtime data read 
    # from board hardware. This PD function is called on individual board
    # from pxe_install.sh script. There are no PI card types at this stage.
    # hence the use of PD boardtype.
    declare -F pd_board_specific_override &>/dev/null &&                                \
        pd_board_specific_override $boardtype 
else 
    # during steady state install, install manager orchestrates the
    # install operation. Install manager has access only to PI card types at
    # that point. Hence it uses PI card type to customize install params
    # PI card type defines: /common/system-generic/system_common.proto

    declare -F pd_card_specific_override &>/dev/null &&                                 \
        pd_card_specific_override $cardtype
fi 

# Dump all install specific parameters retrieved from the calvados_bootstrap
# file (by sourcing spirit_pd.sh) & overridden by pd_board_specific_override 
# or pd_card_specific_override into another output file. 

if [ -n "$outfile" ]; then
    dump_board_install_parameters_to_file $outfile
fi


