#!/bin/bash
#
# Copyright (c) 2017-2018 by Cisco Systems, Inc.
# All rights reserved.
#

KERNEL_CFG_PATH=/var/run/cisco/kernel.boot.cfg
CFG_PATH=/var/run/cisco/ios.xr.cfg
XRV9K_CFG_PATH=/var/run/cisco/ios.xrv9k.cfg
DEBUG_LOGSPEC=/var/log/xr_sysctl.log

# In lines
MAX_LOG_SIZE=1000

#
# Provide (partial) functionality of update-rc.d to
# setup init.d symbolic links
#
# $1 == The script name (should exist in $1/etc/init.d/.)
# $2 == Priority.  This is the prefix applied in $1/etc/rc?.d/.
# ... == The run levels to which the script should be installed
#
function update_sys_v_start()
{
    local script_name="$1"
    local priority="$2"
    local run_levels=("${@:3}")

    if [ -e "/etc/init.d/${script_name}" ]; then
        for run_level in "${run_levels[@]}"; do
           mkdir -p "/etc/rc${run_level}.d"
           ln -sf "../init.d/${script_name}" \
              "/etc/rc${run_level}.d/${priority}${script_name}"
        done
    else
        echo "ERROR: ${FUNCNAME} failed for $*" >&2
    fi
}

#
# Check if the log file should be truncated
#
function truncate_if_necessary()
{
    mkdir -p "${DEBUG_LOGSPEC%/*}" || true
    if [[ -w "${DEBUG_LOGSPEC}" ]]; then
        local lines
        lines=$(wc -l < "${DEBUG_LOGSPEC}")
        if [[ ${lines} -ge ${MAX_LOG_SIZE} ]]; then
            # If sed fails, we're probably out of space.
            # In that case, reinitialize.
            sed -i -e "1,$((lines/2))d" "${DEBUG_LOGSPEC}" || {
                # Remove in case noclobber is set
                rm -f "${DEBUG_LOGSPEC}" ;
                echo "$(date) log reinitialized" > "${DEBUG_LOGSPEC}";
            } || true
        fi
    fi
}

#
# Log the contents of the given text file to a debug log
#
# $1 == The text filename to log
#
function log_file_contents()
{
    local src_spec="$1"
    local prefix="$(date) ${src_spec##*/}: "

    truncate_if_necessary
    if [[ -r "${src_spec}" ]]; then
        sed -e "s|^|${prefix//|/\\|}|" ${src_spec} >> "${DEBUG_LOGSPEC}" || true
    fi
}

#
# Log a message to the debug log
#
# $* == The message to log
#
function log_debug()
{
    truncate_if_necessary
    echo "$(date): $*" >> "${DEBUG_LOGSPEC}" || true
}

#
# Log an error to the debug log, and to logger
#
# $* == The message to log
#
function log_error()
{
    log_debug "ERROR: $*"
    logger -t xr_sysctl.sh "ERROR: $*" || true
}

if [[ ! -f /usr/bin/env ]]; then
    if [[ ! -f /bin/env ]]; then
        log_error "No /bin/env nor /usr/bin/env"
    else
        ln -sf /bin/env /usr/bin/env || true
    fi
fi

log_debug "INFO: $1"
case "$1" in
    reload|restart|start)
        rm -rf ${CFG_PATH} ${XRV9K_CFG_PATH} ${KERNEL_CFG_PATH}
        /usr/bin/xr_sysctl --system || \
            log_error 'xr_sysctl --system failed'
        unset ${!IOS_XR_*} XR_SYSCTL_RC
        log_file_contents /proc/cmdline
        log_file_contents ${CFG_PATH}
        log_file_contents ${XRV9K_CFG_PATH}
        touch ${CFG_PATH}.boot || true
        ;;
    status)
        log_file_contents /proc/cmdline
        log_file_contents ${CFG_PATH}
        log_file_contents ${XRV9K_CFG_PATH}
        q="'"
        if [[ "$(tr ${q}\\t\\r\\n '    ' < /proc/cmdline)" != "$(/usr/bin/xr_sysctl -n kernel.boot.cmdline)" ]]; then
            log_error 'Inconsistent cmdline settings! Resetting...'
            /usr/bin/xr_sysctl --system || true
            log_file_contents /proc/cmdline
            log_file_contents ${CFG_PATH}
            log_file_contents ${XRV9K_CFG_PATH}
        fi
        ;;
    stop)
        rm -rf ${CFG_PATH}.boot
        unset ${!IOS_XR_*} XR_SYSCTL_RC
        ;;
    install)
        update_sys_v_start xr_sysctl.sh S00 2 3 4 5

        #
        # For a SMU, initialize xr_sysctl
        #
        /etc/rc.d/init.d/xr_sysctl.sh start || true

        #
        # Cleanup from previous possible SMU
        #
        rm -f /lib/libxr_sysctl.so /usr/lib64/libxr_sysctl.so > /dev/null 2>&1 || true
        ;;
    *)
        echo "Usage: $0 {start|stop|restart|reload|status|install}" >&2
        exit 2
        ;;
esac
