#!/bin/bash
#
# calvados_launch_asr9k_pd.sh
#
# Copyright (c) 2014-2016, 2018 by Cisco Systems, Inc.
# All rights reserved.

BOARDTYPE=`cat /proc/cmdline | sed 's/^.*boardtype=//' | cut -d" " -f1`

VIRT_SMP=1

# Why this is here ? - This is a pd scecific check where we wait 
#                      on Multicast and Hp for UIO devices to be
#                      ready. All LXC platforms do not support this 
# Description: 
# Start the calvados once the uio dev proxy_hp and muilticast_irq
# have been create if uio are listed on platform, if not then we 
# cannot mknod the uio devices on the calvados/xr.
# 
# Platform Specific need : ASR9k
# - Not sure so will leave it implemented 
#   the default way it was implemented 
function cal_pd_lxc_start_device_ready {
    local wait_for_uio=0
    local uio_irq_multicast_exist=0
    local uio_proxy_hp_exists=0
    devlistconf=/etc/sysconfig/Devlist_calvados.conf
    # Check if /dev/uio* is listed in the devices, if not
    # don't need to wait for uio proxy_irq and multicast to be ready
    if [[ -f $devlistconf ]]; then
        while read line; do
            if strstr  "$line" "uio*"; then
                wait_for_uio=1
            fi
        done <  "$devlistconf"
    fi
    # If uio is listed, we need to wait for these devices to be created
    if [[ $wait_for_uio -eq 1 ]] ; then
        while true; do
            # Check all the uio devices to see if the proxy_hp and 
            # muliticast are already created.
            for uio_dev in /dev/uio*
            do
                # extract the device instance from the path
                uio_dev_inst=${uio_dev##*/}
                # Check if this is the milticast_irq device
                if [[ uio_irq_multicast_exist -eq 0 ]] ; then
                    cal_cmn_dev_exists \
                        "/sys/devices/platform/uio_multicast_pdriver.0/uio/$uio_dev_inst/name" \
                        "pci_proxy_irq_multicast" uio_irq_multicast_exist
                fi
                # Check if this is the milticast_irq device
                if [[ uio_proxy_hp_exists -eq 0 ]] ; then
                    cal_cmn_dev_exists  \
                        "/sys/devices/platform/uio_hp_pdriver.0/uio/$uio_dev_inst/name" \
                        "pci_proxy_hp" uio_proxy_hp_exists
                fi
            done
            if [[ uio_irq_multicast_exist -eq 1 && uio_proxy_hp_exists -eq 1 ]] ; then
                break;
            fi
            sleep 1
        log_message "Waiting for the uio devices to be ready \
                     $uio_irq_multicast_exist $uio_proxy_hp_exists"
        done
    fi
}

function cal_asr9k_build_os_kvm_xml {
cat << end_build_os_kvm_xml >> $OUT_XML_FILE
  <os>
    <type arch='x86_64' machine='pc-q35-1.5'>hvm</type>
    <boot dev='hd'/>
  </os>
end_build_os_kvm_xml
}

function dummyIP_for_qemu {
  #
  # Need to set an IPv4 address on the PF interface
  # before we spawn calvados, otherwise qemu fails to bind the port
  # to which we redirect the console.
  # We put the IP on an out-of-the-way VLAN subint to minimize any
  # potential confusion.
  #
  DUMMY_VLAN=4094 # 0xffe
  DUMMY_ADDR=169.254.147.201/16
  local pf_if="eth-pf1" #default case for LC

  if [ "$BOARDTYPE" == "RP" ]; then
      local lb_cnt=$(lspci -d :37cd | wc -l)
      if [[ "${lb_cnt}" != "0" ]]; then
          # RP3 or RSP5
          local fv_cnt=$(lspci -d :1581 | wc -l)
          local no_fv=$(grep no_fv /proc/cmdline | wc -l)
          if [[ "${fv_cnt}" == "0" ]] || [[ ${no_fv} != "0" ]]; then
              # F/V not present OR don't use F/V
              pf_if="eth5" 
          else
              # F/V present and used
              pf_if="eth1"
          fi
      else
          # not RP3 and not RSP5
          pf_if="eth1"
      fi    
  fi

  PF=$(ifconfig | grep ${pf_if} | grep -v '\.' | cut -f1 -d' ')
  if [ ! -z ${PF} ]; then
    if [ ! -f /proc/net/vlan/${PF}.${DUMMY_VLAN} ]; then
      vconfig   add ${PF} ${DUMMY_VLAN}
      ifconfig      ${PF}.${DUMMY_VLAN} ${DUMMY_ADDR} # Note: did not 'up' it.
    fi
  fi
}

function pci_args {
  # find the pci virtual devices to launch calvados with:
  lspci -nn | awk /$1/' {print "-device pci-assign,host="$1; exit}'
}

function calvados_dev_args {
  # Gather up the PCI IDs for Niantic and IPU
  VNIC_PCI_BDF=$(grep vnic1 /tmp/eth_bdf | awk '{print $2}' | cut -c 6-)
  VNIC_PCI_ARGS="-device pci-assign,host=$VNIC_PCI_BDF"
  get_board_type
  if [ "$BOARDTYPE" == "RP" ]; then
    MGMT_PCI_BDF=$(grep mgmt2 /tmp/eth_bdf | awk '{print $2}' | cut -c 6-)
    MGMT_PCI_ARGS="-device pci-assign,host=$MGMT_PCI_BDF"
  fi
  echo "$VNIC_PCI_ARGS \
        $MGMT_PCI_ARGS"
}

#build interface
function cal_asr9k_build_interface_xml {
cat << end_build_interface_xml >> $OUT_XML_FILE
    <interface type='ethernet'>
      <mac address='52:54:00:29:92:7e'/>
      <script path='/opt/cisco/hostos/bin/calvados_ifup.sh'/>
      <model type='virtio'/>
    </interface>
end_build_interface_xml
}

function cal_asr9k_build_udrv_hwid_xml {
    UDRV_HWID_DIR=/dev/udrv_hwid
    if [ ! -d $UDRV_HWID_DIR ]; then
        mkdir -p $UDRV_HWID_DIR
    fi  
cat << end_build_udrv_xml >> $OUT_XML_FILE
    <$VIRT_ARCH:arg value='-chardev'/>
    <$VIRT_ARCH:arg value='socket,id=client_$VMNAME,path=$UDRV_HWID_DIR/client_$VMNAME.sock,server,nowait'/>
    <$VIRT_ARCH:arg value='-device'/>
    <$VIRT_ARCH:arg value='virtserialport,id=vmdev_$VMNAME,chardev=client_$VMNAME,name=udrv_hwid_client'/>
end_build_udrv_xml
}

function cal_asr9k_build_dev_args_xml {
    
    dev_args=$(calvados_dev_args)

    #translate qemu command line to xml 
    OLDIFS="$IFS"
    IFS=' ' read -a array <<< $dev_args
    IFS="$OLDIFS"
    index=0
    while [[ $index -lt ${#array[@]} ]];
    do
        #echo "$index ${array[$index]}"
        OPTION=${array[$index]}
        ((index = index + 1))
        OPTION_ARG=${array[$index]}
        ((index = index + 1))
        echo "$OPTION" 
        echo "$OPTION_ARG"
        cal_cmn_build_arg_xml
    done
}

function cal_asr9k_build_misc_xml {
cat << end_build_misc_xml >> $OUT_XML_FILE
    <$VIRT_ARCH:arg value='-rtc'/>
    <$VIRT_ARCH:arg value='hostsync=on'/>
    <$VIRT_ARCH:arg value='-readconfig'/>
    <$VIRT_ARCH:arg value='/usr/share/qemu-153/asr9k-q35-chipset.cfg'/>
    <$VIRT_ARCH:arg value='-device'/>
    <$VIRT_ARCH:arg value='hwid,debug=1'/>
end_build_misc_xml

}

# Passthrough eusb partition
function cal_asr9k_build_disk_pt_xml {
    eusb_part2=$(readlink /dev/eusb2)
    if [[ $? == 0 && -b /dev/$eusb_part2 ]]; then
        DRIVE_SRC_DEV="/dev/$eusb_part2"
        DRIVE_TGT_DEV="vde"
        ALIAS_NAME="virtio-disk5"
        cal_cmn_build_disk_xml
    fi

    eusb_part3=$(readlink /dev/eusb3)
    if [[ $? == 0 && -b /dev/$eusb_part3 ]]; then
        DRIVE_SRC_DEV="/dev/$eusb_part3"
        DRIVE_TGT_DEV="vdf"
        ALIAS_NAME="virtio-disk6"
        cal_cmn_build_disk_xml
    fi
}

function cal_asr9k_canbus_ttyS_xml {
    # Serial 0,1 are used by virtconsole
    local char_ser_dev=2
    local ttyS_list=("$@")

    for i in "${ttyS_list[@]}"
        do

cat  << end_build_canbus_ttyS_xml >> $OUT_XML_FILE
    <$VIRT_ARCH:arg value='-chardev'/>
    <$VIRT_ARCH:arg value='tty,id=charserial${char_ser_dev},path=/dev/ttyS${i},raw'/>
    <$VIRT_ARCH:arg value='-device'/>
    <$VIRT_ARCH:arg value='isa-serial,chardev=charserial${char_ser_dev},id=serial${char_ser_dev}'/>
end_build_canbus_ttyS_xml

            let "char_ser_dev++"
        done
}

#create the config file in xml format 
function cal_asr9k_build_xml {
    cal_cmn_build_virt_base_cfg_xml #build the basic piece for virtualization 
    
    cal_asr9k_build_os_kvm_xml
    cal_cmn_build_devices_start_xml #start device tag

    VIRT_EMULATOR="/usr/bin/kvm-qemu-system-x86_64"
    cal_cmn_build_emulator_xml

    cal_cmn_build_block_devices_xml #build disk device

    CONSOLE_IPADDR="127.0.0.1" #get it from config file
    cal_asr9k_build_interface_xml

    # Setup eUSB passthrough
    if [ "$BOARDTYPE" != "LC" ]; then
        cal_asr9k_build_disk_pt_xml
    fi

    cal_cmn_build_console_access_xml
    cal_cmn_build_devices_end_xml

    cal_cmn_build_cmdline_start_xml
    cal_cmn_build_common_args_xml

    if [ "$BOARDTYPE" != "LC" ]; then
         sed -i -e "s;socket,id=vserial0,host=$CONSOLE_IPADDR,port=$HVC0PORT,telnet,server,nowait;tty,id=vserial0,path=/dev/ttyS1;" $OUT_XML_FILE
    fi

    if [ "$BOARDTYPE" == "LC" ]; then
        board_type_ttyS=("1")
    else
        board_type_ttyS=("2" "3")
    fi
        
    cal_asr9k_canbus_ttyS_xml "${board_type_ttyS[@]}"

    cal_asr9k_build_udrv_hwid_xml
    cal_asr9k_build_dev_args_xml
    cal_asr9k_build_misc_xml

    cal_cmn_build_cmdline_end_xml
    cal_cmn_build_domain_end_xml
}

function pd_get_disk_size {
    local dev=${1}
    max=$(fdisk -l ${dev} 2>/dev/null | grep "Disk /dev"  | cut -d ',' -f2)
    max=${max%bytes}
    let PD_DISK_SIZE=max/1000000
}
 
extend_calvados_misc_disk1 () {
  # 
  # Extend the calvados /misc/disc1 partition by 5% of total disk size
  #   
  if [[ "$BOARDTYPE" != "RP" ]]; then
      return
  fi
  
  pd_get_disk_size /dev/sdb
  if [[ -z $PD_DISK_SIZE ]]; then
      kmsg_log "Failed to get disk size. exiting.."
      return
  fi
  local lv_extend_size=$(($PD_DISK_SIZE*5/100))

  local vgfree=$(vgdisplay pci_disk1 --units M | grep "Free  PE" | awk '{print $7}' | awk -F'.' '{print $1}')
  local lvsize=$(lvdisplay /dev/pci_disk1/ssd_disk1_calvados_1  | grep "LV Size"  | awk '{print $3}' | awk -F'.' '{print $1}')
  local lvsize_g=$(lvdisplay /dev/pci_disk1/ssd_disk1_calvados_1  | grep "LV Size"  | awk '{print $3}')
  
  if [[ ${lvsize} -le 7 && ${vgfree} -gt ${lv_extend_size} ]]; then
      kmsg_log "Extending partition size from " $lvsize_g"G"
      e2fsck -f -p  /dev/pci_disk1/ssd_disk1_calvados_1 >/dev/null 2>&1
      rc=$?
      # e2fsck return code 
      #    0 - No errors
      #    1 - File system errors corrected
      if [[ $rc -gt 1 ]]; then
          kmsg_log "e2fsck returned " $rc
      fi
      lvextend -r -L+${lv_extend_size}M /dev/pci_disk1/ssd_disk1_calvados_1 >/dev/null 2>&1
      rc=$?
      if [[ $rc -ne 0 ]]; then
          kmsg_log "lvextend failed " $rc
      fi
      e2fsck -f -p  /dev/pci_disk1/ssd_disk1_calvados_1 >/dev/null 2>&1
      rc=$?
      if [[ $rc -gt 1 ]]; then
          kmsg_log "e2fsck returned " $rc
      fi
  else 
      kmsg_log "Retaining disk1 size at " $lvsize_g
  fi
}
 
function calvados_start_pd {
    readonly CALVADOS_COMMON="/opt/cisco/hostos/bin/calvados_launch_common.sh"
    if [ -f $CALVADOS_COMMON ]; then
        . $CALVADOS_COMMON
    fi
 
    if [ "$VIRT_METHOD" == "lxc" ]; then
        VIRT_ARCH="lxc"
    else 
        VIRT_ARCH="qemu"
    fi

    OUT_XML_FILE="/misc/config/${VMNAME}-${VIRT_ARCH}-${PLATFORM}-${SIMULATION}.xml" 
    rm $OUT_XML_FILE

    get_board_type

    if [ "$BOARDTYPE" == "RP" ]; then
        VIRT_SMP=$VIRT_RP_SMP
        VIRT_MEM=$VIRT_RP_MEM
    elif [ "$BOARDTYPE" == "LC" ]; then
        VIRT_SMP=$VIRT_LC_SMP
        VIRT_MEM=$VIRT_LC_MEM
    fi

    touch /tmp/niantic_devices

    if [ "$VIRT_ARCH" == "lxc" ]; then
       cal_cmn_build_lxc_xml no_eobc
    else
       # Need some interface up, otherwise QEMU complains
       dummyIP_for_qemu
       #create the config file in xml format
       cal_asr9k_build_xml
    fi
    
    # Extend the calvados partition size 
    extend_calvados_misc_disk1

    # start the guest
    cal_cmn_virsh_start
}

function calvados_stop_pd {
    cal_cmn_virsh_stop
}
