#!/bin/bash
# -------------------------------------------------------------------------------
# show_tech_host
#
# April 2015, Diana Chris
#
# Copyright (c) 2015-2016 by cisco Systems, Inc.
# All rights reserverd.
# -------------------------------------------------------------------------------

# This script collects logs from the hostOS 
#

# ---------------------
#Logs to collect:
# 1.  /var/log/ on host:
#     a.  host-install0.log
#     b.  host-install.log
#     c.  sysadmin-install.log
#     d.  dmesg
#     e.  ctrace - hushd
#     f.  libvirt/lxc/ and libvirt/qemu
#     g.  auth.log
#     h.  syslog.log
#     i.  messages
#     j.  cron.log
#     k. lxc_hook_script.log
#     l. debug logs dumped in /misc/scratch 
#     m. pstore logs
# 2.  /proc/mounts
# 3.  Active libvirt lxc: virsh -c lxc:/// list --all
# 4.  Domain xmls: virsh -c lxc:/// dumpxml <domain>
# 5.  Loopback devices
# 6.  n/w stats: /proc/net/dev
# 7.  intf stats: ifconfig
# 8.  brctl info
# 9.  disk utilization: df 
# 10. mem utilization: free -m
# 11. Cgroup mem info

# ---------------------

standalone=0

while [ "$#" -gt "0" ]; do
    case "$1" in
        -t) cardtype="$2"; shift 2;;
        -f) media_path="$2"; shift 2;;
        -n) nodename="$2"; shift 2;;
        -c) clean="1"; shift 1;;
         *) shift;;
    esac
done

if [ -z "$media_path" ]; then
    media_path="/misc/scratch/showtech"
    standalone=1
fi

if [ ! -z "$nodename" ]; then
    nodename_plus_host="${nodename}_host"
else
    nodename_plus_host="host"
fi    

#All logs etc will be accumulated into $fullpath
fullpath=$media_path/$nodename_plus_host

if [[ "$clean" == "1" ]]; then
     rm ${fullpath}.tgz
     rmdir $media_path; RCX=$?
     exit 0
fi

mkdir -p $fullpath ; RC=$?
if [ $RC -ne 0 ]; then
    exit 0
fi

FILENAME="$fullpath/host_showtech_log"

# Bootstrap file
BOOTSTRAP_FILE="/etc/init.d/calvados_bootstrap.cfg"

HUSHD_LOG="$fullpath/hushd.log"
OUTPUT_TAR="$fullpath"

if [[ -f "$BOOTSTRAP_FILE" ]]; then
    source $BOOTSTRAP_FILE
fi    

# ---------------------
# General Functions
# ---------------------

function print_main_heading {
    echo "" >> $FILENAME
    echo "-------------------------------------------------------------------------------" >> $FILENAME
    echo "                          tech support Host                                  " >> $FILENAME
    echo "-------------------------------------------------------------------------------" >> $FILENAME
    echo "" >> $FILENAME
}

function print_end {
    echo "" >> $FILENAME
    echo "-------------------------------------------------------------------------------" >> $FILENAME
    echo "                          tech support Host End                                 " >> $FILENAME
    echo "-------------------------------------------------------------------------------" >> $FILENAME
    echo "" >> $FILENAME
}


function print_heading() {
    echo "" >> $FILENAME
    echo "++++++++++ $@ ++++++++++" >> $FILENAME
    echo "" >> $FILENAME
}


function print_sub_heading() {
    echo "" >> $FILENAME
    echo "---------- $@ ----------" >> $FILENAME
    echo "" >> $FILENAME
}

function print_tail {
    echo "" >> $FILENAME
    echo "------------------------" >> $FILENAME
    echo "" >> $FILENAME
}

function print_sub_tail {
    echo "" >> $FILENAME
}

function print_file() {
    local file=$1
    if [[ -f "$file" ]]; then
        print_sub_heading "$file"
        cat $file >> $FILENAME
        print_sub_tail
    fi
}

function exec_cmd() {
    local cmd=$@
    print_sub_heading "$cmd"
    $cmd 2> /dev/null >> $FILENAME
    print_sub_tail
}

# ---------------------

# ---------------------
# Functions
# ---------------------

function collect_host_install_logs {
    
    ln -s "/var/log/host-install0.log" "$fullpath/host-install0.log" 2>/dev/null
    ln -s "/var/log/host-install.log" "$fullpath/host-install.log" 2>/dev/null
}

function collect_sysadmin_install_logs {
    ln -s "/var/log/sysadmin-install.log" "$fullpath/sysadmin-install.log" 2>/dev/null
}

function collect_dmesg_log {
    ln -s  "/var/log/dmesg" "$fullpath/dmesg" 2>/dev/null
}

function collect_hushd_ctrace {
    if [[ -f "$HUSHD_LOG" ]]; then
        mv "$HUSHD_LOG" "$HUSHD_LOG.old"
    fi
    /opt/cisco/hostos/bin/ctracedec -Agtkdp hushd 2>/dev/null > $HUSHD_LOG
}

function collect_libvirt_logs {
    if [[ "$VIRT_METHOD" == "lxc" ]]; then
        dir="/var/log/libvirt/lxc/"
    else
        dir="/var/log/libvirt/qemu/"
    fi    
    N=`ls $dir 2>/dev/null | wc -l`
    if [ $N -ne 0 ]; then
        for name in $dir/*
        do
            targname=`basename $name`
            ln -s $name $fullpath/$targname 2>/dev/null
        done
    fi
}

function collect_auth_log {
    ln -s "/var/log/auth.log" "$fullpath/auth.log" 2>/dev/null
}

function collect_syslog {
    ln -s "/var/log/syslog" "$fullpath/syslog" 2>/dev/null
}

function collect_syslog_log {
    ln -s "/var/log/syslog.log" "$fullpath/syslog.log" 2>/dev/null
}

function collect_kern_log {
    ln -s "/var/log/kern.log" "$fullpath/kern.log" 2>/dev/null
}

function collect_messages_log {
    ln -s "/var/log/messages" "$fullpath/messages" 2>/dev/null
}

function collect_cron_log {
    ln -s "/var/log/cron.log" "$fullpath/cron.log" 2>/dev/null
}

function collect_lxc_hook_script_log {
    if [[ "$VIRT_METHOD" = "lxc" ]]; then
        ln -s "/var/log/lxc_hook_script.log" "$fullpath/lxc_hook_script.log" 2>/dev/null
    fi    
}

function collect_debug_logs {
    N=`ls -1d /misc/scratch/sysadmin* 2>/dev/null 1| wc -l`
    if [ $N -ne 0 ]; then
        for name in /misc/scratch/sysadmin*
        do
            targname=`basename $name`
            ln -s $name $fullpath/$targname 2>/dev/null
        done
    fi

    N=`ls -1d /misc/scratch/default-sdr* 2>/dev/null 1| wc -l`
    if [ $N -ne 0 ]; then
        for name in /misc/scratch/default-sdr*
        do
            targname=`basename $name`
            ln -s $name $fullpath/$targname 2>/dev/null 
        done
    fi
}

function collect_pstore_logs {
    if [ -d "/mnt/pstore" ]; then
        ln -s  "/mnt/pstore" "$fullpath/pstore" 2>/dev/null
    fi
}

function print_kernel_info {
    print_heading "Kernel Version: uname -a"
    uname -a 2>/dev/null >> $FILENAME
    print_tail
    print_heading "Kernel cmdline: /proc/cmdline"
    cat /proc/cmdline 2>/dev/null >> $FILENAME
    print_tail
    print_heading "Ulimits: ulimit -a"
    ulimit -a 2>/dev/null >> $FILENAME
    print_tail
    uname=$(uname -r)
    print_heading "Kernel flags: /boot/config-$uname"
    cat "/boot/config-$uname" 2>/dev/null >> $FILENAME
    print_tail
    print_heading "Kernel Settings: sysctl -a"
    sysctl -a 2>/dev/null >> $FILENAME
    print_tail
}

function print_pci_info {
    print_heading "PCI info: lspci"
    lspci 2>/dev/null >> $FILENAME
    print_tail
}

function print_mounts {
    print_heading "Active mounts"
    print_file "/proc/mounts"
    print_tail
}

function print_active_vms {
    local root="/var/run/libvirt/"
    if [[ "$VIRT_METHOD" = "lxc" ]]; then
        print_heading "Active/Defined LXCs"
    else
        print_heading "Active/Defined VMs"
    fi
    echo "Domain: PID" >> $FILENAME
    for file in $(find "$root" -name '*.pid' -print 2>/dev/null)
    do
        pid=`cat "$file"`
        kill -0 $pid 2>/dev/null
        if [[ $? = "0" ]]; then
            domain=$(basename $file | cut -d"." -f1 2>/dev/null)
            echo "$domain: $pid" >> $FILENAME
        fi    
    done    
    print_tail
}

function print_libvirt_xml {

    local xml_root
    local mod_xml_root
    if [[ "$VIRT_METHOD" = "lxc" ]]; then
        xml_root="/etc/libvirt/lxc"
        mod_xml_root="/var/run/libvirt/lxc"
    else
        xml_root="/etc/libvirt/qemu"
        mod_xml_root="/var/run/libvirt/qemu"
    fi    
    print_heading "Persistent Libvirt XMLs"
    for file in $(find "$xml_root" -name '*.xml' -print 2>/dev/null)
    do
        print_file "$file"
    done
    print_tail
    print_heading "Libvirt modified XMLs"
    for file in $(find "$mod_xml_root" -name '*.xml' -print 2>/dev/null)
    do
        print_file "$file"
    done
    print_tail
}

function print_loop_devices {
    print_heading "Loop devices"
    ps aux | grep "\[loop" | grep -v grep >> $FILENAME
    exec_cmd "losetup -a"
    print_tail
}

function print_network_stats {
    print_heading "Network statistics"
    print_file "/proc/net/dev"
    print_tail
}

function print_intf_status {
    print_heading "Interface status"
    exec_cmd "ip addr show"
    print_tail
}

function print_iptables {
    print_heading "IP table"
    exec_cmd "route"
    print_tail
}

function print_arp_cache {
    print_heading "ARP Cache"
    exec_cmd "arp -n"
    print_tail
}

function print_brctl_info {
    print_heading "Host Bridge information"
    exec_cmd "brctl show"
    print_tail
}

function print_disk_utilization {
    print_heading "Host disk utilzation"
    exec_cmd "df"
    print_tail
if [ "$cardtype" == "RP" -o "$cardtype" == "LC" ]; then
    print_heading "Logical Volumes"
    exec_cmd "lvscan -a"
    print_tail
    print_heading "Attributes of Logical Volumes"
    exec_cmd "lvdisplay"
    print_tail
    print_heading "Physical Volumes"
    exec_cmd "pvscan"
    print_tail
    print_heading "Lists partition layout on all block devices"
    exec_cmd "parted -l"
    print_tail
fi
}

function print_mem_utilization {
    print_heading "Host memory utilization"
    print_file "/proc/meminfo"
    print_tail
    print_heading "Virtual memory statistics"
    exec_cmd "vmstat"
    print_tail
}

function print_cgroup_memory_info {
   local root="/dev/cgroup/memory"
   
   print_heading "Cgroup Memory information"
   if [[ ! -d "$root" ]]; then
       msg="Cgroup memory information not available on this platform"
       echo "$msg" >> $FILENAME
       return
   fi    
   dir_list=$(find $root -name "*" -type d -print 2>/dev/null)
   for dir in $dir_list
   do
       print_heading $dir
       print_sub_heading "Memory limit in bytes"
       print_file "$dir/memory.limit_in_bytes"
       print_sub_tail
       print_sub_heading "Memory usage in bytes"
       print_file "$dir/memory.usage_in_bytes"
       print_sub_tail
       print_sub_heading "Memory MAX usage in bytes"
       print_file "$dir/memory.max_usage_in_bytes"
       print_sub_tail
       print_sub_heading "Memory statistics"
       print_file "$dir/memory.stat"
       print_sub_tail
       print_tail
   done
}

function print_processes {
    print_heading "Processes"
    ps -ef 2>/dev/null >> $FILENAME
    print_tail
    print_heading "Top Processes"
    top -b -n 1 2>/dev/null >> $FILENAME
    print_tail
}

function main() {

    # validate filename option
    if [[ -f "$FILENAME" ]]; then
        mv "$FILENAME" "$FILENAME.old"
    fi
    touch $FILENAME

    # create empty archive file
    if [ $standalone -eq 1 ]; then
        timestamp=`date "+%Y-%b-%d-%H%M%S-%Z"`
        OUTPUT_TAR=$OUTPUT_TAR-$timestamp
        echo "HOST logs start time: `date`"
    fi

    # Collect logs
    print_main_heading

    #    Kernel Info
    print_kernel_info

    #    PCI info
    print_pci_info
    
    #2.  /proc/mounts
    print_mounts

    #3.  Active libvirt lxc: virsh -c lxc:/// list --all
    print_active_vms

    #4.  Domain xmls: virsh -c lxc:/// dumpxml <domain>
    print_libvirt_xml

    #5.  Loopback devices
    print_loop_devices

    #6.  n/w stats: /proc/net/dev
    print_network_stats

    #7.  intf stats: ifconfig
    print_intf_status

    #    ip_tables
    print_iptables

    #    arp cache
    print_arp_cache

    #8.  brctl info
    print_brctl_info

    #9.  disk utilization: df
    print_disk_utilization

    #10. mem utilization: free -m
    print_mem_utilization

    #11. Cgroup mem info
    print_cgroup_memory_info

    #1. collect logs from /var/log/ directory 
    #a.  host-install0.log
    #b.  host-install.log
    collect_host_install_logs

    #c.  sysadmin-install.log
    collect_sysadmin_install_logs

    #d.  dmesg 
    collect_dmesg_log

    #e.  ctrace - hushd
    collect_hushd_ctrace

    #f.  libvirt/lxc/default-sdr--1.log and sysadmin.log
    collect_libvirt_logs 
    
    #g.  auth.log
    collect_auth_log

    #h.  syslog.log
    collect_syslog_log

    #i.  messages
    collect_messages_log

    #j.  cron.log
    collect_cron_log

    #k.  lxc_hook_script.log
    collect_lxc_hook_script_log

    #l.  collect logs dumped by debug script
    collect_debug_logs

    #m.  collect logs from persistent storage (dmesg, ...)
    collect_pstore_logs

    #n. Collect syslog
    collect_syslog

    #o. collect kern.log
    collect_kern_log

    #    ps -ef and top processes
    print_processes

    print_end

    ( cd $media_path
      tar --dereference -czf $nodename_plus_host.tgz $nodename_plus_host 1>/dev/null 2>&1
    )

   if [ $standalone -eq 1 ]; then
       echo "Host logs: $OUTPUT_TAR.tgz"
   else
       echo "$OUTPUT_TAR.tgz"
   fi
   rm -rf "$fullpath"

}

# Execute the script
main 
