#!/usr/bin/perl

# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
# script to collect ltrace memory usage on XR (for Spirit/eXR platforms)
#
# Copyright (c) 2017-2019 by Cisco Systems, Inc.
# All rights reserved.
# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx

use strict;
use warnings;
use File::Basename;
use File::Copy;
use Getopt::Std;
use vars qw/ %opt /;

use lib qw ( /pkg/opt/cisco/pam/ /opt/cisco/calvados/pam/ /opt/pam/ );
use pam_ltrace;

getopts( "p:AWXh", \%opt );

sub usage {
    print STDERR "\n", "~" x 80, "\n";
    print STDERR "\tUsage:\n";
    print STDERR "\t$0 - will print overall ltrace usage summary (default).\n";
    print STDERR "\t$0 -p <process name>, will print ltrace usage for the process.\n";
    print STDERR "\t$0 -A to print ltrace, shmwin, and xos/xdt\n";
    print STDERR "\t$0 -W to sort by shmwin (used with -A together), default sort by ltace\n";
    print STDERR "\t$0 -X to sort by XOS/XDT (used with -A together), default sort by ltace\n";
    print STDERR "\t -W and -X are mutually exclusive\n";
    print STDERR "\te.g.,\n";
    print STDERR "\t$0 -p emsd\n";
    print STDERR "\t -h - will print this messsage.\n";
    print STDERR "~" x 80, "\n";
    exit(0);
}

sub get_ltrace_breakdowns($) {
    my $file = shift;
    my $info;
    $info->{total_size} = 0;
    $info->{total_rss} = 0;
    $info->{total_pss} = 0;
    if (!open(FD, $file)) {
        return $info;
    }
    my $total_size = 0;
    my $total_rss = 0;
    my $total_pss = 0;

    my $sub_total_size = 0;
    my $sub_total_rss = 0;
    my $sub_total_pss = 0;

    my $is_ltrace = 0;
    local $/ = 'VmFlags:';
    my $function = "";
    my @functionList = ();
    my %isKnownFunc;
    my $val = $1;
    while(my $buf=<FD>) {
        if ($buf =~ /\s+\/dev\/.*\/ltrace/) {
            $is_ltrace = 1;
            foreach my $line (split(/\n/,$buf)) {
                if ($line =~ /\s+\/dev\/.*\/(ltrace.*)/) {
                    my $new_function = $1;
                    if (!$isKnownFunc{$new_function}) {
                        $isKnownFunc{$new_function} = 1;
                        push @functionList, $new_function;
                        if ($function ne "") {
                            $info->{$function}->{sub_total_size} = 0.0;
                            if ((defined($sub_total_size)) && ($sub_total_size ne "")) {
                                $info->{$function}->{sub_total_size} = $sub_total_size;
                            }
                            $info->{$function}->{sub_total_rss} = 0;
                            if ((defined($sub_total_rss)) && ($sub_total_rss ne "")) {
                                $info->{$function}->{sub_total_rss} = $sub_total_rss;
                            }
                            $info->{$function}->{sub_total_pss} = 0.0;
                            if ((defined($sub_total_pss)) && ($sub_total_pss ne "")) {
                                $info->{$function}->{sub_total_pss} = $sub_total_pss;
                            }
                        }
                        $function = $new_function;
                        $sub_total_size = 0;
                        $info->{$function}->{sub_total_size} = 0;
                        $sub_total_rss = 0;
                        $info->{$function}->{sub_total_rss} = 0;
                        $sub_total_pss = 0;
                        $info->{$function}->{sub_total_pss} = 0;
                    } else {
                        $function = $new_function;
                        $sub_total_size = $info->{$function}->{sub_total_size};
                        $sub_total_rss = $info->{$function}->{sub_total_rss};
                        $sub_total_pss = $info->{$function}->{sub_total_pss};
                    }
                    next;
                } elsif ($line =~ /^Size:\s+(\d+)\s+kB/i) {
                    $val = $1;
                    $total_size += $val;
                    $sub_total_size += $val;
                } elsif ($line =~ /^Rss:\s+(\d+)\s+kB/i) {
                    $val = $1;
                    $total_rss += $val;
                    $sub_total_rss += $val;
                } elsif ($line =~ /^Pss:\s+(\d+)\s+kB/i) {
                    $val = $1;
                    $total_pss += $val;
                    $sub_total_pss += $val;
                    last;
                }
            } ;# foreach my $line (split(/\n/,$buf))
        } elsif ($is_ltrace) {
            if (0) {
            if ($buf =~ /\s+00000000\s+00:00\s+0(?!\w)/) {
                foreach my $line (split(/\n/,$buf)) {
                    if ($line =~ /^Size:\s+(\d+)\s+kB/i) {
                        $val = $1;
                        $total_size += $val;
                        $sub_total_size += $val;
                    } elsif ($line =~ /^Rss:\s+(\d+)\s+kB/i) {
                        $val = $1;
                        $total_rss += $val;
                        $sub_total_rss += $val;
                    } elsif ($line =~ /^Pss:\s+(\d+)\s+kB/i) {
                        $val = $1;
                        $total_pss += $val;
                        $sub_total_pss += $val;
                        last;
                    }
                }
            } else {
                $is_ltrace = 0;
            }
            } ;# if (0)
        }
    }
    close(FD);
    undef $/;

    $info->{$function}->{sub_total_size} = $sub_total_size;
    $info->{$function}->{sub_total_rss} = $sub_total_rss;
    $info->{$function}->{sub_total_pss} = $sub_total_pss;

    $info->{total_size} = $total_size;
    $info->{total_rss} = $total_rss;
    $info->{total_pss} = $total_pss;

    $info->{functionList} = \@functionList;
    return $info;
} ;# sub get_ltrace_breakdowns($)

if (defined($opt{h})) {
    usage();
}
if ((defined($opt{W})) && (defined($opt{X}))) {
    usage();
}
if ((defined($opt{A}) && (!defined($opt{W}) && !defined($opt{X}))) ||
    (!defined($opt{A}) && (defined($opt{W}) || defined($opt{X})))) {
    usage();
}
if (defined($opt{p}) && ($opt{p} =~ /\w+/)) {
    my $_pids = `pidof $opt{p}`;
    $_pids =~ s/[\r\n]//g;
    my @pids = split(/\s+/, $_pids);
    if (!scalar(@pids)) {
        print "Cannot find process PID for '$opt{p}' (make sure the process is running?).\n";
    }
    foreach my $pid (@pids) {
        if ((! -f "/proc/$pid/exe") ||
            (! -f "/proc/$pid/smaps") ||
            (! -f "/proc/$pid/cmdline")) {
            #print "Cannot find process PID for '$pid' (make sure the process is running?).\n";
            next;
        } else {
            my $smape_file = "/proc/$pid/smaps";
            my $ltrace_info = get_ltrace_breakdowns($smape_file);
            my @tmp =();
            foreach my $func (@{$ltrace_info->{functionList}}) {
                my $_tmp = "$func,$ltrace_info->{$func}->{sub_total_size}";
                $_tmp .= ",$ltrace_info->{$func}->{sub_total_rss}";
                push @tmp, $_tmp;
            }
            my @sorted = sort { (split /,/, $b)[2] <=> (split /,/, $a)[2] } @tmp;

            print "==>ltrace breakdown (by function) for $opt{p} (pid=$pid)<==\n";
            my $msg = <<END;
------------------------------------------------- ----------
END
            $msg .= sprintf("%-50s", "Function");
            $msg .= sprintf("%-10s", "Usage (kB)");
            $msg .= "\n";
            $msg .= <<END;
------------------------------------------------- ----------
END
            print $msg;
            foreach my $_tmp (@sorted) {
                my ($func, $size, $rss) = split(/,/, $_tmp);
                $func =~ s/.*ltrace\///;
                $msg = sprintf("%-49s", $func);
                $msg .= sprintf("%11.1f", $rss);
                $msg .= "\n";
                print $msg;
            }
        }
    }
} else {
    my $info = collect_local_ltrace_memory();
    if (!$opt{A}) {
        my @tmp = @{$info->{ltraces}};
        my @sorted = sort { (split /,/, $b)[2] <=> (split /,/, $a)[2] } @tmp;
        my $grand_total_rss = 0;
        my $msg = <<END;
--------------------------------- -------- -----------
Process                           PID      Ltrace (kB)
--------------------------------- -------- -----------
END
        print $msg;
        foreach my $_tmp (@sorted) {
            my ($process, $pid, $lt_rss, $shmwin_rss, $xdt_rss) = split(/,/, $_tmp);
            $msg = sprintf("%-34s", $process);
            $msg .= sprintf("%-8s", $pid);
            $msg .= sprintf("%12.1f", $lt_rss);
            $msg .= "\n";
            print $msg;
            $grand_total_rss += $lt_rss;
        }
        $msg = sprintf("%-34s", "Grand total");
        $msg .= sprintf("%-8s", "");
        $msg .= sprintf("%12.1f", $grand_total_rss);
        $msg .= "\n";
        print $msg;
    } else {
        my @tmp = @{$info->{ltraces}};
        my $idx = 2;
        if ($opt{W}) {
            $idx = 3;
        } elsif ($opt{X}) {
            $idx = 4;
        }
        my @sorted = sort { (split /,/, $b)[$idx] <=> (split /,/, $a)[$idx] } @tmp;
        my $ltrace_total_size = 0;
        my $shmwin_total_rss = 0;
        my $xdt_total_rss = 0;
        my $msg = <<END;
--------------------------------- -------- ----------- ----------- -----------
Process                           PID      Ltrace (kB) Shmwin (kB)    XOS (kB)
--------------------------------- -------- ----------- ----------- -----------
END
        print $msg;
        foreach my $_tmp (@sorted) {
            my ($process, $pid, $lt_rss, $shmwin_rss, $xdt_rss) = split(/,/, $_tmp);
            $msg = sprintf("%-34s", $process);
            $msg .= sprintf("%-8s", $pid);
            $msg .= sprintf("%12.1f%12.1f%12.1f", $lt_rss,$shmwin_rss,$xdt_rss);
            $msg .= "\n";
            print $msg;
            $ltrace_total_size += $lt_rss;
            $shmwin_total_rss += $shmwin_rss;
            $xdt_total_rss += $xdt_rss;
        }
        $msg = sprintf("%-34s", "Grand total");
        $msg .= sprintf("%-8s", "");
        $msg .= sprintf("%12.1f%12.1f%12.1f", $ltrace_total_size, $shmwin_total_rss, $xdt_total_rss);
        $msg .= "\n";
        print $msg;
    }
}
