#!/usr/bin/python
#-----------------------------------------------------------------------
# install-functions.py - Generic place holder to put pythonic functions 
#                        related to Install
# 
# This script is independent of install_functions.sh file. Initially, 
# there is no intersection of functions with the functions from file
# install_functions.sh
# 
# In this script, Install class is the main class, whose functions will 
# be invoked from outside of this script (i.e. mainly from C code). 
#
# Copyright (c) 2017-2019 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------------------------
#


#######################################################
#
# <<<<<<<<<<<<<<<<<<   PRECAUTION   >>>>>>>>>>>>>>>>>>
#                      ----------
#
# Please do not put any 'print' statement if value is 
# not intended to return to the caller of this script
#
# Instead use logger object for printing the debug 
# statements, which can be found in file
# /var/log/install/install_functions_py.log
#
#######################################################


from __future__ import print_function, with_statement
import sys
import os
import re
from subprocess import Popen, PIPE
import logging
import logging.handlers
import datetime as dt
import hashlib
from urlparse import urlparse

RPM_BIN='rpm'
PLATFORM_STR1='PLATFORM_EXT'
PLATFORM_STR2='PLATFORM'
BOOTSTRAP_FILE='/etc/rc.d/init.d/calvados_bootstrap.cfg'
CMD_FILE='/proc/cmdline'
LOG_FILE = '/var/log/install/install_functions_py.log'



def runcmd(cmd, success_ret_values = [0]):
  '''
    Runs cmd command in shell.

    It will print cmd as well as its output to logger.
    If some error occurs then output contains error.
  '''  

  sprc = 0
  process = Popen(cmd, stdout=PIPE, stderr=PIPE, shell=True)
  out, error = process.communicate()
  sprc = process.returncode

  if len(out) > 1 and out[-1] == '\n':
    out = out[:-1]
    
  #for commands like grep, return value will be non-zero even in no-error case
  if sprc == None or sprc not in success_ret_values:
      logger.error('Error in executing CMD="%s" , output="%s", ' \
                   'error="%s"' %(cmd, out, error))
      sys.exit(1)
    
  logger.debug('cmd="%s" output="%s"'%(cmd,out))
  return dict(rc=sprc, output=out)


def get_file_md5(file_name):
  '''
    Get md5 checksum of a file
    
  '''
  
  hash_md5 = hashlib.md5()
  try:
    with open(file_name, 'rb') as file:
      for chunk in iter(lambda: file.read(4096), b''):
        hash_md5.update(chunk)
  except EnvironmentError:
    return ''
  
  return hash_md5.hexdigest()


class MyFormatter(logging.Formatter):
  '''Class to format timestamp in logger to include miliseconds'''
     
  converter=dt.datetime.fromtimestamp

  def formatTime(self, record, datefmt=None):
    ct = self.converter(record.created)
    if datefmt:
      s = ct.strftime(datefmt)
    else:
      t = ct.strftime('%Y-%m-%d %H:%M:%S')
      s = '%s,%03d' % (t, record.msecs)
    return s


class Install:
  '''
    Class containing all pythonic general purpose functions for install
    
     
    Common agruments for most of the functions of this class 
    1. self: object created in __main__
    2. ret(Default value=False): If it is given as True then function will not 
       print anything on stdout, instead values will be returned to the 
       calling function. By default (which is the case when it will be called 
       from C code), all functions will print to stdout and caller has to 
       interpret it. 
  '''

  def __init__(self):
    '''
      Install class initialization.

      Following object variables are initialized:
        platform
        vmtype
        group
        arch
    ''' 

    bootDict = {}
    with open(BOOTSTRAP_FILE) as bootFile:
      for line in bootFile:
        line = line.split('#')[0].strip()
        if not line:
          continue
        (key, value) = line.split('=')
        key = key.strip()
        value = value.strip()

        bootDict[key] = value  

    if PLATFORM_STR1 in bootDict:
      self.platform = bootDict[PLATFORM_STR1]
    elif PLATFORM_STR2 in bootDict:
      self.platform = bootDict[PLATFORM_STR2]
    else:
      logger.error('ERROR: Could not able to get platform from ' 
                   + BOOTSTRAP_FILE)
      sys.exit(1)

    self.vmtype = ''
    archStr = ''
    if 'VMTYPE' in bootDict:  
      self.vmtype = bootDict['VMTYPE']
    else:
      out_dict = runcmd('/usr/bin/xr_sysctl -n kernel.boot.vmtype')
      if out_dict['rc']:
        logger.error('ERROR: Could not able to get VM Type from '
                      + BOOTSTRAP_FILE + ' or ' + CMD_FILE)
        sys.exit(1)
      else:
        self.vmtype = out_dict['output']


    if self.vmtype == 'xr-vm':
      archStr='XR_SUPPORTED_ARCHS'
      self.group = 'IOS-XR'
    elif self.vmtype == 'sysadmin-vm':
      archStr='CALV_SUPPORTED_ARCHS'
      self.group = 'sysadmin'
    else:
      logger.error('ERROR: Unknown vm-type: ' + self.vmtype)
      sys.exit(1)

    localArch = runcmd("arch")
    localArch = localArch['output'].strip()

    if archStr in bootDict:
      supportedArchs = bootDict[archStr].split(',')
    else:
      supportedArchs = [ 'x86_64' ]
    bFound = False
    for arch in supportedArchs:
      if arch == localArch or arch in localArch :
        localArch = arch
        bFound = True

    if not bFound:
      logger.error('ERRROR: Unknown architecture: ' + localArch)
      sys.exit(1)
    
    self.arch = localArch

    logger.info('PLATFORM: ' + self.platform + ' VM Type: ' + self.vmtype + 
                 ' ARCH: ' + self.arch)

     
  def _is_smu(self, smu):
    if smu and re.search('CSC[a-z]{2}\d{5}', smu):
      return True
    return False


  def _is_tp_smu(self, smu):
    if self._is_smu(smu) and not smu.startswith(self.platform):
      return True
    return False


  def _is_hostos_smu(self, smu):
    if self._is_smu(smu):
      if "sysadmin-hostos" in smu or "spirit-boot" in smu:
        return True
    return False


  def _strip_arch(self, pkg, bDebugLog=False):
    '''
      Strip architecture from a string or list of a string
    '''
    
    if type(pkg) == list:
      pkg_list = []
    
      for p in pkg:
        p = p.replace('.x86_64','')
        p = p.replace('.arm','')
        pkg_list.append(p)
      
      pkg = pkg_list
      
    elif type(pkg) == str:
      pkg = pkg.replace('.x86_64','')
      pkg = pkg.replace('.arm','')
    
    else:
      logger.error('ERROR: ' + str(type(pkg)) + ' type\'s argument passed to '
                   + '_strip_arch function is not supported')
    
    if bDebugLog:
      logger.debug('Package Name(s) with stripped arch: ' + str(pkg))
    return pkg


  def _initialize_installed_smu_list(self):
    PATH_PREFIX = '/install/instdb/local'
    LDPATH_FILE = ''
    LOGGER_BIN = ''

    if self.vmtype == 'sysadmin-vm':
      LDPATH_FILE = '{}/calvados_ldpath.txt'.format(PATH_PREFIX)
      LOGGER_BIN = '/opt/cisco/calvados/bin/cal_logger'
    elif self.vmtype == 'xr-vm':
      LDPATH_FILE = '{}/xr_ldpath.txt'.format(PATH_PREFIX)
      LOGGER_BIN = '/pkg/bin/logger'
    else: #This is handled in __init__, but just kept here for clarity
      logger.error('Unknown vmtype: {}'.format(self.vmtype))  
      return ([], {}) 

    try:
      with open(LDPATH_FILE) as fLoadpath:
        lines = fLoadpath.readlines()
        new_lines = []
        for x in lines:
          x = x.strip().split('#')[0]
          if x :
            new_lines.append(x)
        lines = new_lines    

        logger.info('{} file Content:'.format(LDPATH_FILE))
        logger.info('{}'.format(str(lines)))

        if len(lines) != 1:
          err_msg = 'Loadpath file {} seems to be corrupted; supersede info ' \
                    'may be wrong.'.format(LDPATH_FILE)
          logger.error(err_msg)
          os.system("{} -s crit {}".format(LOGGER_BIN, err_msg))
          return ([], {})

        installed_smu_list = []   # list of valid SMUs from loadpath file
        installed_smu_tuples = [] # list of (name, version, release) for
                                  # above SMUs
        supersede_dict_temp = {}  # Temporary dict to hold (n, v, r) tuples  
                                  # according to each package SMUs
                                  # {key: first SMU (n, v, r) tuple in loadpath
                                  #       for a given package
                                  #  values: (n, v,r) tuples of remaining SMUs 
        supersede_dict = {}       # {key: top-level SMU rpm name
                                  #  value: all obsoleted SMU rpms of a given 
                                  #         top-level rpm

        for pkg in lines[0].split(':'):
          pkg = self._strip_arch(pkg)

          if not self._is_smu_supersedable(pkg):
            continue

          smu_name_group = re.search(r'(.*)-(.*)-(.*\.CSC[a-z]{2}\d{5}.*)', 
                                     pkg)
          if smu_name_group:
            (name, version, release) = smu_name_group.groups()
            installed_smu_tuples.append((name, version, release)) 
            installed_smu_list.append(pkg)
          else: # Ignoring this loadpth entry
            logger.error('SMU name {} is not in proper format.'.format(pkg))
            logger.error('It could be due to possible corruption '\
                          'of {} file'.format(LDPATH_FILE))

        logger.info('SMUs after filtering: {}'.format(str(installed_smu_list)))

        for (name, version, release) in installed_smu_tuples:
          top_smu_entry = ''
          # checking for existing entries in supersede_dict_temp
          # if same package entry is there then just append it
          for (name_key, ver_key, rel_key) in supersede_dict_temp:
            if name == name_key:
              top_smu_entry = (name_key, ver_key, rel_key)
              break

          # Assumption: First SMU will always be superseding later SMUs in a 
          #             given loadpath
          if top_smu_entry:
            supersede_dict_temp[top_smu_entry].append((name,version,release))
          else:
            supersede_dict_temp[(name,version,release)] = []

        for (name, version, release) in supersede_dict_temp:
          top_smu = '{}-{}-{}'.format(name, version, release)
          supersede_dict[top_smu] = []
          #TODO: can soring be done here based on version?
          for (obs_name, obs_version, obs_release) in \
              supersede_dict_temp[(name, version, release)]:
            obs_smu = '{}-{}-{}'.format(obs_name, obs_version, obs_release)
            supersede_dict[top_smu].append(obs_smu)

        for smu in supersede_dict:
          supersede_dict[smu].sort()

        logger.info('Supersede dict: {}'.format(str(supersede_dict)))

    except:
      logger.exception('Exception occurred in _initialize_installed_smu_list ')
      logger.info('Returning None(s)')
      installed_smu_list = []
      supersede_dict = {}

    return (installed_smu_list, supersede_dict)


  def _is_smu_supersedable(self, smu):
    ret_value = True

    try:
      smu = self._strip_arch(smu)

      if not self._is_smu(smu):
        #logger.debug("Ignoring package {} as it is not SMU".format(smu))
        ret_value = False
      elif self._is_tp_smu(smu):
        #logger.debug("Ignoring SMU {} as it is a TP SMU".format(smu))
        ret_value = False
      elif self._is_hostos_smu(smu):
        #logger.debug("Ignoring SMU {} as it is a hostos SMU".format(smu))
        ret_value = False
      else:
        smu_name_group = re.search(r'(.*)-(.*)-(.*\.CSC[a-z]{2}\d{5})', smu)
        if smu_name_group:
          (name, version, release) = smu_name_group.groups()
          release_group = re.search(r'(.*).(CSC[a-z]{2}\d{5})', release)
          if release_group:
            (rel_ver, ddts) = release_group.groups()
            if rel_ver[-1].isalpha():
              logger.debug("Ignoring SMU {} as it is an engg. SMU".format(smu))
              ret_value = False
          else:
            ret_value = False
        else:
          ret_value = False
    except:
      logger.exception('Exception occurred in _is_smu_supersedable function')
      logger.info('Returning False')
      ret_value = False


    return ret_value  


  def is_smu_superseded(self, smu):
    '''
      Return True if given SMU in_smu has been superseded by any another 
      installed SMU.

    '''

    ret_value = False

    try:
      smu = self._strip_arch(smu)

      if self._is_smu_supersedable(smu):
        (smu_list, supersede_dict) = self._initialize_installed_smu_list()

        if smu in smu_list:
          if smu not in supersede_dict:
            ret_value = True
        else:
          logger.warning("SMU {} is not in installed SMU list".format(smu))
    except:
      logger.exception('Exception occurred in is_smu_superseded')
      ret_value = False

    logger.info('Final print : {}'.format(str(ret_value).upper()))
    print(str(ret_value).upper())


  def are_smus_superseded(self, input_smus):
    '''
      Return a string containing 0/1 indicating supersede detail of SMUs
    ''' 
    
    return_str = ''

    try:
      (smu_list, supersede_dict) = self._initialize_installed_smu_list()

      for smu in input_smus.split(':'):
        smu = self._strip_arch(smu)

        if not self._is_smu_supersedable(smu):
          return_str += '0'

        elif smu in smu_list:
          if smu in supersede_dict:
            return_str += '0'
          else:
            return_str += '1'
        else:
          return_str += '0'
    except:
      logger.exception('Exception occurred in are_smus_superseded function')
      return_list = '0'*len(input_smus)

    logger.info('Final print : {}'.format(return_str))
    print(return_str)


  def get_all_superseded_smus (self, input_smus, ret_option=False): 
    '''
      Return a string containing supsersede info for all input SMUs
      
      Output string format: "2:SUP_SMUa:SUP_SMUb:1:SUP_SMUc" 
      Superseded SMUs in output will be in correspendence to the sequence of  
      SMUs given in input_smus argument
    '''

    return_str = ''
    total_sup_smus = 0

    try:
      (smu_list, supersede_dict) = self._initialize_installed_smu_list()
      # What if input_smus are not same as smu_list??
      # This should be a transient issue during install operation

      for smu in input_smus.split(':'):
        smu = self._strip_arch(smu)

        if not self._is_smu_supersedable(smu):
          return_str += '0:'

        elif smu in smu_list:
          if smu in supersede_dict:
            total_sup_smus += len(supersede_dict[smu])
            return_str += '{}:'.format(len(supersede_dict[smu]))
            for sup_smu in supersede_dict[smu]:
              return_str += '{}:'.format(sup_smu)
          else:
            return_str += '0:'
        else:
           return_str += '0:'
    except:
      logger.exception('Exception occurred in get_all_superseded_smus')
      return_str = '0:'*len(input_smus)

    if ret_option:
      logger.info('Final return : {}'.format(total_sup_smus))
      return total_sup_smus
    else:
      logger.info('Final print : {}'.format(return_str))
      print(return_str)


  def get_all_superseded_smus_count(self, input_smus):
    '''
      Get number of supersiding SMUs all input smus
    ''' 

    total_smu_count = 0

    try:
       total_smu_count = self.get_all_superseded_smus(input_smus, True)
    except:
      logger.exception('Exception occurred in get_all_superseded_smus_count')
      total_smu_count = 0

    logger.info('Final print : {}'.format(total_smu_count))
    print(total_smu_count)      


  def get_username (self, path):
    '''
      to retrieve username from source path in install add and update
    '''
    path = path.replace("\\","")
    path = urlparse(path)
    if(path.username):
        logger.info('Username retrieved: ' + path.username)
        print(path.username)
    else:
        logger.info('Username is not present')

  def get_password (self, path):
    '''
      to retrieve password from source path in install add and update
      Returns "@@$$^^" if password is not present
    '''
    path = path.replace("\\","")
    path = urlparse(path)
    if(path.password):
        print(path.password)
    else:
        logger.info("Password is not present")
        print("@@$$^^")

  def get_server_ip (self, path):
    '''
      to retrieve server ip from source path in install add and update
    '''
    path = path.replace("\\","")
    path = urlparse(path)
    logger.info('Server ip addr retrieved: ' + path.hostname)
    print(path.hostname)

  def get_source_dir (self, path):
    '''
      to retrieve source dir from source path in install add and update
    '''
    path = path.replace("\\","")
    parsed_path = urlparse(path)
    logger.info('Server dir retrieved: ' + parsed_path.path)
    print(parsed_path.path)
    
  def get_path_without_password (self, path):
    '''
      in install add and update, to remove password from path
    '''
    path = path.replace("\\","")
    parsed_path = urlparse(path)
    if(parsed_path.password):
        pwdless_path = path.replace(':'+parsed_path.password,"")
        logger.info('Path without password: ' + pwdless_path)
        print(pwdless_path)
    else:
        logger.info("Password is not present")
        print("@@$$^^")
    
  def get_password_encoded_path (self, path):
    '''
      in install add and update, to remove password from path
    '''
    path = path.replace("\\","")
    parsed_path = urlparse(path)
    if(parsed_path.password):
        encoded_pwd = "*"*len(parsed_path.password)
        encoded_path = path.replace(parsed_path.password,encoded_pwd)
        logger.info('Password Encoded path: ' + encoded_path)
        print(encoded_path)
    else:
        logger.info("Password is not present")
        print("@@$$^^")
    
  def get_modified_path (self, path):
    '''
      path after removing '\'
    '''
    path = path.replace("\\","")
    logger.info('Modified path: ' +path)
    print(path)

 # get RPM Path from giso_info.txt
  def get_rpm_path_from_giso(self, giso_info):
    """
     Check if RPM_PATH exist in giso_info.txt
     if doesn't exist return NORPMPATH
    """
    for line in open(giso_info, 'r'):
        data = line.find("RPM_PATH:")
        if data != -1:
           return line
    return ("NORPMPATH")

  def is_signed_651_ncs5500_rpm_path(self, line):
     """ 
      check for path like giso/boot/initrd.img/iso/system_image.iso
      in giso_info.txt
     """
     s=line.find("system_image.iso")
     if s != -1:
        if (line.count("boot") == 1):
            return True
     return False

  def is_signed_rpm_path(self, line):
    """
      check for path like giso/boot/initrd.img
      in giso_info.txt
    """
    s = line.find("system_image.iso")
    if s == -1:
       if(line.count("boot") == 1):
          return True
       else:
           return False
    return False

  def is_signed_ncs5500_rpm_path(self, line):
    """ 
     check for path like 
     giso/boot/initrd.img/iso/system_image.iso/boot/initrd.img in
     giso_info.txt return True if exist such path in RPM_PATH
    """
    s = line.find("system_image.iso")
    if s != -1:
      if (line.count("boot") == 2):
          return True
    return False

  def get_giso_rpm_path(self, giso_info):
    """ 
      typedef enum {
          SIGNED_NCS5500_RPM_PATH       = 1,
          SIGNED_651_NCS5500_RPM_PATH   = 2,
          SIGNED_RPM_PATH               = 3,
          DEFAULT_RPM_PATH              = 4,
          /* not golden ISO */
          NO_GISO_ISO_PATH             = 12345,
      } giso_rpm_path_t;

      Checks RPM_PATH in golden iso returns one of the above enum defined 
      based on value RPM_PATH
    """
    path = self.get_rpm_path_from_giso(giso_info)
    result = path.find("NORPMPATH")
    if result == -1:
       if self.is_signed_ncs5500_rpm_path(path):
          print(1)
       elif self.is_signed_651_ncs5500_rpm_path(path): 
          print(2)
       elif self.is_signed_rpm_path(path):
          print(3)
       else:
          print(4)
    else:
        print(4) 

  def check_iso_isgiso(self, path):
      """ 
         check is it golden ISO by looking giso_info.txt
         if giso_info.txt doesn't find returns 
         NO_GISO_ISO_PATH enum defined in caller 
      """
      ret = runcmd("isoinfo -R -l -i " + path + "| grep giso_info.txt",
                    success_ret_values = [0, 1])
      if ret["rc"] == 1:
         print(12345)
      else:
         print(0)

  def get_packages_from_swp_file(self, file_name):
    '''this function will read the .txt swp and returns the list of packages'''
    file_lines = []
    try:
        f = open(file_name, "r")
        for line in f:
          file_lines.append(line)
        f.close()
    except:
        logger.debug("Failed to read swp {}".format(file_name))
        return []

    lines_len = len(file_lines)
    pkg_list = []

    for i in range(lines_len):
        val = file_lines[i].strip()
        val = val.split(":")
        if len(val) > 1 and (' pkg ' in val[0] or ' SP ' in val[0]):
            val = val[1]
        else:
            continue
        # pkg 0 rpm 0: ncs5500-sysadmin-hostos-7.0.1.105-r70199I.CSCho99999.admin
        #SP 0: ncs5500-sp4-x-7.0.1.99I
        m = re.search(r'(.*/)*(.*)-(.*)-(.*)\.(.*)', val)
        if m :
            pkg_list.append(val)

    return pkg_list

  def check_if_system_committed(self):
    ''' this function will verify if the active and committed swp .txt'''
    ''' files have the same packages '''
    # get the list of swp profiles
    cmd  = 'ls /install_repo/local/*swprofile*active*txt'
    out_dict = runcmd(cmd)
    if out_dict['rc']:
        print(-1)
    else:
        output = out_dict['output'].splitlines()
        for x in output:
            logger.debug("for file {}".format(x))
            pkg_list_active = self.get_packages_from_swp_file(x)
            logger.debug(pkg_list_active)
            y = x.replace('active','committed')
            pkg_list_committed = self.get_packages_from_swp_file(y)
            logger.debug(pkg_list_committed)
            if set(pkg_list_active) != set(pkg_list_committed):
                print(1)
                break
        else:
            print(0)

if __name__ == '__main__':
  '''
    Generic python script for various install functions.

    First argument should be function name. 
    Rest of the argument should be the arguments for function passed as first 
    argument.
    In main, first logger is intialized and then respective functions are 
    called after instantiating Install class.
  '''

  # create logger
  logger = logging.getLogger('install_functions_logger')
  logger.setLevel(logging.DEBUG)
  formatter = MyFormatter('%(asctime)s  #%(process)d:: %(message)s',
                                '%Y-%m-%d %H:%M:%S.%f')

  # Logs to logfile
  fh = logging.handlers.RotatingFileHandler(LOG_FILE, 
                                            maxBytes=(1024*10000), 
                                            backupCount=2)
  fh.setLevel(logging.DEBUG)
  fh.setFormatter(formatter)
  logger.addHandler(fh)

  logger.debug('+'*80)
  logger.debug(' '.join(sys.argv))

  inst= Install()
  if not hasattr(inst, sys.argv[1]):
    logger.error('No function name ' + sys.argv[1] + ' in ' + sys.argv[0])
    sys.exit(1)

  function = getattr(inst, sys.argv[1])
  if len(sys.argv) > 2:
    function(','.join(sys.argv[2:]))
  else:
    function()


