#!/bin/bash

# Copyright (c) 2015-2019 by cisco Systems, Inc.
# All rights reserved.
# Created by: Santhosh N

SCRIPTPATH="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
source "$SCRIPTPATH/disk_space_precheck.sh"

function run_cmd() {
    echo -e "\nRun -> $@"
    eval $@
    echo ""
}

function validate_iso() {
    miniiso=$1
    isomnt=`mktemp -d /tmp/.su_isomnt.XXXXXX`
    sync

    run_cmd mount -o loop $miniiso $isomnt

    initrd_md5=$(md5sum $isomnt/boot/initrd.img | awk '{ print $1 }' )
    if grep "^Initrd" $isomnt/iso_info.txt; then
       iso_md5=$(grep "^Initrd" $isomnt/iso_info.txt | awk '{print $3}' )
    else
       umount $isomnt
       rm -rf $isomnt
       return 0
    fi

    umount $isomnt
    rm -rf $isomnt
    sync

    if [ "$initrd_md5" == "$iso_md5" ]; then 
        return 0 
    else
        return 1  
    fi;
  }

function get_disk1_dev() {
   local retval="Failed"

   echo "Function get_disk1_dev begin" > /tmp/get_disk1_dev
   set +e
   for device in `ls /dev/vd* `; do
        out=$(blkid -s LABEL -o value ${device} 2>/dev/null)
        echo $out  $device >> /tmp/get_disk1_dev
        if [ "${out}" == "DISK_ONE" ]; then
            echo "found the Recovery partition " >> /tmp/get_disk1_dev
            retval=$device
            break
        fi
    done
    echo $retval
    echo "Ret val " $retval >> /tmp/get_disk1_dev
}

function modify_prep_chkpt() {
    new_restart_type=$1
    prep_chkpt_file=$2
    sed -i "s/^\(restart\_type\s\+\).*/\1$new_restart_type/" $prep_chkpt_file
}

function inst_get_obsoletes_from_rpm()
{
   pkg_name=$1
   is_su_operation=$2
   admin=$3

   if [ "$is_su_operation" = "TRUE" ]; then
       rm -rf /tmp/my_sup_dir
       pkg_path=$(find /install_repo/cache/ -name "$pkg_name")
       mkdir -p /tmp/my_sup_dir
       cd /tmp/my_sup_dir
       echo "$pkg_path"
       tar vxf $pkg_path >/dev/null
       if [ $? = 0 ]; then
          if [ "$admin" = "FALSE" ]; then
            ddts_lines=$(rpm -qp --obsoletes *.rpm | cut -d = -f1 | cut -d - -f3)
          else
            ddts_lines=$(rpm -qp --obsoletes *.rpm | cut -d - -f2 | cut -d = -f1)
          fi
          for line in $ddts_lines
          do
           echo ${line%.*} >>/tmp/my_sup_dir/sup_pkgs_from_rpms
          done
          if [ -e "/tmp/my_sup_dir/sup_pkgs_from_rpms" ]; then
              cat /tmp/my_sup_dir/sup_pkgs_from_rpms | sort -u >/tmp/obsoletes_smus_data_in_bundle
          fi
       else
          echo "package or path doesn't exist"
       fi
       rm -rf /tmp/my_sup_dir
   else
       pkg_pfx=$(echo ${pkg_name%-*})
       ddts="${pkg_pfx##*.}"
       echo "ddts: $ddts"
       rm -rf /tmp/sup_pkgs_from_rpms

       if [ "$admin" = "FALSE" ]; then
         pkgs=$(ls /opt/cisco/XR/packages/ | grep "$ddts")
       else
         pkgs=$(ls /opt/cisco/calvados/packages/ | grep "$ddts")
       fi

       if [ -z "$pkgs" ]; then
           echo "No packages found !!"
       else
         if [ "$admin" = "FALSE" ]; then
           ddts_lines=$(rpm -q --obsoletes $pkgs | cut -d = -f1 | cut -d - -f3)
         else
           ddts_lines=$(rpm -q --obsoletes $pkgs | cut -d - -f2 | cut -d = -f1)
         fi
         for line in $ddts_lines
         do
           echo ${line%.*} >>/tmp/sup_pkgs_from_rpms
         done
         if [ -e "/tmp/sup_pkgs_from_rpms" ]; then
             cat /tmp/sup_pkgs_from_rpms | sort -u >/tmp/obsoletes_smus_data_in_bundle
         fi
       fi
       rm -rf /tmp/sup_pkgs_from_rpms
   fi
}

###########################################################
# XRVM  superseded infra changes
#
############################################################

SYSTEM_RPM_DATA="/install/obsoletes/__installed_rpm_smu__.txt"
SUPERSEDED_SMUS="/install/obsoletes/__superseded_smus__.txt"
SUP_LOG_FILE="/var/log/install/superseded__check.logs"

#
# get obsoletes data from the installed SMUs
# and cache 
# IN: file containing rpm -qa data 
# OUT: list of DDTS which are fully superseded
#      will be added to SUPERSEDED_SMUS file
#

function get_superseded_data_from_smus()
{
 set +o pipefail
 set +e 
 rpm_qa="$1"
 #$1 is input rpm -qa data
 smu_list=$(cat $rpm_qa | grep "CSC" | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/' | sort -u)
 if [ -z "$smu_list" ]; then
     return
 fi

 for smu in ${smu_list[@]}
 do
   echo "Requires: $smu " >>$SUP_LOG_FILE
   inst_rpm=$(grep "$smu" "$rpm_qa")

   # Requires 
   grep $smu "$rpm_qa" | xargs rpm -qR | grep "CSC" | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/' | sort -u >/tmp/rlist
   if [ -e /tmp/rlist ]; then
        cat /tmp/rlist >>$SUP_LOG_FILE
   else
       touch /tmp/rlist
   fi

   #echo "Obsoletes ######"
   s_list=$(grep $smu "$rpm_qa" | xargs rpm -q --obsoletes  |\
         grep "CSC" | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/' | sort -u)

   echo "obsoletes: $s_list">>$SUP_LOG_FILE

   for ddts in ${s_list[@]}
   do
     ddts_in_req=$(grep $ddts /tmp/rlist)
     if [ -z "$ddts_in_req" ]; then
         ddts_active=$(grep $ddts "$rpm_qa")
         if [ -z "$ddts_active" ]; then
             continue
         else
            echo "$ddts">>$SUPERSEDED_SMUS
         fi
     else
         continue
     fi
     done
 done
}

function inst_get_superseded_smus()
{
 set +o pipefail
 set +e
 smu=$2

 if [ -e /tmp/superseded_data.txt ]; then
      rm /tmp/superseded_data.txt
 fi
  
 grep $smu "$1" | xargs rpm -qR | grep "CSC" | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/' | sort -u >/tmp/rlist
 if [ -e /tmp/rlist ]; then
     cat /tmp/rlist >>$SUP_LOG_FILE
 else
      touch /tmp/rlist
 fi 

 #echo "Obsoletes ######"
 s_list=$(grep $smu "$1" | xargs rpm -q --obsoletes  |\
      grep "CSC" | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/' | sort -u) 

 echo "inst_get_superseded_smus:obsoletes: $s_list">>$SUP_LOG_FILE
 
 for ddts in ${s_list[@]}
   do
     ddts_in_req=$(grep $ddts /tmp/rlist)
     if [ -z "$ddts_in_req" ]; then
         ddts_active=$(grep $ddts "$1")
         if [ -z "$ddts_active" ]; then
             echo ""
         else
            echo "$ddts">>/tmp/superseded_data.txt
         fi
     else
        echo ""
     fi
   done
 if [ -e /tmp/rlist ]; then
       rm -rf /tmp/rlist
 fi
}

#
# Check SMU fully supersedes any other installed SMU
# on the system 
# IN: SMU name
# OUT: creates file with SMU name and stores 
#      SMU IDs which are fully superseded by input SMU 
#
function inst_get_superseded_info()
{
 set +o pipefail
 set +e

    if [ -e /install/obsoletes/$1 ]; then
        cat /install/obsoletes/$1 
        return 
    fi

    in_smu=$(echo $1 | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/')
    if [ -e /install/obsoletes ]; then
        echo ""
    else
        mkdir -p /install/obsoletes
    fi

    rpm -qa >$SYSTEM_RPM_DATA
    touch /install/obsoletes/$1
    inst_get_superseded_smus "$SYSTEM_RPM_DATA" $in_smu
    if [ -e /tmp/superseded_data.txt ]; then
       cp  /tmp/superseded_data.txt /install/obsoletes/$1 
       cat /install/obsoletes/$1
    fi

}

function inst_superseded_validate_cache()
{
   rm -rf /install/obsoletes/*
}

function inst_get_sup_display()
{
  smu="$1"
  ddts="$2"
  echo $smu | sed  's/CSC\w\+/'$ddts'/'

}

####
# check given smu is present in superseded list
####
function inst_check_smu_superseded()
{
 set +o pipefail
 set +e

   in_smu=$(echo $1 | sed 's/.*\(CSC[A-Za-z0-9]\{7\}\).*/\1/')
   #is it optional package ?
   if [ -z "$in_smu" ]; then
       return "FALSE"
   fi

   if [ -e "$SUPERSEDED_SMUS" ]; then
       is_in_sup=$(grep "$in_smu" $SUPERSEDED_SMUS)
       if [ -z "$is_in_sup" ]; then
           echo  "FALSE" 
       else
           echo "TRUE" 
       fi
   else
       #running show install active before running show install superseded
       mkdir -p /install/obsoletes
       rpm -qa >$SYSTEM_RPM_DATA
       get_superseded_data_from_smus $SYSTEM_RPM_DATA

       if [ -e "$SUPERSEDED_SMUS" ]; then
           is_in_sup=$(grep "$in_smu" $SUPERSEDED_SMUS)
           if [ -z "$is_in_sup" ]; then
               echo "FALSE" 
           else
               echo "TRUE" 
            fi
       else
           # no superseded package exist, so just touch it
           # to avoid sup check 
           touch $SUPERSEDED_SMUS 
           echo "FALSE" 
       fi
   fi
}

function check_release()
{
    set +o pipefail
    set +e
    local IS_NEW_PKG_FMT=$2
    releases_new_pkg_fmt=("r601" "r602" "r611" "r612" "r6131" "r6121" "r613" "r614" "r622" "r623" "r631" "r60" "r61" "r62")
    arraylth_new=${#releases_new_pkg_fmt[@]}

    releases_old_fmt=("5.2.5." "5.2.6." "6.0.1." "6.0.2." "6.1.1." "6.1.2." "6.1.31." "6.1.21." "6.1.3." "6.1.4." "6.2.2." "6.3.1." "6.0." "6.1." "6.2.")
    arraylth_old=${#releases_old_fmt[@]}

    i=0
    if [ $IS_NEW_PKG_FMT -eq 1 ]
    then
        for (( i=1; i<${arraylth_new}+1; i++ ));
        do
            echo "$1" | grep ${releases_new_pkg_fmt[i-1]}
            if (( $? == 0 )); then
                break;
            fi
        done
        if [ $i -gt ${arraylth_new} ]
        then
            return 0
        fi
    else
        for (( i=1; i<${arraylth_old}+1; i++ ));
        do
            echo "$1" | grep ${releases_old_fmt[i-1]}
            if (( $? == 0 )); then
                break;
            fi
        done
        if [ $i -gt ${arraylth_old} ]
        then
            return 0
        fi
    fi

    return 1
}
function check_tp_smu_and_rel()
{
    set +o pipefail
    set +e
    local RPM=$1
    local IS_NEW_PKG_FMT=$2
    tp_smu_rel=`rpm -qp --queryformat "[%{xrrelease}\n]" $RPM`
    tp_smu_rel="$(echo -e "${tp_smu_rel}" | tr -d ' ')"

    if [ ! -z "$tp_smu_rel" ]
    then
        if [ `echo $tp_smu_rel | grep ^[0-9]` ]
        then
            tp_smu_rel="r$tp_smu_rel"
            check_release tp_smu_rel $IS_NEW_PKG_FMT
            return $?
        fi
    fi
    return 2
}
function verify_signature()
{
  set +o pipefail
  set +e

  local RPM_PATH=$1
  local retval="Success"
  local IS_NEW_PKG_FMT=0
  local is_tp_smu_and_verify=0
  local check_rel_ret_val=-1

  local RELEASE=`rpm -qp --queryformat "[%{RELEASE}\n]" $RPM_PATH`
  RELEASE="$(echo -e "${RELEASE}" | tr -d ' ')"
  if [ ! -z "$RELEASE" ]
  then
      if [ `echo "$RELEASE" | grep ^r` ]
      then
          IS_NEW_PKG_FMT=1
          check_tp_smu_and_rel $RPM_PATH $IS_NEW_PKG_FMT

          check_rel_ret_val=$?

          if [ $check_rel_ret_val -eq 2 ]
          then
              check_release $RELEASE $IS_NEW_PKG_FMT
              check_rel_ret_val=$?
          fi

      elif [ `echo "$RELEASE" | grep "Default"` ]
      then
          check_release $RPM_PATH $IS_NEW_PKG_FMT
          check_rel_ret_val=$?

      elif [ `echo "$RELEASE" | grep "SMU"` ]
      then
          check_release $RPM_PATH $IS_NEW_PKG_FMT
          check_rel_ret_val=$?
      fi
  fi
  if [ $check_rel_ret_val -eq 0 ]
  then
      rpm -K $RPM_PATH >> /dev/null
      if (( $? != 0 )); then
          retval="Failed"
      else
          rpm -Kv $RPM_PATH | grep -i "signature:" | grep ": OK"
          if (( $? != 0 )); then
              retval="Failed"
          fi
      fi
  fi

  echo $retval
}

function ip_active_sdr_instmgr() {
    nodename=`/pkg/bin/placed_show -p sdr_instmgr | grep sdr_instmgr | awk '{print $4}'`
    node_ipaddr=`/pkg/bin/show_platform_sysdb -v | grep -e "^$nodename" | awk '{print $7}'`
    echo $node_ipaddr
}

function update_label() {
    label="$1"
    node_ip=`ip_active_sdr_instmgr`

    echo "Label = $label"
    echo "Node-IP = $node_ip"

    cmd="echo 'Label : $label' > /tmp/label-info.txt; kill -s HUP `pgrep sdr_instmgr`"
    ssh $node_ip $cmd
}

function sign_error() {
    set +e
    set +o pipefail
    rpm_name="$1"
    base_name="$(basename $rpm_name)"
    err_msg=""

    has_signature="$(rpm -qpi $rpm_name 2>/dev/null | grep "Signature" | grep "Key ID" | cut -d"," -f3)"

    if [[ -z "$has_signature" ]]
    then
        err_msg="$(echo -e "RPM: $base_name is not signed")"
    else
        signature2="$(rpm -qa | head -1 | xargs -I '{}' rpm -qi "{}" | grep Signature | cut -d "," -f3)"

        if [[ $has_signature != $signature2 ]]
        then
            err_msg="$(echo "Key Mismatch Error for $base_name. Please check if both image and RPMS are signed with same keys")"
        fi
    fi
    echo "$err_msg"
    set -e
}

# This function will print node_name:node_ip:directory_usage(%):available_space(kb) 
# of given directory  in all the XR vm in the system
# First argument should be the path of the directory
# if directory exist then it will give the usage in percent
# If directory doesn't exist tehn it will give -1 as usage in percent
# Output would be like follow
# [xr-vm_node0_RSP0_CPU0:~]$xr_dir_usage_in_pcent /var/log
# 0/0/CPU0:192.0.0.3:5:7#0/RSP0/CPU0:192.0.4.4:8:20#0/RSP1/CPU0:192.0.8.4:94:37
# [xr-vm_node0_RSP0_CPU0:~]$
function xr_dir_usage_in_pcent ()
{
   local node_name=""
   local node_ip=""
   local usage_avail=""
   local usage_pcent=""
   local avail_kb=""

   name_ip_arr=( `show_platform_sysdb -v | grep " Band" | awk '{ printf "%s:%s\n", $1, $7 }'` )
   for element in "${name_ip_arr[@]}"
   do
       node_name=`echo ${element} | cut -d ':' -f1`
       node_ip=` echo ${element} | cut -d ':' -f2`
       # if directory exist get value else -1
       if ssh  -q $node_ip  [[ -d "$1" ]]; then
           usage_avail=`ssh $node_ip "df -k --output=pcent,avail $1 | tail -n1"`
           usage_pcent=`echo $usage_avail | cut -d " " -f1 | sed "s/[^0-9]//g"`
           avail_kb=`echo $usage_avail | cut -d " " -f2`
       else
           usage_pcent=-1
           avail_kb=-1
       fi
       echo -n "$node_name:$node_ip:$usage_pcent:$avail_kb#"
   done
   echo
}
