#!/usr/bin/python

# Copyright (c) 2018-2019 by Cisco Systems, Inc.
# All rights reserved.

import time
import sys
import os
from   os import path as PATH
import json
import re
import fcntl
import subprocess
from   subprocess import Popen, PIPE, call
import datetime
import shutil
import crypt
import hashlib
import pam_edcd_utils as pam_edcd

ONDEMAND_EDCD_PATH    = '/misc/disk1/cisco_support/edcd/'
EDCD_VARIABLES        = ['alias', 'component',
                         'created', 'ddts', 'identifier',
                         'updated', 'vers', 'wiki',
                         'comment']
PLATFORMS             = {'all': 'All', 'xrv9k': 'Sunstone',
                         'sunstone': 'Sunstone', 'asr9k': 'ASR9K',
                         'ncs6k': 'Panini', 'fretta': 'Fretta',
                         'ncs5.5k': 'Fretta', 'zermatt': 'Fretta',
                         'ncs5k': 'Skywarp', 'ncs4k': 'Scapa',
                         'ncs1k': 'Rosco',
                         'iosxrwb-xgs': 'iosxrwb-xgs'}

ONDEMAND_ATTRS        = ['do_async',
                         'do_tgz',
                         'identifier',
                         'logdir',
                         'noRetainLogs',
                         'retainLogs',
                         'tag']

DEBUG                 = False
platform              = "All"
operSys               = 'XR'
on_lindt              = False


def display_edcd_cmd_data(identifiers):
    json_data = pam_edcd.ReadJsonFile(getattr(args, 'cmd_json_file'))
    for identifier in identifiers:
        if identifier not in json_data:
#           print("ignore %s" % identifier)
            continue
        print("="*60)
        print(" "*15 + "Identifier: " + identifier)
        print("="*60)
        count = 1
        commands = [x for x in json_data[identifier]["commands"].split("\n")
                                                         if x.strip() != ""]
        for command in commands:
            space = " "
            if count >= 10:
                space = ""
            print(space + str(count) + ": " + command)
            count += 1
        print("-"*60)
        print("")


def add_eem(args):
    edcd_path = getattr(args, 'edcd_path')
    if not os.path.exists(edcd_path):
        os.makedirs(edcd_path)
        os.chmod(edcd_path, 0o777)

    eem_json_file = getattr(args, 'eem_syslog_json_file')
    eem_json_data = pam_edcd.ReadJsonFile(eem_json_file)
    keyword = getattr(args, 'keyword')
    subpattern = getattr(args, 'subpattern')
    throttle_window = getattr(args, 'throttle_window')

    timestamp = datetime.datetime.now()
    timestamp_str = timestamp.strftime("%Y%m%d%H%M%S.%f")

    if keyword not in eem_json_data:
        eem_json_data[keyword] = {}
    if subpattern not in eem_json_data[keyword]:
        eem_json_data[keyword][subpattern] = {}

    field = eem_json_data[keyword][subpattern]
    if getattr(args, 'debug'):
        print json.dumps(field, indent=1)
    if 'created' not in field:
        field['created'] = "Created by PAM agent at " + timestamp_str
    field["modified"] = "Modified by PAM agent at " + timestamp_str
    field['throttle_window'] = str(throttle_window)
    for k in ONDEMAND_ATTRS:
        v = getattr(args, k)
        if v is not None and v != '':
            field[k] = v
        elif v is False:
            field[k] = "false"
        else:
            field[k] = ""
        if getattr(args, 'debug'):
            print("[k=%s]=%s" % (k, v))

    if getattr(args, 'debug'):
        print json.dumps(eem_json_data, indent=1)
    f = open(eem_json_file, 'w')
    fcntl.lockf(f, fcntl.LOCK_EX)
    f.write(json.dumps(eem_json_data, sort_keys=True, indent=4, separators=(',', ': ')))
    fcntl.lockf(f, fcntl.LOCK_UN)
    f.close()
    os.chmod(eem_json_file, 0o777)
    to_ip = pam_edcd.GetVmIP()
    if not on_lindt and len(to_ip) > 0:
        try:
            pam_edcd.ScpFiles(to_ip, edcd_path, json_file)
        except ValueError as verr:
            print("Copy %s to other RPs failed (error: %s)" % _(json_file, verr))

    print("\nEDCD has been updated. Please update configuration by running configure event manager pam-agent.")
    print("(Execute 'show edcd eem database' to display all new EDCD entries.)\n")


def trigger_eem(args):
    edcd_path = getattr(args, 'edcd_path')
    if not os.path.exists(edcd_path):
        os.makedirs(edcd_path)
        os.chmod(edcd_path, 0o777)
    eem_json_file = getattr(args, 'eem_syslog_json_file')
    eem_json_data = pam_edcd.ReadJsonFile(eem_json_file)
    eem_throttle_file_path = getattr(args, 'eem_throttle_json')
    keyword = getattr(args, 'keyword')
    syslog = getattr(args, 'syslog')
    trigger = getattr(args, 'trigger')
    new_entry = False
    timestamp = datetime.datetime.now()
    timestamp_str = timestamp.strftime("%Y%m%d%H%M%S.%f")
    syslog = getattr(args, 'syslog')

    new_eem_data = pam_edcd.get_eem_syslog_data(args, eem_json_data, keyword, syslog)
#   BE VERY CAREFUL - not print any keywork
#   print json.dumps(new_eem_data,indent=1)
    if not new_eem_data or new_eem_data == "undefined" or \
                new_eem_data == None or new_eem_data == "{}":
        print("No valid identifier found - skip data collection.")
        exit()
    is_throttled = pam_edcd.check_throtle(args, new_eem_data, eem_throttle_file_path)
    if is_throttled == 1:
        print("The same commands have been executed within throttling window. Skip executing (execute 'show edcd eem logs' to display details)")
        exit()
    elif is_throttled == -1:
        print("No commands defined in EDCD. Skip executing\n")
        exit()

    triggerfile_path = ONDEMAND_EDCD_PATH + timestamp_str + '-pam_eem_syslog.json'
    if not os.path.exists(ONDEMAND_EDCD_PATH):
        os.makedirs(ONDEMAND_EDCD_PATH)
        os.chmod(ONDEMAND_EDCD_PATH, 0o777)
    f = open(triggerfile_path, 'w')
    fcntl.lockf(f, fcntl.LOCK_EX)
    f.write(json.dumps(new_eem_data, sort_keys=True, indent=4, separators=(',', ': ')))
    fcntl.lockf(f, fcntl.LOCK_UN)
    f.close()
    os.chmod(triggerfile_path, 0o777)
    print("\nEEM action has been triggered. execute 'show edcd eem log' to verify.\n")


def do_eem(args):
    edcd_path = getattr(args, 'edcd_path')
    if not os.path.exists(edcd_path):
        os.makedirs(edcd_path)
        os.chmod(edcd_path, 0o777)

    eem_json_file = getattr(args, 'eem_syslog_json_file')
    eem_json_data = pam_edcd.ReadJsonFile(eem_json_file)
    eem_throttle_file_path = getattr(args, 'eem_throttle_json')

    operation = getattr(args, 'operation')
    keyword = getattr(args, 'keyword')
    subpattern = getattr(args, 'subpattern')
    identifier = getattr(args, 'identifier')
    syslog = getattr(args, 'syslog')
    throttle_window = getattr(args, 'throttle_window')
    trigger = getattr(args, 'trigger')

    if subpattern is not None:
        try:
            re.compile(subpattern)
        except:
            print("Please give a valid regex. ( %s ) is not valid." % subpattern )
            exit()

    if operation == 'add-update':
        timestamp = datetime.datetime.now()
        timestamp_str = timestamp.strftime("%Y%m%d%H%M%S.%f")

        if keyword not in eem_json_data:
            eem_json_data[keyword] = {}
        if subpattern not in eem_json_data[keyword]:
            eem_json_data[keyword][subpattern] = {}

        field = eem_json_data[keyword][subpattern]
        if getattr(args, 'debug'):
            print json.dumps(field, indent=1)

        if 'created' not in field:
            field['created'] = "Created by PAM agent at " + timestamp_str
        field["modified"] = "Modified by PAM agent at " + timestamp_str

        field['throttle_window'] = str(throttle_window)
        for k in ONDEMAND_ATTRS:
            v = getattr(args, k)
            if v is not None and v != '':
                field[k] = v
            elif v is False:
                field[k] = "false"
            else:
                field[k] = ""
            if getattr(args, 'debug'):
                print("[k=%s]=%s" % (k, v))

        if getattr(args, 'debug'):
            print json.dumps(eem_json_data, indent=1)
        f = open(eem_json_file, 'w')
        fcntl.lockf(f, fcntl.LOCK_EX)
        f.write(json.dumps(eem_json_data, sort_keys=True, indent=4, separators=(',', ': ')))
        fcntl.lockf(f, fcntl.LOCK_UN)
        f.close()
        os.chmod(eem_json_file, 0o777)
        to_ip = pam_edcd.GetVmIP()
        if not on_lindt and len(to_ip) > 0:
            try:
                pam_edcd.ScpFiles(to_ip, edcd_path, eem_json_file)
            except ValueError as verr:
                print("Copy %s to other RPs failed (error: %s)" % (eem_json_file, verr))

        print("\nEDCD has been updated. Please update configuration by running configure event manager pam-agent.")
        print("(Execute 'show edcd eem database' to display all new EDCD entries.)\n")

    elif operation == 'add-and-trigger':
        add_eem(args)
        trigger_eem(args)

    elif operation == 'delete':
        new_json_data = {}
        if keyword not in eem_json_data:
            print("No entries found for keyword '%s'" % keyword)
            exit()
        kw_data = eem_json_data[keyword]
        if (subpattern is None or subpattern == '') and \
                               subpattern not in kw_data:
            for k, v in eem_json_data.iteritems():
                if keyword == k:
                    continue
                new_json_data[k] = v
        else:
#           go through each keyword and subpattern
            for k, v in eem_json_data.iteritems():
                if keyword != k:
                    new_json_data[k] = v
            new_pat_data = {}
            for subpat, vv in kw_data.iteritems():
                if subpattern != subpat:
                    new_pat_data[subpat] = vv
            timestamp = datetime.datetime.now()
            timestamp_str = timestamp.strftime("%Y%m%d%H%M%S.%f")
            if new_pat_data:
                new_json_data[keyword] = new_pat_data
        f = open(eem_json_file, 'w')
        fcntl.lockf(f, fcntl.LOCK_EX)
        f.write(json.dumps(new_json_data, sort_keys=True, indent=4, separators=(',', ': ')))
        fcntl.lockf(f, fcntl.LOCK_UN)
        f.close()
        os.chmod(eem_json_file, 0o777)
        to_ip = pam_edcd.GetVmIP()
        if not on_lindt and len(to_ip) > 0:
            try:
                pam_edcd.ScpFiles(to_ip, edcd_path, eem_json_file)
            except ValueError as verr:
                print("Copy %s to other RPs failed (error: %s)" % (eem_json_file, verr))

        print("\nEDCD has been updated. Please update configuration by running configure event manager pam-agent.")
        print("(Execute 'show edcd eem database' to display all new EDCD entries.)\n")

    elif operation == 'delete-all':
        f = open(eem_json_file, 'w')
        fcntl.lockf(f, fcntl.LOCK_EX)
        f.write("")
        fcntl.lockf(f, fcntl.LOCK_UN)
        f.close()
        os.chmod(eem_json_file, 0o777)
        to_ip = pam_edcd.GetVmIP()
        if not on_lindt and len(to_ip) > 0:
            try:
                pam_edcd.ScpFiles(to_ip, edcd_path, eem_json_file)
            except ValueError as verr:
                print("Copy %s to other RPs failed (error: %s)" % (eem_json_file, verr))

        print("\nAll EEM EDCD are deleted. Please remove all config by running no configure event manager pam-agent <>.")
        print("(Also execute 'show edcd eem database' to verify.)\n")

    elif operation == 'display':
        if type(eem_json_data) != dict or \
             len(eem_json_data) == 0 or \
             keyword not in eem_json_data:
            print ("Entry not found in database")
            exit()

        cmd_json_data = pam_edcd.ReadJsonFile(getattr(args, 'cmd_json_file'))

        data     = eem_json_data[keyword]
        patList  = data.keys()
        print("="*60)
        print("Keyword: '%s'" % keyword)
        print("="*60)
        for pat in patList:
            identifier = data[pat]['identifier']
            _commands = []
            if subpattern is not None and pat != subpattern:
                continue
            identifier = data[pat]['identifier']
            if identifier in cmd_json_data:
                _commands = [x for x in cmd_json_data[identifier]['commands'].split('\n')
                                                                if x.strip() != ""]
            print("-"*60)
            print("Subpattern: '%s':" % pat)
            print("-"*60)
            print("Commands:")
            for command in _commands:
                line = " "*4 + command
                print(line)
            for k in ['identifier', 'tag', 'throttle_window']:
                if k in data[pat] and data[pat][k] != "":
                    if k == 'throttle_window':
                        line = k + " (minutes): " + data[pat][k]
                    else:
                        line = k + ": " + data[pat][k]
                    print(line)
            print("-"*60)

    elif operation == 'display-all':
        cmd_json_data = pam_edcd.ReadJsonFile(getattr(args, 'cmd_json_file'))
        for kw in eem_json_data.keys():
            print("="*60)
            print("Keyword: '%s'" % kw)
            print("="*60)
            for pat in eem_json_data[kw].keys():
                pat_data = eem_json_data[kw][pat]
                identifier = pat_data['identifier']
                _commands = []
                if identifier in cmd_json_data:
                    _commands = [x for x in cmd_json_data[identifier]['commands'].split('\n')
                                                                     if x.strip() != ""]
                print("-"*60)
                print("Subpattern: '%s':" % pat)
                print("-"*60)
                print("Commands:")
                for command in _commands:
                    line = " "*4 + command
                    print(line)
                for k in ['identifier', 'tag', 'throttle_window', 'do_async']:
                    if k in eem_json_data[kw][pat] and eem_json_data[kw][pat][k] != "":
                        if k == 'throttle_window':
                            line = str(k) + " (minutes): " + str(eem_json_data[kw][pat][k])
                        else:
                            line = str(k) + ": " + str(eem_json_data[kw][pat][k])
                        print(line)
            print("")

    elif operation == 'display-eem-log':
        eem_log_file = getattr(args, 'eem_log_file')
        if os.path.exists(eem_log_file):
            with open(eem_log_file) as fd:
                line = fd.readline()
                while line:
                    print(line.strip())
                    line = fd.readline()
        else:
            print("No EEM logs yet.")

    elif operation == 'trigger':
        trigger_eem(args)


def add_ondemand(args):
    edcd_path = getattr(args, 'edcd_path')
    json_file = getattr(args, 'cmd_json_file')
    json_data = pam_edcd.ReadJsonFile(json_file)
    identifier    = getattr(args, 'identifier', '')
    exec_commands = getattr(args, 'exec_commands', '')
    if exec_commands:
        exec_commands = exec_commands.strip()

    new_entry = False
    timestamp = datetime.datetime.now()
    timestamp_str = timestamp.strftime("%Y%m%d%H%M%S.%f")
    if identifier not in json_data:
        json_data[identifier] = {}
        new_entry = True

    if identifier.lower() == "all":
        print("'All' is a reserved identifier and can not be used. Please use another identifier.")
        exit()
    if 'pdb' in sys.modules:
        sys.modules['pdb'].set_trace()
    exec_commands = exec_commands.strip()
    if exec_commands == '':
        print("Misss XR CLI commands")
        exit()
    ShellRegex = r'(?i)^\s*run\s+'
    exec_commands = re.sub(ShellRegex, 'shell ', exec_commands)

    exec_commands = exec_commands.replace(';', '\n')
    exec_commands = exec_commands.replace('\\n', '\n')
    new_exec_commands = [x.strip() for x in exec_commands.split('\n') if x.strip() != '']
    new_shell_commands = []
    final_exec_commands = []
    shellRegex = r'^\s*shell\s+'
    for command in new_exec_commands:
        out = pam_edcd.VerifyCommand(command, on_lindt=False)
        if not out:
            continue

        shellMatches = re.match(shellRegex, command)
        if shellMatches != None:
            command = re.sub(r'(?i)^\s*shell\s+', '', command).strip()
            final_exec_commands.append("run <ssh> \"" + command + "\"")
            new_shell_commands.append("run <ssh> \"" + command + "\"")
        elif not command.startswith("admin"):
            final_exec_commands.append(command)
            new_shell_commands.append("xr_cli \"" + command + "\"")
        else:
            final_exec_commands.append(command)
            command = re.sub(r'(?i)^\s*admin\s+', '', command).strip()
            new_shell_commands.append("admin-cli-proxy-xr_run_cmd \"" + command + "\"")
    new_exec_commands = final_exec_commands

    if 'pdb' in sys.modules:
       sys.modules['pdb'].set_trace()

    field = json_data[identifier]
    if 'created' not in field:
        field['created'] = "Created by PAM agent at " + timestamp_str
    field["modified"] = "Modified by PAM agent at " + timestamp_str

    if len(new_exec_commands) == 0:
        if new_entry:
            print("None of the commands given are valid so no new entry is going to be created for identifier %s" 
                % identifier)
            exit(1)
        else:
            print("None of the commands given are valid so existing entry for identifier %s is not going to be updated" 
                % identifier)
            exit(1)
    _exec_commands = []
    shell_commands = []
    if "commands" in field:
        _exec_commands = field['commands'].replace(';', '\n').split('\n')
        shell_commands = field['shell_commands'].replace(';', '\n').split('\n')
    if _exec_commands == ['']:
        _exec_commands = []
    if shell_commands == ['']:
        shell_commands = []
    for index in range(len(new_exec_commands)):
        command = new_exec_commands[index].strip()
        if command == '' or command in _exec_commands:
            continue
        _exec_commands.append(command)
        shell_commands.append(new_shell_commands[index].strip())
    field['commands'] = '\n'.join(_exec_commands)
    field['shell_commands'] = '\n'.join(shell_commands)

    if getattr(args, 'tag', ''):
        field['tag'] = getattr(args, 'tag', '')
    if getattr(args, 'do_tgz', False):
        field['do_tgz'] = "true"
    if getattr(args, 'do_async', False):
        field['do_async'] = "true"
    if getattr(args, 'logdir', ''):
        logdir = getattr(args, 'logdir', '')
        _regex = r'^(/misc/disk1/|\s*[/]?harddisk:)'
        _matches = re.findall(_regex, logdir)
        if len(_matches) == 0:
            print("\nInvalid path (it must start with /misc/disk1/ - default is /misc/disk1/cisco_support/.\n")
            exit()
        field['logdir'] = logdir
    if getattr(args, 'retainLogs', ''):
        field['retainLogs'] = getattr(args, 'retainLogs', '')
    if getattr(args, 'noRetainLogs', ''):
        field['noRetainLogs'] = getattr(args, 'retainLogs', '')
    if getattr(args, 'debug'):
        for k in ONDEMAND_ATTRS:
            print("field[k=%s]=%s" % (k, field[k]))

    #edcd_path = getattr(args, 'edcd_path')
    if not os.path.exists(edcd_path):
        os.makedirs(edcd_path)
        os.chmod(edcd_path, 0o777)
    f = open(json_file, 'w')
    fcntl.lockf(f, fcntl.LOCK_EX)
    f.write(json.dumps(json_data, sort_keys=True, indent=4, separators=(',', ': ')))
    fcntl.lockf(f, fcntl.LOCK_UN)
    f.close()
    os.chmod(json_file, 0o777)
    to_ip = pam_edcd.GetVmIP()
    if not on_lindt and len(to_ip) > 0:
        try:
            pam_edcd.ScpFiles(to_ip, edcd_path, json_file)
        except ValueError as verr:
            print("Copy %s to other RPs failed (error: %s)" % (json_file, verr))
    print("\nOndemand EDCD has been updated (execute 'show edcd ondemand database' to verify.)\n")


def trigger_ondemand(args):
    edcd_path = getattr(args, 'edcd_path')
    json_file = getattr(args, 'cmd_json_file')
    json_data = pam_edcd.ReadJsonFile(json_file)
    identifier    = getattr(args, 'identifier', '')

    timestamp = datetime.datetime.now()
    timestamp_str = timestamp.strftime("%Y%m%d%H%M%S.%f")
    if identifier.lower() == "all":
        print("'All' is a reserved identifier and can not be used. Please use another identifier.")
        exit()
    if identifier not in json_data:
        print("Unable to find %s in EDCD ondemand database. Please ensure that entry exists is %s."
              % (identifier, json_file))
        exit()
    output = {}
    output[identifier] = json_data[identifier]

    if getattr(args, 'tag', ''):
        output[identifier]['tag'] = getattr(args, 'tag', '')
    if getattr(args, 'do_tgz', False):
        output[identifier]['do_tgz'] = "true"
    if getattr(args, 'do_async', False):
        output[identifier]['do_async'] = "true"
    if getattr(args, 'logdir', ''):
        logdir = getattr(args, 'logdir', '')
        _regex = r'^(/misc/disk1/|\s*[/]?harddisk:)'
        _matches = re.findall(_regex, logdir)
        if len(_matches) == 0:
            print("\nInvalid path (it must start with /misc/disk1/ - default is /misc/disk1/cisco_support/.\n")
            exit()
        output[identifier]['logdir'] = logdir
    if getattr(args, 'retainLogs', ''):
        output[identifier]['retainLogs'] = getattr(args, 'retainLogs', '')
    if getattr(args, 'noRetainLogs', ''):
        output[identifier]['noRetainLogs'] = getattr(args, 'retainLogs', '')
    if getattr(args, 'debug'):
        for k in ONDEMAND_ATTRS:
            print("field[k=%s]=%s" % (k, field[k]))

    triggerfile_path = ONDEMAND_EDCD_PATH + timestamp_str + '-pam_cmd.json'
    if not os.path.exists(ONDEMAND_EDCD_PATH):
        os.makedirs(ONDEMAND_EDCD_PATH)
        os.chmod(ONDEMAND_EDCD_PATH, 0o777)
    f = open(triggerfile_path, 'w')
    fcntl.lockf(f, fcntl.LOCK_EX)
    f.write(json.dumps(output, sort_keys=True, indent=4, separators=(',', ': ')))
    fcntl.lockf(f, fcntl.LOCK_UN)
    f.close()
#    print("\nEDCD action has been triggered. execute 'show edcd ondemand log' to verify.\n")


def CheckCollectionInProgress():
    """
    This function checks if PAM is currently collecting data on the router or not
    """
    try:
        command = "find /misc/disk1/cisco_support/showtech/* -mmin -4"
        p = subprocess.Popen(command, stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE, shell=True)
        output, err = p.communicate()
        regex = r'\S+\.txt'
        file_count = 0
        output = output.split('\n')
        for line in output:
            print(line)
            match = re.match(regex, line, re.I)
            if match:
                file_count += 1
        print("Show Techs still getting collected: %d" % file_count)
    except:
        print("Unable to get any data. Please contact pam-support@cisco.com")


def do_ondemand(args):
    edcd_path = getattr(args, 'edcd_path')
    json_file = getattr(args, 'cmd_json_file')
    json_data = pam_edcd.ReadJsonFile(json_file)

    operation     = getattr(args, 'operation', '')
    identifier    = getattr(args, 'identifier', '')
    exec_commands = getattr(args, 'exec_commands', '')
    if exec_commands:
        exec_commands = exec_commands.strip()

    if operation == "add-update":
        add_ondemand(args)

    elif operation == 'add-and-trigger':
        add_ondemand(args)
        trigger_ondemand(args)

    elif operation == "add-replace":
#       TODO
#       add_replace_ondemand(args)
        print("Not implemented yet")
        pass

    elif operation == 'delete':
        timestamp = datetime.datetime.now()
        timestamp_str = timestamp.strftime("%Y%m%d%H%M%S.%f")
        if identifier not in json_data:
            print("identifier '%s' not found in EDCD - skip updating EDCD." % identifier)
            exit()
        new_json_data = {}
        new_cmd_data = {}
        id_data = json_data[identifier]

        if exec_commands == '' or exec_commands is None:
            for k, v in json_data.iteritems():
                if identifier == k:
                    continue
                new_json_data[k] = v
        if exec_commands != '' and exec_commands is not None:
            for k, v in json_data.iteritems():
                if identifier != k:
                    new_json_data[k] = v

            shellRegex = r'^\s*shell\s+'
            _exec_commands = []
            old_exec_commands = id_data['commands'].split('\n')
            exec_commands = exec_commands.replace(';', '\n')
            exec_commands = exec_commands.replace('\\n','\n')
            delete_exec_commands = []
            for cmd in exec_commands.split('\n'):
                cmd = cmd.strip()
                if cmd == '':
                    continue
                shellMatches = re.match(shellRegex, cmd)
                cmd = re.sub(r'(?i)<ssh>\s+', ' ', cmd)
                if shellMatches != None:
                    cmd = re.sub(r'(?i)^\s*shell\s+', '', cmd)
                    delete_exec_commands.append("run <ssh> \"" + cmd + "\"")
                else:
                    out = pam_edcd.VerifyCommand(cmd, on_lindt)
                    if not out:
                        continue
                    delete_exec_commands.append(cmd)
            _exec_commands = [c for c in old_exec_commands if c not in delete_exec_commands]
            new_cmd_data = {}
            if len(_exec_commands):
                new_cmd_data = {}
                timestamp = datetime.datetime.now()
                timestamp_str = timestamp.strftime("%Y%m%d%H%M%S.%f")
                new_shell_commands = []
                new_cmd_data['commands'] = ('\n').join(_exec_commands)
                for command in _exec_commands:
                    new_shell_commands.append("xr_cli \"" + command + "\"")
                    new_cmd_data['shell_commands'] = '\n'.join(new_shell_commands)
                for kk, vv in id_data.iteritems():
                    if kk == 'commands' or kk == 'shell_commands':
                        continue
                    elif kk == 'created' or kk == 'modified':
                        new_cmd_data["modified"] = "Modified by PAM agent at " + timestamp_str
                    else:
                        new_cmd_data[kk] = vv
        if new_cmd_data:
            new_json_data[identifier] = new_cmd_data
        if getattr(args, 'debug'):
            print json.dumps(new_json_data, indent=1)
        f = open(json_file, 'w')
        fcntl.lockf(f, fcntl.LOCK_EX)
        f.write(json.dumps(new_json_data, sort_keys=True, indent=4, separators=(',', ': ')))
        fcntl.lockf(f, fcntl.LOCK_UN)
        f.close()
        os.chmod(json_file, 0o777)
        to_ip = pam_edcd.GetVmIP()
        if not on_lindt and len(to_ip) > 0:
            try:
                pam_edcd.ScpFiles(to_ip, edcd_path, json_file)
            except ValueError as verr:
                print("Copy %s to other RPs failed (error: %s)" % (json_file, verr))
        print("\nOndemand EDCD has been updated (execute 'show edcd ondemand database' to verify.)\n")
        exit()

    elif operation == 'delete-all':
        f = open(json_file, 'w')
        fcntl.lockf(f, fcntl.LOCK_EX)
        f.write("")
        fcntl.lockf(f, fcntl.LOCK_UN)
        f.close()
        os.chmod(json_file, 0o777)
        to_ip = pam_edcd.GetVmIP()
        if not on_lindt and len(to_ip) > 0:
            try:
                pam_edcd.ScpFiles(to_ip, edcd_path, json_file)
            except ValueError as verr:
                print("Copy %s to other RPs failed (error: %s)" % (json_file, verr))
        print("\nAll ondemand EDCD are deleted (execute 'show edcd ondemand database' to verify.)\n")

    elif operation == 'display':
        if identifier.lower() == "all":
            keys = list(json_data.keys())
            if len(keys) == 0 or (len(keys) == 1 and "all" in keys):
                print ("Entry not found in database")
            else:
                print("="*60)
                print(" " + "EDCD Identifier" + " " * 20 + "CLI Count")
                print("="*60)
                count = 1
                for iden in keys:
                    if iden.lower() == "all":
                        continue
                    space = " "
                    if count >= 10:
                        space = ""
                    additional_spaces = 40 - len(iden)
                    if additional_spaces <= 0:
                        additional_spaces = 1
                    traces_length = str(len(json_data[iden]["commands"].split('\n'))) \
                                                if "commands" in json_data[iden] else 0
                    print(space + str(count) + ": " + iden + " " * additional_spaces
                                                    + str(traces_length))
                    count += 1
                print("-"*60)
        else:
            print("="*60)
            print(" "*15 + "Identifier: " + identifier)
            print("="*60)
            count = 1
            if identifier not in json_data:
                print("No entries found for identifier '%s'" % identifier)
                exit()
            if not getattr(args, 'print_shell_commands', False):
                commands = [x for x in json_data[identifier]["commands"].split("\n")
                                if x.strip() != ""]
            else:
                commands = [x for x in json_data[identifier]["shell_commands"].split("\n")
                                if x.strip() != ""]
            for command in commands:
                space = " "
                if count >= 10:
                    space = ""
                print(space + str(count) + ": " + command)
                count += 1
            print("-"*60)

    elif operation == 'display-all':
        keys = list(json_data.keys())
        if len(keys) == 0 or (len(keys) == 1 and "all" in keys):
            print ("No entry found in database")
            exit()
        display_edcd_cmd_data(keys)

    elif operation == 'collection-status':
        CheckCollectionInProgress()

    elif operation == 'trigger':
        trigger_ondemand(args)


def add_pam(args):
    edcd_path = getattr(args, 'edcd_path')
    event = getattr(args, 'event')
    if event in ('syslog'):
        json_file = getattr(args, 'pattern_json_file')
    else:
        json_file = getattr(args, 'event_json_file')
    json_data = pam_edcd.ReadJsonFile(json_file)
    identifier = getattr(args, 'identifier', '')
    process = getattr(args, 'process', '')
    pattern = getattr(args, 'pattern', '')
    timestamp = datetime.datetime.now()
    timestamp_str = timestamp.strftime("%Y%m%d%H%M%S.%f")
    if platform not in json_data:
        json_data[platform] = {}
    if operSys not in json_data[platform]:
        json_data[platform][operSys] = {}
    field = json_data[platform][operSys]

    if event in ('syslog'):
        if pattern not in field:
            field[pattern] = {}
        field = field[pattern]
    else:
        if event not in field:
            field[event] = {}
        field = field[event]
        if process not in field:
            field[process] = {}
        field = field[process]

    for key in EDCD_VARIABLES:
        if key not in field:
            field[key] = ""
    field['identifier'] = identifier
    if not field['created']:
        field['created'] = "Created by PAM agent at " + timestamp_str
    field["modified"] = "Modified by PAM agent at " + timestamp_str

    if not os.path.exists(edcd_path):
        os.makedirs(edcd_path)
        os.chmod(edcd_path, 0o777)
    f = open(json_file, 'w')
    fcntl.lockf(f, fcntl.LOCK_EX)
    f.write(json.dumps(json_data, sort_keys=True, indent=4, separators=(',', ': ')))
    fcntl.lockf(f, fcntl.LOCK_UN)
    f.close()
    os.chmod(json_file, 0o777)
    to_ip = pam_edcd.GetVmIP()
    if not on_lindt and len(to_ip) > 0:
        try:
            pam_edcd.ScpFiles(to_ip, edcd_path, json_file)
        except ValueError as verr:
            print("Copy %s to other RPs failed (error: %s)" % (json_file, verr))
    print("\nPAM built-in EDCD has been updated (execute 'show edcd <>' to verify.)\n")


def delete_pam(args):
    edcd_path = getattr(args, 'edcd_path')
    event = getattr(args, 'event')
    if event in ('syslog'):
        json_file = getattr(args, 'pattern_json_file')
    else:
        json_file = getattr(args, 'event_json_file')
    json_data = pam_edcd.ReadJsonFile(json_file)
    identifier = getattr(args, 'identifier', '')
    process = getattr(args, 'process', '')
    pattern = getattr(args, 'pattern', '')
    timestamp = datetime.datetime.now()
    timestamp_str = timestamp.strftime("%Y%m%d%H%M%S.%f")
    if platform not in json_data:
        json_data[platform] = {}
    if operSys not in json_data[platform]:
        json_data[platform][operSys] = {}
    field = json_data[platform][operSys]

    if event in ('syslog'):
        if pattern not in field:
            print("pattern '%s' not found in '%s'. Skip deleting" %
                    (pattern, json_file))
            exit()
        json_data[platform][operSys].pop(pattern, None)
    else:
        if event not in field:
            print("event '%s' not found in '%s'. Skip deleting" %
                        (pattern, json_file))
            exit()
        if process is not None and process not in field[event]:
            print("process '%s' not found in '%s'. Skip deleting" %
                    (pattern, json_file))
            exit()

        if event != '' and process is None:
            json_data[platform][operSys].pop(event, None)
        elif event != '' and process is not None:
            json_data[platform][operSys][event].pop(process, None)
            if not json_data[platform][operSys][event]:
                json_data[platform][operSys].pop(event, None)

    if not json_data[platform][operSys]:
        json_data[platform].pop(operSys, None)
    if not json_data[platform]:
        json_data.pop(platform, None)

    if not os.path.exists(edcd_path):
        os.makedirs(edcd_path)
        os.chmod(edcd_path, 0o777)
    f = open(json_file, 'w')
    fcntl.lockf(f, fcntl.LOCK_EX)
    f.write(json.dumps(json_data, sort_keys=True, indent=4,
                        separators=(',', ': ')))
    fcntl.lockf(f, fcntl.LOCK_UN)
    f.close()
    os.chmod(json_file, 0o777)
    to_ip = pam_edcd.GetVmIP()
    if not on_lindt and len(to_ip) > 0:
        try:
            pam_edcd.ScpFiles(to_ip, edcd_path, json_file)
        except ValueError as verr:
            print("Copy %s to other RPs failed (error: %s)" % (json_file, verr))
    print("\nPAM build-in EDCD has been updated (execute 'show edcd <>' to verify.)\n")


def display_pam_pattern(args):
    pam_json_data = pam_edcd.ReadJsonFile(getattr(args, 'pattern_json_file'))
    cmd_json_data = pam_edcd.ReadJsonFile(getattr(args, 'cmd_json_file'))

    pattern = getattr(args, 'pattern')
    platform = pam_edcd.get_platform()
    os = pam_edcd.get_os_type()

    pat_data = {}
    prev_pat_len = 0
    pat = ''
    for _plt, v in pam_json_data.iteritems():
        if _plt not in ('All', 'ALL', platform):
            continue
        for _os, vv in v.iteritems():
            if _os not in ('All', 'ALL', os):
                continue
            for _pat, vvv in vv.iteritems():
                _regex = r'' + _pat
                _matches = re.findall(_regex, pattern)
                if len(_matches) == 0:
                    continue
#               take the longest match
                if len(_pat) > prev_pat_len:
                    pat == _pat
                    prev_pat_len = len(_pat)
                    pat_data = vvv

    if pat_data:
        print("="*60)
        print("syslog pattern='%s'" % pattern)
        print("="*60)
        identifier = pat_data['identifier']
        _commands = []
        print("-"*60)
        print("identifier: '%s':" % identifier)
        print("-"*60)
        if identifier in cmd_json_data:
            _commands = [x for x in cmd_json_data[identifier]['commands'].split('\n')
                                    if x.strip() != ""]
        print("Commands:")
        for command in _commands:
            line = " "*4 + command
            print(line)
            for k in ['tag', 'throttle_window']:
                if k in pat_data and pat_data[k] != "":
                    if k == 'throttle_window':
                        line = k + " (minutes): " + pat_data[k]
                    else:
                        line = k + ": " + pat_data[k]
                    print(line)
        print("-"*60)


def display_pam_pattern_all(args):
    edcd_path = getattr(args, 'edcd_path')
    pam_json_data = pam_edcd.ReadJsonFile(getattr(args, 'pattern_json_file'))
    cmd_json_data = pam_edcd.ReadJsonFile(getattr(args, 'cmd_json_file'))

    platform = pam_edcd.get_platform()
    os = pam_edcd.get_os_type()

    for _plt, v in pam_json_data.iteritems():
        if _plt not in ('All', 'ALL', platform):
            continue
        for _os, vv in v.iteritems():
            if _os not in ('All', 'ALL', os):
                continue
            for _pat, vvv in vv.iteritems():
                pat_data = vvv
                if pat_data:
                    print("="*60)
                    print("syslog pattern='%s'" % _pat)
                    print("="*60)
                    identifier = pat_data['identifier']
                    _commands = []
                    print("-"*60)
                    print("identifier: '%s':" % identifier)
                    print("-"*60)
                    if identifier in cmd_json_data:
                        _commands = [x for x in cmd_json_data[identifier]['commands'].split('\n')
                                    if x.strip() != ""]
                    print("Commands:")
                    for command in _commands:
                        line = " "*4 + command
                        print(line)
                        for k in ['tag', 'throttle_window']:
                            if k in pat_data and pat_data[k] != "":
                                if k == 'throttle_window':
                                    line = k + " (minutes): " + pat_data[k]
                                else:
                                    line = k + ": " + pat_data[k]
                                print(line)
                    print("-"*60)
                print("")


def display_pam_process(args):
    pam_json_data = pam_edcd.ReadJsonFile(getattr(args, 'event_json_file'))
    cmd_json_data = pam_edcd.ReadJsonFile(getattr(args, 'cmd_json_file'))

    event = getattr(args, 'event')
    process = getattr(args, 'process')
    platform = pam_edcd.get_platform()
    os = pam_edcd.get_os_type()
    process_data = {}
    prev_match_cnt = 0
    for _plt, v in pam_json_data.iteritems():
        match_cnt = 0
        if _plt not in ('All', 'ALL', platform):
            continue
        match_cnt += 2 if _plt == platform else match_cnt + 1
        for _os, vv in v.iteritems():
            if _os not in ('All', 'ALL', os):
                continue
            for _evt, vvv in vv.iteritems():
                if _evt not in ('All', 'ALL', event):
                    continue
                for _pr, vvvv in vvv.iteritems():
                    if _pr not in ('All', 'ALL', process):
                        continue
                    match_cnt += 4 if _pr == process else match_cnt + 3
#                    take the closest match
#                          process match overrides 'All' process,
#                          process match overrides platform,
#                          platform match overrides 'All' platform,
                    if match_cnt > prev_match_cnt:
                        prev_match_cnt = match_cnt
                        process_data = vvvv

    if process_data:
        print("="*60)
        print("event='%s', process='%s'" % (event, process))
        print("="*60)
        identifier = process_data['identifier']
        _commands = []
        print("-"*60)
        print("identifier: '%s'" % identifier)
        print("-"*60)
        if identifier in cmd_json_data:
            _commands = [x for x in cmd_json_data[identifier]['commands'].split('\n')
                        if x.strip() != ""]
        print("Commands:")
        for command in _commands:
            line = " "*4 + command
            print(line)
            for k in ['tag', 'throttle_window']:
                if k in process_data and process_data[k] != "":
                    if k == 'throttle_window':
                        line = k + " (minutes): " + process_data[k]
                    else:
                        line = k + ": " + process_data[k]
                    print(line)
        print("-"*60)


def display_pam_process_all(args):
    pam_json_data = pam_edcd.ReadJsonFile(getattr(args, 'event_json_file'))
    cmd_json_data = pam_edcd.ReadJsonFile(getattr(args, 'cmd_json_file'))

    platform = pam_edcd.get_platform()
    os = pam_edcd.get_os_type()
    for _plt, v in pam_json_data.iteritems():
        if _plt not in ('All', 'ALL', platform):
            continue
        for _os, vv in v.iteritems():
            if _os not in ('All', 'ALL', os):
                continue
            for _evt, vvv in vv.iteritems():
                for _pr, vvvv in vvv.iteritems():
                    process_data = vvvv
                    if not process_data:
                        continue
                    if 'identifier' not in process_data:
                        continue
                    print("="*60)
                    print("event='%s', process='%s'" % (_evt, _pr))
                    print("="*60)
                    identifier = process_data['identifier']
                    _commands = []
                    print("-"*60)
                    print("identifier: '%s'" % identifier)
                    print("-"*60)
                    if identifier in cmd_json_data:
                        _commands = [x for x in cmd_json_data[identifier]['commands'].split('\n')
                                    if x.strip() != ""]
                        print("Commands:")
                        for command in _commands:
                            line = " "*4 + command
                            print(line)
                            for k in ['tag', 'throttle_window']:
                                if k in process_data and process_data[k] != "":
                                    if k == 'throttle_window':
                                        line = k + " (minutes): " + process_data[k]
                                    else:
                                        line = k + ": " + process_data[k]
                                    print(line)
                        print("-"*60)
                    print("")


def do_pam(args):
    edcd_path = getattr(args, 'edcd_path')
    operation = getattr(args, 'operation', '')
    event = getattr(args, 'event')
    if event in ('syslog'):
        json_file = getattr(args, 'pattern_json_file')
    else:
        json_file = getattr(args, 'event_json_file')
    json_data = pam_edcd.ReadJsonFile(json_file)

    identifier = getattr(args, 'identifier', '')
    process = getattr(args, 'process', '')
    pattern = getattr(args, 'pattern', '')

    if operation == 'add-update':
        add_pam(args)

    elif operation == 'delete':
        delete_pam(args)

    elif operation == 'delete-all':
        pattern_json_file = getattr(args, 'pattern_json_file')
        event_json_file = getattr(args, 'event_json_file')
        for _json_file in (pattern_json_file, event_json_file):
            f = open(_json_file, 'w')
            fcntl.lockf(f, fcntl.LOCK_EX)
            f.write("")
            fcntl.lockf(f, fcntl.LOCK_UN)
            f.close()
            os.chmod(_json_file, 0o777)
            to_ip = pam_edcd.GetVmIP()
            if not on_lindt and len(to_ip) > 0:
                try:
                    pam_edcd.ScpFiles(to_ip, edcd_path, _json_file)
                except ValueError as verr:
                    print("Copy %s to other RPs failed (error: %s)" % (_json_file, verr))
        print("\nAll EDCD entries are deleted (execute 'show edcd <>' to verify.)\n")

    elif operation == 'display':
        if event in ('syslog'):
            display_pam_pattern(args)
        else:
            display_pam_process(args)

    elif operation == 'display-all':
        display_pam_pattern_all(args)
        display_pam_process_all(args)

    elif operation == 'display-all-pattern':
        display_pam_pattern_all(args)

    elif operation == 'display-all-event':
        display_pam_process_all(args)

if __name__ == '__main__':
    args = pam_edcd.InitializeArgs()
    json_data = pam_edcd.ReadJsonFile(getattr(args, 'cmd_json_file'))
    rc = pam_edcd.VerifyInputs(args, json_data)
    if not rc:
        exit()

    platform = pam_edcd.get_platform()
    if platform in ('spitfire', 'ncs540l'):
        os.environ["LD_LIBRARY_PATH"] = \
            "/pkg/lib:/pkg/lib/cerrno:/pkg/lib/mib:/pkg/lib/spp_plugins"
        on_lindt = True

    if getattr(args, 'debug'):
        DEBUG = True
    if 'pdb' in sys.modules:
        sys.modules['pdb'].set_trace()
    mode = getattr(args, 'mode')
    if mode == 'ondemand':
        rc = do_ondemand(args)
    if mode == 'pam':
        rc = do_pam(args)
    if mode == 'eem':
        rc = do_eem(args)

    exit()
#   sample tests
    print("displaying EDCD data") 
    identifiers = ['12345', 'bgp-showtech']
    display_edcd_cmd_data(identifiers)



