 #------------------------------------------------------------------
 # CLI Tcl Library: execute CLI commands in Tcl.
 #
 # March, 2000, Yu Zhang
 #
 # Copyright (c) 2000-2001, 2005-2007, 2009-2015 by cisco Systems, Inc.
 # All rights reserved.
 #------------------------------------------------------------------

namespace eval ::cisco::eem {
    namespace export cli_open cli_exec cli_close
    namespace export cli_write cli_read cli_read_pattern 
    namespace export cli_read_line cli_read_drain
    namespace export cli_get_ttyname
}

### EEM #######################################################################
namespace eval ::cisco::eem {
    namespace export cli_open cli_exec cli_close
    namespace export cli_write cli_read cli_read_pattern 
    namespace export cli_read_line cli_read_drain
    namespace export cli_get_ttyname
}

proc ::cisco::eem::cli_open {} {
    return [::cisco::eem::cli_open]
}
proc ::cisco::eem::cli_write {fd cmd} {
    return [::cisco::eem::cli_write $fd $cmd] 
}
proc ::cisco::eem::cli_read_pattern {fd ptn} {
    return [::cisco::eem::cli_read_pattern $fd $ptn]
}
proc ::cisco::eem::cli_read {fd} {
    return [::cisco::eem::cli_read $fd]
}
proc ::cisco::eem::cli_exec {fd cmd} {
    return [::cisco::eem::cli_exec $fd $cmd]
}
proc ::cisco::eem::cli_close {fd tty_id} {
    return [::cisco::eem::cli_close $fd $tty_id]
}
proc ::cisco::eem::cli_read_line {fd} {
    return [::cisco::eem::cli_read_line $fd]
}
proc ::cisco::eem::read_drain {fd} {
    return [::cisco::eem::read_drain $fd]
}
proc ::cisco::eem::cli_read_drain {fd} {
    return [::cisco::eem::cli_read_drain $fd]
}
proc ::cisco::eem::cli_get_ttyname {tty_id} {
    return [::cisco::eem::cli_get_ttyname $tty_id]
}

###############################################################################

# a typical usage of cli library:
#   set result [cli_open]
#   array set cli1 $result
#   set result [cli_exec $cli1(fd) "en"]
#   process result...
#   set result [cli_exec $cli1(fd) "config"]
#   set result [cli_exec $cli1(fd) "interface Ethernet1/0"]
#   set result [cli_exec $cli1(fd) "no shut"]
#   set result [cli_exec $cli1(fd) "end"]
#   cli_close $cli1(fd) $cli1(tty_id)

# or

#   set result [cli_open]
#   array set cli1 $result
#   cli_write $cli1(fd) "en"
#   cli_write $cli1(fd) "show run"
# next time before cli_exec:
#   cli_read_drain $cli1(fd)
#   set result [cli_exec $cli1(fd) "show run"]
#   cli_close $cli1(fd) $cli1(tty_id)


# Spawn an exec, open a channel to it and return the slave side
# channel handler. 
# Possible error raised:
# 1. cannot get pty for exec
# 2. cannot spawn exec
# 3. error reading the first prompt

proc ::cisco::eem::cli_open {} {
    global _cerrno _cerr_sub_num _cerr_sub_err _cerr_posix_err _cerr_str


    # Connect the tty to exec
    if [catch {::cisco::eem::tty_connect } result] {
	# puts [format "Result - 2 = %d" $result]
	return -code error "cannot get pty for exec: $result"
    } else {
	array set arr1 $result
	# returns:
	#   tty_id = internal tty ID
        #   tty_fd = tty fd (channel name)
    }

#    if [catch {open $arr1(tty_name) {RDWR}} result] {
#        return -code error "cannot open pty for exec: $result"
#   } else {
#        set arr1(fd) $result
#   }

    set arr1(fd) $arr1(tty_fd)
    # both read and write won't be buffered
    fconfigure $arr1(fd) -buffering none
    # non-blocking read and write
    fconfigure $arr1(fd) -blocking 0

    # drain the first prompt
    if [catch {cli_read $arr1(fd)} result] {
	# puts [format "Result - 3 = %d" $result]
	return -code error "error reading the first prompt: $result"
    } else {
	# Set the terminal length to 0
	if [catch {cli_exec $arr1(fd) "term length 0"} result] {
	    return -code error "error reading the channel: $result"
	}

        return [array get arr1]
	# arr1 contains
	#   tty_id = tty ID
	#   tty = tty device name
	#   fd = open Tcl FD
    }
}

# Write the command 'cmd' to the channel whose handler is given
# by 'fd' to execute the command.
# Possible error raised:
# 	None

proc ::cisco::eem::cli_write {fd cmd} {
    puts $fd $cmd
    return
}

# Read the command output from the CLI channel whose handler is given 
# by 'fd' until the pattern 'ptn' occurs in the contents read. Return 
# all the contents read up to the match.
# Possible error raised:
#  	None

proc ::cisco::eem::cli_read_pattern {fd ptn} {
    set result ""
    set is_end 0
    set str1 ""
    set i 1
# lets not spin forever. try for maximum 50000 times
    while {$is_end == 0 && $i<=50000 } {
        after 50
        set str [read $fd]
        if {$str == ""} {
            incr i;
            after 50
            continue
        } else {
                append result $str
        }
        # double quotes (don't change to curly braces!)
        set is_end [regexp "(.*)?($ptn)" $result str1]
    }
    return $result
}

# Read the command output from the CLI channel whose handler is given 
# by 'fd' until the pattern of the router prompt occurs in the contents 
# read. Return all the contents read up to the match.
# Possible error raised:
# 1. cannot get router name

proc ::cisco::eem::cli_read {fd} {
    global _cerrno _cerr_sub_num _cerr_sub_err _cerr_posix_err _cerr_str

    set routername [info hostname]
    if {[string match "" $routername]} {
        return -code error "Host name is not configured"
    }
    set prompt [format "(\\(((admin-)?config\[^\n\]*|admin)\\))|%s(#|>)" $routername]
    set result [::cisco::eem::cli_read_pattern $fd $prompt]
    return $result
}

proc ::cisco::eem::cli_exec_read_drain {fd } {
    global _cerrno _cerr_sub_num _cerr_sub_err _cerr_posix_err _cerr_str
    set result ""
    set is_end 0
    set str1 ""
    set i 1

    #set router prompt
    set routername [info hostname]
    if {[string match "" $routername]} {
        return -code error "Host name is not configured"
    }
    set ptn [format "(\\(((admin-)?config\[^\n\]*|admin)\\))|%s(#|>)" $routername]
    
    #read till the first router promt is received or  worst case read twice
    #incase router promt not found.
    while {$is_end == 0 && $i<=2 } {
        set str [read $fd]        
        set is_end [regexp ".*?$ptn" $str str1]
        if {$is_end} {
            append result $str1
            return $result
        } else {
            append result $str
        }
        after 100
        incr i
    }
    return $result    
}

# Write the command 'cmd' to the channel whose handler is given 
# by 'fd' to execute the command. Then read the output of the
# command from the channel and return the output. 
# Possible error raised:
# 1. error reading the channel

proc ::cisco::eem::cli_exec {fd cmd} {
    # Drain the FD before executing the command, so we don't pollute the result with old data
    set drain_result [cli_exec_read_drain $fd]
    #puts "drain_result:$drain_result"

    cli_write $fd $cmd
    if [catch {cli_read $fd} result] {
        return -code error "error reading the channel: $result"
    } else {
        return $result
    }
}

# Close the exec process and the CLI channel connected to it given
# the handler of the channel 'fd'.
# Possible error raised:
# 1. cannot close channel

proc ::cisco::eem::cli_close {fd tty_id} {
    # Drain the FD before closing it
    set drain_result [cli_read_drain $fd]

    cli_write $fd "exit"
    after 100
    if [catch {::cisco::eem::tty_disconnect $tty_id} result] {
	return -code error "cannot close the channel: $result" 
    }
    if [catch {close $fd} result] {
#	return -code error "cannot close the channel: $result" 
    }
}

# Read one line of the command output from the CLI channel whose handler 
# is given by 'fd'. Return the line read.
# Possible error raised:
# None

proc ::cisco::eem::cli_read_line {fd} {
    set result ""
    set is_end 0
    set str1 ""
    set i 1
# lets not spin forever. try for maximum 500 times
    while {$is_end == 0 && $i<=500 } {
        set str [read $fd]
        set is_end [regexp {[^\n](.*)\n} $str - str1]
        append result $str1
        after 50
        incr i
    }
    return $result
}

proc ::cisco::eem::read_drain {fd} {
    set result ""
    for {set i 0} {$i < 500} {incr i} {
	    set str [read $fd]
        append result $str
        after 1

# no need to check this line because automore is disabled for exec
#        if {[string match "*--More--*" $str]} {
#            puts -nonewline $fd " "
#        }
    }

    return $result
}



# Read and drain the command output of the CLI channel whose handler 
# is given by 'fd'. Return all the contents read.
# Possible error raised:
#	None

proc ::cisco::eem::cli_read_drain { fd { wait_time 200 } } {
    fileevent $fd readable [set result [::cisco::eem::read_drain $fd]]
    after $wait_time
    fileevent $fd readable 
    return $result
} 

# Return both the real and pseudo tty names for this cli connection.
# Possible error raised:
# 1. cannot get tty name

proc ::cisco::eem::cli_get_ttyname {tty_id} {
    if [catch {::cisco::eem::tty_get_ttyname $tty_id} result] {
	return -code error "cannot get tty name: $result"
    }
    array set arr1 $result
    return [array get arr1]
    # arr1 contains
    #   pty = pty device name
    #   tty = tty device name
} 








