#!/usr/bin/perl

# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
# Agent for moniroting show logging
# 
# Feb 2016, Jieming Wang
# 
# Copyright (c) 2016-2019 by Cisco Systems, Inc.
# All rights reserved.
# xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx


use lib qw ( /pkg/opt/cisco/pam/ /opt/cisco/calvados/pam/ /opt/pam/ );

use pam;
use strict;

sub get_old_show_log($);
sub get_new_xr_show_logging($$$);
sub get_new_calv_show_log($$);

use Getopt::Std;
use Date::Calc qw(:all);
use vars qw/ %opt /;
use strict;
use warnings;
use File::Basename;
use File::Copy;
use Logfile::Rotate;

use Expect;

my $month_map = {
    'Jan' => 1,
    'Feb' => 2,
    'Mar' => 3,
    'Apr' => 4,
    'May' => 5,
    'Jun' => 6,
    'Jul' => 7,
    'Aug' => 8,
    'Sep' => 9,
    'Oct' => 10,
    'Nov' => 11,
    'Dec' => 12,
};
my $month_pat = "";
foreach my $mon (keys %{$month_map}) {
    $month_pat .= "|" . $mon;
}
$month_pat =~ s/^\|//; $month_pat =~ s/^/(/;
$month_pat =~ s/\|$//; $month_pat =~ s/$/)/;

##########################################
# parameters
##########################################
my ($master_pam_conf, $master_conf_info, $ret);

my ($sys_info,
    $osType,
    $boardtype,
    $platform,
    $vf1_3073_ip,
    $version);

my ($cpu_arch_dir);
my ($localtime, $node_info);

my $pid_dir = "/opt/cisco/pam/run/";
my $pid_file = $pid_dir . "/pid";

$master_pam_conf = "/opt/pam/pam.conf";
my $log_dir = get_PamLogDir();
mkdir $log_dir if (! -d $log_dir);
my $bucket = "monitor_agent";
my $event_type = "show_logging";
my $local_log_arch_dir = $log_dir . "/event_track/";
mkdir $local_log_arch_dir if (! -d $local_log_arch_dir );
#add timestamp, and cleanup old ones
my $logFile = $local_log_arch_dir . "/" . "known_logs.txt";
my $show_tech_root = "/misc/disk1/showtech/";

getopts( "I:d", \%opt );

my $debug = $opt{d};
my $refresh_interval =  $opt{I} || 600;

$sys_info = &getOsType();
$osType = $sys_info->{hostType};
$version = &getWS($osType);
$boardtype = $sys_info->{boardtype};
$platform = $sys_info->{platform};
#local eth-vf1.3073 IP address:
$vf1_3073_ip = $sys_info->{vf1_3073_ip};

if ($boardtype !~ /R[S]?P|CC/i) {
    print "$0 can only run on RP.\n";
    exit(1);
}
my $mode = "exec";
if ($osType =~ /calvados|sysadmin/i) {
    $mode = "sysadmin";
}

my $my_pid = $$;

my $this = $0;
my $bname = $this;
$bname =~ s/.*\///g;

########################################
#make sure only one instance running
########################################
$ret = &check_process($bname, $sys_info, $my_pid);
if ((scalar(@$ret)) > 0) {
    print "process $bname (pid=@$ret) is already running.\n";
    exit(1);
}

#Set LD_LIBRARY_PATH to make sure it will not use non-system libray
if ( $osType =~ /xr/i) {
    $ENV{LD_LIBRARY_PATH} = "/pkg/lib:/pkg/lib/cerrno:";
    $ENV{LD_LIBRARY_PATH} .= "/pkg/lib/mib:/pkg/lib/spp_plugins";
} elsif ( $osType =~ /calvados/i) {
    $ENV{LD_LIBRARY_PATH} = "";
}

my @old_local_logs = ();

my $log_info = &get_old_show_log($logFile);
my $last_timestamp = $log_info->{timestamp};

umask 0002;

$SIG{'TERM'} = 'INT_handler';
my $min_free_threshold = 307200; #300MB:
#Limit its own memory usage - perl doesn't free memory!
#it will be be restarted!

# cap memory usage!
my $max_threshold = 30000; #30 MB?
my $memfree = get_memfree();
#Only start when there is enough memory to prevent exhausting memory
if ($memfree < $min_free_threshold) {
    #free cache on calvados in case its too low:
    if ($osType =~ /calvados/) {
        `sysctl vm.drop_caches=3`;
    }
    $memfree = get_memfree();
}
if ($memfree < $min_free_threshold) {
    sleep 60;
    unlink $pid_file if (-f $pid_file);
    exit;
}

my @alert_patterns = ();

my $minimum_refresh_interval = 30;
my $prev_mtime = 0;
my ($log_conf);

my ($sbt_node_pid, @sbt_node_pids, %isKnownSbt, $dlls);
my (%isKnownPat);
my @oldLogs = ();
if (defined($log_info->{oldLogs})) {
    @oldLogs = @{$log_info->{oldLogs}};
}
my $sbt = get_sbt(\@oldLogs);
my $sbt_node_pids = @{$sbt->{node_pids}};
foreach $sbt_node_pid (@sbt_node_pids) {
    if (!$isKnownSbt{$sbt_node_pid}) {
        $isKnownSbt{$sbt_node_pid} = 1;
    }
}

my $version_content = &getVersionContent("/");
my (@newLogs, @allLogs);

#if ($osType =~ /xr/i) {
#    $routername = `/pkg/bin/node_list_generation -f MY`;
#}
my $chassis_id = get_chassis_id($sys_info);
if ($chassis_id !~ /^\d+$/) {
    print "Invalid chassis ID: $chassis_id (expect integer).\n";
    exit(1);
}

$node_info = get_active_rp_nodes_by_chassis_id($sys_info, $chassis_id);

my $wsfd;

my %isCmdSeen;
#Need both JSON files:
#User defined EDCD files are under /opt/pam/etc/:
my $user_edcd_dir = "/opt/pam/etc/";
my @pam_event_db_files=('/opt/cisco/pam/pam_event.json');
my @pam_pattern_db_files=('/opt/cisco/pam/pam_pattern.json');
if ($osType =~ /xr/i) {
    @pam_event_db_files=('/pkg/opt/cisco/pam/pam_event.json');
    @pam_pattern_db_files=('/pkg/opt/cisco/pam/pam_pattern.json');
}
my $edcd_info = get_user_pattern_files($user_edcd_dir);
my @user_event_files = @{$edcd_info->{event_files}};
my @user_pattern_files = @{$edcd_info->{pattern_files}};
@pam_event_db_files=(@pam_event_db_files, @user_event_files);
@pam_pattern_db_files=(@pam_pattern_db_files, @user_pattern_files);

#in EDCD, there are 2 type of commands: commands, and shell_commands
#take shell command for XR (for now)
my $cmd_type = "shell_commands";
if ((($osType =~ /xr/i) && (-f "/pkg/bin/xr_cli")) ||
     ($osType =~ /calvados|sysadmin/i)) {
    $cmd_type = "commands";
}
my @default_commands = ('show_logging',
                        'ng_show_version',
                        'sdr_instcmd show install active',
                        'show_platform_sysdb');
if ($osType =~ /calvados|sysadmin/i) {
    $cmd_type = "commands";
    @default_commands = ('show logging',
                         'show version',
                         'show install active',
                         'show platform');
}
my $pat_cmd_maps = get_edcd_pattern_commands($platform,
                                             $osType,
                                             \@pam_pattern_db_files);
my $cmd_map_info = $pat_cmd_maps->{$cmd_type};
my @log_patterns= (@{$cmd_map_info->{patterns}});

@newLogs = ();
unless (fork) {
    $my_pid = $$;
    while (1) {
        my $showtech_cnt = 0;
        $memfree = get_memfree();
        if ($memfree < $min_free_threshold) {
            #free cache on calvados in case its too low:
            if ($osType =~ /calvados/) {
                `sysctl vm.drop_caches=3`;
            }
            $memfree = get_memfree();
        }
        if ($memfree < $min_free_threshold) {
            #restart to free memory (as perl cannot free)
            sleep 60;
            unlink $pid_file if (-f $pid_file);
            exit;
        }
        my $total_rss = get_process_total_memory($my_pid);
        if ($total_rss >= $max_threshold) {
            #restart to free memory (as perl cannot free)
            unlink $pid_file if (-f $pid_file);
            print "Memory exceeds limit. Restart to free memory.\n" if ($debug);
            exit;
        }
        #Skip if harddisk is missing
        if ($log_dir !~ /^\s*[\/]*(harddisk:|misc\/+disk1)/) {
            sleep 60;
            print "No harddisk mounted, skip.\n";
            next;
        }
        #Only run on the active RP
        #to collect show logging on that chassis
        $node_info = get_active_rp_nodes_by_chassis_id($sys_info, $chassis_id);
        if (!$node_info->{isActive}) {
            sleep 60;
            print "Not active RP, skip.\n";
            next;
        }
        if ($osType =~ /calvados/i) {
            my $first_calv_ip = get_first_calv_ip($node_info);
            if ($vf1_3073_ip ne $first_calv_ip) {
                sleep 60;
                print "Not active RP, skip.\n";
                next
            }
        }

        #get show log data:
        if ($osType =~ /xr/i) {
            @newLogs = &get_new_xr_show_logging($refresh_interval,
                                                \@log_patterns,
                                                $chassis_id);
            my $isAlertFound = 0;

        } elsif ($osType =~ /calv/i) {
            $log_info = &get_new_calv_show_log($last_timestamp, $chassis_id);
        }
        my @tmp_newLogs = @newLogs;

        my $update_log = 0;
        if (open($wsfd, ">$logFile")) {
            $update_log = 1;
        }

        my $isTracebackFound = 0;
        my $matchedPatterns = "";
        my $matchedLogs = "";
        my %isUniqPat;

        my $show_tech_executed = 0;
        my @executed_commands = ();
        my @saved_log_files = ();
        my %isExecuted;

        ###############################################################
        #I - process tracebacks
        ###############################################################
        if ($osType =~ /xr/i) {
            $sbt = get_sbt(\@tmp_newLogs);
            @sbt_node_pids = @{$sbt->{node_pids}};
            foreach my $sbt_node_pid (@sbt_node_pids) {
                my $show_tech_executed = 0;
                $showtech_cnt = 0;
                my @executed_commands = ();
                my @saved_log_files = ();
                my %isExecuted;
                next if ( (!defined($sbt->{$sbt_node_pid}->{proc})) ||
                          ($sbt->{$sbt_node_pid}->{proc} !~ /\w+/) );
                my $sbt_proc = $sbt->{$sbt_node_pid}->{proc};
                #according to CSCvb94402, ignore repeative tracebacks.
                if (!$isKnownSbt{$sbt_proc}) {
                    $isKnownSbt{$sbt_proc} = 1;
                    #TODO - Remove TRUNCATED line
                    #LC/0/5/CPU0:Jul 12 06:07:08.567 : fib_mgr..4(TRUNCATED)

                    my $sbt_line = $sbt->{$sbt_node_pid}->{line};
                    $matchedLogs .= $sbt_line . "\n";;
                    $isTracebackFound++;
                    my $sbt_trace = $sbt->{$sbt_node_pid}->{trace};
    
                    print "matchedPatterns=Traceback,matchedLogs=$matchedLogs\n"
                                                                    if ($debug);
                    my $sbt_node = $sbt->{$sbt_node_pid}->{node};
                    #TOOO - HACK for spitfire
                    $sbt_node = "0_RP0_CPU0" if ($sys_info->{is_thinxr});
                    my $jid = $sbt->{$sbt_node_pid}->{jid};
                    my $pid = $sbt->{$sbt_node_pid}->{pid};
                    #Verify if process is still running, if not skip
                    my $pid_info = get_xr_pid_info($sbt_proc, $sbt_node);
                    if (!defined($pid_info->{pid})) {
                        next;
                    }
    
                    my %isSeen;
                    my @traceCommands = ();
    
                    my $_event_type = "traceback";
                    my $cmd_info1 = get_default_commands($_event_type,
                                                         $osType);
                    my @default_stb_commands = 
                                @{$cmd_info1->{$cmd_type}->{allCommands}};
                    foreach my $cmd_mode (@default_stb_commands) {
                        $cmd_mode .= ",";
                        if ($cmd_mode =~ /<location|node>/i) {
                            $cmd_mode .= $sbt_node;
                        }
                        if (!$isSeen{$cmd_mode}) {
                            $isSeen{$cmd_mode} = 1;
                            push @traceCommands, $cmd_mode;
                        }
                    }
    
                    # check any additional traceback specific commands
                    # to run on top of default
                    my $cmd_info = get_edcd_event_commands($platform,
                                                           $osType,
                                                           $_event_type,
                                                           $sbt_proc,
                                                         \@pam_event_db_files);
                    my $_commandList = $cmd_info->{$cmd_type}->{commandList};
                    foreach my $cmd (@$_commandList) {
                        $cmd =~ s/\s+/ /g;
                        my $_cmd_mode = $cmd . "," . $mode . ",";
                        $_cmd_mode .= $sbt_node;
                        if (!$isSeen{$_cmd_mode}) {
                            $isSeen{$_cmd_mode} = 1;
                            push @traceCommands, $_cmd_mode;
                        }
                    }

                    my $_pid = $pid;
                    #assume location will be mapped to
                    # -n nodeid (except show tech)
                    my $tmp_pattern = 'show_dll\s+\-p\s+0x<pid>\s+';
                    $tmp_pattern .= '\-n\s+<location>';
                    foreach my $cmd_mode (@traceCommands) {
                        my $_process = $sbt_proc;
                        my ($_cmd, $_mode, $_node) = split(/,/, $cmd_mode);
                        if ($_cmd =~ /$tmp_pattern/) {
                            $_cmd =~
                                s/pid\s+<pid>/pid $sbt->{$sbt_node_pid}->{pid}/;
                        } elsif ($_cmd =~ /pid\s+0x<pid\s+of\s+([\w\-]+)>/) {
                            $_process = $2;
                            my $_proc_info = get_xr_pid_info($_process, $_node);
                            if ( (!defined($_proc_info->{pid})) ||
                                ($_proc_info->{pid} !~ /^\d+$/) ) {
                                next;
                            } else {
                                $_pid = $_proc_info->{pid};
                                $_cmd =~s/pid\s+0x<pid\s+of\s+[\w\-]+>/pid 0x<pid>/;
                            }
                        } else {
                            #next;
                        }
                        if ($_cmd =~ /show.*tech/i) {
                            $show_tech_executed++;
                            $showtech_cnt++;
                            sleep 1;
                        }
                        #skip executed!
                        next if ($isExecuted{$cmd_mode});
                        $isExecuted{$cmd_mode} = 1;
                        #TOOO - HACK ..
                        $_node = "0_RP0_CPU0" if ($sys_info->{is_thinxr});
                        my $cli_info = cli_agent_shell($sys_info,
                                                       $_event_type,
                                                       $_process,
                                                       $_pid,
                                                       $_cmd,
                                                       $_node);
                                                       #chassis_id
                        if (!$cli_info->{rc}) {
                            my $msg = $cli_info->{msg};
                            &pam_logger($sys_info, $log_dir, $bucket, $msg);
                        } else {
                            next if (!defined($cli_info->{log_files}));
                            push @executed_commands, $cli_info->{cmd};
                            my @log_files = @{$cli_info->{log_files}};
                            foreach my $_log_file (@log_files) {
                                #for show dll, add tracebacks:
                                #show append the same traces to all show_dll???
                                if ($_log_file =~ /show_dll/) {
                                    my $tmp_file = $_log_file . ".tmp";
                                    if (!open(WD, ">$tmp_file")) {
                                        my $msg =
                                               "Unable to create $tmp_file: $!";
                                        &pam_logger($sys_info,
                                                    $log_dir,
                                                    $bucket,
                                                    $msg);
                                        next;
                                    }
                                    if (!open(FD, "$_log_file")) {
                                        my $msg="Unable to open $_log_file: $!";
                                        &pam_logger($sys_info,
                                                    $log_dir,
                                                    $bucket,
                                                    $msg);
                                        next;
                                    }
                                    print WD $sbt->{$sbt_node_pid}->{line},"\n";
                                    while(<FD>) {
                                        print WD $_;
                                    }
                                    close(FD);
                                    close(WD);
                                    unlink $_log_file;
                                    rename($tmp_file, $_log_file);
                                }
                                push @saved_log_files, $_log_file;
                            }
                        } ;# if success
                    } ;#foreach my $cmd (@traceCommands)
    
                    ############################################################
                    #save summary/archive for tracebacks!
                    ############################################################
                    my $summary_log = &save_show_log_summary($log_dir,
                                                            $matchedPatterns,
                                                            $matchedLogs,
                                                            \@executed_commands,
                                                            \@saved_log_files,
                                                            $show_tech_executed,
                                                            $show_tech_root,
                                                            $bucket);
                    push @saved_log_files, $summary_log if ($summary_log);
                    if ( @saved_log_files ) {
                        my $delete_original = 1;
                        my @retained_logs = ();
                        my $pr_info;
                        $pr_info->{node} = $sbt_node;
                        $pr_info->{event_type} = $_event_type;
                        $pr_info->{procName} = $sbt->{$sbt_node_pid}->{proc};
                        $pr_info->{delete_original} = $delete_original;
                        $pr_info->{showtech_cnt} = $showtech_cnt;
                        my $tar_name = &create_log_archive($log_dir,
                                                           \@saved_log_files,
                                                           \@retained_logs,
                                                           $pr_info);
                    }
                } ;# if (!$isKnownSbt{$sbt_node_pid})
            } ;# foreach my $sbt_node_pid (@{$sbt->{node_pids}})
        } ;# if ($osType =~ /xr/i)
        ###############################################################
        #I - done with handling tracebacks
        ###############################################################

        ###############################################################
        #II - process other (non-traceback) patterns from show logging 
        ###############################################################
        #don't use (@{${}}) as it's causing leaks...
        @tmp_newLogs = @newLogs;
        foreach my $log (@tmp_newLogs) {
            my $isAlertFound = 0;
            $show_tech_executed = 0;
            $showtech_cnt = 0;
            @executed_commands = ();
            undef %isExecuted;
            $matchedLogs = "";
            $matchedPatterns = "";
            @saved_log_files = ();
            my @exec_cmds = ();

            #Don't append as it will cause memory leak!!
            #$log .= "." if ($log !~ /[\.,;]$/);
            if ($update_log) {
                print $wsfd $log, "\n";
            }
            my $isMatched = 0;

            #For now: only take location
            my %isCmdSeen;
            my $node = "";
            foreach my $pat (@log_patterns) {
                #skip Traceback as it is handled separately
                next if ($pat =~ /Traceback/i);
                #TODO - lindt where nodename is missing
                if ( $log =~ /^\s*(\S+)?:?${month_pat}\s+.*${pat}/) {
                    $node = $1;
                    #TOOO - HACK ..
                    $node = "0_RP0_CPU0" if ($sys_info->{is_thinxr});
                    print "log=$log\n" if ($debug);
                    $matchedLogs .= $log . "\n";;
                    $node =~ s/^(LC|R[S]?P)\///;
                    $node =~ s/\/ADMIN\d+//;
                    if (!$isKnownPat{$pat}) {
                        $isKnownPat{$pat} = 1;
                    } else {
                        next;
                    }
                    $isMatched++;
                    my @cmd_modes = @{$cmd_map_info->{$pat}};
                    foreach my $cmd_mode (@cmd_modes) {
                        $cmd_mode .= "," . $mode . ",";
                        if ($cmd_mode =~ /<location|node>/i) {
                            $cmd_mode .= $node;
                        }
                        if (!$isCmdSeen{$cmd_mode}) {
                            $isCmdSeen{$cmd_mode} = 1;
                            push @exec_cmds, $cmd_mode;
                        }
                    }
                    foreach my $cmd_mode (@default_commands) {
                        $cmd_mode .= "," . $mode . ",";
                        if ($cmd_mode =~ /<location|node>/i) {
                            $cmd_mode .= $node;
                        }
                        if (!$isCmdSeen{$cmd_mode}) {
                            $isCmdSeen{$cmd_mode} = 1;
                            push @exec_cmds, $cmd_mode;
                        }
                    }
                    $isAlertFound++;
                    if (!$isUniqPat{$pat}) {
                        $isUniqPat{$pat} = 1;
                        $matchedPatterns .= " '" . $pat . "',";
                    }
                }
            } ;# foreach my $pat (@log_patterns)
            next if (!$isMatched);

            $matchedPatterns =~ s/,$//;
            my $process = "";
            foreach my $cmd_mode (@exec_cmds) {
                my ($cmd, $mode, $_node) = split(/,/, $cmd_mode);
                next if ($isExecuted{$cmd});
                $isExecuted{$cmd} = 1;
                if ($cmd =~ /show.*tech/i) {
                    $show_tech_executed++;
                    $showtech_cnt++;
                    sleep 1;
                }
                #TOOO - HACK ..
                $node = "0_RP0_CPU0" if ($sys_info->{is_thinxr});
                my $cli_info = cli_agent_shell($sys_info,
                                               $event_type,
                                               $process,
                                               "",
                                               $cmd,
                                               $node);
                if (!$cli_info->{rc}) {
                    my $msg = $cli_info->{msg};
                    &pam_logger($sys_info, $log_dir, $bucket, $msg);
                } else {
                    if (defined($cli_info->{log_files})) {
                        push @executed_commands, $cli_info->{cmd};
                        foreach my $_log_file (@{$cli_info->{log_files}}) {
                            push @saved_log_files, $_log_file;
                        }
                    }
                }
            } ;# foreach my $cmd_mode (@exec_cmds)
            if ($isAlertFound>0) {
                $matchedLogs =~ s/[\r\n]*$//;
                my $summary_log = &save_show_log_summary($log_dir,
                                                         $matchedPatterns,
                                                         $matchedLogs,
                                                         \@executed_commands,
                                                         \@saved_log_files,
                                                         $show_tech_executed,
                                                         $show_tech_root,
                                                         $bucket);
                push @saved_log_files, $summary_log if ($summary_log);
                if ( @saved_log_files ) {
                    my $delete_original = 1;
                    my $_event_type = "show_logging_patterns";
                    my @retained_logs = ();
                    my $pr_info;
                    $pr_info->{node} = $node;
                    $pr_info->{event_type} = $_event_type;
                    $pr_info->{procName} = "";
                    $pr_info->{delete_original} = $delete_original;
                    $pr_info->{showtech_cnt} = $showtech_cnt;
                    my $tar_name = &create_log_archive($log_dir,
                                                       \@saved_log_files,
                                                       \@retained_logs,
                                                       $pr_info);
                }
            } ;# if ($isAlertFound>0)
        } ;# foreach my $log (@newLogs)
        if ($update_log) {
            close($wsfd);
        }
        print "111 Restart...\n" if ($debug);
        sleep $refresh_interval;
    } ;# while (1) 
    print "Fail to fork\n";
} ;# unless (fork)

sub INT_handler {
    print "KILL signal received. Closing all files.\n";
    if (defined($wsfd)) {
        close($wsfd);
    }
    exit(0);
}

sub get_old_show_log ($) {
    my $logFile = shift;

    my $max_delta = 11 * 30 * 24 * 60 * 3600;
    my $log_retention = 2 * 3600;

    my $log_info;
    my @oldLogs = ();
    $log_info->{newLogs} = \@oldLogs;
    $log_info->{timestamp} = 0;
    $log_info->{ymdhms} = "0:0:0:0:0:0";
    my $gmt;
    my ($year,$c_mon,$c_day, $c_hour,$c_min,$c_sec, $c_doy,$c_dow,$c_dst) =
                                                       System_Clock([$gmt]);
    my ($actual_year,$mon,$day,$hour,$min,$sec, $msec);
    if (! -f $logFile) {
        return $log_info;
    }
    my ($dev,$ino,$_mode,$nlink,$uid,$gid,
        $rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks)
                                            = stat($logFile);
    if (defined($size) && ($size >= 1000000)) {
        unlink $logFile;
        print "Log size is too big ($size bytes > 1000000), delete.\n";
        return $log_info;
    }
    my $delta = time() - $mtime;
    if ($delta > $log_retention) {
        unlink $logFile;
        print "Log is too old (created $delta seconds ago), delete.\n";
        return $log_info;
    }

    if ( -f $logFile ) {
        if (!open(FD, $logFile)) {
            print "Failed to open $logFile: $!\n";
            return $log_info;
        }

        my $latest_timestamp = 0;
        my $month_pat = 'Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec';
        my $pattern = "\\d+:\\s*($month_pat)\\s+(\\d+)\\s+";
        $pattern .= '(\d+):(\d+):(\d+)\.(\d+)\s*';
        while (my $log = <FD>) {
            if ( $log =~ /$pattern/) {
                chomp $log;
                push @oldLogs, $log;
                $mon =  $1;
                $day =  $2;
                $hour = $3;
                $min =  $4;
                $sec =  $5;
                $msec = $6;
                $mon = $$month_map{$mon};
                my $timestamp = Date_to_Time($year,$mon,$day,$hour,$min,$sec);
                $timestamp .= "." . $msec;
                if ($mon eq 12) {
                    my $delta = $timestamp - $last_timestamp;
                    if ($delta > $max_delta) {
                        $actual_year = $year - 1;
                        $timestamp =
                          Date_to_Time($actual_year,$mon,$day,$hour,$min,$sec);
                    }
                } else {
                    $actual_year = $year;
                }
                if ($timestamp > $latest_timestamp) {
                    $latest_timestamp = $timestamp;
                }
            }
        }
        close(FD);
        if (scalar(@oldLogs) > 0) {
            $log_info->{newLogs} = \@oldLogs;
            $log_info->{timestamp} = $latest_timestamp;
            $log_info->{ymdhms} = "$actual_year:$mon:$day:$hour:$min:$sec";
        }
    }
    $log_info->{oldLogs} = \@oldLogs;
    return $log_info;
} ;#sub get_old_show_log

# get date of behind - used for show logging on XR:
sub get_show_log_start_date($$$) {
    my $Dh = shift;
    my $Dm = shift;
    my $Ds = shift;
    my $Dd = 0;

    my $clock_binary = "/pkg/bin/iosclock";
    if (( -f $clock_binary) && (-x "/pkg/bin/iosclock")) {
        my $clock_cmd = "/pkg/bin/iosclock -e 0x0";
	if (!open(FD, "$clock_cmd |")) {
		print "Cannot run $clock_cmd $!\n";
	}
	else {
	    while (my $log = <FD>) {
                if ($log =~ m/(\d+)\:(\d+)\:(\d+)\.(\d+)\s+(\w+)\s+\w+\s+(\w+)\s+(\d+)\s+(\d+)/) {
                    my $c_hour = $1;
                    my $c_min = $2;
		    my $c_sec = $3;
		    my $c_msec = $4;
                    my $timezone = $5;
		    my $c_mon = $$month_map{$6};
		    my $c_day = $7;
                    my $c_year = $8;
                    if ($timezone ne 'UTC') {
			my ($year,$month,$day, $hour,$min,$sec) =
			    Add_Delta_DHMS($c_year,$c_mon,$c_day,$c_hour,$c_min,$c_sec,
					   $Dd,$Dh,$Dm,$Ds);
			return "$year,$month,$day,$hour,$min,$sec";
		    }
		}
	    } ;# while (my $line = <FD>)
	}
   }

    my $gmt = Gmtime();
    my ($c_year,$c_mon,$c_day, $c_hour,$c_min,$c_sec,
                     $c_doy,$c_dow,$c_dst) = System_Clock([$gmt]);

    my ($year,$month,$day, $hour,$min,$sec) =
            Add_Delta_DHMS($c_year,$c_mon,$c_day,$c_hour,$c_min,$c_sec,
                                $Dd,$Dh,$Dm,$Ds);
    my $month_map = {
        1 => 'january',
        2 => 'february',
        3 => 'march',
        4 => 'april',
        5 => 'may',
        6 => 'june',
        7 => 'july',
        8 => 'august',
        9 => 'september',
       10 => 'october',
       11 => 'november',
       12 => 'december',
    };
    return "$year,$month,$day,$hour,$min,$sec";
} ;# sub get_show_log_start_date($$$)

sub get_new_calv_show_log($$) {
    my $chassis_id = shift;
    my $last_timestamp = shift || 0;

    my $max_delta = 11 * 30 * 24 * 60 * 3600;

    my $log_info;
    my $timestamp;
    my @newLogs = ();

    my $month_pat = 'Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec';
    #get the current year:
    my $gmt = Gmtime();
    my ($year,$c_mon,$c_day, $c_hour,$c_min,$c_sec,
                         $c_doy,$c_dow,$c_dst) = System_Clock([$gmt]);

    my ($mon,$day,$hour,$min,$sec,$msec);
    my $delta;

    #TODO - only collect last 2000 lines from the same chassis!
    my $cmd = "logging";
    if ($chassis_id =~ /^\d+$/) {
        $cmd .= " | include \"(F)?${chassis_id}/(R[S]?P|SC|CB)?[0-9]+/ADMIN\"";
    }
    my $syslog = confd_cli_wrapper($cmd);

    my $latest_timestamp = $last_timestamp;
    my $pattern = '\d+:\s*(' . $month_pat . ')\s+(\d+)\s+(\d+):(\d+):';
    $pattern .= '(\d+)\.(\d+)\s*';
    foreach my $log (split(/\n/, $syslog)) {
        if ( $log =~ /$pattern/) {
            $mon =  $1;
            $day =  $2;
            $hour = $3;
            $min =  $4;
            $sec =  $5;
            $msec = $6;
            $mon = $$month_map{$mon};
            $timestamp = Date_to_Time($year,$mon,$day,$hour,$min,$sec);
            $timestamp .= "." . $msec;

            # TODO - if month is Dec, and delta is > 11 month,
            # then need to shift year back 
            if ($mon eq 12) {
                $delta = $timestamp - $last_timestamp;
                if ($delta > $max_delta) {
                    my $actual_year = $year - 1;
                    $timestamp =
                      Date_to_Time($actual_year,$mon,$day,$hour,$min,$sec);
                }
            }
            if ( ($timestamp > $last_timestamp) ) {
                push @newLogs, $log;
                $last_timestamp = $timestamp;
            }
        }
    }
    $log_info->{timestamp} = $last_timestamp;
    $log_info->{newLogs} = \@newLogs;
    if (scalar(@newLogs) > 0) {
        $log_info->{ymdhms} = "$year:$mon:$day:$hour:$min:$sec";
    }
    return $log_info;
} ;#sub get_new_calv_show_log

sub get_new_xr_show_logging($$$) {
    my $interval      = shift; #minutes
    my $_log_patterns = shift;
    my $chassis_id    = shift;

    my @newLogs = ();
    my $month_pat = 'Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec';
    #get the current year:
    my $gmt = Gmtime();
    my ($year,$c_mon,$c_day, $c_hour,$c_min,$c_sec,
                          $c_doy,$c_dow,$c_dst) = System_Clock([$gmt]);
    my ($mon,$day,$hour,$min,$sec,$msec);

    #To minimize memory usage, take samples starting within last 2 intervals
    $interval = sprintf("%d", 2 * $interval/60);
    my $Dh = 0;
    my $Dm = -$interval;
    my $Ds = 0;
    my $dateStrings = get_show_log_start_date($Dh, $Dm, $Ds);
    my ($year1,$month1,$day1,$hour1,$min1,$sec1) = split(/,/, $dateStrings);
    my $_y = sprintf("0x%x", $year1);
    my $_o = sprintf("0x%x", $month1-1);
    my $_d = sprintf("0x%x", $day1);
    my $_h = sprintf("0x%x", $hour1);
    my $_m = sprintf("0x%x", $min1);
    my $_s = sprintf("0x%x", $sec1);
    my $syslog_cmd = "/pkg/sbin/show_logging";
    if ((! -f $syslog_cmd) && (-x "/pkg/bin/show_logging")) {
        $syslog_cmd = "/pkg/bin/show_logging";
    }
    $syslog_cmd .= " -o $_o -d $_d -h $_h -m $_m -s $_s -z $_y";

    if ($chassis_id =~ /^\d+$/) {
        $syslog_cmd .= " | egrep \"^(B|LC|R[S]?P)/$chassis_id/(R[S]?P|CB)?[0-9]+\"";
    }

    my @log_patterns = @$_log_patterns;

    #TODO:
    #single (less efficient but less memory)
    #combined (efficient but more memory)
    my $patterns = "";
    foreach my $pat (@log_patterns) {
        $patterns .= $pat . "|";
    }
    $patterns =~ s/\^\s*\|//;
    $patterns =~ s/\s*\|$//;
    if (!open(FD, "$syslog_cmd |")) {
        print "Cannot run $syslog_cmd $!\n";
        return @newLogs;
    }
    my $traceback_pat = '(\w+)\s+:\s+\(PID=\d+\)\s+:\s+\-Traceback=';
    my %isBtSeen;
    while (my $log = <FD>) {
        if ($log =~ /${traceback_pat}/) {
            my $_proc = $1;
            if (!$isBtSeen{$_proc}) {
                push @newLogs, $log;
                $isBtSeen{$_proc} = 1;
            }
        } else {
            foreach my $pat (@log_patterns) {
                if ( $log =~ /:?(${month_pat})\s+.*(${patterns})/) {
                    push @newLogs, $log;
                    last;
                }
            }
        }
    } ;# while (my $line = <FD>)
    close (FD);
    return @newLogs;
} ;#sub get_new_xr_show_logging
