#!/usr/bin/python
#
# July 2017
#
#  Copyright (c) 2017-2019 by cisco Systems, Inc.
#  All rights reserved.
#
# Test Optics DOM sensor monitoring by overriding sensor values

import os
import sys
import re
import argparse
import time

parser = argparse.ArgumentParser(description='Test optics DOM alarms on eXR')

parser.add_argument('-s', help='Show the sensor values for the interface', action="store_true")
parser.add_argument('-c', help='Clear all sensor values for interface', action="store_true")
parser.add_argument('--temp', default="12.345", help='Specify temperature value for lane in Celcius')  # 12345/1000
parser.add_argument('--tx_pwr', default="1.2345", help='Specify TX power value for lane in mW')   # default: 12345/100000
parser.add_argument('--rx_pwr', default="1.2345", help='Specify RX power value for lane in mW')   # default: 12345/10000
parser.add_argument('--bias', default="24.69", help='Specify TX bias (current) value for lane in mA')  #default: 12345/500
parser.add_argument('--voltage', default="1.2345", help='Specify Voltage value for lane in Volts')     #default: 12345/10000
parser.add_argument('--lane', help='Specify the lane to operate (0 to N, or -1 for base module)')
parser.add_argument('interface', help='Interface to operate on')
args = parser.parse_args()

bay_port = None

def find_port_parameters(interface):

    global bay_port

    interface_internal = interface.replace("/", "_")
    cmd = "/pkg/bin/ethernet_show_controller -i %s -s internal" % interface_internal
    #print ("Running command: %s\n" % cmd)
    output = os.popen(cmd).read()
    
    #Quick validation
    if not "Bay Number" in output:
        # We didn't ge what we wanted
        print("Looks like invalid interface %s is passed!!\nGot output:%s" % (interface_internal, output))
        sys.exit()

    for line in output.split('\n'):
        #Ifinst Subport      : 48
        match = re.search("Ifinst\s+Subport\s+:\s+(\d+)", line)
        if match != None:
            bay_port = int(match.group(1))

    if bay_port == None :
        print("Could not derive parameters for %s (internal error).\nOutput:%s", (interface, output))
        sys.exit()


interface = args.interface

#Get board type
find_port_parameters(interface)

#Build sensor db
sensor_file = "/tmp/%d_sensor_values" % (bay_port)
sensors = list()
if os.path.exists(sensor_file):
    output = os.popen("cat %s" % sensor_file).read()

    for line in output.split('\n'):
        match = re.search("^\s*(\S+)\s+(\S+)\s+(\S+)\s+(\d+)\s+(\d+)\s+(\d+)", line)
        if match != None:
            sensor_obj = {}
            sensor_obj['lane'] = int(match.group(1))
            sensor_obj['current'] = int(match.group(2))
            sensor_obj['temp'] = int(match.group(3))
            sensor_obj['rx_pwr'] = int(match.group(4))
            sensor_obj['tx_pwr'] = int(match.group(5))
            sensor_obj['vcc'] = int(match.group(6))
            sensors.append(sensor_obj)

if args.s:
    cmd = "/pkg/bin/lda_test client xcvr alarms %d | grep -E 'Base|Lane'" % (bay_port) 
    #print("Running command: %s" % cmd)
    output = os.popen(cmd).read()
    print(output)
    sys.exit()

if args.c:
    if args.lane == None:
        # Lane not specified, assume all lanes
        del(sensors[:])
        print("Removed sensor values for all lanes")
    else:
        #Remove the lane from the sensors list
        for obj in sensors:
            if obj['lane'] == int(args.lane):
                sensors.remove(obj)
else:

    if args.lane == None:
        print("Lane not specified\n")
        sys.exit()

    obj = None
    obj_found = False
    for obj in sensors:
        if obj['lane'] == int(args.lane):
            obj_found = True
            print("Trying to overwrite existing lane: %d\n" % int(args.lane))
            break

    if obj_found == False:
        obj = {}
        print("Trying to add new lane info: %d\n" % int(args.lane))
        obj['lane'] = int(args.lane)
        sensors.append(obj)

    obj['current'] = int(float(args.bias) * 500)
    obj['tx_pwr'] = int(float(args.tx_pwr) * 10000)
    obj['rx_pwr'] = int(float(args.rx_pwr) * 10000)
    obj['temp'] = int(float(args.temp) * 1000)
    obj['vcc'] = int(float(args.voltage) * 10000)


retry = 0
sensor_fd = None
while retry < 3:
    sensor_fd = open(sensor_file, "w")
    if sensor_fd != None:
        break
    time.sleep(50)
    retry = retry + 1

for obj in sensors:
    cmd = "%d %d %d %d %d %d\n" % (obj['lane'], obj['current'], obj['temp'], obj['rx_pwr'], obj['tx_pwr'], obj['vcc'])
    sensor_fd.write(cmd)
sensor_fd.close()

