#!/pkg/bin/perl
##
 # dataplane_subs.pl
 #
 #------------------------------------------------------------------
 # Copyright (c) 2012-2013 by Cisco Systems, Inc.
 # All rights reserved.
 # Author: Joshua Jodrey
 #
 # version 1.0 (tested in 4.3)
 #------------------------------------------------------------------
 ##

#require 'platforms/viking/lc/prm/scripts/dataplane_lc_config.pl';
require '/pkg/bin/dataplane_lc_config.pl';
#require 'dataplane_lc_config.pl';
#require '/harddisk:/dataplane_lc_config.pl';

BEGIN { $SIG{'__WARN__'} = sub { &warn_error ($_[0]); }}


#**********************************#
#       Linecard Subroutines       #
#**********************************#
sub dataplaneMap {
	my %hwMap = %{$_[0]};
	my $locationString = $_[1];
	
	my @interfaceArray = ();
	my $interfaceArg = "";

        # If an interface is passed in, convert to the form "GigE..."
        # and then filter all keys in the hardware map against it
	if ($_[2]) {
            $interfaceArg = convertInterface($_[2]);
            $interfaceArg =~ /(\w*Gig)\w*E\w*(\d+)_(\d+)_(\d+)_(\d+)/;
            $interfaceArg = "$1E$4_$5";
	}
	
	$locationString =~ /(\d+)\/(\d+)\/CPU0/;
	my $rack = $1;
	my $slot = $2;
	
	# Get all of the interface keys into a string array and sort
	# For each interface in the array, walk the dataplane until you hit the XBAR
	# TenGigE0/0/0/0 <-> (XAUI 4) NP 0 (ILKN 11) <-> (NW 0) FIA 0 (FB 0, 1) <-> (PORT 3, 4) XBAR
	# TenGigE0/0/0/1 <-> (XAUI 5) NP 0 (ILKN 11) <-> (NW 0) FIA 0 (FB 0, 1) <-> (PORT 3, 4) XBAR
	# TenGigE0/0/0/2 <-> (XAUI 6) NP 0 (ILKN 11) <-> (NW 0) FIA 0 (FB 0, 1) <-> (PORT 3, 4) XBAR
	
	print "KEY: --> Ingress  <-- Egress <-> Bidirectional\n";
	print "     Interface (PORT-TYPE port#, load-balance-port#) HW-OBJ inst [inst,load-balance-inst]\n\n";
	
	for my $key (keys %hwMap) {
            if ($key =~ /GigE\d+_(\d+)/) {
                push(@interfaceArray, $key);
            }
        }
	
	foreach my $interface (@interfaceArray) {
            if (($interfaceArg ne "") && ($interfaceArg ne $interface)) {
                next;
            }
            my $portType = $hwMap{$interface}{IF_TYPE};
            my @portNums = @{$hwMap{$interface}{PORT_IN}};
            
            my $portString = "";
            foreach my $port (@portNums) {
                $portString .= "$port, ";
            }

            chop($portString);
            chop($portString);
		
            $interface =~ /(.*GigE)(\d+)_(\d+)/;
            my $ep = $2;
            my $int = $3;
            my $prettyInterface = "$1$rack/$slot/$ep/$int";
            
            printf("%20s <-> (%s %s) ", $prettyInterface, $portType, $portString);

            my $hwObject = $interface;
            while(exists $hwMap{$hwObject}{CONNECTS_TO}) {
                $hwObject = $hwMap{$hwObject}{CONNECTS_TO};
                if ($hwObject =~ /NP(\d+)/) {
                    my $np = $1;
                    my @portNums = @{$hwMap{$hwObject}{PORT_OUT}};
                    my $outputPortString = "";
                    foreach my $port (@portNums) {
                        $outputPortString .= "$port, ";
                    }
                    chop($outputPortString);
                    chop($outputPortString);
                    
                    @portNums = @{$hwMap{$hwObject}{PORT_IN}};
                    my $inputPortString = "";
                    foreach my $port (@portNums) {
                        $inputPortString .= "$port, ";
                    }
                    chop($inputPortString);
                    chop($inputPortString);
                    
                    printf("NP %d (ILKN %s) <-> (NW %s) ", $np, $outputPortString, $inputPortString);
                    
                } elsif ($hwObject =~ /FIA(\d+)/) {
                    my $fia = $1;
                    my @portNums = @{$hwMap{$hwObject}{PORT_OUT}};
                    my $outputPortString = "";
                    foreach my $port (@portNums) {
                        $outputPortString .= "$port, ";
                    }
                    chop($outputPortString);
                    chop($outputPortString);
                    
                    @portNums = @{$hwMap{$hwObject}{PORT_IN}};
                    my $inputPortString = "";
                    foreach my $port (@portNums) {
                        $inputPortString .= "$port, ";
                    }
                    chop($inputPortString);
                    chop($inputPortString);
                    
                    printf("FIA %s (FB %s) <-> (PORT %s) XBAR", $fia, $outputPortString, $inputPortString);
                    
                } elsif ($hwObject =~ /XBAR(\d+)/) {
                    
                } elsif ($hwObject =~ /RSP/) {
                    
                } else {
                    printf("Unknown HW Object: '%s'\n", $hwObject);
                }
            }
            printf("\n");
	}
    }

sub dataplaneMapJuggernaut {
	my %hwMap = %{$_[0]};
	my $locationString = $_[1];
	
	my @interfaceArray = ();
	my $interfaceArg = "";
	if ($_[2]) {
		$interfaceArg = convertInterface($_[2]);
		$interfaceArg =~ /(.*GigE)(\d+)_(\d+)_(\d+)_(\d+)/;
		$interfaceArg = "$1$4_$5";
	}
	
	$locationString =~ /(\d+)\/(\d+)\/CPU0/;
	my $rack = $1;
	my $slot = $2;
	
	# Get all of the interface keys into a string array and sort
	# For each interface in the array, walk the dataplane until you hit the XBAR
	# HundredGigE0/0/0/0 --> (ILKN 10) NP 2 (ILKN 10) --> BRIDGE 1 --> (NW 0, 1) FIA COMPLEX [2,3] (FB 0, 1) --> (PORT 1, 2, 3, 4) XBAR
	# HundredGigE0/0/0/0 <-- (ILKN 10) NP 3 (ILKN 10) <-- BRIDGE 1 <-- (NW 0, 1) FIA COMPLEX [2,3] (FB 0, 1) <-- (PORT 1, 2, 3, 4) XBAR
	
	print "KEY: --> Ingress  <-- Egress <-> Bidirectional\n";
	print "     Interface (PORT-TYPE port#, load-balance-port#) HW-OBJ inst [inst,load-balance-inst]\n\n";
	
	for my $key (keys %hwMap) {
        if ($key =~ /GigE\d+_(\d+)/) {
			$interfaceArray[$1] = $key;
		}
    }
	
	foreach my $interface (@interfaceArray) {
		if (($interfaceArg ne "") && ($interfaceArg ne $interface)) {
			next;
		}
		my $dirArrow = "-->";
		my $egressString = "";
	
		my $portType = $hwMap{$interface}{IF_TYPE};
		my @portNums = @{$hwMap{$interface}{PORT_IN}};
		my $portString = "";
		foreach my $port (@portNums) {
			$portString .= "$port, ";
		}
		chop($portString);
		chop($portString);
		
		$interface =~ /(.*GigE)(\d+)_(\d+)/;
		my $ep = $2;
		my $int = $3;
		my $prettyInterface = "$1$rack/$slot/$ep/$int";
		if ($egressString eq "") {
			printf("%20s %s (ILKN %s) ", $prettyInterface, $dirArrow, $portString);
		} else {
			$egressString = sprintf("%20s %s (ILKN %s) ", $prettyInterface, $dirArrow, $portString)
							. $egressString;
			last;
		}

		my $hwObject = $interface;
		while(exists $hwMap{$hwObject}{CONNECTS_TO}) {
			$hwObject = $hwMap{$hwObject}{CONNECTS_TO};
			if ($hwObject =~ /GigE/) {
				my @portNums = @{$hwMap{$interface}{PORT_IN}};
				my $portString = "";
				foreach my $port (@portNums) {
					$portString .= "$port, ";
				}
				chop($portString);
				chop($portString);
				$egressString = sprintf("%20s %s (ILKN %s) ", $prettyInterface, $dirArrow, $portString)
								. $egressString;
				last;
			} elsif ($hwObject =~ /NP(\d+)/) {
				my $np = $1;
				my @portNums = @{$hwMap{$hwObject}{PORT_OUT}};
				my $outputPortString = "";
				foreach my $port (@portNums) {
					$outputPortString .= "$port, ";
				}
				chop($outputPortString);
				chop($outputPortString);

				if ($egressString eq "") {
					printf("NP %d (ILKN %s) %s ", $np, $outputPortString, $dirArrow);
				} else {
					$egressString = sprintf("NP %d", $np, $outputPortString, $dirArrow)
									. $egressString;
				}
				
			} elsif ($hwObject =~ /BRIDGE(\d+)/) {
				my $bridge = $1;
				@portNums = @{$hwMap{$hwObject}{PORT_IN}};
				my $inputPortString = "";
				foreach my $port (@portNums) {
					$inputPortString .= "$port, ";
				}
				chop($inputPortString);
				chop($inputPortString);
				
				if ($egressString eq "") {
					printf("BRIDGE %d %s (NW %s) ", $bridge, $dirArrow, $inputPortString);
				} else {
					$egressString = sprintf(" (ILKN %s) %s BRIDGE %d %s ", $inputPortString, $dirArrow, $bridge, $dirArrow)
									. $egressString;
				}
				
			} elsif ($hwObject =~ /FIA_COMPLEX_(\d+)_(\d+)/) {
				my $firstFia = $1;
				my $secondFia = $2;
				my @portNums = @{$hwMap{$hwObject}{PORT_OUT}};
				my $outputPortString = "";
				foreach my $port (@portNums) {
					$outputPortString .= "$port, ";
				}
				chop($outputPortString);
				chop($outputPortString);
				
				@portNums = @{$hwMap{$hwObject}{PORT_IN}};
				my $inputPortString = "";
				foreach my $port (@portNums) {
					$inputPortString .= "$port, ";
				}
				chop($inputPortString);
				chop($inputPortString);
			
				if ($egressString eq "") {
					printf("FIA COMPLEX [%s,%s] (FB %s) %s (PORT %s)",
						   $firstFia, $secondFia, $outputPortString, $dirArrow, $inputPortString);
				} else {
					$egressString = sprintf("(NW %s) FIA COMPLEX [%s,%s]", $outputPortString, $firstFia, $secondFia)
									. $egressString;
				}	   
				
			} elsif ($hwObject =~ /XBAR(\d+)/) {
				printf(" XBAR\n");
				$dirArrow = "<--";
				
				my @portNums = @{$hwMap{$hwObject}{PORT_OUT}};
				my $outputPortString = "";
				foreach my $port (@portNums) {
					$outputPortString .= "$port, ";
				}
				chop($outputPortString);
				chop($outputPortString);
				
								@portNums = @{$hwMap{$hwObject}{PORT_IN}};
				my $inputPortString = "";
				foreach my $port (@portNums) {
					$inputPortString .= "$port, ";
				}
				chop($inputPortString);
				chop($inputPortString);
				
				$egressString = sprintf(" (FB %s) %s (PORT %s) XBAR\n", $inputPortString, $dirArrow, $outputPortString);
			} else {
				printf("Unknown HW Object: '%s'\n", $hwObject);
			}
		}
		printf("%s", $egressString);
	}
}

#**********************************#
#       Interface Subroutines      #
#**********************************#

sub interfaceDataplaneClear {
	my %hwMap = %{$_[0]};
	my $location = $_[1];
	my $slotNum = locationStringToInteger($location);
	my $interface = $_[2];

	$interface =~ /(\w*GigE)(\d+)_(\d+)_(\d+)_(\d+)/;
	my $startingKey = $1 . $4 . "_" . $5;
	my $prettyInterface = "$1$2/$3/$4/$5";
								
	$hwObject = $startingKey;
	while(exists $hwMap{$hwObject}{CONNECTS_TO}) {
            
		if ($hwObject =~ /GigE(\d+)/) {
			clearInterfaceMACCounters($interface, $prettyInterface);
			
		} elsif ($hwObject =~ /NP(\d+)/) {
			$ingressNP = $1;
			clearNPCounters($ingressNP, $location);
			
		} elsif ($hwObject =~ /BRIDGE(\d+)/) {
			# These are cleared when FIA is cleared
		
		} elsif ($hwObject =~ /FIA/) {
			clearFIACounters($location);
		
		} elsif ($hwObject =~ /XBAR(\d+)/) {
			clearXBARCounters($location);
			
			if (%hwMap->{NAME} eq "Juggernaut") {
				# Need to also clear egress NP counters
				clearNPCounters($ingressNP-1, $location);
			
			}
			
			# Reached the end, so exit loop
			last;
			
		} else {
			printf("Unknown HW Object %s\n", $hwObject);
		}
		
		$hwObject = $hwMap{$hwObject}{CONNECTS_TO};
	}
	
	printf("Finished clearing datapath counters for %s\n", $prettyInterface);
}

sub interfaceDataplaneWalk {
	my %hwMap = %{$_[0]};
	my $location = $_[1];
	my $slotNum = locationStringToInteger($location);
	my $interface = $_[2];
	$interface =~ /(\w*GigE)(\d+)_(\d+)_(\d+)_(\d+)/;
	my $startingKey = $1 . $4 . "_" . $5;
	our $prettyInterface = "$1$2/$3/$4/$5";
	my $currentDirection = "INGRESS";

	@egressStrings = ();
	@ingressStrings = ();

    printCountersHeader();
	printf("%41s Dataplane\n\n", $prettyInterface);

    printf("INGRESS    HW Objects    Counters            Rate           Drops\n");
    printCountersHeader();

	$hwObject = $startingKey;	
	while(exists $hwMap{$hwObject}{CONNECTS_TO}) {

		if ($hwObject =~ /GigE(\d+)/) {
			$retString = showGoodInterfaceMACCounters($interface, $currentDirection);
			
		} elsif ($hwObject =~ /NP(\d+)/) {
			$retString = showGoodNPCounters($1, \@{$hwMap{$hwObject}{PORT_OUT}}, $location, $currentDirection);
			
		} elsif ($hwObject =~ /\bFIA(\d+)/) {
			$retString = showGoodFIACounters($1, \@{$hwMap{$previousObject}{PORT_IN}}, \@{$hwMap{$hwObject}{PORT_OUT}}, $location, $currentDirection);
			
		} elsif ($hwObject =~ /XBAR(\d+)/) {
			$retString = showGoodXBARCounters($1, \@{$hwMap{$previousObject}{PORT_IN}}, $location, $currentDirection);
				
		} else {
			print "Unknown HW Object $hwObject\n";
		}
		
		if ($currentDirection eq "INGRESS") {
			# add return string to the END of the ingress string array
			push(@ingressStrings, $retString);

		} elsif ($currentDirection eq "EGRESS") {
			# add return string to the START of the egress string array
			unshift(@egressStrings, $retString);
		}
		
		$previousObject = $hwObject;
		$hwObject = $hwMap{$hwObject}{CONNECTS_TO};
		
		if ($hwObject eq "RSP") {
                    if ($currentDirection eq "INGRESS") {
			# print ingress strings
			foreach (@ingressStrings) {
				printf("%s", $_);
			}
		
			# start back at the interface
			$hwObject = $startingKey;
			
			$currentDirection = "EGRESS";

                        printf("\nEGRESS\n");
			printCountersHeader();       
				
                    } elsif ($currentDirection eq "EGRESS") {
			# print engress strings
			foreach (@egressStrings) {
				printf("%s", $_);
			}
                    }
		}
	}
}

sub interfaceDataplaneWalkJuggernaut {
	my %hwMap = %{$_[0]};
	my $location = $_[1];
	my $slotNum = locationStringToInteger($location);
	my $interface = $_[2];
	$interface =~ /(\w+GigE)(\d+)_(\d+)_(\d+)_(\d+)/;
	my $startingKey = $1 . $4 . "_" . $5;
	our $prettyInterface = "$1$2/$3/$4/$5";
	my $currentDirection = "INGRESS";
	
	@egressStrings = ();
	@ingressStrings = ();
 
	printCountersHeader();
	printf("%41s Dataplane\n\n", $prettyInterface);

    printf("INGRESS    HW Objects    Counters            Rate           Drops\n");
    printCountersHeader();
        
	$hwObject = $startingKey;	
	while(exists $hwMap{$hwObject}{CONNECTS_TO}) {

		if ($hwObject =~ /GigE(\d+)/) {
			$retString = showGoodInterfaceMACCounters($interface, $currentDirection);
			
			if ($currentDirection eq "EGRESS") {
				# When we reach the EGRESS MAC, we're done
				print $retString;
				last;
			}
		} elsif ($hwObject =~ /NP(\d+)/) {
			$retString = showGoodNPCounters($1, \@{$hwMap{$hwObject}{PORT_OUT}}, $location, $currentDirection);
			
		} elsif ($hwObject =~ /BRIDGE(\d+)/) {
			$retString = showGoodBridgeCounters($1, $location, $currentDirection);
			
		} elsif ($hwObject =~ /FIA_COMPLEX_(\d+)/) {
			$retString = showGoodFIAComplexCounters($1, \@{$hwMap{$previousObject}{PORT_IN}}, \@{$hwMap{$hwObject}{PORT_OUT}}, $location, $currentDirection);
		
		} elsif ($hwObject =~ /XBAR(\d+)/) {
			if ($currentDirection eq "INGRESS") {
				$xbar_port_ref = \@{$hwMap{$previousObject}{PORT_IN}};
			}
			
			$retString = showGoodXBARCounters($1, $xbar_port_ref, $location, $currentDirection);
			
		} else {
			print "Unknown HW Object $hwObject\n";
		}
		
		print $retString;
		
		if ($currentDirection eq "INGRESS") {
			# add return string to the END of the ingress string array
			#push(@ingressStrings, $retString);
			
		} elsif ($currentDirection eq "EGRESS") {
			# add return string to the END of the egress string array
			#push(@egressStrings, $retString);
		}
		
		$previousObject = $hwObject;
		$hwObject = $hwMap{$hwObject}{CONNECTS_TO};
		
		if ($hwObject =~ /EGRESS/) {
			if ($currentDirection eq "INGRESS") {
				# print ingress strings
				foreach (@ingressStrings) {
					printf("%s", $_);
				}
				
				# Back up one object to get the XBAR EGRESS counters;
				$hwObject = $previousObject;
		
				$currentDirection = "EGRESS";
                printf("\nEGRESS\n");
				printCountersHeader();                               
			}
		}
	}
	
	# print engress strings
	foreach (@egressStrings) {
		printf("%s", $_);
	}
}

sub interfaceDataplaneDrops {
	my %hwMap = %{$_[0]};
	my $location = $_[1];
	my $slotNum = locationStringToInteger($location);
	my $interface = $_[2];
	$interface =~ /(\w*GigE)(\d+)_(\d+)_(\d+)_(\d+)/;
	my $startingKey = $1 . $4 . "_" . $5;
	our $prettyInterface = "$1$2/$3/$4/$5";
	my $currentDirection = "INGRESS";

	@egressStrings = ();
	@ingressStrings = ();

    printDropHeader();
	printf("                 $prettyInterface Dataplane\n\n");

    printf("%-30s%-15s\n", "INGRESS", "Drop Counts");
	printDropHeader();
	
	$hwObject = $startingKey;	
	while(exists $hwMap{$hwObject}{CONNECTS_TO}) {

		if ($hwObject =~ /GigE(\d+)/) {
			$retString = showInterfaceMACDrops($interface, $currentDirection);
			
		} elsif ($hwObject =~ /NP(\d+)/) {
			$retString = showNPDrops($1, \@{$hwMap{$hwObject}{PORT_OUT}}, $location, $currentDirection);
			
		} elsif ($hwObject =~ /\bFIA(\d+)/) {
			$retString = showFIADrops($1, \@{$hwMap{$previousObject}{PORT_IN}}, \@{$hwMap{$hwObject}{PORT_OUT}}, $location, $currentDirection);
			
		} elsif ($hwObject =~ /XBAR(\d+)/) {
			$retString = showXBARDrops($1, \@{$hwMap{$previousObject}{PORT_IN}}, $location, $currentDirection);
				
		} else {
			print "Unknown HW Object $hwObject\n";
		}
		
		if ($currentDirection eq "INGRESS") {
			# add return string to the END of the ingress string array
			push(@ingressStrings, $retString);

		} elsif ($currentDirection eq "EGRESS") {
			# add return string to the START of the egress string array
			unshift(@egressStrings, $retString);
		}
		
		$previousObject = $hwObject;
		$hwObject = $hwMap{$hwObject}{CONNECTS_TO};
		
		if ($hwObject eq "RSP") {
			if ($currentDirection eq "INGRESS") {
				# print ingress strings
				foreach (@ingressStrings) {
					printf("%s", $_);
				}
			
				# start back at the interface
				$hwObject = $startingKey;
				
				$currentDirection = "EGRESS";

                printf("\nEGRESS\n");
				printDropHeader();      
				
			} elsif ($currentDirection eq "EGRESS") {
				# print engress strings
				foreach (@egressStrings) {
					printf("%s", $_);
				}
			}
		}
	}
}

sub interfaceDataplaneDropsJuggernaut {
	my %hwMap = %{$_[0]};
	my $location = $_[1];
	my $slotNum = locationStringToInteger($location);
	my $interface = $_[2];
	$interface =~ /(\w*GigE)(\d+)_(\d+)_(\d+)_(\d+)/;
	my $startingKey = $1 . $4 . "_" . $5;
	our $prettyInterface = "$1$2/$3/$4/$5";
	my $currentDirection = "INGRESS";

	@egressStrings = ();
	@ingressStrings = ();

    printDropHeader();
	printf("                 $prettyInterface Dataplane\n\n");

    printf("%-30s%-15s\n", "INGRESS", "Drop Counts");
	printDropHeader();
	
	$hwObject = $startingKey;	
	while(exists $hwMap{$hwObject}{CONNECTS_TO}) {

		if ($hwObject =~ /GigE(\d+)/) {
			$retString = showInterfaceMACDrops($interface, $currentDirection);
			
			if ($currentDirection eq "EGRESS") {
			    # When we reach the EGRESS MAC, we're done
				push(@egressStrings, $retString);
				last;
			}
			
		} elsif ($hwObject =~ /NP(\d+)/) {
			$retString = showNPDrops($1, \@{$hwMap{$hwObject}{PORT_OUT}}, $location, $currentDirection);
					
		} elsif ($hwObject =~ /BRIDGE(\d+)/) {
			$retString = showBridgeDrops($1, $location, $currentDirection);
			
		} elsif ($hwObject =~ /FIA_COMPLEX_(\d+)/) {
			$retString = showFIAComplexDrops($1, \@{$hwMap{$previousObject}{PORT_IN}}, \@{$hwMap{$hwObject}{PORT_OUT}}, $location, $currentDirection);
		
		} elsif ($hwObject =~ /XBAR(\d+)/) {
			if ($currentDirection eq "INGRESS") {
				$xbar_port_ref = \@{$hwMap{$previousObject}{PORT_IN}};
			}
			$retString = showXBARDrops($1, $xbar_port_ref, $location, $currentDirection);
				
		} else {
			print "Unknown HW Object $hwObject\n";
		}
		
		print $retString;
		
		if ($currentDirection eq "INGRESS") {
			# add return string to the END of the ingress string array
			#push(@ingressStrings, $retString);

		} elsif ($currentDirection eq "EGRESS") {
			# add return string to the END of the egress string array
			#push(@egressStrings, $retString);
		}
		
		$previousObject = $hwObject;
		$hwObject = $hwMap{$hwObject}{CONNECTS_TO};
		
		if ($hwObject =~ /EGRESS/) {
			if ($currentDirection eq "INGRESS") {
				# print ingress strings
				foreach (@ingressStrings) {
					printf("%s", $_);
				}
				
				# Back up one object to get the XBAR EGRESS counters;
				$hwObject = $previousObject;
		
				$currentDirection = "EGRESS";
                printf("EGRESS\n");
				printDropHeader();                               
			}
		}
	}
	
	# print engress strings
	foreach (@egressStrings) {
		printf("%s", $_);
	}
}

#*******************************************#
#           Show Drops Subroutines          #
#*******************************************#

#***************************************************************************
# Name: showInterfaceMACDrops
#
# Desc: Parses all of the MAC interface drops and error counts from
#		the "show controllers <interface>" CLI.
#
# Arguments: [0] The interface string to retrieve MAC counters for
#			 [1] The current walk direction, INGRESS/EGRESS
#
# Returns: A single string containing all the MAC drop and error strings
#***************************************************************************
sub showInterfaceMACDrops {
	my $this_function = (caller(0))[3];
	my $interface = convertInterface($_[0]);
	my $currentDirection = $_[1];
	my $retString = "";
	
	if ($currentDirection eq "INGRESS") {
		$direction = "Input";
	} elsif ($currentDirection eq "EGRESS") {
		$direction = "Output";
	}
	
	my $cliString = "show controllers <interface> stats";
	my $exec = "ethernet_show_controller -i $interface -s stats";
	my $output = `$exec`;

	if ($verbose) { print $output; }

    my @dropStrings = ();
    my @errorStrings = ();
    my @outputLines = split(/\n/, $output);
	my $total = 0;

    foreach my $line (@outputLines) {
		# Match only non zero drop/error counts
        if ($line =~ /${direction}\s+(drop.*?)\s+=\s+([1-9][0-9]*)/) {
            push(@dropStrings, sprintf("%s%-10lu\n", "$1: ", $2));
			$total += $2;
        } elsif ($line =~ /${direction}\s+(error.*?)\s+=\s+([1-9][0-9]*)/) {
            push(@errorStrings, sprintf("%s%-10lu\n", "$1: ", $2));
			$total += $2;
        }
    }
	
	my $linkStatus = getMACLinkStatus($interface);
	if ($linkStatus ne "UP") {
		my @linkStatus = split(/\n/, $linkStatus);
		foreach my $line (@linkStatus) {
			unshift(@dropStrings, sprintf("!!$line!!\n", ""));
		}
	}
	
	$retString = sprintf("%30s%lu%s\n", "$prettyInterface MAC: ", $total, ($total ? " total drops" : ""));
	
	if(@dropStrings || @errorStrings) {
		$retString = $retString . getLineSeparator("top");
		foreach my $line (@dropStrings) {
			$retString = $retString . $line;
		}
		foreach my $line (@errorStrings) {
			$retString = $retString . $line;
		}
		$retString = $retString . getLineSeparator("bottom");
	}

	return $retString;
}

#***************************************************************************
# Name: showNPDrops
#
# Desc: Parses the error and drop counter values
#		from the "show controllers np counters" CLI.
#
#		Parses the NP Interlaken error and drop counter values from the
#		"show controllers np fabric-counters tx/rx <np> <location>" CLI.
#
# Arguments: [0] The NP to query
#            [1] Reference to the NP Fabric port array (to FIA)
#			 [2] The location string
#			 [3] The current walk direction, INGRESS/EGRESS
#
# Returns: A single string containing all the NP drop and error strings
#***************************************************************************
sub showNPDrops {
	my $this_function = (caller(0))[3];
    my ($np, $np_port_ref) = @_;
	my $slot = locationStringToInteger($_[2]);
	my $currentDirection = $_[3];
	
	my $retString = "";

    my @np_ports = @{$np_port_ref};
	
	my $cliString = "show controllers np counters np$np $_[1]";
	my $exec = "prm_np_show counters -e 0x$np -s $slot";
    my $output = `$exec`;
  	
	my @outputLines = split(/\n/, $output);
	if ($verbose) { print $output; }
	
	my @egressStrings = ();
	my @ingressStrings = ();
	my @generalStrings = ();
	
	my $directionTotal = 0;
	my $generalTotal = 0;

    foreach my $line (@outputLines) {
		# Match only non zero drop/error counts
        if ($line =~ /\d+\s+(\w+_DROP|\w+_ERR|\w+_EXCD)\s+(\d+)/) {
			my $desc = $1;
			my $count = $2;
			if ($currentDirection eq "INGRESS" && $line =~ /_ING/) {
				#print "\nINGRESS LINE: $line :: DESC: $1  :: COUNT: $2\n\n";
				push(@ingressStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
				$directionTotal += $count;
			} elsif ($currentDirection eq "EGRESS" && $line =~ /_EGR/) {
				push(@egressStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
				$directionTotal += $count;
			} else {
				if ($currentDirection eq "INGRESS") {
					if ($line =~ /_EGR/) {
						next;
					}
					push(@generalStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
					$generalTotal += $count;
				}
				if ($currentDirection eq "EGRESS") {
					if ($line =~ /_ING/) {
						next;
					}
					push(@generalStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
					$generalTotal += $count;
				}
			}
        }
    }
	
	$capDir = ucfirst lc($currentDirection);
	$retString = $retString . sprintf("%30s%lu%s\n", "NP $np uCode $capDir: ", $directionTotal, ($directionTotal ? " total drops" : ""));
	
	if ($currentDirection eq "INGRESS") {
		if (@ingressStrings) {
			$retString = $retString . getLineSeparator("top");
			foreach my $line (@ingressStrings) {	
				$retString = $retString . $line;
			}
			$retString = $retString . getLineSeparator();
		}
	} else {
		if (@egressStrings) {
			$retString = $retString . getLineSeparator("top");
			foreach my $line (@egressStrings) {	
				$retString = $retString . $line;
			}
			$retString = $retString . getLineSeparator();
		}
	}
	
	$retString = $retString . sprintf("%30s%lu%s\n", "NP $np uCode (Dir Unknown): ", $generalTotal, ($generalTotal ? " total drops" : ""));

	if (@generalStrings) {
		$retString = $retString . getLineSeparator("top");
		foreach my $line (@generalStrings) {	
			$retString = $retString . $line;
		}
		$retString = $retString . getLineSeparator();
	}
	
	if ($currentDirection eq "INGRESS") {
		$cliString = "show controllers np fabric-counters tx <np> <location>";
		$exec = "prm_np_show xaui -x 0 -t -e 0x$np -s $slot";
	} elsif ($currentDirection eq "EGRESS") {
		$cliString = "show controllers np fabric-counters rx <np> <location>";
		$exec = "prm_np_show xaui -x 0 -r -e 0x$np -s $slot";
	}
	
	$output = `$exec`;
	if ($verbose) { print $output; }

 	@outputLines = split(/\n/, $output);
	
	@egressStrings = ();	
	@ingressStrings = ();
	
	my $total = 0;
	my $npPort = -1;
	
	foreach (@np_ports) {
		foreach my $line (@outputLines) {
			if ($line =~ /interface\s(\d+)\scounters/) {
					$npPort = $1;
			}
			if ($line =~ /(\w+_BAD\w*|\w+_ERROR\w*|\w+_FAILURE\w*)\s+(0x\w+)\s+(\w+)/) {
				my $count = oct($2 . $3);
				if ($count == 0) {
					next;
				}
				if ($currentDirection eq "INGRESS") {
					push(@ingressStrings, sprintf("%s%-10lu\n", "IKLN $npPort  $1: ", $count));
					$total += $count;
				} elsif ($currentDirection eq "EGRESS") {
					push(@egressStrings, sprintf("%s%-10lu\n", "IKLN $npPort  $1: ", $count));
					$total += $count;
				}
			}
		}
	}
	
	if ($currentDirection eq "INGRESS") {
		if ($globalHwMap_ref->{NAME} ne "Juggernaut") {
			$retString = $retString . sprintf("%30s%-10s\n", "NP $np TM: ", "N/A");
		}
		$retString = $retString . sprintf("%30s%lu%s\n", "NP $np Output: ", $total, ($total ? " total drops" : ""));
		if (@ingressStrings) {
			$retString = $retString . getLineSeparator("top");
			foreach my $line (@ingressStrings) {	
				$retString = $retString . $line;
			}
			$retString = $retString . getLineSeparator();
		}
	} else {
		if (@egressStrings) {
			$retString = getLineSeparator()  . $retString;
			foreach my $line (@egressStrings) {	
				$retString = $line . $retString;
			}
			$retString = getLineSeparator("top") . $retString;
		}
		$retString = sprintf("%30s%lu%s\n", "NP $np Input: ", $total, ($total ? " total drops" : "")) . $retString;
		if ($globalHwMap_ref->{NAME} ne "Juggernaut") {
			$retString = $retString . sprintf("%30s%-10s%12s\n", "NP $np TM: ", "N/A", "");
		} else {
			$cliString = "show controllers np tm counters <np> <location>";
			$exec = "prm_np_show TM counters -e 0x$np -s $slot";
			$output = `$exec`;
			if ($verbose) { print $output; }
			my $secondNP = $np + 1;
			if ($output =~ /drop paks: (\d+)/) {
                my $count = $1;

                $exec = "prm_np_show TM counters -e 0x$secondNP -s $slot";
                $output = `$exec`;
                if ($verbose) { print $output; }

					if ($output =~ /drop paks: (\d+)/) {
						$count += $1;
					}
                
					if ($count eq "0") {
						$retString = $retString . sprintf("%30s%lu%s\n", "NP[$np,$secondNP] TM: ", 0, "");
					} else {	
						$retString = $retString . sprintf("%30s%lu%s\n", "NP[$np,$secondNP] TM: ", $count, " total drops");
					}
                } else {
                    print "Parsing $_[1] TM counters failed for '$exec'\n";
                    print "Use the following CLI to gather these stats:\n";
                    print "'$cliString'\n";
                    return "";
                }
		}
	}
	
	return $retString;
}

#***************************************************************************
# Name: showBridgeDrops
#
# Desc: Parses error and drop counter values from the
#		"show controllers fabric bridge stats" CLI.
#
# Arguments: [0] The IMUX/Bridge to query
#			 [1] The location string
#			 [2] The current walk direction, INGRESS/EGRESS
#***************************************************************************
sub showBridgeDrops {
	my $this_function = (caller(0))[3];
	my $bridge = $_[0];
	my $slot = locationStringToInteger($_[1]);
	my $currentDirection = $_[2];
	
	my $retString = "";
	my @dropStrings = ();
	
	$cliString = "show controllers fabric fia bridge stats location $_[1]";
	$exec = "fsi_show -n $slot -f -b -S";
	$output = `$exec`;
	if ($verbose) { print $output; }
	
	my @outputLines = split(/\n/, $output);
	
	my $onRightBridge = 0;
	my $dir;
	if ($currentDirection eq "INGRESS") {
		$dir = "in";
	} elsif ($currentDirection eq "EGRESS") {
		$dir = "eg";
	}
	
	my $total = 0;
		
	foreach my $line (@outputLines) {
		if ($line =~ /Category: bridge_${dir}-(\d+)/) {
			if ($1 == $bridge) {
				$onRightBridge = 1;
			} else {
				$onRightBridge = 0;
			}
		}
		if ($onRightBridge) {
			if ($line =~ /\s\s+(.*?Err.*?|.*?drp.*?)\s\s+([1-9][0-9]*)/) {	
				my $desc = $1;
				my $count = $2;
				
				push(@dropStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
				$total += $count;
			}
		}
	}
		
	my @linkStatus = getBridgeLinkStatus($bridge, $slot);
	if (@linkStatus) {
		foreach my $line (@linkStatus) {
			unshift(@dropStrings, sprintf("!!$line!!\n", ""));
		}
	}
	
	$retString = sprintf("%30s%lu%s\n", "Bridge $bridge: ", $total, ($total ? " total drops" : ""));
	
	if (@dropStrings) {
		$retString = $retString . getLineSeparator("top");
		foreach my $line (@dropStrings) {
			$retString = $retString . $line;
		}
		$retString = $retString . getLineSeparator();
	}
	
	return $retString;
}

#***************************************************************************
# Name: showFIADrops
#
# Desc: Parses the error and drop counter values of the specified NP to the FIA
#
# Arguments: [0] The FIA to gather counters from
#			 [1] Reference to the FIA network port array (from NP)
#			 [2] Reference to the FIA fabric port array (to XBAR)			 
#			 [3] The location string
#			 [4] The current walk direction INGRESS/EGRESS
#***************************************************************************
sub showFIADrops {
	my ($fia, $fia_nw_ref, $fia_fb_ref) = @_;
	my $slot = locationStringToInteger($_[3]);
	my $currentDirection = $_[4];
	my $retString = "";
	
	@fia_nw = @{$fia_nw_ref};
	@fia_fb = @{$fia_fb_ref};
	
	$cliString = "show controllers fabric fia stats <location>";
	$exec = "fsi_show -n $slot -f -s";
	my $output = `$exec`;

	if ($verbose) { print $output; }
	
	@dropStrings = ();
	
	my $total = 0;
	
	if ($currentDirection eq "INGRESS" && $output =~ /FIA-${fia}.*?Ingress\sdrop:\s+([1-9][0-9]*)/s) {
		# Found ingress drops, execute FIA drops CLI and grab all non-zero values
		
		$exec = "fsi_show -n $slot -f -d -N";
		$output = `$exec`;
		
		if ($verbose) { print $output; }

		@outputLines = split(/\n/, $output);
		my $currentFIA = 0;
		
		foreach my $line (@outputLines) {
			if ($line =~ /FIA-(\d+)/) {
				if ($1 == $fia) {
					$currentFIA = 1;
				} else {
					$currentFIA = 0;
				}
			}
			if ($currentFIA) {
				if ($line =~ /(\w+.*?)\s+([1-9][0-9]*)/) {	
					my $desc = $1;
					my $count = $2;
					if ($1 =~ /(\d)/) {
						my $port = $1;
						if (grep( /^${port}$/, @fia_nw)) {
							push(@dropStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
							$total += $count;
						}
					} else {
						push(@dropStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
						$total += $count;
					}
				}
			}
		}
		
		# Found ingress errors, execute FIA errors CLI and grab all non-zero values
		$exec = "fsi_show -n 2049 -f -e -N";
		$output = `$exec`;
							 
		if ($verbose) { print $output; }

		@outputLines = split(/\n/, $output);
		
		$currentFIA = 0;
		foreach my $line (@outputLines) {
			if ($line =~ /FIA-(\d+)/) {
				if ($1 == $fia) {
					$currentFIA = 1;
				} else {
					$currentFIA = 0;
				}
			}
			if ($currentFIA) {
				if ($line =~ /(\w+.*?)\s+([1-9][0-9]*)/) {
					my $desc = $1;
					my $count = $2;
					
					push(@dropStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
					$total += $count;
				}
			}
		}
	}
	
	if ($currentDirection eq "EGRESS" && $output =~ /FIA-${fia}.*?Egress\sdrop:\s+([1-9][0-9]*)/s) {
		# Found egress drops, execute FIA drops CLI and grab all non-zero values
		
		$exec = "fsi_show -n $slot -f -d -E";
		$output = `$exec`;
		
		if ($verbose) { print $output; }

		@outputLines = split(/\n/, $output);
		
		$currentFIA = 0;
		foreach my $line (@outputLines) {
			if ($line =~ /FIA-(\d+)/) {
				if ($1 == $fia) {
					$currentFIA = 1;
				} else {
					$currentFIA = 0;
				}
			}	
			if ($currentFIA) {
				if ($line =~ /(\w+.*?)\s+([1-9][0-9]*)/) {	
					my $desc = $1;
					my $count = $2;
					if ($1 =~ /(\d)/) {
						my $port = $1;
						if (grep( /^${port}$/, @fia_nw)) {
							push(@dropStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
							$total += $count;
						}
					} else {
						push(@dropStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
						$total += $count;
					}
				}
			}
		}
		
		# Found egress errors, execute FIA errors CLI and grab all non-zero values
		
		$exec = "fsi_show -n 2049 -f -e -N";
		$output = `$exec`;
							 
		if ($verbose) { print $output; }

		@outputLines = split(/\n/, $output);
		
		$currentFIA = 0;
		foreach my $line (@outputLines) {
			if ($line =~ /FIA-(\d+)/) {
				if ($1 == $fia) {
					$currentFIA = 1;
				} else {
					$currentFIA = 0;
				}
			}
			if ($currentFIA) {
				if ($line =~ /(\w+.*?)\s+([1-9][0-9]*)/) {	
					my $desc = $1;
					my $count = $2;
					
					push(@dropStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
					$total += $count;
				}
			}
		}
	}
	
	my @linkStatus = getFIALinkStatus($fia, $fia_nw_ref, $slot);
	if (@linkStatus) {
		foreach my $line (@linkStatus) {
			unshift(@dropStrings, sprintf("!!$line!!\n", ""));
		}
	}
	
	$retString = sprintf("%30s%lu%s\n", "FIA $fia: ", $total, ($total ? " total drops" : ""));
	
	if (@dropStrings) {
		$retString = $retString . getLineSeparator("top");
		foreach my $line (@dropStrings) {
			$retString = $retString . $line;
		}
		$retString = $retString . getLineSeparator();
	}
	
	return $retString;
}

#***************************************************************************
# Name: showFIAComplexDrops
#
# Desc: Parses the error and drop counter values of the specified NP to the FIA Complex
#
# Arguments: [0] The first FIA in the complex to gather counters from
#			 [1] Reference to the FIA network port array (from NP)
#			 [2] Reference to the FIA fabric port array (to XBAR)			 
#			 [3] The location string
#			 [4] The current walk direction INGRESS/EGRESS
#***************************************************************************
sub showFIAComplexDrops {
	my ($firstFIA, $fia_nw_ref, $fia_fb_ref) = @_;
	my $slot = locationStringToInteger($_[3]);
	my $currentDirection = $_[4];
	my $retString = "";
	
	@fia_nw = @{$fia_nw_ref};
	@fia_fb = @{$fia_fb_ref};
	
	$cliString = "show controllers fabric fia stats $_[3]";
	$exec = "fsi_show -n $slot -f -s";
	$output = `$exec`;
    if ($verbose) { print $output; }
	
	@dropStrings = ();
	
	my $total = 0;
	
	for ($fia = $firstFIA; $fia <= ($firstFIA + 1); $fia++) {
		if ($currentDirection eq "INGRESS" && $output =~ /FIA-${fia}.*?Ingress\sdrop:\s+([1-9][0-9]*)/s) {
			# Found ingress drops, execute FIA drops CLI and grab all non-zero values
			
			$exec = "fsi_show -n $slot -f -d -N";
			$output = `$exec`;
			
			if ($verbose) { print $output; }

			@outputLines = split(/\n/, $output);
			my $currentFIA = 0;
			
			foreach my $line (@outputLines) {
				if ($line =~ /FIA-(\d+)/) {
					if ($1 == $fia) {
						$currentFIA = 1;
					} else {
						$currentFIA = 0;
					}
				}
				if ($currentFIA) {
					if ($line =~ /(\w+.*?)\s+([1-9][0-9]*)/) {	
						my $desc = $1;
						my $count = $2;
						if ($1 =~ /(\d)/) {
							my $port = $1;
							if (grep( /^${port}$/, @fia_nw)) {
								push(@dropStrings, sprintf("%s%-10lu\n", "FIA $fia  $desc: ", $count));
								$total += $count;
							}
						} else {
							push(@dropStrings, sprintf("%s%-10lu\n", "FIA $fia  $desc: ", $count));
							$total += $count;
						}
					}
				}
			}
			
			# Found ingress errors, execute FIA errors CLI and grab all non-zero values
			$exec = "fsi_show -n 2049 -f -e -N";
			$output = `$exec`;
								 
			if ($verbose) { print $output; }

			@outputLines = split(/\n/, $output);
			
			$currentFIA = 0;
			foreach my $line (@outputLines) {
				if ($line =~ /FIA-(\d+)/) {
					if ($1 == $fia) {
						$currentFIA = 1;
					} else {
						$currentFIA = 0;
					}
				}
				if ($currentFIA) {
					if ($line =~ /(\w+.*?)\s+([1-9][0-9]*)/) {
						my $desc = $1;
						my $count = $2;
						
						push(@dropStrings, sprintf("%s%-10lu\n", "FIA $fia  $desc: ", $count));
						$total += $count;
					}
				}
			}
		}
		
		if ($currentDirection eq "EGRESS" && $output =~ /FIA-${fia}.*?Egress\sdrop:\s+([1-9][0-9]*)/s) {
			# Found egress drops, execute FIA drops CLI and grab all non-zero values
			
			$exec = "fsi_show -n $slot -f -d -E";
			$output = `$exec`;
			
			if ($verbose) { print $output; }

			@outputLines = split(/\n/, $output);
			
			$currentFIA = 0;
			foreach my $line (@outputLines) {
				if ($line =~ /FIA-(\d+)/) {
					if ($1 == $fia) {
						$currentFIA = 1;
					} else {
						$currentFIA = 0;
					}
				}	
				if ($currentFIA) {
					if ($line =~ /(\w+.*?)\s+([1-9][0-9]*)/) {	
						my $desc = $1;
						my $count = $2;
						if ($1 =~ /(\d)/) {
							my $port = $1;
							if (grep( /^${port}$/, @fia_nw)) {
								push(@dropStrings, sprintf("%s%-10lu\n", "FIA $fia  $desc: ", $count));
								$total += $count;
							}
						} else {
							push(@dropStrings, sprintf("%s%-10lu\n", "FIA $fia  $desc: ", $count));
							$total += $count;
						}
					}
				}
			}
			
			# Found egress errors, execute FIA errors CLI and grab all non-zero values
			
			$exec = "fsi_show -n 2049 -f -e -N";
			$output = `$exec`;
								 
			if ($verbose) { print $output; }

			@outputLines = split(/\n/, $output);
			
			$currentFIA = 0;
			foreach my $line (@outputLines) {
				if ($line =~ /FIA-(\d+)/) {
					if ($1 == $fia) {
						$currentFIA = 1;
					} else {
						$currentFIA = 0;
					}
				}
				if ($currentFIA) {
					if ($line =~ /(\w+.*?)\s+([1-9][0-9]*)/) {	
						my $desc = $1;
						my $count = $2;
						
						push(@dropStrings, sprintf("%s%-10lu\n", "$desc: ", $count));
						$total += $count;
					}
				}
			}
		}
		
		my @linkStatus = getFIALinkStatus($fia, $fia_nw_ref, $slot);
		if (@linkStatus) {
			foreach my $line (@linkStatus) {
				unshift(@dropStrings, sprintf("!!FIA $fia  $line!!\n", ""));
			}
		}
	}
	
	my $secondFIA = $firstFIA + 1;
	$retString = sprintf("%30s%lu%s\n", "FIA[$firstFIA,$secondFIA] Complex: ", $total, ($total ? " total drops" : ""));

	if (@dropStrings) {
		$retString = $retString . getLineSeparator("top");
		foreach my $line (@dropStrings) {
			$retString = $retString . $line;
		}
		$retString = $retString . getLineSeparator();
	}
	
	return $retString;
}

#***************************************************************************
# Name: showXBARDrops
#
# Desc: 
#
# Arguments: [0] The XBAR instance to query
#			 [1] The XBAR port array (from FIA)
#			 [2] The location string
#			 [3] The current walk direction
#***************************************************************************
sub showXBARDrops {
    my ($xbar, $xbar_port_ref) = @_;
    my $slot = locationStringToInteger($_[2]);
    my $currentDirection = $_[3];
	my @dropStrings = ();
	my $retString = "";
    
    my @xbar_ports = @{$xbar_port_ref};
	# Add RSP ports to this array since we want to check link status on them as well
	my @rsp_ports = ('9', '12', '14', '16', '7', '11', '15', '17');
	push(@xbar_ports, @rsp_ports);
	
	$xbar_port_ref = \@xbar_ports;
	
	my @linkStatus = getXBARLinkStatus($xbar, $xbar_port_ref, $slot);
	if (@linkStatus) {
		foreach my $line (@linkStatus) {
			unshift(@dropStrings, sprintf("!!$line!!\n", ""));
		}
	}
	
	# Currently don't have any drop or error counts to parse for XBAR	
	$retString = sprintf("%30s%lu\n", "XBAR $xbar: ", 0);
	
	if (@dropStrings) {
		$retString = $retString . getLineSeparator("top");
		foreach my $line (@dropStrings) {
			$retString = $retString . $line;
		}
		$retString = $retString . getLineSeparator();
	}
	
	return $retString;
}

#*******************************************#
#          Show Counters Subroutines        #
#*******************************************#

#***************************************************************************
# Name: showGoodInterfaceMACCounters
#
# Desc: Prints the total amount of MAC interface input packets from the
#		"show controllers <interface>" CLI and prints the results.
#
# Arguments: [0] The interface string to retrieve MAC counters for
#			 [1] The current walk direction, INGRESS/EGRESS
#***************************************************************************
sub showGoodInterfaceMACCounters {
	my $this_function = (caller(0))[3];
	my $interface = convertInterface($_[0]);
	my $currentDirection = $_[1];
	my $retString = "";
	
	# Get total number of drops for interface/MAC
	my $drops = -1;
	my $dropOutput = showInterfaceMACDrops($_[0], $currentDirection);
	if ($dropOutput =~ /MAC: (\d+)/) {
		if ($1 eq "0") {
	        $drops = $1;
		} else {
			$drops = "-" . $1;
		}
	}
	
	if ($currentDirection eq "INGRESS") {
		$direction = "Input";
	} elsif ($currentDirection eq "EGRESS") {
		$direction = "Output";
	}

	$cliString = "show controllers <interface> stats";
	$exec = "ethernet_show_controller -i $interface -s stats";
	$output = `$exec`;
	if ($verbose) { print $output; }

    if ($output =~ /${direction} total packets\s+=\s+(\d+)/) {
		$retString = sprintf("%25s%-10lu%15s%15s\n", "$prettyInterface MAC: ", $1, "", $drops);
    } else {
		print "Parsing $_[1] interface MAC counters failed for '$exec'\n";
		print "Use the following CLI to gather these stats:\n";
		print "'$cliString'\n";
		return "";
    }
	
	return $retString;
}

#***************************************************************************
# Name: showGoodNPCounters
#
# Desc: Prints the PARSE_ENET_RECEIVE_CNT and MDF_TX_FABRIC counter values
#		from the "show controllers np counters" CLI.
#
#		Prints the NP Interlaken transmit counter values from the
#		"show controllers np fabric-counters tx/rx <np> <location>" CLI.
#
#		Prints the TM counter values from the
#		"show controller np tm count all <location>" CLI.
#
# Arguments: [0] The NP to query
#            [1] Reference to the NP Fabric port array (to FIA)
#			 [2] The location string
#			 [3] The current walk direction, INGRESS/EGRESS
#***************************************************************************
sub showGoodNPCounters {
	my $this_function = (caller(0))[3];
    my ($np, $np_port_ref) = @_;
	my $slot = locationStringToInteger($_[2]);
	my $currentDirection = $_[3];
	
	my $retString = "";
	my $tm = 0;
    my $parseIngRate, $parseEgrRate, $modifyIngRate, $modifyEgrRate;

    @np_ports = @{$np_port_ref};
	
	# Get total number of drops for NP uCode and Fabric Ports
	my $uCodeDrops = -1;
	my $fabricDrops = -1;
	my $dropOutput = showNPDrops($np, $np_port_ref, $_[2], $currentDirection);
	my $capDir = ucfirst lc($currentDirection);
	if ($dropOutput =~ /uCode ${capDir}: (\d+).*?Unknown\): (\d+)/s) {
		$uCodeDrops = $1 + $2;
		if ($uCodeDrops ne "0") {
			$uCodeDrops = "-" . $uCodeDrops;	
		}
	} else {
		print "NO MATCH!\n\n$dropOutput\n";
	}
	
	if ($dropOutput =~ /NP ${np} \w+: (\d+)/) {
		$fabricDrops = $1;
		if ($1 eq "0") {
			$fabricDrops = $1;
		} else {
			$fabricDrops = "-" . $1;
		}
	} else {
		print "NO MATCH!\n\n$dropOutput\n";
	}
	
	$cliString = "show controllers np counters np$np $_[1]";
	$exec = "prm_np_show counters -e 0x$np -s $slot";
	$output = `$exec`;
        $output = `$exec`;
        $output = `$exec`;
        
	if ($verbose) { print $output; }
	
	if ($currentDirection eq "INGRESS") {
		if ($output =~ /PARSE_ENET_RECEIVE_CNT\s+(\d+)\s+(\d+)/) {
			$parseCount = $1;
                        $parseIngRate = $2;
		} else {
			$parseCount = 0;
                        $parseIngRate = 0;
		}
		if ($output =~ /MDF_TX_FABRIC\s+(\d+)\s+(\d+)/) {
			$modifyCount = $1;
                        $modifyIngRate = $2;
		} else {
			$modifyCount = 0;
                        $modifyIngRate = 0;
		}
		$temp = sprintf("%25s%-10lu%15s%15s\n", "NP $np uCode Input: ", $parseCount, ($parseIngRate . " pps"), $uCodeDrops);
		$retString = $retString . $temp;
		
		$temp = sprintf("%25s%-10lu%15s%15s\n", "NP $np uCode Output: ", $modifyCount, ($modifyIngRate . " pps"), 0);
		$retString = $retString . $temp;
		
	} elsif ($currentDirection eq "EGRESS") {
		if ($output =~ /PARSE_FAB_RECEIVE_CNT\s+(\d+)\s+(\d+)/) {
			$parseCount = $1;
            $parseEgrRate = $2;
		} else {
			$parseCount = 0;
            $parseEgrRate = 0;
		}
		
		if ($output =~ /MDF_TX_WIRE\s+(\d+)\s+(\d+)/) {
			$modifyCount = $1;
            $modifyEgrRate = $2;
		} else {
			$modifyCount = 0;
            $modifyEgrRate = 0;
		}
		$temp = sprintf("%25s%-10lu%15s%15s\n", "NP $np uCode Input: ", $parseCount, ($parseEgrRate . " pps"), $uCodeDrops);
		$retString = $retString . $temp;
		
		$temp = sprintf("%25s%-10lu%15s%15s\n", "NP $np uCode Output: ", $modifyCount, ($modifyEgrRate . " pps"), 0);
		$retString = $retString . $temp;
	}

	# TM COUNTERS DEEMED NOT HELPFUL, NEED TO DISTINGUISH BETWEEN EGRESS/INGRESS
	# $cliString = "show controller np tm counters np$np tm tm$tm location $_[1]";
	# $exec = "prm_np_show TM counters -e 0x$np -s $slot -t 0x$tm";
	# $output = `$exec`;
	
	# if ($output =~ /NP\s+${np}\s+TM\s+${tm}.*?\n.*?\nxmt paks:\s+(\d+)/,) {
		# $temp = sprintf("%30s: %d\n", "NP $np TM", $1);
		# $retString = $retString . $temp;
	# } else {
		# printf("Parsing NP %s TM %s counters failed for '%s'\n", $np, $tm, $exec);
		# printf("Use the following CLI to gather these stats:\n");
		# printf("'%s'\n\n", $cliString);
	# }
	
	if ($currentDirection eq "INGRESS") {
		$cliString = "show controllers np fabric-counters tx <np> <location>";
		$exec = "prm_np_show xaui -x 0 -t -e 0x$np -s $slot";
	} elsif ($currentDirection eq "EGRESS") {
		$cliString = "show controllers np fabric-counters rx <np> <location>";
		$exec = "prm_np_show xaui -x 0 -r -e 0x$np -s $slot";
	}
	
	$output = `$exec`;
	if ($verbose) { print $output; }
	
    if ($currentDirection eq "INGRESS") {
        $txCount = 0;
        foreach (@np_ports) {
            $np_port = $_ - 10; # Interlaken 10-12 maps to NP Fabric ports 0-2
			if ($output =~ /interface\s${np_port}\scounters.*?\n\s*INTERLAKEN_TX_PACKETS\s+(0x\w+)\s+(\w+)/) {
						eval { $txCount += oct($1 . $2); };
			} else {
				print "Parsing Ingress NP to Fabric counter failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
        }
		if ($globalHwMap_ref->{NAME} ne "Juggernaut") {
			$retString = $retString . sprintf("%25s%-10s%15s%15s\n", "NP $np TM Output: ", "N/A", "", "N/A");
		}
        $temp = sprintf("%25s%-10lu%15s%15s\n", "NP $np Output: ", $txCount, "", $fabricDrops);
        $retString = $retString . $temp;
	} elsif ($currentDirection eq "EGRESS") {
        $rxCount = 0;
        foreach (@np_ports) {
            $np_port = $_ - 10; # Interlaken 10-12 maps to NP Fabric ports 0-2
            if ($output =~ /interface\s${np_port}\scounters.*?\n\s*INTERLAKEN_RX_PACKETS\s+(0x\w+)\s+(\w+)/) {
                eval { $rxCount += oct($1 . $2); };
            } else {
				print "Parsing Egress NP Fabric counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
            }
        }
        $temp = sprintf("%25s%-10lu%15s%15s\n", "NP $np Input: ", $rxCount,  "", $fabricDrops);
        $retString = $temp . $retString;
		
		if ($globalHwMap_ref->{NAME} ne "Juggernaut") {
			$retString = $retString . sprintf("%25s%-10s%15s%15s\n", "NP $np TM Output: ", "N/A", "", "N/A");
		} else {
			$cliString = "show controllers np fabric-counters rx <np> <location>";
			$exec = "prm_np_show TM counters -e 0x$np -s $slot";
			$output = `$exec`;
			if ($verbose) { print $output; }
			my $secondNP = $np + 1;
			if ($output =~ /drop paks: (\d+)/) {
				$retString = $retString . sprintf("%25s%-10s%15s%15s\n", "NP[$np,$secondNP] TM Output: ", "N/A", "", 0);
			}
		}
	}
	
	return $retString;
}

#***************************************************************************
# Name: showGoodBridgeCounters
#
# Desc: Parses the to/from FIA/NP counter values from the
#		"show controllers fabric fia bridge stats" CLI.
#
# Arguments: [0] The IMUX/Bridge to query
#			 [1] The location string
#			 [2] The current walk direction, INGRESS/EGRESS
#***************************************************************************
sub showGoodBridgeCounters {
	my $this_function = (caller(0))[3];
	my $bridge = $_[0];
	my $slot = locationStringToInteger($_[1]);
	my $currentDirection = $_[2];
	
	my $retString = "";

	# Get total number of drops for FIA bridge
	my $drops = -1;
	my $dropOutput = showBridgeDrops($bridge, $_[1], $currentDirection);
	if ($dropOutput =~ /Bridge ${bridge}: (\d+)/) {
		if ($1 eq "0") {
	        $drops = $1;
		} else {
			$drops = "-" . $1;
		}
	}
	
	$cliString = "show controllers fabric fia bridge stats location $_[1]";
	$exec = "fsi_show -n $slot -f -b -S";
	$output = `$exec`;
	if ($verbose) { print $output; }
	
	if ($currentDirection eq "INGRESS") {
		if ($output =~ /bridge_in-${bridge}.*?\s+To FIA0 L-0\s+(\d+).*?\s+To FIA0 L-1\s+(\d+).*?\s+To FIA1 L-0\s+(\d+).*?\s+To FIA1 L-1\s+(\d+).*?Frm NPU Pkt\s+(\d+)/s) {
			$outputCount = $1 + $2 + $3 + $4;
			$inputCount = $5;
		} else {
			print "Parsing $_[1] interface MAC counters failed for '$exec'\n";
			print "Use the following CLI to gather these stats:\n";
			print "'$cliString'\n";
			return "";
		}

		$temp = sprintf("%25s%-10lu%15s%15s\n", "Bridge $bridge Input: ", $inputCount, "", $drops);
		$retString = $retString . $temp;
			
		$temp = sprintf("%25s%-10lu%15s%15s\n", "Bridge $bridge Output: ", $outputCount, "", 0);
		$retString = $retString . $temp;
		
	} elsif ($currentDirection eq "EGRESS") {
		if ($output =~ /bridge_eg-${bridge}.*?\s+FIA0 L-0 Pkt\s+(\d+).*?\s+FIA0 L-1 Pkt\s+(\d+).*?\s+FIA1 L-0 Pkt\s+(\d+).*?\s+FIA1 L-1 Pkt\s+(\d+).*?To NPU\s+(\d+)/s) {
			$inputCount = $1 + $2 + $3 + $4;
			$outputCount = $5;
		} else {
			print "Parsing $_[1] interface MAC counters failed for '$exec'\n";
			print "Use the following CLI to gather these stats:\n";
			print "'$cliString'\n";
			return "";
		}
		
		$temp = sprintf("%25s%-10lu%15s%15s\n", "Bridge $bridge Input: ", $inputCount, "", $drops);
		$retString = $retString . $temp;
			
		$temp = sprintf("%25s%-10lu%15s%15s\n", "Bridge $bridge Output: ", $outputCount, "", 0);
		$retString = $retString . $temp;
	}
	
	return $retString;
}

#***************************************************************************
# Name: showGoodFIACounters
#
# Desc: Prints the count of packets from the specified NP to the FIA
#
#		Prints the aggregate count of packets from the specified FIA to the XBAR
#
# Arguments: [0] The FIA to gather counters from
#			 [1] Reference to the FIA network port array (from NP)
#			 [2] Reference to the FIA fabric port array (to XBAR)			 
#			 [3] The location string
#			 [4] The current walk direction INGRESS/EGRESS
#***************************************************************************
sub showGoodFIACounters {
	my ($fia, $fia_nw_ref, $fia_fb_ref) = @_;
	my $slot = locationStringToInteger($_[3]);
	my $currentDirection = $_[4];
	my $retString = "";
	
	@fia_nw = @{$fia_nw_ref};
	@fia_fb = @{$fia_fb_ref};
	
	# Get total number of drops for FIA
	my $drops = -1;
	my $dropOutput = showFIADrops($fia, $fia_nw_ref, $fia_fb_ref, $_[3], $currentDirection);
	if ($dropOutput =~ /FIA ${fia}: (\d+)/) {
		if ($1 eq "0") {
	        $drops = $1;
		} else {
			$drops = "-" . $1;
		}
	}
	
	$cliString = "show controllers fabric fia stats <location>";
	$exec = "fsi_show -n $slot -f -s";
	$output = `$exec`;
    if ($verbose) { print $output; }
	
	if ($currentDirection eq "INGRESS") {
		$rxCount = 0;
		foreach (@fia_nw) {
			if ($output =~ /FIA-${fia}.*?From Line Interface\[${_}\]\s+(\d+)/s) {
				$rxCount += $1;
			} else {
				print "Parsing Ingress NP to FIA counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
		
		$txCount = 0;
		foreach (@fia_fb) {
			if ($output =~ /FIA-${fia}.*?To Unicast Xbar\[${_}\]\s+(\d+)/s) {
				$txCount += $1;
			} else {
				print "Parsing Ingress FIA to XBAR counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
	} elsif ($currentDirection eq "EGRESS") {
	
		$txCount = 0;
		foreach (@fia_nw) {
			if ($output =~ /FIA-${fia}.*?To Line Interface\[${_}\]\s+(\d+)/s) {
				$txCount += $1;
			} else {
				print "Parsing Egress FIA to NP counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
		
		$rxCount = 0;
		foreach (@fia_fb) {
			if ($output =~ /FIA-${fia}.*?From Unicast Xbar\[${_}\]\s+(\d+)/s) {
				$rxCount += $1;
			} else {
				print "Parsing Egress XBAR to FIA counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
	}

	$retString = $retString . sprintf("%25s%-10lu%15s%15s\n", "FIA $fia Input: ", $rxCount, "", $drops);
	$retString = $retString . sprintf("%25s%-10lu%15s%15s\n", "FIA $fia Output: ", $txCount, "", 0);
	
	return $retString;
}

#***************************************************************************
# Name: showGoodFIAComplexCounters
#
# Desc: Prints the count of packets from the specified NP to the FIA Complex
#		Prints the aggregate count of packets from the specified FIA Complex to the XBAR
#		(Currently only exists on Juggernaut 2x100GE cards)
#
# Arguments: [0] The first FIA in the complex to gather counters from
#			 [1] Reference to the FIA network port array (from NP)
#			 [2] Reference to the FIA fabric port array (to XBAR)			 
#			 [3] The location string
#			 [4] The current walk direction INGRESS/EGRESS
#***************************************************************************
sub showGoodFIAComplexCounters {
	my ($fia, $fia_nw_ref, $fia_fb_ref) = @_;
	my $slot = locationStringToInteger($_[3]);
	my $currentDirection = $_[4];
	my $retString = "";
	
	@fia_nw = @{$fia_nw_ref};
	@fia_fb = @{$fia_fb_ref};
	
	# Get total number of drops for FIA Complex
	my $drops = -1;
	my $dropOutput = showFIAComplexDrops($fia, $fia_nw_ref, $fia_fb_ref, $_[3], $currentDirection);
	if ($dropOutput =~ /Complex: (\d+)/) {
		if ($1 eq "0") {
	        $drops = $1;
		} else {
			$drops = "-" . $1;
		}
	}
	
	$cliString = "show controllers fabric fia stats $_[3]";
	$exec = "fsi_show -n $slot -f -s";
	$output = `$exec`;
    if ($verbose) { print $output; }
	
	if ($currentDirection eq "INGRESS") {
		$rxCount = 0;
		foreach (@fia_nw) {
			if ($output =~ /FIA-${fia}.*?From Line Interface\[${_}\]\s+(\d+)/s) {
				$rxCount += $1;
			} else {
				print "Parsing Ingress NP to FIA counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
		$txCount = 0;
		foreach (@fia_fb) {
			if ($output =~ /FIA-${fia}.*?To Unicast Xbar\[${_}\]\s+(\d+)/s) {
				$txCount += $1;
			} else {
				print "Parsing Ingress FIA to XBAR counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
		
		$fia++;
		foreach (@fia_nw) {
			if ($output =~ /FIA-${fia}.*?From Line Interface\[${_}\]\s+(\d+)/s) {
				$rxCount += $1;
			} else {
				print "Parsing Ingress NP to FIA counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
		foreach (@fia_fb) {
			if ($output =~ /FIA-${fia}.*?To Unicast Xbar\[${_}\]\s+(\d+)/s) {
				$txCount += $1;
			} else {
				print "Parsing Ingress FIA to XBAR counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
        $firstFIA = $fia - 1;
		$temp = sprintf("%25s%-10lu%15s%15s\n", "FIA[$firstFIA,$fia] Complex Input: ", $rxCount, "", $drops);
		$retString = $retString . $temp;
		
		$temp = sprintf("%25s%-10lu%15s%15s\n", "FIA[$firstFIA,$fia] Complex Output: ", $txCount, "", 0);
		$retString = $retString . $temp;
	} elsif ($currentDirection eq "EGRESS") {
	
		$txCount = 0;
		foreach (@fia_nw) {
			if ($output =~ /FIA-${fia}.*?To Line Interface\[${_}\]\s+(\d+)/s) {
				$txCount += $1;
			} else {
				print "Parsing Egress FIA to NP counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
		$rxCount = 0;
		foreach (@fia_fb) {
			if ($output =~ /FIA-${fia}.*?From Unicast Xbar\[${_}\]\s+(\d+)/s) {
				$rxCount += $1;
			} else {
				print "Parsing Egress XBAR to FIA counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
		
		$fia++;
		foreach (@fia_nw) {
			if ($output =~ /FIA-${fia}.*?To Line Interface\[${_}\]\s+(\d+)/s) {
				$txCount += $1;
			} else {
				print "Parsing Egress FIA to NP counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
		foreach (@fia_fb) {
			if ($output =~ /FIA-${fia}.*?From Unicast Xbar\[${_}\]\s+(\d+)/s) {
				$rxCount += $1;
			} else {
				print "Parsing Egress XBAR to FIA counters failed for '$exec'\n";
				print "Use the following CLI to gather these stats:\n";
				print "'$cliString'\n";
			}
		}
		
        $firstFIA = $fia - 1;
		$temp = sprintf("%25s%-10lu%15s%15s\n", "FIA[$firstFIA,$fia] Complex Input: ", $rxCount, "", $drops);
		$retString = $retString . $temp;
		
		$temp = sprintf("%25s%-10lu%15s%15s\n", "FIA[$firstFIA,$fia] Complex Output: ", $txCount, "", 0);
		$retString = $retString . $temp;
	}
	
	return $retString;
}

#***************************************************************************
# Name: showGoodXBARCounters
#
# Desc: 
#
# Arguments: [0] The XBAR instance to query
#			 [1] The XBAR port array (from FIA)
#			 [2] The location string
#			 [3] The current walk direction
#***************************************************************************
sub showGoodXBARCounters {
    my ($xbar, $xbar_port_ref) = @_;
    my $slot = locationStringToInteger($_[2]);
    my $currentDirection = $_[3];
    
    my @xbar_ports = @{$xbar_port_ref};
    
    $cliString = "show controllers fabric crossbar statistics instance 0 $_[2]";
    $exec = "xbar_show packet-stats -i 0x0 -s $slot";
    $output = `$exec`;
    if ($verbose) { print $output; }
	
    my $count = 0;

    my @ingressCounters = ();
    my @egressCounters = ();
    my @outputLines = split(/\n/, $output);
    my $currentPort = -1;

    foreach my $line (@outputLines) {
        if ($line =~ /xbar:${xbar}\s+port:(\d+)/) {
            $currentPort = $1;
        } elsif ($line =~ /Ingress Packet Count.*?:\s+(\d+)/) {
            $ingressCounters[$currentPort] = $1;
        } elsif ($line =~ /Egress Packet Count.*?:\s+(\d+)/) {
            $egressCounters[$currentPort] = $1;
        }
    }
    
    foreach my $port (@xbar_ports) {
        if ($currentDirection eq "INGRESS") {
            $count += $ingressCounters[$port];
        }
        if ($currentDirection eq "EGRESS") {
            $count += $egressCounters[$port];
        }
    }
    
    $retString = sprintf("%25s%-10lu%15s%15s\n", "XBAR (superframes): ", $count, "", 0);
    
    return $retString;
}

#**********************************#
#         Clear Subroutines        #
#**********************************#

sub clearInterfaceMACCounters {
	my $interface = convertInterface($_[0]);
	my $prettyInterface = $_[1];
	
	$prettyInterface =~ /(\w+)(\d+\/\d+\/\d+\/\d+)/;
	my $cli = "clear controller $1 $2 stats";
	my $exec = "ethernet_clear_controller -s -I $interface";
	my $output = `$exec`;
	
	if ($output) {
		printf("%s\n\nFAILED to clear MAC counters, to clear manually use '%s'\n", $output, $cli);
	} else {
		printf("Cleared %s counters via '%s'\n", $prettyInterface, $cli);
	}
	
}

sub clearNPCounters {
	my $np = $_[0];
	my $slot = locationStringToInteger($_[1]);
	my $tm = 0;
	
	my $cli = "clear controller np counters np$np location $_[1]";
	my $exec = "prm_clear_controller_np counters -n 0x$np -s $slot";
	my $output = `$exec`;
	
	if ($output =~ /successfully/) {
		printf("Cleared NP%s uCode counters via '%s'\n", $np, $cli);
	} else {
		printf("Clearing NP%s uCode counters FAILED:\n%s", $np, $output);
	}
	# Clear TM counters NOT CURRENTLY USING
	$cli = "clear controller np tm counters np$np tm tm$tm location $_[1]";
	$exec = `prm_clear_controller_tm tm_clear_counters -n 0x$np -t 0x$tm -s $slot`;
	
	printf("Cleared NP%s TM counters via '%s'\n", $np, $cli);
	
	$cli = "clear controller np fabric-counters all np$np location $_[1]";
	$exec = "prm_clear_controller_np xaui -n 0x$np -s $slot -a";
	$output = `$exec`;

	if ($output =~ /successfully/) {
		printf("Cleared NP%s fabric counters via '%s'\n", $np, $cli);
	} else {
		printf("Clearing NP%s fabric counters FAILED:\n%s", $np, $output);
	}
}

sub clearFIACounters {
	my $slot = locationStringToInteger($_[0]);

	my $cli = "clear controller fabric fia location $_[0]";
	my $exec = "fsi_clear -n $slot";
	my $output = `$exec`;
	
	if ($output) {
		printf("%s\n\nFAILED to clear FIA counters, to clear manually use '%s'\n", $output, $cli);
	} else {
		printf("Cleared FIA counters via '%s'\n", $cli);
	}
}

sub clearXBARCounters {
	my $slot = locationStringToInteger($_[0]);

	my $cli = "clear controller fabric crossbar-counters location $_[0]";
	my $exec = "xbar_clear counters -n $slot";
	my $output = `$exec`;
	
	if ($output) {
		printf("%s\n\nFAILED to clear XBAR counters, to clear manually use '%s'\n", $output, $cli);
	} else {
		printf("Cleared XBAR counters via '%s'\n", $cli);
	}
}

#**********************************#
#    Get Link Status Subroutines   #
#**********************************#
sub getMACLinkStatus {
	my $interface = $_[0];
        
	$interface =~ /(\w*Gig\w*)(\d+)_(\d+)_(\d+)_(\d+)/;
        my $prettyInterface = "$1$2/$3/$4/$5";
	
	$prettyInterface =~ /(\w+)(\d+\/\d+\/\d+\/\d+)/;
	my $ifNum = $2;
	my $cli = "show interface $1 $ifNum brief";
	my $exec = "show_interface -i $interface -o 0x2";
	my $output = `$exec`;
	
	if ($output =~ /\w+${ifNum}\s+(\S+)\s+(\S+)/) {
		my $intStatus = $1;
		my $linePStatus = $2;
		
		if ($intStatus =~ /down/ && $linePStatus =~ /down/) {
			return "INTERFACE IS DOWN\nLINE PROTOCOL IS DOWN";
		} elsif ($intStatus =~ /down/) {
			return "INTERFACE IS DOWN";
		} elsif ($linePStatus =~ /down/) {
			return "LINE PROTOCOL IS DOWN";
		}
	} else {
		print "Parsing interface link status failed for '$exec'\n";
		print "Use the following CLI to gather this information:\n";
		print "'$cli'\n";
	}
	
	return "UP";
}

sub getBridgeLinkStatus {
	my ($bridge, $slot) = @_;
	
	my @retArray = ();	
	
	my $cli = "show controllers fabric fia bridge sync-status <location>";
	my $exec = "fsi_show -n $slot -f -b -L";
	my $output = `$exec`;
	
	my @outputLines = split(/\n/, $output);
	my $onRightBridge = 0;
	foreach my $line (@outputLines) {
		if ($line =~ /Category: bridge_sync-(\d+)/) {
			if ($1 == $bridge) {
				$onRightBridge = 1;
			} else {
				$onRightBridge = 0;
			}
		}
		
		if ($onRightBridge) {
			if ($line =~ /\s\s+(\w+.*?)\s\s+(\S+)/) {
				$desc = uc $1;
				$status = uc $2;
				if ($status ne "UP") {
					push(@retArray, "$desc IS $status");
				}
			}
		}
	}
	
	return @retArray;
}

sub getFIALinkStatus {
	my ($fia, $fia_nw_ref, $slot) = @_;
	
	my @retArray = ();
	
	@fia_nw = @{$fia_nw_ref};
	my %checkPorts = map { $_ => 1 } @fia_nw;	
	
	my $cli = "show controllers fabric fia link-status <location>";
	my $exec = "fsi_show -n $slot -f -l";
	my $output = `$exec`;
	
	my @outputLines = split(/\n/, $output);
	my $currentFIA = "";
	foreach my $line (@outputLines) {
		if ($line =~ /FIA-(\d+)/) {
			$currentFIA = $1;
		} else {
			if ($currentFIA eq $fia) {
				if ($line =~ /\s+(\w+\s+link-(\d))\s+(\S+)/) {
					$desc = uc $1;
					$thisPort = $2;
					$status = uc $3;
					if ($status ne "UP" && exists($checkPorts{$thisPort})) {
						push(@retArray, "$desc IS $status");
					}
				}
				
			}
		}
	}
	
	return @retArray;
}

sub getXBARLinkStatus {
	my ($xbar, $xbar_port_ref, $slot) = @_;
	
	my @retArray = ();
	
	@xbar_ports = @{$xbar_port_ref};
	my %checkPorts = map { $_ => 1 } @xbar_ports;
	
	my $cli = "show controllers fabric crossbar link-status instance $xbar location <location>";
	my $exec = "xbar_show link-status -i 0x$xbar -s $slot";
	my $output = `$exec`;
	
	my @outputLines = split(/\n/, $output);
	
	foreach my $line (@outputLines) {
		if ($line =~ /(\d+)\s+\d+\s+\d+\s+\d+\s+(\w+)/) {
			my $thisPort = $1;
			# Strip off leading zero if exists
			if (substr($thisPort, 0, 1) eq "0") {
				$thisPort = substr($thisPort, 1, 1);
			}
			my $status = uc $2;
			if($status ne "UP" && exists($checkPorts{$thisPort})) {
				push(@retArray, "PORT $thisPort IS $status");
			}
		}
	}
	
	return @retArray;
}
#**********************************#
#         Helper Subroutines       #
#**********************************#

sub getHardwareMap {
	my $location = $_[0];
	$location =~ /(\d+)\/(\d+)\/CPU0/;
	my $ep_location = "$1/$2/";
	my @ep0_interfaces = ();
	my @ep1_interfaces = ();
	my @ep0_port_nums = ();
	my @ep1_port_nums = ();
	my $ep0InterfaceType = "";
	my $ep1InterfaceType = "";
	my $linecard = "";
	my $showPlatResults = `show_platform_vkg -e`;
	
	if ($showPlatResults =~ /${location}\s+(\S+)/) {
		$linecard = $1;

		# Check for EPs
		if ($showPlatResults =~ /${location}.*?\n${ep_location}(\d)\s+(\S+)/) {
			if ($1 == 0) {
				if (exists $EP_string_map{$2}) {
					$ep0_interfaces_ref = $EP_string_map{$2};
					@ep0_interfaces = @{$ep0_interfaces_ref};
					if ($showPlatResults =~ /${location}.*?\n${ep_location}\d\s+\S+.*?\n${ep_location}(\d)\s+(\S+)/) {
						if ($1 == 1) {
							if (exists $EP_string_map{$2}) {
								$ep1_interfaces_ref = $EP_string_map{$2};
								@ep1_interfaces = @{$ep1_interfaces_ref};
							} else {
								print "EP type $2 in slot 1 not supported!\n";
								exit;
							}
						}
					}
				} else {
					print "EP type $2 in slot 0 not supported!\n";
					exit;
				}
			} elsif ($1 == 1) {
				if (exists $EP_string_map{$2}) {
                                    $ep1_interfaces_ref = $EP_string_map{$2};
                                    @ep1_interfaces = @{$ep1_interfaces_ref};
				} else {
                                    print "EP type $2 in slot 1 not supported!\n";
                                    exit;
				}
			}
		}
	} else {
		print "Location $location not found!\n";
		exit;
	}
	
	if (exists $LC_string_map{$linecard}) {
		#Check for a modular LC
                #                 Weapon-X                  Ironman
		if (($linecard =~ /A9K-MOD(\d+)-SE|TE/) || ($linecard =~ /ASR9001-LC/)) {
			$numOfNPs = 2;

			if ($1 == 160) {
                            $numOfNPs = 4;
			}
                        
			#Get partial hardware map, and then add interface mappings for each EP
			$hw_map_ref =  $LC_string_map{$linecard};
			$num_of_ep0_interfaces = @ep0_interfaces;
			$num_of_ep1_interfaces = @ep1_interfaces;

                        if ($num_of_ep0_interfaces != 0) {
                            $ep0_interfaces[0] =~ /(\w+)\d/;
                            $ep0Key = $num_of_ep0_interfaces . "_" . $1 . "_EP";
                            if (exists $hw_map_ref->{$ep0Key}->{IF_TYPE}) {
				@ep0_port_nums = @{$hw_map_ref->{$ep0Key}{PORT_NUMS}};
				$ep0InterfaceType = $hw_map_ref->{$ep0Key}{IF_TYPE};
                            } else {
				# EP0 type not supported for this linecard
				print "EP Type not supported in bay 0\n";
				@ep0_interfaces = ();
                            }
                        }

                        if ($num_of_ep1_interfaces != 0) {
                            $ep1_interfaces[0] =~ /(\w+)\d/;
                            $ep1Key = $num_of_ep1_interfaces . "_" . $1 . "_EP";
                            if (exists $hw_map_ref->{$ep1Key}{IF_TYPE}) {
				@ep1_port_nums = @{$hw_map_ref->{$ep1Key}{PORT_NUMS}};
				$ep1InterfaceType = $hw_map_ref->{$ep1Key}{IF_TYPE};
                            } else {
				# EP1 type not supported for this linecard
				print "EP Type not supported in bay 1\n";
				@ep1_interfaces = ();
                            }
                        }
			
			if (@ep0_interfaces) {
                            if ($numOfNPs == 2) {
                                for ($i = 0; $i < $num_of_ep0_interfaces; $i++) {
                                    $ep0_interfaces[$i] =~ /(\w*?GigE)(\d+)/;
                                    $key = $1 . "0_" . $2;
                                    $hw_map_ref->{$key} = {"IF_TYPE" => "$ep0InterfaceType", "CONNECTS_TO" => "NP0", "PORT_IN" => [ "$ep0_port_nums[$i]" ]};
                                }
                            } else { # numOfNPs == 4
                                # Split the interfaces across both NPs
                                for ($i = 0; $i < $num_of_ep0_interfaces/2; $i++) {
                                    $ep0_interfaces[$i] =~ /(\w*?GigE)(\d+)/;
                                    $key = $1 . "0_" . $2;
                                    $hw_map_ref->{$key} = {"IF_TYPE" => "$ep0InterfaceType", "CONNECTS_TO" => "NP0", "PORT_IN" => [ "$ep0_port_nums[$i]" ]};
                                }
                                $j = 0;
                                for ($i = ($num_of_ep0_interfaces/2); $i < ($num_of_ep0_interfaces); $i++) {
                                    $ep0_interfaces[$i] =~ /(\w*?GigE)(\d+)/;
                                    $key = $1 . "0_" . $2;						
                                    $hw_map_ref->{$key} = {"IF_TYPE" => "$ep0InterfaceType", "CONNECTS_TO" => "NP1", "PORT_IN" => [ "$ep0_port_nums[$j]" ]};
                                    $j++;
                                }
                            }
			}
			if (@ep1_interfaces) {
                            if ($numOfNPs == 2) {
                                for ($i = 0; $i < $num_of_ep1_interfaces; $i++) {
                                    $ep1_interfaces[$i] =~ /(\w*?GigE)(\d+)/;
                                    $key = $1 . "1_" . $2;
                                    $hw_map_ref->{$key} = {"IF_TYPE" => "$ep1InterfaceType", "CONNECTS_TO" => "NP1", "PORT_IN" => [ "$ep1_port_nums[$i]" ]};
                                }
                            } else { # numOfNPs == 4
                                for ($i = 0; $i < $num_of_ep1_interfaces/2; $i++) {
                                    $ep1_interfaces[$i] =~ /(\w*?GigE)(\d+)/;
                                    $key = $1 . "1_" . $2;
                                    $hw_map_ref->{$key} = {"IF_TYPE" => "$ep1InterfaceType", "CONNECTS_TO" => "NP2", "PORT_IN" => [ "$ep1_port_nums[$i]" ]};
                                }
                                $j = 0;
                                for ($i = $num_of_ep1_interfaces/2; $i < $num_of_ep1_interfaces; $i++) {
                                    $ep1_interfaces[$i] =~ /(\w*?GigE)(\d+)/;
                                    $key = $1 . "1_" . $2;
                                    $hw_map_ref->{$key} = {"IF_TYPE" => "$ep1InterfaceType", "CONNECTS_TO" => "NP3", "PORT_IN" => [ "$ep1_port_nums[$j]" ]};
                                    $j++;
                                }
                            }
			}
			
			return $hw_map_ref;
			
		} else {
			return $LC_string_map{$linecard};
		}
	} else {
		print "Linecard Type '$linecard' Not Supported!\n";
		exit;
	}
}

sub locationStringToInteger {

    # Megatron: 22 slots, first two are RSPs
    my %chassis9922Mapping = (
                              "0/0/CPU0" => "2081",
                              "0/1/CPU0" => "2097",
                              "0/2/CPU0" => "2113",
                              "0/3/CPU0" => "2129",
                              "0/4/CPU0" => "2145",
                              "0/5/CPU0" => "2161",
                              "0/6/CPU0" => "2177",
                              "0/7/CPU0" => "2193",
                              "0/8/CPU0" => "2209",
                              "0/9/CPU0" => "2225",
                              "0/10/CPU0" => "2241",
                              "0/11/CPU0" => "2257",
                              "0/12/CPU0" => "2273",
                              "0/13/CPU0" => "2289",
                              "0/14/CPU0" => "2305",
                              "0/15/CPU0" => "2321",
                              "0/16/CPU0" => "2337",
                              "0/17/CPU0" => "2353",
                              "0/18/CPU0" => "2369",
                              "0/19/CPU0" => "2385"
                              );
    
    #Starscream: 12 slots, first two are RSPs
    my %chassis9912Mapping = (
                              "0/0/CPU0" => "2081",
                              "0/1/CPU0" => "2097",
                              "0/2/CPU0" => "2113",
                              "0/3/CPU0" => "2129",
                              "0/4/CPU0" => "2145",
                              "0/5/CPU0" => "2161",
                              "0/6/CPU0" => "2177",
                              "0/7/CPU0" => "2193",
                              "0/8/CPU0" => "2209",
                              "0/9/CPU0" => "2225"
                              );
    
    # Red October: 10 slots, middle two are RSPs
    my %chassis9010Mapping = (
                              "0/0/CPU0" => "2049",
                              "0/1/CPU0" => "2065",
                              "0/2/CPU0" => "2081",
                              "0/3/CPU0" => "2097",
                              "0/4/CPU0" => "2145",
                              "0/5/CPU0" => "2161",
                              "0/6/CPU0" => "2177",
                              "0/7/CPU0" => "2193"
                              );

    # Samurai: 6 slots, top two are RSPs
    my %chassis9006Mapping = (
                              "0/0/CPU0" => "2081",
                              "0/1/CPU0" => "2097",
                              "0/2/CPU0" => "2113",
                              "0/3/CPU0" => "2129"
                              );

    # Ironman: 1 slot, contains RSP and LC
    my %chassis9001Mapping = (
                              "0/0/CPU0" => "2081"
                              );

    if ($globalChassisType eq "ASR-9001") {
        return $chassis9001Mapping{$_[0]};
    } elsif ($globalChassisType eq "ASR-9006") {
        return $chassis9006Mapping{$_[0]};
    } elsif ($globalChassisType eq "ASR-9010") {
        return $chassis9010Mapping{$_[0]};
    } elsif ($globalChassisType eq "ASR 9912") {
        return $chassis9912Mapping{$_[0]};
    } elsif ($globalChassisType eq "ASR 9922") {
        return $chassis9922Mapping{$_[0]};
    } else {
        print "Invalid chassis type: $globalChassisType\n";
        exit;
    }
}

sub convertInterface {
	if ($_[0] =~ /GigabitEthernet(\d+)_(\d+)_(\d+)_(\d+)/) {
		return "GigE$1_$2_$3_$4";
	} else {
		if ($_[0] =~ /^GigE(\d+)_(\d+)_(\d+)_(\d+)/) {
			return "GigabitEthernet$1_$2_$3_$4";
		} else {
			return $_[0];
		}
	}
}

sub getChassisType {
    my $versionInfo = `show_version -b`;
    
    if ($versionInfo =~ /(ASR.9\d\d\d)/) {
        return $1;
    } else {
        print "ERROR: Could not determine chassis type from 'show version brief'\n";
        exit;
    }
}

sub warn_error {
    
    my ($msg ) = @_;
    if ($msg =~ /Integer overflow/){
        print "One or more counters have exceeded 32-bits and maxed out. \nConsider 'clear controller dataplane'\n";
    }else{
        die "Error: $msg\n";
    }
}

sub getLineSeparator {
	my $position = $_[0];
	my $numDashes = 31;
	if ($position eq "top") {
		return sprintf("%s^\n", "-" x ($numDashes - 1)); 
	} else {
		return sprintf("%s\n", "-" x $numDashes); 
	}
}

sub printDropHeader {
	my $numDashes = 60;
	printf("%s\n", "=" x $numDashes);
}

sub printCountersHeader {
	my $numDashes = 80;
	printf("%s\n", "=" x $numDashes);
}

sub confirmClear {
    print "This command will clear all counters associated with this datapath.\n";
    print "Is this okay? [y/n] ";
    $answer = <STDIN>;
    chomp($answer);
    if ($answer =~ /^y/i) {
        return;
    } else {
        exit;
    }
}
