::cisco::eem::event_register_syslog pattern "ROUTING-ISIS-5-ADJCHANGE ?: ?Adjacency to.*${intf_bundle_pattern}.*" maxrun 60
# ---------------------------------------------------------------------
# isis_adj_group_routing_xr.tcl
#
# Chandra Vijayaprasad <chanvija@cisco.com>
#
# Copyright (c) 2011 by cisco Systems, Inc.
# All rights reserved.
#--------------------------------------------------------------------
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
#
# This policy watches for a specific syslog message indicating that an
# interface is down.  It then checks to see if that interface is part of
# a logical ECMP bundle.  If so, it will adjust the ISIS metric of each
# interface in the bundle to a new value to make the bundle less desirable.
# Once all of the interfaces in the bundle are back up, the ISIS metric will
# be adjusted back to the original value after a configurable amount of delay
# time.
#
# This script requires quite a few EEM environment variables.  If the script
# is running on IOS-XR, the following environment variable needs to be set:
#
# Please refer to EDCS-964813 for details on usage.

namespace import ::cisco::eem::*
namespace import ::cisco::lib::*

#proc to print syslogs on if debug flag is enabled.
#isis_adj_debug eem variable needs to be set to 1
proc _action_syslog {msg} {
    if {[info exists ::isis_adj_debug]} {
        if {$::isis_adj_debug} {
            action_syslog priority debug msg "$msg"
        }
    }
}

#proc to convert interfae names in short frmat to long one
#Eg., BE101 to Bundle-Ether 101
proc get_intf_long_name {intf} {
    if {![regexp -nocase {([a-zA-Z\-]+)(.*)} $intf . name rsm .]} {
        return $intf
    }
    switch -regexp -- $name {
        {BP} { return "Bundle-POS$rsm" }
        {BE} { return "Bundle-Ether$rsm" }
        {PO} { return "POS$rsm" }
        {Gi} { return "GigabitEthernet$rsm" }
        {Te} { return "Tengig$rsm" }
    }
    return "$name $rsm"
}

#an abstraction for cli_exec to debug any issues with cli_exec
proc my_cli_exec { fd cmd } {
    return [cli_exec $fd $cmd]
}

# Extract the user policy directory so we can register new Tcl policies.
proc get_pol_dir { fd } {
    set res {}

    set output [cli_exec $fd "show event manager directory user policy"]
    set output [string trim $output]
    regsub -all "\r\n" $output "\n" result

    set lines [split $result "\n"]
    foreach line $lines {
        if { $line == "" } {
            #skip null lines
            continue
        }
        if { ! [regexp {\s} $line] && ! [regexp {#$} $line] } {
            set res $line
            break
        }
    }

    if { $res == {} } {
        return -code error "The user policy directory has not been configured"
    } else {
        _action_syslog "user policy directory $res"
    }

    return $res
}

#clean up procedure to do any last minute things.
proc cleanup { fd tty } {
    catch {cli_close $fd $tty} ignore
}

array set arr_einfo [event_reqinfo]
set msg $arr_einfo(msg)
set min_links_up -1

global state cliarr bundle_name intflst num_neighbors_up down_intf

# Determine what interface is generating the message.  We need to use ISIS
# for the moment because PKT_INFRA-LINK-3-UPDOWN messages cannot be intercepted
# by EEM on IOS-XR.
if { ! [regexp "ROUTING-ISIS-5-ADJCHANGE ?: Adjacency to \\S+ ?\\(\(${intf_bundle_pattern}\\S+\)\\) ?\\(\(\\S+\)\\) ?\(\\S+\)," $msg . down_intf isis_type state] } {
# We cannot resend the original message, or we will get stuck in a loop
# due to CSCte05382.
    action_syslog priority err msg "Failed to parse syslog message"
    return
}

_action_syslog "State $state type $isis_type intf $down_intf"

if {($state != "Up") && ($state != "Down")} {
    _action_syslog "State $state...how to handle this?"
    return
}

# IOS-XR requires us to register our policies with a username.
if { ! [info exists intf_bundle_eem_user] } {
    set result "ERROR: Policy cannot be run: variable intf_bundle_eem_user has not been set"
    action_syslog priority err msg $result
    return
}

# Open the CLI session.
if { [catch {cli_open} result] } {
    action_syslog priority err msg "Failed to open CLI session" $result
    return
}

array set cliarr $result
my_cli_exec $cliarr(fd) "term length 0"

# Find all bundles.  Each bundle will have an "intf_bundle_name_*" environment
# variable set.
set output [my_cli_exec $cliarr(fd) "show event manager environment | include intf_bundle_name_"]

set found 0
set matched_bundle {}
set lines [split $output "\n"]
set intflst [list]
set bundle_name {}

foreach line $lines {
# We need to be able to get the full value of each bundle environment variable
# in both IOS and IOS-XR.  This means we have to run the "show event manager
# environment VAR" command until we find the bundle which contains the
# interface which generated the message.
    if { [regexp {(intf_bundle_name_\S+) } $line -> matched_bundle] } {
        set output [my_cli_exec $cliarr(fd) "show event manager environment $matched_bundle"]
        regsub -all "\r\n" $output "\n" output
        set olines [split $output "\n"]
        foreach oline $olines {
            set oline [string trim $oline]
            if { $oline == "" } {
                continue
            }
            if { ! [regexp {\s} $oline] && ! [regexp {#$} $oline] } {
                set intflst [split $oline ","]
                if { [lsearch -exact $intflst $down_intf] != -1 } {
                    set bundle_name [string range $matched_bundle [string length "intf_bundle_name_"] end]
                    set found 1
                    break
                }
            }
        }
        if { $found } {
            break
        }
    }
}

# If the interface which generated the message is not in a configured bundle,
# just ignore the message, and exit gracefully.
if { ! $found } {
    _action_syslog "No matching interfaces found"
    cleanup $cliarr(fd) $cliarr(tty_id)
    return
}

# Check if algorithm is provided
if { ! [info exists "::intf_bundle_algorithm_${bundle_name}"] } {
    action_syslog priority err msg "Environment variable \
    intf_bundle_algorithm_${bundle_name} is not set"
    cleanup $cliarr(fd) $cliarr(tty_id)
    return
}
set algo [expr \$::intf_bundle_algorithm_$bundle_name]

if { ! [info exists "::intf_bundle_isis_proc_${bundle_name}"] } {
    action_syslog priority err msg "Environment variable \
    intf_bundle_isis_proc_${bundle_name} is not set"
    cleanup $cliarr(fd) $cliarr(tty_id)
    return
}

# We have to be careful.  If we see a syslog message which says that a given
# interface is now "Up," we can't just assume ALL of the interfaces in its
# bundle are up.  So, we check the ISIS neighbor list to make sure each
# interface in the bundle is still up.  If so, we will proceed to change
# the metric on the
# interfaces back to their original value.  If not, we assume that the
# "Down" code path was previously executed, and we just exit gracefully.

_action_syslog "Checking isis neighbor count"
set output [my_cli_exec $cliarr(fd) "show isis neighbors"]
array set nbrs [list]
foreach line [split $output "\n"] {
    set line [string trim $line]
    regsub -all {\s+} $line { } line
    set parts [split $line]
    set intf [get_intf_long_name [lindex $parts 1]]
    if { [lsearch -exact $intflst $intf] != -1 } {
        set nbrs($intf) 1
    }
}

#check if all neigbors on interfaces in the bundle have come up.
set num_neighbors_up 0
foreach i $intflst {
    if {[info exists nbrs($i)] } {
        incr num_neighbors_up
    }
}

proc algorithm_1 {} {
    global cliarr bundle_name intflst state num_neighbors_up

    # Now we need to do per-bundle environment variable checks.  Each of these
    # environment variables will need to be set for each bundle name.
    if { ! [info exists "::intf_bundle_orig_metric_$bundle_name"] } {
        action_syslog priority err msg "Environment variable \
        intf_bundle_orig_metric_${bundle_name} is not set"
        return
    }

    if { ! [info exists "::intf_bundle_new_metric_${bundle_name}"] } {
        action_syslog priority err msg "Environment variable \
        intf_bundle_new_metric_${bundle_name} is not set"
        return
    }

    if { ! [info exists "::intf_bundle_delay_${bundle_name}"] } {
        action_syslog priority err msg "Environment variable \
        intf_bundle_delay_${bundle_name} is not set"
        return
    }

    if {[info exists "::intf_bundle_min_adj_${bundle_name}"]} {
        set min_links_up [expr \$::intf_bundle_min_adj_${bundle_name}]
        if {($min_links_up <= 0) || \
            ($min_links_up > [llength $intflst])} {
            action_syslog priority err msg \
            "intf_bundle_min_adj_${bundle_name} has incorrect value :
            $min_links_up. Range is 1 thru [llength $intflst]"
            return
        }
    } else {
        set min_links_up [llength $intflst]
    }

    set isis_proc [expr \$::intf_bundle_isis_proc_${bundle_name}]

    #Check if action has been taken for bundle_up already. Optimization

    if {[catch {array set up_down [context_retrieve \
            ISIS_ADJ_GRP_ALG1_UP]} err]} {
        #probably this is the first time
        _action_syslog "Could not retrieve context for \
        ISIS_ADJ_GRP_ALG1_UP. Error : $err"
    }
    foreach var [array names up_down]  {
        #Workaround for CSCtf18759. Save back all variables, so that the
        #final context_save will push them back
        _action_syslog "$var $up_down($var)"
        eval set $var $up_down($var)
    }

    #bundle_up_<bundle_name> = 1 means the bundle costs have been
    #reduced due to interfaces coming up
    #bundle_down_<bundle_name> = 1 means the bundle costs have been
    #increased due to interfaces going down

    if {[regexp -nocase {up} $state ]} {
        if {[catch { eval set value \
                $up_down(bundle_up_${bundle_name})}]} {
            set value 0
        }
        if {$value} {
            _action_syslog "Metric on $bundle_name has already been \
            lowered. Exiting rc=0..."
            if {[catch {eval [context_save ISIS_ADJ_GRP_ALG1_UP \
                    bundle_*]} err]} {
                eval [_action_syslog "Context save for \
                bundle_* failed. Error $err"]
            } else {
                eval [_action_syslog "Context save for \
                bundle_* successful"]
            }

            return
        }
    } else {
        if {[catch { eval set value \
                $up_down(bundle_down_${bundle_name})}]} {
            set value 0
        }
        if {$value} {
            action_syslog priority info msg "Metric on $bundle_name \
            has already been lowered."

            if {[catch {eval [context_save ISIS_ADJ_GRP_ALG1_UP \
                    bundle_*]} err]} {
                eval [_action_syslog "Context save for \
                bundle_* failed. Error $err"]
            } else {
                eval [_action_syslog "Context save for \
                bundle_* successful"]
            }
            return
        }
    }

    if { $state == "Up" } {
        if {$num_neighbors_up >= $min_links_up} {
            action_syslog priority info msg "Number of active \
            links($num_neighbors_up) is\
            better or equal to configured threshold($min_links_up). \
            Restoring the original metric on bundle($bundle_name) links"
        } else {
            action_syslog priority info msg "Number of active \
            links($num_neighbors_up) is\
            less than configured threshold($min_links_up). Not restoring\
            the original metric on bundle($bundle_name) links"
            if {[catch {eval [context_save ISIS_ADJ_GRP_ALG1_UP \
                    bundle_*]} err]} {
                eval [_action_syslog "Context save for \
                bundle_* failed. Error $err"]
            } else {
                eval [_action_syslog "Context save for \
                bundle_* successful"]
            }
            return
        }
    } else {
        if {$num_neighbors_up >= $min_links_up} {
            action_syslog priority info msg "Number of active \
            links($num_neighbors_up) is\
            higher than or equal to configured \
            threshold($min_links_up). Not lowering\
            the metric on bundle($bundle_name) links"
            if {[catch {eval [context_save ISIS_ADJ_GRP_ALG1_UP \
                    bundle_*]} err]} {
                eval [_action_syslog "Context save for \
                bundle_* failed. Error $err"]
            } else {
                eval [_action_syslog "Context save for \
                bundle_* successful"]
            }
            return
        } else {
            action_syslog priority info msg "Number of active \
            links($num_neighbors_up) is\
            lower than configured threshold($min_links_up). Lowering\
            the metric on bundle($bundle_name) links"
        }
    }

    set old_metric [expr \$::intf_bundle_orig_metric_${bundle_name}]
    set new_metric [expr \$::intf_bundle_new_metric_${bundle_name}]

    set poldir [get_pol_dir $cliarr(fd)]
    set pname "tm_start_${bundle_name}.tcl"
    set polname "${poldir}/${pname}"

    if { $state == "Up" } {
    # If the policy file already exists, we don't have to schedule
    # a new instance of the policy.  Just exit gracefully.
        if { [file exists $polname] } {
            action_syslog priority info msg "$polname exists. Not continuing.."
            if {[catch {eval [context_save ISIS_ADJ_GRP_ALG1_UP \
                    bundle_*]} err]} {
                eval [_action_syslog "Context save for \
                bundle_* failed. Error $err"]
            } else {
                eval [_action_syslog "Context save for \
                bundle_* successful"]
            }
            return
        }

        set timer_time [expr \$::intf_bundle_delay_${bundle_name}]

    # Build a policy which will run after a specified number of seconds which will
    # reset the metric of all interfaces in a given bundle to the original value.
    # This policy will only run once.
        set fd [open $polname "w"]
        puts $fd "::cisco::eem::event_register_timer countdown time $timer_time name start_${bundle_name}_timer"
        puts $fd {namespace import ::cisco::eem::*}
        puts $fd {namespace import ::cisco::lib::*}
        puts $fd {
            proc my_cli_exec { fd cmd } {

                return [cli_exec $fd $cmd]
            }
        }
        puts $fd "if { \[catch {cli_open} result] } {"
        puts $fd {    action_syslog priority info msg $result}
        puts $fd "}"
        puts $fd {array set cliarr $result}
        puts $fd "action_syslog priority notice msg \"Number of ISIS \
        adjacencies($num_neighbors_up) \
        in bundle $bundle_name is higher than threshold($min_links_up); \
        setting metric back to $old_metric\""
        puts $fd "my_cli_exec \$cliarr(fd) \"delete /noprompt $polname\""
        puts $fd {my_cli_exec $cliarr(fd) "config t"}
        puts $fd "my_cli_exec \$cliarr(fd) \"router isis $isis_proc\""
        foreach i $intflst {
            puts $fd "my_cli_exec \$cliarr(fd) \"interface $i\""
            puts $fd "my_cli_exec \$cliarr(fd) \"address-family ipv4 unicast\""
            puts $fd "my_cli_exec \$cliarr(fd) \"metric $old_metric\""
        }
        puts $fd {my_cli_exec $cliarr(fd) "commit"}
    # We remove the policy here to avoid having it re-run if the router is
    # rebooted.
        puts $fd "my_cli_exec \$cliarr(fd) \"no event manager policy $pname\""
        puts $fd {my_cli_exec $cliarr(fd) "commit"}
        puts $fd {my_cli_exec $cliarr(fd) "end"}
        close $fd

        my_cli_exec $cliarr(fd) "config t"
        my_cli_exec $cliarr(fd) "event manager policy $pname username $::intf_bundle_eem_user"
        my_cli_exec $cliarr(fd) "commit"
        my_cli_exec $cliarr(fd) "end"

        #metric for the link will be lowered when the script runs. 
        #Set bundle_up context to 1 and bundle_down
        #context to 0
        eval set bundle_up_${bundle_name} 1
        eval set bundle_down_${bundle_name} 0
        if {[catch {eval [context_save ISIS_ADJ_GRP_ALG1_UP \
                bundle*]} err]} {
            eval [_action_syslog "Context save for \
            bundle_* failed. Error $err"]
        } else {
            eval [_action_syslog "Context save for \
            bundle_* successful"]
        }

    } else {
    # The bundle is down.  We will change the metric on  each interface in
    # the bundle to the new, 
    # higher value.  This code can be run multiple times on a given bundle, but
    # doing so should not cause a problem.
        _action_syslog "Removing $pname and lowering metric"
        my_cli_exec $cliarr(fd) "config t"
        my_cli_exec $cliarr(fd) "no event manager policy $pname"
        my_cli_exec $cliarr(fd) "commit"
        my_cli_exec $cliarr(fd) "router isis $isis_proc"
        foreach i $intflst {
            _action_syslog "Lowering for $i"
            my_cli_exec $cliarr(fd) "interface $i"
            my_cli_exec $cliarr(fd) "address-family ipv4 unicast"
            my_cli_exec $cliarr(fd) "metric $new_metric"
        }

        my_cli_exec $cliarr(fd) "commit"

        #metric on the link is lowered. Set bundle_up context to 0 and bundle_down
        #context to 1
        eval set bundle_up_${bundle_name} 0
        eval set bundle_down_${bundle_name} 1
        if {[catch {eval [context_save ISIS_ADJ_GRP_ALG1_UP \
                bundle_*]} err]} {
            eval [_action_syslog "Context save for \
            bundle_* failed. Error $err"]
        } else {
            eval [_action_syslog "Context save for \
            bundle_* successful"]
        }

        my_cli_exec $cliarr(fd) "end"

        my_cli_exec $cliarr(fd) "delete /noprompt $polname"
    }
}

proc algorithm_2 {} {
    global cliarr bundle_name intflst state num_neighbors_up down_intf
    
    #check if the multiplier has been provided
    if { ! [info exists "::intf_bundle_multiplier_$bundle_name"] } {
        action_syslog priority err msg "Environment variable \
        intf_bundle_multiplier_${bundle_name} is not set"
        return
    }

    #check if the default metrics are provided for this bundle
    #this is needed to avoid conditions at bootup for process restart.
    #without this information, we might end up with metric 0 due to
    #repeated Up state transitions.
    if { ! [info exists "::intf_bundle_default_metric_$bundle_name"] } {
        action_syslog priority err msg "Environment variable \
        intf_bundle_multiplier_${bundle_name} is not set"
        return
    }

    #find out if the peering is at L1 or L2 or L1+L2 per interface
    foreach intf $intflst {
        set metric \
        [my_cli_exec $cliarr(fd) "show isis interface $intf | inc \
        \"Metric|Circuit Type\""]

        if {![regexp {Metric\s+\(L1/L2\):\s+(\d+)/(\d+)} $metric . \
            l1_cost l2_cost]} {
            action_syslog priority err msg "Could not derive L1/L2 cost\
            for interface $intf"
            return
        } else {
            _action_syslog "$intf : L1 $l1_cost L2 $l2_cost"
            set intf_data($intf,l1_cost) $l1_cost
            set intf_data($intf,l2_cost) $l2_cost
        }

        if {![regexp {Circuit Type:\s+(\S+)} $metric . type]} {
            action_syslog priority err msg "Could not derive\
            circuit-type for interface $intf"
            return
        } else {
            _action_syslog "$intf : type $type"
            set intf_data($intf,type) $type
        }
    }

    set isis_proc [expr \$::intf_bundle_isis_proc_${bundle_name}]
    set multiplier [expr \$::intf_bundle_multiplier_${bundle_name}]
    set default_metric [expr \$::intf_bundle_default_metric_${bundle_name}]
    set new_metric [expr round(pow($multiplier,\
       [expr [llength $intflst] - $num_neighbors_up])) * $default_metric]
    _action_syslog "new_metric $new_metric multiplier $multiplier \
       num_neighbors_up $num_neighbors_up default_metric $default_metric"

    my_cli_exec $cliarr(fd) "config terminal"
    my_cli_exec $cliarr(fd) "router isis $isis_proc"

    if {[catch {array set up_down [context_retrieve \
            ISIS_ADJ_GRP_ALG2_UP]} err]} {
        #probably this is the first time
        _action_syslog "Could not retrieve context for \
        ISIS_ADJ_GRP_ALG2_UP. Error : $err"
    }
    foreach var [array names up_down]  {
        #Workaround for CSCtf18759. Save back all variables, so that the
        #final context_save will push them back
        _action_syslog "$var $up_down($var)"
        eval set $var $up_down($var)
    }
    regsub -all "/" $down_intf "_" _down_intf
    if {$state == "Up"} {

        set commit 0

        if {[catch { eval set value \
            $up_down(bundle_up_${bundle_name}_${_down_intf})}]} {
            set value 0
        }
        if {$value} {
            #Optimization to avoid multiple commits for same interface
            #with L1 and L2 peering
            _action_syslog "Up event for $down_intf bundle\
            $bundle_name already handled"
        } else {
            foreach intf $intflst {
                regsub -all "/" $intf "_" _intf
                my_cli_exec $cliarr(fd) "interface $intf"
                my_cli_exec $cliarr(fd) "address-family ipv4 unicast"
                switch -exact $intf_data($intf,type) {
                    "level-1" {
                        if {($intf_data($intf,l1_cost) <= \
                            $default_metric)} {

                            action_syslog priority info msg "L1 Cost of\
                            interface\
                            $intf in bundle $bundle_name is at configured\
                            default. Cannot reduce below that value"

                            continue
                        }
                        if {$new_metric != $intf_data($intf,l1_cost)} {
                            _action_syslog  "Updating L1 Cost of\
                                interface $intf in bundle $bundle_name"
                            my_cli_exec $cliarr(fd) \
                                "metric $new_metric level 1"
                            set commit 1
                        } else {
                            _action_syslog  "L1 Cost of\
                                interface $intf in bundle $bundle_name is\
                                at new metric $new_metric"
                        }
                    }
                    "level-1-2" {
                        if {($intf_data($intf,l1_cost) <= \
                            $default_metric)} {

                            action_syslog priority info msg "L1 Cost of\
                            interface\
                            $intf in bundle $bundle_name is at configured\
                            default. Cannot reduce below that value"

                            #check l2 cost also

                        } else {
                            if {$new_metric != $intf_data($intf,l1_cost)} {
                                _action_syslog  "Updating L1 Cost of\
                                    interface $intf in bundle $bundle_name"
                                my_cli_exec $cliarr(fd) \
                                    "metric $new_metric level 1"
                            } else {
                                _action_syslog  "L1 Cost of\
                                    interface $intf in bundle $bundle_name is\
                                    at new metric $new_metric"
                            }
                        }
                        if {($intf_data($intf,l2_cost) <= \
                            $default_metric)} {

                            action_syslog priority info msg "L2 Cost of\
                            interface\
                            $intf in bundle $bundle_name is at configured\
                            default. Cannot reduce below that value"

                            continue
                        } else {
                            if {$new_metric != $intf_data($intf,l2_cost)} {
                                _action_syslog  "Updating L2 Cost of\
                                    interface $intf in bundle $bundle_name"
                                my_cli_exec $cliarr(fd) \
                                    "metric $new_metric level 2"
                                set commit 1
                            } else {
                                _action_syslog  "L2 Cost of\
                                    interface $intf in bundle $bundle_name is\
                                    at new metric $new_metric"
                            }
                        }
                    }
                    "level-2-only" {
                        if {($intf_data($intf,l2_cost) <= \
                            $default_metric)} {

                            action_syslog priority info msg "L2 Cost of\
                            interface\
                            $intf in bundle $bundle_name is at configured\
                            default. Cannot reduce below that value"

                            continue
                        }

                        if {$new_metric != $intf_data($intf,l2_cost)} {
                            _action_syslog  "Updating L2 Cost of\
                                interface $intf in bundle $bundle_name"

                            my_cli_exec $cliarr(fd) \
                                "metric $new_metric level 2"
                            set commit 1
                        } else {
                            _action_syslog  "L2 Cost of\
                                interface $intf in bundle $bundle_name is\
                                at new metric $new_metric"
                        }
                    }
                }
            }
            if {$commit} {
                action_syslog priority info msg "ISIS Adjacency has come up\
                on some links. Reducing the metric on the bundle\
                $bundle_name"
            }
        }
        eval set bundle_up_${bundle_name}_${_down_intf} 1
        eval set bundle_down_${bundle_name}_${_down_intf} 0
    } else {
        set commit 0
        if {[catch {eval set value \
            $up_down(bundle_down_${bundle_name}_${_down_intf})}]} {
            set value 0
        }
        if {$value} {
            _action_syslog "Down event for $down_intf bundle\
            $bundle_name already handled"
        } else {
            foreach intf $intflst {
                regsub -all "/" $intf "_" _intf
                my_cli_exec $cliarr(fd) "interface $intf"
                my_cli_exec $cliarr(fd) "address-family ipv4 unicast"
                switch -exact $intf_data($intf,type) {
                    "level-1" {
                        _action_syslog  "Updating L1 Cost of\
                            interface $intf in bundle $bundle_name"

                        my_cli_exec $cliarr(fd) \
                            "metric $new_metric level 1"
                        set commit 1
                    }
                    "level-1-2" {
                        _action_syslog  "Updating L1+L2 Cost of\
                            interface $intf in bundle $bundle_name"

                        my_cli_exec $cliarr(fd) \
                            "metric $new_metric level 1"
                        my_cli_exec $cliarr(fd) \
                            "metric $new_metric level 2"
                        set commit 1
                    }
                    "level-2-only" {
                        _action_syslog  "Updating L2 Cost of\
                            interface $intf in bundle $bundle_name"

                        my_cli_exec $cliarr(fd) \
                            "metric $new_metric level 2"
                        set commit 1
                    }
                }
            }
            if {$commit} {
                action_syslog priority info msg "ISIS Adjacency has gone\
                down on some links. Increasing the metric on the bundle \
                $bundle_name"
            }
        }

        eval set bundle_down_${bundle_name}_${_down_intf} 1
        eval set bundle_up_${bundle_name}_${_down_intf} 0
    }

    if {[catch {eval [context_save ISIS_ADJ_GRP_ALG2_UP \
            bundle_*]} err]} {
        eval [_action_syslog "Context save for \
        bundle_* failed. Error $err"]
    } else {
        eval [_action_syslog "Context save for \
        bundle_* successful"]
    }

    if {$commit} {
        my_cli_exec $cliarr(fd) "commit"
    }
}

switch $algo {
    "1" {
        algorithm_1
    }
    "2" {
        algorithm_2
    }
    default {
        action_syslog priority err msg "Environment variable \
        intf_bundle_algorithm_${bundle_name} value is out of range.\
        Permitted value is 1 or 2 only."
    }
}
cleanup $cliarr(fd) $cliarr(tty_id)
_action_syslog "Exiting..."
