#!/bin/bash
#-----------------------------------------------------------------------------
#
# Copyright (c) 2012-2015, 2017-2018 by Cisco Systems, Inc.
# All rights reserved.
#
# This script is used to gather the show tech information for ESD and MLAP.
#
# This script doesn't collect any trace now. See details in CSCur77441.
#
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
#
# ExecuteCli
#
# A function to execute a single CLI command
#
#-----------------------------------------------------------------------------
function ExecuteCli {
    echo "$@"
    $EXEC_CONFD_CMD <<EOF
$@ | nomore
EOF
    echo " "
}

#-----------------------------------------------------------------------------
#
# ExecuteHiddenCli
#
# A function to execute a single hidden CLI command. This function first
# unlocks the hidden commands, then runs the command. The unhide is not 
# permanent and is only for the length of this command.
#
#-----------------------------------------------------------------------------
function ExecuteHiddenCli {
    echo "$@"
    $EXEC_CONFD_CMD <<EOF
activate advanced
yes
$@ | nomore
deactivate advanced
EOF
    echo " "
}

#-----------------------------------------------------------------------------
#
# ExecuteGetPortList
#
# A function to parse out the port list from a summary switch or MLAP CLI.
#
#-----------------------------------------------------------------------------
function ExecuteGetPortList {
    $EXEC_CONFD_CMD <<EOF | awk '/^[0-9]+/ && NF >= 5 { print $1 }'
$@ | nomore
EOF
}

#-----------------------------------------------------------------------------
#
# Main Program Starts Here
#
#-----------------------------------------------------------------------------

# Set the name of the CLI proxy tool that will find the active CONFD to query
EXEC_CONFD_CMD="/opt/cisco/calvados/bin/admin-cli-proxy -n"
#EXEC_CONFD_CMD="confd_cli"

# First, let's capture the list of reachable switches so we can
# build a list of locations for further commands.
SWITCH_OUTPUT=`$EXEC_CONFD_CMD <<EOF | awk ' NF >= 3 && !/(---|Rack)/ { print $1"/"$2"/"$3 }'
show controller switch reachable | nomore
EOF`

# Now, let's also capture the list of reachable MLAP instances
# so we can also build a list of locations for further commands.
MLAP_OUTPUT=`$EXEC_CONFD_CMD <<EOF | awk ' NF >= 3 && !/(---|Rack)/ { print $1"/"$2"/"$3 }'
show controller switch mlap reachable | nomore
EOF`

#capture all the sysadmin vm ip's 
VM_OUTPUT=`$EXEC_CONFD_CMD <<EOF | awk ' NF >= 4 && /(sysadmin)/ { print $3 }'
show vm | nomore
EOF`
 
#capture all the sysadmin vm's locations
LOCATION_OUTPUT=`$EXEC_CONFD_CMD <<EOF | awk ' NF >= 1 && /(Location:)/ { print $2 }'
show vm | nomore
EOF`

#get platform
PLATFORM=`$EXEC_CONFD_CMD <<EOF | awk '{ print $2 }'
show platform | include /RP | until /RP | nomore
EOF`

 
# List the ESDMA connections information
echo "PLATFORM : $PLATFORM"
echo " "
echo "---------------------------------------------------------------------"
echo "ESDMA Connections Information"
echo "---------------------------------------------------------------------"
echo " "
ExecuteHiddenCli "show controller switch mgmt-agent connections"

# List the reachable SWITCH instances 
echo " "
echo "---------------------------------------------------------------------"
echo "ESD Reachability Information"
echo "---------------------------------------------------------------------"
echo " "
ExecuteCli "show controller switch reachable"

echo " "
echo "---------------------------------------------------------------------"
echo "ESD SDR Information (notified from SDR-MGR)"
echo "---------------------------------------------------------------------"
echo " "
ExecuteCli "show controller switch vlan information"

if [ "$PLATFORM" == "NC6-RP" ] || [ "$PLATFORM" == "NCS4K-RP" ]
then
    ExecuteCli "show controller switch summary location"
    ExecuteCli "show controller switch statistics location"
    ExecuteCli "show controller switch fdb location"
    ExecuteCli "show controller switch fdb statistics location"
    ExecuteCli "show controller switch vlan information location"
    ExecuteCli "show controller switch vlan membership location"
    ExecuteCli "show controller switch vlan rules location"
    ExecuteCli "show controller switch sdr port-statistics location"
    ExecuteCli "show controller switch bridge statistics location"
    ExecuteHiddenCli "show controller switch debug counters location"
    ExecuteCli "show controller switch statistics detail"
else 
    # Now, for each SWITCH instance, issue commands to each
    for SWITCH_LOCATION in $SWITCH_OUTPUT ; do
        echo " "
        echo "---------------------------------------------------------------------"
        echo "ESD Location $SWITCH_LOCATION "
        echo "---------------------------------------------------------------------"
        echo " "

        ExecuteCli "show controller switch summary location $SWITCH_LOCATION"
        ExecuteCli "show controller switch statistics location $SWITCH_LOCATION"
        ExecuteCli "show controller switch fdb location $SWITCH_LOCATION"
        ExecuteCli "show controller switch fdb statistics location $SWITCH_LOCATION"
        ExecuteCli "show controller switch vlan information location $SWITCH_LOCATION"
        ExecuteCli "show controller switch vlan membership location $SWITCH_LOCATION"
        ExecuteCli "show controller switch vlan rules location $SWITCH_LOCATION"
        ExecuteCli "show controller switch sdr port-statistics location $SWITCH_LOCATION"
        ExecuteCli "show controller switch bridge statistics location $SWITCH_LOCATION"
        ExecuteHiddenCli "show controller switch debug counters location $SWITCH_LOCATION"

        # Get the list of switch ports for this ESD instance so we can do detail outputs
        SWITCH_PORTS=`ExecuteGetPortList "show controller switch summary location $SWITCH_LOCATION"`

        # Now loop around each port on the switch
        for PORT in $SWITCH_PORTS ; do
            ExecuteCli "show controller switch statistics detail location $SWITCH_LOCATION $PORT"
        done
    done
fi

# List the reachable MLAP instances 
echo " "
echo "---------------------------------------------------------------------"
echo "MLAP Reachability Information"
echo "---------------------------------------------------------------------"
echo " "
ExecuteCli "show controller switch mlap reachable"

if [ "$PLATFORM" == "NC6-RP" ] || [ "$PLATFORM" == "NCS4K-RP" ]
then
    ExecuteCli "show controller switch mlap location"
    ExecuteCli "show controller switch mlap statistics location"
    ExecuteHiddenCli "show controller switch mlap debug counters location"
    ExecuteCli "show controller switch mlap detail location"
else
    # Now, for each MLAP instance, issue commands to each
    for MLAP_LOCATION in $MLAP_OUTPUT ; do
        echo " "
        echo "---------------------------------------------------------------------"
        echo "MLAP Location $MLAP_LOCATION"
        echo "---------------------------------------------------------------------"
        echo " "

        ExecuteCli "show controller switch mlap location $MLAP_LOCATION"
        ExecuteCli "show controller switch mlap statistics location $MLAP_LOCATION"
        ExecuteHiddenCli "show controller switch mlap debug counters location $MLAP_LOCATION"
    
        # Get the list of MLAP ports for this MLAP instance for detail commands
        MLAP_PORTS=`ExecuteGetPortList "show controller switch mlap location $MLAP_LOCATION"`

        # Now run the detailed commands per port
        for PORT in $MLAP_PORTS ;   do
            ExecuteCli "show controller switch mlap detail location $MLAP_LOCATION $PORT"
        done
    done
fi

#port registers dump
if [ "$PLATFORM" == "NC6-RP" ] || [ "$PLATFORM" == "NCS4K-RP" ]
then
    ExecuteCli "show controller switch mac location"
fi

#lspci output 
if [ "$PLATFORM" == "NC6-RP" ] || [ "$PLATFORM" == "NCS4K-RP" ]
then
    VM_OUTPUT_ARRAY=($VM_OUTPUT)
    LOCATION_OUTPUT_ARRAY=($LOCATION_OUTPUT)
    # Now, for each location, run lspci commnad 
    for (( i=0; i<${#VM_OUTPUT_ARRAY[@]}; i++ )); do
        echo " "
        echo "---------------------------------------------------------------------"
        echo "Host PCIE tree for Location : ${LOCATION_OUTPUT_ARRAY[i]}"
        echo "---------------------------------------------------------------------"
        echo " "
    
        ssh -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no -o LogLevel=error ${VM_OUTPUT_ARRAY[i]} ssh -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no -o LogLevel=error 10.0.2.2 lspci -tv
    
        echo " "
        echo "---------------------------------------------------------------------"
        echo "Guest PCIE tree for Location : ${LOCATION_OUTPUT_ARRAY[i]}"
        echo "---------------------------------------------------------------------"
        echo " "
    
        ssh -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no -o LogLevel=error ${VM_OUTPUT_ARRAY[i]} lspci -tv
    done
fi
