#!/usr/bin/env python
"""

  dossier.py

  Dossier generation utility

  Copyright (c) 2020 by cisco Systems, Inc.
  All rights reserved.

"""
import json
import subprocess
from collections import OrderedDict
import sys, os
import time
import json
import xml.etree.cElementTree as ET
import getopt
import re
import logging
import json.encoder as jsonencoder
import fcntl

model_namespace = "http://cisco.com/ns/yang/Cisco-IOS-XR-ama"
model_revision = "2019-08-05"
#JSON parser requires memory ~ dozen times of the buffer passed to it
JSON_PARSER_MEM_NEEDS = 12
memory_limit = 314000000
''' enum support missing in older XR releases
class Status(str, Enum):
    # To convey the result status of gathering data
    Success = 'Success'
    TryAgain = 'TryAgain'
    NotSupported = 'NotSupported'
    ResourceFailure = 'ResourceFailure'
'''


def gen_dict_extract(key, var, exact_match = True, endswith = False):
    if hasattr(var,'items'):
        non_leaf = []
        for k, v in var.items():
            #print ("here " + k)
            if key == k:
                yield (k, v)
            elif endswith and k.endswith(":"+key):
                yield(k,v)
            elif not exact_match and key in k:
                yield (k, v)
            if isinstance(v, dict) or isinstance(v, list):
                non_leaf.append(v)
        for v in non_leaf:
            if isinstance(v, dict):
                for (key_name, result_value) in gen_dict_extract(key, v, exact_match, endswith ):
                    yield (key_name, result_value)
            elif isinstance(v, list):
                for d in v:
                    for (key_name, result_value) in gen_dict_extract(key, d, exact_match, endswith):
                        yield (key_name, result_value)

class Status():
    # To convey the result status of gathering data
    Success = 'Success'
    TryAgain = 'TryAgain'
    NotSupported = 'NotSupported'
    ResourceFailure = 'ResourceFailure'

class netconf_collector:
    # Global variables
    HELLO = """
    <?xml version="1.0" encoding="UTF-8"?>
    <hello xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
      <capabilities>
        <capability>urn:ietf:params:netconf:base:1.1</capability>
      </capabilities>
    </hello>
    ]]>]]>"""

    CLOSE = """
        <close-session/>
    """

    dossier_json_schema = '/pkg/bin/Cisco-IOS-XR-ama.jsonschema'

    def __init__(self):
        self.BUFSIZE = 65536
        self.capabilities = dict()
        self.connected = False
        (self.pipe, hello) = self.connect_onbox(None)
        if (self.pipe is not None):
           self.connected = True
           self.populate_capabilities(hello)
        if os.path.exists(self.dossier_json_schema):
            f = open(self.dossier_json_schema, 'r')
            schema = f.read()
            f.close()
        else:
            schema = '{}'
            logging.error("Schema file does not exist {}".format(self.dossier_json_schema))
        self.js = json.loads(schema, object_pairs_hook=OrderedDict)

    def populate_capabilities(self, hello):
        hello_response = json.loads(self.xml2json(hello))
        re_model = '^([^\?]+)\?module=([^&]+)&?revision=([^&]+)?'
        if 'hello' in hello_response and 'capabilities' in hello_response['hello'] and 'capability' in \
                hello_response['hello']['capabilities']:
            for element in hello_response['hello']['capabilities']['capability']:
                m = re.search(re_model, element)
                if m:
                    self.capabilities[m.group(2)] = [m.group(1), m.group(3)]
        #print(self.capabilities)

    def strip_tag(self, tag):
        strip_ns_tag = tag
        split_array = tag.split('}')
        if len(split_array) > 1:
            strip_ns_tag = split_array[1]
            tag = strip_ns_tag
        return tag

    def elem_to_internal(self, elem, strip_ns=1, strip=1, schema={}, indent=""):
        """Convert an Element into an internal dictionary (not JSON!)."""

        d = OrderedDict()
        elem_tag = elem.tag

        if strip_ns:
            elem_tag = self.strip_tag(elem.tag)
        for key, value in list(elem.attrib.items()):
            d['@' + key] = value

        # loop over subelements to merge them
        for subelem in elem:

            tag = subelem.tag
            objtype = "string"
            if strip_ns:
                tag = self.strip_tag(subelem.tag)
            s = {}
            if schema:
                find_schema = gen_dict_extract(tag, schema, exact_match=True, endswith=True)
                s = {}
                for (n, s) in find_schema:
                    try:
                        typefromsch = gen_dict_extract('type', s)
                        for (x, objtype) in typefromsch:
                            if isinstance(objtype, dict):
                                for objtype in objtype.keys():
                                    logging.debug("{}{} is of type {}".format(indent, tag, objtype))
                            break
                    except KeyError:
                        logging.debug("no type in {} {}".format(n, s))
                        objtype = None
                    # print("Found schema for {} {}".format(n, objtype ))
                    sub_schema = s
                    break
                if not s:
                    logging.debug("No schema found for {}".format(tag))
            #Remove parent type before passing the schema, to prevent name collision with a member of type
            parenttype = s.pop('type', None)
            v = self.elem_to_internal(subelem, strip_ns=strip_ns, strip=strip, schema=s, indent=" " + indent)
            #Restore parent type
            if parenttype:
                s['type'] = parenttype
            value = v[tag]

            try:
                # add to existing list for this tag
                d[tag].append(value)
            except AttributeError:
                # turn existing entry into a list
                d[tag] = [d[tag], value]
            except KeyError:
                # add a new non-list entry
                if (objtype == "array"):
                    logging.debug("{}printing {} as array".format(indent, tag))
                    d[tag] = [value]
                elif (objtype == "number" or objtype == "integer"):
                    try:
                        logging.debug("{}printing {} as {}".format(indent, tag, objtype))
                        d[tag] = int(value)
                    except ValueError:
                        d[tag] = value
                elif (objtype == "enum"):
                    logging.debug("{}printing {} as enum {}".format(indent, tag, value))
                    d[tag] = value
                elif (objtype == "boolean"):
                    logging.debug("{}printing {} as boolean {}".format(indent, tag, value))
                    if (value == "true"):
                        d[tag] = True
                    elif (value == "false"):
                        d[tag] = False
                else:
                    d[tag] = value
        text = elem.text
        tail = elem.tail
        if strip:
            # ignore leading and trailing whitespace
            if text:
                text = text.strip()
            if tail:
                tail = tail.strip()
        if d:
            # use #text element if other attributes exist
            if text:
                d["#text"] = text
        else:
            # text is the value if no attributes
            d = text or None
        return {elem_tag: d}

    def elem2json(self, elem, strip_ns=1, strip=1, schema={}):
        """Convert an ElementTree or Element into a JSON string."""

        if hasattr(elem, 'getroot'):
            elem = elem.getroot()

        return json.dumps(self.elem_to_internal(elem, strip_ns=strip_ns, strip=strip, schema=schema))


    def xml2json(self, xmlstring, strip_ns=1, strip=1, model="", schema={}):

        """Convert an XML string into a JSON string.
        logging.debug("<{}>".format(model))
        logging.debug(xmlstring)
        logging.debug("</{}>".format(model))"""
        try:
            elem = ET.fromstring(xmlstring)
        except ET.ParseError as err:
            logging.error("failed (%s)\n" % str(err))
            return None
        return self.elem2json(elem, strip_ns=strip_ns, strip=strip, schema=schema)

    def connect_onbox(self, user):
        if user is None:
            # Username not provided. Get it from env variable
            login_user = os.getenv("AAA_USER")
            if login_user is None or len(login_user) == 0:
                login_user = 'lab'
        else:
            login_user = user

        # Connect with netconf agent via ssh proxy client
        STDIN = '0'  # Assuming stdin will be always 0 on any env
        STDOUT = '1'  # Assuming stdout will be always 1 on any env
        logging.debug("- Connecting to netconf agent on the box...")
        try:
            p = subprocess.Popen(
                ['netconf_sshd_proxy', '-i', STDIN, '-o', STDOUT,
                 '-u', login_user],
                bufsize=self.BUFSIZE,
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT,
                shell=False)
        except Exception as err:
            logging.debug("failed (%s)\n" % str(err))
            logging.error("Failed to start netconf session!")
            logging.error("Please make sure you have 'netconf-yang agent ssh'")
            logging.error("configured on your router.")
            return (None, None)

        logging.debug("success")
        (stdouterr, stdin) = (p.stdout, p.stdin)

        # Wait for hello message from agent
        logging.debug('- Connected to NETCONF agent. Waiting for <hello> message...')

        buf = ''
        while True:
            data = stdouterr.readline()
            if not data:
                logging.debug('failed. No data received!')
                print
                "\nFailed to start netconf session!"
                print
                "Please make sure you have 'netconf-yang agent ssh'",
                print
                "configured on your router."
                return (None, None)
            buf += data.strip()
            if buf.endswith(']]>]]>') or buf.endswith('</hello>'):
                # hello message has been received
                break
        logging.debug("success")

        logging.debug("- <hello> message was received from NETCONF agent.")
        logging.debug("- Now sending our <hello> message...")

        stdin.write(self.HELLO)
        stdin.flush()

        logging.debug("success")
        return (p, buf)

    def get_schema_for_model(self, model):
        map_for_model = OrderedDict()
        try:
            find_schema = gen_dict_extract(model + ":", self.js['properties']['Cisco-IOS-XR-ama:ama-response'],
                                           exact_match=False)
            for (name, schema) in find_schema:
                key = name
                if model + ":" in name:
                    key = name[len(model + ":"):]
                map_for_model[key] = schema
        except KeyError:
            logging.error("Error fetching schema for "+model)
        return (map_for_model)

    def get_json_from_model(self, model, container):
        if (self.connected == False):
            return Status.ResourceFailure, None
        else:
            p = self.pipe
        if (model not in self.capabilities):
            return Status.NotSupported, None
        if p is not None:
            ns = self.capabilities[model][0]
            filter = '<{} xmlns="{}"/>'.format(container, ns)

            get_req = """ 
<get>
<filter>
"""
            get_req += filter
            get_req += """
</filter>
</get>
"""
            rpc = """
<rpc message-id="101" xmlns="urn:ietf:params:xml:ns:netconf:base:1.0">
"""
            rpc += get_req
            rpc += """
</rpc>
"""
            req_str = '\n#' + str(len(rpc) - 2) + rpc + '##\n'
            (stdouterr, stdin) = (p.stdout, p.stdin)
            stdouterr.flush()
            try:
                stdin.write(req_str)
                stdin.flush()
            except:
                print
                "failed to send data!"
                exit
            buf = ''
            flags = fcntl.fcntl(p.stdout, fcntl.F_GETFL) # get current p.stdout flags
            fcntl.fcntl(p.stdout, fcntl.F_SETFL, flags | os.O_NONBLOCK)
            bytes_recvd = 0
            logging.debug("Begin Collection of {}".format(get_req))
            last_rd_time = time.time()
            while True:
                line = ""
                try:
                    line = os.read(p.stdout.fileno(), self.BUFSIZE)
                except:
                    time.sleep(0.5)
                    logging.debug("no more data {}".format(len(line)))
                bytes_recvd += len(line)
                cur_time = time.time()
                if cur_time-last_rd_time > 60:
                    print_dossier_progress(model, bytes_recvd)
                    last_rd_time = cur_time
                logging.debug(len(line))
                if line.strip().endswith('##'):
                    logging.debug("line is {}".format(line))
                    buf += line
                    break
                buf += line
            logging.debug("End Collection of {}, bytes received = {}".format(get_req, bytes_recvd))
            start_at = buf.find('<{} '.format(container))
            end_at = buf.rfind('</{}>'.format(container)) + len('</{}>'.format(container))
            if (start_at == -1 or end_at == -1):
                return(Status.Success, "{}")
            buf = buf[start_at:end_at]
            logging.debug("Received {} of len {}".format(container, len(buf)))
            map_for_model = self.get_schema_for_model(model)
            jsondata = self.xml2json(buf, 1, 1, model=model,schema = map_for_model)
            if jsondata is not None:
                return(Status.Success, jsondata)
            else:
                return(Status.ResourceFailure, None)
        else:
            return(Status.ResourceFailure, None)


def does_bin_exists(program):
    import os
    def is_exe(fpath):
        return os.path.isfile(fpath) and os.access(fpath, os.X_OK)

    fpath, fname = os.path.split(program)
    if fpath:
        if is_exe(program):
            return True
    else:
        for path in os.environ["PATH"].split(os.pathsep):
            exe_file = os.path.join(path, program)
            if is_exe(exe_file):
                return True
    return False

clistrmap = { # dossier subtree : ['cli','parameters']
    "version": ["ng_show_version"],
    #"rollback-history" : ["show_config_changes", "-r -n 0x64"],
    "license-udi": ["smartliccmd","show license smart udi"],
    "platform": ["show_platform_sysdb"],
    "running-config": ["nvgen",  "-c -l 1 -t 1 -o 1"],
    #"packages": ["xrinstall", "show-packages active"], # "packages": ["sdr_instcmd", "show install active"],
}

sys_intergrity_jsonmap = { # dossier subtree : ['cli','parameters']
    "identity-certificates" : ["attestation_app", "-t 1 -l all -C CiscoSUDI -z -j"], #-x 123456
    #"attestation-certificates" : ["attestation_app", "-t 1 -l all -C CiscoAIK -z -j"], #-x 123456
    #"system-boot-integrity" : ["attestation_app", "-t 2 -l all -z -j"],
    "system-ima": ["attestation_app", "-t 4 -z -j -u -l all"],
    #"platform-config-registers": ["attestation_app", '-t 3 -l all -p 0,1,2,3,4,5,6,7,8,9,10,15 -z -j'],
    #"hardware-integrity": ["attestation_app", "-t 5 -l all -z -j"],
    }

model_map = { # dossier subtree : ['model-name','container-name']
    "system-inventory":["Cisco-IOS-XR-invmgr-oper","inventory", "inventory"],
    "reboot-history":["Cisco-IOS-XR-linux-os-reboot-history-oper","reboot-history"],
    "rollback-history":["Cisco-IOS-XR-config-cfgmgr-exec-oper","config-manager"],
    "packages": ["Cisco-IOS-XR-install-oper", "install","install"]
}

alternate_model_map = {
        "system-inventory":["Cisco-IOS-XR-spi-invmgr-oper","inventory","lindt-inventory"],
        "packages": ["Cisco-IOS-XR-spirit-install-instmgr-oper", "software-install","software-install"]
}

def get_alternate_model_map(topic):
    sublevel = None
    if (topic in alternate_model_map):
        if len(alternate_model_map[topic]) == 3:
            sublevel = alternate_model_map[topic][2]
        return(alternate_model_map[topic][0], alternate_model_map[topic][1],sublevel)
    return(None,None, None)

def get_command_out(command, parameters):
    clioutput = b""
    try:
        if (does_bin_exists(command)):
            clioutput = subprocess.Popen(parameters, stdout=subprocess.PIPE)
            status = Status.Success
            return(status, clioutput.communicate()[0])
        else:
            status = Status.NotSupported
    except subprocess.CalledProcessError as e:
        clioutput = e.output
        status = Status.ResourceFailure
    return(status, clioutput)

def exclude_default_data(exclude_data):
   exclude_data["system-integrity-snapshot"] = 1
   exclude_data["running-config"] = 1
   exclude_data["system-inventory"] = 1
   exclude_data["rollback-history"] = 1
   exclude_data["reboot-history"] = 1
   exclude_data["packages"] = 1

def print_usage(cmd):
    print("{} [-h --system-integrity-snapshot=0x1 --running-config=0x1 \
    --system-inventory=0x1 --rollback-history=0x1 --reboot-history=0x1 \
    --packages=0x1 --nonce <hex string of nonce> --start-ima-event <event number> --max-ima-events <number>]".format(cmd))

first = True
def print_header(headerelements):
    global first
    jse = jsonencoder.JSONEncoder()
    sys.stdout.write("{")
    for key in headerelements:
        if not first:
            sys.stdout.write(",")
        sys.stdout.write("\"" + key + "\":")
        sys.stdout.write(jse.encode(headerelements[key]))
        first = False

def print_trailer(trailelements):
    global first
    jse = jsonencoder.JSONEncoder()
    for key in trailelements:
        if not first:
            sys.stdout.write(",")
        sys.stdout.write("\"" + key + "\":")
        sys.stdout.write(jse.encode(trailelements[key]))
        first = False
    sys.stdout.write("}")

def print_clioutput(cli,output):
    global first
    jse = jsonencoder.JSONEncoder()
    if not first:
        sys.stdout.write(",")
    sys.stdout.write("\"" + cli + "\":")
    sys.stdout.write(jse.encode(output))
    first = False

def print_dossier_progress(model, bytes_received):
    global first
    if not first:
        sys.stdout.write(",")
    sys.stdout.write("\"dossier-progress\":{")
    sys.stdout.write("\"stage\":\""+model+"\"")
    sys.stdout.write(",\"bytes\":"+str(bytes_received)+"}")
    first = False

def print_model_header(model, headerelements):
    global first
    firstele = True
    jse = jsonencoder.JSONEncoder()
    if not first:
        sys.stdout.write(",")
    sys.stdout.write("\""+model+"\":{")
    for key in headerelements:
        if not firstele:
            sys.stdout.write(",")
        sys.stdout.write("\"" + key + "\":")
        sys.stdout.write(jse.encode(headerelements[key]))
        firstele = False
    first = False
    return firstele

def print_model_trailer(firstele, trailelements):
    jse = jsonencoder.JSONEncoder()
    if trailelements is not None:
        for key in trailelements:
            if not firstele:
                sys.stdout.write(",")
            sys.stdout.write("\"" + key + "\":")
            sys.stdout.write(jse.encode(trailelements[key]))
            firstele = False
    sys.stdout.write("}")

def print_model_content(firstele, content):
    jse = jsonencoder.JSONEncoder()
    thisfirst = True
    if not content:
        return firstele
    if not firstele:
        sys.stdout.write(",")
    for key in content:
        if not thisfirst:
            sys.stdout.write(",")
        sys.stdout.write("\"" + key + "\":")
        sys.stdout.write(jse.encode(content[key]))
        thisfirst = False
    firstele = False
    return firstele


def print_integrity_header(headerelements):
    global first
    jse = jsonencoder.JSONEncoder()
    if not first:
        sys.stdout.write(",")
    sys.stdout.write("\"system-integrity-snapshot\":{")
    first = False
    firstele = True
    if not headerelements:
        return firstele
    for key in headerelements:
        if not firstele:
            sys.stdout.write(",")
        sys.stdout.write("\"" + key + "\":")
        sys.stdout.write(jse.encode(headerelements[key]))
        firstele = False
    return firstele

def print_integrity_trailer(firstele, trailelements):
    jse = jsonencoder.JSONEncoder()
    if trailelements is not None:
        for key in trailelements:
            if not firstele:
                sys.stdout.write(",")
            sys.stdout.write("\"" + key + "\":")
            sys.stdout.write(jse.encode(trailelements[key]))
            firstele = False
    sys.stdout.write("}")

def print_integrity_command_out(command, parameters, firstele, header):
    try:
        if (does_bin_exists(command)):
            if not firstele:
                sys.stdout.write(",")
            firstele = False
            sys.stdout.write("\""+header+"\":")
            sys.stdout.flush()
            subprocess.check_call(parameters, stdout=sys.stdout)
            sys.stdout.flush()
            status = Status.Success
            return(status, firstele)
        else:
            status = Status.NotSupported
    except subprocess.CalledProcessError as e:
        status = Status.ResourceFailure
        sys.stdout.write("{}")
    except RuntimeError:
        logging.error(RuntimeError)
        logging.error("Failed to collect data due to RuntimeError {} {}".format(command, parameters))
        sys.stdout.write("{}")
    return(status,firstele)

def no_json_data():
    header = OrderedDict()
    header["collection-start-time"] = time.time()
    header["model-name"] = model_namespace
    header["model-revision"] = model_revision
    header["result-code"] = Status.TryAgain
    print_header(header)
    trailer = OrderedDict()
    trailer["collection-end-time"] = time.time()
    print_trailer(trailer)

def main(cmd, argv):
    exclude_data = dict()
    exclude_default_data(exclude_data)
    nonce=""
    start_ima_event = 0
    max_ima_events = 0

    try:
        opts, args = getopt.getopt(argv, "ha:r:i:o:b:p:n:s:m:",["system-integrity-snapshot=","running-config=",
                                                          "system-inventory=","rollback-history=",
                                                          "reboot-history=","packages=",
                                                                "nonce=","start-ima-event=", "max-ima-events="])
    except getopt.GetoptError:
        print_usage(cmd)
        sys.exit(2)
    for opt, arg in opts:
        if opt == '-h':
            print_usage(cmd)
            sys.exit()
        elif opt in ("-a", "--system-integrity-snapshot") and arg == "0x1":
            exclude_data.pop("system-integrity-snapshot", None)
        elif opt in ("-r", "--running-config") and arg == "0x1":
            exclude_data.pop("running-config", None)
        elif opt in ("-i", "--system-inventory") and arg == "0x1":
            exclude_data.pop("system-inventory", None)
        elif opt in ("-o", "--rollback-history") and arg == "0x1":
            exclude_data.pop("rollback-history", None)
        elif opt in ("-b", "--reboot-history") and arg == "0x1":
            exclude_data.pop("reboot-history", None)
        elif opt in ("-p", "--packages") and arg == "0x1":
            exclude_data.pop("packages", None)
        elif opt in ("-n", "--nonce"):
            nonce=arg
        elif opt in ("-s", "--start-ima-event"):
            start_ima_event = arg
        elif opt in ("-m", "--max-ima-events"):
            max_ima_events = arg
    header = OrderedDict()
    header["collection-start-time"] = time.time()
    header["model-name"] = model_namespace
    header["model-revision"] = model_revision
    print_header(header)

    for cli in clistrmap:
        if cli in exclude_data:
            continue
        printcli = OrderedDict()
        clioutput = ""
        command = clistrmap[cli][0]
        parameters = []
        parameters.append(command)
        if (len(clistrmap[cli]) > 1):
            parameters.extend((clistrmap[cli][1]).split())
        (status, clioutput) = get_command_out(command, parameters)
        printcli["result-code"] = status
        if (status == Status.Success):
            printcli[cli] = clioutput.decode("utf-8")
        print_clioutput(cli,printcli)

    nc = None
    for getnc in model_map:
        if getnc in exclude_data:
            continue
        if not nc:
            nc = netconf_collector()
        model_name = None
        container = None
        sublevel = None
        modelout = OrderedDict()
        if (model_map[getnc][0] not in nc.capabilities):
            (alt_model, container, sublevel) = get_alternate_model_map(getnc)
            if (alt_model != None and alt_model in nc.capabilities):
                model_name = alt_model
            else:
                modelout["result-code"] = Status.NotSupported
                firstele = print_model_header(getnc, modelout)
                mode_name = None
                print_model_trailer(firstele, None)
        else:
            model_name = model_map[getnc][0]
            container = model_map[getnc][1]
            if len(model_map[getnc]) == 3: 
                sublevel = model_map[getnc][2]
        if (model_name != None):
            (status, data) = nc.get_json_from_model(model_name, container)
            modelout["result-code"] = status
            modelout["model-name"] = model_name
            modelout["model-revision"] = nc.capabilities[model_name][1]
            firstele = print_model_header(getnc, modelout)
            if (status == Status.Success):
                content = json.loads(data)
                contentout = OrderedDict()
                if content.has_key(container) and sublevel:
                    contentout[sublevel] = content[container]
                    firstele = print_model_content(firstele, contentout)
                elif content.has_key(container):
                    firstele = print_model_content(firstele, content[container])
            print_model_trailer(firstele, None)

    if ("system-integrity-snapshot" not in exclude_data):
        sys_int_out = OrderedDict()
        model_name = "Cisco-IOS-XR-remote-attestation-act"
        received_output = 0
        if not nc:
            nc = netconf_collector()
        if (model_name not in nc.capabilities):
            sys_int_out["result-code"] = Status.NotSupported
            firstele = print_integrity_header(sys_int_out)
            print_integrity_trailer(firstele, None)
        else:
            sys_int_out["model-name"] = model_name
            sys_int_out["model-revision"] = nc.capabilities[model_name][1]
            firstele = print_integrity_header(sys_int_out)
            sys_int_trailer = OrderedDict()
            sys_int_trailer["result-code"] = Status.Success
            for cli in sys_intergrity_jsonmap:
                clioutput = ""
                command = sys_intergrity_jsonmap[cli][0]
                parameters = []
                parameters.append(command)
                if (len(sys_intergrity_jsonmap[cli]) > 1):
                    parameters.extend((sys_intergrity_jsonmap[cli][1]).split())
                if (len(nonce) != 0):
                    parameters.extend(["-x",nonce])
                if (start_ima_event != 0):
                    parameters.extend(["-s",start_ima_event])
                if (max_ima_events != 0):
                    parameters.extend(["-e", max_ima_events])
                (status, firstele) = print_integrity_command_out(command, parameters, firstele, cli)
                if (status != Status.Success):
                    sys_int_trailer["result-code"] = status
            print_integrity_trailer(firstele, sys_int_trailer)
    trailer = OrderedDict()
    trailer["collection-end-time"] = time.time()
    print_trailer(trailer)

brute_resource = False
try:
  import resource
except ImportError:
  brute_resource = True
  import ctypes
  from ctypes import *
  class rlimit(Structure):
              _fields_ = [("rlim_cur", ctypes.c_ulonglong), ("rlim_max", ctypes.c_ulonglong)]

def has_memory(needed):
  current = getCurrentMemoryUsage()
  if (memory_limit - current) < needed:
      return False
  return True

def getrlimit():
    ''' Memory limits in bytes '''
    mlimiti = 0
    if not brute_resource:
        mlimiti = resource.getrlimit(resource.RLIMIT_DATA)[0]
    if mlimiti < 0 or brute_resource:
        ''' Python 2.7.3's getrlimit implementation is not considering long long values. See   \
            Modules/resource.c:114 in the Python-2.7.3 source code. HAVE_LONG_LONG is not       \
        defined '''
        with open('/proc/self/limits') as f:
            mlimit = f.read()
            mlimit = mlimit.split('Max data size')[1].split("\n")[0].strip().split(' ')[0]
            mlimiti = int(mlimit.strip())
            f.close()
    logging.debug("Process memory limits : {} ".format(mlimiti))
    return (mlimiti)

def setrlimitdata():
    ''' Increasing rlimit  '''
    limit = 1024 * 1024 *1024 
    if not brute_resource:
        ''' Brute force approach is required for Lindt platforms which don't
            import the resource package'''
        resource.setrlimit(resource.RLIMIT_DATA, (limit, resource.RLIM_INFINITY))
    else:
        try:
            libmem = CDLL("libmemdbg.so")
            newlim = rlimit(limit, limit)
            libmem.setrlimit(2, byref(newlim))
        except:
            logging.error("Failed to set rlimit data to {}".format(limit))

def getCurrentMemoryUsage():
    ''' Memory usage in kB '''
    if brute_resource:
        with open('/proc/self/status') as f:
            memusage = f.read()
            memusage = memusage.split('VmRSS:')[1].split('\n')[0][:-3]
            memusagei = int(memusage.strip()) * 1024
    else:
          memusagei = resource.getrusage(resource.RUSAGE_SELF).ru_maxrss
    return(memusagei)

if __name__ == "__main__":
   logging.basicConfig(format='%(asctime)s %(levelname)-8s %(message)s',
           filename='/var/log/dossier.log', level=logging.INFO, datefmt='%Y-%m-%d %H:%M:%S')
   fp = open('/tmp/.dossier.lock', 'w')
   try:
       fcntl.lockf(fp, fcntl.LOCK_EX | fcntl.LOCK_NB)
   except IOError:
       no_json_data()
       logging.error('One instance of dossier is already running')
       sys.exit(1)
   setrlimitdata()
   memory_limit = getrlimit()
   logging.info("Started {}".format(os.getpid()))
   main(sys.argv[0], sys.argv[1:])
   logging.info("Ended {}".format(os.getpid()))



