#!/bin/bash
# -------------------------------------------------
# disk_space_precheck.sh
# Script for disk space precheck
#
# Aug 2016, Anushree Jangid
# 
# Copyright (c) 2016-2019 by Cisco Systems, Inc.
# All rights reserved.
# -------------------------------------------------

log_device="/var/log/install"
log_file="$log_device/disk_space-""`date +%Y-%b-%d.%H%M%S.%Z`"".log"
log_error_file="$log_device/disk_space_err-""`date +%Y-%b-%d.%H%M%S.%Z`"".log"
SCRIPTPATH="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
DATA_MISC_SCRATCH="/misc/scratch"
DATA_MISC_CONFIG="/misc/config"

if [ -f /etc/cisco/xr_sysctl.rc ]; then
    source /etc/cisco/xr_sysctl.rc
fi

echo_info() {
  echo "INFO:`date +%Y-%b-%d.%H%M%S.%Z` $1 ">>"$log_file"
}

echo_success() {
  echo "$1" 
  echo  "SUCCESS: `date +%Y-%b-%d.%H%M%S.%Z` $1 ">>"$log_file"
}

echo_err() {
  echo "ERROR: `date +%Y-%b-%d.%H%M%S.%Z` $1 ">>"$log_error_file"
}

echo_warn() {
  echo "WARNING: `date +%Y-%b-%d.%H%M%S.%Z` $1 ">>"$log_error_file"
}

echo_debug(){
  echo "DEBUG: `date +%Y-%b-%d.%H%M%S.%Z` $1 ">>"$log_file" 
}

################################################################################
# Enable error check
################################################################################
enable_err(){
  set -e
  set -o pipefail
}

################################################################################
# Disable error check
################################################################################
disable_err(){
  set +e
  set +o pipefail
}

###############################################################################
# This function will help to validate a package against valid characters 
# allowed in a package name
###############################################################################
function validate_package_name() {
   pkg_name="$1"
   rc=0
   if [[ "pkg$pkg_name" == "pkg"|| "$pkg_name" =~ [^A-Za-z0-9_.\-] ]] ; then
      rc=1
   fi
   return $rc
}

###############################################################################
# This function calculates the size of package specified as part of install
# operation. If the package is not found size is returned as 0.
# A temporary file pkg_info is created which stores the type of package,
# name of package and size.
##############################################################################
function store_package_size() {
  disable_err
  echo_info "In function ${FUNCNAME[*]}"
  pkg_name="$1"
  xr="/install_repo/gl/xr"
  cal="/install_repo/gl/instdb/clos-master"
  aswp="-swprofile-active.txt"
  cswp="-swprofile-committed.txt"
  swp_paths="$cal$aswp $cal$cswp $xr$aswp $xrcswp"
  mini_path="/misc/disk1/tftpboot/"
  pkg_info_file="$log_device/pkg_info"
  xr_repo="/install_repo/gl/xr/"
  cal_repo="/install_repo/gl/calvados/"   
  validate_package_name "$pkg_name"
  ret_code=$?
  if [ $ret_code -ne 0 ]; then
      echo_info "Invalid Package: $pkg_name"
      enable_err
      return
  fi
  #If mini or full image, then package would be present in tftpboot

  if [[ $pkg_name =~ mini ]] || [[ $pkg_name =~ full ]]; then
    echo_info "Package to be processed is mini/full image"
    size=$(ls -lL --block-size=M "$mini_path$pkg_name"  | cut -d ' ' -f5 )
    if [ -z $size ]; then
      enable_err 
      return
    else
      bsize=${size: -1}
      nsize=$(echo ${size/%?} 1.2 1| awk '{ printf "%d", $1 * $2 + $3}')
      size=$nsize$bsize
      echo_success $size
      pkg_type="MINI"
      echo_success "Setting pkg type as "$pkg_type
      echo "$pkg_type,""$pkg_name,"${size/%?}  >> $pkg_info_file 
      echo_info "$pkg_type,""$pkg_name,"${size/%?}
    fi
  elif [[ $pkg_name =~ golden ]]; then
    HOSTRPMS="/host_rpms/"                                                    
    CALRPMS="/calvados_rpms/"                                                 
    XRRPMS="/xr_rpms/"  
    ISOMNT=$(mktemp -d /tmp/isomnt.XXXXXX)
    #this is golden ISO. process it accordingly
    if [ -f $mini_path$pkg_name ];then
      size=$(ls -lL --block-size=M "$mini_path$pkg_name"  | cut -d ' ' -f5 )
      if [ -z $size ]; then
        enable_err 
        return
      else
        pkg_type="MINI"
        echo_success "Setting pkg type as "${pkg_type}
        echo "$pkg_type,""$pkg_name,"${size/%?}  >> $pkg_info_file 
        echo_info "$pkg_type,""$pkg_name,"${size/%?}
      fi
      
      mount -r -o loop $mini_path$pkg_name ${ISOMNT}
      ret_code=$?
      if [ $ret_code -ne 0 ]; then
        echo_err "ERROR: could not mount $mini_path$pkg_name on ${ISOMNT}"
        enable_err
        return
      fi
      if [ -d ${ISOMNT}/${HOSTRPMS} ]; then
        pkg_type="HOST"
        pkg_info_str=$(ls -lL --block-size=M ${ISOMNT}/${HOSTRPMS}/* | awk -v \
                      pkg_type=$pkg_type '{print pkg_type "," $9 "," int($5)}')
        echo "${pkg_info_str}" >> ${pkg_info_file}
        echo_info "${pkg_info_str}"
      fi

      if [ -d ${ISOMNT}/${CALRPMS} ]; then
        pkg_type="CAL"
        pkg_info_str=$(ls -lL --block-size=M ${ISOMNT}/${CALRPMS}/* | awk -v \
                      pkg_type=$pkg_type '{print pkg_type "," $9 "," int($5)}')
        echo "${pkg_info_str}" >> ${pkg_info_file}
        echo_info "${pkg_info_str}"
      fi

      if [ -d ${ISOMNT}/${XRRPMS} ]; then
        pkg_type="XR"
        pkg_info_str=$(ls -lL --block-size=M ${ISOMNT}/${XRRPMS}/* | awk -v \
                      pkg_type=$pkg_type '{print pkg_type "," $9 "," int($5)}')
        echo "${pkg_info_str}" >> ${pkg_info_file}
        echo_info "${pkg_info_str}"
      fi
    fi
    [ -d "${ISOMNT}" ] && umount ${ISOMNT} && rmdir ${ISOMNT}
  else
    # Check if package exists in committed or active software profile.
    # If exists then return 0 as disk space check is not required
    # Otherwise check the package in calvados or xr repo.
    gstatus=$(grep -s "$pkg_name" $swp_paths 2>&1 1>/dev/null || true | wc -l)
    if [[ -z "$gstatus" ]]; then
      echo 0
      enable_err 
      return 0
    else
       echo_info "check if package is already in ${pkg_info_file}"
       grep -i "${pkg_name}" "${pkg_info_file}"
       ret_code=$?
       if [ "${ret_code}" -eq 0 ]; then
         echo_info "file ${pkg_name} already has an entry in ${pkg_info_file}"
         enable_err
         return 0;
       fi
       echo_info "Check for existence in XR repo"
       pkg_path="$(find "$xr_repo" -name "$pkg_name*" 2>/dev/null | \
                   head -n1)"
       if [ "pkg$pkg_path" != "pkg" ]; then
         pkg_type="XR"
       else
         echo_info "Check for existence in Calvados repo"
         pkg_path="$(find "$cal_repo" -name "$pkg_name*" 2>/dev/null | \
                     head -n1)"
         if [ "pkg$pkg_path" = "pkg" ]; then
           echo_info "Package not found.. Ignore.."
           echo 0
           enable_err
           return 0;
         fi

         # Get platform name for panini and scapa platform as their packages
         # are tar bundles. Other platforms are rpm based.
         platform=${IOS_XR_PLATFORM}
         case "$platform" in
           panini)
             platform="ncs6k"
               ;;
           scapa)
             platform="ncs4k"
               ;;
           *)
             echo_info "no change in platform "$platform
         esac
         # Packages present in calvados repo can be of host or calvados type
         # Vm type is set to "Host" if
         #  1. TP package contains .host. 
         #  2. Sysadmin hostos smu contains -sysadmin-hostos
         # Otherwise its a "Calvados" pacakge

         if [[ $pkg_path =~ \.admin\.* ]]; then
           pkg_type="CAL"
         elif [[ $pkg_path =~ \.host\.* ]]; then  
           pkg_type="HOST"
         
         # For packages that are tar archives for eg, ncs6k/ncs4k pkgs
         # Extract pkg_type from bundle_info file.

         elif [[ $pkg_path == ncs4k* ]]; then
           tar -xvf $pkg_path bundle_info.txt
           pkg_type=$(grep "Package-type:" bundle_info.txt)
           echo_info "Package type from bundle info "$pkg_type
           if [[ $pkg_type =~ HOST ]]; then
             pkg_type="HOST"
           else
             pkg_type="CAL"
           fi
         elif [[ $pkg_path =~ \-sysadmin\- ]]; then
           if [[ $pkg_path =~ \-hostos\- ]]; then
             pkg_type="HOST"
           else
             pkg_type="CAL"
           fi
         fi # $pkg_path =~ \.admin\.* 
       fi
       echo_info "pkg type "$pkg_type
       size=$(ls -lL --block-size=M "$pkg_path"  | cut -d ' ' -f5 )
       echo "$pkg_type,""$pkg_name,"${size/%?}  >> $pkg_info_file 
       echo_info "$pkg_type,""$pkg_name,"${size/%?}
    fi # Check for swp
  fi # Check for mini/full or other packages 
  enable_err
}

###############################################################################
# Create disk space data string. This will be used to query the platform 
# specific required size datasheet. 
# The string is constructed by reading i
# calvados/system_pkg/install/inst_common/scripts/disk_space_data.csv file.
# Sample data string "CAL-XR <> CAL: SUM=9999997, MAX=9999997, MIN=9999997 ;
# XR: SUM=22222288, MAX=22222227, MIN=1 ; 
# HOST: SUM=0, MAX=0, MIN=0 ; MINI: SUM=0, MAX=0, MIN=0"

###############################################################################
function get_disk_space_data_string() { 
  disable_err 
  echo_info "In function ${FUNCNAME[*]}"
  host_sum=0
  host_max=0
  host_min=0
  cal_sum=0
  cal_max=0
  cal_min=0
  xr_sum=0
  xr_max=0
  xr_min=0
  mini_sum=0
  mini_max=0
  mini_min=0
  pkg_info_file="/$log_device/pkg_info"
  package_category=$(cat $pkg_info_file | cut -d',' -f1 | sort | uniq | \
                     paste -s -d-)
  echo_info "Package category: "$package_category
  package_sizes=$(cat $pkg_info_file | cut -d',' -f1,3 | sort)
  package_sum_grp=$(echo "$package_sizes" | awk -F, '{
    a[$1]+=$2;
    }
    END{
      for(i in a)print i","a[i];
    }'
  )
  package_max_grp=$(echo "$package_sizes" | awk -F, '{
    if (a[$1] < $2)a[$1]=$2;
    }
    END{
      for(i in a){print i,a[i];}
    }' OFS=, 
  )

  package_min_grp=$(echo "$package_sizes" | awk -F, '{
    if (a[$1] == "")a[$1]=$2; 
    if (a[$1] > $2)a[$1]=$2;
    }
    END{
      for(i in a){print i,a[i];}
    }' OFS=, 
  )

  cal_sum=$(echo $(echo "$package_sum_grp" | grep "CAL" || echo "SUM=0") | \
                   sed 's/.*,/SUM=/')
  cal_max=$(echo $(echo "$package_max_grp" | grep "CAL" || echo "MAX=0" ) | \
                   sed 's/.*,/MAX=/')
  cal_min=$(echo $(echo "$package_min_grp" | grep "CAL" || echo "MIN=0" ) | \
                   sed 's/.*,/MIN=/')

  xr_sum=$(echo $(echo "$package_sum_grp" | grep "XR" || echo "SUM=0" ) | \
                  sed 's/.*,/SUM=/')
  xr_max=$(echo $(echo "$package_max_grp" | grep "XR" || echo "MAX=0" ) | \
                  sed 's/.*,/MAX=/')
  xr_min=$(echo $(echo "$package_min_grp" | grep "XR" || echo "MIN=0" ) | \
                  sed 's/.*,/MIN=/')

  host_sum=$(echo $(echo "$package_sum_grp" | grep "HOST" || echo "SUM=0" ) |\
                    sed 's/.*,/SUM=/')
  host_max=$(echo $(echo "$package_max_grp" | grep "HOST"|| echo "MAX=0" ) |\
                    sed 's/.*,/MAX=/')
  host_min=$(echo $(echo "$package_min_grp" | grep "HOST" || echo "MIN=0" ) |\
                    sed 's/.*,/MIN=/')

  mini_sum=$(echo $(echo "$package_sum_grp" | grep "MINI" || echo "SUM=0" ) |\
                    sed 's/.*,/SUM=/')
  mini_max=$(echo $(echo "$package_max_grp" | grep "MINI" || echo "MAX=0" ) |\
                    sed ' s/.*,/MAX=/')
  mini_min=$(echo $(echo "$package_min_grp" | grep "MINI" || echo "MIN=0" ) |\
                    sed  's/.*,/MIN=/')
  disk_space_data_string="$(echo "$package_category <>"\
                            "CAL: $cal_sum, $cal_max, $cal_min ;"\
                            "XR: $xr_sum, $xr_max, $xr_min ;"\
                            "HOST: $host_sum, $host_max, $host_min ;"\
                            "MINI: $mini_sum, $mini_max, $mini_min")" 
  echo "$disk_space_data_string"
  disk_space_data_string="$(echo $disk_space_data_string | \
                            awk -F'<>' '{ print $2 }')"
  echo_info "disk_space_data string is "$disk_space_data_string
  enable_err
}

###############################################################################
# Calculate free disk space in the given location. 
##############################################################################
function do_diskspace_precheck() {
  disable_err
  echo_info "In function ${FUNCNAME[*]}"
  loc="$1"
  req_space="$2"
  df_out="$3"
  rc=0 
  avail_space=$(echo "$df_out" |grep % | tail -n1 |  sed 's/\s\+/ /' | \
                awk -F' ' '{ print $(NF-2) }')
  avail_space=$(echo ${avail_space/%?})
  extra_space=$(($avail_space-$req_space))
  echo_info "Available space: "$avail_space
  echo_info "Required space: "$req_space
  echo_info "Extra space: "$extra_space
  to_print_loc=$(echo "$loc" | \
    sed 's-^/misc/disk1$-harddisk:-; s-^/install_repo$-install:-; s-^/$-rootfs:-')
  if [ "$extra_space" -lt 0 ]; then
    #echo "$to_print_loc partition requires at least $req_space MB free space."
    echo "$to_print_loc,$req_space"
    enable_err 
    return 1 
  fi
  enable_err 
  return 0
}

################################################################################
# Find the primary mount of a give location
################################################################################
function get_primary_mount() {
  loc="$1"
  if [ -z "$2" ]; then
    df_out="$(df -P --block-size=M $1)"
  else
    df_out="$2"
  fi
  loc=$(echo "$df_out" |grep % | tail -n1 |  sed 's/\s\+/ /' | \
                awk -F' ' '{ print $(NF) }')
  fs="$(df -a "$loc" | awk '{print $1}' | tail -n1)"
  loc="$(cat /proc/*/mountinfo | sort | uniq |grep '/ /' | \
         grep $fs | awk '{ print $5}')"
  echo "$loc"
}

################################################################################
# Verify disk space information on each node. 
# This script is called at all the nodes for calculating the required space
# and available space.
# Returns: The amount of space required output message in case of
#          insufficient disk space
#
# Sample function call:
# verify_disk_space "CAL-HOST-MINI-XR <> CAL: SUM=4, MAX=1, MIN=1 ; \
# XR: SUM=2, MAX=1, MIN=1 ; HOST: SUM=4, MAX=1, MIN=1 ; \
# MINI: SUM=2902, MAX=1091, MIN=1091;"
###############################################################################
function verify_disk_space() {
  disable_err 
  # Log cleanup
  ls -dt "$log_device/disk_space"* 2>/dev/null | tail -n +6 | 
          xargs -I {} rm -- {}
  echo_info "In function ${FUNCNAME[*]}"
  rm "$log_device/pkg_info" 2>/dev/null || true
  rc=0
  disk_space_data_string="$1"
  datasheet="$SCRIPTPATH""/disk_space_data.csv"
  
  # E.g CAL-HOST-MINI-XR
  package_category=$(echo $disk_space_data_string | \
                     awk -F' <>' '{ print $1 }')
  # String without package_category
  disk_space_data_string="$(echo $disk_space_data_string | \
                            awk -F' <>' '{ print $2 }')"
  # Get node type to find which node disk space check will be done
  node_type=${IOS_XR_BOARD_TYPE}
  echo_info "Package category: $package_category Node type: $node_type "

  # Extract the header from datasheet(startswith PLATFORM/TYPE/LOC)
  # and row containing package_category. Get the location, type and the 
  # expression to calculate disk space
  # sample output= 
  # output='RP:CAL:/install_repo:SUM(SUM(CAL), MAX(HOST), MAX(MINI), SUM(XR))
  #         RP:CAL:/:MAX(MAX(MINI), MAX(HOST), SUM(CAL), SUM(XR))
  #         RP:XR:/:SUM(SUM(SUM(XR), SUM(CAL), MAX(HOST), MAX(MINI)),\
  #                 MAX(SUM(XR), SUM(CAL), MAX(HOST), MAX(MINI)))'
  # Read the output as On RP node, CAL vm , /install_repo loc expression is xyz

  output=$(cat "$datasheet" | \
    grep -i "^$package_category:\|^PLATFORM\|^TYPE:\|^LOC:" | awk -F':' '{ 
      for (i=1; i<=NF; i++)  {
        a[NR,i] = $i
      }
    }
    NF>p { p = NF }
    END {    
      for(j=1; j<=p; j++) {
        str=a[1,j]
        for(i=2; i<=NR; i++){
          str=str":"a[i,j];
        }
        print str
      }
    }' | grep -i "^$node_type:" 
  ) 
  
  # Calculate the vm type in order to excude disk space information
  # On sysadmin-vm(CAL) XR disk space info is not required.

  vmtype=${IOS_XR_VM_TYPE}
  if [[ "$vmtype" = "sysadmin-vm" ]]; then 
    exclude_vmtype="XR"
  else 
    exclude_vmtype="CAL"; 
  fi
  echo_info "$output"
  
  disk_data=$(echo "$output" | awk  -F':' -v exclude_vmtype="$exclude_vmtype"\
              '$2 != exclude_vmtype { print $0 }'| cut -d':' -f3,4 ) 
  disk_data=$(echo "$disk_data" | grep -v NA)
  echo_info "$disk_data"
  loc_values=$(echo "$disk_data" | cut -d':' -f2)
  loc_data=$(echo "$disk_data" | cut -d':' -f1)
  echo_info "$loc_values $loc_data "
  loc=($loc_data)
  count=0

  # loc_values='SUM(SUM(CAL), MAX(HOST), MAX(MINI), SUM(XR))
  #       MAX(MAX(MINI), MAX(HOST), SUM(CAL), SUM(XR))'
  # loc_data='/install_repo
  #         /'
  if [ ${#loc[@]} -eq 0 ]; then
    echo_info "No location to do disk space precheck"
    enable_err
    return 0
  fi

  # disk space check is not required if there is no location

  if [ "$(echo -ne ${loc_values} | wc -m)" -eq 0 ]; then
    echo_info "disk space check not required"
  else
    all_space_info=""
    while IFS=$'\n' read -r line; do
      echo_debug "$line";
      space_req=$(parse_tab "$line" "$disk_space_data_string")
      df_out=$(df -P --block-size=M ${loc[$count]})
      if [ "df$df_out" = "df" ]; then
          echo_info "No disk space information available for ${loc[$count]}"
          continue
      fi
      prim_mount=$(get_primary_mount "${loc[$count]}" "$df_out")
      echo_info "Disk space information"
      echo_info "$df_out"
      echo_info "Actual: ${loc[$count]} Req Space: $space_req Primary: $prim_mount"
      if [ -n "$space_req" ]; then
          if [ "as$all_space_info" = "as" ]; then
              all_space_info="$prim_mount,$space_req"
          else
            all_space_info="$(echo -e "$all_space_info\n$prim_mount,$space_req")"
          fi
      fi
      count=$((count+1))
    done <<< "$(echo "$loc_values")"
    echo_info "all space info $all_space_info"
    space_sum_grp=$(echo "$all_space_info" | awk -F, '{
        a[$1]+=$2;
        }
        END{
           for(i in a){print i,a[i];}
        }' OFS=, 
    )
    echo_info "Final list of locations after grouping"
    echo_info "$space_sum_grp"
    all_space_info=""
    space_info="" 
    while IFS=$'\n' read -r line; do
      disk_loc="$(echo $line | awk -F',' '{ print $1 }')"
      space_req="$(echo $line | awk -F',' '{ print $2 }')"
      df_out=$(df -P --block-size=M $disk_loc)
      if [ "df$df_out" = "df" ]; then
          echo_info "No disk space information available for $disk_loc"
          continue
      fi
      echo_info "Disk space information"
      echo_info "$df_out"
      space_info=$(do_diskspace_precheck $disk_loc "$space_req" "$df_out")
      if [ -n "$space_info" ]; then
          if [ "as$all_space_info" = "as" ]; then
              all_space_info="$space_info"
          else
            all_space_info="$(echo -e "$all_space_info\n$space_info")"
          fi
          rc=1
      fi
    done <<< "$(echo "$space_sum_grp")"
  fi
  echo_info "$all_space_info"
  while IFS=$'\n' read -r line; do
      disk_loc="$(echo $line | awk -F',' '{ print $1 }')"
      req_space="$(echo $line | awk -F',' '{ print $2 }')"
      space_str="$disk_loc partition requires at least $req_space MB free space."
      echo
      echo "                     $space_str"
  done <<< "$(echo "$all_space_info")"
  echo_info "---- Disk space check ends ---"
  enable_err 
  return $rc
}

###############################################################################
# Solve the disk space requirement expressions. This will give the amount of
# space required.
# Example format: SUM(SUM(XR),SUM(XR))
# Input:  String containing 
# These are stored in disk_space_data.csv file
# The expression should be interpreted as 
#   Sum of all XR pkgs + Sum of all XR pkgs which means that space required
#   at a given location is 2*Sum of all XR packages
# The expression is solved in the same way as infix operation is solved.
#   SUM(SUM(XR),SUM(XR)) = SUM(XR)+SUM(XR)
# Here the outermost SUM is applied as an add operator.
##############################################################################
function parse_tab(){
  disable_err 
  stmt="$1"
  input="$2"
  curr_func=${stmt%%(*}
  curr_args=$(extract_args "$stmt")
  case "$curr_func" in
  SUM)
    echo_debug "Firstcall $curr_func::$curr_args"
    echo $(calc_SUM "$curr_args" "$input")
    ;;  
  MAX)
    echo_debug "Firstcall $curr_func::$curr_args"
    echo $(calc_MAX "$curr_args" "$input")
    ;;  
  MIN)
    echo_debug "Firstcall $curr_func::$curr_args"
    echo $(calc_MIN "$curr_args" "$input")
    ;;  
  *)  
    echo "Invalid"
    ;;  
  esac
  enable_err
}

function calc_SUM(){
  result=0
  stmt="$1"
  input="$2" 
  if [[ "$stmt" =~ [A-Z]+ ]]; then
    echo_debug "InnerCall SUM::$stmt"
    if [[ "$stmt" == *\(* ]]; then
      echo_debug "STMTPRE:  $stmt"
      rest_stmt=$(extract_rest_args "$stmt")
      echo_debug "STMTPOST:  $rest_stmt"
      curr_func=$( echo "${stmt%%(*}" | xargs)
      curr_args=$(extract_args "$stmt")
      prefix="calc"
      echo_debug "Calling $curr_func::$curr_args"
      curr_result=$($prefix'_'$curr_func "$curr_args" "$input")
      echo_debug "Returned $curr_result"
      rest_result=$($prefix'_SUM' "$rest_stmt" "$input")
      echo_debug "Result $result:$curr_result:$rest_result"
      result=$((result+curr_result+rest_result))
    else
      curr_args=$( echo $stmt | xargs)
      result=$(extract_data $curr_args "SUM" "$input")
    fi  
  fi
  echo $result
}

function calc_MAX(){
  stmt="$1"
  input="$2" 
  result=0
  if [[ "$stmt" =~ [A-Z]+ ]]; then
    echo_debug "InnerCall MAX::$stmt"
    if [[ "$stmt" == *\(* ]]; then
      echo_debug "STMT:  $stmt"
      rest_stmt=$(extract_rest_args "$stmt")
      echo_debug "STMTPOST:  $rest_stmt"
      curr_func=$( echo "${stmt%%(*}" | xargs)
      curr_args=$(extract_args "$stmt")
      prefix="calc"
      echo_debug "Calling $curr_func::$curr_args"
      curr_result=$($prefix'_'$curr_func "$curr_args" "$input")
      echo_debug "Returned $curr_result"
      rest_result=$($prefix'_MAX' "$rest_stmt" "$input")
      if [[ $curr_result -gt $result ]]; then
        result=$curr_result
      fi
      if [[ $rest_result -gt $result ]]; then
        result=$rest_result
      fi
    else
      curr_args=$( echo $stmt | xargs)
      result=$(extract_data $curr_args "MAX" "$input")
    fi
  fi
  echo $result
}

function calc_MIN(){
  stmt="$1"
  input="$2"  
  result=99999999999999999
  if [[ "$stmt" =~ [A-Z]+ ]]; then
    echo_debug "InnerCall MIN::$stmt"
    if [[ "$stmt" == *\(* ]]; then
      echo_debug "STMT:  $stmt"
      rest_stmt=$(extract_rest_args "$stmt")
      echo_debug "STMTPOST:  $rest_stmt"
      curr_func=$( echo "${stmt%%(*}" | xargs)
      curr_args=$(extract_args "$stmt")
      prefix="calc"
      echo_debug "Calling $curr_func::$curr_args"
      curr_result=$($prefix'_'$curr_func "$curr_args" "$input")
      echo_debug "Returned $curr_result"
      rest_result=$($prefix'_MIN' "$rest_stmt" "$input")
      if [[ $curr_result -le $result ]]; then
        result=$curr_result
      fi
      if [[ $rest_result -le $result ]]; then
        result=$rest_result
      fi
    else
      curr_args=$( echo $stmt | xargs)
      result=$(extract_data $curr_args "MIN" "$input")
    fi
  fi
  echo $result
}

function extract_args(){
  echo $( echo "$1" | awk -F'`' '{
    for (i=1; i<=length; ++i) {
      if (numLeft == 0 && substr($0, i, 1) == "(") {
        leftPos = i
        numLeft = 1
      } else if (substr($0, i, 1) == "(") {
         ++numLeft
      } else if (substr($0, i, 1) == ")") {
         ++numRight
      }
      if (numLeft && numLeft == numRight) {
         print substr($0, leftPos+1, i-leftPos-1)
        next
      }
    }
  }')
}

function extract_rest_args(){
  string="$1"
  string=$(echo $( echo "$string" | awk -F'`' '{
    for (i=1; i<=length; ++i) {
      if (numLeft == 0 && substr($0, i, 1) == "(") {
        leftPos = i
        numLeft = 1
      } else if (substr($0, i, 1) == "(") {
          ++numLeft
      } else if (substr($0, i, 1) == ")") {
          ++numRight
      }
      if (numLeft && numLeft == numRight) {
         print substr($0, i+1)
         next
      }
    }
  }'))
  echo $string | sed 's/^\s*,*//'
}

function extract_data(){
  class=$1
  type=$2
  input="$3" 
  result=$(echo "$input" | tr ";" "\n" | grep "$class:" | head | awk -F':' '{
    print $2}' | tr "," "\n" | grep "$type=" | awk -F'=' '{ print $2}')
  echo $result
  echo_debug "extract_data $1 $2 \"$input\""
}

##############################################################################
# xr_issu_data_partition_usage():
# check usage of /misc/scratch and /misc/config
# if usage touches 95% or more in these partitions return 1 else 0
##############################################################################
function xr_issu_data_partition_usage(){

   result=0
   disable_err
   ## check for /misc/scratch
   part=$(df -k | grep "$DATA_MISC_SCRATCH" | cut -d '%' -f1)
   used=$(echo ${part##* })
   echo_debug "The /misc/scratch used $used"

   if [[ "$used" -ge 95 ]];
      then
        result=1
   fi

   ## check for /misc/config
   part=$(df -k | grep "$DATA_MISC_CONFIG" | cut -d '%' -f1)
   used=$(echo ${part##* })
   echo_debug "The /misc/config used $used"
   if [[ "$used" -ge 95 ]];
      then
        result=1
   fi
   enable_err
   return $result
}

function unmount_deleted_files(){
   #find the deleted files.Following is the output format
   #/misc/disk1/tftpboot/ncs5500-mini-x-6.1.3copy2 (deleted) on 
   #/var/volatile/tmp/t2 type iso9660 (ro,relatime)
   disable_err
   out=$(mount |  grep -o "deleted) on .* type")
   if [ -z "$out" ]; then
      echo "INFO: No files found to unmount"
   else
      echo "$out"
      echo "$out" | grep -o "/[^ ]*" | xargs -L1 umount 
      result="$?"
      if [ "$result" -ne 0 ]; then
         echo "ERROR: Unable to unmount rc = $result"
      else
         echo "SUCCESS: Unmount done"
      fi
   fi
   enable_err
}

#
# Kill the process instmgr_reserve_ports if it is running.
#
function instmgr_reserve_ports_end(){

   ps aux | grep "[i]nstmgr_reserve_ports"
   if [[ $? == 0 ]]; then
        killall -e /opt/cisco/calvados/bin/instmgr_reserve_ports
        if [[ $? == 0 ]]; then
             echo "INFO: process terminated"
             echo 0
        else
             echo "ERROR: failed to ternminate the process"
             echo 1
        fi
   else
        #in any case grep fails but process exist?
        echo "WARNING: No such process"
        #any case try killing it to avoid problems, no harm in trying
        killall -e /opt/cisco/calvados/bin/instmgr_reserve_ports
        echo $?
   fi
}

function instagt_reserve_ports_end() {
   ps aux | grep "[i]nstagt_reserve_ports"
   if [[ $? == 0 ]]; then
        killall -e /opt/cisco/calvados/bin/instagt_reserve_ports
        if [[ $? == 0 ]]; then
             echo "INFO: process terminated"
             echo 0
        else
             echo "ERROR: failed to ternminate the process"
             echo 1
        fi
   else
        #in any case grep fails but process exist?
        echo "WARNING: No such process"
        #any case try killing it to avoid problems, no harm in trying
        killall -e /opt/cisco/calvados/bin/instagt_reserve_ports
        echo $?
   fi
}

#
# Check how many logs exist on the system and if we have
# more than 100 logs of install log(show install log), oldest
# log will be deleted, and if logs are less than 100 in number 
# no action
# CAL logs @ __LOG_DIR="/install_repo/gl/instdb/log/"
# XR logs @  __LOG_DIR="/var/log/install/instdb/log"
#

function clear_install_logs()
{
   __LOG_DIR="$1"
   __MAX_LOG_NUMBER=202 # 200 + 2 (. & ..)

   MAX_ITERATIONS=0
   #echo "The file is $__LOG_DIR"
   while true; do
       __logs=$(ls -l $__LOG_DIR | wc -l)
       echo "The number of log files $__logs"
       # if we have more than 100 log files then delete oldest log file
       if [ $__logs -ge $__MAX_LOG_NUMBER ]; then
         # get the oldest file leaving most recent 100 log files
         __out=$(ls -ltr $__LOG_DIR | grep -v total |  head -1)
         __log_file=$(echo ${__out##* })
         if [ -z $__log_file ]; then
             echo "ERROR: Failed to find log file name"
             echo 1
             break
         else
             rm -rf $__LOG_DIR$__log_file
             echo "Removing file $__LOG_DIR$__log_file" 
         fi
       else
         echo "No more log files to delete"
         echo 0
         break
       fi

       # limiting Max 1000 iteration at a time to avoid
       # unwanted delay in case if we have huge logs after upgrades
       if [ $MAX_ITERATIONS -le 1000 ]; then
           MAX_ITERATIONS=$[$MAX_ITERATIONS+1]
       else
             break
       fi
   done
}


function get_sp_regex()
{
    disable_err
    bootstrap_cfg="/etc/init.d/calvados_bootstrap.cfg"
    platform="$(grep "^[  ]*PLATFORM_EXT=" "$bootstrap_cfg" | cut -d'=' -f2)"
    if [[ -z $platform ]]; then 
        platform="$(grep "^[  ]*PLATFORM=" "$bootstrap_cfg" | cut -d'=' -f2)"
        if [[ -z $platform ]]; then 
            platform="[a-zA-Z0-9]+"
        fi   
    fi   
    sp_regex="($platform)-(sp[0-9]+)-(x|x64)-([0-9]{1,3}\.[0-9a-zA-Z\.]+)"
    echo "$sp_regex"
    enable_err
}

function is_file_name_sp()
{
    disable_err
    sp_regex="^$(get_sp_regex)\$"
    if [[ -n $1 ]]; then
        filename="$(basename $1)"
        if [[ "$filename" =~ $sp_regex ]]; then
            echo "${BASH_REMATCH[0]}, ${BASH_REMATCH[1]}, ${BASH_REMATCH[2]}, ${BASH_REMATCH[3]}, ${BASH_REMATCH[4]}"
            return 0
        else
            return 1
        fi
    else
        return 1
    fi
    enable_err
}
# Find the input rpms which are part of sp present in harddisk
function get_sp_rpms_from_input() {
    disable_err
    sp_regex="$(get_sp_regex)"
    spdir="$1"
    shift
    op_string="$1"
    shift
    input_list="$(echo "$@" | sed 's:\s\+:\n:g')"
    echo_debug "Input $input_list"
    sps="$(find "$spdir" -maxdepth 1 -type f 2>/dev/null| grep -E "/$sp_regex")"
    echo_debug "SPs:\n$sps"
    output=""
    if [[ -n $sps ]]; then
        for rpm in $input_list
        do
            ret=$(echo $rpm | grep -E "$sp_regex")
            if [[ -n $ret ]]; then
                continue
            fi
            found="$(echo "$sps" | xargs grep -H "Boxname:" | grep "$rpm" | cut -d':' -f1,3 | sort -u | xargs -L1)"
            if [[ "x$found" != "x" ]]; then
                found_sps="$(echo "$found" | cut -d':' -f1 | xargs -L1 basename | sort -u | paste -s -d',' | xargs)"
                output="$(echo -e "$output\n$rpm is part of $found_sps")"
            fi
        done
        if [[ -n $output ]]; then
            output="$(echo "$output" | xargs -L1 echo "             ")"
            output_count="$(echo "$output" | wc -l)"
            if [[ $output_count > 0 ]]; then
                echo "Error: Following SMU(s) will not be $op_string:"
                echo "$output"
                return $output_count
            fi
        fi
    fi
    enable_err
    return 0
}

function get_dependent_smus_from_sp_rpms() {
    disable_err
    optional_rpm="$(basename "$1")"
    if [[ -z $optional_rpm ]]; then
        enable_err
        return 0
    fi
    shift
    sp_rpms="$@"
    echo_debug "Optional rpm: $optional_rpm"
    echo_debug "Sp rpms: $sp_rpms"
    #Create regex for finding optional pkg's smu
    #opt_regex="$(echo $optional_rpm | sed 's/\([0-9a-zA-Z\-]\+\-\)\([^\-]\+\)\-\([^\.]\+\)\.\(.*\)/\1[0-9\\.]\\+-\3\\.CSC[a-zA-Z0-9]\\+\\.\4/')"
    opt_regex="$(echo $optional_rpm | sed 's/\([0-9a-zA-Z\-]\+\-\)\([^\-]\+\)\-\([^\.]\+\)\(.*\)/\1[0-9\\.]\\+-\3\\.CSC[a-zA-Z0-9]\\+\4/')"
    if [[ -z $opt_regex ]]; then
        enable_err
        return 1 
    fi
    dep_rpm="$(echo "$sp_rpms" | sed 's/\s\+/\n/g'  | grep "$opt_regex")"
    if [[ -z $dep_rpm ]]; then
        enable_err
        return 1
    fi
    echo "$dep_rpm"
    return 0
}

# input is a smu and it check for its dependent base rpm 
function check_base_rpms_for_smu() {
    disable_err
    smu_rpm="$(basename "$1")"
    if [[ -z $smu_rpm ]]; then
        enable_err
        return 0
    fi
    shift
    mini_rpms="$@"
    #Create regex for finding optional pkg's smu
    #opt_regex="$(echo $optional_rpm | sed 's/\([0-9a-zA-Z\-]\+\-\)\([^\-]\+\)\-\([^\.]\+\)\.\(.*\)/\1[0-9\\.]\\+-\3\\.CSC[a-zA-Z0-9]\\+\\.\4/')"
    opt_regex="$(echo $smu_rpm | sed 's/\([0-9a-zA-Z\-]\+\-\)\([^\-]\+\)\-\([^\.]\+\)\(.*\)/\1[0-9\\.]\\+-\3/')"
    if [[ -z $opt_regex ]]; then
        enable_err
        return 1 
    fi
    dep_rpm="$(echo "$mini_rpms" | sed 's/\s\+/\n/g'  | grep "$opt_regex")"
    if [[ -z $dep_rpm ]]; then
        enable_err
        return 1
    fi
    echo "$dep_rpm"
    return 0
}

# Function to filter add_id file
# Remove all the sp rpms from add_id file if sp info file is present in add_id dir

function filter_sp_rpms_in_addid_file() {
     disable_err
     addid_dir="$1"
     addid_file_path="$2"
     filteredList="/install/tmp/filteredlist"
     if [[ -z $addid_dir ]]; then
         enable_err
         return
     fi
     if [[ -z $addid_file_path ]]; then
         enable_err
         return
     fi
     cd $addid_dir
     rm -f $filteredList
     ls "$addid_dir"/*sp* | xargs -L1 basename | xargs -L1 -i echo {} 3 | sort -u >> $addid_file_path
     cut -d' ' -f1 "$addid_file_path" | grep -h -F -f - "$addid_dir"/*sp* | \
         grep Boxname | cut -d':' -f2 | xargs -L1 | \
          sort -u | grep -v -F -f - "$addid_file_path" >| $filteredList

     result="$?"
     if [ "$result" -ne 0 ]; then
          echo "ERROR: Unable to edit the addid file $addid_file_path rc = $result"
     else
         cat $filteredList | sort -u | xargs -L1 >| $addid_file_path
         echo "SUCCESS: Successfully altered the addid file"
         cat $addid_file_path
     fi
     enable_err
}

#Check the number of sp's present in a directory
function are_multiple_sps_specified() {
    sp_regex="$(get_sp_regex)"
    spdir="$1"
    sps_in_dir="$(file -L "${spdir}"/* | grep SERVICEPACK | cut -d":" -f 1)"
    sps_output=""
    sps_in_dir_count=0
    if [[ -n $sps_in_dir ]]; then
        sps_in_dir_count=$(echo "$sps_in_dir" | wc -l)
        for spfile in ${sps_in_dir};
        do
            spname="$(basename "$spfile")"
            sps_output="$(echo -e "$sps_output\n${spname} on CLI")"
        done
    fi
    tars_in_dir="$(file -L "${spdir}"/*  | grep tar |cut -d ':' -f1)"
    sps_in_tar_count=0
    if [[ -n $tars_in_dir ]]; then
        for tarfile in ${tars_in_dir};
        do
            sps_in_tar="$(tar tf $tarfile | grep -h -E "^$sp_regex")"
            if [[ -n $sps_in_tar ]]; then
                tarname="$(basename $tarfile)"
                sps_in_current_tar_count=$(echo "$sps_in_tar" | wc -l)
                sps_in_tar_count=$(($sps_in_tar_count + $sps_in_current_tar_count))
                sps_inline="$(echo "$sps_in_tar" | paste -s -d',' | xargs)"
                sps_output="$(echo -e "${sps_output}\n${sps_inline} in ${tarname}")"
            fi
        done
    fi
    sps_output="$(echo "$sps_output" | xargs -L1 | cat -n | xargs -L1 echo "                ")"
    sps_count=$(($sps_in_tar_count + $sps_in_dir_count))
    if [[ $sps_count > 1 ]]; then
        echo -e "Please pick one of the following:"
        echo "$sps_output"
        return $sps_count
    fi
    return 0
}

function hide_sp_rpms(){
    disable_err
    sp_dir="$1"
    input_file="$2"
    if [[ ! -d "$sp_dir" ]]; then
        echo "$sp_dir doesn't exist"
        return
    fi
    if [[ ! -f "$input_file" ]]; then
        echo "$input_file doesn't exist"
        return
    fi
    sp_regex="$(get_sp_regex)"
    input_sps="$(cat "${input_file}" | grep -E "$sp_regex")"
    if [[ -z $input_sps ]]; then
        enable_err
        return
    fi
    sps="$(find -L "$sp_dir" -maxdepth 1 -type f 2>/dev/null | grep -F "${input_sps}")"
    sp_rpms="$(echo "${sps}" | xargs cat | grep "Boxname:" | cut -d":" -f2 | xargs -L1)"
    input_filtered="$(echo "${sp_rpms}" | grep -v -F -f - ${input_file})"
    if [[ -z $input_filtered ]]; then
        enable_err
        return
    fi
    echo "$input_filtered" >| "${input_file}"
    enable_err
}


#Check if tar file has SP

function is_sp_specified_in_tar() {
    disable_err
    sp_regex="$(get_sp_regex)"
    spdir="$1"
    tars_in_dir="$(file -L "${spdir}"/*  | grep tar |cut -d ':' -f1)"
    retval="False"

    if [[ -n $tars_in_dir ]]; then
        for tarfile in ${tars_in_dir};
        do
            sps_in_tar="$(tar tf $tarfile | grep -h -E "^$sp_regex")"
            if [[ -n $sps_in_tar ]]; then
                retval="True";
            fi
        done
    fi
    echo $retval;
    enable_err
}

#Check if add id dir has SP

function add_id_dir_has_sp() {
    disable_err
    search_dir="/var/log/install/instdb/add_ids"

    sp_regex="$(get_sp_regex)"

    for entry in $search_dir/*
    do
       if [[ -n $entry ]]; then
        filename="$(basename $entry)"
        if [[ "$filename" =~ $sp_regex ]]; then
            echo "$filename"
            enable_err
            return 0
        fi
       fi
    done
    echo "False"
    enable_err
    return 1
}

function get_sp_name_from_tar() {
    disable_err
    sp_regex="$(get_sp_regex)"
    spdir="$1"
    tar_in_dir="$(file -L "${spdir}"/*  | grep tar |cut -d ':' -f1)"

    if [[ -n $tar_in_dir ]]; then
        sp_in_tar="$(tar tf $tar_in_dir | grep -h -E "^$sp_regex")"
        if [[ -n $sp_in_tar ]]; then
            sp_name="$(basename $sp_in_tar .iso)"
            echo "$sp_name"
            enable_err
            return 0
        fi
    fi
    enable_err
    return 1
}

# check existence of SP symlinks
# if link exists then return 1, else 0 
# If repo itself doesnt exist, it is not an RP or CC card, return 0

function verify_sp_symlinks ()
{
    disable_err
    local fname=""
    local lfile=""
    local repo_path="/install_repo/gl/"

    echo "****Function verify_sp_symlinks****" >> /tmp/sp_link_verify_log
    if [ -d $repo_path ]; then
        for fname in `find $repo_path -type l`; do
            if is_file_name_sp $fname >/dev/null; then
                lfile=`readlink $fname 2>&1` 
                echo "$fname is a link to $lfile" >> /tmp/sp_link_verify_log
                if [ ! -f $lfile ]; then
                    echo $lfile
                    enable_err
                    return 1
                fi
            fi
        done
    fi
    rm -rf /tmp/sp_link_verify_log 2> /dev/null
    enable_err
    return 0
}

function verify_sp_rpms ()
{
    disable_err
    local sp_path="/misc/disk1/tftpboot/"
    local repo_dir="/install_repo/gl/"
    local rpm_list=""
    for fname in `find $sp_path`; do
        if is_file_name_sp $fname >/dev/null; then
           sp_rpms=`echo $fname | xargs grep -H "Boxname:" | cut -d':' -f3 | sort -u | xargs -L1` 
           if [-z "$sp_rpms" ]; then
               echo $fname
               enable_err
               return 1
           fi
           rpm_list+=$sp_rpms
           rpm_list+=" "
        fi
    done
    for rpm_name in `echo $rpm_list | sort | uniq`; do
        res=`find $repo_dir -name $rpm_name` 
        if [ -z "$res" ]; then
            echo $rpm_name
            enable_err
            return 1
        fi
    done
    enable_err
    return 0
}

function get_sp_name_if_sp_rpm ()
{
    disable_err
    input_file=$1 
    local sp_path="/misc/disk1/tftpboot/"
    if [ -d $sp_path ]; then
        for fname in `find $sp_path | sort -r`; do
            if is_file_name_sp $fname >/dev/null; then
               res=`grep $input_file $fname`
               if [ ! -z "$res" ]; then
                   echo `basename $fname`
                   enable_err
                   return 0
               fi
            fi
        done
    fi
    enable_err
    return 0 
} 
