#! /bin/bash
# ----------------------------------------------------------------------
# attestation_ima_script -- Fetch IMA logs script
#
# May, 2018, Saiprasad Muchala
#
# Copyright (c) 2018 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------------------------

# Extract the options passed by the parser module
# -l: location of node from which IMA logs needs to be copied
# -b: Device to which IMA logs to be copied
# -v: filename with which IMA logs will be copied

#STDOUT is passed as a PIPE if getting executed from the CLI. If STDOUT is a
#pipe, filesys_cli executable get_line function throwing error. Hence closing 
#the STDOUT and assigning it to current tty fd.
exec 1<&-;
exec 1<>`tty`;

if [ $# -lt 2 ]
then
    echo "Usage: $0 -l <source location> -b <Destination Device location> -v <Destination File name>"
    exit
fi

location="";
IMA_LOG="/sys/kernel/security/ima/ascii_runtime_measurements";

#If user gives to_file name, finally move the tar created to the specified name
while getopts l:b:v: ch
do      case "$ch" in
        l)      location="$OPTARG";;
        b)      to_device="$OPTARG";;
        v)      to_file="$OPTARG";;
        esac
done

if [ -z "$to_device" ]
then
    echo "Destination Device location is mandatory"
    echo "Usage: $0 -l <source location> -b <Destination Device location> -v <Destination File name>"
    exit
fi

if [ ! -z "$to_file" ] && [ $to_file != "-" ] && [[ $to_file != *tar.gz* ]]; then
    echo "Destination file should be specified in tar.gz format";
    exit
fi

if [ ! -f $IMA_LOG ]; then
    echo "IMA not enabled on the platform";
    exit
fi

script_working_directory=$PWD;
timestamp=`date '+%Y_%m_%d__%H_%M_%S'`;
unique_folder=$$_$timestamp;
tmp_ima_device=disk0:;
tmp_ima_directory=/$tmp_ima_device/tmp_$unique_folder;
tmp_ima_file_dir=tmp_$unique_folder;

if [ -z "$to_file" ] || [ $to_file == "-" ]; then
    to_file=ascii_ima_log_$timestamp.tar.gz;
fi

## Get the ip address of node
get_node_ip()
{
    node=$1
    ip_addr=`nodename_to_ipaddress -n $node 2>&1`

    if echo $ip_addr |grep -q ".\+\..\+\..\+\..\+"; then
        echo $ip_addr ; 
    else echo "Could not get ip address of $node"
        exit
    fi
}

# This function changes the current working directory to tmp/unique_folder directory
# and creates the tar file with all ima logs copied on to that location and
# copies the resultant tar file to requested location on the router.

create_ima_tar_file () {
    cd $tmp_ima_directory;
    tar -czf $to_file ascii_ima_log_*;
    cd $script_working_directory;
}

# This function will get IMA logs from the current node. 
get_local_node_ima_file () {
    local_node=`echo "$1" | sed -r 's/[/]+/_/g'`;
    cp $IMA_LOG $tmp_ima_directory/ascii_ima_log_$local_node;
}

# This function will get IMA lo
get_remote_node_ima_file () {
    ip=`get_node_ip $1`;
    ssh -o UserKnownHostsFile=/dev/null -o LogLevel=quiet -o                  \
        StrictHostKeychecking=no $ip bash -c "' \
        mkdir /tmp/$unique_folder; \
        cp $IMA_LOG /tmp/$unique_folder/ascii_runtime_measurements;'";
    local_node=`echo "$1" | sed -r 's/[/]+/_/g'`;
    out=`scp -o UserKnownHostsFile=/dev/null -o LogLevel=quiet -o               \
StrictHostKeychecking=no $ip:/tmp/$unique_folder/ascii_runtime_measurements        \
/tmp/$unique_folder/ascii_ima_log_$local_node`;
    ssh -o UserKnownHostsFile=/dev/null -o LogLevel=quiet -o                   \
        StrictHostKeychecking=no $ip bash -c "' \
        rm -rf /tmp/$unique_folder;'";
    cp /tmp/$unique_folder/ascii_ima_log_$local_node $tmp_ima_directory/;
}

#This function will copy IMA logs from one node
get_ima_log_fm_node ()
{
    local_node_orig=`node_list_generation -f MY`;
    if [ $local_node_orig == $1 ]; then
        get_local_node_ima_file $1;
    else
        get_remote_node_ima_file $1;
    fi
}

get_ima_log_all_nodes ()
{
    all_nodes=`node_list_generation -f ALL 2>&1`;
    lp_itr=0;
    for name in $all_nodes;
    do
        name=`echo $name | sed -r 's/[_]+/\//g'`;
        node_name[$lp_itr]=${name:4};
        lp_itr=$[$lp_itr+1];
    done
    number_of_nodes=$lp_itr;

    #run command on each node
    lp_itr=0; while [ $lp_itr -lt $number_of_nodes ]; do
        get_ima_log_fm_node ${node_name[$lp_itr]};
        lp_itr=$[$lp_itr+1];
    done
}

#create the tmp directory to copy the IMA logs
mkdir $tmp_ima_directory;

if [ -z "$location" ] 
then
#if location absent, fetch the current node IMA logs only.
{
    local_node_orig=`node_list_generation -f MY`;
    get_local_node_ima_file $local_node_orig
}
#if location is all, fetch the IMA logs from all locations.
elif [ "$location" == "all" ]
then
{
    mkdir /tmp/$unique_folder;
    get_ima_log_all_nodes;
    rm -rf /tmp/$unique_folder;
}
else
{
    mkdir /tmp/$unique_folder;
    get_ima_log_fm_node $location;
    rm -rf /tmp/$unique_folder;
}
fi

create_ima_tar_file;
filesys_cli copy -a $tmp_ima_device -u $tmp_ima_file_dir/$to_file -b $to_device -v $to_file;
rm -rf $tmp_ima_directory;

