#! python

import sys
sys.dont_write_bytecode = True
import re
import subprocess
import os
import json
import signal

def run_cmd (cmd, logger = None):
    process = subprocess.Popen(cmd, stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE, shell=True)
    out, error = process.communicate()
    sprc = process.returncode
    if sprc is None or sprc != 0:
        out = error
        raise RuntimeError("Error CMD=%s returned --->%s" % (cmd, out))
    else:
        if logger:
            logger.debug ("CMD=%s OUTPUT=%s"%(cmd, out))
        pass
    return out.strip()

def check_if_sp(file_name):
    """This function returns True if the file_name is service pack else False"""
    m = re.search(r'(.*-sp\d+-.*)-(\d+.\d+.\d+.*)\.iso', file_name)
    if m:
        return(True)
    else:
        m = re.search(r'(.*-sp\d+-.*)-(\d+.\d+.\d+.*)', file_name)
        if m:
            return(True)
    return(False)

def get_from_version ():
    """
    Get current base software version running
    """
    version_re = re.compile(r'.*version=(?P<version>\S+).*[Boot image]')
    cmd = "sdr_instcmd show install active"
    output = run_cmd(cmd)

    version = version_re.search(output)
    if version:
        version = version.group('version')
    if not version:
        logger.error("Failed to get SW version : %s" % (output))
        exit_with_errors()
    return version

def check_platform_supports_optim_wf (platform):
    allow_optim_workflows_plat = [ "ncs5500", "asr9k" ]
    if platform not in allow_optim_workflows_plat:
        return False
    else:
        return True

def get_to_version (iso_name):
    '''
        Get version of iso.
    '''
    m = re.search(r'(.*/)*(.*)-(\d+.\d+.\d+.\d+\w*)-(\w*)', iso_name)
    if not m:
        m = re.search(r'(.*/)*(.*)-(\d+.\d+.\d+\w*)-(\w*)', iso_name)
    if not m:
        m = re.search(r'(.*/)*(.*)-(\d+.\d+.\d+.\w*)', iso_name)
    if not m:
        m = re.search(r'(.*/)*(.*)-(\d+.\d+.\d+\w*)', iso_name)
    if m:
        logger.debug("INFO: version = %s" %(m.groups()[2]))
        return m.groups()[2]
    return None 

def decode_progress (mdata_dict):
    ret_str = ''
    vm_dict = {}
    mdata_vms = {}

    with open (VM_INFO_FILE, 'r') as fd:
        mdata_vms = json.loads (fd.read())

    for key, value in mdata_vms.iteritems ():  
        if value.has_key ('sysadmin'):
            vm_dict[value['sysadmin'][0]] = key  

    if mdata_dict:
        table_str = ''
        op_id = mdata_dict["OID"]
        ret_str += "\n\nUpdates from sysadmin for this operation.\n"
        ret_str += "Install prepare operation %s is in progress\n"%(op_id)
        if mdata_dict ['Updates']:
            ret_str += '\n'.join (mdata_dict['Updates'])
        
        if mdata_dict['Completed'] or mdata_dict['Progress'] or mdata_dict['Abort']:
            table_str += "\n{:<25} {:<25} {:<25}".format('Node(s)','State','Action')

        if mdata_dict['Completed']:
            for node in mdata_dict["Completed"]:
                if vm_dict.has_key (node):
                    card = vm_dict[node]
                else:
                    card = node
                table_str += "\n{:<25} {:<25} {:<25}".format(card, "Completed", "Partition preparation completed")
    
        if mdata_dict['Progress']:
            for node in mdata_dict["Progress"]:
                if vm_dict.has_key (node):
                    card = vm_dict[node]
                else:
                    card = node
                table_str += "\n{:<25} {:<25} {:<25}".format(card, "In Progress", "Partition preparation in progress")
    
        if mdata_dict['Abort']:
            for node in mdata_dict["Abort"]:
                if vm_dict.has_key (node):
                    card = vm_dict[node]
                else:
                    card = node
                table_str += "\n{:<25} {:<25} {:<25}".format(card, "Abort", "Partition preparation aborted")

        ret_str += table_str

    return ret_str

def is_cisco_rpm (rpm, platform):
    return ((platform in rpm)) 

def is_same_release_rpm (pkg, to_version):
    release = ''
    rpm = str(pkg)
    rel_to_match = 'r' + ''.join(to_version.split('.'))
    rpm = rpm.replace ('.x86_64', '')
    rpm = rpm.replace ('.arm', '')
    import re
    # Release could have DDTS ID, etc. split on '.' and extract 0th field.
    mre = re.search(r'(.*)-(.*)-(.*)', rpm)
    if mre:
        release = mre.groups()[2].split('.')[0]
    if release == rel_to_match:
        return True
    return False    

def sanitize_input_list (pkglist, to_version, platform):
    sanitized_list = []
    skipped_list = []
    # Only worry about Cisco rpms.
    cisco_pkgs = filter (lambda x: is_cisco_rpm (x, platform), pkglist)
    for pkg in cisco_pkgs:
        if '-mini-' in pkg or '-golden' in pkg:
            sanitized_list.append (pkg)
            continue
        if is_same_release_rpm (pkg, to_version):
            sanitized_list.append (pkg)
        else:
            skipped_list.append(pkg)
    return sanitized_list, skipped_list 

def signal_handler(signal, frame):
    sys.exit(0)

if __name__ == '__main__':
    mini_in_input = False
    optim_disable = False
    plat_name = ''
    iso = None
    pkglist = []
    signal.signal(signal.SIGINT, signal_handler)

    logfile = '/var/log/install/instcmd_cli_hook.log'
    os.environ['MP_LOG_FILE'] = logfile
    if 'pkg' in sys.argv:
        for idx, pkg in enumerate(sys.argv[5:]):
            if '*' in pkg:
                sys.argv[idx+5] = "\'%s\'"%(pkg)

    try:
        from logger_init import get_logger
        import logging
        from constants import *
    except:
        cmd = "sdr_instcmd %s"%(' '.join(sys.argv[1:]))
        subprocess.call (cmd, shell=True)
        sys.exit (0)
    
    logger = get_logger(logfile)

    logger.debug ('Input arguments %s'%(','.join(sys.argv)))
        
    # Check platform to see if it supports optim workflows.
    try:
        cmd = "/pkg/bin/install-functions.py get_platform_name"
        plat_name = run_cmd (cmd)
        if not check_platform_supports_optim_wf (plat_name):
            cmd = "sdr_instcmd %s"%(' '.join(sys.argv[1:]))
            logger.debug ("Platform doesnt support optim workflow. Call %s"%(cmd))
            try:
                subprocess.call (cmd, shell=True)
            except:
                pass
            sys.exit (0)
    except SystemExit as e:
        if e.code != 0:
            logger.debug ("Was unable to retrieve platform support for optim " \
                          "workflow. Exit with -1")
            sys.exit (-1)
        if e.code == 0:
            sys.exit (0)

    if sys.argv[1] == "show":
        cmd = "sdr_instcmd %s"%(' '.join(sys.argv[1:]))
        if sys.argv[3] == "request":
            status_request_str = ''
            if os.path.isfile (UPDATE_STATUS_FILE_XR):
                with open (UPDATE_STATUS_FILE_XR) as fd:
                    status_request_str = fd.read()
                ''' 
                Optimised parallel prepare in progress.
                '''
                if os.path.isfile (UPDATE_STATUS_FILE):
                    cmd = "cat %s"%(UPDATE_STATUS_FILE)
                    try:
                        data = run_cmd (cmd)
                        data = data.replace ("'", "\"")
                        mdata_dict = json.loads(data)
                        status_request_str += decode_progress (mdata_dict)
                    except:
                        pass
                print (status_request_str)
                sys.exit (0)
        subprocess.call (cmd, shell=True)
        sys.exit (0)

    if 'nooptim' in sys.argv:
        sys.argv.remove ('nooptim')
        cmd = "sdr_instcmd %s"%(' '.join(sys.argv[1:]))
        logger.debug ('No optim detected. Call %s'%(cmd))
        subprocess.call (cmd, shell=True)
        sys.exit (0)

    if len(sys.argv) <=3:
	if 'activate' in sys.argv and 'optim' not in sys.argv:    
            cmd = "sdr_instcmd %s"%(' '.join(sys.argv[1:]))
            # If install activate is triggered without any other parameters, 
            # peek into xr prepare checkpoint file to see if optim needs to be set.
            file2check = "/var/log/install/instdb/prepare_chkpt"
            if os.path.isfile (file2check):
                with open (file2check, 'r') as fp:
                    buff = fp.read()
                    system_op = buff[buff.find("system_op"):].split()[1]
                    base_iso = buff[buff.find("base_iso"):].split()[1]
                    if system_op == '1' and ('-mini-' in base_iso or '-golden' in base_iso):
                        cmd = "sdr_instcmd %s optim"%(' '.join(sys.argv[1:]))
        else:
            cmd = "sdr_instcmd %s"%(' '.join(sys.argv[1:]))
        logger.debug ('No packages in list. Call %s'%(cmd))
        subprocess.call (cmd, shell=True)
        if 'prepare' in sys.argv and 'clean' in sys.argv:
	    file2remove = "/var/log/install/instdb/prepare_chkpt"
            file2removeadmin = "/install_repo/gl/instdb/instmgr_prep_chkpt"
	    if os.path.isfile (file2remove):
		#logger.info ("Prepare may have been done in optimized mode.")
		#logger.info ("Ignore any abort faced as part of this prepare clean operation")
	        os.remove (file2remove)
                cmd = "/pkg/bin/install_exec_sysadmin \"rm -f %s\""%(file2removeadmin)
                try:
                    run_cmd (cmd)
                except:
                    pass
        sys.exit (0)

    ctrl_argv = int(sys.argv[4], 16)
    if ctrl_argv:
        cmd = "sdr_instcmd %s"%(' '.join(sys.argv[1:]))
        logger.debug ('Install command asoociated with options. Call %s'%(cmd))
        subprocess.call (cmd, shell=True)
        sys.exit (0)

    if 'id' in sys.argv:
        try:
            add_id_dir = '/var/log/install/instdb/add_ids'
            id_files_dir = [os.path.join (add_id_dir, x) for x in os.listdir (add_id_dir) if 
                                os.path.isfile (os.path.join (add_id_dir, x))]
            for idx, idnum in enumerate(sys.argv[5:]):
                if not idnum.isdigit():
                    break
                id_file_considered = os.path.join (add_id_dir, 'add_id.%s'%(idnum))
                if id_file_considered not in id_files_dir:
                    logger.debug ('Invalid add operation id: %s'%(idnum))
                    raise
                with open (id_file_considered, 'r') as fd:
                    for data in fd.readlines():
                        pkg = data.split()[0]
                        if '-mini-' in pkg or '-golden' in pkg:
                            if not mini_in_input:
                                mini_in_input = True
                                iso = pkg
                            else:
                                print 'Multiple ISOs in input parsed as part of id %s'%(idnum)
                                sys.exit (0)
                        if pkg not in pkglist:
                            pkglist.append (pkg)
            if not pkglist:
                raise
        except:
            cmd = "sdr_instcmd %s"%(' '.join(sys.argv[1:])) 
            logger.debug ("Issue while parsing id files. Go with %s"%(cmd))
            subprocess.call (cmd, shell=True)
            sys.exit (0)

    if 'pkg' in sys.argv:
        for idx, pkg in enumerate(sys.argv[5:]):
            if '*' in pkg:
                optim_disable = True
                break
            if '-mini-' in pkg or '-golden' in pkg:
                mini_in_input = True
                iso = pkg
            if pkg not in pkglist:
                pkglist.append (pkg)

    for pkg in pkglist:
        if check_if_sp (pkg):
            optim_disable = True

    try:
        if not mini_in_input or optim_disable:
            cmd = "sdr_instcmd %s"%(' '.join(sys.argv[1:]))
            logger.debug ('No mini in input or wildcard in input. Call %s'%(cmd))
            subprocess.call (cmd, shell=True)
            sys.exit (0)
        plat_name = iso.split('-')[0]
        if not check_platform_supports_optim_wf (plat_name):
            cmd = "sdr_instcmd %s"%(' '.join(sys.argv[1:]))
            logger.debug ('Platform doesnt support optim workflow. Call %s'%(cmd))
            subprocess.call (cmd, shell=True)
            sys.exit (0)
        from_version = get_from_version ()
        to_version = get_to_version (iso)

        if from_version == to_version:
            if 'replace' not in sys.argv:
                cmd = "sdr_instcmd %s"%(' '.join(sys.argv[1:]))
                logger.debug ('V2 version same as running version without replace. ' 
                                'Go ahead with %s'%(cmd))
                subprocess.call (cmd, shell=True)
                sys.exit (0)

        '''
            Create command to call install script with appropriate inputs.
            Sanitize pkglist for version matching to_version.
        '''
        pkglist, skipped_pkglist = sanitize_input_list (pkglist, to_version, plat_name)

        if skipped_pkglist:
            print ("Error: Mixed release upgrade not supported.\n"
                                "Following packages are not compatible with "
                                "ISO version %s given as input for system upgrade: \n\t%s"
                                %(to_version, '\n\t'.join (skipped_pkglist)))
            print ("Package list retrieved was: \n%s"%('\n'.join(pkglist)))
            sys.exit (-1)

        logger.debug ("Package List to proceed with system upgrade: \n%s"%('\n'.join(pkglist)))
        run_background = False
        cmd_install = ['install']
        if 'replace' in sys.argv:
            cmd_install.append ("replace")
        if 'prepare' in sys.argv:
            cmd_install.append ('prepoptim')
            run_background = True
        if 'activate' in sys.argv:
            cmd_install.append ('actioptim')
        cmd_install.extend (pkglist)
        if 'noprompt' in sys.argv:
            cmd_install.append ("noprompt")
            run_background = True
        if 'synchronous' in sys.argv:
            cmd_install.append ("synchronous")
            run_background = False
        cmd = ' '.join (cmd_install)
        logger.debug ("Command executed %s"%(cmd))
        if run_background:
            cmd_install.append ("background")
            user = 'root'
            oid = -1
            try:
                import getpass
                user  = getpass.getuser()
                INSTALL_DB_DIR = '/var/log/install/instdb/'
                op_id_file = INSTALL_DB_DIR + 'operation_id.txt'
                oidfile = open(op_id_file, 'r')
                oid = int(oidfile.read()) + 1
                oidfile.close()
            except:
                oid = 1
                pass
            formatter_console = logging.Formatter('%(asctime)s %(message)s',"%b %d %H:%M:%S")
            ch = logging.StreamHandler(sys.stdout)
            ch.setLevel(logging.INFO)
            ch.setFormatter(formatter_console)
            logger.addHandler(ch)
            print_list = sys.argv[1:]
            print_list.remove ('0x0')
            logger.info ("Install operation %d started by %s: \n  %s"%(oid, 
                                                        user, ' '.join(print_list)))
            logger.info ("Package list:")
            for pkg in pkglist:
                logger.info ("\t%s"%(pkg))
            logger.info ("Install operation will continue in the background")
            subprocess.Popen (cmd_install, cwd='/', stdout = sys.stdout, 
                            stderr = sys.stdout, preexec_fn=os.setpgrp)
        else:
            subprocess.call (cmd, shell=True)
    except:
        import traceback
        exc_info = sys.exc_info()
        TB = traceback.format_exc()
        logger.debug (TB)
