#!/usr/bin/python

import re
import sys
import os
import time
import argparse
import commands
import shutil
import glob
import logger
import errno
import traceback
import subprocess
import datetime
from logger_init import get_logger
import logging

VALIDATION = True
lead_xr_vm = None
rps_cal_vms = []
sp_rpms = []
platform = None
platform_prefix = "/opt/cisco/calvados/bin/vrfch.sh CTRL_VRF"
log = get_logger("/var/log/install/inst_update.log")

class LocalError(Exception):
    def __init__(self,msg):
        self.msg = msg

class CalError(Exception):
    def __init__(self,msg):
        self.msg = msg

def run_cmd (cmd):
    process = subprocess.Popen(cmd, stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE, shell=True)
    out, error = process.communicate()
    sprc = process.returncode
    if sprc is None or sprc != 0:
        out = error
        raise RuntimeError("Error CMD=%s returned --->%s" % (cmd, out))
    else:
        log.debug ("CMD=%s OUTPUT=%s"%(cmd, out))
        pass
    return out.strip()

def create_dirs(dir_path):
    try:
        os.makedirs(dir_path)
    except OSError, e :
        raise(LocalError("Failed to create dir : %s : %s" % (dir_path, os.strerror(e.errno))))

def mount_iso(bundle_iso, tmp_mount_path):
    if os.path.exists(tmp_mount_path):
        os.system('umount %s' % (tmp_mount_path))
        shutil.rmtree(tmp_mount_path)
    create_dirs(tmp_mount_path)
    cmd = 'mount -o loop %s %s' % (bundle_iso, tmp_mount_path)
    log.debug(cmd)
    status,output = commands.getstatusoutput(cmd)
    if status :
        log.debug("Cmd:%s\nOutput:%s"%(cmd,output))
        raise(LocalError("Failed to mount %s at %s" % (bundle_iso,tmp_mount_path)))
        
def umount_iso(tmp_mount_path):
    ''' Unmount and remove given path '''

    if os.path.exists(tmp_mount_path):
        # In case previous umount failed try again
        try:
            cmd = 'umount %s' % (tmp_mount_path)
            status,output = commands.getstatusoutput(cmd)
            if status :
                log.debug("Cmd:%s\nOutput:%s"%(cmd,output))
            shutil.rmtree(tmp_mount_path)
        except:
            pass

def check_if_sp(file_name):
    """This function returns True if the file_name is service pack else False"""
    m = re.search(r'(.*-sp\d+-.*)-(\d+.\d+.\d+.*)\.iso', file_name)
    if m:
        return(True)
    else:
        return(False)

"""
Class Name: collect_vm_info
This is used to collect info. about all CAL/ XR
VMs and as well as store XR VM IP where adr_instmgr is running
"""
class collect_vms_info():
    def __init__(self):
        self.xr_vm_ip = []
        self.lead_xr_vm = ""
        self.collect_sysadmin_vm_info()

    def set_sdr_instmgr_xrvm_for_sdr(self, xr_vm_ip):
        global lead_xr_vm
        cmd = "ssh %s /pkg/bin/placed_show sdr_instmgr | grep \"Role: REDUNDANCY_ACTIVE\"" % (
            xr_vm_ip)
        log.debug(cmd)
        cmd_op = commands.getoutput(cmd)
        log.debug(cmd_op)
        if cmd_op:
            if "REDUNDANCY_ACTIVE" in cmd_op:
                lead_xr_vm = xr_vm_ip
                self.lead_xr_vm = xr_vm_ip

    def collect_sysadmin_vm_info(self):
        """This function is used to collect the Cals VM Info."""
        global rps_cal_vms
        global log
        log.debug("Collecting Sysadmin VMs Information")

        cmd1 = "/pkg/bin/install_exec_sysadmin \"source /opt/cisco/calvados/bin/install-functions.sh ; %s /opt/cisco/calvados/bin/show_cmd 'show vm'  \"" % (platform_prefix)
        for i in range(0,3):
            status, cmd = commands.getstatusoutput(cmd1)
            if status:
                log.debug ("Show vm failed. Retry count %d"%(i))
                continue
            elif "application timeout" in cmd:
                log.debug ("Show vm failed with application timeout. Retry count %d"%(i))
                continue
            else:
                log.debug(cmd)
                break
        if not cmd:
            raise(LocalError("Failed to fetch VM information.")) 
        else:
            for line in cmd.split(os.linesep):
                if "Location" in line:
                    location = line.split()[1]
                if "running" in line:
                    vm_name = line.split()[0]
                    if "sysadmin" in vm_name:
                        vm_ip = re.findall(r'[0-9]+(?:\.[0-9]+){3}', line)
                        if ("RP" in location or "RSP" in location or "CB" in location):
                            rps_cal_vms.append(vm_ip[0])
                    elif ("RP" in location or "RSP" in location) and not lead_xr_vm:
                        # if SDR inst mgr runnig here set as lead XR vm ?
                        # This is node from where packages will be distributed
                        vm_ip = re.findall(r'[0-9]+(?:\.[0-9]+){3}', line)
                        self.xr_vm_ip.append(vm_ip[0])
                        self.set_sdr_instmgr_xrvm_for_sdr(vm_ip[0])
        if not lead_xr_vm:
            raise(LocalError("Failed to get IP for lead XR VM."))  


def validate_sp(mount_path):
    cmd = '/pkg/bin/install-functions.py validate_sp_rpms %s' % (mount_path)
    status,output = commands.getstatusoutput(cmd)
    if output:
        log.debug("validate_sp_rpms :%s" % (output))
        raise(LocalError("SP Validation failed %s" % (output)))

def validate_iso(package):
    if "-mini-" not in package and "-goldenk9-" not in package  \
            and "-golden-" not in package:
        return
    # Since ISO is validate in install update , skip validation here
    tmp_mnt_path = os.path.join("/tmp", "mnt"+os.path.basename(package))
    if os.path.exists(tmp_mnt_path):
        # If leftover from previous execution
        umount_iso(tmp_mnt_path)

    mount_iso(package, tmp_mnt_path)
    iso_info = os.path.join(tmp_mnt_path, "iso_info.txt")
    md5_origin_cmd = "cat %s | grep initrd " % (iso_info)
    md5sum_orig = commands.getoutput(md5_origin_cmd).split(" ")[-1]
    inird_img = os.path.join(tmp_mnt_path, "boot/initrd.img")
    md5_cmd = "md5sum %s" % (inird_img)
    md5sum_new = commands.getoutput(md5_cmd).split(" ")[0].strip()
    umount_iso(tmp_mnt_path)
    if not md5sum_new == md5sum_orig:
        raise(LocalError("MD5 verification failed for: %s" % (os.path.basename(package))))

def validate_packages(pkg):
    global VALIDATION
    if pkg.endswith('.rpm'):
        if not os.path.exists(pkg):
            raise(LocalError("Package %s not found for verification" %(pkg)))
        cmd = "rpm -Kv %s|grep -i signature | grep  OK " % (pkg)
        status, output = commands.getstatusoutput(cmd)
        if status:
            raise(LocalError("Signature verification failed for RPM : %s " %(pkg)))
    elif pkg.endswith('.iso'):
        return validate_iso(pkg)
    elif '.iso' in pkg:
        return validate_iso(pkg)
    else:
        raise(LocalError("Unknown file type which is not supported : %s " %(pkg)))
    return
 
def create_repo(pargs):
    global sp_rpms
    xr_repo = os.path.join(pargs.repo, "install_repo/gl/xr")
    cal_repo = os.path.join(pargs.repo, "install_repo/gl/calvados")
    tftp_repo = os.path.join(pargs.repo, "misc/disk1/tftpboot")
    pkgs_n_repos = {xr_repo: [], cal_repo: [], tftp_repo: []}
    pkgs_n_repos['BASE_REPO'] = pargs.repo
    for package in set(pargs.packages):
        src = os.path.join(pargs.repo, package)
        validate_packages(src)
        new_name = os.path.splitext(package)[0]
        if '-sysadmin-' in package or '.host.' in package or '.admin.' in package:
            if not os.path.exists(cal_repo):
                create_dirs(cal_repo)
            dest = os.path.join(cal_repo, new_name)
            os.rename(src, dest)
            pkgs_n_repos[cal_repo].append(new_name)

        elif ("-mini-" in package or "-golden-" in package or
              "-goldenk9-" in package) and ".iso" in package:
            # add to /misc/disk1/tftpboot
            if not os.path.exists(tftp_repo):
                create_dirs(tftp_repo)
            # Check if the package naming corresponds to old format.
            # Add accordingly to the repo.
            if 'golden' in package and not package.endswith('.iso'):
                m = re.search(r'(.*/)*(.*)\.iso-(\d+.\d+.\d+.\d+\w*)\.(.+)', package)
                if not m:
                    m = re.search(r'(.*/)*(.*)\.iso-(\d+.\d+.\d+\w*)\.(.+)', package)
                if m:
                    path, name, version, label = m.groups()
                    new_name = "%s-%s-%s"%(name, version, label)

            dest = os.path.join(tftp_repo, new_name)
            os.rename(src, dest)
            pkgs_n_repos[tftp_repo].append(new_name)

        elif check_if_sp(package):
            # Its SP ,
            cmd = 'source /pkg/bin/install-functions.sh; are_multiple_sps_specified %s' % (pargs.repo)
            status,output = commands.getstatusoutput(cmd)
            if status:
                raise(LocalError("Multiple SP's specified in input\n %s" % (output)))
            tmp_mnt_path = os.path.join(pargs.repo, "mnt"+package.strip())
            mount_iso(os.path.join(pargs.repo, package), tmp_mnt_path)
            validate_sp(tmp_mnt_path)
            # Move sp_info.txt
            src = os.path.join(tmp_mnt_path, "sp_info.txt")
            dest = os.path.join(tftp_repo, new_name)
            if not os.path.exists(os.path.dirname(dest)):
                create_dirs(os.path.dirname(dest))
            os.system("cp %s %s" % (src, dest))
            pkgs_n_repos[tftp_repo].append(new_name)

            # Get rpms in in SP folder
            cal_sp_rpms = os.path.join(tmp_mnt_path, 'calvados_rpms')
            xr_sp_rpms = os.path.join(tmp_mnt_path, 'xr_rpms')
            host_sp_rpms = os.path.join(tmp_mnt_path, 'host_rpms')
            if os.path.exists(cal_sp_rpms):
                # Read all RPMs and copy with renamed without extention
                cal_sp_rpm_files = glob.glob("%s/*.rpm" % (cal_sp_rpms))
                if not os.path.exists(cal_repo):
                    create_dirs(cal_repo)
                for a in cal_sp_rpm_files:
                    new_name = os.path.splitext(os.path.basename(a))[0]
                    dest = os.path.join(cal_repo, new_name)
                    os.system("cp %s %s" % (a, dest))
                    sp_rpms.append(os.path.basename(dest))
                    pkgs_n_repos[cal_repo].append(new_name)
            if os.path.exists(xr_sp_rpms):
                # Read all RPMs and copy with renamed without extention
                xr_sp_rpm_files = glob.glob("%s/*.rpm" % (xr_sp_rpms))
                if not os.path.exists(xr_repo):
                    create_dirs(xr_repo)
                for b in xr_sp_rpm_files:
                    new_name = os.path.splitext(os.path.basename(b))[0]
                    dest = os.path.join(xr_repo, new_name)
                    os.system("cp %s %s" % (b, dest))
                    sp_rpms.append(os.path.basename(dest))
                    pkgs_n_repos[xr_repo].append(new_name)
            if os.path.exists(host_sp_rpms):
                # Read all RPMs and copy with renamed without extention
                host_sp_rpm_files = glob.glob("%s/*.rpm" % (host_sp_rpms))
                if not os.path.exists(cal_repo):
                    create_dirs(cal_repo)
                for c in host_sp_rpm_files:
                    new_name = os.path.splitext(os.path.basename(c))[0]
                    dest = os.path.join(cal_repo, new_name)
                    os.system("cp %s %s" % (c, dest))
                    sp_rpms.append(os.path.basename(dest))
                    pkgs_n_repos[cal_repo].append(new_name)
            umount_iso(tmp_mnt_path)

        else:
            if not os.path.exists(xr_repo):
                create_dirs(xr_repo)
            dest = os.path.join(xr_repo, new_name)
            os.rename(src, dest)
            pkgs_n_repos[xr_repo].append(new_name)

    return pkgs_n_repos

def clean_sysadmin_repo(pkgs_n_repos,sdr_name):
    local_dir = pkgs_n_repos['BASE_REPO']
    for key in pkgs_n_repos.keys():
        if key == 'BASE_REPO':
            continue
        destp = key.replace(local_dir, "/")
        if pkgs_n_repos[key]:
            remote_cmd = "/opt/cisco/calvados/bin/utils/install_add_replicate.py \
            -c rm -s %s  -d %s -p %s -x %s -a %s -sdr %s"  % (key, destp.replace('//','/'), \
            ' '.join(pkgs_n_repos[key]), lead_xr_vm, ' '.join(rps_cal_vms), sdr_name)
            cmd = "/pkg/bin/install_exec_sysadmin \'%s\'" % (remote_cmd)
            status,output = commands.getstatusoutput(cmd)

def push_to_sysadmin(pkgs_n_repos,sdr_name):
    local_dir = pkgs_n_repos['BASE_REPO']
    for key in pkgs_n_repos.keys():
        if key == 'BASE_REPO':
            continue
        destp = key.replace(local_dir, "/")
        if pkgs_n_repos[key]:
            remote_cmd = "/opt/cisco/calvados/bin/utils/install_add_replicate.py \
            -c cp -s %s  -d %s -p %s -x %s -a %s -sdr %s" % (key, destp.replace('//','/'), \
            ' '.join(pkgs_n_repos[key]), lead_xr_vm, ' '.join(rps_cal_vms), sdr_name)
            cmd = "/pkg/bin/install_exec_sysadmin \'%s\'" % (remote_cmd)
            print(cmd)
            log.debug(cmd)
            try:
                out = run_cmd(cmd)
                if 'Error' in out:
                    raise(CalError("Encountered error in sysadmin during add operation."))
            except:
                exc_info = sys.exc_info()
                TB = traceback.format_exc()
                log.debug(TB)
                raise(CalError("Encountered error in sysadmin during add operation."))
    # Make the RP Sync directories dirty 
    try: 
        cmd = "/pkg/bin/install_exec_sysadmin \"ls /install_repo\" %d"%(2) 
        log.debug(cmd)
        run_cmd(cmd)
    except:
        log.debug ("Error encountered when marking repository dirty")
        raise(CalError("Error encountered when marking repository dirty"))

def create_add_id(add_id,repo):
    global sp_rpms
    op_id = 1
    repo_id = 1
    op_id_dir = "/var/log/install/instdb/add_ids"
    try:
        os.makedirs (op_id_dir)
    except OSError as e:
        if e.errno != errno.EEXIST:
            raise 
    if not add_id :
        # pick next ID from /var/log/install/instdb/operation_id.txt
        op_id_file = "/var/log/install/instdb/operation_id.txt"
        if not os.path.exists(op_id_file):
            add_id = 1
            os.system("echo 1 >> %s"%(op_id_file))
        else :
            fd = open(op_id_file,'r')
            add_id = fd.read()
            fd.close()
        pass 
    add_id_file = "/var/log/install/instdb/add_ids/add_id.%s"%(add_id) 
    fd = open(add_id_file,'w')
    for key in repo.keys():
        if key == "BASE_REPO" :
            continue
        elif '/install_repo/gl/calvados' in key :
            repo_id = 2
        elif '/misc/disk1/tftpboot' in key :
            repo_id = 3
        elif '/install_repo/gl/xr' in key :
            repo_id = 3
        if repo[key] :
            print repo[key]
            for pkg in repo[key]:
                if os.path.basename(pkg) not in sp_rpms:
                    fd.write("%s %d\n"%(pkg,repo_id))
                else :
                    log.debug("Skipping %s from add_id"%(pkg))
    fd.close()

def parsecli():
    parser = argparse.ArgumentParser(description="quick install add utility")

    mandatory_args = parser.add_argument_group('required arguments')
    mandatory_args.add_argument('-r', '--repo', dest='repo', type=str,
                                required=True, action='store',
                                help='Path to find packages')
    parser.add_argument('-p', '--packages', required=True,
                        nargs='*', help='list of packages')
    mandatory_args.add_argument('-i', '--addid', dest='addid', type=str,
                                action='store', default=None,
                                help='install add ID if known')
    parser.add_argument('-sdr','--sdr',help='SDR where packages need to be added')
    pargs = parser.parse_args()
    return pargs


def main():
    try:
        pargs = parsecli()
        INSTALL_DB_DIR = '/var/log/install/instdb/'
        LOGFILE1 = INSTALL_DB_DIR + 'log/install_operation_' + str(pargs.addid) +'.log'
        try:
            os.makedirs (os.path.dirname(LOGFILE1))
        except OSError as e:
            if e.errno != errno.EEXIST:
                raise LocalError ("Unable to create directory %s"%(os.path.dirname(LOGFILE1))) 

        try:
            if not os.path.isfile (LOGFILE1):
                formatter = logging.Formatter('%(asctime)s %(message)s',"%b %d %H:%M:%S")
                fh1 = logging.handlers.RotatingFileHandler(LOGFILE1, maxBytes=(1024*100), backupCount=3)
                fh1.setLevel(logging.INFO)
                fh1.setFormatter(formatter)
                log.addHandler(fh1)
        except:
            pass

        collect_vms_info()
        repo = create_repo(pargs)
        log.debug("Triggering Add Distribute\n")
        # Once data collection is done, log install add operation 
        # as per current syntax
        try:
            import getpass
            user  = getpass.getuser()
        except:
            user = 'root'
            pass

        log.info ("Install operation %s started by %s:" % (pargs.addid, user))
        log.info ("install add source %s %s"%(pargs.repo, ' '.join(pargs.packages)))
        log.info ("Action 1: install add action started")
        log.info ("Install operation will continue in the background")
        push_to_sysadmin(repo,pargs.sdr)
        log.info ("Packages added:")
        for pkg in pargs.packages:
            log.info ("\t%s"%(pkg))
        log.info ("Action 1: install add action completed successfully")
        log.info ("Install operation %s finished successfully"%(pargs.addid))
        log.info ("Ending operation %s"%(pargs.addid))
        log.debug("Distribute Done\n")
        create_add_id(pargs.addid,repo)
    except LocalError as error:
        log.error(error.msg)
        log.error ("Install operation %s aborted"%(pargs.addid))
        log.error ("Ending operation %s"%(pargs.addid))
        TB = traceback.format_exc()
        log.debug(TB)
        xr_repo = os.path.join(pargs.repo, "install_repo/")
        tftp_repo = os.path.join(pargs.repo, "misc")
        log.debug("Triggering Add Cleanup\n")
        try:
            if os.path.exists(pargs.repo):
                shutil.rmtree(pargs.repo)
        except:
            log.info("Unable to remove %s, but continuing..."%(pargs.repo))
        sys.exit(-1)
    except CalError as error:
        log.error(error.msg)
        log.error ("Install operation %s aborted"%(pargs.addid))
        log.error ("Ending operation %s"%(pargs.addid))
        TB = traceback.format_exc()
        log.debug(TB)
        log.debug("Triggering Add Cleanup\n")
        try:
            if os.path.exists(pargs.repo):
                shutil.rmtree(pargs.repo)
        except:
            log.info("Unable to remove %s, but continuing..."%(pargs.repo))
        clean_sysadmin_repo(repo,pargs.sdr)
        sys.exit(-1)

if __name__ == "__main__":
    main()
