#!/usr/bin/python
#------------------------------------------------------------------
# install -
#
# Copyright (c) 2015-2018 by cisco Systems, Inc.
# All rights reserved.
#------------------------------------------------------------------
#

import sys
import os
import re
import subprocess
import traceback
import logging 
import logging.handlers
import time

oper_bg_re = re.compile(r'Install operation will continue in the background')
oper_none_re = re.compile(r'No install operation in progress')
oper_id_re = re.compile(r'Auto commit operation (?P<oper_id>\d+) started for .*')
oper_success_re = re.compile(r'Install operation \d+ finished successfully')


MAX_RETRY_COUNT = 3

def commit():
    """ Commit the software packages, retry twice to avoid transient issue """
    output = '' 
    retry_count = 0 

    LOGFILE = '/var/log/install/autocommit.log'
    if not os.path.exists('/var/log/install'):
        LOGFILE = './autocommit.log'
    # create logger
    logger = logging.getLogger('install_logger')
    logger.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s::  %(message)s',"%Y-%m-%d %H:%M:%S")

    #Console message
    ch = logging.StreamHandler(sys.stdout)
    ch.setLevel(logging.INFO)
    logger.addHandler(ch)

    # Logs to logfile Just keep 10 KB of logs max for deebugging last failure 
    fh = logging.handlers.RotatingFileHandler(LOGFILE, maxBytes=(4096*10), backupCount=1)
    fh.setLevel(logging.DEBUG)
    fh.setFormatter(formatter)
    logger.addHandler(fh)
    try :
        while retry_count < MAX_RETRY_COUNT:
            retry_count = retry_count + 1
            try :
                cmd = ['sdr_instcmd', 'install', 'commit']
                proc = subprocess.Popen(cmd,stdout=subprocess.PIPE)
                proc.wait()
                output = proc.stdout.read()
            except :
                exc_info = sys.exc_info()
                TB = traceback.format_exc()
                logger.debug(TB)
                logger.error("""
                    Error: An exception was hit while waiting for commit to complete.
                    """)
                logger.error(output)
    
            id_re = oper_id_re.search(output)
            if id_re :
                op_id = id_re.group(1)
            elif not id_re :
                logger.error("Failed to get operation ID ")
                logger.error(output)
                # wait for 5 sec and retry
                time.sleep(5)
                continue
    
            if oper_bg_re.search(output) and oper_success_re.search(output):
                logger.debug(output)
                logger.debug("Install commit operation successful")
                retry_count = 0
                break
    
            elif oper_bg_re.search(output) and not oper_success_re.search(output):
                # If operation went in backgrount wait for it to complete
                logger.debug("Operation going on in background, waiting ...")
                in_progress = True
                cmd = ['sdr_instcmd', 'show', 'install', 'request']
        
                while in_progress :
                    try :
                        proc = subprocess.Popen(cmd,stdout=subprocess.PIPE)
                        proc.wait()
                        output1 = proc.stdout.read()
                    except :
                        exc_info = sys.exc_info()
                        TB = traceback.format_exc()
                        logger.debug(TB)
                        logger.error("show install request failed.")
                        logger.error(output1)
                        # Dont bother about "show install request failure" 
                    if oper_none_re.search(output1):
                        in_progress = False
        
                # Test commit status 
                cmd = ['sdr_instcmd', 'show', 'install', 'log', op_id]
                try :
                    proc = subprocess.Popen(cmd,stdout=subprocess.PIPE)
                    proc.wait()
                    output2 = proc.stdout.read()
                except:
                    exc_info = sys.exc_info()
                    TB = traceback.format_exc()
                    logger.debug(TB)
                    logger.error(output2)
                    logger.error("Install commit operation failed.")
                    # Wait for 5 sec and retry
                    logger.info("Retrying commit again... ")
                    time.sleep(5)
                    continue 
                if oper_success_re.search(output2):
                    logger.info(output2)
                    logger.debug("Install commit operation successful")
                    retry_count = 0
                    break
                else:
                    logger.debug(output2)
                    logger.debug("Install commit failed")
                    # Wait for 5 sec and retry
                    logger.info("Retrying commit again... ")
                    time.sleep(5)
                    continue 
            elif not oper_success_re.search(output):
                logger.debug(output)
                logger.debug("Install commit failed")
                # Wait for 5 sec and retry
                logger.info("Retrying commit again...")
                time.sleep(5)
                continue 
        if retry_count == MAX_RETRY_COUNT :
            message = """
            Tried 'install commit' %d times and it is still failing, stopping retry.\
            Please trigger 'install commit' after issue causing the failure is resolved.\
            Details of 'install commit' failure can be checked from 'show install log reverse'
            """ % (retry_count)
            os.system("/pkg/bin/logger -s crit %s"%(message.strip()))
            if os.path.exists("/var/log/install/instdb/replace_commit_marker_file.txt"):
                os.remove("/var/log/install/instdb/replace_commit_marker_file.txt")
    except:
        exc_info = sys.exc_info()
        TB = traceback.format_exc()
        logger.debug(TB)
        logger.error("Error:install autocommit failed, please use 'install commit' command to commit")
        if os.path.exists("/var/log/install/instdb/replace_commit_marker_file.txt"):
            os.remove("/var/log/install/instdb/replace_commit_marker_file.txt")
        message = """install autocommit failed, Please trigger 'install commit' manually."""
        os.system("/pkg/bin/logger -s crit %s"%(message.strip()))
        sys.exit(1)


if __name__ == "__main__":
    ''' Main progrm , forks child for doing the job '''
    try:
        pid = os.fork()
        if pid > 0:
            # Exit parent process
            sys.exit(0)
    except OSError, e:
        if os.path.exists("/var/log/install/instdb/replace_commit_marker_file.txt"):
            os.remove("/var/log/install/instdb/replace_commit_marker_file.txt")
        print >> sys.stderr, "Error: install autocommit failed, please use 'install commit' command to commit"
        message = """install autocommit failed, Please trigger 'install commit' manually."""
        os.system("/pkg/bin/logger -s crit %s"%(message.strip()))
        sys.exit(1)

    # Execute commit command in background in child process
    try:
        commit()
    except OSError, e:
        if os.path.exists("/var/log/install/instdb/replace_commit_marker_file.txt"):
            os.remove("/var/log/install/instdb/replace_commit_marker_file.txt")
        print >> sys.stderr, "Error: install autocommit failed, please use 'install commit' command to commit"
        message = """install autocommit failed, Please trigger 'install commit' manually."""
        os.system("/pkg/bin/logger -s crit %s"%(message.strip()))
        sys.exit(1)

