#!/usr/bin/python
#------------------------------------------------------------------
# install -
#
# Copyright (c) 2015-2019 by cisco Systems, Inc.
# All rights reserved.
#------------------------------------------------------------------
#

import sys
import os
import re
import subprocess
import traceback
import logging 
import logging.handlers
import time

oper_bg_re = re.compile(r'Install operation will continue in the background')
oper_none_re = re.compile(r'No install operation in progress')
oper_id_re = re.compile(r'Install operation (?P<oper_id>\d+) started by .*')
oper_success_re = re.compile(r'Install operation \d+ finished successfully')
oper_no_inactive_pkgs = re.compile(r'Error: No inactive package.*')

MAX_RETRY_COUNT = 3

def remove_inactive_all():
    """ Remove the inactive software post auto commit, retry twice to avoid transient issue """
    output = '' 
    retry_count = 0 

    LOGFILE = '/var/log/install/autoremove.log'
    if not os.path.exists('/var/log/install'):
        LOGFILE = './autoremove.log'
    # create logger
    logger = logging.getLogger('install_logger')
    logger.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s::  %(message)s',"%Y-%m-%d %H:%M:%S")

    #Console message
    ch = logging.StreamHandler(sys.stdout)
    ch.setLevel(logging.INFO)
    logger.addHandler(ch)

    # Logs to logfile Just keep 10 KB of logs max for debugging last failure 
    fh = logging.handlers.RotatingFileHandler(LOGFILE, maxBytes=(4096*10), backupCount=1)
    fh.setLevel(logging.DEBUG)
    fh.setFormatter(formatter)
    logger.addHandler(fh)
    try :
        while retry_count < MAX_RETRY_COUNT:
            retry_count = retry_count + 1
            try :
                time.sleep(5)
                cmd = ['sdr_instcmd', 'install', 'remove', 'inactive', 'all']
                proc = subprocess.Popen(cmd,stdout=subprocess.PIPE)
                proc.wait()
                output = proc.stdout.read()
            except :
                exc_info = sys.exc_info()
                TB = traceback.format_exc()
                logger.debug(TB)
                logger.error("""
                    Error: An exception was hit while waiting for remove to complete.
                    """)
                logger.error(output)
    
            id_re = oper_id_re.search(output)
            if id_re :
                op_id = id_re.group(1)
            elif not id_re :
                logger.error("Failed to get operation ID ")
                logger.error(output)
                # wait for 5 sec and retry
                time.sleep(5)
                continue
    
            if oper_bg_re.search(output) and oper_success_re.search(output):
                logger.debug(output)
                logger.debug("Install remove operation successful")
                retry_count = 0
                break
    
            elif oper_bg_re.search(output) and not oper_success_re.search(output):
                logger.debug(output)
                # If operation went in background wait for it to complete
                logger.debug("Operation going on in background, waiting ...")
                in_progress = True
                cmd = ['sdr_instcmd', 'show', 'install', 'request']
        
                while in_progress :
                    try :
                        proc = subprocess.Popen(cmd,stdout=subprocess.PIPE)
                        proc.wait()
                        output1 = proc.stdout.read()
                    except :
                        exc_info = sys.exc_info()
                        TB = traceback.format_exc()
                        logger.debug(TB)
                        logger.error("show install request failed.")
                        logger.error(output1)
                        # Dont bother about "show install request failure" 
                    if oper_none_re.search(output1):
                        logger.debug(output1)
                        in_progress = False
        
                # Test instal remove inactive all status 
                cmd = ['sdr_instcmd', 'show', 'install', 'log', op_id]
                try :
                    proc = subprocess.Popen(cmd,stdout=subprocess.PIPE)
                    proc.wait()
                    output2 = proc.stdout.read()
                except:
                    exc_info = sys.exc_info()
                    TB = traceback.format_exc()
                    logger.debug(TB)
                    logger.error(output2)
                    logger.error("Install remove operation failed.")
                    # Wait for 5 sec and retry
                    logger.info("Retrying remove again... ")
                    time.sleep(5)
                    continue 
                if oper_success_re.search(output2):
                    logger.info(output2)
                    logger.debug("Install remove operation successful")
                    retry_count = 0
                    break
                elif oper_no_inactive_pkgs.search(output2):
                #Ignore error due to no inactive pkgs and do not retry
                    logger.debug(output2)
                    message = """install remove inactive all failed as no inactive packages were found in the repository."""
                    os.system("/pkg/bin/logger -s crit %s"%(message.strip()))
                    retry_count = 0
                    break
                else:
                    logger.debug(output2)
                    logger.debug("Install remove failed")
                    # Wait for 5 sec and retry
                    logger.info("Retrying remove again... ")
                    time.sleep(5)
                    continue 
            elif not oper_success_re.search(output):
                logger.debug(output)
                logger.debug("Install remove failed")
                # Wait for 5 sec and retry
                logger.info("Retrying remove again...")
                time.sleep(5)
                continue 
        if retry_count == MAX_RETRY_COUNT :
            message = """
            Tried 'install remove inactive all'  %d times and it is still failing, stopping retry.\
            Please trigger 'install remove inactive all' after issue causing the failure is resolved.\
            Details of 'install remove inactive all' failure can be checked from 'show install log reverse'
            """ % (retry_count)
            os.system("/pkg/bin/logger -s crit %s"%(message.strip()))
            if os.path.exists("/var/log/install/instdb/rep_commit_remove_inactive_all_marker_file"):
                os.remove("/var/log/install/instdb/rep_commit_remove_inactive_all_marker_file")
    except:
        exc_info = sys.exc_info()
        TB = traceback.format_exc()
        logger.debug(TB)
        logger.error("Error:install remove inactive all failed, please use 'install remove inactive all' command to remove inactive packages")
        if os.path.exists("/var/log/install/instdb/rep_commit_remove_inactive_all_marker_file"):
            os.remove("/var/log/install/instdb/rep_commit_remove_inactive_all_marker_file")
        message = """install remove inactive all failed, Please trigger 'install remove inactive all' manually."""
        os.system("/pkg/bin/logger -s crit %s"%(message.strip()))
        sys.exit(1)


if __name__ == "__main__":
    ''' Main progrm , forks child for doing the job '''
    try:
        pid = os.fork()
        if pid > 0:
            # Exit parent process
            sys.exit(0)
    except OSError, e:
        if os.path.exists("/var/log/install/instdb/rep_commit_remove_inactive_all_marker_file"):
            os.remove("/var/log/install/instdb/rep_commit_remove_inactive_all_marker_file")
        print >> sys.stderr, "Error: install remove inactive all failed, please use 'install remove inactive all' command to remove inactive packages"
        message = """install remove inactive all failed, Please trigger 'install remove inactive all' manually."""
        os.system("/pkg/bin/logger -s crit %s"%(message.strip()))
        sys.exit(1)

    # Execute install remove inactive all command in background in child process
    try:
        remove_inactive_all()
    except OSError, e:
        if os.path.exists("/var/log/install/instdb/rep_commit_remove_inactive_all_marker_file"):
            os.remove("/var/log/install/instdb/rep_commit_remove_inactive_all_marker_file")
        print >> sys.stderr, "Error: install remove inactive all failed, please use 'install remove inactive all' command to remove inactive packages"
        message = """install remove inactive all failed, Please trigger 'install remove inactive all' manually."""
        os.system("/pkg/bin/logger -s crit %s"%(message.strip()))
        sys.exit(1)

