#!/usr/bin/env perl
#------------------------------------------------------------------
# collect_cef_data.pl -- Script to collect ipv4/ipv6/mpls 
# fib/rib/lsd/ldp/ifh/adj data for a particular prefix/label
#
# February 2019, Payal Bhatia
#
# Copyright (c) 2019-2020 by cisco Systems, Inc. 
# All rights reserved.
#------------------------------------------------------------------

use strict;
use warnings;

use Data::Dumper qw(Dumper);
use List::Util;
use POSIX qw(strftime);

sub is_valid_ip_address;
sub is_valid_mpls_lbl;
sub run_rib_redist_hist_cmd;
sub run_pfx_show_commands;
sub run_lbl_show_commands;

my $scriptname          = $0;
my $loc                 = "";
my $gbl_vrf             = "";
my $proto               = "";
my $lbl                 = "";
my $lbl_hex             = "";
my $pfx                 = "";
my $pfx_hex             = "";
my $pfx_len             = 0;
my @already_printed_lbl = ();
my @already_printed_pfx = ();
my @already_printed_ifh = ();
my $dir                 = "";
my $zipfile             = "";
if ( -d "/harddisk:" ) {
    $dir = "/harddisk:";
}
elsif ( -d "/disk1:" ) {
    $dir = "/disk1:";
}
else {
    $dir = "/disk0:";
}
my $rib_redist_file_name = "$dir/rib_redist_hist_all.txt";
my $rib_file;

my $USAGE = <<EOF;
usage: $scriptname [-h] [-p ipv4/ipv6 -v vrfname -P <pfx/pfx_len>] -l <location>
        OR
       $scriptname [-h] [-p mpls -L <lbl>] -l <location> 
This script parses System C simulator output log file.
-h show help
-p protocol name ipv4/ipv6/mpls
-v vrf name
-P Prefix/Prefix_Length
-L Label
-l location name

Example:
/pkg/bin/collect_cef_data.pl -p ipv4 -v default -P 192.168.0.1/32 -l 0/1/CPU0
/pkg/bin/collect_cef_data.pl -p ipv6 -v default -P 192::1/120 -l 0/1/CPU0
/pkg/bin/collect_cef_data.pl -p mpls -L 24001 -l 0/1/CPU0
EOF

#getopts('hp:v:P:L:l:');
my $argnum;

foreach $argnum ( 0 .. $#ARGV ) {
    if ( $ARGV[$argnum] eq '-h' ) {
        print $USAGE;
        exit 1;
    }
    if ( $ARGV[$argnum] eq '-v' ) {
        $gbl_vrf = $ARGV[ $argnum + 1 ];
        $argnum  = $argnum + 1;
    }
    if ( $ARGV[$argnum] eq '-p' ) {
        $proto  = $ARGV[ $argnum + 1 ];
        $argnum = $argnum + 1;
    }
    if ( $ARGV[$argnum] eq '-P' ) {
        $pfx = $ARGV[ $argnum + 1 ];
        my @split_pfx = split /\//, $pfx;
        if ( scalar(@split_pfx) > 1 ) {
            $pfx_len = $split_pfx[1] + 0;
        }
        $pfx    = $split_pfx[0];
        $argnum = $argnum + 1;
    }
    if ( $ARGV[$argnum] eq '-l' ) {
        $loc    = uc $ARGV[ $argnum + 1 ];
        $argnum = $argnum + 1;
    }
    if ( $ARGV[$argnum] eq '-L' ) {
        $lbl    = $ARGV[ $argnum + 1 ];
        $argnum = $argnum + 1;
    }

}

if ( $proto eq '' ) {
    print "++++++++++++++Protocol is missing++++++++++++++\n";
    print $USAGE;
    exit 1;
}

if (    ( $proto eq 'ipv4' or $proto eq 'ipv6' )
    and ( $pfx eq '' or $loc eq '' or $gbl_vrf eq '' or $pfx_len eq 0 ) )
{
    print
"++++++++++++++Pfx OR Pfx len OR Location OR VRF is missing++++++++++++++\n";
    print $USAGE;
    exit 1;
}
if ( $proto eq 'mpls' ) {
    if ( $lbl eq '' or $loc eq '' ) {
        print "++++++++++++++Label OR Location is missing++++++++++++++\n";
        print $USAGE;
        exit 1;
    }
    else {
        if ( $gbl_vrf eq '' ) {
            $gbl_vrf = 'default';
        }
    }
}

print
"Input - vrf:$gbl_vrf location:$loc proto:$proto ";
if ($pfx ne '') {
    print "pfx - $pfx/$pfx_len\n";
} 
if ($lbl ne '') {
    print "lbl:$lbl\n";
}

my $node_rsi = `node_conversion -N $loc`;
$node_rsi = trim($node_rsi);
my $datestring = strftime "%F.%Hh%Mm%Ss", localtime;
my $filename;
if ( $proto eq 'mpls' ) {
    $filename =
        "$dir/collect_cef_data_" 
      . $proto . "_" 
      . $gbl_vrf . "_" 
      . $lbl . "_"
      . $node_rsi
      . "_$datestring.log";
     $zipfile = 
        "$dir/collect_cef_data_" 
      . $proto . "_" 
      . $gbl_vrf . "_" 
      . $lbl . "_"
      . $node_rsi
      . "_$datestring.zip";
}
else {
    $filename =
        "$dir/collect_cef_data_" 
      . $proto . "_" 
      . $gbl_vrf . "_" 
      . $pfx . "_"
      . $node_rsi
      . "_$datestring.log";
    $zipfile = 
        "$dir/collect_cef_data_" 
      . $proto . "_" 
      . $gbl_vrf . "_" 
      . $pfx . "_"
      . $node_rsi
      . "_$datestring.zip";
}
open( my $fh, '>', $filename ) or die "Could not open file '$filename' $!";

if ( $proto eq 'ipv4' or $proto eq 'ipv6' ) {
    if ( is_valid_ip_address( $pfx, $pfx_len, $proto ) eq 0 ) {
        print "This is invalid $proto Address: $pfx/$pfx_len";
        exit 1;
    }
    run_rib_redist_hist_cmd($proto);
    run_pfx_show_commands( $proto, $pfx, $pfx_len, $gbl_vrf );
}
elsif ( $proto eq 'mpls' ) {
    if ( is_valid_mpls_lbl($lbl) eq 0 ) {
        print "This is Invalid mpls label $lbl";
        exit 1;
    }
    run_lbl_show_commands( $proto, $lbl, $gbl_vrf );
}

close $fh;

#Log collection
my $err1 = 0; my $err2 = 0;
if (-e "$filename") {
    $err1 = system("zip '$zipfile' '$filename'");
}
if (-e "$rib_redist_file_name") {
    $err2 = system("zip '$zipfile' '$rib_redist_file_name'");
}
if ($err1 ne 0 or $err2 ne 0) {
    #Encountered a zipping error. Just pass the files along as is
    print
    "\nPlease collect the following files and pass them along for " 
    . "further debugging - \n";
    print "\t $filename \n\t $rib_redist_file_name\n";
    if ( -e "$zipfile" ) {
        unlink $zipfile;
        if ($! ne "") {
            print "Error $! - $zipfile\n Please delete file manually\n";
        }
    }
}
if ($err1 eq 0 and $err2 eq 0) {
    #zipping successful. Remove the individual files
    print
    "\nPlease collect the following files and pass them along for " 
    . "further debugging - \n";
    print "\t $zipfile \n";
    if (-e "$filename") {
        unlink $filename;
        if ($! ne "") {
            print "Error $! - $filename\n Please delete file manually\n";
        }
    }
    if (-e "$rib_redist_file_name") {
        unlink $rib_redist_file_name;
        if ($! ne "") {
            print "Error - $! - $rib_redist_file_name\n Please delete file manually\n";
        }
    }
}

print
"\n***Please remember to clean up the files from the router after "
. "copying them.***\n\n";

############ End of script ############

#Arguments pfx, pfx_len, proto
sub is_valid_ip_address () {

    #TODO add valid checks for IP addresses
    if ( $_[2] eq "ipv4" ) {
        if ( $_[1] gt 32 ) {
            return 0;
        }
    }
    elsif ( $_[2] eq "ipv6" ) {
        if ( $_[1] > 128 ) {
            return 0;
        }
    }
    return 1;
}

#Arguments pfx, pfx_len, proto
sub is_valid_mpls_lbl () {

    #TODO add valid checks for IP addresses
    if ( $_[0] <= 1048575 ) {
        return 1;
    }
    else {
        return 0;
    }
}

sub location_to_int {
    my $location = shift;
    my $node_rsi = trim(`node_conversion -N $location`);
    my $node     = trim(`node_conversion -i $node_rsi`);

    #print ("Location $location, $node_rsi, and node id = $node\n");
    return $node;
}

sub write_output_to_file {
    my $array_ref = shift;
    my @out       = @$array_ref;
    $array_ref = shift;
    my @cmd     = @$array_ref;
    my $cmd_str = shift;

    #print ("And Command is @cmd\n");
    my $cur = strftime "%e-%a-%b-%Y %H:%M:%S", localtime;
    my $cmd_val;
    print $fh "*" x 80;
    print $fh "\n";
    print $fh $cmd_str;
    print $fh "\n";
    print $fh join( " ", @cmd ), "\n";
    print $fh " ---------- Saved at: $cur ---------- \n\n";
    print $fh join( "", @out ), "\n";
    print $fh "\n";

    print $fh "*" x 80;
    print $fh "\n\n";
}

#Arguments proto

=head
Show commands run:
show rib clients redist history all
Files gathered:
rib_redist_hist_all.txt
=cut

sub run_rib_redist_hist_cmd {
    my $mproto = shift;
    my $p;
    if ( $mproto eq 'ipv4' ) {
        $p = '0x1';
    }
    else {
        $p = '0x2';
    }
    my @ip_rib_cmd = "/pkg/bin/show_" . $mproto . "_rib -X $p -G 0x0 ";
    my $cmd_str    = 'show rib clients redist history all';

    open( $rib_file, '>', $rib_redist_file_name )
      or die "Couldn't open file $rib_redist_file_name $!";
    my @out = qx(@ip_rib_cmd);
    print $rib_file join( " ", @ip_rib_cmd ), "\n";

    print $rib_file "$cmd_str\n";
    print $rib_file join( "", @out ), "\n";
    close $rib_file;
}

sub ip2dec ($) {
    unpack N => pack CCCC => split /\./ => shift;
}

sub trim { my $s = shift; $s =~ s/^\s+|\s+$//g; return $s; }

sub convert_ifname { my $ifname = shift; $ifname =~ tr/\//_/; return $ifname; }

=head
If interface is on an LC, these will be gathered for the LC as well
Show commands run:
show cef vrf <vrfname> <proto> interface <ifname> internal location <location>
show cef vrf <vrfname> <proto> interface <ifname> detail location <location>
show cef vrf <vrfname> <proto> interface <ifname> internal location <location>
show cef vrf <vrfname> <proto> adjacency <ifname>remote internal location <location>
show adjacency <proto> <ifname> internal detail location <location>
show adjacency <proto> <ifname> remote internal detail location <location>
show im database interface <ifname> detail location all
show arp vrf default <ifname> detail
OR
show ipv6 neighbors vrf <vrfname> <ifname>
=cut

sub process_ifh_info {
    my ( $ifh, $proto, $vrf ) = @_;
    my $p;
    my @fib_sh_cmd;
    my $cmd_str;
    my @cmd;
    my @out;

    #sometimes the interface is printed as e.g. Gi0/0/0/0. Filter out for net
    #TODO extend it
    my $regex = '[A-Z][a-z][0-9]';
    if ( ( $ifh =~ $regex ) == 1 ) {
        print("ifh is short $ifh, $regex\n");
        return;
    }

    if ( grep( /^$ifh$/, @already_printed_ifh ) ) {
        print "Already covered interface $ifh, ifh list: "
          . join( ", ", @already_printed_ifh ) . "\n";
        return;
    }
    else {
        push( @already_printed_ifh, $ifh );
    }

    if ( $proto eq "ipv4" ) {
        $p = '0x0';
    }
    else {
        $p = '0x1';
    }
    my @locations;
    my $cur_loc;
    push( @locations, $loc );
    my $ifname = $ifh;
    my @split_ifh = split /\//, $ifh;
    if ( @split_ifh + 0 > 1 ) {
        $ifname = convert_ifname($ifh);

    }

    #Covers interfaces e.g. Gi0/0/0/0 OR MgmtEth0/RP0/CPU0/0
    if (   $ifh =~ /(\d+)\/(\d+)\/(\d+)\/(\d+)/
        || $ifh =~ /(\d+)\/RP(\d+)\/CPU(\d+)\/(\d+)/ )
    {
        $ifname = convert_ifname($ifh);

        #print "split ifh --- $1 and $2 and $3 and $4 $ifh and $ifname\n";
        $cur_loc = "0/$2/CPU0";
        if ( $cur_loc ne $loc ) {
            push( @locations, $cur_loc );
        }
    }
    my $location;
    my $i = 0;
    while ( $i < @locations ) {
        $location = $locations[$i];
        print(
        "+++++++++ Gathering IFH related show commands for $ifh for $location "
        . "+++++++++\n"
        );

        #show cef vrf <vrfname> ipv4/ipv6 interface <ifhname> internal location <loc>
        @fib_sh_cmd =
            "/pkg/bin/fib_show_command -I -O $p -f $vrf "
          . " -i $ifname -N "
          . location_to_int($location);
        $cmd_str =
          "show cef vrf $vrf $proto interface $ifh internal location $location";
        @cmd = @fib_sh_cmd;
        push( @cmd, "--internal" );
        @out = qx(@cmd);
        write_output_to_file( \@out, \@cmd, $cmd_str );

        #show cef vrf <vrfname> ipv4/ipv6 <ifhname> detail location <loc>
        $cmd_str =
          "show cef vrf $vrf $proto interface $ifh detail location $location";
        @cmd = @fib_sh_cmd;
        @out = qx(@cmd);
        write_output_to_file( \@out, \@cmd, $cmd_str );

        #show cef vrf<vrfname> ipv4/ipv6 adjacency <ifname> internal location 0/0/CPU0
        @fib_sh_cmd =
            "/pkg/bin/fib_show_command -Z -O $p -f $vrf "
          . " -9 $p -i $ifname --internal -N "
          . location_to_int($location);
        $cmd_str =
          "show cef vrf $vrf $proto adjacency $ifh internal location $location";
        @cmd = @fib_sh_cmd;
        @out = qx(@cmd);
        write_output_to_file( \@out, \@cmd, $cmd_str );

        #show cef vrf<vrfname> ipv4/ipv6 adjacency <ifname> remote int loc 0/0/CPU0
        @fib_sh_cmd =
            "/pkg/bin/fib_show_command -Z -O $p -f $vrf "
          . " -9 $p -i $ifname -B --internal -N "
          . location_to_int($location);
        $cmd_str =
        "show cef vrf $vrf $proto adjacency $ifh remote internal location $location";
        @cmd = @fib_sh_cmd;
        @out = qx(@cmd);
        write_output_to_file( \@out, \@cmd, $cmd_str );

        #show adjacency <ifname> internal detail location 0/0/CPU0
        my @aib_sh_cmd =
            "/pkg/bin/aib_show_command -E -P $proto -D -I "
          . "$ifname -N "
          . location_to_int($location);
        $cmd_str =
          "show adjacency $proto $ifh internal detail location $location";
        @cmd = @aib_sh_cmd;
        @out = qx(@cmd);
        write_output_to_file( \@out, \@cmd, $cmd_str );

        #show adjacency <ifname> remote internal detail location 0/0/CPU0
        @aib_sh_cmd =
            "/pkg/bin/aib_show_command -E -P $proto -D -I "
          . "$ifname -N "
          . location_to_int($location);
        $cmd_str =
        "show adjacency $proto $ifh remote internal detail location $location";
        @cmd = @aib_sh_cmd;
        push( @cmd, "-R" );
        @out = qx(@cmd);
        write_output_to_file( \@out, \@cmd, $cmd_str );

        $i = $i + 1;
    }

    #show im database interface gigabitEthernet 0/0/0/2 det location all
    my @im_sh_cmd = "/pkg/bin/im_show database -n $ifname -l 0x3 -a";
    $cmd_str = "show im database interface $ifh detail location all";
    @cmd     = @im_sh_cmd;
    @out     = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    if ( $proto eq "ipv4" ) {
        my @arp_sh_cmd = "/pkg/bin/arp_command show -I $ifname -A -V $vrf -E";
        $cmd_str = "show arp vrf $vrf $ifh detail";
        @cmd     = @arp_sh_cmd;
        @out     = qx(@cmd);
        write_output_to_file( \@out, \@cmd, $cmd_str );
    }
    else {
        my @nd_sh_cmd = "/pkg/bin/show_ipv6_neighbors -v $vrf -I $ifname";
        $cmd_str = "show ipv6 neighbors vrf $vrf $ifh";
        @cmd     = @nd_sh_cmd;
        @out     = qx(@cmd);
        write_output_to_file( \@out, \@cmd, $cmd_str );
    }
}

sub process_next_hops {
    my $proto     = shift;
    my $vrf       = shift;
    my $array_ref = shift;
    my @out       = @$array_ref;
    my $toFind    = 'via';
    my $i         = 0;

    print("\n++++++++++++++++++++++++++++++++++++\n");
    foreach my $line (@out) {
        my $have_ifh;
        my $have_pfx;
        my $nh;
        my $ifh;
        my $via_pfx = '';
        my $nh_vrf  = '';
        my $id;
        $line = trim($line);
        $id = index( $line, $toFind );
        my @words = split / /, $line;

        if ( $id eq 0 ) {
            $have_ifh = 0;
            $have_pfx = 0;
            if ( ( $words[1] =~ /^\d/ ) || ( grep( /:/, $words[1] ) ) ) {
                $nh       = substr $words[1], 0, -1;
                $have_pfx = 1;
                $ifh      = $words[2];
                if ( $ifh =~ /^[a-zA-Z]/ ) {
                    if ( ( substr $ifh, -1 ) =~ ',' ) {
                        $ifh = substr $ifh, 0, -1;
                    }
                    $have_ifh = 1;
                }
                my $j = 1;

                #we will process this via block
                while ( ( $i + $j < ( 0 + @out ) )
                    and not( index( trim( $out[ $i + $j ] ), $toFind ) =~ 0 ) )
                {
                    my $ln = trim( $out[ $i + $j ] );
                    my @wd = split / /, $ln;
                    if ( @wd + 0 eq 0 ) {
                        $j = $j + 1;
                        next;
                    }
                    if ( $wd[0] eq 'next' ) {

                        #process
                        #next hop 18.18.18.1/32 via 18.18.18.0/24
                        if ( $wd[-2] eq $toFind ) {

                            #process
                            #next hop 18.18.18.1/32 via 18.18.18.0/24
                            $via_pfx  = $wd[-1];
                            $j        = $j + 1;
                            $have_pfx = 1;
                            next;
                        }
                        if ( $wd[2] eq 'VRF' ) {

                            #process
                            #next hop VRF - 'red', table - 0xe0000011
                            $nh_vrf = $wd[4];

                            #remove the single quotes
                            $nh_vrf = substr $nh_vrf, 1, -2;
                            $j = $j + 1;
                            next;
                        }
                        if ( $wd[-1] =~ /^\d/ ) {
                            $via_pfx = $wd[-1];
                            my @split_pfx = split /\//, $via_pfx;
                            $pfx      = $split_pfx[0];
                            $pfx_len  = $split_pfx[1] + 0;
                            $have_pfx = 1;
                            $j        = $j + 1;
                            next;
                        }
                    }
                    if (    $wd[0] eq 'local'
                        and ( 0 + @wd > 2 )
                        and $wd[2] =~ /^\d/ )
                    {

                        #process  local label 25001
                        my $p = 'mpls';
                        run_lbl_show_commands( 'mpls', $wd[2], $vrf );
                    }
                    $j = $j + 1;
                }
                if ( $have_pfx eq 1 ) {
                    if ( $nh_vrf eq '' ) {
                        $nh_vrf = 'default';
                    }
                    if ( $nh ne $via_pfx ) {
                        my @split_pfx = split /\//, $nh;
                        my $pfx1 = $split_pfx[0];
                        run_pfx_show_commands( $proto, $pfx1, 0, $nh_vrf );
                    }
                    if ( $via_pfx ne '' ) {
                        run_pfx_show_commands( $proto, $pfx, 0, $nh_vrf );
                    }
                    $have_pfx = 0;
                    if ( $have_ifh eq 1 ) {
                        process_ifh_info( $ifh, $proto, $vrf );
                        $have_ifh = 0;
                    }
                }
            }
            else {

                #we have an interface
                my $ifh = $words[1];
                if ( ( substr $ifh, -1 ) =~ ',' ) {
                    $ifh = substr $ifh, 0, -1;
                }
                process_ifh_info( $ifh, $proto, $vrf );
            }
        }
        $i = $i + 1;
    }
    print("\n++++++++++++++++++++++++++++++++++++\n");
}

sub expand_ipv6_address {
    my ($pfx) = shift;
    my @sp        = split /::/, $pfx;
    my $l_octet_cnt = 0;
    my $r_octet_cnt = 0;
    my $i         = 0;
    my $count     = 0;
    my @addr      = '';
    my @res       = '';
    my @left_res  = '';
    my @rt_res    = '';
    my $combined_pfx = '';
    if (index($pfx, "::") eq -1) {
        #non shortened prefix - life is easy. Just expand each octet
        @addr = split /:/, $pfx;
        while ($i < 8) {
            $res[$l_octet_cnt] = sprintf( "%04d", $addr[$l_octet_cnt++] );
            $i++;
        }
        $combined_pfx = join( "", @res );
        return $combined_pfx;
    }
    #cases
    #::ipv6 address
    #::ipv4 address
    #<octet1>:..::<octet7>:<octet8>
    #<octet1>:..::
    #<octet1>::
    #<octet1>::<octet8>
    $i = 0;
    @addr = split /:/, $sp[0];
    if (scalar(@addr) ne 0) {
       while ($i lt scalar(@addr)) {
           $left_res[$l_octet_cnt] = sprintf( "%04s", $addr[$l_octet_cnt++] );
           $i++;
       }
    }
    
    if (scalar(@sp) gt 1) {
    @addr = split /:/, $sp[1];
    if (scalar(@addr) ne 0) {
        $i = 0;
        while ($i lt scalar(@addr)) {
            if ($i+1 == scalar(@addr) and index($sp[1], '.') ne -1) {
                #we have an ipv4 address, should be the end of the IPv6 addr
                my $ipv4_pfx =  $addr[$i];
                my $ipv4_vrf = shift;
                $rt_res[$r_octet_cnt] = sprintf( "%04x", ( ip2dec($ipv4_pfx)) );
                $r_octet_cnt = $r_octet_cnt+2;
                $i = $i+1;
                run_pfx_show_commands('ipv4', $ipv4_pfx, 0, $ipv4_vrf );
                next;
            } else {
                $rt_res[$r_octet_cnt++] = sprintf( "%04s", $addr[$i] );
                $i++;
            }
        }
        } 
    }
    #fill the rest with 0s (may be in between, may be at the end)
    while ($l_octet_cnt + $r_octet_cnt < 8) {
        $left_res[$l_octet_cnt+1] = sprintf( "%04d", 0);
        $l_octet_cnt++;
    }
    $combined_pfx = join( "", @left_res );
    $combined_pfx = $combined_pfx . join( "", @rt_res );
    return $combined_pfx;
}

#Argument proto, pfx, pfx_len, vrf

=head
Show commands run:
show cef vrf <vrfname> <proto> <pfx>[optional /pfx_len] detail location <location>
show cef vrf <vrfname> <proto> <pfx>[optional /pfx_len] internal location <location>
show cef vrf <vrfname> <proto> <pfx>[optional /pfx_len] backup detail location <location>
=cut

sub run_cef_show_commands() {
    my ( $proto, $pfx, $pfx_len, $vrf ) = @_;
    print("\n+++++++++ Gathering CEF show commands for vrf $vrf "
          . "pfx $pfx +++++++++\n");

    my @fib_sh_cmd = ("/pkg/bin/fib_show_command -t");
    my $p;
    my $t;
    my $ip_hex;
    my $pfx_len_hex  = 0;
    my $mask_str     = '';
    my $mask_str_hex = '';
    if ( $proto eq 'ipv4' ) {
        $p      = '0x0';
        $t      = '0x2';
        $ip_hex = sprintf( "%x", ip2dec($pfx) );
    }
    else {
        $p      = '0x1';
        $t      = '0xa';
        $ip_hex = expand_ipv6_address($pfx, $vrf);
    }
    $pfx_len_hex = sprintf( "%x", $pfx_len );

    if ( $pfx_len ne 0 ) {
        $mask_str     = "/$pfx_len";
        $mask_str_hex = "-M $pfx_len_hex";
    }
    @fib_sh_cmd =
"/pkg/bin/fib_show_command -t -O $p -f $vrf -P $ip_hex $mask_str_hex -T $t";
    my @cmd = @fib_sh_cmd;
    push( @cmd, "-d" );
    push( @cmd, "-N" );
    my $loc_id = location_to_int($loc);
    push( @cmd, $loc_id );
    my @out = qx(@cmd);
    my $str_cmd =
      ("show cef vrf $vrf $proto $pfx$mask_str detail location $loc\n");
    my @x = @out;
    write_output_to_file( \@out, \@cmd, $str_cmd );

    $str_cmd =
      ("show cef vrf $vrf $proto $pfx$mask_str internal location $loc\n");
    @cmd = @fib_sh_cmd;
    push( @cmd, "--internal" );
    push( @cmd, "-N" );
    $loc_id = location_to_int($loc);
    push( @cmd, $loc_id );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $str_cmd );

    $str_cmd =
      ("show cef vrf $vrf $proto $pfx$mask_str backup detail location $loc\n");
    @cmd = @fib_sh_cmd;
    push( @cmd, "-d" );
    push( @cmd, "-b" );
    push( @cmd, "-N" );
    $loc_id = location_to_int($loc);
    push( @cmd, $loc_id );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $str_cmd );

    process_next_hops( $proto, $vrf, \@x );
}

#Argument proto, pfx, pfx_len, vrf

=head
Show commands run:
show route vrf <vrfname> <proto> <pfx>[optional /pfx_len] backup
show route vrf <vrfname> <proto> <pfx>[optional /pfx_len] detail private
=cut

sub run_rib_show_commands() {
    my ( $proto, $pfx, $pfx_len, $vrf ) = @_;
    print("\n+++++++++ Gathering RIB show commands for vrf $vrf "
          . "pfx $pfx +++++++++\n");
    my $p;
    my $pfx_len_hex;
    my $ip_hex;
    my @ip_rib_cmd;
    my $cmd_str;
    my @out;
    my @cmd;
    my $mask_str     = '';
    my $mask_str_hex = '';

    if ( $proto eq "ipv4" ) {
        $p = '0x1';
        $ip_hex = sprintf( "%x", ip2dec($pfx) );
    }
    else {
        $p      = '0x2';
        $ip_hex = $pfx;
    }

    $pfx_len_hex = sprintf( "%x", $pfx_len );
    if ( $pfx_len ne 0 ) {
        $mask_str     = "/$pfx_len";
        $mask_str_hex = "-l $pfx_len_hex";
    }

    @ip_rib_cmd =
        "/pkg/bin/show_" 
      . $proto
      . "_rib -X $p -Y 0x1 -Z $vrf -V ________ "
      . "-f $pfx $mask_str_hex -3 $proto $vrf";
    $cmd_str = "show route vrf $vrf $proto $pfx$mask_str backup";
    @cmd     = @ip_rib_cmd;
    @out     = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    @ip_rib_cmd =
        "/pkg/bin/show_" 
      . $proto
      . "_rib -X $p -Y 0x1 -Z $vrf -V ________ " 
      . "-p $ip_hex $mask_str_hex -3 $proto $vrf";
    $cmd_str = "show route vrf $vrf $proto $pfx$mask_str detail private";
    @cmd     = @ip_rib_cmd;
    push( @cmd, "--internal" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    my $lines = join( "", @out );
    if ( index( $lines, "Network not in table" ) != -1 ) {
        return;
    }

    #trim the prefix so we remove the last byte/octet from the prefix
    my $grep_pfx = $pfx;
    if ( $pfx_len eq 0 ) {
        if ( $proto eq "ipv4" ) {
            my @parts = split /\./, $pfx;
            $parts[-1] = '';
            $grep_pfx = join( ".", @parts );
        }
        else {
            my @parts = split /::/, $pfx;
            $parts[-1] = '';
            $grep_pfx = join( "::", @parts );
        }
    }

    @out     = "\n";
    $cmd_str = "show rib $proto clients redist history all | i $grep_pfx";
    @cmd = "/pkg/bin/show_" . $proto . "_rib -X $p -G 0x0 | grep -E $grep_pfx";
    open( $rib_file, '<', $rib_redist_file_name )
      or die "Couldn't open file $rib_redist_file_name $!";
    while ( my $line = <$rib_file> ) {
        if ( $line =~ /$grep_pfx/ ) {
            push( @out, $line );
        }
    }
    write_output_to_file( \@out, \@cmd, $cmd_str );
}

#Argument proto, pfx, pfx_len, vrf
sub run_pfx_show_commands() {
    my ( $proto, $pfx, $pfx_len, $vrf ) = @_;
    if ( grep( /^$pfx/, @already_printed_pfx ) ) {
        print "Already covered prefix $pfx, Pfx list: "
          . join( ", ", @already_printed_pfx ) . "\n";
        return;
    }
    else {
        push( @already_printed_pfx, $pfx );
    }
    &run_cef_show_commands;
    &run_rib_show_commands;

}

=head
Show commands run:
show cef vrf <vrfname> mpls local-label <lbl> eOS internal location <location>
show cef vrf <vrfname> mpls local-label <lbl> eOS detail location <location>
show cef vrf <vrfname> mpls local-label <lbl> non-EOS internal location <location>
show cef vrf <vrfname> mpls local-label <lbl> non-EOS detail location <location>
=cut

sub run_cef_lbl_show_commands {
    my ( $proto, $lbl, $vrf ) = @_;
    my @fib_sh_cmd;
    my @cmd;
    my @out;
    my $cmd_str;
    print(
        "\n+++++++++ Gathering CEF MPLS show commands for label $lbl +++++++++\n"
    );

    my $hex_lbl = sprintf( "0x%x", $lbl );
    @fib_sh_cmd =
        "fib_show_command -t -O 0x2 -f $vrf --mpls_local_label $hex_lbl -N "
      . location_to_int($loc)
      . " --mpls_eos";
    $cmd_str =
      "show cef vrf $vrf mpls local-label $lbl eOS internal location $loc";

    @cmd = @fib_sh_cmd;
    push( @cmd, "0x1" );
    push( @cmd, "--internal" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    $cmd_str =
      "show cef vrf $vrf mpls local-label $lbl eOS detail location $loc";
    @cmd = @fib_sh_cmd;
    push( @cmd, "0x1" );
    push( @cmd, "-d" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    $cmd_str =
      "show cef vrf $vrf mpls local-label $lbl non-EOS internal location $loc";
    @cmd = @fib_sh_cmd;
    push( @cmd, "0x0" );
    push( @cmd, "--internal" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    $cmd_str =
      "show cef vrf $vrf mpls local-label $lbl non-EOS detail location $loc";
    @cmd = @fib_sh_cmd;
    push( @cmd, "0x0" );
    push( @cmd, "-d" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );
}

=head
Show commands run:
show mpls forwarding labels <lbl> debug detail location <location>
show mpls forwarding labels <lbl> hardware egress debug detail location <location>
show mpls forwarding labels <lbl> eos0 debug detail location <location>
show mpls forwarding labels <lbl> both-eos detail location <location>
show mpls lsd forwarding labels <lbl> detail
show mpls label table label <lbl> detail private
show mpls label table label <lbl> history
show mpls label table label <lbl> detail
show mpls label table summary
show mpls ldp forwarding local-label <lbl> detail
show mpls ldp vrf default forwarding local-label <lbl> detail 
show mpls ldp vrf default bindings local-label <lbl> detail
=cut

sub run_mpls_show_commands {
    my ( $proto, $lbl, $vrf ) = @_;
    my @mpls_sh_cmd;
    my @lbl_cmd, my @cmd;
    my @out;
    my $cmd_str;
    print(
    "\n+++++++++ Gathering MPLS LSD/LDP show commands for label $lbl +++++++++\n"
    );

    my $hex_lbl = sprintf( "%x", $lbl );
    @mpls_sh_cmd = '/pkg/bin/fib_mpls_show_fwding';
    $cmd_str = "show mpls forwarding labels $lbl debug detail location $loc";
    @lbl_cmd = "@mpls_sh_cmd -h " . location_to_int($loc) . " -l $hex_lbl";
    @cmd     = @lbl_cmd;
    push( @cmd, "-V" );
    push( @cmd, "-D" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    $cmd_str =
    "show mpls forwarding labels $lbl hardware egress debug detail location $loc";
    @cmd = @lbl_cmd;
    push( @cmd, "-E" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    $cmd_str =
      "show mpls forwarding labels $lbl eos0 debug detail location $loc";
    @cmd = @lbl_cmd;
    push( @cmd, "-X" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    $cmd_str = "show mpls forwarding labels $lbl both-eos detail location $loc";
    @cmd     = @lbl_cmd;
    push( @cmd, "-V" );
    push( @cmd, "-Y" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    $cmd_str = "show mpls lsd forwarding labels $lbl detail";
    my @lsd_sh_cmd = "/pkg/bin/mpls_lsd_fwd_show";
    @lbl_cmd = "@lsd_sh_cmd -b $hex_lbl -e $hex_lbl -d -Z";
    @cmd     = @lbl_cmd;
    @out     = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    my @mpls_lsd_sh_cmd = "/pkg/bin/mpls_lsd_show";
    @lbl_cmd = "@mpls_lsd_sh_cmd -t 0x0 -l $hex_lbl";
    $cmd_str = "show mpls label table label $lbl detail private";
    @cmd     = @lbl_cmd;
    push( @cmd, "-d" );
    push( @cmd, "-P" );
    push( @cmd, "-Z" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    $cmd_str = "show mpls label table label $lbl history";
    @cmd     = @lbl_cmd;
    push( @cmd, "-h" );
    push( @cmd, "-Z" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    $cmd_str = "show mpls label table label $lbl detail";
    @cmd     = @lbl_cmd;
    push( @cmd, "-d" );
    push( @cmd, "-Z" );
    @out = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    $cmd_str = "show mpls label table summary";
    @lbl_cmd = "@mpls_lsd_sh_cmd  -t 0x0 -s -Z";
    @cmd     = @lbl_cmd;
    @out     = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    my @mpls_ldp_sh_cmd = "/pkg/bin/mpls_ldp_show";
    @lbl_cmd = "@mpls_ldp_sh_cmd  -t 0x8 -v $vrf 0x1 -Z false "
      . " -d -u 0x6 -L $hex_lbl";
    $cmd_str = "show mpls ldp forwarding local-label $lbl detail";
    @cmd     = @lbl_cmd;
    @out     = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    @lbl_cmd =
        "@mpls_ldp_sh_cmd -t 0x8 -v $vrf 0x1 -N "
      . location_to_int($loc)
      . " -d -u 0x6 -L $hex_lbl";
    $cmd_str = "show mpls ldp vrf $vrf forwarding local-label $lbl detail ";
    @cmd     = @lbl_cmd;
    @out     = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );

    $cmd_str = "show mpls ldp vrf $vrf bindings local-label $lbl detail";
    @lbl_cmd = "@mpls_ldp_sh_cmd -t 0x2 -v $vrf 0x1 -Z false -b -i $hex_lbl";
    @cmd     = @lbl_cmd;
    @out     = qx(@cmd);
    write_output_to_file( \@out, \@cmd, $cmd_str );
}

#Argument proto, lbl, vrf
sub run_lbl_show_commands() {
    my ( $proto, $lbl, $vrf ) = @_;
    if ( grep( /^$lbl/, @already_printed_lbl ) ) {
        print "Already covered label $lbl, Lbl list: "
          . join( ", ", @already_printed_lbl ) . "\n";
        return;
    }
    else {
        push( @already_printed_lbl, $lbl );
    }
    &run_cef_lbl_show_commands( $proto, $lbl, $vrf );
    run_mpls_show_commands( $proto, $lbl, $vrf );
}

