#!/usr/bin/python
#Author - Arko Dasgupta
#Date	- 07/04/2018

import argparse
import subprocess
import misc_utils 

import logging

def setup_logger(args):
	if args.verbose:
		log_level = logging.DEBUG
	else:
		log_level = logging.INFO

	logging.basicConfig(level=log_level, format="%(asctime)s:%(levelname)s:%(message)s")


def get_parser():
	"""
	Build Arguments
	"""
	parser = argparse.ArgumentParser(description="Utility to dump Broadcom data")
	parser.add_argument("-u", "--unit",	     action="store", help="NPU Unit Number", required=True)
	parser.add_argument("-l", "--lc",	     action="store", help="Line Card Number", required=True)
	parser.add_argument("-p", "--prefix",	     action="store", help="Prefix")
	parser.add_argument("-ll", "--local_label",  action="store", help="Local Label")
	parser.add_argument("-tl", "--tunnel_label", action="store", help="RSVP-TE/SR-TE Label")
	parser.add_argument("-f", "--fec",	     action="store", help="FEC in decimal")
	parser.add_argument("-e", "--encap",	     action="store", help="EEDB Encap in hex")
	parser.add_argument("-v", "--verbose",	     action="store_true", help="Enable Verbose Mode", default=False)
	parser.add_argument("-d", "--db_id",	     action="store", help="Dump Prefix/Route DB. IDs - LEM=0, TCAM=1, KBP=2", default=False)

	return parser

class BroadcomDiag:
	"""
	Access functions to get Broadcom Jericho NPU data
	"""
	def __init__(self, unit, lc):
		self.unit = int(unit, 16)
		self.lc = int(lc)

	def get_diagshell_data(self, diag_string, unit=None):
		"""
		Build the fia diagshell command
		"""
		if unit is None:
			unit = self.unit
                nodeid = self.lc*256
		cmd = "fia_driver_show -c \"%s\" -u 0x%x -n %d" % (diag_string, unit, nodeid)
		cmd_data = misc_utils.run_cmd(cmd)
		return cmd_data

	def get_fec_data(self, fec):
		"""
		Get FEC data
		"""
		# ECMP FEC
		if int(fec) < 4096:
			fec_string = 'dump IHB_FEC_ECMP %s 1' % (fec)
		else:
		# NON ECMP
			fec_string = 'dune \\\"ppd_api frwrd_fec entry_get fec_ndx %s\\\"' % (fec)
		fec_data = self.get_diagshell_data(fec_string)
		return fec_data

	def get_eedb_data(self, unit, encap):
		"""
		Get EEDB Encap data
		"""
		encap_string = "diag pp LIF_show type=out id=%s gl=1" % (encap)
		encap_data = self.get_diagshell_data(encap_string, unit)
		return encap_data

	def get_db_dump_data(self, db_id):
		"""
		Get dump of Route/Prefix Database 
		"""
		db_dump_string = "diag dbal db_dump %s" % (db_id)
		db_dump_data = self.get_diagshell_data(db_dump_string)
		return db_dump_data

	def get_leaf_label_data(self, label):
		"""
		Get Local Label data
		"""
		leaf_string = "dune \\\"ppd_api frwrd_ilm get in_label %s\\\"" % (label)
		leaf_data = self.get_diagshell_data(leaf_string)
		return leaf_data
	def get_prefix_info(self, prefix):
		"""
		Get Encap for Prefix 
		"""
                ip,subnet = prefix.split('/')
                ipv6 = misc_utils.is_ipv6(ip)
                ipv4_tag = "Inx  Vrf   Address             FEC        Flags Flags2 Encap id Hit"
                ipv6_tag = "Inx  Vrf   Address                                     FEC       Flags Flags2 Hit"
                # converting ipv6 in hw readable format
                if ipv6:
                   data = () 
                   data = misc_utils.cidr_to_prefix_hex(prefix, True)
                   str = ''
                   cnt = 0
                   for c in data[0]:
                       if cnt == 4:
                          str = str + ":" + c
                          cnt = 1
                       else:
                          str = str + c
                          cnt = cnt + 1
                   prefix = str + "/" + data[1]
                # geting lc ip to do ssh and dumping prefix info
                ipslot = misc_utils.get_ip_lc(int(self.lc))
                cmd = "ssh %s '/pkg/bin/fia_test pci_stats -r -u %s; cat /tmp/route_summary_%s | grep %s; rm -rf /tmp/route_summary_%s'" % (ipslot,self.unit,self.unit,prefix,self.unit) 
                if ipv6:
                   logging.info(ipv6_tag)
                else:
                   logging.info(ipv4_tag)
                cmd_data = misc_utils.run_cmd(cmd,True)
                logging.info(cmd_data[0])
 
                return 1 

def get_bcm_data(args):
	"""
	This function parses the arguments and executes the appropriate BCM command
	"""
	bcm = BroadcomDiag(args.unit, args.lc)
	if args.fec:
		bcm.get_fec_data(args.fec)
	if args.encap:
		bcm.get_eedb_data(int(args.unit, 16), args.encap)
	if args.local_label:
		bcm.get_leaf_label_data(args.local_label)
	if args.db_id:
		bcm.get_db_dump_data(args.db_id)	
	if args.prefix:
		bcm.get_prefix_info(args.prefix)

if __name__ == "__main__":

	args = get_parser().parse_args()
	setup_logger(args)
	get_bcm_data(args)

