#!/usr/bin/python
#Author - Arko Dasgupta
#Date  - 04/07/2018
# Copyright (c) 2018-2019 by Cisco Systems, Inc.
# All rights reserved

import argparse
import subprocess
import logging
import math
import sys
import re
import misc_utils
import ConfigParser

class FrettaScale:
    """
    This class holds the scale for various HW resources
    """

    def __init__(self):
        self.encaps = None
   

def setup_logger(args):
    if args.verbose:
        log_level = logging.DEBUG
    else:
        log_level = logging.INFO
 
    logging.basicConfig(level=log_level, format="%(asctime)s:%(levelname)s:%(message)s")

def get_parser():
    """
    Build Arguments
    """
    parser = argparse.ArgumentParser(description="Utility to validate hardware resource usage on NCS5500")
    parser.add_argument("-g", "--get_scale",  action="store", help="Get scale numbers for various features")
    parser.add_argument("-d", "--verbose", action="store_true", help="Enable Debug Mode", default=False)
    return parser

def get_config_scale(args):
    """
    Get HW scale values for various features
    """
    scale = FrettaScale() 
    default_config_file = "/pkg/bin/grid_scale.cfg"
    config_file = default_config_file

    config = ConfigParser.ConfigParser()
    config.read(config_file)
    scale.encaps = dict(config.items("GLIF"))
    
    return scale    
    

def compute_scale(scale):
    """
    Get scale numbers for various features and compute the actual usage in hardware and software
    """
    compute_encap_scale(scale.encaps)


def compute_encap_scale(encaps):
    """
    Parse various CLIs to compute total encaps used by HW and SW
    """
    for feature in encaps:
        logging.info("FEATURE %s", feature)
        logging.info("Supported Scale is %s", encaps[feature])
        display_parsed_encap_cli(feature)

def display_parsed_encap_cli(feature):
    """
    Parse various XR commands related to encaps
    """
    if feature == "arp_nd":
        parse_grid_cmd(2, 0)
        parse_ipnh_from_dpa_cmd()
        parse_adjacency_cmd()
    elif feature == "ldp":
        parse_grid_cmd(2, 1)
        parse_ldp_cmd()

def parse_ldp_cmd():
    """
    Parse LDP command for labelled prefixes
    """
    cmd = "mpls_ldp_show -t 0x8 0x1 -Z false -S"
    ldp_op = misc_utils.run_cmd(cmd)
    labelled_paths = int(ldp_op[-3].split(':')[1].strip(' ').split(' ')[0])
    logging.info('Total Labelled Paths in LDP is %d', labelled_paths)

def parse_grid_cmd(pool, bank):
    """
    Parse GRID command to get max ids and available ids
    """
    cmd = "grid_show -P 0x%x -b 0x%x" % (pool, bank)
    grid_op = misc_utils.run_cmd(cmd)
    for line in grid_op:
        if "Max Bank Size" in line:
            max_ids = int(line.split(':')[1].strip(' ').split(' ')[0])
        elif "Available resource IDs" in line:
            ids_left = int(line.split(':')[1].strip(' ').split(' ')[0])

    logging.info('Max IDs in GRID %d', max_ids)
    logging.info('Available IDs in GRID is %d', ids_left)  

def parse_ipnh_from_dpa_cmd():
    """
    Parse DPA command to get total ipnh and ip6nh across all LCs
    """
    cmd = "ofa_sysdb_show_hw_resources -t 0x2 -l all"
    dpa_cmd_op = misc_utils.run_cmd(cmd)
    total_ipnh = 0
    total_ip6nh = 0
    for line in dpa_cmd_op:
        if 'ipnh' in line:
            num = line.split(':')[1].strip(' ').split(' ')[0]
            total_ipnh += int(num)
        elif 'ip6nh' in line:
            num = line.split(':')[1].strip(' ').split(' ')[0]
            total_ip6nh += int(num)

    logging.info('Total ipnh objects across all LCs is %d', total_ipnh)
    logging.info('Total ip6nh objects across all LCs is %d', total_ip6nh)

def parse_adjacency_cmd():
    """
    Parse PI adjacency command
    """
    cmd = "aib_show_command -S"
    adj_cmd_op = misc_utils.run_cmd(cmd)

    ipv4_nh = 0
    ipv6_nh = 0
    for line in adj_cmd_op:
        if "complete adjacencies of type IPv4" in line:
            ipv4_nh += int(line.split()[0])
        elif "complete adjacencies of type IPv6" in line:
            ipv6_nh += int(line.split()[1])

    logging.info('Total IPv4 adjacencies in XR is %d', ipv4_nh)
    logging.info('Total IPv6 adjancencies in XR is %d', ipv6_nh)

if __name__ == "__main__":

    args = get_parser().parse_args()
    setup_logger(args)
    scale = get_config_scale(args)
    compute_scale(scale)

