#!/usr/bin/python
#Author - Arko Dasgupta
#Date  - 04/07/2018

from broadcom_diag import BroadcomDiag
import misc_utils
import argparse
import subprocess
import logging
import math
import sys
import re

def setup_logger(args):
    if args.verbose:
        log_level = logging.DEBUG
    else:
        log_level = logging.INFO

    logging.basicConfig(level=log_level, format="%(asctime)s:%(levelname)s:%(message)s")


def get_parser():
    """
    Build Arguments
    """
    parser = argparse.ArgumentParser(description="Utility to dump CEF harware and trace data")
    parser.add_argument("-p", "--prefix",  action="store", help="IPv4 or IPv6 prefix in CIDR notation ")
    parser.add_argument("-ll", "--local_label",  action="store", help="MPLS Local Label")
    parser.add_argument("-tl", "--tunnel_label",  action="store", help="RSVP-TE/SR-TE Local Label")
    parser.add_argument("-l", "--lc",  action="store", help="Line Card Number", required=True)
    parser.add_argument("-v", "--vrf",   action="store", help="VRF id")
    parser.add_argument("-d", "--verbose", action="store_true", help="Enable Debug Mode", default=True)
    parser.add_argument("-t", "--trace", action="store_true", help="Enable PD FIB and OFA tracedump", default=False)
    parser.add_argument("-f", "--file", action="store", help="File containing the output of show cef hardware")
    return parser

class FrettaCefParser:
    """
    Parse the CEF output  
    """
    cef_tokens = ["LEAF:", "RSHLDI:", "LWLDI:", "SHLDI:", "TX-NHINFO:"]

    def __init__(self, args):
        if args.prefix is None and args.local_label is None:
            logging.error("Please enter Prefix or Local Label")
            sys.exit(1)
        if args.local_label:
            self.local_label = args.local_label
            self.tunnel_label = None
        elif args.tunnel_label:
            self.tunnel_label = args.tunnel_label
            self.local_label = None
        else:
            self.prefix = args.prefix 
            self.is_ipv6 = misc_utils.is_prefix_ipv6(args.prefix)
            self.network, self.mask = misc_utils.cidr_to_prefix_hex(args.prefix, self.is_ipv6)
            self.local_label = None
            self.tunnel_label = None

        self.lc    = int(args.lc)
        self.ip    = misc_utils.get_ip_lc(self.lc)
        self.vrf   = args.vrf
        self.file   = args.file
        self.en_trace = args.trace
 
    def run_cef_cmd(self):
        """
        Build the CEF HW command
        """
        nodeid = self.lc * 256
        if self.local_label:
            cmd = "fib_mpls_show_fwding -h %d -l 0x%x -E" % (nodeid, int(self.local_label)) 
        elif self.tunnel_label:
            cmd = "fib_mpls_show_fwding -h %d -l 0x%x -E -P" % (nodeid, int(self.tunnel_label)) 
        else:
            if self.is_ipv6:
                base_cmd = "fib_show_command -t -O 0x1 -H egress -P %s -M 0x%x -T 0xa -d -N %d" % (self.network, int(self.mask), nodeid)
            else:
                base_cmd = "fib_show_command -t -O 0x0 -H egress -P %s -M 0x%x -T 0x2 -d -N %d" % (self.network, int(self.mask), nodeid)
	    if self.vrf:
                cmd = "%s -f %s" % (base_cmd, self.vrf)
            else:
                cmd = base_cmd

        cmd_data = misc_utils.run_cmd(cmd)
        return cmd_data
    
    def get_fib_trace(self, trans_id):
        """
        This function fetches FIB traces associated with the transaction id
        """
        nodeid = self.lc * 256
        cmd = "ssh %s '/pkg/bin/show_fib_hal_ltrace -i %d | grep ,trans_id,%s,'" % (self.ip, nodeid, trans_id)
        fib_trace_op = misc_utils.run_cmd(cmd)
        
	

    def get_dpa_trace(self, trans_id):
        """
        This function fetches DPA traces associated with the transaction id
        """
        cmd = "ssh %s '/pkg/bin/ofa_show_ltrace | grep ,trans_id,%s,'" % (self.ip, trans_id)
        dpa_trace_op = misc_utils.run_cmd(cmd)
        
    def get_grid_res_data(self, encap):	
        """
        Get GRID data associated with encap
        """
        # Res history command
        cmd = "grid_show -P 0x2 -r %s" % (encap)
        grid_res_hist_op = misc_utils.run_cmd(cmd)
        # Second last word of last line is the client name
        client = grid_res_hist_op[-2].split()[-1].split('(')[0]
        # Res client command
        cmd = "grid_show -P 0x2 -t %s -r %s" % (client, encap)
        grid_res_client_op = misc_utils.run_cmd(cmd) 

    def get_next_empty_line_index(self, cef_slice):
        """
        This function returns the indice of the line in
        the slice that corresponds to the first empty
        line
        """
        cef_slice_len = len(cef_slice)
        for index, line in enumerate(cef_slice):
            if (len(line) == 0) or (index == (cef_slice_len - 1)):
                return index

    def line_has_cef_token(self, cef_line):
        """
        This function returns the the token if it is present in
        cef line else it returns None
        """
        for cef_token in self.cef_tokens:
            if cef_token in cef_line: 
                return cef_token
        return None 

    def get_rev_from_cef(self, data):
        """
        This function retrieves the fib and dpa transaction ids from the data
        """
        indexes = [index for index, word in enumerate(data) if word.startswith("rev:")]
        rev_string = []
        for index in indexes:
            rev_string.append((data[index].split(':'))[1])

        indexes = [index for index, word in enumerate(data) if "dpa-rev:" in word]
        dpa_rev_string = []
        for index in indexes:
            dpa_rev_string.append((data[index].split(':'))[1])


        return (rev_string, dpa_rev_string)

    def get_encaps_from_cef(self, cef_objects):
        """
        This function retrieves the encap data from the cef output
        """
        encaps = {}
        # Get LL Encap / TX-NHINFO data, SRV6NH, MPLSNH 
        cef_encap_keys = ["TX-NHINFO:", "RSHLDI:", "LWLDI:"]
        for encap_key in cef_encap_keys:
            if encap_key in cef_objects:  
                nhinfo = cef_objects[encap_key]
            else:
                nhinfo = None
            if nhinfo is not None:
                for info in nhinfo:
                    nhdata = str(info).split(' ')

                    index = [index for index, word in enumerate(nhdata) if word == "id:" or word == "Id:"]
                    # TX incomplete case
                    if len(index) == 0:
                        continue;
                    else:
                        index = index[0]
                    # Derive the encap nibbles from the next word
                    encap = "0x" + nhdata[index + 1][5:10]
                    index = [index for index, word in enumerate(nhdata) if word == "npu_mask:"]
                    # NPU Mask only for ARP encap
                    if len(index) == 0:
                        unit = 0
                    else:
                        index = index[0]
                        unit = int(math.log(int((nhdata[index + 1])[0]), 2)) 
                    encaps[encap] = {"unit": unit} 

                    # Get index for revision
                    rev_string, dpa_rev_string = self.get_rev_from_cef(nhdata)
                    encaps[encap]["trans_id"] = rev_string
                    encaps[encap]["dpa_trans_id"] = dpa_rev_string
        if self.file:
            logging.debug("ENCAP %s", encaps)

        return encaps  

    def get_fecs_from_cef(self, cef_objects):
        """
        This function retrieves the fec data from the cef output
        """
        fecs = {}
        cef_fec_keys = ["SHLDI:", "RSHLDI:", "LWLDI:"]
        for fec_key in cef_fec_keys:  
            if fec_key in cef_objects:
                ldi = cef_objects[fec_key]
            else:
                ldi = None
            if ldi is not None:
                for info in ldi:
                    ldi_data = str(info).split(' ')
                    indexes = [index for index, word in enumerate(ldi_data) if word == "index:"]
                    for index in indexes:
                        # Derive the fec from the next word
                        fec_string = ldi_data[index + 1]
                        fec = fec_string[fec_string.index('(') + 1 : fec_string.index(')')]
                        fecs[fec] = {} 
                        rev_string, dpa_rev_string = self.get_rev_from_cef(ldi_data)
                        fecs[fec]["trans_id"] = rev_string
                        fecs[fec]["dpa_trans_id"] = dpa_rev_string
        if self.file:
            logging.debug("FEC %s", fecs)

        return fecs 

    def get_hw_objects(self):
        """
        This function extracts the HW data from the CEF output
        """
        cef_objects = self.parse_cef()
        encaps = self.get_encaps_from_cef(cef_objects)
        leaf = {} 

        fecs = self.get_fecs_from_cef(cef_objects)
        if self.file is None:
            bcm = BroadcomDiag("0", self.lc)

            if (self.local_label):
                leaf["data"] = bcm.get_leaf_label_data(self.local_label)
            else:
                # Need Prefix LEM/LPM/TCAM commands
                leaf["data"] = bcm.get_prefix_info(self.prefix) 

            for fec in fecs:
                fecs[fec]["data"] = bcm.get_fec_data(fec)
                if self.en_trace:
                    for trans_id in fecs[fec]["trans_id"]:
                        self.get_fib_trace(trans_id)
                    for trans_id in fecs[fec]["dpa_trans_id"]:
                        self.get_dpa_trace(trans_id)
                     
            for encap in encaps:
                encaps[encap]["data"] = bcm.get_eedb_data(encaps[encap]["unit"], encap)
                self.get_grid_res_data(encap)
                if self.en_trace:
                    for trans_id in encaps[encap]["trans_id"]:
                        self.get_fib_trace(trans_id)
                    for trans_id in encaps[encap]["dpa_trans_id"]:
                        self.get_dpa_trace(trans_id)

        return (leaf, fecs, encaps)

    def parse_cef(self):
        """
        This function runs the cef command and parses the output
        """
        cef_objects = {}
        # Read from file
        if self.file:
            with open(self.file) as f:
                cef_output = f.read()
                cef_lines = cef_output.splitlines()
        # Run command
        else:
            cef_lines = self.run_cef_cmd()
            # Check if output is valid
            if "Prefix not found" in cef_lines[0] or "not configured or does not exist" in cef_lines[0]:
                logging.error("Invalid Prefix/Label")
                sys.exit(1)   

        # Parse Tokens 
        cur_index = 0
        while cur_index < len(cef_lines):
            # Check if the line has a token
            cef_token = self.line_has_cef_token(cef_lines[cur_index]) 
            if cef_token is not None:
                # Get Slice end index
                object_end_index = self.get_next_empty_line_index(cef_lines[cur_index:])
                # Slice contains all data associated with the token
                if cef_token not in cef_objects:
                    cef_token_data = []
                else:
                    cef_token_data = cef_objects[cef_token]
                cef_token_data.append(cef_lines[cur_index + 1: cur_index + object_end_index]) 
                cef_objects[cef_token] = cef_token_data
            # Move index to the next token
                cur_index = cur_index + object_end_index
            else:
                cur_index = cur_index + 1
        return cef_objects

    def build_dag(self, leaf, fecs, encaps):
        """
        This function builds and verifies the HW chain (DAG) by 
        parsing the HW objects
        leaf -> fecs -> encaps
        """
        #TO DO
        return None

def get_parsed_cef(args):
    """
    This function prints the parsed cef output
    """
    cef_parser = FrettaCefParser(args)
    (leaf, fecs, encaps) = cef_parser.get_hw_objects()
    hw_dag = cef_parser.build_dag(leaf, fecs, encaps)

if __name__ == "__main__":

    args = get_parser().parse_args()
    setup_logger(args)
    get_parsed_cef(args)



