#!/usr/bin/python
# Author - Arko Dasgupta
# Date   - 04/15/2018

import socket
import binascii
import subprocess
import argparse
import re
import logging
from argparse import RawTextHelpFormatter

helper_function = ['get_grid_lsd_keys {rp0,rp1}', 
                   'cidr_to_prefix_hex {ipv4/subnet}', 
                   'cidr_to_prefix_hex {ipv6/subnet}']

def setup_logger(args):
    if args.verbose:
        log_level = logging.DEBUG
    else:
        log_level = logging.INFO

    logging.basicConfig(level=log_level, format="%(asctime)s:%(levelname)s:%(message)s")

def get_parser():
    """
    Build Arguments
    """
    parser = argparse.ArgumentParser(formatter_class=RawTextHelpFormatter,description="Utility to dump CEF harware and trace data")
    parser.add_argument("-f", "--function",  action="store", help=helper_func())
    parser.add_argument("value",  action="store", help="value")
    parser.add_argument("-v", "--verbose",   action="store_true", help="Enable Verbose Mode", default=False)
    return parser

def helper_func():
    return '\n'.join(helper_function)

def run_cmd(cmd, quiet=False):
    """
    Execute the CEF command on the RP
    """
    if quiet is False:
       logging.debug("Executing ### %s ###", cmd)
    cmdpipe = subprocess.Popen(["%s" % cmd ] ,stdout=subprocess.PIPE, shell=True)
    result = cmdpipe.communicate()
    data = str(result[0]).split("\n")
    if quiet is False:
       logging.debug("%s", str(result[0]))
    return data

def is_ipv6(s):
    pieces = s.split('.')
    if len(pieces) != 4: return True
    else: return False

def is_prefix_ipv6(prefix_string):
    if ':' in prefix_string:
       return True
    else:
       return False

def get_ip_lc(lc):
    """
    This function returns the XR VM IP address for the corresponding LC
    """
    # Get IPs from sysdb command
    cmd = "/pkg/bin/show_platform_sysdb -v"
    cmd_data = run_cmd(cmd,True) 
    # Sample O/P - 0/0/CPU0  LC (ACTIVE)  NONE  FINAL Band  192.0.0.6
    for vminfo in cmd_data:
        vminfo = vminfo.strip()
        #Skip for RP
        if re.search('RP',vminfo):
           continue
        if not re.search('CPU0',vminfo):
           continue
        res = re.split("\s\s+",vminfo)
        lc_num = int(str(res[0]).split('/')[1])
        if lc_num == lc:
           # return IP address for this LC
           return res[4]

def cidr_to_prefix_hex(cidr, is_ipv6):
    """
    This function converts a IP network prefix in CIDR formatted
    to a hex format
    """
    ip, subnet = cidr.split('/') 
    subnet_bits = int(subnet)
    mask = 0
    if not is_ipv6:
        ip_hex = binascii.hexlify(socket.inet_pton(socket.AF_INET, ip))
        # Buld MASK
        for i in range(0, 32):
            if subnet_bits:
                mask = (mask << 1) | 1 
                subnet_bits = subnet_bits - 1
            else:
                mask = (mask << 1) | 0 
        
        ip_hex = int(ip_hex, 16) & mask
        return (str(hex(ip_hex))[2:], subnet)
    else:
        ip_hex = binascii.hexlify(socket.inet_pton(socket.AF_INET6, ip))
        i = 0
        ip_hex_num = '' 
        # This twisted code changes endianness
        while(i < 32):
            num = ip_hex[i:i+4]
            i = i + 4
            ip_hex_num = ip_hex_num + str(num) 
        # Incorporate Mask
        ip_hex_num = ip_hex_num[0:subnet_bits/4] + str().zfill(32 - subnet_bits/4)  
        return (ip_hex_num, subnet)

def get_grid_lsd_keys(rp):
    """
    This function find out label corresponding to keys 
    """
    temp_list =[]
    str = "Client Key"
    if rp == 'rp0':
       cmd = "grid_show -P 0x2 -t lsd -r -1 -l 4352"
    elif rp == 'rp1':
       cmd = "grid_show -P 0x2 -t lsd -r -1 -l 4096"
    else:
       print("wrong rp location")
       return
    output = run_cmd(cmd, True)
    for op in output:
        logging.info(op) 
        if op != '' and op is not None and (op.split(':')[0].strip()) == str:
              key = op.split(':')[1]
              logging.info("Label                         : %d" %(int(key[9:17],16)))

def parse_func(args):
    if args.function == "get_grid_lsd_keys":
       get_grid_lsd_keys(args.value)
    if args.function == "cidr_to_prefix_hex":
       ipv6,subnet = args.value.split('/')
       logging.info(cidr_to_prefix_hex(args.value, is_prefix_ipv6(ipv6)))

if __name__ == "__main__":

    args = get_parser().parse_args()
    setup_logger(args)
    parse_func(args)


