#!/usr/bin/python

import subprocess
import sys
import json
import os
import logging
import logging.handlers
import re
import copy
import itertools
import signal
# Needed as SMU can change the
# location of the yaml.py or
# this file.
try:
    import yaml
except:
    result = []
    name = 'cyaml.py'
    path = '/opt/cisco/XR/packages/'
    for root, dirs, files in os.walk(path):
        if name in files:
            result.append(root)
    sys.path.extend(result)
    sys.path.insert(0, "/pkg/bin")
    import yaml

LOGFILE = '/var/log/license/smart_lic_detail.log'
RUNNING_CFG_FILE = "sl_run_cfg"
MISC_DISK1 = "/misc/disk1/"
TMP = "/tmp/"
VAR_LOG_LIC = "/var/log/license/"
PIDFILE = '/pkg/bin/smartlic_pids.yaml'
UNSUPP_PIDFILE = '/pkg/bin/unsupported_lc_pids'
intf_dict = { 'Hu' : 'HundredGigE',
              'FH' : 'FourHundredGigE',
              'Te' : 'TenGigE',
              'Fo' : 'FortyGigE',
              'Fi' : 'FiftyGigE',
              'TF' : 'TwentyFiveGigE',
              'TH' : 'TwoHundredGigE',
              'Gi' : 'GigabitEthernet'}

breakout_dict = { 10  : 'TenGigE',
                  25  : 'TwentyFiveGigE',
                  40  : 'FortyGigE',
                  50  : 'FiftyGigE',
                  100 : 'HundredGigE',
                  200 : 'TwoHundredGigE',
                  400 : 'FourHundredGigE'}

bandwidth_dict = { 'TenGigE' : 10000000,
                   'TwentyFiveGigE' : 25000000,
                   'FortyGigE' : 40000000,
                   'FiftyGigE' : 50000000,
                   'HundredGigE' : 100000000,
                   'TwoHundredGigE' : 200000000,
                   'GigabitEthernet' : 1000000,
                   'FourHundredGigE' : 400000000}

unsupported_pid_list = set()
ess_intf_list = []
feature_list = []
feature = {}
destination = '/misc/disk1/'
feat_lst_json_file = 'feature_list'
supp_lc_pid = []
lc_dict = {}
adv_intf_exclusion_list = []
running_cfg = ""


class TimeOut(Exception):
    """Base class for other exceptions"""
    pass

class NotEnoughSpace(Exception):
    """Base class for other exceptions"""
    pass


def storeData(data):
    logger.info("Entering Function " + storeData.__name__)
    filepath = os.path.join(destination,feat_lst_json_file)
    fp = open(filepath,'w')
    try:
        json.dump(data,fp)
    except Exception as e:
        logger.error("failed to write in %s due to error %s" %(filepath,e))
    fp.close()

'''
loadData prints the output in the below format. Sample of output:

Feature                         Location                       Interface List
--------------------------------------------------------------------------------
ESSENTIAL                      0/2/CPU0                       HundredGigE0/2/0/11
ESSENTIAL                      0/2/CPU0                       HundredGigE0/2/0/7
L2VPN(Advantage)                0/2/CPU0                       HundredGigE0/2/0/7
'''
def loadData(print_size=None):
    logger.info("Entering Function " + loadData.__name__)
    line_sep = "-"
    feat_intf_list = []
    filepath = os.path.join(destination,feat_lst_json_file)
    try:
        dbfile = open(filepath, 'r')
        db = json.load(dbfile)
    except:
        logger.error("Failed to load json file %s" %(filepath))
    else:
        heading = "%s %s %s"%('{0: <30}'.format("Feature"), \
           '{0: <30}'.format("Location"),'{0: <30}'.format("Interface List"))
        feat_intf_list.append(heading)
        feat_intf_list.append(line_sep*80)
        for keys in db:
            for name, card_info, intf_li in zip(keys['name'], \
                                  keys['card_info'],keys['intf_list']):
                if( len(intf_li) > 0):
                    new_str = "%s %s %s"%('{0: <30}'.format(name),\
                       '{0: <30}'.format(card_info),'{0: <30}'.format(intf_li))
                    feat_intf_list.append(new_str)
        feat_intf_list = "\n   ".join(feat_intf_list)
        if (print_size != None):
            print(len(feat_intf_list))
        else:
            print(feat_intf_list)
    finally:
        dbfile.close()

def loadListSize():
    logger.info("Entering Function " + loadListSize.__name__)
    feat_intf_list = []
    filepath = os.path.join(destination,feat_lst_json_file)
    try:
        dbfile = open(filepath, 'r')
        
        db = json.load(dbfile)
    except Exception as e:
        logger.error("Failed to load json file %s Error %s" %(filepath,e))
    else:
        for keys in db:
            for name, card_info, intf_li in zip(keys['name'], \
                                  keys['card_info'],keys['intf_list']):
                if( len(intf_li) > 0):
                    new_str = "%s %s %s"%('{0: <30}'.format(name),\
                       '{0: <30}'.format(card_info),'{0: <30}'.format(intf_li))
                    feat_intf_list.append(new_str)
        print(len(feat_intf_list))
    finally:
        dbfile.close()

def loadFourHundredGigeBandwidthData(type):
    logger.info("Entering Function " + loadFourHundredGigeBandwidthData.__name__)
    feat_intf_list = []
    bandwidth = 0
    filepath = os.path.join(destination,feat_lst_json_file)
    try:
        dbfile = open(filepath, 'r')

        db = json.load(dbfile)
    except:
        logger.error("Failed to load json file %s" %(filepath))
    else:
        for keys in db:
            for name, card_info, intf_li in zip(keys['name'], \
                                  keys['card_info'],keys['intf_list']):
                if((len(intf_li) > 0) and (re.search(type,name))) :
                    if(re.match("FourHundredGigE",intf_li)) :
                        bandwidth += bandwidth_dict['FourHundredGigE']

                    new_str = "%s"%(intf_li.strip())
                    feat_intf_list.append(new_str)
        bandwidth = bandwidth/1000000
        print(bandwidth)
    finally:
        dbfile.close()

def loadBandwidthData(type):
    logger.info("Entering Function " + loadBandwidthData.__name__)
    feat_intf_list = []
    bandwidth = 0
    filepath = os.path.join(destination,feat_lst_json_file)
    try:
        dbfile = open(filepath, 'r')
        db = json.load(dbfile)
    except:
        logger.error("Failed to  load json file %s" %(filepath))
    else:
        for keys in db:
            for name, card_info, intf_li in zip(keys['name'], \
                                  keys['card_info'],keys['intf_list']):
                if((len(intf_li) > 0) and (re.search(type,name))) :
                    if(re.match("HundredGigE",intf_li)) :
                        bandwidth += bandwidth_dict['HundredGigE']
                    elif(re.match("TenGigE",intf_li)) :
                        bandwidth += bandwidth_dict['TenGigE']
                    elif(re.match("FortyGigE",intf_li)) :
                        bandwidth += bandwidth_dict['FortyGigE']
                    elif(re.match("FiftyGigE",intf_li)) :
                        bandwidth += bandwidth_dict['FiftyGigE']
                    elif(re.match("TwentyFiveGigE",intf_li)) :
                        bandwidth += bandwidth_dict['TwentyFiveGigE']
                    elif(re.match("TwoHundredGigE",intf_li)) :
                        bandwidth += bandwidth_dict['TwoHundredGigE']
                    elif(re.match("GigabitEthernet",intf_li)) :
                        bandwidth += bandwidth_dict['GigabitEthernet']

                    new_str = "%s"%(intf_li.strip())
                    feat_intf_list.append(new_str)
        bandwidth = bandwidth/1000000
        print(bandwidth)
    finally:
        dbfile.close()

def run_cmd(cmd):
     process = subprocess.Popen(cmd, stdout=subprocess.PIPE,
                                stderr=subprocess.PIPE, shell=True)
     out, error = process.communicate()
     sprc = process.returncode
     if sprc is None or sprc != 0:
         out = str.encode("error")
         logger.error("Error CMD=%s returned --->%s" % (cmd, out))
     outstr = out.decode("ascii")
     return dict(rc=sprc, output=outstr)

def get_card_info(intf_list):
    card_list = []
    for intf in intf_list:
        card_info = re.search(r".*?(\d+)\/(\d+)\/(\d+)",intf)
        if card_info:
            card_name = card_info.group(1) + "/" + card_info.group(2) +"/CPU" \
                        + card_info.group(3)
            card_list.append(card_name)
    return(card_list)

def get_bundle_intf(intf_li):
    bundle_intf_list = []
    for key,intf in itertools.product(intf_dict,intf_li):
        bundle_intf = re.match("^[A-Za-z]{2}[0-9]", intf)
        if bundle_intf:
            if key in intf:
                new_intf  = intf.replace(key,intf_dict[key])
                bundle_intf_list.append(new_intf)
        else :
            logger.info("Unmatched bundle interface {}".format(intf))
            bundle_intf_list.append(intf)
    bundle_intf_list = list(set(bundle_intf_list))
    return(bundle_intf_list)

def adv_intf_priority_list_exclusion(intf_list):
    global adv_intf_exclusion_list
    final_intf_li = list((set(intf_list) - set(adv_intf_exclusion_list)))
    adv_intf_exclusion_list = list(set(adv_intf_exclusion_list + final_intf_li))
    return final_intf_li

def get_breakout_intf(breakout_intf, breakout_intf_list):
    global RUNNING_CFG_FILE   
    ret = 0
    intf_name = ""
    breakout = re.split('[a-zA-Z]+', breakout_intf.group())
    breakout = ' '.join(breakout).split()
    if breakout :
        breakout_card_loc = breakout[0]
        breakout_card_loc = breakout_card_loc[:-2]
        if breakout_card_loc not in breakout_intf_list :
            cmd = "cat  %s | grep -A1 \"controller\" \
                | grep -A1 \"%s\" | \
                grep breakout | awk '{print $2}'" %(RUNNING_CFG_FILE,breakout_card_loc)
            result = run_cmd(cmd)
            if (result['rc'] == 0):
                breakout_bw = result['output'].rstrip()
                bandwidth = breakout_bw.split("x")
                bandwidth = ' '.join(bandwidth).split()
                if bandwidth :
                    totalbandwidth = int(bandwidth[0]) * int(bandwidth[1])
                    if totalbandwidth in breakout_dict :
                        intf_name = str(breakout_dict[totalbandwidth]) + breakout_card_loc
                        breakout_intf_list.append(breakout_card_loc)
                        ret = 1
                else :
                    intf_name = breakout_intf.group() 
                    intf_name = intf_name.strip()
                    ret = 1

    return dict(rc=ret, intfname=intf_name)

'''
Snippet of the output

 Port                  Device           State        Port ID         B/W, kbps
  --------------------  ---------------  -----------  --------------  ----------
  Hu0/2/0/7             Local            Active       0x8000, 0x0000   100000000
      Link is Active
  Hu0/2/0/8             Local            Configured   0x8000, 0x0000   100000000
      Link is down
  Hu0/2/0/9             Local            Configured   0x8000, 0x0000   100000000
      Link is down
  Hu0/2/0/10            Local            Configured   0x8000, 0x0000   100000000
      Link is down
  Hu0/2/0/11            Local            Active       0x8000, 0x0000   100000000
      Link is Active
'''
def get_intf_from_bundle(bd_id):
    new_intf_list = []
    bundle_intf = []
    breakoutlist = []
    breakout_intf = []

    cmd = "/pkg/bin/bundlemgr_show -b -B "+ bd_id
    result = run_cmd(cmd)
    if(result['rc'] == 0):
        new_intf_list = result['output'].strip().split('\n')
        for new_intf in new_intf_list:
            intf = re.search("(\w+\/\d+\/\d+\/\d+)",new_intf)
            if intf:
                breakout = re.search("(\w+\/\d+\/\d+\/\d+\/\d+)", new_intf)
                if breakout:
                    ret = get_breakout_intf(breakout, breakoutlist)
                    if (ret['rc'] == 1):
                        breakout_intf.append(ret['intfname'])
                    else :
                        continue
                else :
                    bundle_intf.append(intf.group())

    bundle_intf = bundle_intf + breakout_intf
    if(len(bundle_intf) > 0):
        bundle_intf = get_bundle_intf(bundle_intf)
        bundle_intf = format_list(bundle_intf)
    return bundle_intf

'''
Example of the output

Legend: pp = Partially Programmed.
Bridge group: bg1, bridge-domain: 1, id: 0, state: up, ShgId: 0, MSTi: 0
  Aging: 300 s, MAC limit: 64000, Action: none, Notification: syslog
  Filter MAC addresses: 0
  ACs: 1 (0 up), VFIs: 0, PWs: 0 (0 up), PBBs: 0 (0 up), VNIs: 0 (0 up)
  List of ACs:
    BE1, state: unresolved, Static MAC addresses: 0
  List of Access PWs:
  List of VFIs:
  List of Access VFIs:
Bridge group: bg4, bridge-domain: 4, id: 3, state: up, ShgId: 0, MSTi: 0
  Aging: 300 s, MAC limit: 64000, Action: none, Notification: syslog
  Filter MAC addresses: 0
  ACs: 1 (0 up), VFIs: 0, PWs: 0 (0 up), PBBs: 0 (0 up), VNIs: 0 (0 up)
  List of ACs:
    Hu0/2/0/7, state: unresolved, Static MAC addresses: 0
  List of Access PWs:
  List of VFIs:
  List of Access VFIs:

'''
def show_l2vpn_bridge_domain(output):
    intf_list = []
    intf_list1 = []
    breakoutlist = []

    result = output.strip().split('\n')
    for line in result:
        z = re.search(r"(\w+\/\d+\/\d+\/\d+).*|(BE\d+)",line)
        if z:
            if re.match("BE\d+",z.group()):
                bundle_name = z.group().replace("BE","Bundle-Ether")
                intf_in_bundle = get_intf_from_bundle(bundle_name)
                intf_list1 = intf_list1 + intf_in_bundle
            else:
                breakout = re.search("(\w+\/\d+\/\d+\/\d+\/\d+)", line)
                if breakout:
                    ret = get_breakout_intf(breakout, breakoutlist)
                    if (ret['rc'] == 1):
                        intf = [ret['intfname']]
                        intf = get_bundle_intf(intf)
                        intf_list = intf_list + intf
                    else:
                        continue
                else:
                    intf = []
                    intf.append(z.group(1))
                    intf = get_bundle_intf(intf)
                    intf_list = intf_list + intf

    final_intf_list  = list(set(intf_list1 + intf_list))
    final_intf_list = list(dict.fromkeys(final_intf_list))
    final_intf_list = format_list(final_intf_list)
    return final_intf_list

def format_list(intf_li):
    global ess_intf_list
    final_intf_li = ((set(ess_intf_list) & set(intf_li)))
    return list(final_intf_li)

def read_unsupported_pids():
    global unsupported_pid_list
    if os.path.exists(UNSUPP_PIDFILE):
        with open(UNSUPP_PIDFILE, 'r') as fd:
            unsupported_pid_list = set([ line.strip() for line in fd.readlines()
                                   if line.strip() and not line.strip().startswith('#') ])
    logger.info("Unspported pids read: {}".format(unsupported_pid_list))
 
def get_up_interface():
    global ess_intf_list
    global feature_list
    global lc_dict
    ess_license = {}
    feat_name = ['ESSENTIAL']
    cmd = "show_ip_interface -b -v all"
    cmd1 = "/pkg/bin/show_inventory -e"
    lc_name = ""
    pid = ""

    result = run_cmd(cmd1)
    if(result['rc'] == 0):
        lc_list = result['output'].strip().split('\n')
        for lc in lc_list:
            if re.search("NAME",lc):
                lc_name = lc.split(",")[0].split(":")[1].replace('"','').strip()
            elif re.search("PID",lc):
                pid = lc.split(",")[0].split(":")[1].replace('"','').strip()
            if(len(lc_name) and len(pid)):
                lc_dict[lc_name] = pid

    result = run_cmd(cmd)
    if(result['rc'] == 0):
        intf_list = result['output'].strip().split('\n')
        breakoutlist = []
        intf_regex = re.compile("(\w+\/\d+\/\d+\/\d+).*?(Up)")
        breakout_regex = re.compile("(\w+\/\d+\/\d+\/\d+\/\d+)") 
        for intf in intf_list:
           z = intf_regex.match(intf)
           if z:
               intfname = z.group(1)
               logger.info("INTF_NAME {}".format(intfname))
               breakout = breakout_regex.match(intf)
               if breakout:
                   ret = get_breakout_intf(breakout, breakoutlist)
                   if (ret['rc'] == 1):
                       intfname = ret['intfname']
                   else :
                       continue
               ess_intf_list.append(intfname)

        ess_intf_list = list(set(ess_intf_list))
        logger.info("ess int list {}".format(ess_intf_list))
        if(len(ess_intf_list) > 0):
            # repeat feature name equal to the number of interface
            feat_name = feat_name * len(ess_intf_list)
            ess_license['name'] = feat_name
            ess_license['intf_list'] = ess_intf_list
            card_info = get_card_info(ess_license['intf_list'])
            ess_license['card_info'] = card_info
            feature_list.append(ess_license)
            remove_unsupported_card()

'''
Example of output

l2vpn
 bridge group bg1
  bridge-domain 1
   interface Bundle-Ether1
   !
  !
 !
 bridge group bg4
  bridge-domain 4
   interface HundredGigE0/2/0/7
   !
  !
 !
!
'''
def get_l2vpn_interface():
    global feature_list
    feat_name = ['L2VPN(Advantage)']
    l2_feature = {}
    l2vpn_intf = []

    cmd = "/pkg/sbin/nvgen -c -q gl/l2vpn/"
    result = run_cmd(cmd)
    if not result["rc"]:
         intf_list = result['output'].strip().split('\n')
         breakoutlist = []
         for intf in intf_list:
             z = re.search(r"(\w+\/\d+\/\d+\/\d+)|(\w+[-]\w+\d+)",intf)
             if z:
                 if "Bundle" in z.group():
                     intf_in_bundle = get_intf_from_bundle(z.group())
                     l2vpn_intf = l2vpn_intf + intf_in_bundle
                 else:
                     intfname = z.group(1)
                     breakout = re.search("(\w+\/\d+\/\d+\/\d+\/\d+)", intf)
                     if breakout:
                         ret = get_breakout_intf(breakout, breakoutlist)
                         if (ret['rc'] == 1):
                              intfname = ret['intfname']
                         else :
                              continue
                     l2vpn_intf.append(intfname)

         l2vpn_intf = list(set(l2vpn_intf))
         l2vpn_intf = list(dict.fromkeys(l2vpn_intf))
         l2vpn_intf = format_list(l2vpn_intf)
         l2vpn_intf = adv_intf_priority_list_exclusion(l2vpn_intf)
         if(l2vpn_intf):
            feat_name = feat_name * len(l2vpn_intf)
            l2_feature['name'] = feat_name
            l2_feature['intf_list'] = l2vpn_intf
            card_info = get_card_info(l2vpn_intf)
            l2_feature['card_info'] = card_info
            feature_list.append(l2_feature)

def get_l3vpn_interface():
    global interface_list
    global feature_list
    l3_intf_li = []
    l3_feature = {}
    feat_name = ['L3VPN(Advantage)']
    cmd = "mpls_vpn_show -a | sed '1d;/^ /d' | wc -l"
    result = run_cmd(cmd)
    vrf_count = int(result['output'])
    if vrf_count > 8:
        cmd = "show_ip_interface -b -v all"
        result = run_cmd(cmd)
        breakoutlist = []
        if not result['rc']:
            vrf_list = result['output'].strip().splitlines()
            for vrf in vrf_list:
                if "default" not in vrf :
                    z = re.search(r"(\w+\/\d+\/\d+\/\d+)|(\w+[-]\w+\d+)", vrf)
                    if z :
                        if "Bundle" in z.group():
                            intf_in_bundle = get_intf_from_bundle(z.group())
                            l3_intf_li = l3_intf_li + intf_in_bundle
                        else:
                            intfname = z.group(1)
                            breakout = re.search("(\w+\/\d+\/\d+\/\d+\/\d+)", vrf)
                            if breakout:
                                ret = get_breakout_intf(breakout, breakoutlist)
                                if (ret['rc'] == 1):
                                    intfname = ret['intfname']
                                else :
                                    continue
                            l3_intf_li.append(intfname)
            l3_intf_li = format_list(l3_intf_li)
            l3_intf_li = adv_intf_priority_list_exclusion(l3_intf_li)
            if(len(l3_intf_li) > 0):
            # repeat feature name equal to the number of interface
                feat_name = feat_name * len(l3_intf_li)
                l3_feature['name'] = feat_name
                l3_feature['intf_list'] = l3_intf_li
                card_info = get_card_info(l3_intf_li)
                l3_feature['card_info'] = card_info
                feature_list.append(l3_feature)

'''
Example of output
RP/0/RP0/CPU0:peyto#show macsec mka interface
Mon Jan 13 13:24:27.091 UTC
==========================================================================
   Interface-Name      KeyChain-Name   Fallback-KeyChain     Policy Name
==========================================================================
     TF0/0/0/27             kc               - NA -       *DEFAULT POLICY*
     TF0/0/0/39             kc               - NA -       *DEFAULT POLICY*

'''
def get_macsec_interface():
    global interface_list
    global feature_list
    macsec_intf = []
    macsec_breakout_intf = []
    feat_name = ['MACSEC(Advantage)']
    macsec_feature = {}
    cmd = "/pkg/bin/show_macsec_mka -m -i all 2>/dev/null"
    result = run_cmd(cmd)
    if not result["rc"]:
        breakoutlist = []
        intf_list = result['output'].strip().split('\n')
        for intf in intf_list:
            z = re.search("(\w+\/\d+\/\d+\/\d+)",intf)
            if z:
                breakout = re.search("(\w+\/\d+\/\d+\/\d+\/\d+)", intf)
                if breakout:
                    ret = get_breakout_intf(breakout, breakoutlist)
                    if (ret['rc'] == 1):
                        macsec_breakout_intf.append(ret['intfname'])
                    else :
                        continue
                else :
                    macsec_intf.append(z.group(1))

        macsec_intf = macsec_intf + macsec_breakout_intf
        macsec_intf =  get_bundle_intf(macsec_intf)
        macsec_intf = format_list(macsec_intf)
        macsec_intf = adv_intf_priority_list_exclusion(macsec_intf)
        if(len(macsec_intf) > 0):
            # repeat feature name equal to the number of interface
            feat_name = feat_name * len(macsec_intf)
            macsec_feature['name'] = feat_name
            macsec_feature['intf_list'] = macsec_intf
            card_info = get_card_info(macsec_intf)
            macsec_feature['card_info'] = card_info
            feature_list.append(macsec_feature)

def traffic_eng_configured():
    flag = False

    global RUNNING_CFG_FILE
    cmd = "cat %s | grep 'traffic-eng'" %(RUNNING_CFG_FILE) 
    result = run_cmd(cmd)
    if not result['rc']:
        if(re.search("traffic-eng",result['output'])):
            flag = True

    return flag

def service_layer_api_configured():
    flag = False

    if(os.path.exists('/pkg/bin/sl_show')):
        cmd = "/pkg/bin/sl_show -G | grep 'config on'"
        result = run_cmd(cmd)
        if not result['rc']:
            config_on = result['output'].split(':')[1].strip()
            if(config_on == 'YES'):
                flag = True
    return flag

def get_peer_scale():
    flag = False
    global ess_intf_list
    global feature_list
    feat_name = ['Peering-Scale(Advantage)']
    peer_scale_feature = {}
    traffic_eng_feature = {}
    service_layer_api = {}
    feat_name1 = ['Traffic-Engineering(Advantage)']
    feat_name3 = ['Service-Layer(Advantage)']

    card_info = get_card_info(ess_intf_list)
    if(traffic_eng_configured()):
        feat_name1 = feat_name1 * len(ess_intf_list)
        traffic_eng_feature['name'] = feat_name1
        traffic_eng_feature['card_info'] = card_info
        traffic_eng_feature['intf_list'] = ess_intf_list
        feature_list.append(traffic_eng_feature)
        flag = True

    elif(service_layer_api_configured()):
        feat_name3 = feat_name3 * len(ess_intf_list)
        service_layer_api['name'] = feat_name3
        service_layer_api['card_info'] = card_info
        service_layer_api['intf_list'] = ess_intf_list
        feature_list.append(service_layer_api)
        flag = True

    else:
        feat_name = feat_name * len(ess_intf_list)
        cmd = "show_ipv4_rib -X 0x1 -Y 0x1 -Z ________ -V ________ -S ipv4 ________"
        result = run_cmd(cmd)
        if not result['rc'] :
            routes = result['output'].strip().splitlines()
            for route in routes:
                if "Total" in route:
                    z = re.match("(Total)\s+(\d+)",route)
                    if z:
                        total_routes = int(z.group(2))
                        if(total_routes > 512000):
                            peer_scale_feature['name'] = feat_name
                            peer_scale_feature['card_info'] = card_info
                            peer_scale_feature['intf_list'] = ess_intf_list
                            feature_list.append(peer_scale_feature)
                            flag = True

        cmd1 = "show_ipv6_rib -X 0x2 -Y 0x1 -Z ________ -V ________ -S ipv6 __________"
        result = run_cmd(cmd1)
        if not result['rc']:
            routes = result['output'].strip().splitlines()
            for route in routes:
                if "Total" in route:
                    z = re.match("(Total)\s+(\d+)",route)
                    if z:
                        total_routes = int(z.group(2))
                        if(total_routes > 50000):
                            peer_scale_feature['name'] = feat_name
                            peer_scale_feature['card_info'] = card_info
                            peer_scale_feature['intf_list'] = ess_intf_list
                            feature_list.append(peer_scale_feature)
                            flag = True

    return flag

def remote_imcast_route_configured():
    cmd = "/pkg/bin/evpn_show 0x7 | grep 'Number of Remote IMCAST Routes'"
    flag = False

    result = run_cmd(cmd)
    if not result["rc"]:
        num_imcast_route = int(result['output'].strip().split(':')[1])
        if num_imcast_route >  0:
            flag = True
    return flag

def active_vpls_pw_to_remote_configured():
    cmd = "/pkg/bin/l2vpn_show 0xa | grep 'Number of PWs'"
    flag = False

    result = run_cmd(cmd)
    if not result["rc"]:
        out = result['output'].split(' ')
        idx = int(out.index('Up:')+1)
        active_vpls = int(out[idx].strip(','))
        if(active_vpls > 0):
            flag = True
    return flag

'''
Example of output

sh iccp group
Tue Dec 18 10:35:52.668 EST
Redundancy Group 105

member ip:10.10.10.10 (SRv6-HUB10), up (connected)
monitor: route-watch (up)
backbone interface BE5001: up
enabled applications: PseudoMLACP
isolation recovery delay timer: 30 s, not running

enabled applications can be either PseudoMLACP or mLACP or mLACP, PseudoMLACP
'''
def iccp_group_enabled_application_configured():
    flag = False

    if(os.path.exists('/pkg/bin/iccp_show')):
        cmd = "/pkg/bin/iccp_show -t 0x1 | grep 'enabled applications'"
        result = run_cmd(cmd)
        if not result['rc']:
            output = result['output'].strip().split('\n')
            for line in output:
                iccp_group = line.split(':')[1].strip().split(',')[0]
                if (iccp_group == 'PseudoMLACP' or
                    iccp_group == 'mLACP'):
                    flag = True
    return flag

def evpn_ethernet_segment_configured():
    flag = False

    cmd = "/pkg/bin/evpn_show -d 0x3 | grep 'Operational'"
    result = run_cmd(cmd)
    if not result['rc']:
        evpn_detail = result['output'].split(':')[1].strip()
        if(evpn_detail == 'MH'):
            flag = True
    return flag

def get_evpn_interface():
    global feature_list
    evpn_intf = []
    evpn_feature  = {}
    bridge_domain_name = []
    feat_name = ['EVPN(Advantage)']
    cmd = "/pkg/bin/evpn_show -d 0x5 | grep EVPN | grep MPLS"

    result = run_cmd(cmd)
    if not result['rc']:
        output = result['output'].strip().split('\n')
        for line in output:
            bridge_domain_name.append(line.split()[2])

    if (len(bridge_domain_name) > 0):
        for bd in bridge_domain_name:
            cmd = "l2vpn_show -b "+ bd + " 0x9"
            result = run_cmd(cmd)
            if not result['rc']:
                intf_list = show_l2vpn_bridge_domain(result['output'])
                evpn_intf = evpn_intf + intf_list
    evpn_intf = list(set(evpn_intf))
    evpn_intf = format_list(evpn_intf)
    evpn_intf = adv_intf_priority_list_exclusion(evpn_intf)
    if(len(evpn_intf) > 0):
        feat_name = feat_name * len(evpn_intf)
        evpn_feature['name'] = feat_name
        evpn_feature['intf_list'] = evpn_intf
        card_info = get_card_info(evpn_intf)
        evpn_feature['card_info'] = card_info
        feature_list.append(evpn_feature)

def get_vpls_interface():
    global feature_list
    vpls_feature = {}
    vpls_intf_new = []
    feat_name = ['VPLS(Advantage)']

    cmd = "l2vpn_show 0x9"
    result = run_cmd(cmd)
    if not result['rc']:
        vpls_intf = show_l2vpn_bridge_domain(result['output'])

        vpls_intf_new = adv_intf_priority_list_exclusion(vpls_intf)
        if(len(vpls_intf_new) > 0):
            feat_name = feat_name * len(vpls_intf_new)
            vpls_feature['name'] = feat_name
            vpls_feature['intf_list'] = vpls_intf_new
            card_info = get_card_info(vpls_intf_new)
            vpls_feature['card_info'] = card_info
            feature_list.append(vpls_feature)

def get_iccp_license_count():
    iccp_intf = []
    global feature_list
    iccp_feature = {}
    feat_name = ['ICCP(Advantage)']

    cmd = "/pkg/bin/bundlemgr_show -m all"
    result = run_cmd(cmd)
    if(result['rc'] == 0):
        lines = result['output'].strip().split('\n')
        for line in lines:
            if re.search(r"(Bundle-Ether)|(BE)",line):
                z = re.search(r"(Bundle-Ether\d+)|(BE\d+)",line)
                if z:
                    intf_in_bundle = get_intf_from_bundle(z.group())
                    iccp_intf = iccp_intf + intf_in_bundle
        iccp_intf = list(dict.fromkeys(iccp_intf))
        iccp_intf = format_list(iccp_intf)
        iccp_intf = adv_intf_priority_list_exclusion(iccp_intf)
    if(len(iccp_intf) > 0):
        feat_name = feat_name * len(iccp_intf)
        iccp_feature['name'] = feat_name
        iccp_feature['intf_list'] = iccp_intf
        card_info = get_card_info(iccp_intf)
        iccp_feature['card_info'] = card_info
        feature_list.append(iccp_feature)

def get_l2_services():

    is_remote_imcast_route_config      = remote_imcast_route_configured();
    is_active_vpls_pw_to_remote_config = active_vpls_pw_to_remote_configured();
    is_iccp_grp_enable_appln_config    = iccp_group_enabled_application_configured();
    is_evpn_ethernet_segment_config    = evpn_ethernet_segment_configured();

    if (not is_remote_imcast_route_config) and \
       (not is_active_vpls_pw_to_remote_config) and \
       (not is_iccp_grp_enable_appln_config) and \
       (not is_evpn_ethernet_segment_config):
       return  

    if  is_remote_imcast_route_config:
        get_evpn_interface()

    if is_active_vpls_pw_to_remote_config:
        get_vpls_interface()

    if is_iccp_grp_enable_appln_config or is_evpn_ethernet_segment_config:
        get_iccp_license_count()

    get_l2vpn_interface()

def remove_unsupported_card():
    global feature_list
    global ess_intf_list
    global lc_dict
    regex_lccpu = re.compile(r"/CPU(\d+)$")
    # check_lc_supported(lc_dict)
    for feature in feature_list:
        unsupported_indices = []
        # name, intf_list, card_info
        for i, card in enumerate(feature['card_info']):
            pid = lc_dict.get(card)
            if not pid:
                pid = lc_dict.get(regex_lccpu.sub(r"", card))
            if pid and pid in unsupported_pid_list:
                unsupported_indices.append(i)
        unsupported_indices.reverse()
        for i in unsupported_indices:
            popped = [feature['name'].pop(i),
                      feature['intf_list'].pop(i),
                      feature['card_info'].pop(i)]
            logger.info(" Unsupported feature, removing {}".format(popped))
        
def get_intf_size():
    logger.info("Entering Function " + get_intf_size.__name__)
    read_unsupported_pids()
    get_up_interface()
    if get_peer_scale() is False:
        get_l3vpn_interface()
        get_macsec_interface()
        get_l2_services()
    storeData(feature_list)
    loadListSize()

def get_all_feat_intf_size():
    logger.info("Entering Function " + get_all_feat_intf_size.__name__)
    loadData(True)

def get_all_feat_intf():
    logger.info("Entering Function " + get_all_feat_intf.__name__)
    loadData()


def get_essential_bandwidth():
    logger.info("Entering Function " + get_essential_bandwidth.__name__)
    loadBandwidthData("ESSENTIAL")

def get_essential_FHGig_bandwidth():
    logger.info("Entering Function " + get_essential_FHGig_bandwidth.__name__)
    loadFourHundredGigeBandwidthData("ESSENTIAL")

def get_advance_bandwidth():
    logger.info("Entering Function " + get_advance_bandwidth.__name__)
    loadBandwidthData("(Advantage)")

def get_advance_FHGig_bandwidth():
    logger.info("Entering Function " + get_advance_FHGig_bandwidth.__name__)
    loadFourHundredGigeBandwidthData("(Advantage)")

def initial_run_cfg_cleanup():
     logger.info("Entering Function " + initial_run_cfg_cleanup.__name__)
     global MISC_DISK1
     global TMP
     global VAR_LOG_LIC
     global RUNNING_CFG_FILE
     if os.path.exists(MISC_DISK1 + RUNNING_CFG_FILE):
         os.remove(MISC_DISK1 + RUNNING_CFG_FILE)

     if os.path.exists(TMP + RUNNING_CFG_FILE):
         os.remove(TMP + RUNNING_CFG_FILE)
  
     if os.path.exists(VAR_LOG_LIC + RUNNING_CFG_FILE):
         os.remove(VAR_LOG_LIC + RUNNING_CFG_FILE)
    
def get_free_space_dir():
     logger.info("Entering Function " + get_free_space_dir.__name__)
     global MISC_DISK1
     global TMP
     global VAR_LOG_LIC

     statvfs = os.statvfs(MISC_DISK1)
     b = (statvfs.f_frsize * statvfs.f_bavail)/(1024*1024)
     if b > 20:
        return MISC_DISK1
     statvfs = os.statvfs(TMP)
     b = (statvfs.f_frsize * statvfs.f_bavail)/(1024*1024)
     if b > 20:
        return TMP
     statvfs = os.statvfs(VAR_LOG_LIC)    
     if b > 20:
        return VAR_LOG_LIC
     return ''

def collect_running_cfg():
    logger.info("Entering Function " + collect_running_cfg.__name__)
    global running_cfg
    global RUNNING_CFG_FILE
    available_dir = get_free_space_dir()
    if available_dir:
       RUNNING_CFG_FILE = available_dir + RUNNING_CFG_FILE
    else :
       raise NotEnoughSpace 
    cmd = "nvgen -c -l 1 -t 1 -o 1 2>/dev/null"
    result = run_cmd(cmd)
    if (result['rc'] == 0):
       running_cfg = str(result['output'])
       f = open(RUNNING_CFG_FILE,"w")
       f.write(running_cfg)
       f.close()
       logger.info("Wrote runing config to file {}".format(RUNNING_CFG_FILE))
    else:
       logger.info("Could not collect running config")

def timeout_handler(signum, frame):
    logger.info("Timeout! Exiting...")
    raise TimeOut 

if __name__ == '__main__':

    logger = logging.getLogger('sl_logger')
    logger.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s::  %(message)s',"%Y-%m-%d %H:%M:%%%S")
    formatter_console = logging.Formatter('%(asctime)s %(message)s',"%b %d %H:%MM:%S")
    logdir = os.path.dirname(LOGFILE)
    if not os.path.exists(logdir):
        os.makedirs(logdir)

    handler = logging.handlers.RotatingFileHandler(LOGFILE, maxBytes=(1024*100),backupCount=3)
    handler.setLevel(logging.DEBUG)
    handler.setFormatter(formatter)
    logger.addHandler(handler)

    signal.signal(signal.SIGALRM, timeout_handler)
    #Keeping a timeout of 30 mins.
    signal.alarm(1800)
    try:
       initial_run_cfg_cleanup()
       if (len(sys.argv) > 2 and (re.search("no_running_cfg",sys.argv[2]))):
            logger.info("No running config collected")
       else:
            collect_running_cfg()
       function = eval(sys.argv[1])
       function()
    except TimeOut:
          logger.info("Script Timed out while collecting config")
    except NotEnoughSpace:
          logger.info("No free space available to write config file!")
    finally:
           if os.path.exists(RUNNING_CFG_FILE):
               os.remove(RUNNING_CFG_FILE)
