#! /usr/bin/python
#
# Copyright (c) 2020 by cisco Systems, Inc.
# All rights reserved.
#
# This script will be invoked like this:
# Enable:  /pkg/bin/single_router_control_script.py --node 1.1.1.1 --enable TenGigE0_0_0_0
# Disable: /pkg/bin/single_router_control_script.py --node 1.1.1.1 --disable TenGigE0_0_0_0
#
# --node <ipaddr> indicates target LC node's IP address
# --enable  <if_name> indicates which l1port with given if_name should be enabled
# --disable <if_name> indicates which l1port with given if_name should be enabled
#
# Enable case
# 1. Query target node of rcy port number
#    ssh 1.1.1.1 /pkg/bin/single_router_control -r 100
#    This will print single line
#    > 0x32
# 2. Save output value into json
#    Format is shown in this file, %s to be replaced by above result "0x32"
# 3. Enable single router mode on LC
#    ssh 1.1.1.1 /pkg/bin/single_router_control -e 100 -q
# 4. Double check single router mode is on
# 5. Start pktio process, save pid into /var/run/pktio.100.pid
#
# Disable case
# 1. Kill pktio process, pid can be picked up from /var/run/pktio.100.pid
# 2. Disable single router mode on LC
#    ssh 1.1.1.1 /pkg/bin/single_router_control -d 100 -q
# 3. Check single router mode is off
#

import argparse
import sys
import subprocess
import json
import os
import signal

SSH_CMD="ssh -o UserKnownHostsFile=/dev/null -o LogLevel=ERROR"

def debug_print(in_args, text):
    if in_args.debug:
        print("%s" % text)

# Build dictionary for further parsing
def build_status(in_args):
    """
TenGigE0_0_0_29,360,disabled(default),168,30,0,1
if_name,ifh,enable-disable,rcy,pp_port,npu_id,is_j2
<snip>
    """
    d = {}
    cmd = "%s %s /pkg/bin/single_router_control -L" % (SSH_CMD, in_args.node)
    debug_print(in_args, "Collecting node's status: [%s]" % (cmd))
    proc = subprocess.Popen(cmd.split(), stdout=subprocess.PIPE)
    for line in iter(proc.stdout.readline,''):
        (if_name, ifh, en_dis, rcy, pp_port_id, npu_id,is_j2) = line.split(',')
        d[if_name] = {"ifh" : int(ifh),
                      "en_dis" : en_dis,
                      "rcy_port_id" : int(rcy),
                      "pp_port_id" : int(pp_port_id),
                      "npu_id" : int(npu_id),
                      "is_j2" : int(is_j2)}
    return d

def gen_json(slot, intf, if_name):
    # This is json file for J1 card, passed to single_router_packet_proxy process
    # <json_format_j1>
    #
    # 5 parameters to fill
    # Dst MAC for TX injection header "0x4e", "0x41", "0x50", "0x00", "%s", "%s"   << 0x0x:0x1y, x=(slot), y=0x10+(npu+1)
    # Src MAC for TX injection header "0x4e", "0x41", "0x50", "0x00", "%s", "0x01" << 0x11 for RP0, 0x12 for RP1
    # Inject pp_port                  "0x80", "%s"                                 << pp_port of l1port
    # Effective rcy_port              "data"   : [ "%s" ]                          << rcy_port of l1port

    rcy_port_id = intf['rcy_port_id']
    pp_port_id = intf['pp_port_id']
    npu_id = intf['npu_id']
    rp_id = "0x11"
    npu_type_is_j2 = intf['is_j2']

    json_format_template="""
    {
         "tx_embed"  : [
                         {
                           "offset" : 0,
                           "len"    : 16,
                           "data"   : [ "0x4e", "0x41", "0x50", "0x00", "0x%02x", "0x%02x",
                                        "0x4e", "0x41", "0x50", "0x00", "%s", "0x01",
                                        "0x87", "0x6F",
                                        "0xa0", "0x%02x" ]
                         }
                       ],
         "rx_match"  : [
                         {
                           "offset" : %d,
                           "len"    : 1,
                           "data"   : [ "0x%02x" ]
                         }
                       ],
         "rx_decap_length"  : %d
    }"""

    if npu_type_is_j2:
        rx_match_offset = 42
        rx_decap_len = 49
    else:
        rx_match_offset = 36
        rx_decap_len = 43
    json_format = json_format_template % (
                                          (0x00+int(slot)), (0x10+int(npu_id)+1),
                                          rp_id,
                                          int(pp_port_id),
                                          rx_match_offset,
                                          int(rcy_port_id),
                                          rx_decap_len)

    j = json.loads(json_format)

    with open("/tmp/single_router.%s" % (if_name), "w") as f:
        json.dump(j, f, indent=2)

def handle_proxy_process(in_args, intf, if_name):
    # enable case, start process
    print ("handle_proxy_process")
    config_file = "/tmp/single_router.%s" % (if_name)
    state_file = "/var/run/single_router.%s.pid" % (if_name)
    pid = 0
    if in_args.enable:
        # enable case
        try:
            from subprocess import DEVNULL
        except:
            DEVNULL = open(os.devnull, 'r+b')

        if in_args.debug:
            dbgoutput = "/tmp/proxy.%s" % (if_name)
            DEVNULL = open(dbgoutput, "w+")
            debug_print(in_args, "Debug-mode: output will be saved into %s, please don't run forever" % (dbgoutput))

        veth = in_args.north_bound_intf
        tx_intf='eth-vf2' if 'eth-vf2' in  os.listdir('/sys/class/net/') else 'ps-inb.1538'
        rx_intf='ps-inb.1778' if 'ps-inb.1778' in  os.listdir('/sys/class/net/') else 'eth-vf5'
        cmd = "ip netns exec xrnns /pkg/bin/single_router_pkt_proxy -n %s -s %s -r %s -i %s" % (veth, tx_intf, rx_intf,  config_file)
        print cmd
        p = subprocess.Popen(cmd.split(), stdout=DEVNULL, stderr=DEVNULL)
        pid = p.pid
        with open(state_file, "w") as f:
            f.write("%s" % (pid))
        print ("Started proxy process %d" % (pid))
    else:
        # disable case
        with open(state_file, "r") as f:
            pid = int(f.readline())
        try:
            os.kill(pid, signal.SIGTERM)
            print ("Killed proxy process %d" % (pid))
        except:
            print ("Failed to kill %d, please check the status yourself" % (pid))

def parse_input():
    parser=argparse.ArgumentParser(description='SingleRouterControlScript')
    parser.add_argument('--enable', type=str)
    parser.add_argument('--disable', type=str)
    parser.add_argument('--slot', type=str)
    parser.add_argument('--node', type=str)
    parser.add_argument('--north_bound_intf', type=str)
    parser.add_argument('--list', default=False, action='store_true')
    parser.add_argument('--singleline', default=False, action='store_true')
    parser.add_argument('--debug', default=False, action='store_true')

    return parser.parse_args(sys.argv[1:])

def main():
    intf = {}
    if_name = ""
    en_dis_cmd = ""
    ifh = ""
    in_args = parse_input()

    if in_args.node == None:
        print ("Node not provided, quit. [args: %s]" % (sys.argv))
        sys.exit(0)

    if in_args.enable:
        if_name = in_args.enable
    elif in_args.disable:
        if_name = in_args.disable
    else:
        if in_args.list:
            # this is pretty list command
            cmd = "%s %s /pkg/bin/single_router_control -l" % (SSH_CMD, in_args.node)
        else:
            # this is singleline list command
            cmd = "%s %s /pkg/bin/single_router_control -L" % (SSH_CMD, in_args.node)
        debug_print(in_args, "show status: [%s]" % (cmd))
        subprocess.call(cmd.split())
        sys.exit(0)

    # Let's build this node's single_router status
    lc_state = build_status(in_args)

    try:
        intf = lc_state[if_name]
    except:
        print("%s is not available on this node, abort." % (if_name))

    # Dump json file for enable case
    en_dis = intf['en_dis']
    ifh = intf['ifh']
    if in_args.enable:
        en_dis_cmd = "-e"
        gen_json(in_args.slot, intf, if_name)
    else:
        en_dis_cmd = "-d"

    # Skip duplicate command, or let it go through anyways? Operations are idempotent
    if in_args.enable and en_dis == 'enabled':
        print("Single-router mode already enabled for [%s]" % (if_name));
        sys.exit(0)

    if in_args.disable and en_dis == 'disabled(default)':
        print("Single-router mode already disabled for [%s]" % (if_name));
        sys.exit(0)


    # Enable/Disable Single Router mode on physical interface on LC
    cmd = "%s %s /pkg/bin/single_router_control %s %d -q" % (SSH_CMD, in_args.node, en_dis_cmd, ifh)
    debug_print(in_args, "Execute [%s]" % (cmd))
    proc = subprocess.call(cmd.split())

    # Start proxy process
    handle_proxy_process(in_args, intf, if_name)

if __name__ == "__main__":
    main()


