#!/pkg/bin/perl
#-----------------------------------------------------------------------------
# packet_drops.pl - Script to collect drop-counters in packet path
#
# Copyright (c) 2020-2021 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------------------------------

############# GLOBAL Definitions ####################
# Drop file list processing
my $GRAMMER_FILE = "packet_drops.list";
my @drop_file_list = ( "/pkg/etc/$GRAMMER_FILE" );  # File containing grammer for drop-commands
my $command_running = 0;
my $command_completed = 0;
my %cmd_hash = { };   # Database built after parsing the grammer
my $module_found = 0;

# Command specific variables
my $cmd_name = "";
my $group_name = "";
my $drop_string = "";
my $drop_value = "";

my %old_counters = { };  #DB of counters from last read

# Argument processing
$num_args = scalar @ARGV;
if ($num_args != 5) { 
   die "Usage: $0 <module> <ongoing on/off> <loc nodeID> <ver ON/OFF> <eXR>\n";
}


my $module = $ARGV[0];
my $ongoing = 0;
my $nodes = $ARGV[2];
my $verbose = 0;
my $self_test = 0;  # Test the output and highlight all patterns that  match
my $platform = $ARGV[4];

my $user_path = "/disk0a:/usr";
if ($platform eq "eXR"){
    $user_path = "/disk0:/usr";
}

if (-e "${user_path}/$GRAMMER_FILE") {
    print "Found grammer file $GRAMMER_FILE at $user_path. \n";
    @drop_file_list = ( "${user_path}/$GRAMMER_FILE" );
}
if ($module eq "self_test") {
    $module = "all";
    $self_test = 1;
}


if ($ARGV[1] eq "ON") {
    $ongoing = 1;  #Print counter's delta since last read
    #print "Running in ongoing mode";
}

if ($ARGV[3] eq "ON") {
   $verbose = 1;
   print "Entering verbose mode...\n";
}



############# Subroutines #####################
# Trim whitespaces from left and right
sub trim($)
{
	my $string = shift;
	$string =~ s/^\s+//;
	$string =~ s/\s+$//;
	return $string;
}

sub stdlog
{
    if ($verbose == 1) {
        print $_[0];
    }
}

sub node_name_to_id
{
    my $nodename = $_[0];
    my $node_rsi = `node_conversion -N $nodename`;
    my $node_id = `node_conversion -i $node_rsi`;
    return $node_id;
}


sub find_and_print_drops
{
    my $node_id = $_[0];
    my $nodename = $_[1];

    if ($self_test == 1) {
        print "\n\n=================================\n";
        print "Running self-test on $nodename\n";
        print "=================================\n";
    } elsif ($module ne "commands") {
       
        my $ongoing_text = "";
        if ($ongoing) {
            $ongoing_text = " ongoing";
        }
        print "\n\n=====================================\n";
        print "Checking for$ongoing_text drops on $nodename\n";
        print "=====================================\n";
    }

    #In ongoing mode, make a hash of old values
    my $drop_values_file = "/tmp/packetOldDropValues_${node_id}";
    if ($ongoing) {
        open OLD_VALUES, "<$drop_values_file";
        
        while (<OLD_VALUES>) {
            if (/^(.*):(\d+)\s*$/) {
                $key = trim($1);
                $old_counters{"$key"} = trim($2);
            }
        }
        close OLD_VALUES;
        
        stdlog "Dumping old-counters hash\n"; 
        stdlog "$_ $old_counters{$_}\n" for (keys %old_counters);
    }
    
    #Delete any existing old-counters file and replace with new counter values
    unlink $drop_values_file;
    open NEW_VALUES, ">$drop_values_file";
    
    foreach $drop_file (@drop_file_list) {
    
        # Open the file containing grammer for commands and their output
        # interpretations.
        open CMDS, "<$drop_file";
        stdlog "Opening $drop_file\n";
    
        while (<CMDS>) {
    
            $_ =~ s/(^.*?)#(.*)$/\1/;  #Limit line scope till any (#) char
        
            if (/\[commandstart\]/) {
                # Starting a new command. Flush previous data
                $command_running = 1;
                %cmd_hash = { };
                stdlog "Found command start, cleared hash\n";
            } elsif (/\[commandend\]/) {
                $command_running = 0;
                $command_completed = 1;
                stdlog "Command data end. Hash built\n";
            } elsif ($command_running == 1 && /^(.*?)=(.*)$/) { #use lazy quantifier to extract key and value correctly 
                # Some command running, build hash
        
                $key = trim($1);
                $val = trim($2);
                $cmd_hash{$key} = $val;
                stdlog "Set $key = $val in cmd_hash\n";
            }
        
            if ($command_completed == 1) {
                # Command data collected, now run command and process output
        
                #Dump the hash for debug
                stdlog "Dumping cmd_hash:\n";
                stdlog "$_ $cmd_hash{$_}\n" for (keys %cmd_hash);
        
                #Reset flag for further use
                $command_completed = 0;
        
                # Run the command
                if (!exists($cmd_hash{"cmd_name"})) {
                    print "Could not find cmd_name. Skipping command.\n";
                    next;
                } else {
                    $cmd_name = $cmd_hash{"cmd_name"};
                }
        
                if (!exists($cmd_hash{'cmd_exec'})) {
                    print "Didnt find cmd_exec for cmd \"$cmd_name\". Skip \n";
                    next;
                }
    
                if (!exists($cmd_hash{"module"})) {
                    print "Module not found for command \"$cmd_name\". Skip \n";
                    next;
                }
    
                #Check if the module filter is applied
                if ($module eq "commands") {
                    # We are asked to just print all the commands
                    if ($module_found == 0) {
                        $module_found = 1;   #Declare module found
                        printf "%-25s %-15s\n", "Module", "CLI";
                    }
    
                    printf "%-25s %-15s\n", "[$cmd_hash{'module'}]", $cmd_name;
                    next;
                } elsif ($module ne "all" && $module ne $cmd_hash{"module"}) {
                    next;
                }
    
                $module_found = 1;
    
                if (exists($cmd_hash{"ksh_only"}) && $cmd_hash{"ksh_only"} eq "true") {
                    my $node_name = `node_conversion -e $node_id`;
                    chomp($node_name);
                    $cmd_hash{"cmd_exec"} = "iox_on $node_name " . $cmd_hash{"cmd_exec"};
                    stdlog "Ksh-only cmd:  $node_name (node-id:$node_id)\n";
                } else {
                    $cmd_hash{"cmd_exec"} =~ s/\$location/$node_id/;
                }
        
                # todo: Run the command and get the output
                stdlog "Running command: " . $cmd_hash{"cmd_exec"} . "\n";
                open (OUTPUT, $cmd_hash{"cmd_exec"} . " 2>&1 |");  
                #Pipe STDERR as well to STDOUT
        
                $group_name = $cmd_hash{"default_group"};
                $command_printed = 0;
                while (<OUTPUT>) {
                    
                    # Start processing command output here.
                    stdlog "Scanning $_";
                    my $input = $_;
    
        
                    # First check if its a group-name starting
                    if (exists($cmd_hash{"group"})) {
                        if (/$cmd_hash{"group"}/) {
                            $group_name = trim($1);
                            next;
                        }
                    }
        
                    # Check for drop string
                    my $count = 1;
                    my $self_test_print = 0;
                    while (exists($cmd_hash{"drop_regex${count}"})) {
        
                        if (!exists($cmd_hash{"drop_string_loc${count}"})) {
                            $cmd_hash{"drop_string_loc${count}"} = 1;
                        }
        
        
                        if (!exists($cmd_hash{"drop_val_loc${count}"})) {
                            $cmd_hash{"drop_val_loc${count}"} = 2;
                        }
                        
                        stdlog "Trying to match with " . $cmd_hash{"drop_regex${count}"} . "\n";
                        if (/$cmd_hash{"drop_regex${count}"}/) {
        
                            #Drop string matched
                            stdlog "Matched\n";
                            $drop_string = ${$cmd_hash{"drop_string_loc${count}"}};
                            $drop_val = trim(${$cmd_hash{"drop_val_loc${count}"}});
        
                            # Print non-zero drop counters only
                            if ($self_test == 1) {
                                # We are in self-test mode. 
                                #Print all matches even if drop-counter is 0
                                if (!$command_printed) {
                                    $command_printed = 1;
                                    print "\n\n$cmd_name:\n";
                                }
                                #print "******[" . $cmd_hash{"module"} . ":$group_name] $drop_string: $incr$drop_val\n";
                                # Omit the patterns that we are already matching
                                $self_test_print = 1;
    
                            } elsif ($drop_val > 0) {
    
                                my $incr = "";
                                my $key = trim("[" . $cmd_hash{"module"} . ":$group_name] $drop_string");
                                print NEW_VALUES "$key:$drop_val\n";
                                
                                if ($ongoing == 1) {
                                    
                                    if (exists($old_counters{"$key"})) {
                                        $old_drop_value = $old_counters{"$key"};
                                        stdlog "Found old-counter:$old_drop_value, new:$drop_val";
    
    
                                        if ($old_drop_value <= $drop_val) {
                                            $incr = "+";
                                            $drop_val = $drop_val - $old_drop_value;
                                        }
                                    }
                                }
    
                                        
                                if ($drop_val > 0) {
                                    
                                    if (!$command_printed) {
                                        $command_printed = 1;
                                        print "\n\n$cmd_name:\n";
                                    }
            
                                    print "[" . $cmd_hash{"module"} . ":$group_name] $drop_string: $incr$drop_val\n";
                                }
                            }
                            break;
                        }
        
                        $count++;
                    }
                    if (!$self_test_print && $self_test) {
                        if (!$command_printed) {
                            print "\n\n$cmd_name:\n";
                            $command_printed = 1;
                        }
                        print "$input";
                    }
                }
        
                close OUTPUT;
            }
        }
        
        close CMDS;
    }

    if ($module_found == 0) {
        # A filter applied for a module did not match any command block in the list.
        # Probably an incorrect filter string
        # This may give an impression that no drop-counter incremented
        # Alert the user.
        print "module $module did not match any entry\n";
    }
    print "\n\n";
    
    close NEW_VALUES;
}

if ($nodes eq "all") {
    # Need to run for all nodes
    if ($platform eq "eXR"){
        open (NODES, "show_platform_sysdb -v" . " 2>&1 |");   #Pipe STDERR as well to STDOUT
    }    
        
    while (<NODES>) {
        my $nodename = "";
        if ((/^\s*(\S+)\s+.*FINAL/) || (/^\s*(\S+)\s+.*IOS XR RUN/)) {
            $nodename = $1;
            my $nodeid = node_name_to_id $nodename;
            chomp($nodeid);
            stdlog "Finding drops on node $nodename (id:$nodeid)";
            find_and_print_drops($nodeid, $nodename);
        }
    }
    close NODES;
} else {
    # Single node-id specified
    my $nodename = "";
    if ($module ne "commands") {
       $nodename = `node_conversion -e $nodes`;
       chomp($nodename);
       stdlog "Finding drops on node $nodename (id:$nodeid)";
    }

    find_and_print_drops($nodes, $nodename);
}
