#! /usr/bin/env python
# debug-pending-delete.py
#
# Copyright (c) 2020-2021 by cisco Systems, Inc.
# All rights reserved.
#
#
# Usage
#   1. no argument: it will collect objects and run the logic
#   2. -f <file>: read objects from file and run the logic
#   3. -p : read objects from stdin (pipe)
#
# Examples
#  * Run from RP
#   a. show dpa object all location <> | file /harddisk:/obj-all
#   b. run this script with -f argument:
#      => debug-pending-delete.py -f /harddisk:/obj-all
#
#  * Run from LC
#   a. run this script without argument:
#      => debug-pending-delete.py
#   b. run this script through pipe:
#      => extract-obj-dump-from-showtech <showtech.tgz> | debug-pending-delete.py -p
#
import os
import subprocess
import sys
from collections import namedtuple
import argparse

USER_COLLECTED_FILE="/harddisk:/obj-all"
PROGRAM_COLLECTED_FILE="/tmp/obj-all"
SHOW_CLIENT_PROC="/pkg/bin/ofa_show_client"


# Trigger ofa_show_client -a -b > /tmp/obj-all to dump all objects
def collect_all_obj():
    cmd = [SHOW_CLIENT_PROC, "-a", "-b"]
    output = open(PROGRAM_COLLECTED_FILE, "w")
    p = subprocess.Popen(cmd, stdout=output)
    p.communicate()
    output.close()

def parse_args():
    parser = argparse.ArgumentParser(description="Delay delete debug script, run without option to collect data and analyze, or use below options to specify source")
    parser.add_argument('-f', action="store", dest="readfromfile", default="", help="Read from specified file and analyze")
    parser.add_argument('-p', action="store_true", dest="readfrompipe", default=False, help="Read from pipe and analyze")
    return parser.parse_args()

def main():
    # parse arguments
    parse_result = parse_args()

    # collect all object and write to /tmp/obj-all
    # or the data should be pre-collected at /harddisk:/obj-all
    if len(parse_result.readfromfile) > 0:
        print("Will use %s as collected object" % (parse_result.readfromfile))
        try:
            f = open(parse_result.readfromfile)
            print("%s found, using collected object" % (parse_result.readfromfile))
        except:
            print("%s not found, check filename and retry" % (parse_result.readfromfile))
    elif parse_result.readfrompipe == True:
        print("Wait for input from pipe, if you invoke it by accident, use ctrl-c to quit")
        f = sys.stdin
    else:
        print("Will collect object from OFA server")
        collect_all_obj()
        try:
            f = open(PROGRAM_COLLECTED_FILE)
            print("Collection completed")
        except BaseException:
            print("Couldn't collect object from OFA server")
            sys.exit(0)

    # Define namedtuple for the object
    parsed_obj = namedtuple('parsed_obj', ['table', 'handle', 'pending_delete', 'parents', 'children', 'quarantined'])

    # Phase1: detect child->parent
    # list up all objects
    #  capture table-name, handle, and ref-hdl to other object
    #  into namedtuple and then push it to all_obj dictionary
    all_obj = {}
    prev_obj = parsed_obj('dummy', 'dummy', '', [], [], False)   # This is dummy entry for first push to all_obj
    table = ""
    handle = ""
    pending_delete = ""
    parents = []
    children = []
    quarantined = False

    for line in f:
        if "element" in line:
            # Indicates next object
            # First flush previous object to all_obj dictionary
            # At this point children is always empty list
            prev_obj = parsed_obj(table, handle, pending_delete, parents, children, quarantined)
            all_obj[handle] = prev_obj

            # Reset variables for next object
            table = ""
            handle = ""
            pending_delete = ""
            parents = []
            children = []
            quarantined = False

            # Parse next object
            table = line.split(" ")[0]
            handle = line.split(" ")[3].split(":")[1].split(')')[0]

        if "pending(cr/up/dl)" in line and "0/0/0" not in line:
            pending_delete = line.strip()
        if ".refhdl" in line and "not set" not in line:
            parents.append(hex(int(line.split(" ")[4], 0)))
        if "quarantined" in line and "1" in line:
            quarantined = True

    last_obj = parsed_obj(table, handle, pending_delete, parents, children, quarantined)
    all_obj[handle] = last_obj

    f.close()

    # Phase2: connect parent->child
    # iterate over all_obj dict
    # for each 'parent' found, identify parent object
    #  then in parent object, insert child's refhdl into 'children' list
    for handle, child in all_obj.items():
        if len(child.parents) == 0:
            # Root object
            continue
        for parent_hdl in child.parents:
            if parent_hdl not in all_obj:
                print(
                    "parent not found for refhdl %s from obj (%s:%s) - %s" %
                    (parent_hdl, child.table, child.handle, child))
                break
            parent_obj = all_obj[parent_hdl]
            parent_obj.children.append(child.handle)

    def walk_child(obj, level):
        for child_hdl in obj.children:
            child = all_obj[child_hdl]
            print("%sChild: table=%s, handle=%s, quarantined=%s" %
                  ("  " * (level + 1), child.table, child.handle, child.quarantined))
            for next_child in all_obj[child.handle].children:
                walk_child(all_obj[next_child], level + 1)

    # Phase3: print result and next step
    suggest = 0
    print("Inspect %d objects in the system" % (len(all_obj)))
    for handle, obj in all_obj.items():
        if len(obj.pending_delete) > 0:
            suggest = 1
            print(
                "=> Potential pending delete object found:\n  table=%s, handle=%s (attr:%s) children=%d, quarantined=%s" %
                (obj.table, obj.handle, obj.pending_delete, len(obj.children), obj.quarantined))
            walk_child(obj, 1)

    if suggest:
        print
        print("Next step suggestion:")
        print("- Use 'show dpa object <table> object-handle <handle> location <loc>' to inspect pending/child object")
        print("- Dump show ofa trace and look up for <handle>")
    else:
        print("=> No pending delete object found")


if __name__ == "__main__":
    main()
    # Remove temp file
    try:
        os.remove(PROGRAM_COLLECTED_FILE)
    except OSError:
        pass
