#!/bin/bash
#
# update_idprom.sh: Script used to keep proto boards IDPROM updated
#
# May 2015, Chi-Heng Lu
#
# Copyright (c) 2015-2017 by Cisco Systems, Inc.
# All rights reserved.
#
ACT_UTIL=/opt/cisco/calvados/sbin/act_util
IDPROM_UTIL=/opt/cisco/calvados/sbin/ncs5500_idprom_tool
CMGR_CLIENT=/opt/cisco/calvados/sbin/card_mgr_client
CARDINFO_FILE=/tmp/cardinfo_$$.txt
IDPROM_TOOL_FRETTA=/opt/cisco/calvados/sbin/ncs5500_idprom_tool

#
# Functions to respond to the interactive diag tool when modifying an IDPROM
# field. Line with just return character indicates that we are not modifying
# the original value.
#
function lc_fix_6_jericho_pid {
$PROG_CMD << EOF








NC55-36X100G


















27000






















































y
EOF
}

function new_lc_fix_6_jericho_pid {
$PROG_CMD << EOF








NC55-36X100G


















27000





y
EOF
}

function lc_fix_2_jericho_pid {
$PROG_CMD << EOF








NC55-12X100GE-PROT


















27006






















































y
EOF
}

function new_lc_fix_2_jericho_pid {
$PROG_CMD << EOF








NC55-12X100GE-PROT


















27006





y
EOF
}


function lc_fix_1_jericho_pid {
$PROG_CMD << EOF








NC55-6X100GE-PROT


















27001






















































y
EOF
}

function new_lc_fix_1_jericho_pid {
$PROG_CMD << EOF








NC55-6X100GE-PROT


















27001





y
EOF
}


function lc_fix_6_jericho_cpu_pid {
$PROG_CMD << EOF








NC55-36X100G


















27000


















y
EOF
}

function lc_fix_2_jericho_cpu_pid {
$PROG_CMD << EOF








NC55-12X100GE-PROT


















27006


















y
EOF
}

function lc_fix_1_jericho_cpu_pid {
$PROG_CMD << EOF








NC55-6X100GE-PROT


















27001


















y
EOF
}

function rp_fix_pid {
$PROG_CMD << EOF








NC55-RP









































































y
EOF
}

function new_rp_fix_pid {
$PROG_CMD << EOF








NC55-RP
























y
EOF
}

function sup_a_fix_pid {
$PROG_CMD << EOF








NC55-RP-PROTO









































































y
EOF
}

function new_sup_a_fix_pid {
$PROG_CMD << EOF








NC55-RP-PROTO
























y
EOF
}

function sc_fix_pid {
$PROG_CMD << EOF








NC55-SC
























y
EOF
}

function fc_fix_pid {
$PROG_CMD << EOF








NC55-5508-FC
























y
EOF
}

function chassis_fix_pid {
$PROG_CMD << EOF








NCS-5508





































y
EOF
}

function clean_up {

    rm -f $CARDINFO_FILE
    rm -rf /diag/etc
}

# The main board IDPROM content is programmed into a ACT2 chip
# There are two copy of IDPROM in ACT2 chip that diag can access,
# one in the protected region and one in the unprotected region. Existing N9K
# KLM code only uses the unprotected region.
# But there is future plan to move all the access to the protected region, so
# we have to check and correct both regions
function check_n_correct_mb_idprom {
    local TARGET=$1

    echo -n "Reading mainboard $TARGET IDPROM ..."
 
    # NOTE: continue using diag act_util for programming the protected region
    # on x86 cards as the ncs5500_idprom_tool does not have yet support for
    # this.  
    if [ "$TARGET" == "protected" ]; then
        PROG_PID=`$IDPROM_TOOL_FRETTA -c show -t mb -a sec | grep -a "Product ID" | awk '{print $4}'`
    else
        PROG_PID=`$IDPROM_TOOL_FRETTA -c show -t mb -a unsec | grep -a "Product ID" | awk '{print $4}'`
    fi
    echo "[DONE]"

    if [ "$TARGET" == "protected" ]; then
        PROG_CMD="$IDPROM_TOOL_FRETTA -c program -t mb -a sec"
    else
        PROG_CMD="$IDPROM_TOOL_FRETTA -c program -t mb -a unsec"
    fi
    # Correct the PID by calling the right sub-function
    if [ "$VALID_PID" != "$PROG_PID" ]; then
        echo "Updating $CARDTYPE mainboard $TARGET IDPROM PID from '$PROG_PID' to '$VALID_PID' ..."
        if [ "$CARDTYPE" == "LC" ]; then
            if [ "$VALID_PID" == "NC5-36X100GE" ] ||
                [ "$VALID_PID" == "NC55-36X100G" ]; then
                if [ "$TARGET" == "protected" ]; then
                    lc_fix_6_jericho_pid
                else
                    new_lc_fix_6_jericho_pid
                fi
            elif [ "$VALID_PID" == "NC5-12X100GE-PROTO" ] || 
                [ "$VALID_PID" == "NC55-12X100GE-PROT" ]; then
                if [ "$TARGET" == "protected" ]; then
                    lc_fix_2_jericho_pid
                else
                    new_lc_fix_2_jericho_pid
                fi
            elif [ "$VALID_PID" == "NC5-6X100GE-PROTO" ] || 
                [ "$VALID_PID" == "NC55-6X100GE-PROT" ]; then
                if [ "$TARGET" == "protected" ]; then
                    lc_fix_1_jericho_pid
                else
                    new_lc_fix_1_jericho_pid
                fi
            else
                echo "ERROR: Don't know how to correct $CARDTYPE mainboard $TARGET IDPROM with '$VALID_PID' PID"
                clean_up
                exit 1
            fi
        elif [ "$CARDTYPE" == "RP" ]; then
            if [ "$VALID_PID" == "NC5-RP-PROTO" ] || 
                [ "$VALID_PID" == "NC55-RP-PROTO" ]; then
                if [ "$TARGET" == "protected" ]; then
                     sup_a_fix_pid
                else
                     new_sup_a_fix_pid
                fi
            else
                if [ "$TARGET" == "protected" ]; then
                     rp_fix_pid
                else
                     new_rp_fix_pid
                fi
            fi 
        elif [ "$CARDTYPE" == "SC" ]; then
            sc_fix_pid   
        elif [ "$CARDTYPE" == "FABRIC" ]; then
            fc_fix_pid
        fi
    else
        echo "$CARDTYPE mainboard $TARGET IDPROM has correct PID of '$VALID_PID' programmed. No update is required"
    fi
}

# LC has a daughtercard that has an EEPROM accessible directly via CPU to
# store the IDPROM content.
function check_n_correct_dc_idprom {
    echo -n "Reading CPU IDPROM ..."
    # Read that actual programmed PID in the CPU IDPROM
    PROG_PID=`$IDPROM_UTIL -c show -t dc -a unsec | grep -a PRODUCT_NUMBER| awk '{print $3}'`
    echo "[DONE]"
    PROG_CMD="$IDPROM_UTIL -c program -t dc -a unsec"
    if [ "$VALID_PID" != "$PROG_PID" ]; then
        echo "Updating $CARDTYPE CPU IDPROM PID from '$PROG_PID' to '$VALID_PID' ..."
        if [ "$VALID_PID" == "NC5-36X100GE" ] ||
           [ "$VALID_PID" == "NC55-36X100G" ]; then
            lc_fix_6_jericho_cpu_pid    
        elif [ "$VALID_PID" == "NC5-12X100GE-PROTO" ] ||
             [ "$VALID_PID" == "NC55-12X100GE-PROT" ]; then
            lc_fix_2_jericho_cpu_pid    
        elif [ "$VALID_PID" == "NC5-6X100GE-PROTO" ] ||
             [ "$VALID_PID" == "NC55-6X100GE-PROT" ]; then
            lc_fix_1_jericho_cpu_pid
        else
            echo "Don't know how to correct $CARDTYPE CPU IDPROM with '$VALID_PID' PID"\
                 ", It might not be valid or needed for this card"
            clean_up
            exit 1
        fi
    else
         echo "$CARDTYPE CPU IDPROM has correct PID of '$VALID_PID' programmed. No update is required"
    fi
}

# The backplane (aka chassis) IDPROM is accessible via master RP, so we have
# to check for mastership before accessing the content.
function check_n_correct_bp_idprom {
    local INSTANCE=$1
    local TARGET=$2

    echo -n "Reading Chassis instance $INSTANCE $TARGET IDPROM ..."
    # Get the valid chassis PID from card manager
    VALID_PID=`$CMGR_CLIENT -r -c chassisinfo | grep PID | awk '{print $4}'`
    echo "[DONE]"
    if [ "$TARGET" == "protected" ]; then
        PROG_PID=`$IDPROM_TOOL_FRETTA -c show -t chassis -a sec -i $INSTANCE | grep -a "Product ID" | awk '{print $4}'`
        PROG_CMD="$IDPROM_TOOL_FRETTA -c program -t chassis -a sec -i $INSTANCE"
    else
        PROG_PID=`$IDPROM_TOOL_FRETTA -c show -t chassis -a unsec -i $INSTANCE | grep -a "Product ID" | awk '{print $4}'`
        PROG_CMD="$IDPROM_TOOL_FRETTA -c program -t chassis -a unsec -i $INSTANCE"
    fi
    
    if [ "$VALID_PID" != "$PROG_PID" ] || [[ "$PROG_PID" == "N9K"* ]]; then
        echo "Updating Chassis instance $INSTANCE $TARGET IDPROM PID from '$PROG_PID' to '$VALID_PID' ..."
        chassis_fix_pid
    else
        echo "Chassis instance $INSTANCE $TARGET IDPROM has correct PID of '$VALID_PID' programmed. No update is required"
    fi
}

echo -n "Getting current card info ..."
# Get card info from card manager
$CMGR_CLIENT -n -c cardinfo > $CARDINFO_FILE
if [ $? -ne 0 ]; then
    echo ""
    echo "ERROR: Failed to get card info from card manager"
    exit 1
fi
echo "[DONE]"

# Get card type
CARDTYPE=`grep "Card Type" $CARDINFO_FILE | awk '{print $4}'`

# Get the valid PID from card info collected that will be used to compare with
# the PID programmed in the IDPROM
VALID_PID=`grep PID $CARDINFO_FILE | awk '{print $3}'`

# Since we are calling diag tool, create /diag/etc directory so the tool
# is happy to create log file there. We will clean this up before script exits
mkdir -p /diag/etc

# Check and correct the mainboard IDPROM. First do the uprotected region on 
# ACT2 chip, then the preotected one.
check_n_correct_mb_idprom "unprotected"
check_n_correct_mb_idprom "protected"

# If LC, then we need to check the CPU IDPROM
if [ "$CARDTYPE" == "LC" ]; then
    check_n_correct_dc_idprom
fi

# For RP we need to handle chassis IDPROM if the RP has IOFPGA mastership
if [ "$CARDTYPE" == "RP" ]; then
    HW_ARB=`$CMGR_CLIENT -n -c reg_hwarb | grep arbitration | awk '{print $7}'`
    if [ "$HW_ARB" == "Master" ]; then     
        check_n_correct_bp_idprom 1 "unprotected"
        check_n_correct_bp_idprom 1 "protected"
        check_n_correct_bp_idprom 2 "unprotected"
        check_n_correct_bp_idprom 2 "protected"
    else
        # Tell user that we are skipping chassis IDPROM here are we don't
        # have the mastership
        echo "Skipping check of Chassis IDPROM as this RP doesn't have mastership"
    fi
fi

clean_up
exit 0
