#!/bin/bash
# 
# lxc - libvirt pd hook script for platform fretta
# This script is added as a pd hack to pass no host interface from host to XR
# container for fretta
#
# Copyright (c) 2014-2021 by Cisco Systems, Inc.

. /etc/init.d/functions
. /etc/init.d/spirit-functions # populate_rootfs_vars
. /etc/init.d/pd-functions

. /etc/init.d/spirit_pd.sh
#platform_hw_profile_get_settings

# BFD VLAN 1554 moved from here
lc_intf_vlan_vigor=(
         "ps-eobc.3074"
         "spp-inb0.1282"
         "spp-inb1.1298"
    )

lc_intf_pcie_inb_vigor=(
         "spp-inb0.1538"
         "spp-inb1.1538"
         "spp-inb0.1794"
         "spp-inb1.1794"
         "spp-inb0.1778"
         "spp-inb1.1778"
    )

# Vigor-100-Base
lc_intf_vlan_vigor_100_base=(
         "ps-eobc.3074"
         "spp-inb0.1282"
    )

lc_intf_pcie_inb_vigor_100_base=(
         "spp-inb0.1538"
         "spp-inb0.1794"
         "spp-inb0.1778"
    )
 
lc_intf_vlan=(
         "ps-eobc.3074"
         "ps-inb.1282"
         "ps-inb.1298"
         "ps-inb.1314"
         "ps-inb.1330"
         "ps-inb.1346"
         "ps-inb.1362"
         "ps-inb.1538"
         "ps-inb.1554"
         "ps-inb.1570"
         "ps-inb.1794"
         "ps-inb.1778"
    )


rp_intf_vlan=(
         "ps-eobc.3074"
         "ps-inb.1538"
         "ps-inb.1794"
         "ps-inb.1778"
    )


#invoked with $1 $2 $3 $4 as received by the caller
function pd_handle_init_start()
{
   local vm_name=$1
   local func_exit_code=0

   case "$vm_name" in
    sysadmin)
        lxc_set_sysadmin_cpuset_hook $1 $2 $3 $4
	func_exit_code=$?
        ;;

    default-sdr--*)
        lxc_set_sdr_cpuset_hook $1 $2 $3 $4
	func_exit_code=$?
        ;;

    *)
        #Nothing to do
        ;;
    esac

    return $func_exit_code
}
readonly -f pd_handle_init_start

#invoked with $1 $2 $3 $4 as received by the caller
function lxc_set_sysadmin_cpuset_hook()
{
    local vm_name=$1
    if [ -d /dev/cgroup/cpuset/machine/$vm_name.libvirt-lxc ]; then
        log_msg "in lxc_set_sysadmin_cpuset_hook cp_cores $cp_cores"
        echo $cp_cores > /dev/cgroup/cpuset/machine/$vm_name.libvirt-lxc/cpuset..cpus
    else
        log_msg "in lxc_set_sysadmin_cpuset_hook cgroup dir not set"
        return 1
    fi

    return 0
}
readonly -f lxc_set_sysadmin_cpuset_hook

#invoked with $1 $2 $3 $4 as received by the caller
function lxc_set_sysadmin_intf_hook()
{
    local vm_name=$1
    nproc=`nproc`
    if [ "$nproc" -ge "12" ]; then
        cpumask="f00"
    elif [ "$nproc" -ge "8" ]; then
        cpumask="0f0"
    else
        cpumask="00f"
    fi

    get_board_type
    if [ "$BOARDTYPE" = "RP" ]; then
        echo $cpumask > /sys/devices/virtual/net/sysadmin-eth1/queues/rx-0/rps_cpus 
        echo $cpumask > /sys/devices/virtual/net/sysadmin-eth1/queues/tx-0/xps_cpus 
    else
        log_msg "Nothing to do for VM $vm_name"
    fi

    return 0
}
readonly -f lxc_set_sysadmin_intf_hook

#invoked with $1 $2 $3 $4 as received by the caller
function lxc_reclaim_sysadmin_intf_hook()
{
    local vm_name=$1
    log_msg "Nothing to do for VM $vm_name"

    return 0
}
readonly -f lxc_reclaim_sysadmin_intf_hook

#
# invoked with $1 $2 $3 $4 as received by the caller
#
# $1 : Name of the LXC, used in the XML.
#      eg: sysadmin/default-sdr--1
# $2 : Operation of the LXC.
#      (prepare/start/init_start/started/stopped/release/reconnect)
# $3 : Sub-operation of the LXC mentioned in $2
#      (begin/end)
# $4 : One additional argument.
#      For operation init_start, we get the PID of /sbin/init of container.
#
function lxc_reclaim_sdr_intf_hook()
{
    return 0
}

#invoked with $1 $2 $3 $4 as received by the caller
function lxc_set_sdr_cpuset_hook()
{
    local vm_name=$1
    if [ -d /dev/cgroup/cpuset/machine/$vm_name.libvirt-lxc ]; then
        log_msg "in lxc_set_sdr_cpuset_hook cp_cores $cp_cores"
        echo $cp_cores > /dev/cgroup/cpuset/machine/$vm_name.libvirt-lxc/cpuset..cpus
    else
        log_msg "in lxc_set_sdr_cpuset_hook cgroup dir not set"
        return 1
    fi
  
    return 0
}
readonly -f lxc_set_sdr_cpuset_hook

function lxc_set_sdr_intf_lc_hook()
{
    return 0
}

function lxc_set_sdr_intf_rp_hook()
{
    local namespace=$1

    nproc=`nproc`
    if [ "$nproc" -ge "12" ]; then
        cpumask="f00"
    elif [ "$nproc" -ge "8" ]; then
        cpumask="0f0"
    else
        cpumask="00f"
    fi

    echo $cpumask > /sys/devices/virtual/net/xr-eth1/queues/rx-0/rps_cpus 
    echo $cpumask > /sys/devices/virtual/net/xr-eth1/queues/tx-0/xps_cpus 
}

#
# pass interfaces to xr, invoked with $1 $2 $3 $4 as received by caller
#
# $1 : Name of the LXC, used in the XML.
#      eg: sysadmin/default-sdr--1
# $2 : Operation of the LXC.
#      (prepare/start/init_start/started/stopped/release/reconnect)
# $3 : Sub-operation of the LXC mentioned in $2
#      (begin/end)
# $4 : One additional argument.
#      For operation init_start, we get the PID of /sbin/init of container.
#
function lxc_set_sdr_intf_hook()
{
    get_board_type
    if [ "$BOARDTYPE" = "LC" ]; then
        lxc_set_sdr_intf_lc_hook $4
    elif [ "$BOARDTYPE" = "RP" ]; then
        lxc_set_sdr_intf_rp_hook $4
    else
        return 1
    fi

    return 0
}

#
# pd_unmount_lxc_rootfs
#   PI counter part is present in lxc_hook.sh
#
function pd_unmount_lxc_rootfs ()
{
    local vm_name=$1
    local func_exit_code=0

    case "$vm_name" in
    sysadmin|default-sdr--*)
        # Do not unmount the rootfs for XC and FC since these are membooted.
        # If we unmount it, then we need to do pxe_install_memboot, which
        # is what we do on card boot.
        # Besides we don't have any /dev/loop devices on XC and FC.
        get_board_type
        if [ "$BOARDTYPE" != "XC" ] && [ "$BOARDTYPE" != "FC" ]; then
            unmount_lxc_rootfs "$vm_name"
            func_exit_code=$?
        fi
        ;;

    *)
        log_err "pd_unmount_lxc_rootfs: VM $vm_name, not supported"
        ;;

    esac

    return $func_exit_code
}


#do_inb_intf_cleanup_v2
# $1 == Interface Index
# $2 == PCI Bus No.
#Routine to do force cleanup of Inband interfaces
#and Vlans on J2 card
#
function do_inb_intf_cleanup_v2()
{
    log_msg "Removing spp-inb$1 vlan and interface"
    vconfig rem spp-inb$1.1538
    vconfig rem spp-inb$1.1570
    vconfig rem spp-inb$1.1554
    vconfig rem spp-inb$1.1794
    vconfig rem spp-inb$1.1778
    ifconfig spp-inb$1 down
    if [ $1 -eq 0 ]; then
        ifconfig bcm-$2 down
        python /usr/sbin/bcm-knet-init.py stop 0x$2 spp-inb$1
        if [ $? -ne 0 ]; then
            log_msg "Error: Removing spp-inb$1 vlan and interface"
        fi
    fi
    sleep 2
    log_msg "Success: Removal of spp-inb$1 vlan and interface"
}

# Deprecated use do_inb_intf_cleanup_v2 instead
#Routine to do force cleanup of Inband interfaces
#and Vlans on J2 card 
#
function do_inb_intf_cleanup()
{
    for i in 0 1
    do
        log_msg "Removing spp-inb$i vlan and interface"
        vconfig rem spp-inb$i.1538
        vconfig rem spp-inb$i.1570
        vconfig rem spp-inb$i.1554
        vconfig rem spp-inb$i.1794
        ifconfig spp-inb$i down
        if [ $i -eq 0 ]; then
            ifconfig bcm-07 down
            python /usr/sbin/bcm-knet-init.py stop 0x07 spp-inb0 
            if [ $? -ne 0 ]; then
                log_msg "Error: Removing spp-inb$i vlan and interface"
            fi
        else
            ifconfig bcm-09 down
            python /usr/sbin/bcm-knet-init.py stop 0x09 spp-inb1
            if [ $? -ne 0 ]; then
                log_msg "Error: Removing spp-inb$i vlan and interface"
            fi
        fi
        sleep 2
        log_msg "Success: Removal of spp-inb$i vlan and interface"
    done
}  
#
# Small routine to check if the Vigor inband interface exists
#
function check_vigor_spp_intf_exists ()
{
    if [ -d /sys/devices/virtual/net/$1 ]; then
        return 0
    fi
    return 1
}

#
# wait_for_vigor_spp_intf_rem
#  Bridge and inband interfaces removal has to be happened 
#  before PCI device removal
#  This Routine to use as handshake method to identify Vigor Inband
#  interface removal. Once Inband interfaces got revoed then  
#  it will break loop and proceed with PCI device removal 
#
function wait_for_vigor_spp_intf_rem()
{
    local MAX_RETRY=90
    local retry_count=0
    local intf_list=("${!1}")
    for i in "${intf_list[@]}";
    do
        while [ $retry_count -lt $MAX_RETRY ]
        do
            check_vigor_spp_intf_exists $i
            if [ $? -ne 1 ]; then
                log_msg "Inband Intf not removed yet:$retry_count intf:$i"
                let retry_count+=1
                if [ $retry_count -ge $MAX_RETRY ]; then
                    log_msg "FATAL ERROR TIME's UP waiting for Inband Interface deletion"
                    log_msg "removal of PCI device before spp-inb vlan may cause kernel panic"
                    return 1
                fi
                sleep 1
            else
                log_msg "Inband interface:$i is removed already" 
                break;
            fi
        done
    done
    return 0
}

readonly -f pd_unmount_lxc_rootfs

IOFPGA_HOST_EVENT_REG_ADDR=0x18
DM2_PRESHUTDOWN_NOTIF=1
# Steps to bring down the device
# 0. Notify card_mgr about DM2 going down
# 1. rmmod of the kernel module
# 2. remove the devices from the pci heirarchy
# 3. Power down the device
function lxc_xr_lc_disable_dnx_devices()
{
    #
    # Before turning off the power domain 2, we need to notify card_mgr
    # via scratch register about this, so sensors on DM2 that is currently
    # being monitored could be stopped to prevent getting bogus reading
    # when the power is turned off.
    # We will give card_mgr 2 seconds timeout to process this message.
    local retry=0
    local reg_val=$DM2_PRESHUTDOWN_NOTIF
    local SI5346_REG=0x410
    log_msg "Notifying card_mgr about DM2 pre-shutdown event ..."
    init_iofpga_base_addr
    iofpga_reg_write $IOFPGA_HOST_EVENT_REG_ADDR $reg_val
    until [ $reg_val -eq 0 ]; do
        reg_val=$(iofpga_reg_read $IOFPGA_HOST_EVENT_NOTIF_REG_ADDR)
        reg_val=$(($reg_val & 0xF))
        if [ $reg_val -ne 0 ]; then
            retry=$((retry + 1))
            if [ $retry -gt 20 ]; then
                log_err "Timeout waiting for card_mgr to respond to DM2 pre-shutdown notification"
                break;
            fi
            sleep 0.1
        else
            log_msg "card_mgr responded to DM2 pre-shutdown notification in $((retry * 100)) ms"
        fi
    done


    get_board_type
    if [ "$BOARDTYPE" = "LC" ]; then
       
        get_card_index

        log_msg "Got Card Index CARDINDEX: $CARDINDEX"
  
        # Vigor card index is 27044
        if [ "$CARDINDEX" = "27044" ] ||
           [ "$CARDINDEX" = "27050" ] ||
           [ "$CARDINDEX" = "27046" ] ||
           [ "$CARDINDEX" = "27055" ]; then
           wait_for_vigor_spp_intf_rem lc_intf_pcie_inb_vigor[@] 
           if [ $? -ne 0 ]; then
               log_msg "Doing Force Inband interface cleanup"
               #do_inb_intf_cleanup
               do_inb_intf_cleanup_v2 0 07
               do_inb_intf_cleanup_v2 1 09
           fi
           sleep 1
        fi
        # Vigor-100-Base
        if [ "$CARDINDEX" = "27061" ] ||
           [ "$CARDINDEX" = "27062" ] ||
           [ "$CARDINDEX" = "27063" ] ||
           [ "$CARDINDEX" = "27066" ] ||
           [ "$CARDINDEX" = "27067" ] ||
           [ "$CARDINDEX" = "27068" ] ; then
           # Reset clock to NPU from internal oscillator
           # for next boot as Redwood is getting reset
           iofpga_reg_write_bar1 $SI5346_REG 0x2
           log_msg "Reset clock-output to NPU from internal oscillator"
           wait_for_vigor_spp_intf_rem lc_intf_pcie_inb_vigor_100_base[@]
           if [ $? -ne 0 ]; then
               log_msg "Doing Force Inband interface cleanup"
               #do_inb_intf_cleanup
               if [ "$CARDINDEX" = "27061" ] ||
                  [ "$CARDINDEX" = "27068" ] ; then
                    do_inb_intf_cleanup_v2 0 07
               elif [ "$CARDINDEX" = "27062" ] ; then
                    do_inb_intf_cleanup_v2 0 09
               elif [ "$CARDINDEX" = "27066" ] ||
                    [ "$CARDINDEX" = "27067" ] ; then
                    do_inb_intf_cleanup_v2 0 02
                fi
           fi
           sleep 1
        fi

        #Potenza+10C
        if [ "$CARDINDEX" = "27060" ] ||
           [ "$CARDINDEX" = "27064" ]; then
            # Reset clock to NPU from internal oscillator
            # for next boot as Redwood is getting reset
            iofpga_reg_write_bar1 $SI5346_REG 0x2
            log_msg "Reset clock-output to NPU from internal oscillator"
        fi
        # Remove Jericho/Qumran
        if [ "$CARDINDEX" = "27066" ] ||
           [ "$CARDINDEX" = "27067" ]; then
            jericho_dev_id=88
        else
            jericho_dev_id=86
        fi
    fi

    log_msg "jericho pci device id : $jericho_dev_id"
    for bdf in $(lspci -nn | grep "14e4:$jericho_dev_id" | cut -d " " -f 1);
    do
        echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
    done

    # Remove OP TCAM
    for bdf in $(lspci -nn | grep "14e4:98" | cut -d " " -f 1);
    do
        echo 1 > /sys/bus/pci/devices/0000:${bdf}/remove;
    done

    get_card_index

    if [ "$CARDINDEX" = "27012" ]; then
        # Need to remove the device before its turned off 
        echo 1 > /sys/bus/pci/devices/0000\:0f\:00.0/remove
    fi
    if [ "$CARDINDEX" = "27066" ] ||
       [ "$CARDINDEX" = "27067" ]; then
        # Assert slice disconnect signal before putting slices in reset
        reg_val=$(iofpga_db_reg_read 0x3a0)
        reg_val=$(($reg_val | 0x8))
        iofpga_db_reg_write 0x3a0 $(printf 0x%08X $reg_val)
        sleep 0.1

        # Put all slices in reset
        iofpga_db_reg_write 0x3a0 0x0
    else
        # Assert slice disconnect signal before putting slices in reset
        reg_val=$(iofpga_reg_read 0x3a0)
        reg_val=$(($reg_val | 0x8080808))
        iofpga_reg_write 0x3a0 $(printf 0x%08X $reg_val)
        reg_val=$(iofpga_reg_read 0x3a4)
        reg_val=$(($reg_val | 0x808))
        iofpga_reg_write 0x3a4 $(printf 0x%08X $reg_val)
        sleep 0.1

        # Put all slices in reset    
        iofpga_reg_write 0x3a0 0x0
        iofpga_reg_write 0x3a4 0x0
    fi
    # Turn off DM2
    iofpga_reg_write 0x300 0x0
    log_msg "Power domain 2 is now turned off"

    # 3 sec delay is required for succesful power-off of Jericho devices
    # before powering them back up.
    #
    # FRETTA_BRINGUP_HACK_luislu
    # what is the hack: use 6 sec delay instead of 3 sec for Potenza 100 TCAM
    # why is the hack : this delay is required due to IOFPGA bug on this
    #                   specific LC (CSCuy42156).
    # what is required to remove the hack: new rev 0.5 of the IOFPGA
    if [ "$CARDINDEX" = "27011" ]; then
        sleep 6
    else
        sleep 3
    fi

    return 0
}

# Steps to bring down the device
# 2. remove the devices from the pci heirarchy
function lxc_xr_lc_disable_msfpga_devices()
{
    get_card_index
    if [ "$CARDINDEX" = "27012" ]; then
        echo 1 > /sys/bus/pci/devices/0000\:07\:00.0/remove   
        echo 1 > /sys/bus/pci/devices/0000\:08\:00.0/remove   
        echo 1 > /sys/bus/pci/devices/0000\:09\:00.0/remove   
    elif [ "$CARDINDEX" = "27013" ]; then
        echo 1 > /sys/bus/pci/devices/0000\:05\:00.0/remove   
        echo 1 > /sys/bus/pci/devices/0000\:08\:00.0/remove   
        echo 1 > /sys/bus/pci/devices/0000\:09\:00.0/remove   
        echo 1 > /sys/bus/pci/devices/0000\:0a\:00.0/remove   
        echo 1 > /sys/bus/pci/devices/0000\:0f\:00.0/remove   
    fi
   
    return 0
}

# Steps to bring down the  MPA devices 
# 1. From MIFPGA BAR 3 disable to power to both MPA's 
# 2. Once the power is disabled remove the MPA's from the
#    pci tree
# NOTE: Once the LX LXC boot's up it will power on and then do
#       pciscan to bring the MPA FPGA devices back
# Disable for following boards:
# 1) NC55-MOD-A-SE-S    (FPGA ID = 0x01518600)
# 2) NC55-MOD-A-S       (FPGA ID = 0x01518601)
# 3) NCS-57C3-MOD-SYS   (FPGA ID = 0xa11ecf03)
# 4) NCS-57C3-MODS-SYS  (FPGA ID = 0xb11ecf03)
# 5) NC57-MOD-S         (FPGA ID = 0x01518605)
function lxc_xr_lc_disable_mpafpga_devices()
{
    local platform_fpga_id
    
    platform_fpga_id=$(mifpga_reg_read 0 0x4)
#EYRIE_TBD
    case "$platform_fpga_id" in
        0x01518600 | 0x01518601)
                   # power off MPA power enable register 0x14 
                   log_msg "Turing off power Write 0x0 to 0x14 MPA CTRL BAR"
                   mifpga_reg_write 3 0x14 0x0

                   if [ -f /sys/bus/pci/devices/0000\:09\:00.0/remove ]; then 
                       log_msg "MPA1 remove path found removing"
                       echo 1 > /sys/bus/pci/devices/0000\:09\:00.0/remove
                   else 
                       log_msg "MPA1 removal path not found"
                   fi
                   if [ -f /sys/bus/pci/devices/0000\:0c\:00.0/remove ]; then 
                       log_msg "MPA2 remove path found removing"
                       echo 1 > /sys/bus/pci/devices/0000\:0c\:00.0/remove
                   else 
                       log_msg "MPA2 removal path not found"
                   fi
                   ;;
         0x01518605)
                   # power off MPA power enable register 0x14
                   log_msg "Turing off power Write 0x0 to 0x14 MPA CTRL BAR"
                   mifpga_reg_write 3 0x14 0x0

                   if [ -f /sys/bus/pci/devices/0000\:09\:00.0/remove ]; then
                       log_msg "MPA1 remove path found removing"
                       echo 1 > /sys/bus/pci/devices/0000\:09\:00.0/remove
                   else
                       log_msg "MPA1 removal path not found"
                   fi
                   if [ -f /sys/bus/pci/devices/0000\:0c\:00.0/remove ]; then
                       log_msg "MPA2 remove path found removing"
                       echo 1 > /sys/bus/pci/devices/0000\:0c\:00.0/remove
                   else
                       log_msg "MPA2 removal path not found"
                   fi
                   ;;

#Eyrie add MIFPGA fpga id case for SE variant
        0xa11ecf03 | 0xb11ecf03)
                   if [ -f /sys/bus/pci/devices/0000\:05\:00.0/remove ]; then
                       log_msg "MPA1 remove path found removing"
                       setpci -s 0:03.0 0xa0.w=0x50
                       echo 1 > /sys/bus/pci/devices/0000\:05\:00.0/remove
                   else
                       log_msg "MPA1 removal path not found"
                   fi
                   if [ -f /sys/bus/pci/devices/0000\:08\:00.0/remove ]; then
                       log_msg "MPA2 remove path found removing"
                       setpci -s 0:03.3 0xa0.w=0x50
                       echo 1 > /sys/bus/pci/devices/0000\:08\:00.0/remove
                   else
                       log_msg "MPA2 removal path not found"
                   fi
                   if [ -f /sys/bus/pci/devices/0000\:07\:00.0/remove ]; then
                       log_msg "MPA3 remove path found removing"
                       setpci -s 0:03.2 0xa0.w=0x50
                       echo 1 > /sys/bus/pci/devices/0000\:07\:00.0/remove
                   else
                       log_msg "MPA3 removal path not found"
                   fi
                   # power off MPA power enable register 0x14 MPA1 and MA2
                   log_msg "Turing off power Write 0x0 to 0x14 MPA CTRL BAR"
                   mifpga_reg_write 3 0x14 0x0
                   # power off MPA power enable register 0x84 MPA3
                   log_msg "Turing off power Write 0x0 to 0x84 MPA CTRL BAR"
                   mifpga_reg_write 3 0x84 0x0
                   ;;
        *)log_msg "Skipping $platform_fpga_id for MPA device removal"
     esac

    return 0
}

function pd_handle_release ()
{
    local vm_name=$1
    local func_exit_code=0

    case "$vm_name" in
    sysadmin)
        log_msg "pd_handle_release: Nothing to do for VM $vm_name"
        ;;

    default-sdr--*)
        log_msg "pd_handle_release: VM $vm_name"
        get_board_type
        if [ "$BOARDTYPE" = "LC" ]; then

            # On LXC down we need to power down and remove the MPA
            # device. This will apply to only to MPA based platforms 
            # we want to turn of MPA power down before DM2 power in 
            # lxc_xr_lc_disable_dnx_devices goes down and remove 
            # PCI MPA FPGA devices
            # eg - Peyto - 
            #    - MOD LC - 
            log_msg "calling lxc_xr_lc_disable_mpafpga_devices"
            lxc_xr_lc_disable_mpafpga_devices
            log_msg "calling lxc_xr_lc_disable_dnx_devices()"
            lxc_xr_lc_disable_dnx_devices
            log_msg "calling lxc_xr_lc_disable_msfpga_devices()"
            # only for coherent and macsec LC. for other cards this function
            # is nop
            lxc_xr_lc_disable_msfpga_devices
            func_exit_code=$?
        fi
        ;;

    *)
        log_err "pd_handle_release: VM $vm_name, not supported"
        ;;
    esac

    return $func_exit_code
}
readonly -f pd_handle_release

STARTING_XR_VM_NOTIF=2
function lxc_xr_lc_enable_dnx_devices()
{
    # Handling of enabling DM2 and take Jerichos out of reset has been moved
    # to card manager & slice manager plugin running from SysAdmin VM.
    # But we are re-using this hook to write to the sctrach register to
    # notify the card_mgr about starting of XR VM.
    init_iofpga_base_addr
    iofpga_reg_write $IOFPGA_HOST_EVENT_REG_ADDR $STARTING_XR_VM_NOTIF

    local pll_status
    pll_status=$(iofpga_reg_read_fretta_v2 1 0x414)
    log_msg "SI PLL status for the card : 0x$pll_status"

    return 0
}
readonly -f lxc_xr_lc_enable_dnx_devices

#
# Small routine to check if the interface exists
#
function check_intf_exists ()
{
    if [ -d /sys/devices/virtual/net/$1 ]; then
        return 0
    fi
    return 1
}

#
# lxc_xr_add_intf
#   Routine to add vlan interfaces before XR is launched
#
function lxc_xr_add_intf ()
{
    local intf
    local vlanid
    local intf_list=("${!1}")

    for i in "${intf_list[@]}";
    do
        check_intf_exists $i
        if [ $? -ne 0 ]; then
            CMD="vconfig add ${i/./ }"
            printf "%s CMD: %s\n" "$timestamp" "$CMD" >> $LXC_HOOK_LOG_FILE
            declare -F platform_log &>/dev/null && platform_log "$CMD"

            platform_log_exec "$CMD"
            if [ $? -ne 0 ]; then
                platform_log_error "Failed CMD: $CMD"
                return 1
            fi
        fi
    done
}

#
# wait_for_intf
#  Routine to use as handshake method to identify once NPU UP 
#  Once NPU UP and PCIE inband interfaces getting created then 
#  it will break loop and proceed with the routine lxc_xr_Add_intf
#
function wait_for_intf()
{
    local MAX_RETRY=90
    local retry_count=0
    local intf_list=("${!1}")
    for i in "${intf_list[@]}";
    do
        while [ $retry_count -lt $MAX_RETRY ]
        do
            check_intf_exists $i
            if [ $? -ne 0 ]; then
                log_msg "Intf not up count:$retry_count i:$i"
                let retry_count+=1
                if [ $retry_count -ge $MAX_RETRY ]; then
                    log_msg "TIME's UP for waiting PCIE interface creation"
                fi
                sleep 1
            else 
                break;
            fi
        done
    done
}

#
# pd_prelaunch_setup
#   PI counter part is present in lxc_hook.sh
#
function pd_prelaunch_setup ()
{
    local vm_name=$1
    local func_exit_code=0
    local card_index1=$(get_cmdline_card_index)

    case "$vm_name" in
    sysadmin)
        log_msg "pd_prelaunch_setup: Nothing to do for VM $vm_name"
        ;;

    default-sdr--*)
        get_board_type
        if [ "$BOARDTYPE" = "LC" ]; then

            get_card_index

            log_msg "calling lxc_xr_lc_enable_dnx_devices()"
            lxc_xr_lc_enable_dnx_devices
  
            # Vigor card index is 27044
            if [ "$CARDINDEX" = "27044" ] ||
               [ "$CARDINDEX" = "27050" ] ||
               [ "$CARDINDEX" = "27046" ] ||
               [ "$CARDINDEX" = "27055" ]; then
                wait_for_intf lc_intf_pcie_inb_vigor[@] 
                lxc_xr_add_intf lc_intf_vlan_vigor[@]
            # Vigor-100-Base
            elif [ "$CARDINDEX" = "27061" ] ||
                 [ "$CARDINDEX" = "27062" ] ||
                 [ "$CARDINDEX" = "27068" ]; then
                wait_for_intf lc_intf_pcie_inb_vigor_100_base[@]
                lxc_xr_add_intf lc_intf_vlan_vigor_100_base[@]
            elif [ "$CARDINDEX" = "27066" ] ||
                 [ "$CARDINDEX" = "27067" ]; then
                wait_for_intf lc_intf_pcie_inb_vigor_100_base[@]
                lxc_xr_add_intf lc_intf_vlan_vigor_100_base[@]

            else
                wait_for_intf lc_intf_vlan[@] 
                lxc_xr_add_intf lc_intf_vlan[@]
            fi
        else
            lxc_xr_add_intf rp_intf_vlan[@]
        fi
        set_vlan_priority_map ps-eobc.3074

        prelaunch_setup "$vm_name"
        func_exit_code=$?
        ;;

    *)
        log_err "pd_prelaunch_setup: VM $vm_name, not supported"
        ;;
    esac

    return $func_exit_code
}
readonly -f pd_prelaunch_setup

#
# Pass bootstrap CLI file to XR, invoked with $1 $2 $3 $4 as received by caller
# Currently this handles regular config only; if a need to handle admin config
# arises then the code below needs to be replicated with
# 'iosxr_config_admin.txt'.
#
function lxc_cpy_bootstrap_cli()
{
    # $XR_PATH
    populate_rootfs_vars

    # Hardcoded bootstrap CLI name - it must be called
    # this on the provided ISO disk
    local CLI=iosxr_config.txt
    
    # Temporary stored location before copy to XR
    local IN=/mnt/iso
    
    # Final location of CVAC bootstrap CLI which on XR becomes
    # /etc/sysconfig/iosxr_config.txt
    local OUT=$XR_PATH/etc/sysconfig

    log_msg "CVAC: Starting scan for bootstrap CLI on all devices"
    log_msg "CVAC: Config file in location  $IN"
    log_msg "CVAC: Config file out location $OUT"
    
    mkdir -p $IN
    if [[ $? -ne 0 ]]; then
        log_err "CVAC: Failed CMD: mkdir -p $IN"
        exit 1
    fi

    # Scan and mount all /dev/sd*, /dev/hd* and /dev/vd* devices...
    # If successfully mounted, check for bootstrap CLI and copy it to XR
    for device in $( ls /dev ); do
        if [[ $device =~ sd* || $device =~ hd* || $device =~ vd* ]]; then
            mount /dev/$device $IN -o loop
            if [[ $? -eq 0 ]]; then
                # Succesfully mounted - now look for a bootstrap CLI file
                log_msg "CVAC: Mounted /dev/$device to $IN"
                if [[ -e $IN/$CLI ]]; then
                    # Found one, copy it to XR LXC to be parsed when XR comes up
                    log_msg "CVAC: Found bootstrap CLI: $IN/$CLI"
                    cp $IN/$CLI $OUT
                    if [[ $? -eq 0 ]]; then
                        log_msg "CVAC: Copied bootstrap CLI to $OUT/$CLI"
                    else
                        log_err "CVAC: Failed to copy bootstrap CLI: $IN/$CLI to $OUT/$CLI"
                    fi
                fi
                log_msg "CVAC: Unmount /dev/$device from $IN"
                umount $IN -d
            fi
        fi
    done

    return 0
}

#
# lxc_xr_mv_intf
#   Routine to mv interfaces back to host before XR is shut down
#
function lxc_xr_mv_intf ()
{
    local intf_list=("${!1}")
    local vm_name=$2
    local func_exit_code=0

    for i in "${intf_list[@]}";
    do
        # Skip existence test
        log_msg "ip netns exec $vm_name.libvirt ip link set $i netns 1"
        ip netns exec $vm_name.libvirt ip link set $i netns 1
        func_exit_code=$?
        log_msg "result of move cmd is $func_exit_code"
    done
}

#
# handle_release_mv_interfaces
#
# hook invoked from handle_release to move interfaces
# back to host after destroying the container
# This seems necessary when there is KIM/LCND interaction
#
# The hook in lxc_cleanup_helper binary moves the physical interfaces and
#  virtual interfaces underneath them (e.g.: eth0) back to host
# Currently, this moves the ps interface back to host on sdr destroy
#
# Always return 0
#
# FIX_ME
# We have to close cases when libvirt_lxc crashes while running this script
#
function handle_release_mv_interfaces
{
    local vm_name=$1
    local func_exit_code=0

    case "$vm_name" in

    default-sdr--*)
        if [ "$BOARDTYPE" = "LC" ]; then

            get_card_index
            if [ "$CARDINDEX" = "27044" ] ||
               [ "$CARDINDEX" = "27050" ] ||
               [ "$CARDINDEX" = "27046" ] ||
               [ "$CARDINDEX" = "27055" ]; then  
                lxc_xr_mv_intf lc_intf_vlan_vigor[@] $1
            else
                lxc_xr_mv_intf lc_intf_vlan[@] $1  
            fi
#EYRIE_TBD
        else
            lxc_xr_mv_intf rp_intf_vlan[@] $1
        fi
        ;;

    *)
        #Nothing to do
        ;;
    esac

    return $func_exit_code
}

